/*-------------- Telecommunications & Signal Processing Lab ---------------
                             McGill University

Module:
  void AFsetHinfo (const char String[])
  char *AFgetUinfo (float *Sfreq, int *Nchar)

Purpose:
  Set the audio file header information string
  Get the user supplied audio file header information string

Description:
  This routine sets the information string to be written to an AFsp audio file
  header.  This file format allows for an arbitrary information string to be
  written at the end of the header.  By default the header information string
  gives date and user information.  This routine allows the user to specify an
  information string that is used in addition to, or in place of, the standard
  header information string.

  Standard Header Information:
  date:1994/01/25 19:19:39 UTC    date
  sample_rate:8012.5              sampling frequency (only if non-integer)
  user:kabal@k2.EE.McGill.CA      user
  program:CopyAudio               program name (if set using routine UTsetProg)

  Additional information records should follow this format; a named field
  terminated by a colon followed by numeric data or a character string.  For
  the purpose of this routine, records are separated by newline characters.
  The last record need not be terminated by a newline character. In the header
  itself, the newline characters are replaced by nulls.

  If the first record of the header information string is empty, i.e. the first
  character is a newline, the input header information string is concatenated
  with the standard information string.  If not, the input header information
  string replaces the standard information string.

  The header information string can be specified directly as an argument to
  AFsetHinfo or via an environment variable.  If the input string contains has
  a leading '$', the string is assumed to specify the name of an environment
  variable after the '$'.  This routine uses the value of this environment
  variable to get the header information string.

  The procedure AFgetUinfo is used internally by the audio file routines. It
  returns the header information string in a form suitable for writing directly
  to the header.

Parameters:
  AFsetHinfo:
  ->  const char []
  String -	String containing the header information or the name of an
		environment variable (with a leading $)
  AFgetUinfo:
  <-  char []
  AFgetUinfo -	Pointer to the header information string.  This string is Nchar
		characters long.  It is in a form that can be directly written
		to the header.
  ->  float *
  Sfreq -	Sampling frequency.  If this value is non-integer, a
		sample_rate record is put in the standard header information
		string.
  <-  int *
  Nchar -	Number of characters in the header information string

Author / revision:
  P. Kabal  Copyright (C) 1994
  $Revision: 1.8 $  $Date: 1994/02/12 15:24:26 $

-------------------------------------------------------------------------*/

static char rcsid[] = "$Id: AFsetHinfo.c 1.8 1994/02/12 AFsp-V1R2 $";

#include <math.h>
#include <stdlib.h>	/* getenv prototype */
#include <string.h>
#include <libtsp.h>
#include <libtsp/AFnucleus.h>
#include <libtsp/AFparms.h>
#include <libtsp/AUparms.h>

#ifndef __STDC__
#define	const
#endif

#define	MXFSTRING	32	/* "xsample_rate:+9.123456e+100" */
static char *Hstring = NULL;

void
AFsetHinfo (String)

     const char String[];

{
  int nc;
  const char *p;

/* Check for an environment variable */
  if (String[0] == '$') {
    p = getenv (&String[1]);
    if (p == NULL)
      p = "";
  }
  else
    p = String;

/* Copy the header information string to an allocated buffer */
  UTfree (Hstring);
  nc = strlen (p);
  if (nc > 0) {
    Hstring = (char *) UTmalloc (nc + 1);
    strcpy (Hstring, p);
  }
  else
    Hstring = NULL;
}

char *
AFgetUinfo (Sfreq, Nchar)

     float *Sfreq;
     int *Nchar;

{
  static char Hinfo[AU_MXINFO];
  char Csfreq[MXFSTRING+1];
  float Sfreqx;
  char *p;
  int nc;
  int i;

  if (Hstring == NULL || Hstring[0] == '\n') {

    /* Generate the standard header information */
    STcopyMax ("date:", Hinfo, AU_MXINFO);
    STcatMax (UTdate (3), Hinfo, AU_MXINFO);
    Sfreqx = *Sfreq;
    if (Sfreqx != floor ((double) Sfreqx)) {
      sprintf (Csfreq, "\nsample_rate:%.7g", Sfreqx);
      STcatMax(Csfreq, Hinfo, AU_MXINFO);
    }
    STcatMax ("\nuser:", Hinfo, AU_MXINFO);
    STcatMax (UTuserName (), Hinfo, AU_MXINFO);
    p = UTgetProg ();
    if (*p != '\0') {
      STcatMax ("\nprogram:", Hinfo, AU_MXINFO);
      STcatMax (p, Hinfo, AU_MXINFO);
    }

    /* Add the user specified header information */
    if (Hstring != NULL)
      STcatMax (Hstring, Hinfo, AU_MXINFO);
  }
  else {

    /* User specified header information only */
    STcopyMax (Hstring, Hinfo, AU_MXINFO);
  }

/* Change newlines to nulls */
  nc = strlen (Hinfo);
  for (i = 0; i < nc; ++i) {
    if (Hinfo[i] == '\n')
      Hinfo[i] = '\0';
  }

/* Return a pointer to the header information */
  *Nchar = nc;
  return Hinfo;
}
