/*-------------- Telecommunications & Signal Processing Lab ---------------
                             McGill University

Module:
  int AFreadData (FILE *fp, long int offs, float Fbuff[], int Nreq)

Purpose:
  Read data from an audio file (return float values)

Description:
  This routine returns a specified number of samples from an audio file.  The
  data in the file is converted to float format on output.  The sample data in
  the file is considered to be preceded and followed by zero-valued samples.
  Thus if offs is negative or points to beyond the number of samples in the
  file, the appropriate number of zero valued samples are returned.  The file
  must have been opened using routine AFopenRead.

  The following program fragment illustrates the use of this routine to read
  overlapping frames of data.  For the simpler case of sequential access to the
  data without overlap, the variable Lmem should be set to zero.

    fp = AFOpenRead (...);
    ...
    Lmem =...
    Lframe =...
    Nadv = Lframe-Lmem;
    offs = -Lmem;
    while (1) {
      Nout = AFreadData (fp, offs, Fbuff, Lframe);
      offs = offs+Nadv;
      if (Nout == 0)
        break;
      ...
    }

Parameters:
  <-  int
  AFreadData -	Number of data values transferred from the file.  On reaching
		reaching the end of the file, Nout may be less than Nreq, in
		which case the last Nout-Nreq elements are set to zero.  This
		value can be used by the calling routine to determine when
		the data from the file has been exhausted.
		->  FILE *AFgetInf
  fp -		File pointer for a file opened by AFopenRead
  ->  long int
  offs -	Offset into the file in samples.  If offs is positive, the
		first value returned is offs samples from the beginning of the
		data.  If offs is negative, -offs zeros are returned before the
		data in the file.  These zeros before the beginning of the data
		are counted as part of the count returned in Nout.
  <-  float []
  Fbuff -	Array of floats to receive the Nreq samples
  ->  int
  Nreq -	Number of samples requested.  Nreq may be zero.

Author / revision:
  P. Kabal  Copyright (C) 1994
  $Revision: 1.8 $  $Date: 1994/02/12 20:35:43 $

-------------------------------------------------------------------------*/

static char rcsid [] = "$Id: AFreadData.c 1.8 1994/02/12 AFsp-V1R2 $";

#include <stdio.h>
#include <libtsp.h>
#include <libtsp/AFnucleus.h>
#include <libtsp/AFparms.h>

#define MINV(a, b)	(((a) < (b)) ? (a) : (b))

int
AFreadData (fp, offs, Fbuff, Nreq)

     FILE *fp;
     long int offs;
     float Fbuff[];
     int Nreq;

{
  int n;
  int i;
  int N;
  int Nval;
  int Nout;
  struct AF_parms *Fparms;

/* Get the file format */
  Fparms = AFgetFormat (fp);
  if (Fparms->Op != FO_RO)
    UTerrorHalt ("AFreadData: File not opened for read");

/* Fill in zeros at the beginning of data */
  n = MINV (-offs, Nreq);
  for (i = 0; i < n; ++i) {
    Fbuff[i] = 0.0;
    ++offs;
  }

/* Transfer data from file */
  N = Nreq - i;
  switch (Fparms->Format) {

  case FD_FLOAT32:
    Nval = AFreadF4 (fp, offs, &Fbuff[i], N);
    break;

  case FD_LIN16:
    Nval = AFreadI2 (fp, offs, &Fbuff[i], N);
    break;

  case FD_MULAW8:
    Nval = AFreadMulaw (fp, offs, &Fbuff[i], N);
    break;

  default:
    UTerrorHalt ("AFreadData: Unsupported data format");
    break;
  }
  i = i + Nval;
  Nout = i;

/* Zeros at the end of the file */
  for (; i < Nreq; ++i) {
    Fbuff[i] = 0.0;
  }

  return Nout;
}
