/*-------------- Telecommunications & Signal Processing Lab ---------------
                             McGill University

Module:
  char *AFgetHinfo (FILE *fp, int *Ninfo)

Purpose:
  Get an AFsp audio file header information string

Description:
  This routine reads the information string from an AFsp audio file header.
  This file format allows for an arbitrary information string to be stored at
  the end of the header.  This routine retrieves this information from an AFsp
  audio file opened by AFopenRead.

  Standard Header Information:
  date:1994/01/25 19:19:39 UTC    date
  sample_rate:8012.5              sampling frequency (only if non-integer)
  user:kabal@k2.EE.McGill.CA      user
  program:CopyAudio               program name (if set using routine UTsetProg)

  The AFsp audio file header information string consists of a sequence of null
  terminated strings corresponding to the header records.  If the last record
  is not null terminated, a null is added.  The string is Ninfo characters
  long, including this terminating null.  If the audio file is not an AFsp
  file, a pointer to an empty string is returned.

Parameters:
  ->  char *
  AFgetHinfo -	Pointer to the header information string.  This is a pointer to
		a dynamically allocated storage area; each call to this routine
		reallocates this storage area.
  ->  FILE *
  fp -		File pointer for an audio file opened by AFopenRead
  <-  int *
  Ninfo -	Number of characters in the header information string

Author / revision:
  P. Kabal  Copyright (C) 1994
  $Revision: 1.4 $  $Date: 1994/02/12 15:23:12 $

-------------------------------------------------------------------------*/

static char rcsid[] = "$Id: AFgetHinfo.c 1.4 1994/02/12 AFsp-V1R2 $";

#include <libtsp.h>
#include <libtsp/AFnucleus.h>
#include <libtsp/AFparms.h>
#include <libtsp/AUparms.h>

#define MINV(a, b)	(((a) < (b)) ? (a) : (b))

static char Empty[] = "";
static char *Hinfo = NULL;

char *
AFgetHinfo (fp, Ninfo)

     FILE *fp;
     int *Ninfo;

{
  struct AU_head Fhead;
  long int Nbytes;
  int ml;
  int nc;

/* Get the size of the file */
  Nbytes = AFfileSize (fp);
  if (Nbytes < AU_LHMIN)
    UTerrorHalt ("AFgetHinfo: File header too short");

/* Read in the fixed portion of the header */
  (void) AFreadFile (fp, 0, &Fhead, U4, AU_LHMIN / U4);

/* Check the file magic */
  ml = AFbyteOrder ();
  if (Fhead.Magic != FM_SUN[ml] || Fhead.AFsp != FM_AFSP[ml])
    return Empty;

/* Swap the header data if necessary */
  if (ml == SF_EL)
    AFswapBytes (&Fhead, U4, AU_LHMIN / U4);

/* Warnings, error checks */
  if (Fhead.Lhead < AU_LHMIN || Fhead.Lhead > Nbytes) {
    UTwarn ("AFgetHinfo: Invalid header length");
    return Empty;
  }

/* Read the header into an allocated area */
  nc = MINV (AU_MXINFO, Fhead.Lhead - AU_LHMIN);
  UTfree (Hinfo);
  Hinfo = (char *) UTmalloc (nc + 1);
  (void) AFreadFile (fp, AU_LHMIN, Hinfo, 1, nc);

/* Add a trailing null if necessary */
  if (Hinfo[nc-1] != '\0') {
    Hinfo[nc] = '\0';
    ++nc;
  }

/* Return a pointer to the header information */
  *Ninfo = nc;
  return Hinfo;
}
