/*-------------- Telecommunications & Signal Processing Lab ---------------
                             McGill University

Module:
  InfoAudio AFile

Purpose:
  Print information about an audio file

Description:
  This program opens an audio file and prints information about the file to
  standard output.  The intention is that information about the internal
  structure of the file be available in a form that can be easily parsed by a
  shell script.  One possible use is as part of a shell script to invoke a
  playback command that needs to know the data format and sampling frequency
  for the audio file.

  The output for a typical audio file is
     AFsp audio file: /usr/local/src/audio/InfoAudio/xx.au
       Number of samples : 23808  1994/02/04 14:12:50 UTC
       Sampling frequency: 8000 Hz
       Number of channels: 1 (16-bit integer)
    File name: xx.au
    Header length: 104
    Sampling frequency: 8000
    No. samples: 23808
    No. channels: 1
    Data type: integer16      - mu-law8, integer16 or float32
    Data swap: swap           - native, or swap
    --Header information records--
    date:1994/02/04 14:12:50 UTC
    user:library@k2.EE.McGill.CA
    program:CopyAudio

Parameters:
  [options] AFile
  options:
  -P PARMS, --parameters=PARMS  Parameters to be used for headerless input
                              files.  See the description of the environment
                              variable AFNHparms below for the format of the
                              parameter specification.
  -b, --brief                 Print only the data format parameters.
  -h, --help                  Print a list of options and exit.
  -v, --version               Print the version number and exit.

  Environment variables:
  AFNHparms -	Defines the data format for headerless or non-standard input
		audio files.  The string consists of a list of parameters
		separated by commas.  The form of the list is
		"Format, Start, Sfreq, Swapb, Nchan, ScaleF"
		In the following, an asterisk marks the shortest possible short
		form accepted for keyword parameters.
		Format - File data format,
		  "undef*ined" - Headerless files will be rejected
		  "m*u-law8    - 8-bit mu-law data
		  "i*nteger16" - 16-bit twos complement integer data
		  "f*loat32"   - 32-bit floating point
		Start -  byte offset to the start of data (integer value)
		Sfreq -  sampling frequency in Hz (floating point number)
		Swapb  - Data byte swap parameter
		  "native" - no byte swapping
		  "little*-endian" - file data is in little-endian byte order
		     and will be swapped if the current host uses big-endian
		     byte order
		  "big*-endian" - data is in big-endian byte order and will be
		     swapped swapped if the current host uses little-endian
		     byte order
		  "swap" - swap the data bytes
		Nchan -  number of channels, the data consists of interleaved
		  samples from Nchan channels (integer value)
		ScaleF - Scaling applied to the data from the file (floating
		  point number)
		The default values correspond to the following string
		"undefined, 0, 8000., native, 1, 1.0"
  AUDIOFILES -	Specifies a colon separated list of directories to be searched
		when opening the input files.

Author / revision:
  P. Kabal  Copyright (C) 1994
  $Revision: 1.10 $  $Date: 1994/02/14 03:40:44 $

-------------------------------------------------------------------------*/

static char rcsid[] = "$Id: InfoAudio.c 1.10 1994/02/14 AFsp-V1R2 $";

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <libtsp.h>
#include <libtsp/AFparms.h>
#include <libtsp/AFnucleus.h>
#include "InfoAudio.h"

#ifndef EXIT_SUCCESS
#define EXIT_SUCCESS	0	/* Normally in stdlib.h */
#endif

int
main (argc, argv)

     int argc;
     const char *argv[];

{
  char NHparms[MXSTRING+1];
  int Bflag;
  char Fname[FILENAME_MAX+1];
  FILE *fpout;
  FILE *fp;
  long int Nsamp, Nchan;
  float Sfreq;
  char *Hinfo;
  int Ninfo;
  int n;
  struct AF_parms *Fparms;
  static char *DataFormat[] = {
    "undefined",
    "mu-law8",
    "integer16",
    "float32"
  };
  static char *DataSwap[] = {
    "big-endian",
    "little-endian",
    "native",		/* Fparms->Swapb should be native or swap only */
    "swap"
  };

/* Option handling */
  IAoptions (argc, argv, NHparms, &Bflag, Fname);

/* Set up standard audio file handling */
  if (NHparms[0] != '\0')
    AFsetNH (NHparms);
  else
    AFsetNH ("$AFNHparms");
  FLpathList (Fname, "$AUDIOFILES", Fname);

/* Open the audio file */
  fpout = stdout;
  if (Bflag != 0)
    fpout = NULL;
  fp = AFopenRead (Fname, &Nsamp, &Nchan, &Sfreq, fpout);

/* Get the file structure information */
  Fparms = AFgetFormat (fp);

/* Print the information */
  printf ("File name: %s\n", Fname);
  printf ("Header length: %d\n", Fparms->Start);
  printf ("Sampling frequency: %.5g\n", Sfreq);
  printf ("No. samples: %d\n", Nsamp);
  printf ("No. channels: %d\n", Nchan);
  printf ("Data type: %s\n", DataFormat[Fparms->Format]);
  printf ("Data swap: %s\n", DataSwap[Fparms->Swapb]);

/* Print the header information strings */
  if (Bflag == 0) {
    Hinfo = AFgetHinfo (fp, &Ninfo);
    if (Ninfo > 0) {
      printf ("--Header information records--\n");
      n = 0;
      while (n < Ninfo) {
	printf ("%s\n", &Hinfo[n]);
	n = n + strlen (&Hinfo[n]) + 1;
      }
    }
  }

  AFclose (fp);

  return EXIT_SUCCESS;
}
