/*-------------- Telecommunications & Signal Processing Lab ---------------
                             McGill University

Module:
  FiltAudio [options] -f FilterFile AFileIn AFileOut

Purpose:
  Filter data from an audio file

Description:
  This program takes an audio file and a filter coefficient file as input and
  produces a filtered output audio file.  Subsampling and interpolation factors
  can be optionally specified.  This program supports three types of filters,
  FIR, all-pole, and general IIR.  Filters are specified in filter files.

  Filter Files:
  The first record of a filter file indicates the type of filter.
    !FIR	- FIR filter, direct form
    !IIR	- IIR filter, cascade of biquad sections
    !ALL	- All-pole filter, direct form
  Subsequent records contain filter coefficients.  Comment records ('!' in the
  first position of the record) can be interspersed amongst the data.  Data
  records are free form, with data values separated by white space (blanks,
  tabs and newlines).  Commas can also be used to separate data values, but
  only within records, i.e. a comma should not appear at the end a record.
  Records are limited to 1024 characters.

  FIR filters are specified by the direct-form coefficients h[i],
            N-1       -i
    H(z) = SUM h[i] z    .
           i=0

  IIR filters are implemented as the cascade of biquadratic filter sections,
  where each section has a z-transform,
             h(i,0)*z**2 + h(i,1)*z + h(i,2)
    H(i,z) = ------------------------------- .
                z**2 + h(i,3)*z + h(i,4)


  All-pole filters are specified by direct-form feeback coefficients,
            1                    N-1       -i
    H(z) = ----  ,  where C(z) = SUM h[i] z    .
           C(z)                  i=0

  For FIR filters, a sample rate change may be affected with interpolation and
  subsampling.  Let Ir and Nsub be the interpolation and subsampling factors,
  respectively.  Conceptually, the operations for FIR filters are as follows.
  1) Ir-1 zeros are inserted between adjacent samples of the frequency shifted
     input to increase the sampling rate by a factor of Ir.
  2) The increased rate signal is filtered.
  3) The result of the filtering is subsampled by a factor of Nsub to form
     the output signal.

  The initial filter alignment and the number of output samples can be
  specified with options.  The filter alignment specifies the position of the
  filter relative to the input date for calculating the first output sample.
  For FIR filters, this alignment is relative to the increased rate input
  sequence.  Specifically, let the number of samples in the input file be Nin.
  The input can be considered to be an array x(0),...,x(Nin-1).  The increased
  rate sequence is xi(.), with xi(k*Ir)=x(k).  The first output sample is
  calculated with the beginning of the impulse response of the filter aligned
  with xi(idoffs).  The array xi(.) can be considered to be of length Nin*Ir;
  the first non-zero sample is xi(0)=x(0), the last non-zero sample is
  xi((Nin-1)*Ir).  Conceptually, the impulse impulse response is moved in
  steps of Nsub to create the output samples.

  The intent is that the output samples be a subset of the values the would be
  obtained if the infinite length sequence formed by padding out the input data
  on either end with zeros were to be filtered.  To this end, the filter
  calculations need warm-up points, particularly for the case that the initial
  filter alignment is not at the beginning of the input data.  For FIR filters,
  this is taken into account by reading previous input values into the filter
  memory.  For IIR filters, previous outputs are also needed as warm-up points.
  If the initial alignment is near the beginning of the data, the IIR filter is
  run from the beginning of the data to generate the warm-up points.  For
  larger alignment offsets, the IIR filter is backed up for a maximum of 1000
  samples to provide the warm-up points.

  If the initial filter alignment is not explicitly specified it is chosen to
  be zero, except for odd-length symmetric or anti-symmetric FIR filters for
  which it is is chosen to be (Ncof-1)/2.  If the number of output samples
  is not explicitly set, it is chosen to be Ir*Nin/Nsub.  For the case of
  Ir=1 and Nsub=1, this results in the same number of output samples as input
  samples.  If the initial filter alignment, offs, is explicitly specified, the
  number of output samples is chosen to be (Ir*Nin-offs)/Nsub.  This value
  can be overridden by explicitly setting the number of output samples.

Parameters:
  [options] -f FiltFile AFileI AFileO
  options:
  -f FILTFILE, --filter_file=FILTFILE
                              Filter file name.
  -s NSUB, --subsample=NSUB   Subsampling factor, default 1.  Subsampling can
                              only be used with FIR filters.
  -i IR, --interpolate=IR     Interpolation factor, default 1.  Interpolation
                              can only be used with FIR filters.
  -a OFFS, --alignment=OFFS   Alignment of data relative to the filter.  The
                              first output sample is calculated with the
                              beginning of the filter response aligned with
                              the specified sample of the interpolated data
                              sequence.  The interpolated data sequence is
                              formed from the data in the input file, by
                              inserting Ir-1 zeros between each input sample.
                              Offset zero corresponds to the first sample from
                              the input file; offset Ir corresponds to the
                              second sample from the input file.  For most
                              filter types, the default alignment is 0.  For
                              symmetric or anti-symmetric FIR filters with an
                              odd number of coefficients, the default is
                              (Ncof-1)/2, corresponding to no delay between the
                              input and output files.  Note that with this
                              value of offset, part of the start-up transient
                              is inevitably lost.
  -n NOUT, --number_samples=NOUT  Number of output samples to be calculated.
  -D DFORMAT, --data_format=DFORMAT
                              Data format for the ouput file, default integer
                              "mu-law8"   - 8-bit mu-law
                              "integer16" - 16-bit twos complement integer
                              "float32"   - 32-bit IEEE float
  -N, --noheader              Create a headerless output file.
  -S SWAP, --swap=SWAP        Output file byte order, SWAP is "big-endian",
                              "little-endian", "native", or "swap", default
                              big-endian.
  -P PARMS, --parameters=PARMS  Parameters to be used for headerless input
                              files.  See the description of the environment
                              variable AFNHparms below for the format of the
                              parameter specification.
  -I INFO, --info=INFO        Header information string.  See the description
                              of the environment variable AFinfo below.
  -h, --help                  Print a list of options and exit.
  -v, --version               Print the version number and exit.

  Environment variables:
  AFNHparms -	Defines the data format for headerless or non-standard input
		audio files.  The string consists of a list of parameters
		separated by commas.  The form of the list is
		"Format, Start, Sfreq, Swapb, Nchan, ScaleF"
		In the following, an asterisk marks the shortest possible short
		form accepted for keyword parameters.
		Format - File data format,
		  "undef*ined" - Headerless files will be rejected
		  "m*u-law8    - 8-bit mu-law data
		  "i*nteger16" - 16-bit twos complement integer data
		  "f*loat32"   - 32-bit floating point
		Start -  byte offset to the start of data (integer value)
		Sfreq -  sampling frequency in Hz (floating point number)
		Swapb  - Data byte swap parameter
		  "native" - no byte swapping
		  "little*-endian" - file data is in little-endian byte order
		     and will be swapped if the current host uses big-endian
		     byte order
		  "big*-endian" - data is in big-endian byte order and will be
		     swapped swapped if the current host uses little-endian
		     byte order
		  "swap" - swap the data bytes
		Nchan -  number of channels, the data consists of interleaved
		  samples from Nchan channels (integer value)
		ScaleF - Scaling applied to the data from the file (floating
		  point number)
		The default values correspond to the following string
		"undefined, 0, 8000., native, 1, 1.0"
  AFinfo -	Sets an information string which is written to the output audio
		file header.  By default date and user information is written
		in the information part of the audio file header.  This
		environment variable allows the user to specify an information
		string that is used in addition to or in place of the standard
		header information string.
		  Standard Header Information:
		  date:1994/01/25 19:19:39 UTC    date
		  user:kabal@k2.EE.McGill.CA      user
		  program:CopyAudio               program name
		Additional information records should follow this format; a
		named field terminated by a colon and then numeric data or a
		character string.  For the purpose of this routine, records
		are terminated by newline characters.  However in the header
		itself, the newline characters are replaced by nulls.
		If the first record of the header information string is empty,
		i.e. the first character is a newline, the header information
		string is concatenated with the standard information string.
		If not, the input header information string replaces the
		standard information string.
  AUDIOFILES -	Specifies a colon separated list of directories to be searched
		when opening the input files.

Author / revision:
  P. Kabal  Copyright (C) 1994
  $Revision: 1.8 $  $Date: 1994/02/16 04:32:44 $

-------------------------------------------------------------------------*/

static char rcsid[] = "$Id: FiltAudio.c 1.8 1994/02/16 AFsp-V1R2 $";

#include <limits.h>		/* INT_MIN */
#include <stdio.h>
#include <stdlib.h>
#include <libtsp.h>
#include <libtsp/FIparms.h>
#include "FiltAudio.h"

#ifndef EXIT_SUCCESS
#define EXIT_SUCCESS	0	/* Normally in stdlib.h */
#endif

int
main (argc, argv)

     int argc;
     const char *argv[];

{
  int FormatO;
  float SfreqI, SfreqO;
  char NHparms[MXINFO+1];
  char Hinfo[MXINFO+1];
  char FiltFile[FILENAME_MAX+1];
  char FnameI[FILENAME_MAX+1];
  char FnameO[FILENAME_MAX+1];
  long int Nsamp, Nchan;
  FILE *fpI;
  FILE *fpO;
  long int idoffs, Nout;
  int FiltType;
  int Nsub, Ir;
  int Dalign;
  int Ncof, Nsec;
  float h[MAXCOF];

/* Get the input parameters */
  FAoptions (argc, argv, &FormatO, &Nout, &idoffs, &Nsub, &Ir, Hinfo, NHparms,
	     FiltFile, FnameI, FnameO);

/* Open the input audio file */
  if (NHparms[0] != '\0')
    AFsetNH (NHparms);
  else
    AFsetNH ("$AFNHparms");
  FLpathList (FnameI, "$AUDIOFILES", FnameI);
  fpI = AFopenRead (FnameI, &Nsamp, &Nchan, &SfreqI, stdout);
  if (Nchan != 1)
    UTerrorHalt ("%s: Multiple input channels not supported", PROGRAM);

/* Read the coefficient file */
  FiltType = FIreadFilt (FiltFile, MAXCOF, h, &Ncof, stdout);
  switch (FiltType) {
  case FI_IIR:
    Nsec = Ncof / 5;
    break;
  case FI_FIR:
  case FI_ALL:
    break;
  default:
    UTerrorHalt ("%s: Invalid filter type", PROGRAM);
    break;
  }
  if (Ncof <= 0)
    UTerrorHalt ("%s: No coeffients specified", PROGRAM);
  if (FiltType != FI_FIR && (Nsub != 1 || Ir != 1))
    UTerrorHalt ("%s: Sample rate change only supported for FIR filters",
		 PROGRAM);

/* Open the output audio file */
  SfreqO = (Ir * SfreqI) / Nsub;
  if (Hinfo[0] != '\0')
    AFsetHinfo (Hinfo);
  else
    AFsetHinfo ("$AFinfo");
  printf ("\n");
  FLbackup (FnameO);
  fpO = AFopenWrite (FnameO, FormatO, Nchan, &SfreqO, stdout);

/* Default alignment */
  Dalign = (idoffs == INT_MIN);
  if (Dalign) {
    if (FiltType == FI_FIR && Ncof % 2 != 0 && ARfSymArray (h, Ncof) != 0)
      idoffs = (Ncof - 1)/2;
    else
      idoffs = 0;
  }
  if (Nout == 0) {
    if (Dalign)
      Nout = (Ir * Nsamp) / Nsub;
    else
      Nout = (Ir * Nsamp - idoffs) / Nsub;
  }

/* Filtering */
  if (FiltType == FI_FIR) {
    if (Nsub == 1 && Ir == 1)
      FAfiltFIR (fpI, fpO, Nout, h, Ncof, idoffs);
    else
      FAfiltSI (fpI, fpO, Nout, h, Ncof, Nsub, Ir, idoffs);
  }
  else if (FiltType == FI_IIR)
    FAfiltIIR (fpI, fpO, Nout, (const float (*)[5]) h, Nsec, idoffs);
  else if (FiltType == FI_ALL)
    FAfiltAP (fpI, fpO, Nout, h, Ncof, idoffs);
  else
    UTerrorHalt ("%s: Unsupported filter type", PROGRAM);

/* Close the audio files */
  AFclose (fpI);
  AFclose (fpO);

  return EXIT_SUCCESS;
}
