/*------------- Telecommunications & Signal Processing Lab -------------
                           McGill University

Module:
void FAoptions (int argc, const char *argv[], int *FormatO, int *Nsub, int *Ir,
		char Hinfo[], char NHparms [],char FiltFile[], char FnameI[],
		char FnameO[])

Purpose:
  Decode options for FiltAudio

Description:
  This routine decodes options for FiltAudio.

Parameters:
  ->  int
  argc -	Number of command line arguments
  ->  const char * []
  argv -	Pointer to argument strings
  <-  int *
  FormatO -	Output file data format code
  <-  char [MXINFO+1]
  Hinfo -	Header information string, default ""
  <-  char [MXSTRING+1]
  NHparms -	Parameters for headerless input files, default ""
  <-  char [FILENAME_MAX+1]
  FnameI -	Input audio file name
  <-  char [FILENAME_MAX+1]
  FiltFile -	Input filter file name
  <-  char [FILENAME_MAX+1]
  FnameO -	Output audio file name

Author / revision:
  P. Kabal  Copyright (C) 1994
  $Revision: 1.4 $  $Date: 1994/02/11 01:23:27 $

----------------------------------------------------------------------*/

static char rcsid[] = "$Id: FAoptions.c 1.4 1994/02/11 AFsp-V1R2 $";

#include <limits.h>		/* INT_MIN */
#include <stdlib.h>		/* prototype for exit */
#include <stdio.h>
#include <libtsp.h>
#include <libtsp/AFparms.h>
#include "FiltAudio.h"

#ifndef EXIT_SUCCESS
#define EXIT_SUCCESS	0	/* Normally in stdlib.h */
#endif

#define	NOPT	((sizeof OptTable / sizeof OptTable[0]) - 1)

/* Option tables and usage message */
static const char *nullTable[] = { NULL };
static const char *OptTable[] = {
  "-f**:", "--f*ilter_file=*",
  "-s**:", "--s*ubsample=*",
  "-i**:", "--int*erpolate=*",
  "-a**:", "--a*lignment=*",
  "-n**:", "--number_samples=*",
  "-D**:", "--d*ata_format=*",
  "-N", "--n*oheader",
  "-S**:", "--sw*ap=*",
  "-P**:", "--p*arameters=*",
  "-I**:", "--inf*o=*",
  "-h", "--h*elp",
  "-v", "--v*ersion",
  "--",
  NULL
};
static const char Usage[] = "\
Usage: %s [options] -f FiltFile AFileI AFileO\n\
Options:\n\
  -f FiltFile, --filter_file=FiltFile  Filter file name.\n\
  -s Nsub, --subsample=Nsub   Subsamping factor.\n\
  -i Ir, --interpolate=Ir     Interpolation factor.\n\
  -a OFFS, --alignment=OFFS   Offset for first output.\n\
  -n NOUT, --number_samples=NOUT  Number of output samples.\n\
  -D DFORMAT, --data_format=DFORMAT  Data format for the ouput file,\n\
			      DFORMAT is \"mulaw\", \"integer\" or \"float\".\n\
  -N, --noheader              Headerless output file.\n\
  -S SWAP, --swap=SWAP        Output file byte order, SWAP is \"big-endian\",\n\
                              \"little-endian\", \"native\", or \"swap\".\n\
  -P PARMS, --parameters=PARMS  Parameters for headerless input files.\n\
  -I INFO, --info=INFO        Header information string.\n\
  -h, --help                  Print a list of options and exit.\n\
  -v, --version               Print the version number and exit.";

void
FAoptions (argc, argv, FormatO, Nout, idoffs, Nsub, Ir, Hinfo, NHparms,
	   FiltFile, FnameI, FnameO)

     int argc;
     const char *argv[];
     int *FormatO;
     long int *Nout;
     long int *idoffs;
     int *Nsub;
     int *Ir;
     char Hinfo[MXINFO+1];
     char NHparms[MXSTRING+1];
     char FiltFile[FILENAME_MAX+1];
     char FnameI[FILENAME_MAX+1];
     char FnameO[FILENAME_MAX+1];

{
  static const char *FormatTab[] = { "m*u-law8", "i*nteger16", "f*loat32" };
  static const char *SwapTab[] =
    {"big*-endian", "little*-endian", "native", "swap" };
  int State;
  const char *OptArg;
  const char **optt;

  int nF;
  int nFF;
  int Dformat;
  int NH;
  int Swapb;
  int nsub, ir;
  int n;
  int offs, nout;

/* Defaults */
  Dformat = FD_LIN16;
  NH = 0;
  Swapb = DS_EB;
  nsub = 1;
  ir = 1;
  offs = INT_MIN;
  nout = 0;
  NHparms[0] = '\0';
  Hinfo[0] = '\0';

/* Initialization */
  UTsetProg (PROGRAM);
  nF = 0;
  nFF = 0;

/* Decode options */
  State = 0;
  optt = OptTable;
  while (1) {
    n = UTgetOption (&State, argc, argv, optt, &OptArg);
    if (n == -1)
      break;
    switch (n) {
    case 0:
      /* Filename argument */
      ++nF;
      if (nF == 1)
	STcopyMax (OptArg, FnameI, FILENAME_MAX);
      else if (nF == 2)
	STcopyMax (OptArg, FnameO, FILENAME_MAX);
      else
	UTerrorHalt ("%s: Too many filenames specified", PROGRAM);
      break;
    case 1:
    case 2:
      /* Filter file */
      nFF = 1;
      STcopyMax (OptArg, FiltFile, FILENAME_MAX);
      break;
    case 3:
    case 4:
      /* Subsampling factor */
      if (STdecInt1 (OptArg, &nsub) || nsub <= 0)
	UTerrorHalt ("%s: Invalid subsampling factor", PROGRAM);
      break;
    case 5:
    case 6:
      /* Interpolation factor */
      if (STdecInt1 (OptArg, &ir) || ir <= 0)
	UTerrorHalt ("%s: Invalid interpolation factor", PROGRAM);
      break;
    case 7:
    case 8:
      /* Alignment */
      if (STdecInt1 (OptArg, &offs))
	UTerrorHalt ("%s: Invalid alignment offset", PROGRAM);
      break;
    case 9:
    case 10:
      /* Number of samples */
      if (STdecInt1 (OptArg, &nout) || nout <= 0)
	UTerrorHalt ("%s: Invalid number of samples", PROGRAM);
      break;
    case 11:
    case 12:
      /* Data format */
      n = STkeyMatch (OptArg, FormatTab);
      switch (n) {
      case 0:
	Dformat = FD_MULAW8;
	break;
      case 1:
	Dformat = FD_LIN16;
	break;
      case 2:
	Dformat = FD_FLOAT32;
	break;
      default:
	UTerrorHalt ("%s: Invalid data format specification", PROGRAM);
      }
      break;
    case 13:
    case 14:
      /* Headerless output file */
      NH = 1;
      break;
    case 15:
    case 16:
      /* Output file byte order */
      n = STkeyMatch (OptArg, SwapTab);
      switch (n) {
      case 0:
	Swapb = DS_EB;
	break;
      case 1:
	Swapb = DS_EL;
	break;
      case 2:
	Swapb = DS_NATIVE;
	break;
      case 3:
	Swapb = DS_SWAP;
	break;
      default:
	UTerrorHalt ("%s: Invalid byte swap specification", PROGRAM);
      }
      break;
    case 17:
    case 18:
      /* Headerless input parameters */
      STcopyMax (OptArg, NHparms, MXSTRING);
      break;
    case 19:
    case 20:
      /* Header information string */
      STcopyMax (OptArg, Hinfo, MXINFO);
      break;
    case NOPT-4:
    case NOPT-3:
      /* Help */
      UTwarn (Usage, PROGRAM);
      exit (EXIT_SUCCESS);
      break;
    case NOPT-2:
    case NOPT-1:
      /* Version */
      printf ("%s: %s\n", PROGRAM, VERSION);
      exit (EXIT_SUCCESS);
      break;
    case NOPT:
      /* Stop interpreting options */
      optt = nullTable;
      break;
    default:
      /* Option error */
      UTerrorHalt (Usage, PROGRAM);
      break;
    }
  }

/* Checks, add defaults */
  if (nF < 2)
    UTerrorHalt ("%s: Too few files specified", PROGRAM);
  if (nFF < 1)
    UTerrorHalt ("%s: No filter file specified", PROGRAM);

/* Set return values */
  *FormatO = Dformat + NH * FD_NH + Swapb * FD_BO;
  *Nout = nout;
  *idoffs = offs;
  *Nsub = nsub;
  *Ir = ir;
}
