/*-------------- Telecommunications & Signal Processing Lab ---------------
                             McGill University

Module:
  CopyAudio [options] AFileA AfileB ... AFileO

Purpose:
  Copy audio files (combine, scale and shift data)

Description:
  This program copies input audio file(s) to an output audio file.  The samples
  in the output file are linear combinations of the input channels.  Options
  are available to change the sampling frequency and data format of the output
  file.  Another option allows for selection of the range of input samples
  which will appear in the output file.

  The sample limits option allows for the shifting of the data in the inputs
  files with respect to the data in the output file.  The data is conceptually
  organized into frames, with samples from individual channels making up a
  frame.  Frame numbers start at 0.  Frame numbers below zero may be specified
  - the samples in these frames have zero values.  Similarly, frame numbers
  beyond the end-of-file may be specified - these are also assumed to have
  zero-valued values.

  For more than one input file, channels assignments are considere to cross
  between input files.  Consider two input files, with the first having 3
  channels, and the second having 2 channels.  For the combined input, these
  channels would be labelled A to E, with channels A, B and C from the first
  input file and channels D and E from the second file.  The channel selection
  option allows a linear combination of the input channels to appear in each
  output channel.

Parameters:
  [options] AFile1 ... AfileN AFileOut
  options:
  -s SFREQ, --srate=SFREQ     Sampling frequency for the output file, default
                              from the input audio file(s)
  -D DFORMAT, --data_format=DFORMAT
                              Data format for the ouput file, default integer
                              "mu-law8"   - 8-bit mu-law
                              "integer16" - 16-bit twos complement integer
                              "float32"   - 32-bit IEEE float
  -l L:U, --limits=L:U        Frame limits (numbered from zero).  The default
                              frame limits correspond to the longest input
                              audio file.
  -cA GAINS, --chanA=GAINS    Channel gain factors for output channel A.  The
                              output data for the specified output channel is
                              created by adding scaled samples from the
                              specified input channels.  The default is have
                              each output channel equal to the corresponding
                              input channel.  Channels are labelled A, B, C,
                              D, and E.
  -cB GAINS, --chanB=GAINS    Channel gain factors for output channel B.
   ...
  -cE GAINS, --chanE=GAINS    Channel gain factors for output channel E.
  -N, --noheader              Create a headerless output file.
  -S SWAP, --swap=SWAP        Output file byte order, SWAP is "big-endian",
                              "little-endian", "native", or "swap", default
                              big-endian.
  -P PARMS, --parameters=PARMS  Parameters to be used for headerless input
                              files.  See the description of the environment
                              variable AFNHparms below for the format of the
                              parameter specification.
  -I INFO, --info=INFO        Header information string.  See the description
                              of the environment variable AFinfo below.
  -h, --help                  Print a list of options and exit.
  -v, --version               Print the version number and exit.

  The gain factor string takes the form
    [+/-] [gain *] chan +/- [gain *] chan ...)
  where chan is A through E.  Consider a case with a two channel input file.
  The output file is to have 3 channels.  Output channel A is input channel
  A.  Output channel B is the average of input channels A and B.  Output
  channel C is input channel B.  The following output channel specifications
  accomplish this mapping.
   --chanA=A --chanB="0.5*A + 0.5*B" --chanC=B
  Note that * is a special character to Unix shells and should appear only
  within quotes to prevent the shell from interpreting it.

  Environment variables:
  AFNHparms -	Defines the data format for headerless or non-standard input
		audio files.  The string consists of a list of parameters
		separated by commas.  The form of the list is
		"Format, Start, Sfreq, Swapb, Nchan, ScaleF"
		In the following, an asterisk marks the shortest possible short
		form accepted for keyword parameters.
		Format - File data format,
		  "undef*ined" - Headerless files will be rejected
		  "m*u-law8    - 8-bit mu-law data
		  "i*nteger16" - 16-bit twos complement integer data
		  "f*loat32"   - 32-bit floating point
		Start -  byte offset to the start of data (integer value)
		Sfreq -  sampling frequency in Hz (floating point number)
		Swapb  - Data byte swap parameter
		  "native" - no byte swapping
		  "little*-endian" - file data is in little-endian byte order
		     and will be swapped if the current host uses big-endian
		     byte order
		  "big*-endian" - data is in big-endian byte order and will be
		     swapped swapped if the current host uses little-endian
		     byte order
		  "swap" - swap the data bytes
		Nchan -  number of channels, the data consists of interleaved
		  samples from Nchan channels (integer value)
		ScaleF - Scaling applied to the data from the file (floating
		  point number)
		The default values correspond to the following string
		"undefined, 0, 8000., native, 1, 1.0"
  AFinfo -	Sets an information string which is written to the output audio
		file header.  By default date and user information is written
		in the information part of the audio file header.  This
		environment variable allows the user to specify an information
		string that is used in addition to or in place of the standard
		header information string.
		  Standard Header Information:
		  date:1994/01/25 19:19:39 UTC    date
		  user:kabal@k2.EE.McGill.CA      user
		  program:CopyAudio               program name
		Additional information records should follow this format; a
		named field terminated by a colon and then numeric data or a
		character string.  For the purpose of this routine, records
		are terminated by newline characters.  However in the header
		itself, the newline characters are replaced by nulls.
		If the first record of the header information string is empty,
		i.e. the first character is a newline, the header information
		string is concatenated with the standard information string.
		If not, the input header information string replaces the
		standard information string.
  AUDIOFILES -	Specifies a colon separated list of directories to be searched
		when opening the input files.

Author / revision:
  P. Kabal  Copyright (C) 1994
  $Revision: 1.10 $  $Date: 1994/02/22 15:44:25 $

-------------------------------------------------------------------------*/

static char rcsid[] = "$Id: CopyAudio.c 1.10 1994/02/22 AFsp-V1R2 $";

#include <stdio.h>
#include <stdlib.h>
#include <libtsp.h>
#include <libtsp/AFnucleus.h>
#include "CopyAudio.h"

#ifndef EXIT_SUCCESS
#define EXIT_SUCCESS	0	/* Normally in stdlib.h */
#endif

#define MAXV(a, b)	(((a) > (b)) ? (a) : (b))

int
main (argc, argv)

     int argc;
     const char *argv[];

{
  int FormatO;
  long int Lim[2];
  char NHparms[MXSTRING+1];
  char Hinfo[MXINFO+1];
  float Chgain[MXCHAN][MXCHAN];
  long int NchanO;
  char Fname[MXFILE][FILENAME_MAX+1];
  char *FnameO;
  int Nfiles, Nifiles;
  long int Nchan[MXIFILE], NchanT;
  long int Nframe;
  FILE *fp[MXIFILE], *fpO;
  int i;
  long int Nsamp;
  float Sfreq[MXIFILE], SfreqO;
  int Sfdiff;
  float Sf;

  UTsetProg (PROGRAM);

/* Get the input parameters */
  CPoptions (argc, argv, &FormatO, &SfreqO, Lim, NHparms, Hinfo, Chgain,
	     &NchanO, Fname, &Nfiles);

/* Open the input files */
  Nifiles = Nfiles - 1;
  if (NHparms[0] != '\0')
    AFsetNH (NHparms);
  else
    AFsetNH ("$AFNHparms");
  NchanT = 0;
  Nframe = 0;
  for (i = 0; i < Nifiles; ++i) {
    FLpathList (Fname[i], "$AUDIOFILES", Fname[i]);
    fp[i] = AFopenRead (Fname[i], &Nsamp, &Nchan[i], &Sfreq[i], stdout);
    NchanT = NchanT + Nchan[i];
    Nframe = MAXV (Nframe, MSiCeil (Nsamp, Nchan[i]));
  }
  if (NchanT > MXCHAN)
    UTerrorHalt ("%s: Too many input channels", PROGRAM);
  printf ("\n");

/* Sampling frequency */
  Sfdiff = 0;
  Sf = Sfreq[0];
  for (i = 1; i < Nifiles; ++i) {
    if (Sfreq[i] != Sfreq[i-1])
      Sfdiff = 1;
    Sf = Sf + Sfreq[i];
  }
  if (Sfdiff)
    UTwarn ("%s: Input sampling frequencies differ", PROGRAM);
  Sf = Sf / Nifiles;
  if (SfreqO == 0.0)
    SfreqO = Sf;

/* Open the output file */
  if (Hinfo[0] != '\0')
    AFsetHinfo (Hinfo);
  else
    AFsetHinfo ("$AFinfo");
  FnameO = Fname[Nfiles-1];
  FLbackup (FnameO);
  if (NchanO == 0)
    NchanO = NchanT;
  fpO = AFopenWrite (FnameO, FormatO, NchanO, &SfreqO, stdout);

/* Scale and copy samples */
  if (Lim[0] > Lim[1]) {
    Lim[0] = 0;
    Lim[1] = Nframe - 1;
  }
  Nchan[Nfiles-1] = NchanO;
  fp[Nfiles-1] = fpO;
  CPcopySamp (fp, Nchan, Nfiles, Lim, Chgain);

/* Close the audio files */
  for (i = 0; i < Nifiles; ++i) {
    AFclose (fp[i]);
  }
  AFclose (fpO);

  return EXIT_SUCCESS;
}
