/*------------- Telecommunications & Signal Processing Lab -------------
                           McGill University

Module:
  void CPoptions (int argc, const char *argv[], int *FormatO, float *SfreqO,
		  long int Lim[], char NHparms[], char Hinfo[],
		  float Chgain[][], long int *NchanO, char Fname[][],
		  int *Nfiles)

Purpose:
  Decode options for CopyAudio

Description:
  This routine decodes options for CopyAudio.

Parameters:
  ->  int
  argc -	Number of command line arguments
  ->  const char * []
  argv -	Pointer to argument strings
  <-  int *
  FormatO -	Output file data format code
  <-  float *
  SfreqO -	Output file sampling frequency
  <-  long int [2]
  Lim -		Sample limits, set to (0, -1) if not explicitly specified
  <-  char [MXSTRING+1]
  NHparms -	Parameters for headerless input files, default ""
  <-  char [MXINFO+1]
  Hinfo -	Header information string, default ""
  <-  float [MXCHAN][MXCHAN]
  Chgain -	Array of channel gains, default unit matrix
  <-  long int *
  NchanO -	Number of output channels, zero if channel gains not specified
  <-  char [MXFILE][FILENAME_MAX+1]
  Fname -	File names
  <-  int *
  Nfiles -	Number of input file names

Author / revision:
  P. Kabal  Copyright (C) 1994
  $Revision: 1.14 $  $Date: 1994/02/22 15:43:36 $

----------------------------------------------------------------------*/

static char rcsid[] = "$Id: CPoptions.c 1.14 1994/02/22 AFsp-V1R2 $";

#include <stdlib.h>		/* prototype for exit */
#include <stdio.h>
#include <libtsp.h>
#include <libtsp/AFparms.h>
#include "CopyAudio.h"

#ifndef EXIT_SUCCESS
#define EXIT_SUCCESS	0	/* Normally in stdlib.h */
#endif

#define	CHST	7		/* Count for first channel gain option */
#define	NOPT	((sizeof OptTable / sizeof OptTable[0]) - 1)

/* Option tables and usage message */
static const char *nullTable[] = { NULL };
static const char *OptTable[] = {
  "-s**:", "--sr*ate=*",
  "-D**:", "--d*ata_format=*",
  "-l**:", "--l*imits=*",
  "-cA**:", "--chanA*=*",
  "-cB**:", "--chanB*=*",
  "-cC**:", "--chanC*=*",
  "-cD**:", "--chanD*=*",
  "-cE**:", "--chanE*=*",
  "-N", "--n*oheader",
  "-S**:", "--sw*ap=*",
  "-p**:", "--p*arameters=*",
  "-I**:", "--i*nfo=*",
  "-h", "--h*elp",
  "-v", "--v*ersion",
  "--",
  NULL
};
static const char Usage[] = "\
Usage: %s [options] AFileI1 AFileI2 ... AFileO\n\
Options:\n\
  -s SFREQ, --srate=SFREQ     Sampling frequency for the output file.\n\
  -D DFORMAT, --data_format=DFORMAT  Data format for the ouput file,\n\
			      DFORMAT is \"mulaw\", \"integer\" or \"float\".\n\
  -l L:U, --limits=L:U        Frame limits.\n\
  -cA GAINS, --chanA=GAINS    Channel gain factors for output channel A.\n\
   ...\n\
  -cE GAINS, --chanE=GAINS    Channel gain factors for output channel E.\n\
  -N, --noheader              Headerless output file.\n\
  -S SWAP, --swap=SWAP        Output file byte order, SWAP is \"big-endian\",\n\
                              \"little-endian\", \"native\", or \"swap\".\n\
  -p PARMS, --parameters=PARMS  Parameters for headerless input files.\n\
  -I INFO, --info=INFO        Header information string.\n\
  -h, --help                  Print a list of options and exit.\n\
  -v, --version               Print the version number and exit.";

void
CPoptions (argc, argv, FormatO, SfreqO, Lim, NHparms, Hinfo, Chgain, NchanO,
	   Fname, Nfiles)

     int argc;
     const char *argv[];
     int *FormatO;
     float *SfreqO;
     long int Lim[2];
     char NHparms[MXSTRING+1];
     char Hinfo[MXINFO+1];
     float Chgain[MXCHAN][MXCHAN];
     long int *NchanO;
     char Fname[MXFILE][FILENAME_MAX+1];
     int *Nfiles;

{
  static const char Ch[] = "ABCDE";
  static const char *FormatTab[] = { "m*u-law8", "i*nteger16", "f*loat32" };
  static const char *SwapTab[] =
    {"big*-endian", "little*-endian", "native", "swap" };
  int State;
  const char *OptArg;
  const char **optt;

  int nF;
  int i;
  int Chset[MXCHAN];
  int n;
  float Sfreq;
  int limL;
  int limU;
  int NH;
  int Swapb;
  int Dformat;
  long int Nchan;

/* Defaults */
  Sfreq = 0.0;
  limL = 0;
  limU = -1;
  Dformat = FD_LIN16;
  NH = 0;
  Swapb = DS_EB;
  NHparms[0] = '\0';
  Hinfo[0] = '\0';
  for (i = 0; i < MXCHAN; ++i) {
    Chset[i] = 0;
    ARfZero (Chgain[i], MXCHAN);
    Chgain[i][i] = 1.0;
  }

/* Initialization */
  nF = 0;

/* Decode options */
  State = 0;
  optt = OptTable;
  while (1) {
    n = UTgetOption (&State, argc, argv, optt, &OptArg);
    if (n == -1)
      break;
    switch (n) {
    case 0:
      /* Filename argument */
      ++nF;
      if (nF > MXFILE)
	UTerrorHalt ("%s: Too many filenames specified", PROGRAM);
      STcopyMax (OptArg, Fname[nF-1], FILENAME_MAX);
      break;
    case 1:
    case 2:
      /* Sampling rate */
      if (STdecFloat1 (OptArg, &Sfreq) || Sfreq <= 0.0)
	UTerrorHalt ("%s: Invalid sampling frequency", PROGRAM);
      break;
    case 3:
    case 4:
      /* Data format */
      n = STkeyMatch (OptArg, FormatTab);
      switch (n) {
      case 0:
	Dformat = FD_MULAW8;
	break;
      case 1:
	Dformat = FD_LIN16;
	break;
      case 2:
	Dformat = FD_FLOAT32;
	break;
      default:
	UTerrorHalt ("%s: Invalid data format specification", PROGRAM);
      }
      break;
    case 5:
    case 6:
      /* Limits specification */
      if (STdecRange (OptArg, &limL, &limU) || limL > limU)
	UTerrorHalt ("%s: Invalid limits specification", PROGRAM);
      break;
    case CHST:
    case CHST+1:
    case CHST+2:
    case CHST+3:
    case CHST+4:
    case CHST+5:
    case CHST+6:
    case CHST+7:
    case CHST+8:
    case CHST+9:
      /* Channel gain expressions */
      i = (n - CHST) / 2;
      CPdecChan (OptArg, Chgain[i]);
      Chset[i] = 1;
      break;
    case 17:
    case 18:
      /* Headerless output file */
      NH = 1;
      break;
    case 19:
    case 20:
      /* Output file byte order */
      n = STkeyMatch (OptArg, SwapTab);
      switch (n) {
      case 0:
	Swapb = DS_EB;
	break;
      case 1:
	Swapb = DS_EL;
	break;
      case 2:
	Swapb = DS_NATIVE;
	break;
      case 3:
	Swapb = DS_SWAP;
	break;
      default:
	UTerrorHalt ("%s: Invalid byte swap specification", PROGRAM);
      }
      break;
    case 21:
    case 22:
      /* Headerless input parameters */
      STcopyMax (OptArg, NHparms, MXSTRING);
      break;
    case 23:
    case 24:
      /* Header information string */
      STcopyMax (OptArg, Hinfo, MXINFO);
      break;
    case NOPT-4:
    case NOPT-3:
      /* Help */
      UTwarn (Usage, PROGRAM);
      exit (EXIT_SUCCESS);
      break;
    case NOPT-2:
    case NOPT-1:
      /* Version */
      printf ("%s: %s\n", PROGRAM, VERSION);
      exit (EXIT_SUCCESS);
      break;
    case NOPT:
      /* Stop interpreting options */
      optt = nullTable;
      break;
    default:
      /* Option error */
      UTerrorHalt (Usage, PROGRAM);
      break;
    }
  }

/* Checks, add defaults */
  if (nF < 2)
    UTerrorHalt ("%s: Too few files specified", PROGRAM);

  for (i = MXCHAN - 1; i >= 0; --i) {
    if (Chset[i] != 0)
      break;
  }
  Nchan = i + 1;
  for (i = 0; i < Nchan - 1; ++i) {
    if (Chset[i] == 0)
      UTerrorHalt ("CPoptions: No specification for output channel %c", Ch[i]);
  }

/* Set return values */
  *FormatO = Dformat + NH * FD_NH + Swapb * FD_BO;
  *SfreqO = Sfreq;
  Lim[0] = limL;
  Lim[1] = limU;
  *NchanO = Nchan;
  *Nfiles = nF;
}
