/*------------- Telecommunications & Signal Processing Lab -------------
                           McGill University

Module:
  CompAudio [options] AFileA AFileB

Purpose:
  Compare audio files, printing statistics

Description:
  This program gathers and prints statistics for one or two input audio files.
  The signal-to-noise ratio (SNR) of the second file relative to the first file
  is printed.  For this calculation, the first audio file is used as the
  reference signal.  The "noise" is the difference between sample values in
  the files.

  For each file of the input files, the following statistical quantities are
  calculated and printed.
  Mean:
    Xm = SUM x(i) / N
  Standard deviation:
    sd = sqrt [ (SUM x(i)**2 - Xm**2) / (N-1) ]
  Max value:
    Xmax = max (x(i))
  Min value:
    Xmin = min (x(i))

  For data which is restricted to the range [-32768,+32767], two additional
  counts (if nonzero) are reported.
  Number of overloads:
    Count of values taking on values -32768 or +32767, along with the number of
    such runs.  For 16-bit data from a saturating A/D converter, the presence
    of "overloads" is a indication of a clipped signal.
  Number of anomalous transitions:
    Dividing the 16-bit data range into 2 positive regions and 2 negative
    regions, an anomalous transition is a transition from a sample value in the
    most positive region directly to a sample value in the most negative region
    or vice-versa.  Such a transition is an indication of a wrapped around
    sample values, or other problems (e.g., byte-swapped data).

  An optional delay range can be specified when comparing files.  The samples
  in file B are delayed relative to those in file A by each of the delay values
  in the delay range.  For each delay, the SNR with optimized gain factor (see
  below) SNR is calculated.  For the delay corresponding to the largest SNR,
  the full regalia of file comparison values is reported.

  Conventional SNR:
                          SUM xa(i)**2
    SNR = -------------------------------------------- .
          SUM xa(i)**2 - 2 SUM xa(i)*xb(i) + SUM xb(i)
    The corresponding value in dB is printed.

  SNR with optimized gain factor:
    SNR = 1 / (1 - r**2) ,
    where r is the correlation coefficient,
                     SUM xa(i)*xb(i)
    r = ---------------------------------------- .
        sqrt [ (SUM xa(i)**2) * (SUM xb(i)**2) ]
   The SNR value in dB is printed.  This SNR calculation corresponds to using
   an optimized gain factor Sf for file B,
        SUM xa(i)*xb(i)
   Sf = --------------- .
         SUM xb(i)**2

  Segmental SNR:
    This is the average of SNR values calculated for 16 ms segments.  For each
    16 ms segment, the SNR is calculated as
                               SUM xa(i)**2
    SS(k) = log10 (1 + ---------------------------) .
                      0.01 + SUM [xa(i)-xb(i)]**2
    The term 0.01 in the denominator prevents a divide by zero.  This value is
    appropriate for data with values significantly larger than 0.01.  The
    additive unity term discounts segments with SNR's less than unity.  The
    final average segmental SNR is calculated as
    SSNR = 10 * log10 ( 10**[SUM SS(k) / N] - 1 ) dB.
    The subtraction of the unity term tends to compensate for the unity term
    in SS(k).

  If any of these SNR values is infinite, only the optimal gain factor is
  printed as part of the message,
    "File A = Sf * File B".

Parameters:
  [options] AFileA AFileB
  options:
  -d DL:DU, --delay=DL:DU     Specify a delay range.  Each delay in the delay
                              range represents a delay of file B relative to
                              file A.  The default range is 0:0.
  -P PARMS, --parameters=PARMS  Parameters to be used for headerless input
                              files.  See the description of the environment
                              variable AFNHparms below for the format of the
                              parameter specification.
  -h, --help                  Print a list of options and exit.
  -v, --version               Print the version number and exit.

  Environment variables:
  AFNHparms -	Defines the data format for headerless or non-standard input
		audio files.  The string consists of a list of parameters
		separated by commas.  The form of the list is
		"Format, Start, Sfreq, Swapb, Nchan, ScaleF"
		In the following, an asterisk marks the shortest possible short
		form accepted for keyword parameters.
		Format - File data format,
		  "undef*ined" - Headerless files will be rejected
		  "m*u-law8    - 8-bit mu-law data
		  "i*nteger16" - 16-bit twos complement integer data
		  "f*loat32"   - 32-bit floating point
		Start -  byte offset to the start of data (integer value)
		Sfreq -  sampling frequency in Hz (floating point number)
		Swapb  - Data byte swap parameter
		  "native" - no byte swapping
		  "little*-endian" - file data is in little-endian byte order
		     and will be swapped if the current host uses big-endian
		     byte order
		  "big*-endian" - data is in big-endian byte order and will be
		     swapped swapped if the current host uses little-endian
		     byte order
		  "swap" - swap the data bytes
		Nchan -  number of channels, the data consists of interleaved
		  samples from Nchan channels (integer value)
		ScaleF - Scaling applied to the data from the file (floating
		  point number)
		The default values correspond to the following string
		"undefined, 0, 8000., native, 1, 1.0"
  AUDIOFILES -	Specifies a colon separated list of directories to be searched
		when opening the input files.

Author / revision:
  P. Kabal  Copyright (C) 1994
  $Revision: 1.11 $  $Date: 1994/02/11 01:29:52 $

----------------------------------------------------------------------*/

static char rcsid[] = "$Id: CompAudio.c 1.11 1994/02/11 AFsp-V1R2 $";

#include <stdlib.h>		/* prototype for exit */
#include <stdio.h>
#include <libtsp.h>
#include "CompAudio.h"

#ifndef EXIT_SUCCESS
#define EXIT_SUCCESS	0	/* Normally in stdlib.h */
#endif

#define SEGTIME	16E-3		/* Segment size in seconds */
#define DSFREQ	8000.		/* Default sampling frequency */

int
main (argc, argv)

     int argc;
     const char *argv[];

{
  char NHparms[MXSTRING+1];
  char FnameA[FILENAME_MAX+1];
  char FnameB[FILENAME_MAX+1];
  int delayL, delayU, delayM;
  int Nfiles;
  FILE *fpA;
  FILE *fpB;
  long int NsampA, NsampB, NchanA, NchanB;
  float Sfreq, SfreqA, SfreqB;
  struct Stats_F StatsA, StatsB;
  struct Stats_T StatsT;
  long int Nsseg;

/*  Get the input parameters */
  CAoptions (argc, argv, &delayL, &delayU, NHparms, FnameA, FnameB);
  if (FnameB[0] == '\0')
    Nfiles = 1;
  else
    Nfiles = 2;

/* Open the input files */
  if (NHparms[0] != '\0')
    AFsetNH (NHparms);
  else
    AFsetNH ("$AFNHparms");
  FLpathList (FnameA, "$AUDIOFILES", FnameA);
  fpA = AFopenRead (FnameA, &NsampA, &NchanA, &SfreqA, stdout);
  if (NchanA > 1)
    UTerrorHalt ("%s: Multiple input channels not supported", PROGRAM);

  if (Nfiles == 2) {
    FLpathList (FnameB, "$AUDIOFILES", FnameB);
    fpB = AFopenRead (FnameB, &NsampB, &NchanB, &SfreqB, stdout);
    if (NchanB > 1)
      UTerrorHalt ("%s: Multiple input channels not supported", PROGRAM);
  }

/* Sampling frequency */
  if (Nfiles == 2) {
    if (NsampA != NsampB)
      UTwarn ("%s: Number of samples differ, %d : %d\n", PROGRAM,
	      NsampA, NsampB);
    if (SfreqA == 0.0)
      Sfreq = SfreqB;
    else if (SfreqB == 0.0)
      Sfreq = SfreqA;
    else
      Sfreq = 0.5 * (SfreqA + SfreqB);
    if (SfreqA != SfreqB && Sfreq > 0.0)
      UTwarn ("%s: Sampling frequencies differ, using %.0f", PROGRAM, Sfreq);
  }
  else
    Sfreq = SfreqA;
  
  if (Sfreq <= 0.0) {
    Sfreq = DSFREQ;
    UTwarn ("%s: Sampling frequency assumed to be %.0f Hz", PROGRAM, Sfreq);
  }

/* Choose a block size which is a multiple of 16 ms long */
  if (Nfiles == 2) {
    Nsseg = MSdNint (SEGTIME * Sfreq);
    if (Nsseg <= 0)
      UTerrorHalt ("%s: Segment size is zero", PROGRAM);
  }

  if (NsampA <= 0 && (Nfiles == 1 || NsampB <= 0))
    exit (EXIT_SUCCESS);

/* Individual file statistics */
  CAstats (fpA, &StatsA);
  if (Nfiles == 2)
    CAstats (fpB, &StatsB);

/* Find the cross file statistics over the delay range */
  if (Nfiles == 2)
    CAcomp (fpA, fpB, Nsseg, delayL, delayU, &delayM, &StatsA, &StatsB,
            &StatsT);

/* Close the files */
  AFclose (fpA);
  if (Nfiles == 2)
    AFclose (fpB);

/* File A statistics */
  printf ("\n");
  if (StatsT.Ndiff > 0 || delayL > delayU)
    printf (" File A:\n");
  CAprstat (&StatsA);

/* File comparisons */
  if (Nfiles == 2) {

    if (StatsT.Ndiff == 0 && delayL <= delayU) {
      /* Identical files */
      if (delayL < delayU)
	printf ("\n File A = File B  (delay = %d)\n", delayM);
      else
	printf ("\n File A = File B\n");
    }
    else {
      /* File B statistics */
      printf (" File B:\n");
      CAprstat (&StatsB);

      if (delayL <= delayU) {
	if (delayL < delayU)
	  printf (" Best match at delay = %d\n", delayM);
	CAprcorr (&StatsA, &StatsB, &StatsT);
      }
    }
  }
  return EXIT_SUCCESS;
}
