/*------------- Telecommunications & Signal Processing Lab --------------
                          McGill University
Module:
  void CAcorr (FILE *fpA, FILE *fpB, int delay, long int Nsseg,
	       struct Stats_T *Stats)

Purpose:
  Gather correlation statistics for two audio files

Description:
  This routine gathers cross-statistics for audio files.  The statistics are
  the sum of cross-products, number of differences, maximum difference,
  number of difference runs, and the accumulated log SNR values for segments
  of length Nsseg.

Parameters:
  ->  FILE *
  fpA -		File pointer for file A
  ->  FILE *
  fpB -		File pointer for file B
  ->  int
  delay -	delay of file B relative to file A
  ->  int
  Nsseg -	Segment length in samples for segmental SNR computations
  <-  struct Stats_T *
  Stats -       Structure containing the file statistics

Author / revision:
  P. Kabal  Copyright (C) 1994
  $Revision: 1.3 $  $Date: 1994/02/12 23:44:34 $

-----------------------------------------------------------------------*/

static char rcsid[] = "$Id: CAcorr.c 1.3 1994/02/12 AFsp-V1R2 $";

#include <math.h>		/* fabs, log10 */
#include <libtsp.h>
#include "CompAudio.h"

#define NBUF	2560
#define MAXV(a, b)	(((a) > (b)) ? (a) : (b))

void
CAcorr (fpA, fpB, delay, Nsseg, Stats)

     FILE *fpA;
     FILE *fpB;
     int delay;
     long int Nsseg;
     struct Stats_T *Stats;

{
  int Inrun;
  static struct Stats_T Init_T = {
    0, 0.0, 0.0, 0, 0, 0.0};
  float Xa[NBUF];
  float Xb[NBUF];
  int i;
  long int ioffs;
  int Na;
  int Nb;
  int N;
  int diffa;
  int diffb;
  double Sx2;
  double Sd2;
  double Sxy;
  int k;

/* Initialization */
  *Stats = Init_T;
  Inrun = 0;
  Sx2 = 0.0;
  Sd2 = 1e-2;
  k = 0;

/* Read the audio files */
  ioffs = 0;
  diffa = MAXV (0, -delay);
  diffb = MAXV (0, delay);

  while (1) {
    Na = AFreadData (fpA, ioffs+diffa, Xa, NBUF);
    Nb = AFreadData (fpB, ioffs+diffb, Xb, NBUF);
    ioffs = ioffs + NBUF;
    if (Na <= 0 && Nb <= 0)
      break;

    /* Data comparisons */
    Sxy = 0.0;
    N = MAXV (Na, Nb);

    for (i = 0; i < N; ++i) {

      if (Xa[i] != Xb[i]) {
        if (Inrun == 0) {
          ++Stats->Nrun;
          Inrun = 1;
        }
        ++Stats->Ndiff;
        Stats->Diffmax = MAXV (Stats->Diffmax, fabs((double)(Xa[i] - Xb[i])));
      }
      else {
        Inrun = 0;
      }

      /* Cross products */
      Sxy = Sxy + (double) Xa[i] * (double) Xb[i];

      /* Segmental SNR update */
      Sx2 = Sx2 + (double) Xa[i] * (double) Xa[i];
      Sd2 = Sd2 + (double) (Xb[i] - Xa[i]) * (double) (Xb[i] - Xa[i]);
      ++k;
      if (k >= Nsseg) {
        Stats->SNRlog = Stats->SNRlog + log10 (1.0 + Sx2 / Sd2);
        ++Stats->Nseg;
        Sx2 = 0.0;
        Sd2 = 1e-2;
        k = 0;
      }

    }
    Stats->Sxy = Stats->Sxy + Sxy;

  }

}
