/*------------- Telecommunications & Signal Processing Lab --------------
                          McGill University
Module:
  void CAcomp (FILE *fpA, FILE *fpB, int Nsseg, int delayL, int delayU,
	       int *delayM, struct Stats_F *StatsA, Stats_F *StatsB,
               struct Stats_T *Stats)

Purpose:
  Gather correlation statistics for two audio files over a range of delays

Description:
  This routine gathers statistics for an audio files over a range of delays.
  The delay values runs from delayL to delayU.  For each delay the gain
  optimized SNR is calculated.  This value is printed for each delay value if
  delayL < delayU.  The cross-file statistics for the delay which maximizes the
  SNR are returned.

Parameters:
  ->  FILE *
  fpA -		File pointer for file A
  ->  FILE *
  fpB -		File pointer for file B
  ->  long int
  Nsseg -	Segment length in samples for segmental SNR computations
  ->  int
  delayL -	Start delay value.  Normally delayL <= delayU.  If this
		condition is not satisfied, no statistics are calculated.
  ->  int
  delayU -	End delay value
  <-  int *
  delayM -	Delay of file B relative to file A which maximizes the gain
		adjusted SNR.
  ->  struct Stats_F *
  StatsA -	Structure containing the file statistics for file A
  ->  struct Stats_F *
  StatsB -	Structure containing the file statistics for file B
  <-  struct Stats_T *
  Stats -       Structure containing the cross-file statistics corresponding
		to the delay value delayM

Author / revision:
  P. Kabal  Copyright (C) 1994
  $Revision: 1.3 $  $Date: 1994/01/31 17:13:28 $

-----------------------------------------------------------------------*/

static char rcsid[] = "$Id: CAcomp.c 1.3 1994/01/31 AFsp-V1R2 $";

#include <math.h>		/* log10 */
#include "CompAudio.h"

#ifndef sun
#include <float.h>
#else
#include <values.h>
#define	DBL_MAX		MAXDOUBLE
#endif

#define DB(x)		(10.0 * log10 (x))

void
CAcomp (fpA, fpB, Nsseg, delayL, delayU, delayM, StatsA, StatsB, StatsT)

     FILE *fpA;
     FILE *fpB;
     long int Nsseg;
     int delayL;
     int delayU;
     int *delayM;
     struct Stats_F *StatsA;
     struct Stats_F *StatsB;
     struct Stats_T *StatsT;

{
  struct Stats_T Statst;
  int delay;
  double SNRmax;
  double SF;
  double denom;
  double SNRdB;

/* Loop over the delays */
  SNRmax = -DBL_MAX;
  for (delay = delayL; delay <= delayU; ++delay) {

    /* Cross product terms */
    CAcorr (fpA, fpB, delay, Nsseg, &Statst);

    if (delayL != delayU) {
      if (StatsB->Sx2 > 0.0)
        SF = Statst.Sxy / StatsB->Sx2;
      else
        SF = 1.0;
      denom = StatsA->Sx2 * StatsB->Sx2 - Statst.Sxy * Statst.Sxy;
      if (denom != 0.0) {
        SNRdB = DB ((StatsA->Sx2 * StatsB->Sx2) / denom);
        printf (" Delay: %2d,  SNR = %-8.5g dB", delay, SNRdB);
        printf ("  (Gain factor for File B = %.5g)\n", SF);
      }
      else {
        SNRdB = DBL_MAX;
        printf (" Delay: %2d,  File A = %.5g * File B\n", delay, SF);
      }
    }
    if (SNRdB > SNRmax) {
      SNRmax = SNRdB;
      *StatsT = Statst;
      *delayM = delay;
    }
  }
}
