#ifndef _nodes_H_
#define _nodes_H_

#include <stdio.h>
#include <stdlib.h>
#include "object.h"

/*                        NODE_

    The base class NODE_ is derived from class SVOBJECT_ (sortable object)
    and defines basic states that will be generated during the search process.
    In other words, it defines the (abstract, since we're dealing with a base
    class) objects that the search space consists of. Every class that is
    derived from NODE_ must have the following functions:

    do_operator(): generates and returns one successor, i.e., a new state,
                   when operator n is applied. Returns NULL when operator n 
                   cannot be applied.
    or expand():   returns a linked list of successors (the linked list must 
                   be built by using the NODE_ *next field). This function
                   is an alternative for do_operator(), either one has to be
                   implemented! 
    equal():       determines if 2 objects are the same, must return 1 if
                   true and 0 if false.
    display():     displays the object.

    Note that the virtual function eval() that is used to determine the order
    of objects is not actually used in class NODE_, therefore it just
    returns 1 (it can't be pure virtual because is it called in solve()).
    For a real usage of this function see class BEST_NODE_ and UNI_NODE_.

*/

class NODE_ : public SVOBJECT_
{
    public:
        NODE_();

        virtual NODE_ *do_operator(int) const;
	virtual NODE_ *expand(int) const;
	     // both of these not pure virtual since either may be implemented 
	virtual int equal(const VOBJECT_ &) const = 0;
        virtual int eval(const SVOBJECT_ &) const;
             // not pure virtual since it's only needed in shortest path search
	virtual void display() const = 0;

	NODE_ *getnext() const;
	void setnext(NODE_ *);
        void setparent(NODE_ *);
        NODE_ *getparent() const;
    private:
        NODE_ *parent,     // to trace the solution path
              *next;
};



/*                        UNI_NODE_

    The base class UNI_NODE_ is derived from class NODE_ and defines
    a special kind of NODE_'s: nodes that will be generated during the
    search process of a uniform cost search. These nodes will be placed in
    an ordered list, the order of 2 nodes is determined by eval() based
    on their 'g-values'.

    G is a cost associated with the node: it's a measure of the cost
    of getting from the start node to the current node. This value is computed
    by compute_g().

*/

class UNI_NODE_ : public NODE_
{
    public:
        UNI_NODE_();
        int eval(const SVOBJECT_ &) const;
        int get_g() const;
        void set_g(int);
    private:
        int g;		// cost of getting from the start node to this node
};



/*                       BEST_NODE_

    The base class BEST_NODE_ is derived from class UNI_NODE_ which in
    turn is derived from class NODE_. Class BEST_NODE_  defines
    a special kind of NODE_'s: nodes that will be generated during the
    search process of a best first search. These nodes will be placed in
    an ordered list, the order of 2 nodes is determined by eval() based
    on their 'f-values'.

    G (see unode.h) and F are costs associated with the node. G is a
    measure of the cost of getting from the start node to the current node
    and F the sum of G and H, the estimated cost of getting from the current
    node to the goal node. These values are computed by compute_g() and
    compute_h() respectively.

*/

class BEST_NODE_ : public UNI_NODE_
{
    public:
        BEST_NODE_();
        int eval(const SVOBJECT_ &) const;
        int get_f() const;
        void set_f(int);
    private:
        int
            f;          // f is g + h
};



/*                       AONODE_

    Class AONODE_ defines nodes that will be generated in an AND-OR search
    process. It is derived from class NODE_. AONODE_ is a base class for
    class ORNODE_ and ANDNODE_ and should never be used for direct derivation.

*/
 
#define OR 0
#define AND 1
#define UNSOLVABLE 0
#define SOLVED 1
#define DUNNO -1

class AONODE_ : public NODE_
{
    public:
	AONODE_();
	AONODE_(int);
	AONODE_(int, int);

        void settype(int);
        void setsolved(int);
        int getsolved() const;
        int gettype() const;
        int getn_left() const;
	void incn_left();
        void decn_left();
    private:
        int type,        // is this an AND node or OR node?
            solved,      // is the node labeled SOLVED, UNSOLVED or DUNNO?
            n_left;      // number of successors left to be solved (AND node),
                         // or: number of successors that may be still be
                         // solved (OR node)
};


/*                     ANDNODE_

    Class ANDNODE_ is derived from class AONODE_. It must be used in a 
    AND-OR search when a set of sub-problems, i.e, a set of new nodes,
    is generated that ALL have to be solved. In this case the user should
    create an ANDNODE_, by new ANDNODE_(), and pass every node representing
    a sub-problem to this ANDNODE_: ANDNODE_::addsucc(some_ornode)).
    Alternatively, an ANDNODE_ may be created by calling the second
    constructor: new ANDNODE_(no_of_nodes) and then the successor nodes
    may be passed by calling ANDNODE_::setsucc(n, node_n).

*/

class ANDNODE_ : public AONODE_
{
    public:
	ANDNODE_();
	ANDNODE_(int no_nodes);
	~ANDNODE_();

	void setsucc(int, AONODE_ *);    // set successor n in succlist to...
	void addsucc(AONODE_ *);         // add a successor to succlist
        int getn_nodes() const;
	AONODE_ *getsucc(int) const;     // get successor n from succlist

	void display() const;
        int equal(const VOBJECT_ &) const;
    private:
	int n_nodes;	                 // number of nodes in succlist
	AONODE_ **succlist;		 // this node's successors
};



/*                        ORNODE_

    Class ORNODE_ is derived from class AONODE_, which in turn is derived
    from class NODE_. It is used in the process of an AND-OR search and
    should be used in conjunction with class AOSEARCH_. Nodes that are meant
    to be generated in an AND-OR search should be derived from class ORNODE_.

*/


class ORNODE_ : public AONODE_
{
    public:
	ORNODE_();

	void setsucc(AONODE_ *);
	AONODE_ *getsucc() const;
    private:
	AONODE_ *succ;			// this node's successor
};

#endif

