;;; -*- Mode: LISP; Syntax: Common-lisp; Package: USER -*-

;;;; Dependency-directed search facility
;;;; Version 7, 4/4/90

;;; Copyright (c) 1986, 1987, 1988, 1989, 1990 Kenneth D. Forbus, 
;;; University of Illinois, Johan de Kleer and Xerox Corporation.  
;;; All rights reserved.

;; This version finds all consistent solutions.
;; By using continuations or flavorizing, it can be turned into
;;  a generator that will produce solutions on demand.

(defvar *debug-dds* t)

(defmacro debug-dds (str &rest args)
  `(if *debug-dds* (format t ,str ,@ args)))

(defun DD-Search (choice-sets end &aux answer marker choices)
  (cond ((null choice-sets)
	 (debug-dds "~%    DDS: Found solution.")
	 (eval end) nil)
	(t (setq marker (list 'DDS (car choice-sets)))
	   (setq choices (car choice-sets))
	   (dolist (choice choices)
	     (debug-dds "~%    DDS: Considering ~A..." choice)
	     (cond ((false? choice) ;skip if known loser
		    (debug-dds "~%    DDS: ~A already known nogood." choice))
		   ((true? choice) ;skip useless if known
		    (debug-dds "~%    DDS: ~A true by implication." choice)
		    (DD-Search (cdr choice-sets) end)
		    (return nil))
		   (t (debug-dds "~%    DDS: Assuming ~A." choice)
		      (with-Contradiction-Handler (ltre-ltms *ltre*)
			#'(lambda (clauses ltms &aux asns)
			    (debug-dds "~%    DDS: Entering handler for ~A with ~A~A."
				    choice clauses
				    (mapcar #'violated-clause? clauses))
			    (dolist (cl clauses)
			      (setq asns (assumptions-of-clause cl))
			      (debug-dds "~%    DDS: Assumptions are: ~A"
				      (mapcar #'view-node asns))
			      (dolist (asn asns)
				(when (or (equal choice (view-node asn))
					  (and (listp choice) (eq (car choice) ':NOT)
					       (equal (cadr choice) (view-node asn))))
				  (throw marker
					 (cons :LOSERS (delete asn asns)))))))
			(setq answer (catch marker
				       (Assuming (list choice) *ltre*
					 (run-rules *ltre*) ;run tests incrementally
				         (DD-Search (cdr choice-sets) end))))
			(when (and (listp answer)
				   (eq (car answer) ':LOSERS))
			  (debug-dds "~%    DDS: ~A inconsistent with ~A."
				     choice (mapcar #'view-node (cdr answer)))
;			  (assert! `(:not (:and ,choice
;					      ,@ (mapcar #'view-node (cdr answer))))
;				   ':dd-search-nogood)
))))))))

;;;; A familiar example

(defun Test-DD-search (&optional (debugging? t))
  (in-LTRE (create-ltre "DDS Test" :debugging debugging?))
  (eval '(rule ((:true A) (:true C))
	       (rassert! (:not (:and A C)) ':domain-nogood)))
  (eval '(rule ((:true B) (:true E))
	       (rassert! (:not (:and B E)) ':domain-nogood)))
  (DD-Search '((A B) (C D) (E F)) 
	     '(show-DD-test-solution)))
	    
(defun show-DD-test-solution (&aux result)
  (dolist (var '(F E D C B A))
    (when (true? var *ltre*) (push var result)))
  (format t "~% Consistent solution: (~A)." result))
