;; -*- Lisp -*-

;;;; Database for Tiny Rule Engine using JTMS
;;;; Version 7, 3/19/1991

;; Copyright (c) 1989, 1990, 1991 Kenneth D. Forbus, Northwestern University,
;; and Johan de Kleer and the Xerox Corporation.
;; All rights reserved.

;;;; Database structure and contents

(defstruct (class :conc-name 
		  (:print-function class-print-procedure))
  name    ; Corresponding symbol
  jtre    ; JTRE it is part of.
  facts   ; Associated facts
  rules)  ; Associated rules

 (defun class-print-procedure (r st ignore)
   (declare (ignore ignore))
   (format st "<Class ~A>" (class-name r)))

(defstruct (datum :conc-name
		  (:print-function datum-print-procedure))
  counter              ; Unique ID for easy lookup
  jtre                 ; The JTRE it is part of
  lisp-form            ; Expression for pattern-matching
  (tms-node nil)       ; Pointer into TMS
  class                ; Class of the corresponding pattern
  (assumption? nil)    ; if non-nil, indicates informant
  (plist nil))         ; local property list

(defun datum-print-procedure (d st ignore)
  (declare (ignore ignore))
  (format st "<Datum ~D>" (datum-counter d)))

;;;; Making statements

(defun assert! (fact just &optional (*JTRE* *JTRE*) &aux datum node)
  (setq datum (referent fact t)
	node (datum-tms-node datum))
  (unless (listp just) (setq just (list just)))
  (debugging-jtre "~%    Asserting ~A via ~A." fact just)
  (justify-node (car just) node
		(mapcar #'(lambda (f) (datum-tms-node (referent f t)))
			(cdr just)))
  datum)

(defun assume! (fact reason &optional (*JTRE* *JTRE*) &aux datum node)
  (setq datum (referent fact t)
	node (datum-tms-node datum))
  (cond	((not (datum-assumption? datum))
	 (setf (datum-assumption? datum) reason)
	 (debugging-jtre "~%    Assuming ~A via ~A." fact reason)
	 (assume-node node)
	 (enable-assumption node))
	((eq reason (datum-assumption? datum)))
	(t (error "Fact ~A assumed because of ~A assumed again because of ~A"
		  (show-datum datum) (datum-assumption? datum) reason)))
  datum)

(defun already-assumed? (fact) (datum-assumption? (referent fact t)))

(defun quiet-assert! (fact just &optional (*JTRE* *JTRE*))
  (without-contradiction-check (jtre-jtms *JTRE*) (assert! fact just)))

(defmacro rassert! (fact just) `(assert! ,(quotize fact) ,(quotize just)))

;;;; Retraction

(defun retract! (fact &optional (just 'user) (quiet? nil) (*JTRE* *JTRE*) &aux datum node)
  (setq datum (referent fact t)
	node (datum-tms-node datum))
  (cond ((not (tms-node-assumption? node))
	 (unless quiet? (format t "~%~A isn't an assumption." (show-datum datum))))
	((not (in-node? node))
	 (unless quiet? (format T "~%The assumption ~A is not currently in." fact)))
	((eq just (datum-assumption? datum))
	 (debugging-jtre "~%    Retracting ~A via ~A." fact just)
	 (setf (datum-assumption? datum) nil)
	 (retract-assumption node))
	((not quiet?)
	 (format t "~%~A not source of assumption for ~A"
		 just fact)))
  node)

(defmacro rretract! (fact &optional (just 'USER))
  `(retract! ,(quotize fact) ,(quotize just)))

(defun contradiction (fact &optional (*JTRE* *JTRE*))
  (make-contradiction (datum-tms-node (referent fact t))))

;;;; Database system

(defun get-class (fact &optional (*JTRE* *JTRE*) &aux class)
  (cond ((null fact) (error "~% NIL can't be a class."))
	((listp fact) (get-class (car fact) *JTRE*))
	((variable? fact)
	 (cond ((boundp fact) (get-class (symbol-value fact) *JTRE*))
	       (t (error "~%Class unbound: ~A" fact))))
	((symbolp fact)
	 (cond ((setq class (gethash fact (jtre-class-table *JTRE*))) class)
	       (t (setq class
			(make-class :NAME fact :FACTS nil
				    :RULES nil :JTRE *JTRE*))
		  (setf (gethash fact (jtre-class-table *JTRE*)) class)
		  class)))
	(t (error "Bad class type: ~A" fact))))

(defun referent (fact &optional (virtual? nil) (*JTRE* *JTRE*))
  (if virtual? (insert fact) (referent1 fact)))

(defun referent1 (fact)
  (dolist (candidate (class-facts (get-class fact)))
	  (when (equal (datum-lisp-form candidate) fact)
		(return candidate))))

(defun insert (fact &aux datum)
  (setq datum (referent1 fact))
  (cond (datum (values datum t))
	(t (setq datum (make-datum :COUNTER (incf (jtre-datum-counter *JTRE*))
				   :JTRE *JTRE*
				   :LISP-FORM fact
				   :CLASS (get-class fact)))
	   (setf (datum-tms-node datum) (tms-create-node (jtre-jtms *JTRE*) datum))
	   (push datum (class-facts (datum-class datum)))
	   (try-rules datum)
	   (values datum nil))))

(defun fetch (pattern &optional (*JTRE* *JTRE*) &aux bindings unifiers)
  (dolist (candidate (get-candidates pattern) unifiers)
    (setq bindings (unify pattern (datum-lisp-form candidate)))
    (unless (eq bindings 'FAIL)
      (push (sublis bindings pattern) unifiers))))

(defun get-candidates (pattern) (class-facts (get-class pattern)))

(defun map-class (proc &optional (*JTRE* *JTRE*))
  (maphash #'(lambda (name class) (declare (ignore name))
	       (funcall proc class))
	   (jtre-class-table *JTRE*)))

;;;; Interface and display of data

(defun in? (fact &optional (*JTRE* *JTRE*) &aux r)
  (when (setq r (referent fact))
	(in-node? (datum-tms-node r))))

(defun out? (fact &optional (*JTRE* *JTRE*) &aux r)
  (when (setq r (referent fact))
	(out-node? (datum-tms-node r))))
    
(defun why? (fact &optional (*JTRE* *JTRE*) &aux r)
  (when (setq r (referent fact))
	(why-node (datum-tms-node r))))

(defun get-tms-node (fact &optional (*JTRE* *JTRE*))
  (datum-tms-node (referent fact t)))

(defun view-node (node) (datum-lisp-form (tms-node-datum node)))

(defun assumptions-of (fact &optional (*JTRE* *JTRE*))
  (mapcar #'view-node 
	  (assumptions-of-node
	   (datum-tms-node (referent fact *jtre* t)))))

(defun wfs (fact &optional (*JTRE* *JTRE*))
  ;; Displays well-founded support for a fact
  (cond ((out? fact) (format t "~% ~A is OUT." fact))
	(t (do ((queue (list (get-tms-node fact)) (nconc (cdr queue) new-antes))
		(so-far (list (get-tms-node fact)))
		(new-antes nil nil))
	       ((null queue) (format t "~%--------") fact)
	     (why-node (car queue)) ;; Say current thing
	     (unless (or (out-node? (car queue)) ;; Should never be true here.  
			 (tms-node-assumption? (car queue))) ;; Might be
	       ;; Go down the support
	       (dolist (ante (just-antecedents (tms-node-support (car queue))))
		 (unless (member ante so-far)
		   (push ante so-far) (push ante new-antes))))))))

;;;; More interrogatives

(defun say-datum-belief (pr &optional (*jtre* *jtre*) (indent ""))
  (format t "~%~A~A: ~A" indent pr
	  (if (in-node? (get-tms-node pr *jtre*)) "IN" "OUT")))

(defun show-justifications (fact &optional (*jtre* *jtre*))
  (format t "~% ~A::" fact)
  (let* ((node (get-tms-node fact *jtre*))
	 (justs (tms-node-justs node)))
    (unless justs
	    (format t " No justifications.")
	    (return-from show-justifications node))
    (dolist (j justs)
	    (format t "~% ~A" (just-informant j))
	    (cond ((just-antecedents j) 
		   (format t ", on:")
		   (dolist (ante (just-antecedents j))
			   (say-datum-belief (view-node ante) *jtre* "  "))
		   (format t "."))
		  (t (format t "."))))))

;;;; More query routines

(defun show-datum (datum) (format nil "~A" (datum-lisp-form datum)))

(defun show-data (&optional (*JTRE* *JTRE*) (stream *standard-output*))
  (format stream 
	  "~%~D facts total." (jtre-datum-counter *JTRE*))
  (map-class  #'(lambda (class)
		  (dolist (datum (class-facts class))
			  (format stream "~%~A: ~A" (show-datum datum)
				  (if (in-node? (datum-tms-node datum)) "IN" "OUT"))))))

(defun get-datum (num &optional (*JTRE* *JTRE*))
  (map-class #'(lambda (class)
		 (dolist (datum (class-facts class))
			 (when (= (datum-counter datum) num)
			       (return-from GET-DATUM datum))))))

(defun get-just (num &optional (*JTRE* *JTRE*))
  (dolist (just (jtms-justs (jtre-jtms *JTRE*)))
    (when (= (just-index just) num) (return-from GET-just just))))
