;; -*- Lisp -*- 

;;;; ATRE definitions and interface 

;; Copyright (c) 1986, 1987, 1988, 1989, 1990, 1991, Kenneth D. Forbus,
;;  Northwestern University, and Johan de Kleer, the Xerox Corporation
;; All rights reserved.

(in-package 'user)

(defstruct (atre :conc-name
		 (:predicate atre?)
		 (:print-function print-atre))
  title                   ; Pretty name
  atms                    ; Pointer to its ATMS
  (classes nil)           ; List of classes
  (class-table nil)       ; Quick index into classes
  (datum-counter 0)       ; Unique ID for asserts
  (rules nil)             ; Index for rules
  (rule-counter 0)        ; Unique ID for rules
  (debugging nil)         ; Show basic operations
  (queue nil)             ; General queue
  (rules-run 0)           ; Statistics
  (in-rules nil)          ; Cache of unsatisfied but :IN-triggered rules
  (focus nil)             ; State of the search, if any.
  (contradiction-rules nil) ; As in Focus paper (AAAI-88)
  (imp-rules nil))   ; Ibid.

(defun print-atre (j st ignore) (declare (ignore ignore))
  (format st "<ATRE: ~A>" (atre-title j)))

(defvar *ATRE* nil) ;; Default ATRE
;;; The binding of this symbol is used inside rules and various
;;; macros to specify which ATRE a rule or fact should be stored in.
;;; The next few procedures encapsulate this choice

(defmacro With-ATRE (atre &rest forms)
   ;; Executes <forms> within <atre>
  `(let ((*ATRE* ,atre)) ,@ forms))

(defun In-ATRE (atre) (setq *ATRE* atre)) ;; Analogy with in-package

(defmacro debugging-atre (msg &rest args)
  `(when (atre-debugging *atre*) (format t ,msg  ,@args)))

;;; Classes, datums, and  rules

(defstruct (class :conc-name 
		  (:print-function print-atre-class))
  name    ; Corresponding symbol
  atre    ; ATRE it is part of.
  facts   ; Associated facts
  rules)   ; Associated rules

(defun print-atre-class  (r st ignore) (declare (ignore ignore))
  (format st "<Class ~A>" (class-name r)))

(defstruct (datum :conc-name
		  (:print-function print-atre-datum))
  counter              ; Unique ID for easy lookup
  atre                 ; The ATRE it is part of
  lisp-form            ; Expression for pattern-matching
  (tms-node nil)       ; Pointer into TMS
  class                ; Class of the corresponding pattern
  (assumption? nil)    ; if non-nil, indicates informant
  (plist nil))         ; local property list

(defun print-atre-datum (d st ignore) (declare (ignore ignore))
  (format st "<Datum ~D>" (datum-counter d)))

(defstruct (rule :conc-name
		 (:print-function (lambda (r st ignore)
				    (declare (ignore ignore))
				    (format st "<Rule ~D>"
					    (rule-counter r)))))
  counter      ; Unique ID for easy lookup
  atre         ; The ATRE it is part of
  class        ; Class of associated pattern
  matcher      ; Procedure that performs the match
  body         ; Procedure that does the rules' work
  in-nodes     ; Must have a jointly non-empty label
  imp-nodes)   ; Must be implied by the focus for the rule to run
  
;;; Setting up ATRE

(defun create-atre (title &key debugging)
 (let ((j (make-atre
	   :TITLE title 
	   :ATMS (create-atms (list :ATMS-OF title) 
			      :node-printer 'view-node)
	   :CLASS-TABLE (make-hash-table :test #'eq)
	   :DEBUGGING debugging))
       (false nil))
   (in-atre j)
   (change-atms (atre-atms j)
		:enqueue-function
		#'(lambda (pair) (enqueue pair j)))
   ;; Create a default contradiction
   (setq false (make-datum :counter (incf (atre-datum-counter j))
			   :atre j :lisp-form 'FALSE
			   :class (get-class 'FALSE)))
   (setf (datum-tms-node false) (atms-contra-node (atre-atms j)))
   (setf (tms-node-datum (datum-tms-node false)) false)
   (push false (class-facts (datum-class false)))
   j))

(defun change-atre (atre &key (debugging nil debugging?))
  (if debugging? (setf (atre-debugging atre) debugging)))

;;;; Running ATRE

#-RT (defun read-form () (read))
#+RT (defun read-form (&aux form)
       (unwind-protect
	   (progn (editor::tty_raw)
		  (setq form (read)))
	 (editor::tty_cooked))
       form)

(defun run (&optional (atre *ATRE*)) ;; Toplevel driver function
    (format T "~%>>")
    (do ((form (read-form) (read-form)))
        ((member form '(quit stop exit abort)) nil)
        (format t "~%~A" (eval form))
        (run-rules atre)
        (format t "~%>>")))

(defun run-forms (forms &optional (atre *ATRE*))
  (dolist (form forms) (eval form) (run-rules atre)))

(defun show (&optional (atre *ATRE*) (stream *standard-output*))
  (format stream "For ATRE ~A:~% Focus = ~A."
	  (atre-title atre) (if (env? (atre-focus atre)) (atre-focus atre)
				"empty"))
  (show-data atre stream) (show-rules atre stream))

(defun solutions (atre choice-sets)
  (interpretations (atre-atms atre)
		   (mapcar #'(lambda (choice-set)
			       (mapcar #'(lambda (f) (get-tms-node f atre))
				       choice-set))
			   choice-sets)))

;;;; Implied-by rules

;; The rule expansion code sets up the necessary tests for
;; seeing if the antecedent nodes are implied by the current
;; focus when the rule is on the queue.  Here we just 
;; re-queue the implied-by rules which were not in the scope
;; of the previous focus for re-examination.

(defun change-focus (env &optional (atre *atre*))
  (unless (atre? atre) ;; Users do slip, sometimes
    (error "Must change the focus of some ATRE, not ~A." atre))
  (when (and (env? env)
	     (not (env-nogood? env)))
    (setf (atre-focus atre) env) ;; change focus
    (setf (atre-queue atre) ;; re-queue implied-by rules
	  (nconc (atre-queue atre) (atre-imp-rules atre)))
    (setf (atre-imp-rules atre) nil)
    env)) ;; return new focus to indicate switch

(defun focus-okay? (atre)
  (and (atre-focus atre)
       (not (env-nogood? (atre-focus atre)))))

(defmacro with-focus (focus atre &rest forms)
  `(let ((old-focus (atre-focus ,atre)))
     (unwind-protect (progn (change-focus ,focus ,atre)
			    ,@ forms)
       (change-focus old-focus ,atre))))

;;;; Contradiction rules

;; This is an extremely simple implementation.  A better one
;; needs more hooks into the ATMS.

(defun contradiction-rule (env proc atre)
  ;; Stores the procedure with the environment in that ATRE's
  ;; contradiction rule table.
  (let ((entry (assoc env (atre-contradiction-rules atre)
		      :test #'eq)))
    (cond ((env-nogood? env) ;; Queue the rule for execution now
	   (enqueue (list proc (list env) nil) atre))
	  (t (unless entry ;; Hold and see if it is nogood.
	       (push (setq entry (cons env nil))
		     (atre-contradiction-rules atre)))
	     (push proc (cdr entry))))))

(defun check-contradiction-rules (atre)
  (dolist (entry (atre-contradiction-rules atre))
    (when (env-nogood? (car entry))
      (dolist (proc (cdr entry))
	(enqueue (list proc (list (car entry)) nil) atre))
      (delete entry (atre-contradiction-rules atre)
	      :count 1))))

