/* Random Number Generator routines */

/* Copyright 1989 Russell G. Almond
   License is granted to copy this program for education or research
   purposes, with the restriction that no portion of this program may
   be copied without also copying this notice.  All other rights
   reserved.  */
 

/* These routines are meant to provide a rapid random number */
 /* generation system for the lisp monte carlo system.   */


#include <math.h>
#include <stdio.h>
#include "rng.h"

/* As these routines may need to interface with many types of lisp */
 /* programs, I have kept the calling protocall as simple as possible. */
 /* All values passed are either (1) an integer, (2) a */
 /* double-precision number, or (3) a character. */

/* Before the simulation begins, some setup is done for each */
 /* distribution.  First a call is made to setup_rng which initializes */
 /* the table of rng_distribution constants.  Next a call is made to */
 /* setup_distribution for each distribution to be drawn from.  This */
 /* returns a distribution number for each distribution set up. */
 /* During the simulation, the draw_distribution routine draws from */
 /* the distribution with the given distribution number. */

typedef struct distribution {
  double (*draw_function)();
  int data_size;
  double *data;
} DISTRIBUTION;


DISTRIBUTION *RNG_TABLE;

int RNG_TABLE_SIZE=0, RNG_TABLE_NEXT=0;


				/* sizes for data for various drawing */
				/* methods */
#define UNIFORM_SIZE 2
#define UNIFORM_RANGE 0
#define UNIFORM_MIN 1

#define NORMAL_SIZE 4
#define NORMAL_MEAN 0
#define NORMAL_SDV 1
#define NORMAL_FLAG 2
#define NORMAL_NEXT 3

#define LOGNORMAL_SIZE 5
#define LOGNORMAL_LOCATION 0
#define LOGNORMAL_SCALE 1
#define LOGNORMAL_SHAPE 2
#define LOGNORMAL_FLAG 3
#define LOGNORMAL_NEXT 4

#define EXPONENTIAL_SIZE 1
#define EXPONENTIAL_SCALE 0

#define BETA1_SIZE 2
#define BETA1_ALPHA 0
#define BETA1_BETA 1

#define BETA_SIZE 8
#define BETA_A0 0
#define BETA_B0 1
#define BETA_A 2
#define BETA_B 3
#define BETA_ALPHA 4
#define BETA_BETA 5
#define BETA_GAMMA 6
#define BETA_FLAG 7
#define LOG4		1.3862943611198905724535279659903608262538909912109375
#define LOG5_1		2.60943791243410050384454734739847481250762939453125

#define GAMMAI_SIZE 2
#define GAMMAI_ALPHA 0
#define GAMMAI_S 1

#define GAMMA_SIZE 25
#define GAMMA_ALPHA 0
#define GAMMA_S 1
#define GAMMA_X0 1
#define GAMMA_X1 2
#define GAMMA_X2 3
#define GAMMA_X3 4
#define GAMMA_X4 5
#define GAMMA_X5 6
#define GAMMA_F0 6
#define GAMMA_F1 7
#define GAMMA_F2 8
#define GAMMA_F3 9
#define GAMMA_F4 10
#define GAMMA_F5 11
#define GAMMA_P0 11
#define GAMMA_P1 12
#define GAMMA_P2 13
#define GAMMA_P3 14
#define GAMMA_P4 15
#define GAMMA_P5 16
#define GAMMA_P6 17
#define GAMMA_P7 18
#define GAMMA_P8 19
#define GAMMA_P9 20
#define GAMMA_P10 21
#define GAMMA_LAML 22
#define GAMMA_LAMR 23
#define GAMMA_D 24

#define BETAIN_SIZE (3+GAMMAI_SIZE+GAMMA_SIZE)
#define BETAIN_ALPHA 0
#define BETAIN_BETA 1
#define BETAIN_FLAG 2
#define BETAIN_GAMMAI_DAT 3
#define BETAIN_GAMMA_DAT (3+GAMMAI_SIZE)


				/* Tuning Constants */

#define GAMMA_METHOD_MAX	10
				/* shape parameter value after which */
				/* the more complex gamma method is used */
#define BETA_METHOD_MIN		100
				/* minimum value for other parameter */
				/* after which 2gamma method of */
				/* generating betas is used */
				/* setup_rng */

extern double draw_uniform(), draw_normal(), draw_lognormal();
extern double draw_exponential(), draw_beta1(), draw_beta();
extern double draw_gammai(), draw_gamma(), draw_betain();


/* This function takes one argument, an integer and sets up the random */
 /* number generator distribution tables for num_dist distributions */
int setup_rng(num_dist)
int num_dist;
{
  if (RNG_TABLE_SIZE !=0) cleanup_rng(); /* cleanup previous rng setup */

  RNG_TABLE_SIZE=num_dist; 
  RNG_TABLE_NEXT=0;
  if( NULL==(RNG_TABLE= (DISTRIBUTION *)calloc(num_dist,sizeof(DISTRIBUTION)))){
    fprintf(stderr, "setup_rng: ran out of C memory\n");
    return(ERROR);
  }
  
  return(num_dist);
}


				/* cleanup_rng */
/* This function frees storage allocated for RNG constants */
int cleanup_rng()
{
  register int i;
  for (i=0 ; i<RNG_TABLE_NEXT; i++)
    free(RNG_TABLE[i].data);
  free(RNG_TABLE);
  RNG_TABLE_SIZE=0;RNG_TABLE_NEXT=0;
  return(0);
}


				/* setup_distribution */
/* This takes 4 arguments.  The first is a distribution type */
 /* represented by a character, the remaning three are assumed to be */
 /* parameters of the distribution.  One or more of the parameters may */
 /* be ignored depending on the distribution type. It returns an index */
 /* into the RNG_TABLE of the address of its distribution description */
 /* block.  */
/* Note, there is not necessarily a one to one correspondence between */
 /* the distribution type and the drawing and setup function for that */
 /* method.  The setup_distribution function selects drawing methods */
 /* based on the parameters. */
/* Currently supported types and drawing methods: */
 /* U -- draw_uniform (2 parameters) */
 /* N -- draw_normal (2 parameters) */
 /* L -- draw_lognormal (3? parameters) */
 /* E -- draw_exponential (1 parameter) */
 /* B -- draw_beta1 (2 parameters) -- either parameter equals 1 */
 /* B -- draw_betain (2 parameters) -- one parameter is an interger */
 /* less than GAMMA_METHOD_MAX, the other is bigger than BETA_METHOD_MIN */
 /* B -- draw_beta (2 parameters) -- otherwise */
 /* G -- draw_gammai (2 paramters) -- first parameter is an integer */
 /* less than GAMMA_METHOD_MAX */
 /* G -- draw_gamma (2 paramters) -- otherwise */
int setup_distribution(typ, par1, par2, par3)
char typ;
double par1, par2, par3;
{
  register int this_dist;
  int errcode;			/* for errors in middle of processing */
  
  this_dist=RNG_TABLE_NEXT++;	/* allocate a distribution in the array */
  
  if( this_dist >= RNG_TABLE_SIZE ) { /* just ran out, allocate a */
				      /* bigger array */
    if (NULL==(RNG_TABLE=(DISTRIBUTION *) realloc(RNG_TABLE,
				 (RNG_TABLE_SIZE+1)*sizeof(DISTRIBUTION)))) {
      fprintf(stderr, "setup_distribution: ran out of C memory\n");
      return(ERROR);
    }
    RNG_TABLE_SIZE++;
  }

  

  switch(typ) {			/* type specific allocation and */
				/* initialization */
    
  case 'U':			/* Uniform random numbers */
  case 'u':			/* parameters are assumed to be lower */
				/* and upper bounds */
    RNG_TABLE[this_dist].draw_function = draw_uniform;
    RNG_TABLE[this_dist].data_size = UNIFORM_SIZE;
    if (NULL==(RNG_TABLE[this_dist].data= (double *) malloc(UNIFORM_SIZE*sizeof(double)))) {
      fprintf(stderr, "setup_distribution: ran out of C memory\n");
      return(ERROR);
    }
    errcode=setup_uniform(RNG_TABLE[this_dist].data,par1,par2,par3);
    break;



  case 'N':			/* Normal random numbers */
  case 'n':			/* parameters are assumed to be mean */
				/* and standard deviation */
    RNG_TABLE[this_dist].draw_function =  draw_normal;
    RNG_TABLE[this_dist].data_size = NORMAL_SIZE;
    if (NULL==(RNG_TABLE[this_dist].data=(double *) malloc(NORMAL_SIZE*sizeof(double)))) {
      fprintf(stderr, "setup_distribution: ran out of C memory\n");
      return(ERROR);
    }
    errcode=setup_normal(RNG_TABLE[this_dist].data,par1,par2,par3);
    break;


  case 'L':			/* Lognormal random numbers */
  case 'l':			/* parameters are assumed to be */
				/* location (usually zero), scale */
				/* (median), and shape */
				/* (var(log(x))*/
    RNG_TABLE[this_dist].draw_function =  draw_lognormal;
    RNG_TABLE[this_dist].data_size = LOGNORMAL_SIZE;
    if (NULL==(RNG_TABLE[this_dist].data=(double *) malloc(LOGNORMAL_SIZE*sizeof(double)))) {
      fprintf(stderr, "setup_distribution: ran out of C memory\n");
      return(ERROR);
    }
    errcode=setup_lognormal(RNG_TABLE[this_dist].data,par1,par2,par3);
    break;


  case 'E':			/* Exponential random numbers */
  case 'e':			/* parameter is assumed to be scale */
    RNG_TABLE[this_dist].draw_function =  draw_exponential;
    RNG_TABLE[this_dist].data_size = EXPONENTIAL_SIZE;
    if (NULL==(RNG_TABLE[this_dist].data=(double *)malloc(EXPONENTIAL_SIZE*sizeof(double)))) {
      fprintf(stderr, "setup_distribution: ran out of C memory\n");
      return(ERROR);
    }
    errcode=setup_exponential(RNG_TABLE[this_dist].data,par1,par2,par3);
    break;



  case 'B':			/* Beta random numbers */
  case 'b':			/* parameters are assumed to be shape1 */
				/* and shape2 */

				/* There are three strategies for */
				/* drawing betas.  If either of the */
				/* parameters is one, we can use the */
				/* inverse cdf method (beta1).  If one */
				/* is a small integer and the other is */
				/* large, then generating two gammas */
				/* will be the simplest method */
				/* (betain).  Finally, otherwise we */
				/* will use beta method from Cheng[1978] */
    if (par1 == 1 || par2 == 1) { /* inverse CDF method */
      
      RNG_TABLE[this_dist].draw_function =  draw_beta1;
      RNG_TABLE[this_dist].data_size = BETA1_SIZE;
      if (NULL==(RNG_TABLE[this_dist].data=(double *)malloc(BETA1_SIZE*sizeof(double)))) {
	fprintf(stderr, "setup_distribution: ran out of C memory\n");
	return(ERROR);
      }
      errcode=setup_beta1(RNG_TABLE[this_dist].data,par1,par2,par3);

    } else if ( (par1 <= GAMMA_METHOD_MAX && par1 == floor(par1) &&
		 par2 > BETA_METHOD_MIN) ||
	        (par2 <= GAMMA_METHOD_MAX && par2 == floor(par2) &&
		 par1 > BETA_METHOD_MIN)                            ){
				/* Generate two gammas */
      
      RNG_TABLE[this_dist].draw_function =  draw_betain;
      RNG_TABLE[this_dist].data_size = BETAIN_SIZE;
      if (NULL==(RNG_TABLE[this_dist].data=(double *)malloc(BETAIN_SIZE*sizeof(double)))) {
	fprintf(stderr, "setup_distribution: ran out of C memory\n");
	return(ERROR);
      }
      errcode=setup_betain(RNG_TABLE[this_dist].data,par1,par2,par3);

    } else {			/* Rejection method of Cheng[1978] */
      
      RNG_TABLE[this_dist].draw_function =  draw_beta;
      RNG_TABLE[this_dist].data_size = BETA_SIZE;
      if (NULL==(RNG_TABLE[this_dist].data=(double *)malloc(BETA_SIZE*sizeof(double)))) {
	fprintf(stderr, "setup_distribution: ran out of C memory\n");
	return(ERROR);
      }
      errcode=setup_beta(RNG_TABLE[this_dist].data,par1,par2,par3);

    }
    break;


  case 'G':			/* Gamma Random Variates */
  case 'g':			/* The two paramters are a shape and a */
				/* scale parameter respectively */

				/* There are two strategies for */
				/* dealing with gamma variates.  If */
				/* the shape parameter is a small */
				/* integer */
    if ( par1 < 10 && floor(par1)==par1) {
      
      RNG_TABLE[this_dist].draw_function =  draw_gammai;
      RNG_TABLE[this_dist].data_size = GAMMAI_SIZE;
      if (NULL==(RNG_TABLE[this_dist].data=(double *) malloc(GAMMAI_SIZE*sizeof(double)))) {
	fprintf(stderr, "setup_distribution: ran out of C memory\n");
	return(ERROR);
      }
      errcode=setup_gammai(RNG_TABLE[this_dist].data,par1,par2,par3);

    } else {

      RNG_TABLE[this_dist].draw_function =  draw_gamma;
      RNG_TABLE[this_dist].data_size = GAMMA_SIZE;
      if (NULL==(RNG_TABLE[this_dist].data= (double *) malloc(GAMMA_SIZE*sizeof(double)))) {
	fprintf(stderr, "setup_distribution: ran out of C memory\n");
	return(ERROR);
      }
      errcode=setup_gamma(RNG_TABLE[this_dist].data,par1,par2,par3);

    }
    break;

  default:
    fprintf(stderr,"setup_distribution: Unreckognized or unsupported distribution type\n");
    RNG_TABLE_NEXT--;
    return(PARAMETER_ERROR);

  }

  if (errcode < 0) {
    free(RNG_TABLE[this_dist].data);
    RNG_TABLE_NEXT--;
    return(errcode);
  }

  return(this_dist);


}


				/* draw_distribution.  This takes a */
				/* distribution and draws from it */
double draw_distribution(dist_num)
int dist_num;
{

				/* draws a distribution by dispatching */
				/* to the appropriate drawing */
				/* function, calling it with the data */
				/* block */
  return( (* (RNG_TABLE[dist_num].draw_function))
	  (RNG_TABLE[dist_num].data));
}





/* Random number generators.  There are two pieces to each random */
 /* number generator.  The first is a setup routine that creates any */
 /* constants needed and stores it in the data block of that */
 /* distributions entry on the table.   The second is a draw routine */
 /* which takes the data block for that distribution as its only */
 /* argument.  */


				/* Uniform */
int setup_uniform(data,par1,par2,par3)
double data[],par1,par2,par3;
{
				/* decide which value is highest */
  if (par1 >par2) {
    data[UNIFORM_MIN] = par2;
    data[UNIFORM_RANGE] = par1-par2;
  } else {
    data[UNIFORM_MIN] = par1;
    data[UNIFORM_RANGE] = par2-par1;
  }
  return(OK);
}

double draw_uniform(data)
double data[];
{
				/* draw and scale a random number */
  double out;
  out=data[UNIFORM_MIN] + data[UNIFORM_RANGE] * RNG();
  return(out);
}


				/* Normal */
				/* Box-Mueler Algorithm */

int setup_normal(data,par1,par2,par3)
double data[],par1,par2,par3;
{
  
				/* store and check mean and sdv */
  data[NORMAL_MEAN]=par1;
  if (par2 < 0) {
    fprintf(stderr,"setup_normal: Negative Standard Deviation\n");
    return(PARAMETER_ERROR);
  }
  data[NORMAL_SDV]=par2;
  data[NORMAL_FLAG]=0;	/* need to generate a number (as */
				/* opposed to having one lying around */
				/* from last iteration) */

  return(OK);
}

double draw_normal(data)
double data[];
{
  double v1,v2,r,fac,out;

  if ( data[NORMAL_FLAG] == 0) {
				/* draw 2 new normals */
    do {			/* draw 2 points on unit circle */
      v1=2.0*RNG() -1.0;
      v2=2.0*RNG() -1.0;
      r=v1*v1+v2*v2;
    } while (r > 1.0);

    fac = sqrt(-2*log(r)/r);
    out=v1*fac;
    data[NORMAL_NEXT]= v2*fac; /* second generated normal is used */
    data[NORMAL_FLAG]= 1.0;	  /* next time out */
    
  } else {
				/* use previously generated normal */
    out=data[NORMAL_NEXT];
    data[NORMAL_FLAG]=0.0;	/* next time generate */

  }

  return( data[NORMAL_MEAN] + data[NORMAL_SDV] * out);

}

				/* Lognormal */
				/* Almost Box-Mueler */
int setup_lognormal(data,par1,par2,par3)
double data[],par1,par2,par3;
{
  
				/* store and check mean and variance */
  data[LOGNORMAL_LOCATION]=par1;
  if (par2 < 0) {
    fprintf(stderr,"setup_lognormal: Negative Scale (Median)\n");
    return(PARAMETER_ERROR);
  }
  data[LOGNORMAL_SCALE]=par2;
  if (par3 < 0) {
    fprintf(stderr,"setup_lognormal: Negative Shape (Var(log(x)))\n");
    return(PARAMETER_ERROR);
  }
  data[LOGNORMAL_SHAPE]=sqrt(par3);
  data[LOGNORMAL_FLAG]=0;	/* need to generate a number (as */
				/* opposed to having one lying around */
				/* from last iteration) */

  return(OK);
}

				/* draw normals (using box-mueler) */
				/* then transform */
double draw_lognormal(data)
double data[];
{
  double v1,v2,r,fac,out;

  if ( data[LOGNORMAL_FLAG] == 0) {
				/* draw 2 new normals */
    do {			/* draw 2 points on unit circle */
      v1=2.0*RNG() -1.0;
      v2=2.0*RNG() -1.0;
      r=v1*v1+v2*v2;
    } while (r > 1.0);

    fac = sqrt(-2*log(r)/r);
    out=v1*fac;
    data[LOGNORMAL_NEXT]= v2*fac; /* second generated normal is used */
    data[LOGNORMAL_FLAG]= 1.0;	  /* next time out */
    
  } else {
				/* use previously generated normal */
    out=data[LOGNORMAL_NEXT];
    data[LOGNORMAL_FLAG]=0.0;	/* next time generate */

  }
  
  out=exp( data[LOGNORMAL_SHAPE] * out);
  out=data[LOGNORMAL_LOCATION] + data[LOGNORMAL_SCALE] * out;

  return(out);

}

				/* exponential */
				/* inverse cdf method */

int setup_exponential(data,par1,par2,par3)
double data[],par1,par2,par3;
{
  data[EXPONENTIAL_SCALE]=par1;

  return(OK);
}


double draw_exponential(data)
double data[];
{
  double out;

  out= - log(RNG()) / data[EXPONENTIAL_SCALE];

  return(out);
}

				/* beta1 */
				/* beta distribution with one */
				/* parameter equal to 1, can use */
				/* inverse cdf method */
int setup_beta1(data,par1,par2,par3)
double data[],par1,par2,par3;
{
  
  if (par1 <0 || par2 <0) {
    fprintf(stderr,"setup_beta: negative parameter\n");
    return(PARAMETER_ERROR);
  }
  data[BETA1_ALPHA]=par1;
  data[BETA1_BETA]=par2;

  return(OK);

}

double draw_beta1(data)		/* inverse cdf method */
double data[];
{

  double out;
  
  if (data[BETA1_ALPHA] == 1.0) { /* par1 = 1.0 */

    out = 1 - pow(RNG(),1/data[BETA1_BETA]);

  } else {			/* par2 = 1.0 */

    out = pow(RNG(),1/data[BETA1_ALPHA]);

  }

  return(out);

}

				/* beta unconstrained */
				/* Cheng[1978] "Generating Beta */
				/* Variates with Nonintegral Shape */
				/* Parameters" Communications ACM, */
				/* Vol. 21, No. 4, pp 317-322. */
				/* Algorithm BB */
int setup_beta(data,par1,par2,par3)
double data[],par1,par2,par3;
{
  
  if (par1 <0 || par2 <0) {
    fprintf(stderr,"setup_beta: negative parameter\n");
    return(PARAMETER_ERROR);
  }
  data[BETA_A0]=par1;
  data[BETA_B0]=par2;

  if (par1 < par2) {		/* a=min(a0,b0), b=max(a0,b0) */
    data[BETA_A]=par1;
    data[BETA_B]=par2;
    data[BETA_FLAG]=1.0;
  } else {
    data[BETA_A]=par1;
    data[BETA_B]=par1;
    data[BETA_FLAG]=0.0;
  }
  
  data[BETA_ALPHA]=par1+par2; /* alpha = a+b */
  data[BETA_BETA]=sqrt((data[BETA_ALPHA]-2)/
                          (2*par1*par2 - data[BETA_ALPHA]));
				/* beta = sqrt( (alpha-2) / (2ab-alpha) ) */
  data[BETA_GAMMA]=data[BETA_A] + 1/data[BETA_BETA];

  return(OK);

}

double draw_beta(data)		/* Algorithm BB from Cheng[1978] */
double data[];
{
  double u1, u2, v, w, z, r, t, out, s;


  do {


				/* draw two uniforms */
    u1=RNG(); u2=RNG();

				/* step 1 */
    v = data[BETA_BETA] * log(u1/(1-u1));
    w = data[BETA_A] * exp(v);
    z = u1*u1*u2;
    r = data[BETA_GAMMA] * v  -  LOG4;
    s = data[BETA_A] + r - w;

				/* step 2 */
    if ( s + LOG5_1 >= 5*z ) break; /* accept */

				/* step 3 */
    t=log(z);
    if ( s >= t) break;		/* accept */

				/* step 4 -- rejection test */
  } while ( r + data[BETA_ALPHA] * 
                log( data[BETA_ALPHA] /(data[BETA_B] +w))
	    < t );		/* reject if true */

				/* we have an accepted guess */

  if ( data[BETA_FLAG] == 1.0) { /* a=a0 */
    out=w /( data[BETA_B] + w );
  } else {			/* a=b0 */
    out=  data[BETA_B]/( data[BETA_B] + w );
  }

  return(out);

}


				/* gamma (small integer shape) */
				/* this can be formed by summing many */
				/* exponentials */

int setup_gammai(data,par1,par2,par3)
double data[],par1,par2,par3;
{
  
  if (par1 < 1.0) {
    fprintf(stderr,"setup_gamma: Nonpositive Shape Parameter\n");
    return(PARAMETER_ERROR);
  }
  
  data[GAMMAI_ALPHA]=par1;
  data[GAMMAI_S]=par2;

  return(OK);

}

double draw_gammai(data)	/* sum of exponetials method */
double data[];
{
  register int i;
  double out;
  
  out=1.0;			/* draw and multiply alpha uniforms */
  for (i=0; i< data[GAMMA_ALPHA]; i++)
    out *= RNG();
  
  out= - log (out)/ data[GAMMA_S];
  return(out);

}


				/* gamma (unconstrained) */
				/* Uses Squeeze method for Generating */
				/* Gammas.  Bruce W. Schmeiser and Ram */
				/* Lal [1980], "Squeeze Methods for */
				/* Generating Gamma Variates", JASA */
				/* Vol. 75, No. 371, pp679-682. */
				/* Algorithm G4PE */
#define FF(y)	(exp(x[3]*log(y/x[3])+x[3]-y))
				/* This definition from Schmeiser and */
				/* Lal [1980] is used to evaluate */
				/* gamma function */

int setup_gamma(data,par1,par2,par3)
double data[],par1,par2,par3;
{

  double *x, *f, *p;
  
  x=&(data[GAMMA_X0]);
  f=&(data[GAMMA_F0]);
  p=&(data[GAMMA_P0]);

  
  if (par1 < 1.0) {
    fprintf(stderr,"setup_gamma: Nonpositive Shape Parameter\n");
    return(PARAMETER_ERROR);
  }
  
  data[GAMMA_ALPHA]=par1;
  data[GAMMA_S]=par2;
  
				/* step 1 */
  x[3]=par1-1;
  data[GAMMA_D]= sqrt (x[3]);
  x[1]=x[2]=f[1]=f[2]=0.0;
  data[GAMMA_LAML]= 1.0;
    
  if (data[GAMMA_D] < x[3]) {
    x[2] = x[3] - data[GAMMA_D];
    x[1] = x[2]*(1-1/data[GAMMA_D]);
    data[GAMMA_LAML] = 1 - x[3]/x[1];
    f[1] = FF(x[1]);
    f[2] = FF(x[2]);
  }

				/* step 2 */

  x[4] = x[3]+data[GAMMA_D];
  x[5] = x[4] * (1+1/data[GAMMA_D]);
  data[GAMMA_LAMR] = 1 - x[3]/x[5];
  f[4] = FF(x[4]);
  f[5] = FF(x[5]);
  f[3] = 1.0;

  p[1] = f[2] * (x[3]-x[2]);
  p[2] = f[4] * (x[4]-x[3]) + p[1];
  p[3] = f[1] * (x[2]-x[1]) + p[2];
  p[4] = f[5] * (x[5]-x[4]) + p[3];
  p[5] = (1-f[2]) * (x[3]-x[2]) + p[4];
  p[6] = (1-f[4]) * (x[4]-x[3]) + p[5];
  p[7] = (f[2]-f[1]) * (x[2]-x[1]) + p[6];
  p[8] = (f[4]-f[5]) * (x[5]-x[4]) + p[7];
  p[9] = - f[1]/data[GAMMA_LAML] + p[8];
  p[10]= f[5]/data[GAMMA_LAMR] +p[9];

  return(OK);			/* whew! */

}

double draw_gamma(data)		/* G4PE from Schmeiser and Lal[1980] */
double data[];
{
  
  double u, w, xx, v, out, w2;
  double *x, *f, *p;
  
  x=&(data[GAMMA_X0]);
  f=&(data[GAMMA_F0]);
  p=&(data[GAMMA_P0]);

  do {
				/* step 3 */
    u=RNG()*p[10];	/* generate uniform variate (0,p10) */
    
    if ( u <= p[4]) {

      if (u <= p[1]) {
	xx = x[2] + u/f[2];
	break;			/* deliever xx */
      }
				/* step 4 */
      if ( u <= p[2]) {
	xx = x[3] + (u-p[1])/f[4];
	break;			/* deliver xx */
      }

				/* step 5 */
      if (u <= p[3]) {
	xx = x[1] + (u-p[2])/f[1];
	break;			/* deliver xx */
      }

				/* step 6 */
      xx = x[4] + (u-p[3])/f[5];
      break;			/* deliver xx */
    
    }
    
				/* step 7 */
    w=RNG();
    
    if ( u <= p[5] ) {
      xx = x[2]+(x[3]-x[2])*w;
      if ( (u-p[4])/(p[5]-p[4]) <= w) {
	break;			/* deliver xx */
      }
      v = f[2] + (u-p[4])/(x[3]-x[2]);
    }

    else if ( u <= p[6] ){	/* step 8 */
      xx = x[3] + (x[4]-x[3])*w;
      if ( (u-p[5])/(p[6]-p[5]) <= w) {
	break;			/* deliver xx */
      }
      v = f[4]+(u-p[5])/(x[4]-x[3]);
    }

    else if (u <= p[8]) {	/* step 9 */
      
      w2=RNG();
      if (w2 > w) w=w2;
      
      if (u <= p[7]) {
	xx = x[1] + (x[2]-x[1])*w;
	v = f[1] + 2*w*(u-p[6])/(x[2]-x[1]);
	if (v <= f[2]*w) break;	/* deliver xx */
      }
      
      else {			/* step 10 */
	xx = x[5] - w*(x[5]-x[4]);
	v = f[5] + 2*w*(u-p[7])/(x[5]-x[4]);
	if (v <= f[4]*w) break;	/* deliever xx */
      }
    }				/* (end step 9 if) */


    else if (u <= p[9]) {	/* step 11 */
      u=(u-p[8])/(p[9]-p[8]);
      xx = x[1] - log(u)/data[GAMMA_LAML];
      v=9999999;		/* needed in while conditions */
      if (xx <=0 ) continue;	/* redraw */
      if (w < (data[GAMMA_LAML]*(x[1]-xx)+1)/u)
	break;			/* deliver xx */
      v = w*f[1]*u;
    }

    else {			/* step 12 */
      u=(u-p[9])/(p[10]-p[9]);
      xx = x[5] - log(u)/data[GAMMA_LAMR];
      if (w < (data[GAMMA_LAMR]*(x[5]-xx)+1)/u)
	break;			/* deliver xx */
      v=w*f[5]*u;
    }

				/* step 13 */
  } while ( xx <= 0 ||
	    log (v) > x[3]*log(xx/x[3]) + x[3] - xx );
				/* continue drawing */

				/* here xx is unscaled gamma */

  out = xx /data[GAMMA_S];

  return (out);

}



				/* betain -- clever beta */
				/* according to Cheng[1980] when one */
				/* parameter is small and the other is */
				/* large, it is advantageous to */
				/* generate two gammas instead of one */
				/* beta.  This is especially true if */
				/* one of the betas paramters is a */
				/* small integer, so that the fast */
				/* gamma method can be used.  This RNG */
				/* handles that case by calling both */
				/* of the Gamma rngs. */

int setup_betain(data,par1,par2,par3)
double data[],par1,par2,par3;
{

  int out;
  
  if (par1 <0 || par2 <0) {
    fprintf(stderr,"setup_beta: negative parameter\n");
    return(PARAMETER_ERROR);
  }
  data[BETAIN_ALPHA]=par1;
  data[BETAIN_BETA]=par2;
  data[BETAIN_FLAG]= ((double) (par1 < par2));


  if (data[BETAIN_FLAG] == 1.0) {
    out = setup_gammai(&(data[BETAIN_GAMMAI_DAT]),par1,1.0,0.0);
    out |= setup_gamma(&(data[BETAIN_GAMMA_DAT]),par2,1.0,0.0);

  } else {

    out = setup_gammai(&(data[BETAIN_GAMMAI_DAT]),par2,1.0,0.0);
    out |= setup_gamma(&(data[BETAIN_GAMMA_DAT]),par1,1.0,0.0);

  }
  return(out);

}

double draw_betain(data)
double data[];
{

  double x,y, out;

  x=draw_gammai(&(data[BETAIN_GAMMAI_DAT]));
  y=draw_gamma(&(data[BETAIN_GAMMA_DAT]));

  if (data[BETAIN_FLAG] == 1.0) {
    out= x / (x+y);
  } else {
    out= y / (x+y);
  }

  return(out);

}

