;;; utils --- genericly useful lisp utilities
;;;
;;; Copyright 1988 Russell G. Almond
;;; License is granted to copy this program for education or research
;;; purposes, with the restriction that no portion of this program may
;;; be copied without also copying this notice.  All other rights
;;; reserved. 


;;; 8/5/89 -- added linear combination operator used in prob package
;;; to form mixture potentials

;;; 2/25/92 --- Version 1.2 Cleaned up documentation.

;(provide 'utils)
(in-package :utils )
;(export '(glue-names add-count make-pairs over-common-cdrs pair+
;	  iota binomial ! add+  elt-array set-elt-array
;	  as-vector as-array float-array long-float linear-comb
;	  2+ 3+ 4+ 5+ 6+ 7+ 8+ 9+ 10+ 11+ 12+ 13+
;	  fifo-list make-fifo-list fifo-list-p fifo-head fifo-last
;	  fifo-pop fifo-append))


;; purge-list -- This function takes two arguments, a list and a
;; predicate.  It returns a list whos values consist of all those
;; members of the list with non null values of the predicate.
;(defun purge-list (test- list)
;  (declare (function test- (t) t) (list list))
;  (cond ((endp list) nil)
;	((eval (test- (car list)))
;	 (cons (car list) (purge-list test- (cdr list))))
;	(t (purge-list test- (cdr list)))))


;;remove-repeats -- This function takes a list and remove from it any
;;repeated (in the equal sense) elements.



;;; Name manipulation functions

;;; glue-names -- This function makes a new symbol with the names
;;; pasted together.
(defun glue-names (n1 n2)
  (declare (type Symbol n1 n2)
	   (:returns (type Symbol n1-n2)))
  "Creates and returns a new symbol by concatenating print names of
<n1> and <n2> resulting in: <n1>-<n2>"
  (intern (concatenate 'string (symbol-name n1) "-" (symbol-name n2))))

;;; add-count -- This adds a reference count to a symbol name so that
;;; we are sure that we are getting unique names.  It adds a property
;;; :ref-count to the symbol.
(defun add-count (sym &optional (package *package*))
  (declare (type Symbol sym) (type (or Package String Symbol) package)
	   (:returns (type Symbol sym-n)))
  "Creates a new symbol in <package> by concatenating a number onto to
the end of <sym>."
  (cond ((null (get sym :ref-count))
	 (setf (get sym :ref-count) 0))
	((numberp (get sym :ref-count))
	 (incf (get sym :ref-count)))
	(t (error "add-count: symbol ~S, has refcount ~S" sym (get sym :ref-count))))
  (intern (concatenate 'string (symbol-name sym) "-" (write-to-string
						      (get sym :ref-count)))
	  package))




;;; make-pairs -- This takes a list with an even number of elements
;;; and pairs up the elements.
;;; 8/8/89 -- changed to return two values, the seond of which is the
;;; odd element out.
(defun make-pairs (list)
  (declare (type List list)
	   (:returns (values (type List pair-list)
			     (type (or nil T) last-item))))
  "Creates a new list from <list> which has as elements pairs of
values from <list>.  If <list> has an odd number of elements, the last
item is returned as a second value.

Thus (make-pairs '(a b c d e)) will return the values ((a b) (c d))
and e ."

  (cond ((endp list) '())
	((eql 1 (length list))
	 (values '() (car list)))
	(t (multiple-value-bind (p-list odd-el)
		     (make-pairs (cddr list))	
		     (values (cons (list (car list) (cadr list)) p-list)
			     odd-el)))))

	

;;; over-common-cdrs -- This takes an alist of which the car is
;;; considered to be some kind of value and the cdr is the units.  It
;;; applies the summary operation (op) to the values with the same
;;; units, and produces a reduceded alist.
(defun over-common-cdrs (op united-list)
  (declare (type (Function (&rest T) T) op)
	   (type List united-list)
	   (:returns (type List reduced-list)))
  "This takes an alist <united-list> of which the car is
considered to be some kind of value and the cdr is the units.  It
applies the summary operation <op> to the values with the same
units, and produces a reduceded alist."
  (if (endp united-list) nil
    (let ((unit (cdar united-list)))
      (cons (cons
	     (apply op
		    (remove unit united-list :key #'cdr :test-not #'equal))
	     unit)
	    (over-common-cdrs op
			      (remove unit united-list :key #'cdr
				      :test #'equal))))))


;;; pair+ --- This function adds a list of two numbers together
(defun pair+ (#1=#:x #2=#:y &rest more)
  (declare (type (List Number) #1# #2#)
	   (:returns (type (List Number) x+y)))
  "Sums two or more lists of pairs of numbers.  Thus 
(pair+ '(x1 x2) '(y1 y2)) returns ((+ x1 y1) (+ x2 y2))."
  (if more (apply #'pair+   (list (+ (car #1#) (car #2#))
			       (+ (cadr #1#) (cadr #2#)))
		  more)
    (list (+ (car #1#) (car #2#))
	  (+ (cadr #1#) (cadr #2#)))))


;; iota -- makes a list from i to j (if add+ is true, makes list from
;; i to j+)
(defun iota (i &optional (j 0) (add+ nil))
  (declare (type Fixnum i) (type Fixnum j))
  "Returns list of integers between <j> and <i>.  If <add+> is true
appends a + to the largest integer (<i>)."
  (if (< i j) (error "iota: negative argument ~S" i))
  (if add+
      (iota-aux (the fixnum (1- i)) j (list (intern (concatenate 'string
						    (write-to-string i) "+")
				       :user)) )
    (iota-aux i j nil))
  )


(defun iota-aux (i j rest)
  (declare (type Fixnum i j) (type List rest))
  (cond ((< i j) rest)
	((= i j) (cons i rest))
	(t (iota-aux (1- i) j (cons i rest)))))


;; add+ -- if  i eq lim return i+ else return i
(defun add+ (i lim)
  (declare (type Integer i lim)
	   (:returns (type Symbol i+)) )
  "If <i> equals <lim> return symbol <i>+ else return <i>."
  (if (eq i lim) (values (intern (concatenate 'string (write-to-string i) "+")))
    i))

;;; binomial (n,m) -- finds the value of (m chose n)
(defun binomial (n m)
  (declare (type Fixnum n) (type Fixnum m)
	   (:returns (type Fixnum n-choose-m)))
  "Calculates the binomial coefficient (n choose n)"
  (cond ((zerop m) 1)
	((eql m n) 1)
	((minusp m) 0)
	((> m n) 0)
	((< (the fixnum (- n m)) m) (binomial n (the fixnum (- n m))))
	(t (binomial-aux n m))))

(defun binomial-aux (n m)
  (declare (type Fixnum n m)
	   (:returns (type Fixnum)))
  (if (zerop m) 1
    (/ (* (the fixnum (1+ (the fixnum (- n m))))
	  (binomial-aux n (the fixnum (1- m)))) m)))


;;; ! -- factorial
(defun ! (n)
  (declare (type Fixnum n)
	   (:returns (type Fixnum n!)))
  "Calculates <n>!."
  (cond ((minusp n) 0)
	((zerop n) 1)
	(t (* n (! (the fixnum (1- n)))))))


;;; These functions deal with setting elements of an array using a
;;; list of the indexies rather than an including the index of the
;;; lists directly.
(declaim (inline elt-array set-elt-array))
(defun elt-array (array indexlist)
  (declare (type Array array) (type List indexlist)
	   (:returns (type T)))
  "Access element marked by <indexlist> in <array>.  Allows
<indexlist> to be treated as a single entity."
  (apply #'aref array indexlist))

(defun set-elt-array (array indexlist value)
  (declare (type Array array) (type List indexlist)
	   (type T value)
	   (:returns (type T value)))
  "Access element marked by <indexlist> in <array>.  Allows
<indexlist> to be treated as a single entity."
  (setf (apply #'aref array indexlist) value))


(defsetf elt-array set-elt-array
    "Access element marked by <indexlist> in <array>.  Allows
<indexlist> to be treated as a single entity."
    )
;(declaim (inline (setf elt-array)))


;;; Type casting functions to switch between arrays and vectors.

(defun as-vector (arr &optional (type t))
  (declare (type Array arr)
	   (:returns (type vector vec)))
  "Unwraps array <arr> into a vector <vec> of the same element type." 
  (make-array (array-total-size arr) :displaced-to arr
	      :element-type (array-element-type arr))) 
						       


(defun as-array (dims vec &optional (type t))
  (declare (type Vector vec) (type List dims)
	   (:returns (type Array arr)))
  "Wraps vector <vec> into an array <arr> of the same element type
with dimensions <dims>." 
  (make-array dims :displaced-to vec
	      :element-type (array-element-type vec))) 

;;; linear-comb -- forms the linear combination of serval arrays given
;;; a series of co-efficients.  As this thing use mapcar to do its
;;; basic operation, if the lists are of unequal length the shorter
;;; one is used.  
(defun linear-comb (array-list weight-list)
  (declare (type (List Array) array-list) (type (List Number) weight-list)
	   (:returns (type Array weighted-sum)))
  "Produces a linear cominbation of arrays in <array-list> using
numbers in <weight-list> to do its work."
  (let ((vec-list (mapcar #'as-vector array-list)))
    (as-array (array-dimensions (car array-list))
	      (coerce				;Damn DEC anyway!
	       (eval `(map 'list #'(lambda (&rest args)
				     (weight-sum ',weight-list args))
			   ,.vec-list))
	       '(vector long-float))
	      'long-float)))

;; weight-sum -- takes a list of weights and then the rest of the
;; arguments are assumed to be things to be summed with the weights
(defun weight-sum (weight-list obj-list)
  (declare (type (List Number) weight-list obj-list)
	   (:returns (type Number linear-comb)))
  "Produces a weighted sum of <obj-list> using weights in
<weight-list>." 
  (reduce #'+ (mapcar #'* weight-list obj-list)))



;;; Type casting function to create arrays of a specific type.

(defun float-array (arr)
  (declare (type (Array Number) arr)
	   (:returns (type (Array Long-float) arr)))
  "Casts each entry in array <arr> as a Long-float."
  (as-array (array-dimensions arr)
	    (coerce
	     (map '(vector long-float) #'long-float (as-vector arr))
	     '(vector long-float))
	    'long-float))

(defun long-float (num)
  (declare (type Number num)
	   (:returns (type Long-Float num)))
  "Casts number <num> a Long-Float."
  (float num 1.0L0))



;;; fifo-lists  -- 

(defstruct (fifo-list (:conc-name fifo-)
		      (:constructor make-fifo-list
				    (head &aux (tail (last head)))))
"This is a list with a pointer to both the first and
last cons cell, so that the list can be rapidly (destructively)
appended to and yet still items can be take from it in order.
Operations: fifo-list make-fifo-list fifo-list-p fifo-head fifo-tail
fifo-pop fifo-append"
  (head nil :type list)
  (tail nil :type list))		;note tail point to last cons
					;cell not last item


;; fifo-pop -- pops an item from off the list and updates the
;; pointers.
(defun fifo-pop (fifo-list)
  (declare (type Fifo-List fifo-list)
	   (:returns (type T first-element)))
  "Pops an Item off the top of <fifo-list>."
  (cond ((pop (fifo-head fifo-list)))
	(t (setf (fifo-tail fifo-list) nil) nil)))

;; fifo-append -- appends a list to a fifo-list and returns fifo-list 
(defun fifo-append (fifo-lst lst)
  (declare (type Fifo-List fifo-lst) (type List lst)
	   (:returns (type Fifo-List fifo-lst)))
  "Nconc's <lst> to the end of <fifo-lst>.  

Warning:  asumes it is safe to destructively modify <lst>.
"
  (if (fifo-tail fifo-lst)
      (setf (cdr (fifo-tail fifo-lst)) lst)
    (setf (fifo-head fifo-lst) lst))
  (setf (fifo-tail fifo-lst) (last lst))
  fifo-lst)


;;; provide when loaded
(bel-provide :utils)


