;;; structures.lisp --- structures data base, common to all routine groups.
;;;
;;; These defing the structures used in the belief function expert
;;; system program.   
;;; 
;;; Copyright 1986 Russell G. Almond
;;; License is granted to copy this program for education or research
;;; purposes, with the restriction that no portion of this program may
;;; be copied without also copying this notice.  All other rights
;;; reserved. 

;;; 7/13/88 - changed ps-sets to include **frame** as a possible ps-set

;;; 7/21/89 - Version 1.1
;;;	* Added distribution structures to support drawing from distributions.

;;; 8/2/89 -- Changed the distribution structure to conform with rng.c
;;; added lognormal 


;;; 2/22/92 -- Version 1.2  Changed documentation specificiations.

;(provide 'structures)
(in-package :basic)
;(export '(*maxvars* *maxvals*
;	   graph make-graph copy-graph graph-p graph-nodes graph-edges
;	   ps-set ps-set-p mt-ps-set mt-set **frame**
;	   ps-set-sym-p ps-set-sym-val vacuous
;	   m-value make-m-value copy-m-value m-value-p m-value-m
;	   m-value-element
;	   val make-val copy-val val-p val-frame
;	   belief-function make-belief-function copy-belief-function
;	   belief-function-p
;	   belief-function-frame belief-function-ms
;	   pot make-pot copy-pot pot-p pot-frame pot-array
;	   distribution make-distribution copy-distribution
;	   distribution-p dist-draw-function dist-setup-function
;	   dist-nominal-function 
;	   dist-rng-calls-per-draw dist-rng-data-1 dist-rng-data-2 
;	   interval make-interval copy-interval interval-p
;	   interval-upper interval-lower
;	   uniform make-uniform copy-uniform uniform-p
;	   uniform-upper uniform-lower
;	   beta-dist make-beta-dist copy-beta-dist beta-dist-p
;	   beta-events beta-trials
;	   gamma-dist make-gamma-dist copy-gamma-dist gamma-dist-p
;	   gamma-events gamma-time
;	   mixture-dist make-mixture-dist copy-mixture-dist mixture-dist-p
;	   mix-of mix-total-weight
;	   lognormal-dist make-lognormal-dist copy-lognormal-dist
;	   lognormal-dist-p lognorm-offset lognorm-median lognorm-error-factor
;	   ))


;;; initializiation constants, placed in lisp-init.lisp
(proclaim '(special *maxvars* *maxvals*))


;;; Structure graph -- this is the basic structure for a graph or a
;;; hyper graph.  A list on nodes and a list of edges.
(defstruct graph
  "Graph representation as nodes and edges."
  (nodes nil :type List)				; The list of nodes
  (edges nil :type List)				; The list of edges--lists of nodes
  )

;;; ps-set -- this is the structure that represents a subset of the
;;; power set over the product space of several variables.  It consists
;;; of a list of arrays each of the size of the number of variables.
;;; The nth element of the array is either a value, or a set of values
;;; for the nth variable.  It represents the union of all such
;;; sets. 
;;; 12/15/86 -- changed from list of arrays to list of lists.  Removed
;;; the structure coating, so that hash tables would work with equal
;;; 7/13/88 -- added the possibility of having the value **frame** as
;;; a ps-set.  **frame** is the frame, and it has **frame** as its
;;; value for convenience.  ps-set is now created as an explicit type
;;; (satisfies ps-set-p)
;(defstruct ps-set
;  (union nil :type list))
(defconstant mt-ps-set '() "Empty PS (TS) set")	;the empty set
(defconstant **frame** '**frame** "The universe.") ; the universe
(defun ps-set-p (#1=#:x)
  (declare (type T #1#)
	   (:returns (type (member T NIL) typecheck)))
  (or (listp #1#) (eq #1# **frame**)))
(deftype ps-set () "Tuple of Sets set" '(satisfies ps-set-p))
;(deftype ps-set () "Tuple of Sets set" (or List (eql **frame**)))


;;; ps-set-sym -- this is a symbol containing ps-set as its value.  It
;;; is used for adding a layer of indirection to the ps-set process.
;;; ps-set-sym's can be tested for eq, rather than an expensive test
;;; for equal that is needed for full ps-sets.  By convention, the
;;; ps-set is put in normal form (list of arrays of values, sorted),
;;; and a hashed reference to that symbol is saved.
(proclaim '(inline ps-set-sym-val))
(defun ps-set-sym-val (symbol)
  (declare (type Symbol symbol)
	   (:returns (type PS-set) symbol-value))
  "Fetches value of named PS-set"
  (symbol-value symbol))
(defun ps-set-sym-p (object)
  (declare (type T object)
	   (:returns (type (member T NIL))))
  "Tests to see if object is a name for a PS-set."
  (if (symbolp object)
      (if (boundp object)
	  (listp (eval object))
	nil)
    nil))
(defconstant mt-set mt-ps-set "Name for mt-ps-set.")

;;; ps-set-hash-table -- this is a hash table for referencing the
;;; ps-set-sym's from the ps-sets.
(defparameter *ps-set-hash-table*
    (make-hash-table :test #'equal :size (* *maxvars*
					    (expt *maxvals* *maxvars*)))
  "Hash table for PS-set name lookup.")
(setf (gethash mt-ps-set *ps-set-hash-table*) 'mt-set)
(setf (gethash **frame** *ps-set-hash-table*) '**frame**)



;;; Structure m-value -- this is an element in the basic probability assignment
;;; representation of a belief function.
(defstruct m-value
  "Belief function Mass/TS-set pair.  
   elment should be a TS(PS) set name."
  (m 0.0 :type Long-float)
  (element mt-set :type Symbol))

;;; vacuous -- ms-list for a vacuous belief function
(defconstant vacuous (list (make-m-value :element **frame** :m 1.0))
  "Focal element of a vacuous belief function.")

;;; val -- This structure could be either a belief function or a
;;; potential, it is defined by the common concept of a frame
(defstruct val
  "Any value defined over a frame of discernment.  In particular, a
belief function or a probability potential."
  (frame nil :type List))




;;; belief-function -- this is the basic probabilty assignment
;;; representation of the belief function.
(defstruct (belief-function (:include val))
  "A belief function represented as a list of focal elements."
  (ms vacuous :type list))


;;; pot -- This is the potential representation of the probability
;;; distribution 
(defstruct (pot (:include val))
  "A (conditional) probability distribution represented as a potential."
  (array (make-array 1 :initial-element 1.0L0 :element-type 'long-float)
	 :type (array long-float)))


			     
;;;; Structures having to do with monte carlo drawing.


;;; distribution -- the basic distribution type has define a
;;; draw-function which takes itself as an argument and returns a list
;;; of parameters (usually and interval for belief functions and a
;;; single value form probabilities).
;;; It also has a setup function which sets and saves various data for
;;; generating random numbers using the rng library.  The setup
;;; function is again called with the structure as an argument and
;;; modifies the structure by setting the parameters rng-data-1 and
;;; rng-data-2 (if needed) to integers giving their offset into a data
;;; table stored in the C space allocated by rng.c.  The slot
;;; rng-calls-per-draw gives the number of time rng must be called to
;;; draw a point or an interval from that distribution (usually 1 for
;;; prob mode and 2 for belief mode) and consequently is given to
;;; setup-rng to tell it how much space to allocate for random number
;;; generation.  
(defstruct (distribution (:conc-name dist-))
  "Probability distribution for a parameter."
  (draw-function #'will:draw-default
		 :type (function (distribution) list))
  (setup-function #'will:setup-default
		  :type (function (distribution) list))
  (nominal-function #'will:nominal-default
		  :type (function (distribution) list))
  (rng-calls-per-draw #!*rng-calls-per-draw* :type integer)
  (rng-data-1 -1 :type integer)
  (rng-data-2 -1 :type integer))


;;; interval -- the basic distribution type has two bounds in belief
;;; mode and a single point value (both elements) in prob mode.
(defstruct (interval (:include distribution (draw-function #'will:draw-interval)
			                    (setup-function #'will:setup-interval)
					    (nominal-function #'will:nominal-interval)
					    (rng-calls-per-draw 0))
		     (:constructor do-make-interval))
  "Fixed interval distribution."
  (upper 1.0 :type long-float)
  (lower 0.0 :type long-float))
(proclaim '(inline make-interval))
(defun make-interval (&key (upper 1.0) (lower 0.0))
  (declare (type Number upper lower)
	   (:returns (type Interval distribution)))
  "Fixed-Interval Construction Function."
  (do-make-interval :upper (float upper) :lower (float lower)))


;;; uniform -- for belief function draws a random interval composed of
;;; two uniforms, for prob draws a single uniform
(defstruct (uniform (:include distribution  (draw-function #'will:draw-uniform)
			                    (setup-function #'will:setup-uniform)
					    (nominal-function #'will:nominal-uniform))
		    (:constructor do-make-uniform))
  "Random interval from two uniforms or random point from one uniform."
  (upper 1.0 :type long-float)
  (lower 0.0 :type long-float))
(proclaim '(inline make-uniform))
(defun make-uniform (&key (upper 1.0) (lower 0.0))
  (declare (type Number upper lower)
	   (:returns (type Uniform distribution)))
  "Uniform distribution structure"
  (do-make-uniform :upper (float upper) :lower (float lower)))


;;; beta-dist -- assumes a beta distribution with so many occurances in so
;;; many trials.  For belief functions it draws a bivariate beta.  For
;;; probs it assumes the prior is added in.
(defstruct (beta-dist (:include distribution (draw-function #'will:draw-beta)
				             (setup-function #'will:setup-beta)
					     (nominal-function #'will:nominal-beta))
		      (:conc-name beta-)
		      (:constructor do-make-beta))
  "Beta distribution or Dempster double-betas."
  (events 0.5 :type float)
  (trials 1.0 :type float))

;; for some reason (I don't know whether its a "feature" of lucid
;; lisp) or a general lisp feature, arguments of type integer are not
;; coerced into floats by do-make-beta.  Therefore we have this
;; wonderful kludge.
(proclaim '(inline make-beta))
(defun make-beta-dist (&key (events 0.5) (trials 1.0))
  (declare (type Number events trials)
	   (:returns (type Beta-Dist distributions)))
  "Beta distribution constructor."
  (do-make-beta :events (float events) :trials (float trials)))


;;; gamma-dist -- assumes a gamma distribution with so many occurances in so
;;; much observation time.  For belief functions it draws a bivariate gamma.  For
;;; probs it assumes the prior is added in.
(defstruct (gamma-dist (:include distribution (draw-function #'will:draw-gamma)
				              (setup-function #'will:setup-gamma)
					      (nominal-function #'will:nominal-gamma))
		      (:conc-name gamma-)
		      (:constructor do-make-gamma))
  "Gamma distribution or Almond double gamma."
  (events 0.5 :type float)
  (time  1.0 :type float))

;; same damn problem
(proclaim '(inline make-gamma))
(defun make-gamma-dist (&key (events 0.5) (time 1.0))
  (declare (type Number events time)
	   (:returns (type Gamma-Dist distributions)))
  "Beta distribution constructor."
  (do-make-gamma :events (float events) :time (float time)))

;;; mixture-dist -- assumes that the distribution is a mixture of
;;; other distribution types.  Takes a single mixture list which is a
;;; alist with elements of the form (weight . distribution) and an optional total
;;; weight which defaults to 1.0
(defstruct (mixture-dist (:include distribution (draw-function #'will:draw-mix)
				               (setup-function #'will:setup-mix)
					       (nominal-function #'will:nominal-mix))
			 (:conc-name mix-)
			 (:constructor do-make-mix))
  "Finite mixture of other distributions."
  (of nil :type list)
  (total-weight 1.0 :type float))

;; make-mixture-dist -- sets the rng-calls-per-draw field properly for
;; mixture distributions.
(defun make-mixture-dist (&rest args &aux dist)
  (declare (type Mixture-Dist dist) (type List args)
	   (:returns (type Mixture-Dist dist)))
  "Mixture distribution constructor.  
<args> should be a list of cons's (weight . dist)"
  (setq dist (apply #'do-make-mix args))
  (setf (dist-rng-calls-per-draw dist)
	(reduce #'+ (mapcar #'(lambda (pair)
				(dist-rng-calls-per-draw (cdr pair)))
			    (mix-of dist))))
  (setf (mix-total-weight dist) (float (mix-total-weight dist)))
  dist)


;;; lognormal-dist -- lognormal distribution  draws a single point
;;; from the given lognormal in prob mode and is unsupported in
;;; belief-mode.
(defstruct (lognormal-dist (:include distribution (draw-function #'will:draw-lognorm)
				               (setup-function #'will:setup-lognorm)
					       (nominal-function #'will:nominal-lognorm))
			   (:conc-name lognorm-)
			   (:constructor do-make-lognorm))
  "Lognormal distribution:  Belief mode is handled by a fixed offset <lower-to-upper>."
  (offset 0.0 :type float)
  (median 1.0e-4 :type float)
  (error-factor 3.0 :type float)
  (lower-to-upper 1.0 :type float))

;; same damn problem as in beta and gamma case.
(proclaim '(inline make-lognormal-dist))
(defun make-lognormal-dist (&key (offset 0.0) (median 1.0e-4)
				 (error-factor 3.0) (lower-to-upper 1.0))
  (declare (type Number offset median error-factor)
	   (:returns (type Lognormal-Dist dist)))
  "Lognormal distribution constructor."
  (do-make-lognorm :offset (float offset) :median (float median)
		   :error-factor (float error-factor)
		   :lower-to-upper (float lower-to-upper)))


;;; provide when loaded
(bel-provide :structures)
