;;; sets.lisp -- this file implements simple set functions needed for
;;; belief function arithmatic 

;;; Copyright 1986 Russell G. Almond
;;; License is granted to copy this program for education or research
;;; purposes, with the restriction that no portion of this program may
;;; be copied without also copying this notice.  All other rights
;;; reserved. 

;;; 12/15/86 moved from list of arrays to list of lists for ps-sets,
;;; because of problems with hash-tables in equalp.  Also removed
;;; ps-set structure "coating" from ps-set for speed.

;;; 12/1/87 added ps-set-complement for conditional embedding

;;; 7/13/88 redefined ps-set to be either a list of lists (as
;;; previously) or the symbol **frame** which represents the entire
;;; frame.  

;;; 2/16/88 added index capability which goes from ps sets to lists of
;;; indexies into arrays.

;;; 2/22/92 Version 1.2 Documentation cleanup.

;(provide 'sets)
(in-package :sets)
(bel-require :structures "structures")
;(use-package 'basic)
;(export '(ps-set-equal ps-set-subsetp ps-set-member ps-set-complement
;		 explode-ps-set sort-ps-set normalize-ps-set
;		 get-ps-set-sym fget-ps-set-sym new-ps-set-symbol
;		 ps-index frame-size ps-sym-intersect))
;(eval-when (compile load eval) (import 'basic::*ps-set-hash-table*))


;;;; ps-set functions

;;; ps-car -- takes the car of a ps-set
;(defmacro ps-car (#1=#:x)
;  (declare (type ps-set #1#))
;  `(the array (car (ps-set-union ,#1#))))

;;; ps-cdr -- takes the cdr of a ps-set
;(defmacro ps-cdr (#1=#:x)
;  (declare (type ps-set #1#))
;  `(make-ps-set :union (cdr (ps-set-union ,#1#))))


;;; ps-set-equal -- returns t if the two ps-sets are equal and nil
;;; otherwise.
(defun ps-set-equal (set-a set-b)
  (declare (type Ps-Set set-a) (type Ps-Set set-b)
	   (:returns (type (member T Nil))))
  "Tests two PS (TS)-sets for equality."
  (and (ps-set-subsetp set-a set-b) (ps-set-subsetp set-b set-a)))

;;; ps-set-subsetp -- returns t if ps-set-a is a subset of ps-set-b
;;; and nil otherwise.
(defun ps-set-subsetp (set-a set-b)
  (declare (type Ps-Set set-a) (type Ps-Set set-b)
	   (:returns (type (member T Nil))))
  "Returns t if <ps-set-a> is a subset of <ps-set-b> and nil
otherwise." 
  (cond ((equalp mt-set set-a) t)
	((eq **frame** set-b) t)
	((eq **frame** set-a) nil)
	((every #'(lambda (el) (ps-set-member el set-b))
	       (explode-ps-element (car set-a)))
	 (ps-set-subsetp (cdr set-a) set-b))
	(t nil)))

;;; ps-set-complement -- complements a ps-set with respect to a given
;;; frame
(defun ps-set-complement (set-a frame)
  (declare (type Ps-Set set-a) (type List frame)
	   (:returns (type Ps-Set frame-set-a)))
  "Complements PS(TS)-set <set-a> with respect to <frame>."
  (cond  ((eq set-a **frame**) mt-set)
	 ((eq set-a mt-set) **frame**)
	 (t (set-difference (explode-ps-set frame) (explode-ps-set set-a)
		    :test #'equal))))

  
;;; ps-set-member -- this takes a single element of the product space
;;; and determines if it is a member of the ps-set.  It returns a
;;; sub-ps-set of which it is a member if so, and nil otherwise.
(defun ps-set-member (el set)
  (declare (type List el) (type PS-Set set)
	   (:returns (type (or T Nil))))
  "Checks a single tuple <el> to see if it is a member of the
PS(TS)-set <set>."
  (cond ((equalp mt-set set) nil)
	((eq set **frame**) t)
	((ps-el-member el (car set)) set)
	(t (ps-set-member el (cdr set)))))

;;; ps-el-member -- this function returns t if el is a member of
;;; set-item and nil if not or if the two lists are of differnt sizes.
;;; el is a member of set-item if each of its items matches or is a list
(defun ps-el-member (el set-item)
  (declare (type List el) (type List set-item)
	   (:returns (type (or T Nil))))
  "Checks to see if tuple <el> is contained within set-tuple
<set-item>."
  (cond ((not (eql (length el) (length set-item))) nil)
	((every #'ps-var-match el set-item) t)
	(t nil)))

;;; ps-var-match -- matches two variables.  Note, the second variable
;;; could be a list, in which case the first variable matches if it is
;;; a member of the second.
(defun ps-var-match (el-var set-item-var)
  (declare (type T el-var set-item-var))
  "<set-item-var> is either an item or a list of items.  Function
returns non-nil if either <el-var> is equal to <set-item-var> or a
member of <set-item-var>."
  (cond ((equalp el-var set-item-var) t)
	((listp set-item-var) (member el-var set-item-var))
	(t nil)))

;;; explode-ps-set -- this takes a ps-set and explodes it to a longer
;;; representation that does not use nested sets.
;;; **frame** is left alone
(defun explode-ps-set (inset)
  (declare (type PS-Set inset)
	   (:returns (type PS-set exploded-set)))
  "This takes a PS(TS)-set (inset) and explodes it into a longer list
of simple tuples (no repeated set elements)."
  (if (eq **frame** inset) **frame**
    (mapcan #'explode-ps-element inset)))



;;; explode-ps-element -- this takes a single entry in a ps-set and
;;; explodes it to be a list of lists who do not have lists as their
;;; elements.
(defun explode-ps-element (el)
  (declare (type List el) (:returns (type PS-set )))
  "Turns a single Tuple of sets <el> into a set of tuples."
  (do ((i (1- (length el)) (1- i))
       (out-list (list el)))
      ((minusp i) out-list)
      (declare (fixnum i))
      (setq out-list (mapcan #'(lambda (elm) (explode-ps-n elm i))
			     out-list))))

;;; explode-ps-n -- explodes the nth element of a ps-list
(defun explode-ps-n (elm n)
  (declare (list elm) (integer n)
	   (:returns (type (PS-set))))
  "Expands the <n>th tuple value of tuple of sets <elm>."
  (if (atom (elt elm n))
      (list elm)
      (mapcar #'(lambda (val &aux arr)
		  (setq arr (copy-seq elm))
		  (setf (elt arr n) val)
		  arr)
	      (elt elm n))))


;;; sort-ps-set -- this sorts the items in a ps-set into normal order.
;;; **frame** is once again left untouched
(declaim (inline sort-ps-set))
(defun sort-ps-set (in-set)
  (declare (type PS-Set in-set)
	   (:returns (type PS-Set out-set)))
  "Sorts a PS (TS) set into canonical order using #'ps-set>"
  (if (eq in-set **frame**) **frame**
    (sort in-set #'ps-set>)))


;;; ps-set> -- compares two elements of a ps-set and returns t if one is
;;; greater than two
(defun ps-set>> (el-one el-two)
  (declare (type List el-one el-two)
	   (:returns (type (or T Nil))))
  "Compares two tuples <el-one> and <el-two> and returns non-nil if
they follow canonical order (alphabetical by arg's name-string)." 
  (cond ((not (eql (length el-one) (length el-two)))
	 (error "ps-set>: incompatatble elements ~S and ~S" el-one el-two))
	(t (do* ((i 0 (1+ i))
		(el-1-val (princ-to-string (elt el-one i)) (princ-to-string (elt el-one i)))
		(el-2-val (princ-to-string (elt el-two i)) (princ-to-string (elt el-two i))))
	       (nil)
	       (declare (fixnum i))
	       (cond ((string> el-1-val el-2-val) (return t))
		     ((string< el-1-val el-2-val) (return nil))
		     ((eql (the fixnum (1+ i)) (length el-one)) (return t))
		     )))))

(defun ps-set> (el-one el-two)
    (declare (type List el-one el-two)
	     (:returns (type (or T Nil))))
  "Compares two tuples <el-one> and <el-two> and returns non-nil if
they follow canonical order (alphabetical by arg's name-string)." 
    (if (not (eql (length el-one) (length el-two)))
	(error "bel:ps-set>: incompatable elements ~S and ~S" el-one el-two)
      (string>= (write-to-string el-one  :level nil :length nil)
	       (write-to-string el-two  :level nil :length nil))))


;;; normalize-ps-set -- puts a ps-set into normal (exploded,sorted)
;;; form.
(defun normalize-ps-set (in-set)
  (declare (type PS-Set in-set)
	   (:returns (type PS-Set out-set)))
  "Takes PS(TS)-set <in-set> and returns PS-set in normal form."
  (if (eq in-set **frame**) **frame**
    (remove-duplicates (sort  (mapcan #'explode-ps-element in-set) #'ps-set>)
		       :test #'equalp)))


;;;; hash table look-up functions

;;; get-ps-set-sym -- this function returns the ps-set-sym associated
;;; with in-set.  If the set is not in the hash table, a new entry is
;;; created for it.
(declaim (inline get-ps-set-sym))
(defun get-ps-set-sym (in-set)
  (declare (type (PS-Set in-set))
	   (:returns (type Symbol ps-set-sym)))
  "This (macro) function tries to look up the name of <ps-set>.  If no name
exists, a new name is created."
  (fget-ps-set-sym (normalize-ps-set in-set)))

;;; fget-ps-set-sym -- this function is a fast version of
;;; get-ps-set-sym.  It assumes that in-set is in normal form.  It
;;; returns the symbol associated with in-set.  If there is no
;;; associated symbol, then it makes a new symbol.
(defun fget-ps-set-sym (in-set)
  (declare (type (PS-Set in-set))
	   (:returns (type Symbol ps-set-sym)))
  "This (macro) function tries to look up the name of <ps-set>.  If no name
exists, a new name is created.  For speed, assumes <ps-set> is in
normalized form."
  (multiple-value-bind (symbol found)
		       (gethash in-set *ps-set-hash-table*)
		       (if found symbol (new-ps-set-symbol in-set))))


;;; new-ps-set-symbol -- this creates a new ps-set symbol with the
;;; value in-set and sets the hash function to add in-set to the hashed
;;; list. (note: generaltes an error if called with **frame**)
(defun new-ps-set-symbol (in-set &aux (symbol (gentemp "focal")))
  (declare (type PS-Set in-set)
	   (:returns (type Symbol name)))
  "Creates a new name corresponding to PS(TS)-set <in-set> and sets
value and hash table entry accordingly."
  (if (eq in-set **frame**) (error "sets: redifining **frame**"))
  (set symbol in-set)
  (setf (gethash in-set *ps-set-hash-table*) symbol))


;;; ps-sym-intersect -- this takes the intersection of the ps-sets
;;; pointed at by two ps-syms and returns a sym for it. (We are
;;; relying on the fact that ps-sets are intered in normal form).
(defun ps-sym-intersect (sym1 sym2)
  (declare (type Symbol sym1 sym2))
  "This takes two PS-Set-names <sym1> and <sym2> and returns a PS-Set
name for the intersection.  Uses a rapid calculation for the
intersection by assuming all PS-Sets are in normal form."
  (cond ((eq sym1 **frame**) sym2)
	((eq sym2 **frame**) sym1)
	(t (let ((set2 (ps-set-sym-val sym2)))
	     (fget-ps-set-sym
	      (remove-if-not #'(lambda (#1=#:x) (member #1# set2 :test #'equal))
			     (ps-set-sym-val sym1)))))))



;;; Index functions  --- These functions take a frame and a ps-set and
;;; return a list of indexies into an array which reference those
;;; elements
(defun ps-index (ps-set frame)
  (declare (type List ps-set) (type List frame)
	   (:returns (type List index-list)))
  "Turns a PS(TS)-set <ps-set> defined over <frame> into a list of
indexes for accessing the corresponding elements in a potential array."
  (cond ((eq mt-set ps-set) nil)
	((eq **frame** ps-set) (ps-index (list frame) frame))
	(t (mapcan #'(lambda (el) (declare (list el))
		       (do ((i (1- (length el)) (1- i))
			    (out-list (list el)))
			   ((minusp i) out-list)
			   (declare (fixnum i))
			   (setq out-list
				 (mapcan #'(lambda (elm)
					     (nexplode-ps-n elm i
							    (elt frame i)))
					 out-list))))
		   ps-set))))

;; nexplode-ps-n--elm n turns the nth element of a ps-list with for
;; framen
;; Thanks Tippit!
(defun nexplode-ps-n (elm n framen)
  (declare (type List elm) (type Fixnum n) (type List framen))
  "Auxilary Function for ps-index."
  (mapcar #'(lambda (val &aux (cop (copy-seq elm)))
	      (declare (list cop))
	      (setf (elt cop n) (position val framen))
	      cop)
	  (if (atom (elt elm n)) (list (elt elm n))
	    (elt elm n))))


;; frame-size -- returns the dimension vector which corresponds to
;; frame (frame is expressed as list of lists)
(declaim (inline frame-size))
(defun frame-size (frame)
  (declare (type List frame)
	   (:returns (type Fixnum length)))
  "Calculates the number of elements in frame <frame>, where <frame>
is a list of outcome spaces."
  (mapcar #'length frame))




;;; provide when loaded
(bel-provide :sets)