;;; readnuke.lisp -- belief function read macros for failure
;;; simulation. 

;;; Copyright 1989 Russell G. Almond
;;; License is granted to copy this program for education or research
;;; purposes, with the restriction that no portion of this program may
;;; be copied without also copying this notice.  All other rights
;;; reserved. 

;;; 8/5/89 -- created file
;;;	* Parent and Child input Macros[w]
;;;	* Multi-event poissons[]


;;; 2/27/92 --- Version 1.2  Cleanup documentation.



;(provide 'readnuke)
(in-package :subshell)
(bel-require :utils "utils")
;(use-package '(utils will basic graphs))
;(export '(defparent defchild repeatdef repeatdef2 repeatdef3
;	   repeatdef4 n-syms
;	  defandpoi deforpoi defreducepoi defval defsink
;	  defandbin deforbin))
;(import 'commonb::get-number)
;(import 'sets::logical-unknown)





;;;; parent and children node definitions.  The defchild and defparent
;;;; macros create parents (nodes with a distribution) and children of
;;;; the same type.  If the dummy attributes are not currently defined
;;;; as attributes, they will be by the defchild and defparent macros.

;;; defparent -- defines parent nodes.
;;; Syntax:
;;;    (defparent name (dummy) &key :model-type model-type
;;;				    :distribution dist
;;;				    :default-rate num :max-failures num
;;;				    :nograph t :sink nil :doc "")
;;; the :model-type keyword is required and is arg must be one of :binomial
;;; or :poisson.  The :distribution keyword is also required and its
;;; value must either be a distribution.
;;; :default-rate and :max-failures are ignored unless the model-type
;;; is :poisson.  Their default values are #!*default-rate-default*
;;; and #!*default-max-failures*.  The :nograph keyword defaults to t
;;; and the :sink  keyword defaults to nil.  There function is similar
;;; to that shown in defbel.  name and dummy must be symbols and are
;;; not evaluated.  name becomes the name of the parent type and dummy
;;; becomes the single attribute which comprises the frame for any
;;; values constructed by this parent.  name and dummy may be the
;;; same.  If dummy does not exist as an attribute, the expansion will
;;; create it using defatt.
;;; the following properties of name are initialized:
;;; :model-type
;;; :distribution -- to the actual distribution produced
;;; :default-rate and :max-failures if poisson
;;; :children -- `()
;;; :values -- to values of attribute
;;; :frame -- `(dummy)
;;; :paramters-drawn -- nil
;;; :nominal-val -- to the value produced by running (impute) on the
;;; distribution 
;;; and the value is set to the nominal-val
;;; name is added to *parent-list* and if nograph is nil graphical
;;; manipulations are performed.  If sink is t then name is added to
;;; sink-list
(defmacro defparent (name (dummy) &key (model-type nil)
			  ((:distribution dist) nil)
			  (default-rate #!*default-rate-default*)
			  (max-failures #!*default-max-failures*)
			  (nograph t) (sink nil) (doc "")
			  &aux (form `()) otherpair)
  (declare (type Symbol name) (type Symbol dummy)
	   (type (or List (member :poisson :binomial)) model-type)
	   (type (or Nil distribution) dist)
	   (type Number default-rate max-failures)
	   (type (member T Nil) nograph)
	   (type (member T Nil) sink)
	   (type String doc)
	   (:returns (type Symbol name)))
  "defparent -- defines parent nodes.
Syntax:
(defparent name (dummy) &key :model-type model-type
			    :distribution dist
			    :default-rate num :max-failures num
			    :nograph t :sink nil :doc "")
the :model-type keyword is required and is arg must be one of :binomial
or :poisson.  The :distribution keyword is also required and its
value must either be a distribution.
:default-rate and :max-failures are ignored unless the model-type
is :poisson.  Their default values are #!*default-rate-default*
and #!*default-max-failures*.  The :nograph keyword defaults to t
and the :sink  keyword defaults to nil.  There function is similar
to that shown in defbel.  name and dummy must be symbols and are
not evaluated.  name becomes the name of the parent type and dummy
becomes the single attribute which comprises the frame for any
values constructed by this parent.  name and dummy may be the
same.  If dummy does not exist as an attribute, the expansion will
"
  (unless (symbolp name)
	  (error "defparent: parent node ~S must be symbol" name))
  (unless (symbolp dummy)
	  (error "defparent: dummy argument of ~S (~S) must be symbol"
		 name dummy))
  (unless (member model-type `(:binomial :poisson))
	  (error "defparent: ~S: illegal or unsupplied :model-type ~S"
		 name model-type))
  (unless (stringp doc)
	  (error "defparent: ~S: illeagal documentation ~S" name doc))
  (unless (distribution-p dist)
	  (unless (distribution-p (setq dist (eval dist)))
		  (error "defparent: ~S: missing or unsupplied distribution"
			 name)))
  (if (get dummy :values)
      (when sink (eval `(defsink ,dummy)))
    (eval `(defatt ,dummy
	     ,.(case model-type (:binomial (list :failure))
		     (:poisson (list :poisson max-failures)))
	     ,doc :nograph ,nograph :sink ,sink)))
					;need to do defatt first
  (push `(setf ,name (get ',name :nominal-val)) form)
  (push `(setf (get ',name :nominal-val)
	       (multiple-value-call #'impute
	          (get ',name :frame) (nominal (get ',name :distribution))
		  :model-type ,model-type
		  :rate ,default-rate :max-events ,max-failures))
	form)
  (push `(setf (get ',name :distribution) ',dist) form)
  (push `(set #?*parent-list* (cons ',name #!*parent-list*)) form)
  (push `(setf (get ',name :parameters-drawn) nil) form)
  (push `(setf (get ',name :frame) '(,dummy)) form)
  (push `(setf (get ',name :values) (get ',dummy :values)) form)
  (push `(setf (get ',name :children) nil) form)
  (push `(setf (get ',name :model-type) ,model-type) form)
  (when (eq :poisson model-type)
	(push `(setf (get ',name :default-rate) ,default-rate) form)
	(push `(setf (get ',name :max-failures) ,max-failures) form))
  (unless nograph
	  (push `(pushnew (quote (,dummy)) (graph-edges ,#?*model-graph*)
			  :test #'equal) form)
	  (if (setq otherpair (assoc (list dummy) #!*val-list* :test #'equal))
	      (warn "defparent:This edge ~S (~S) shadows edge ~S" name
		    (list dummy) (cdr otherpair)))
	  (push `(set #?*val-list* (acons ',name '(,dummy) #!*val-list*)) form))
  (push `(setf (documentation ',name 'function) ,doc) form)
  `(prog1 ,.form))


    
    


;;; defchild -- defines a child node of a given parent.
;;; The syntax for defchild is a follows:
;;;    (defchild child parent  (add) &key :model-type model-type
;;;				      :rate num :max-failures num
;;;				      :nograph nil :sink nil :doc "")
;;; here :model-type is no longer required but if it is supplied, then
;;; its value must agree with that of the parent node.  :rate replaces
;;; default rate.  The default for :nograph is now nil.  Child node
;;; properties are initilized to the following:
;;; :model-type -- (get parent :model-type)
;;; :rate and :max-failures to values supplied
;;; :parent 
;;; :values -- to values of attribute
;;; :frame -- `(att)
;;; :nominal-val -- is copied from parent or re-imputed if the
;;; paremeters are different
;;; name is added to *child-list* and to (get parent :children)
;;; if nograph is nil graphical
;;; manipulations are performed.  If sink is t then name is added to
;;; sink-list if the attribute is being auto-defined
(defmacro defchild (child parent (dummy) &key (model-type (get parent :model-type))
			  (rate (get parent :default-rate))
			  (max-failures (get parent :max-failures))
			  (nograph nil) (sink nil) (doc "")
			  &aux (form `()) otherpair)
  (declare (symbol child) (symbol parent) (symbol dummy)
	   (symbol model-type) (fixnum max-failures) (number rate))
  (unless (symbolp child)
	  (error "defchild: child node ~S must be symbol" child))
  (unless (symbolp parent)
	  (error "defchild: parent node ~S must be symbol" parent))
  (unless (symbolp dummy)
	  (error "defchild: dummy argument of ~S (~S) must be symbol"
		 child dummy))
  (unless (eq model-type (get parent :model-type))
	  (error "defchild: ~S: :model-type ~S must match parent ~S"
		 child model-type parent))
  (unless (stringp doc)
	  (error "defchild: ~S: illeagal documentation ~S" child doc))
  (if (and (get dummy :values)
	   (or nograph (member dummy (graph-nodes #!*model-graph*))))
      (when sink (eval `(defsink ,dummy)))
    (eval `(defatt ,dummy ,.(case model-type (:binomial (list :failure))
				 (:poisson (list :poisson max-failures)))
	     ,doc :nograph ,nograph :sink ,sink)))
					;need to do defatt first
  (push `(setf ,child (get ',child :nominal-val)) form)
  (if (and (eq model-type :poisson)
	   (eql max-failures (the fixnum (get parent :max-failures)))
	   (eql rate (get parent :default-rate)))
      (push `(setf (get ',child :nominal-val)
		   (copy-change-frame (get ',child :frame) (get ',parent :nominal-val)))
	    form)
    (push `(setf (get ',child :nominal-val)
	       (multiple-value-call #'impute
	          (get ',child :frame) (nominal (get ',parent :distribution))
		  :model-type ,model-type
		  :rate ,rate :max-events ,max-failures))
	form))
  (push `(set #?*child-list* (adjoin ',child #!*child-list*)) form)
  (push `(push ',child (get ',parent :children)) form)
  (push `(setf (get ',child :parent) ',parent) form)
  (push `(setf (get ',child :frame) '(,dummy)) form)
  (push `(setf (get ',child :values) (get ',dummy :values)) form)
  (push `(setf (get ',child :model-type) ,model-type) form)
  (when (eq :poisson model-type)
	(push `(setf (get ',child :rate) ,rate) form)
	(push `(setf (get ',child :max-failures) ,max-failures) form))
  (unless nograph
	  (push `(pushnew (quote (,dummy)) (graph-edges ,#?*model-graph*)
			  :test #'equal) form)
	  (if (setq otherpair (assoc (list dummy) #!*val-list* :test #'equal))
	      (warn "defchild:This edge ~S (~S) shadows edge ~S" child
		    (list dummy) (cdr otherpair)))
	  (push `(set #?*val-list* (acons ',child '(,dummy) #!*val-list*)) form))
  (push `(setf (documentation ',child 'function) ,doc) form)
  `(prog1 ,.form))




			  
;;;; Repetition -- These macros allow the repeated description of
;;;; similar system.

;; n-syms -- takes a symbol and a repetition count and returns a list
;; of count new symbols with a count added.
(defun n-syms (symbol count &aux (lst '()) )
  (declare (type Symbol symbol) (type Integer count)
	   (type List lst)
	   (:returns (type (List Symbol) lst)))
  "Takes a <symbol> and a repetition <count> and returns a list of
<count> symbols with names of the form <symbol>-n."
  (dotimes (i count lst)
	   (push (add-count symbol *package*) lst)))


;;; repeatdef -- repeatdefinitions
;;; (repeatdef symbol sym-list . {forms}*)
;;; This executes {forms}* repeatedly in an implicit progn.  Forms are
;;; repeated once for each element of sym-list.  On each iteration any
;;; occurance of symbol is replaced by the current element of sym-lst1.
;;; This macro was written for Bob on his birthday.
(defmacro repeatdef (symbol1 sym-lst1 &body forms )
  (declare (type Symbol symbol1) (type List sym-lst1) (type List forms))
  "(repeatdef <symbol> <sym-lst1> . forms)

This executes <forms> repeatedly in an explicit #'progn.  Forms are
repeated once for each element of <sym-lst1>.  On each iteration any
occurance of <symbol> in <forms> is replaced by the current element of
<sym-lst1>.  <symbol> and <sym-lst> are evaled.  Return value will
depend on <forms>." 
  (setq symbol1 (eval symbol1) sym-lst1 (eval sym-lst1))
  (cons 'progn
	(mapcan #'(lambda (new-sym1)
		    (subst new-sym1 symbol1 forms))
		sym-lst1)))

;;; repeatdef2 -- repeats definitions with two symbols substitutions
(defmacro repeatdef2 (symbol1 symbol2 sym-lst1 sym-lst2
			      &body forms &aux (outform`()) )
  (declare (type Symbol symbol1 symbol2)
	   (type List sym-lst1 sym-lst2) (type List forms))
    "(repeatdef2 <symbol1> <symbol2> <sym-lst1> <sym-lst2> . forms)

This executes <forms> repeatedly in an explicit #'progn.  Forms are
repeated once for each element of <sym-lst1> and <sym-lst2> which
should be the same length.  On each iteration any occurance of
<symbol1> in <forms> is replaced by the current element of <sym-lst1>.
Each occurance of <symbol2> is replaced by the current element of
<sym-list2>.  <symbol>s and <sym-lst>s are evaled.  Return value will
depend on <forms>."
  (setq symbol1 (eval symbol1) sym-lst1 (eval sym-lst1))
  (setq symbol2 (eval symbol2) sym-lst2 (eval sym-lst2))
  (cons 'progn
	(mapcan #'(lambda (new-sym1 new-sym2)
		    (subst new-sym2 symbol2
			    (subst new-sym1 symbol1 forms)))
		sym-lst1 sym-lst2)))


;;; repeatdef3 -- repeats definitions with three symbols substitutions
(defmacro repeatdef3 (symbol1 symbol2 symbol3 sym-lst1 sym-lst2 sym-lst3
			      &body forms &aux (outform`()) )
  (declare (type Symbol symbol1 symbol2 symbol3)
	   (type List sym-lst1 sym-lst2 sym-lst3) (type List forms))
  "(repeatdef3 <symbol1>  <symbol2>  <symbol3>
	       <sym-lst1> <sym-lst2> <sym-lst3> . forms)

This executes <forms> repeatedly in an explicit #'progn.  Forms are
repeated once for each element of <sym-lst1>, <sym-lst2> and
<sym-lst3> which should be the same length.  On each iteration any
occurance of <symbol1> in <forms> is replaced by the current element
of <sym-lst1>.  Each occurance of <symbol2> is replaced by the current
element of <sym-list2>.  And so forth.  <symbol>s and <sym-lst>s are
evaled.  Return value will depend on <forms>."
  (setq symbol1 (eval symbol1) sym-lst1 (eval sym-lst1))
  (setq symbol2 (eval symbol2) sym-lst2 (eval sym-lst2))
  (setq symbol3 (eval symbol3) sym-lst3 (eval sym-lst3))
  (cons 'progn
	(mapcan #'(lambda (new-sym1 new-sym2 new-sym3)
		    (subst new-sym3 symbol3
			    (subst new-sym2 symbol2
				    (subst new-sym1 symbol1 forms))))
		sym-lst1 sym-lst2 sym-lst3)))

;;; repeatdef4 -- repeats definitions with three symbols substitutions
(defmacro repeatdef4 (symbol1 symbol2 symbol3 symbol4
		      sym-lst1 sym-lst2 sym-lst3 sym-lst4
			      &body forms &aux (outform`()) )
  (declare (type Symbol symbol1 symbol2 symbol3 symbol4)
	   (type List sym-lst1 sym-lst2 sym-lst3 sym-lst4)
	   (type List forms))
  "(repeatdef4 <symbol1>  <symbol2>  <symbol3>  <symbol4>
	       <sym-lst1> <sym-lst2> <sym-lst3> <sym-lst3> . forms)

This executes <forms> repeatedly in an explicit #'progn.  Forms are
repeated once for each element of <sym-lst1>, <sym-lst2>, <sym-lst3>
and <sym-lst4> which should be the same length.  On each iteration any
occurance of <symbol1> in <forms> is replaced by the current element
of <sym-lst1>.  Each occurance of <symbol2> is replaced by the current
element of <sym-list2>.  And so forth.  <symbol>s and <sym-lst>s are
evaled.  Return value will depend on <forms>."
  (setq symbol1 (eval symbol1) sym-lst1 (eval sym-lst1))
  (setq symbol2 (eval symbol2) sym-lst2 (eval sym-lst2))
  (setq symbol3 (eval symbol3) sym-lst3 (eval sym-lst3))
  (setq symbol4 (eval symbol4) sym-lst4 (eval sym-lst4))
  (cons 'progn
	(mapcan #'(lambda (new-sym1 new-sym2 new-sym3 new-sym4)
		    (subst new-sym4 symbol4
		       (subst new-sym3 symbol3 
			    (subst new-sym2 symbol2
				    (subst new-sym1 symbol1 forms)))))
		sym-lst1 sym-lst2 sym-lst3 sym-lst4)))



;;;; Poissons functions for dealing with poissons attribute types in
;;;; particular   

;;; defandpoi
;; This sums two belief functions, one of which is a
;; poisson-like with up to max-events events, the other is a list of
;; binomials with max-event repetitions.  The result is a poisson-like
;; with max-events repetitons.  
;; (defandpoibin name (<result> <poi> <bin>) [<doc>] [<cf>])
;; <doc> is optional documentation and <cf> is confidence factor
;; :nograph is not allowed.
(defmacro defandpoi (name (result poi bin) &body bod &aux (doc "")
			     (frame nil) (cf 1.0) (notp nil) 
			     (ps nil) rep )
  (declare (type Symbol name) (type Symbol result poi)
	   (type List bin) (type String doc) (type Number cf)
	   (:returns (type Belief-Function)))
  "(defandpoi <name> (<result> <poi> <bin>) [<doc>] [<cf>])

Builds a belief function <name> which is the and of a poisson events
<poi> and a list of binary events <bin>.  A <result> occurs when both
a <poi> occurs and the corresponding <bin> event occurs.

<Doc-string> is an optional docmentation string.  <cf> is a certainty
factor (defaults to 1.0).
"
  (when (eql 'not (car bin))
	(setq bin (cdr bin) notp t))
  (when (stringp (car bod)) (setq doc (pop bod)))
  (when (get-number (car bod)) (setq cf (get-number (car bod))) )
  (unless (= (the fixnum (setq rep (length bin)))
	     (the fixnum (get (car (rassoc result #!*val-list* :test #'equal-set)) :max-events))
	     (the fixnum (get (car (rassoc poi #!*val-list* :test #'equal-set)) :max-events)))
	  (error "defandpoi: repetitions do not match for ~S, ~S and ~S"
		 result poi bin))
  (setq frame (cons result (cons poi bin)))
  (dotimes (poi-count (1+ (the fixnum (length bin)))) ;iteration over the number of poisson-events
     (dotimes (res-count (1+ poi-count)) ;iteration over the number of
					 ;unchecked events
	(setq ps (append ps (chose-bin res-count poi-count
				        rep bin notp)))))
  `(defval ,name ,frame ,doc ,(list cf ps)))



;;; deforpoi -- produces a poisson event which occurs if either of its
;;; component poisson events occur.
;; This sums two belief functions, both of which are
;; poisson-like the first has up to max-event1 events, the second has
;; up to max-event2 events.  The result is a poisson-like with
;; max-event1+ max-event2 possible events.
;; (deforpoi name (<result> <poi1> <poi2>) [<doc>] [<cf>])
;; <doc> is optional documentation and <cf> is confidence factor
(defmacro deforpoi (name (result poi1 poi2) &body bod &aux (doc "")
			     (frame nil) (cf 1.0) (ps nil))
  (declare (type Symbol name) (type Symbol result poi1 poi2)
	   (type String doc) (type Number cf)
	   (:returns (type Belief-Function)))
  "(deforpoi <name> (<result> <poi1> <poi2>) [<doc>] [<cf>])

Builds a belief function <name> which is the or of two poisson events
<poi1> and <poi2>.  A <result> event occurs when either a <poi1> or a
<poi2> event occurs.

<Doc-string> is an optional docmentation string.  <cf> is a certainty
factor (defaults to 1.0).
"
  (when (stringp (car bod)) (setq doc (pop bod)))
  (when (get-number (car bod)) (setq cf (get-number (car bod)) ))
  (let ((maxt (get (car (rassoc result #!*val-list* :test #'equal-set)) :max-events))
	(max1 (get (car (rassoc poi1 #!*val-list* :test #'equal-set)) :max-events))
	(max2 (get (car (rassoc poi2 #!*val-list* :test #'equal-set)) :max-events)))
    (declare (fixnum maxt max1 max2))
    (unless (eql (+ max1 max2) maxt)
	    (error "deforpoi: repetitions do not match for ~S, ~S and ~S"
		   result poi1 poi2))
    (setq frame (list result poi1 poi2))
    (dotimes (poi1-count (1+ max1)) ;iteration over the number of type
				    ;1 events
       (dotimes (poi2-count (1+ max2)) ;iteration over the number of
				       ;type 2 events 
	  (push (list (add+ (the fixnum (+ poi1-count poi2-count)) maxt)
		      (add+ poi1-count max1) (add+ poi2-count max2)) ps)))
    `(defval ,name ,frame ,doc ,(list cf ps))))

;; defreducepoi -- this takes a poisson of one max-events and reduces
;; it to one of a lower max-events.
;; (dereducepoi name (corser-var finer-var) [<doc>])
;; :nograph is not allowed
(defmacro defreducepoi (name (corser finer) &optional (doc "")
			     &aux corse-n fine-n (bel-el nil))
  (declare (type Symbol name) (type Symbol corser finer)
	   (type Fixnum corse-n fine-n)
	   (:returns (type Belief-Function)))
  "(defreducepoi <name> (<corser> <finer>) [<doc>])

This produces a logical belief function which reduces a poisson event
<finer> to a smaller frame <corser>.  Especially useful for
simplifying complex and hence rare events."
  (setq corse-n (get corser :poisson))
  (setq fine-n (get finer :poisson))
  (unless (numberp corse-n) (error "defreducepoi: malformed poisson ~
                                    attribute ~S" corser))
  (unless (numberp fine-n) (error "defreducepoi: malformed poisson ~
                                    attribute ~S" finer))
  (unless (<= corse-n fine-n)
	  (error "defreducepoi: reducing from granularity ~S to ~S"
		 fine-n corse-n))
  (dotimes (i corse-n) (push (list i i) bel-el))
  (push (list (add+ corse-n corse-n) (iota fine-n corse-n t)) bel-el)
  `(defbel ,name (,corser ,finer) ,doc (1.0 ,bel-el)))




;;; auxilary focal element construction routines.

;; chose-bin -- chose ntrue binomials from ntotal, and pad with
;; unknowns up to rep.  Also append ntrue and ntotal to the front of
;; the list (adding + if necessary).  If notp is true, use the
;; negation of the variable.
(defun chose-bin (ntrue ntotal rep binvars notp )
  (declare (type Fixnum ntrue ntotal rep) (type List binvars)
	   (:returns (type PS-Set)))
  "Builds a PS(TS)-set from repeated binomials.  Assumes that <ntrue>
of the first <ntotal> <binvars> are true.  The result is padded with
<rep> unknowns."
  (let ((true-fun (if notp #'logical-false #'logical-true))
	(false-fun (if notp #'logical-true #'logical-false))
	(binknowns (butlast binvars (- rep ntotal)))
	(restunknowns (mapcar #'logical-unknown (nthcdr ntotal binvars))))
    (mapcar #'(lambda (#1=#:x)
		(cons (add+ ntrue rep)
		      (cons (add+ ntotal rep)
			    (append #1# restunknowns))))
	    (chose-bin-aux ntrue ntotal binknowns true-fun false-fun))))

;; chose-bin-aux -- fits together the true and false values
(defun chose-bin-aux (ntrue ntotal binvars true-fun false-fun)
  (declare (type Fixnum ntrue ntotal) (type List binvars)
	   (ftype (function (t) t) true-fun false-fun)
	   (:returns (type PS-set)))
  (cond ((zerop ntotal) `() )
	((zerop ntrue) (mapcar false-fun binvars))
	((eql ntotal ntrue) (mapcar true-fun binvars))
	((> ntrue ntotal) nil)
	(t (append (mapcar #'(lambda (#1=#:x)
			       (cons (apply true-fun (car binvars)) #1#))
			   (chose-bin-aux (the fixnum (1- ntrue))
					  (the fixnum (1- ntotal)) (cdr binvars)
					  true-fun false-fun))
		   (mapcar #'(lambda (#1#)
			       (cons (apply false-fun (car binvars)) #1#))
			   (chose-bin-aux ntrue (the fixnum (1- ntotal))
					  (cdr binvars)
					  true-fun false-fun))))))




  

;;;; defval -- could be either a defbel or a defpot depending on mode.
(defmacro defval (name attlist &body bod &aux (doc ""))
  (if (stringp (car bod)) (setq doc (pop bod)))
  "Expands to defbel or defpot macro call depending on #!*MODE*."
  (case #!*MODE*
	(:belief `(defbel ,name ,attlist ,doc ,.bod))
	(:prob `(defpot ,name ,attlist ,doc :ps-list ,.bod))))


;;; defsink -- This makes an attribute correspond a sink. 
;;; sinks have three properties 
;;; :sink set to the name of the attribute being sunk 
;;; :sink-source set to the name of a belief function contaning that
;;; sunk attribute from which it should get its:
;;; :sink-value
(defmacro defsink (name &optional (att name))
  (declare (type Symbol name) (type Symbol att)
	   (:returns (type List #?*sink-list*)))
  "Defines a sink with name <name> over attribute <att>."
  (unless (get att :values)
	  (error "defsink: ~S attribute undefined" att))
  `(progn (setf (get ',name :sink) ',att)
	  (pushnew ',name ,#?*sink-list*)))





;;; binomial and and or -- defines very large and and or gates as
;;; collection of smaller gates.
(defmacro defandbin (name (system . components) &key (doc "")
			  &aux (form '() ))
  (declare (type Symbol name) (type Symbol system)
	   (type List components)
	   (:returns (type Belief-Function)))
  "Creates very large and-gates by breaking up into smaller and-gates
and introducing intermediate variables." 
  (case (length components)
	(0 (error "defandbin: Insufficient components in and gate ~S" name))
	(1 (error "defandbin: Insufficient components in and gate ~S" name))
	(2 `(defiff ,name (,system . ,components) ,doc))
	(3 `(defiff ,name (,system . ,components) ,doc))
	(t (multiple-value-bind (pair-list left-over)
				 (make-pairs components)
	     (declare (list pair-list))
             (let ((aux-names (n-syms '|iand| (length pair-list))))
	       (eval `(repeatdef2 'gate 'comp ',aux-names ',pair-list
		       (defatt gate :failure)
		       (defiff gate (gate . comp) ,(symbol-name name))))
	       `(defandbin ,name (,system ,.(if left-over
						(list left-over)
					      nil)
					 . ,aux-names) :doc ,doc))))))

;; deforbin
(defmacro deforbin (name (system . components) &key (doc "")
			  &aux (form '() ))
  (declare (type Symbol name) (type Symbol system)
	   (type List components)
	   (:returns (type Belief-Function)))
  "Creates large or-gates by breaking up into smaller or-gates and
introducing intermediate nodes."
  (case (length components)
	(0 (error "deforbin: Insufficient components in or gate ~S" name))
	(1 (error "deforbin: Insufficient components in or gate ~S" name))
	(2 `(defiffor ,name (,system . ,components) ,doc))
	(3 `(defiffor ,name (,system . ,components) ,doc))
	(t (multiple-value-bind (pair-list left-over)
				 (make-pairs components)
	     (declare (list pair-list))
             (let ((aux-names (n-syms '|ior| (length pair-list))))
	       (eval `(repeatdef2 'gate 'comp ',aux-names ',pair-list
		       (defatt gate :failure)
		       (defiffor gate (gate . comp) ,(symbol-name name))))
	       `(deforbin ,name (,system ,.(if left-over
					       (list left-over)
					     nil)
					 . ,aux-names) :doc ,doc))))))



;;; provide when loaded
(bel-provide :readnuke)