;;; read.lisp -- basic belief function defs.

;;; Copyright 1987 Russell G. Almond
;;; License is granted to copy this program for education or research
;;; purposes, with the restriction that no portion of this program may
;;; be copied without also copying this notice.  All other rights
;;; reserved. 

;;; 1/15/87 -- File created

;;; 12/1/87 -- Conditional imbedding (defbelcond) added.

;;; 2/17/87 -- Added warning for duplicate edges.

;;; 7/13/88 -- Added explicit **frame** at edge of graph.  Also keeps
;;; track of *min-input* for later conditioning.

;;; 2/16/89 -- broke into two pieces to sevre  potentials as well as
;;; belief functions

;;; 7/19/89 -- Version 1.1
;;;	* Accept parameters in place of numbers [d]

;;; 8/4/89
;;;	* Now read macros accept :nograph and :sink keywords (with
;;; optional values).  If either of the keywords is given a value,
;;; then it must be either t or nil. [d]

;;; 8/9/89 
;;;	* Now :sink is reserved for attributes.  


;;; 2/26/92 -- Version 1.2  Documentation Cleanup

;(provide 'read)
(in-package :belief )
(bel-require :lowread "lowread")
(bel-require :arithmatic "arithmatic")
;(use-package '(basic graphs sets))
;(export '(defatt reset-model
;	   ;inherited from lowread.lisp
;	  defbel defif defand defor defifor defis 
;	  defnand defnor defxor defiff defiffor defbelcond
;	  defbel-nograph
;	  ))
;; also inherited are *true-false-values* *model-graph* *val-list*

;;; These definitions are meant to be used in a rule data base.  Thus
;;; the defbel expands to a series of commands that define a new belief
;;; function, and the defatt defines a new attribute (variable).
;;; defatts and defbels can come in any order, but in general, the
;;; defatt defining the attributes used in a belief function must come
;;; before the defbel that uses them (Put the defatts first).  defif,
;;; defand, defor, etc are abbreviated modes that work for logical
;;; belief functions.


;;; Functions for defining belief functions -- defbel
;;; (defbel <name> <frame> [<Doc-string>] {<m-value>}* )
;;; <name> is a name for the belief function
;;; <Doc-string> is the documentation string for the belief function
;;; <frame> is the list of attributes in the frame of the belief
;;; function
;;; <m-value> is a pair consisting of a number and a focal element in
;;; ps-set form.

;; defbel -- constructs the commands for making a belief function and
;; pushes them onto form.  Form is then executed as a progn
(defmacro defbel (name &body bod &aux (form '()) (otherpair nil))
  (declare (type Symbol name) (type List frame)
	   (:returns (type Belief-Function)))
  "Define a new belief function
  (defbel <name> <frame> [<Doc-string>] [:nograph {t | nil}] {<m-value>}*)

<name> is a name for the belief function.  <frame> is the list of
attributes in the frame of the belief function.  <m-value> is a pair
consisting of a generalized number and a focal element in ps-set form.
defbel will take generaized numbers in <m-value> pairs.  In
particular, it will take numbers, constants, parameters, variables,
and expresssions which eval to a number.  Uses #'get-number to check
for number.

<Doc-string> is the documentation string for the belief function

The keyword :nograph t suppresses adding the relationship to
#?*model-graph* and #?*val-list*.

  "
  (let ((frame (pop bod)) (m-list nil) (nograph nil) )
    (unless (listp frame)
	    (error "defbel:~S is not an attribute list" frame))
    (when (stringp (car bod))
	  (push `(setf ,`(documentation (quote ,name) 'function) ,(pop bod))
		form))
    (loop (unless (keywordp (car bod)) (return))
	  (case (pop bod)
		(:nograph (if (or (null (car bod)) (eq t (car bod)))
			      (setq nograph (pop bod))
			    (setq nograph t)))))
    (setq m-list (mapcar #'(lambda (#1=#:x) (setup-focal #1# frame)) bod))
    (setq m-list (append m-list
			 (list (make-m-value :element **frame**
					     :m (reduce #'- (mapcar
							     #'m-value-m 
							     m-list)
							:initial-value 1.0)))))
    (push `(setq ,name (make-belief-function :frame (quote ,frame)
					     :ms (quote ,m-list))) form)
    (unless nograph
	    (if (subsetp frame (graph-nodes #!*model-graph*))
		(push `(pushnew (quote ,frame) (graph-edges ,#?*model-graph*)
				:test #'equal) form)
	      (error "defbel:~S contains undefined attributes" frame))
	    (if (setq otherpair (assoc frame #!*val-list* :test #'equal))
		(warn "defbel:This edge ~S (~S) shadows edge ~S" name frame
		      (cdr otherpair)))
	    (push `(set #?*val-list* (acons ',name ',frame #!*val-list*)) form)))
  `(prog1 ,.form))


;; setup-focal -- turns each focal element into a m-value structure and
;; checks for consistancy.  Returns the m-value
(defun setup-focal (m-pair attlist)
  (declare (type List m-pair) (type List attlist)
	   (:returns (type M-Value)))
  "Conversts a pair (<m-pair>) consisting of a mass and a focal element in
PS(TS)-set form defined over frame <attlist> into a M-Value structure
for use in a Belief-Function."
  (multiple-value-bind (m elm)
      (if (get-number (car m-pair)) (values (get-number (car m-pair)) (cadr m-pair))
	(if (get-number (cadr m-pair)) (values (get-number (cadr m-pair)) (car m-pair))
	  (error "defbel:Bad m-list ~S" m-pair)))
    (update-min-input m)
    (if (check-elm elm attlist)
	(make-m-value :m (long-float m) :element (get-ps-set-sym elm))
      (error "defbel:Bad focal element ~S" elm))))


;; check-elm -- checks a focal element (ps-set) to see if it is valid.
;; returns t if it is valid, does not return (calls error) if it is not.
(defun check-elm (elm attlist)
  (declare (type list Attlist)
	   (type PS-Set elm)
	   (:returns (type (member T Nil))))
  "Checks <elm> to see if it is valid PS(TS)-set in frame <attlist>."
  (if (listp elm) (every #'(lambda (#1=#:x) (check-atts #1# attlist)) elm)
    (error "defbel:~S is not a ps-set" elm)))

;;check-atts -- checks an single entry in a ps-set for validity
(defun check-atts (sub-elm attlist)
  (declare (type List sub-elm) (type List attlist)
	   (:returns (type (member T Nil))))
  "Checks single tuple <sub-elm> in a PS(TS)-set in frame <attlist>."
  (cond ((and (endp sub-elm) (endp attlist)) t)
	((or (endp sub-elm) (endp attlist))
	 (error "defbel:Length mismatch ~S and ~S" sub-elm attlist))
	((listp (car sub-elm))
	 (if (subsetp (car sub-elm) (get (car attlist) :values))
	     (check-atts (cdr sub-elm) (cdr attlist))
	   (error "defbel:Unrecognized values ~S for ~S"
		  (car sub-elm) (car attlist))))
	(t (if (member (car sub-elm) (get (car attlist) :values))
	       (check-atts (cdr sub-elm) (cdr attlist))
	     (error "defbel:Unrecognized value ~S for ~S"
		    (car sub-elm) (car attlist))))))


;;; Logical macroes -- these macroes expand into defbel constructs.
;;; They are of the form:
;;; (def<fn> <name>  <attlist> [<Doc-string>] [<cf>])
;;; <fn> is one of if, and, or, ifor, nand, nor, xor
;;; <name> is the name given to the belief function
;;; <Doc-string> documents the belief function
;;; <attlist> is the list of attributes and '(not attribute)s effected 
;;; <cf> is an optional confidence factor [Default = 1.0]
;;; it is assumed that all attributes are of the :true-false type
;;; The result is a belief function with a single focal element whose
;;; m-value is <cf>.
;;; The focal element varies with <fn> as follows
;;;  if, ifor -- equivalent to the propositions
;;;   if (and (cdr <attlist>)) then (car attlist),
;;;   if (or (cdr <attlist>)) then (car attlist) 
;;; respectively
;;;  iff, iffor -- same with if and only if
;;; and, or, nand, nor, xor -- equivalent to the propositions
;;;   (<fn> <attlist>)
;;; in each case if (not <attribute>) is used instead of <attribute>
;;; then the logical values of true and false will be switch for that
;;; attribute.  It assumes that (car (get <attribute> :values)) is
;;; logical-true and the (cadr (get <attribute> :values)) is
;;; logical-false.

;; defif -- if (and (cdr attributes)) then (car attributes)
(defmacro defif (name (consequent . conditions)
		      &body bod &aux (doc "") (cf 1.0) (nograph nil))
  (declare (type Symbol name) (type List conditions)
	   (:returns (type Belief-Function potential)))
  "(defif <name>  (<consequent> {<condidion>}*) [<Doc>] [:nograph [t|nil]] [<cf>])

Defines a new belief function representing an if-then relationship,
particularly, <consequent> is true if (and <condition>s) is true.  Each
element of the truth table corresponding to this relationship is
assigned value <cf> (default 1.0).

<Doc> is an optional documentation string.

The keyword :nograph t suppresses adding the relationship to
#?*model-graph* and #?*val-list*.

<consequent> and <condition> are generalized attributes.  That is they
can either be (1) an attribute, (2) (<att> . <values>) where <att> is
an attribute and <values> is a list of values to be reguarded as true
or (3) (not <att>) where <att> is another generalized attribute.  
Attributes are passed to #'logical-true and #'logical-false.  Assumes
that (except for case 2) the car of the values list for attribute is
`true'. 

"
  (when (stringp (car bod)) (setq doc (pop bod)))
  (when (eq (car bod) :nograph) (pop bod)
	(if (or (eq (car bod) t) (eq (car bod) nil))
	    (setq nograph (pop bod))
	  (setq nograph t)))
  (when (get-number (car bod)) (setq cf (get-number (car bod))) )
  `(defbel ,name ,(clean-attribute-list (cons consequent conditions)) ,doc
     ,@(if nograph '(:nograph t)) ,(list cf (set-up-if consequent conditions))))


;; defifor -- if (or (cdr attributes)) then (car attributes)
(defmacro defifor (name (consequent . conditions)
		      &body bod &aux (doc "") (cf 1.0) (nograph nil))
  (declare (type Symbol name) (type List conditions)
	   (:returns (type Belief-Function potential)))
    "(defifor <name>  (<consequent> {<condidion>}*) [<Doc>] [:nograph [t|nil]] [<cf>])

Defines a new belief function representing an if-then relationship,
particularly, <consequent> is true if (or <condition>s) is true.  Each
element of the truth table corresponding to this relationship is
assigned value <cf> (default 1.0).

<Doc> is an optional documentation string.

The keyword :nograph t suppresses adding the relationship to
#?*model-graph* and #?*val-list*.

<consequent> and <condition> are generalized attributes.  That is they
can either be (1) an attribute, (2) (<att> . <values>) where <att> is
an attribute and <values> is a list of values to be reguarded as true
or (3) (not <att>) where <att> is another generalized attribute.  
Attributes are passed to #'logical-true and #'logical-false.  Assumes
that (except for case 2) the car of the values list for attribute is
`true'. 

"
  (when (stringp (car bod)) (setq doc (pop bod)))
  (when (eq (car bod) :nograph) (pop bod)
	(if (or (eq (car bod) t) (eq (car bod) nil))
	    (setq nograph (pop bod))
	  (setq nograph t)))
  (when (get-number (car bod)) (setq cf (get-number (car bod))) )
  `(defbel ,name ,(clean-attribute-list (cons consequent conditions)) ,doc
     ,@(if nograph '(:nograph t)) ,(list cf (set-up-ifor consequent conditions))))


;; defiff -- (and (cdr attributes)) if and only if (car attributes)
(defmacro defiff (name (consequent . conditions)
		      &body bod &aux (doc "") (cf 1.0) (nograph nil))
  (declare (type Symbol name) (type List conditions)
	   (:returns (type Belief-Function potential)))
  "(defiff <name>  (<consequent> {<condidion>}*) [<Doc>] [:nograph [t|nil]] [<cf>])

Defines a new belief function representing an equivalence
relationship, particularly, <consequent> is true if and only if (and
<condition>s) is true.  Each element of the truth table corresponding
to this relationship is assigned value <cf> (default 1.0).

<Doc> is an optional documentation string.

The keyword :nograph t suppresses adding the relationship to
#?*model-graph* and #?*val-list*.

<consequent> and <condition> are generalized attributes.  That is they
can either be (1) an attribute, (2) (<att> . <values>) where <att> is
an attribute and <values> is a list of values to be reguarded as true
or (3) (not <att>) where <att> is another generalized attribute.  
Attributes are passed to #'logical-true and #'logical-false.  Assumes
that (except for case 2) the car of the values list for attribute is
`true'. 

"  
  (when (stringp (car bod)) (setq doc (pop bod)))
  (when (eq (car bod) :nograph) (pop bod)
	(if (or (eq (car bod) t) (eq (car bod) nil))
	    (setq nograph (pop bod))
	  (setq nograph t)))
  (when (get-number (car bod)) (setq cf (get-number (car bod))) )
  `(defbel ,name ,(clean-attribute-list (cons consequent conditions)) ,doc
     ,@(if nograph '(:nograph t)) ,(list cf (set-up-iff consequent conditions))))


;; defiffor -- (or (cdr attributes)) if and only if (car attributes)
(defmacro defiffor (name (consequent . conditions)
		      &body bod &aux (doc "") (cf 1.0) (nograph nil))
  (declare (type Symbol name) (type List conditions)
	   (:returns (type Belief-Function potential)))
  "(defiffor <name>  (<consequent> {<condidion>}*) [<Doc>]
 						   [:nograph [t|nil]] [<cf>])

Defines a new belief function representing an equivalence
relationship, particularly, <consequent> is true if and only if (or
<condition>s) is true.  Each element of the truth table corresponding
to this relationship is assigned value <cf> (default 1.0).

<Doc> is an optional documentation string.

The keyword :nograph t suppresses adding the relationship to
#?*model-graph* and #?*val-list*.

<consequent> and <condition> are generalized attributes.  That is they
can either be (1) an attribute, (2) (<att> . <values>) where <att> is
an attribute and <values> is a list of values to be reguarded as true
or (3) (not <att>) where <att> is another generalized attribute.  
Attributes are passed to #'logical-true and #'logical-false.  Assumes
that (except for case 2) the car of the values list for attribute is
`true'. 

"  
  (when (stringp (car bod)) (setq doc (pop bod)))
  (when (eq (car bod) :nograph) (pop bod)
	(if (or (eq (car bod) t) (eq (car bod) nil))
	    (setq nograph (pop bod))
	  (setq nograph t)))
  (when (get-number (car bod)) (setq cf (get-number (car bod))) )
  `(defbel ,name ,(clean-attribute-list (cons consequent conditions)) ,doc
     ,@(if nograph '(:nograph t)) ,(list cf (set-up-iffor consequent conditions))))

;; defand -- (and attributes)
(defmacro defand (name  conditions
		      &body bod &aux (doc "") (cf 1.0) (nograph nil))
  (declare (type Symbol name) (type List conditions)
	   (:returns (type Belief-Function potential)))
  "(defand <name>  ({<condidion>}*) [<Doc>] [:nograph [t|nil]] [<cf>])

Defines a new belief function representing a logical assertion:
particularly, all <condition>s are true.  Each element of the truth
table corresponding to this relationship is assigned value <cf>
(default 1.0). 

<Doc> is an optional documentation string.

The keyword :nograph t suppresses adding the relationship to
#?*model-graph* and #?*val-list*.

<consequent> and <condition> are generalized attributes.  That is they
can either be (1) an attribute, (2) (<att> . <values>) where <att> is
an attribute and <values> is a list of values to be reguarded as true
or (3) (not <att>) where <att> is another generalized attribute.  
Attributes are passed to #'logical-true and #'logical-false.  Assumes
that (except for case 2) the car of the values list for attribute is
`true'. 

"
  (if (listp (car bod)) (setq conditions (pop bod))
    (error "defand:bad attribute list ~S" (car bod)))
  (when (stringp (car bod)) (setq doc (pop bod)))
  (when (eq (car bod) :nograph) (pop bod)
	(if (or (eq (car bod) t) (eq (car bod) nil))
	    (setq nograph (pop bod))
	  (setq nograph t)))
  (when (get-number (car bod)) (setq cf (get-number (car bod))) )
  `(defbel ,name ,(clean-attribute-list conditions) ,doc
     ,@(if nograph '(:nograph t)) ,(list cf (logical-and conditions))))

;; defor -- (or attributes)
(defmacro defor (name conditions
		      &body bod &aux (doc "") (cf 1.0) (nograph nil))
  (declare (type Symbol name) (type List conditions)
	   (:returns (type Belief-Function potential)))
  "(defor <name>  ({<condidion>}*) [<Doc>] [:nograph [t|nil]] [<cf>])

Defines a new belief function representing a logical assertion:
particularly, at least one <condition> is true.  Each element of the
truth table corresponding to this relationship is assigned value <cf>
(default 1.0).

<Doc> is an optional documentation string.

The keyword :nograph t suppresses adding the relationship to
#?*model-graph* and #?*val-list*.

<consequent> and <condition> are generalized attributes.  That is they
can either be (1) an attribute, (2) (<att> . <values>) where <att> is
an attribute and <values> is a list of values to be reguarded as true
or (3) (not <att>) where <att> is another generalized attribute.  
Attributes are passed to #'logical-true and #'logical-false.  Assumes
that (except for case 2) the car of the values list for attribute is
`true'. 

"  (if (listp (car bod)) (setq conditions (pop bod))
    (error "defor:bad attribute list ~S" (car bod)))
  (when (stringp (car bod)) (setq doc (pop bod)))
  (when (eq (car bod) :nograph) (pop bod)
	(if (or (eq (car bod) t) (eq (car bod) nil))
	    (setq nograph (pop bod))
	  (setq nograph t)))
  (when (get-number (car bod)) (setq cf (get-number (car bod))) )
  `(defbel ,name ,(clean-attribute-list conditions) ,doc
     ,@(if nograph '(:nograph t)) ,(list cf (logical-or conditions))))

;; defnand -- (nand attributes)
(defmacro defnand (name conditions
		      &body bod &aux (doc "") (cf 1.0) (nograph nil))
  (declare (type Symbol name) (type List conditions)
	   (:returns (type Belief-Function potential)))
  "(defnand <name>  ({<condidion>}*) [<Doc>] [:nograph [t|nil]] [<cf>])

Defines a new belief function representing a logical assertion:
particularly, at least one <condition> is false.  Each element of the truth
table corresponding to this relationship is assigned value <cf>
(default 1.0). 

<Doc> is an optional documentation string.

The keyword :nograph t suppresses adding the relationship to
#?*model-graph* and #?*val-list*.

<consequent> and <condition> are generalized attributes.  That is they
can either be (1) an attribute, (2) (<att> . <values>) where <att> is
an attribute and <values> is a list of values to be reguarded as true
or (3) (not <att>) where <att> is another generalized attribute.  
Attributes are passed to #'logical-true and #'logical-false.  Assumes
that (except for case 2) the car of the values list for attribute is
`true'. 

"  (if (listp (car bod)) (setq conditions (pop bod))
    (error "defnand:bad attribute list ~S" (car bod)))
  (when (stringp (car bod)) (setq doc (pop bod)))
  (when (eq (car bod) :nograph) (pop bod)
	(if (or (eq (car bod) t) (eq (car bod) nil))
	    (setq nograph (pop bod))
	  (setq nograph t)))
  (when (get-number (car bod)) (setq cf (get-number (car bod))) )
  `(defbel ,name ,(clean-attribute-list conditions) ,doc
     ,@(if nograph '(:nograph t)) ,(list cf (logical-nand conditions))))

;; defnor -- (nor attributes)
(defmacro defnor (name conditions
		      &body bod &aux (doc "") (cf 1.0) (nograph nil))
  (declare (type Symbol name) (type List conditions)
	   (:returns (type Belief-Function potential)))
  "(defnor <name>  ({<condidion>}*) [<Doc>] [:nograph [t|nil]] [<cf>])

Defines a new belief function representing a logical assertion:
particularly, all <condition>s are false.  Each element of the truth
table corresponding to this relationship is assigned value <cf>
(default 1.0). 

<Doc> is an optional documentation string.

The keyword :nograph t suppresses adding the relationship to
#?*model-graph* and #?*val-list*.

<consequent> and <condition> are generalized attributes.  That is they
can either be (1) an attribute, (2) (<att> . <values>) where <att> is
an attribute and <values> is a list of values to be reguarded as true
or (3) (not <att>) where <att> is another generalized attribute.  
Attributes are passed to #'logical-true and #'logical-false.  Assumes
that (except for case 2) the car of the values list for attribute is
`true'. 

"  (if (listp (car bod)) (setq conditions (pop bod))
    (error "defnor:bad attribute list ~S" (car bod)))
  (when (stringp (car bod)) (setq doc (pop bod)))
  (when (eq (car bod) :nograph) (pop bod)
	(if (or (eq (car bod) t) (eq (car bod) nil))
	    (setq nograph (pop bod))
	  (setq nograph t)))
  (when (get-number (car bod)) (setq cf (get-number (car bod))) )
  `(defbel ,name ,(clean-attribute-list conditions) ,doc
     ,@(if nograph '(:nograph t)) ,(list cf (logical-nor conditions))))

;; defxor -- (xor attributes)
(defmacro defxor (name conditions
		      &body bod &aux (doc "") (cf 1.0) (nograph nil))
  (declare (type Symbol name) (type List conditions)
	   (:returns (type Belief-Function potential)))
  "(defxor <name>  ({<condidion>}*) [<Doc>] [:nograph [t|nil]] [<cf>])

Defines a new belief function representing a logical assertion:
particularly, exactly one <condition> is true.  Each element of the truth
table corresponding to this relationship is assigned value <cf>
(default 1.0). 

<Doc> is an optional documentation string.

The keyword :nograph t suppresses adding the relationship to
#?*model-graph* and #?*val-list*.

<consequent> and <condition> are generalized attributes.  That is they
can either be (1) an attribute, (2) (<att> . <values>) where <att> is
an attribute and <values> is a list of values to be reguarded as true
or (3) (not <att>) where <att> is another generalized attribute.  
Attributes are passed to #'logical-true and #'logical-false.  Assumes
that (except for case 2) the car of the values list for attribute is
`true'. 

"  (if (listp (car bod)) (setq conditions (pop bod))
    (error "defxor:bad attribute list ~S" (car bod)))
  (when (stringp (car bod)) (setq doc (pop bod)))
  (when (eq (car bod) :nograph) (pop bod)
	(if (or (eq (car bod) t) (eq (car bod) nil))
	    (setq nograph (pop bod))
	  (setq nograph t)))
  (when (get-number (car bod)) (setq cf (get-number (car bod))) )
  `(defbel ,name ,(clean-attribute-list conditions) ,doc
     ,@(if nograph '(:nograph t)) ,(list cf (logical-xor conditions))))


;;; defis -- this defines a belief function on a single attribute
;;; (defis <name> (<attribute>) [<Doc>] {[:T <cf-true>] | [:F <cf-false>]}*)
;;; It give a belief function with zero, one or two focal elements,
;;; depending on the keywords.  The keyword :T indicates evidence with
;;; m-value <cf-true> for the proposition attribute is true, and the
;;; keyword :F indicates evidence for the proposition attribute is
;;; false.
(defmacro defis (name (attribute) &rest keylist &aux (doc "")
		       (cf-true nil) (cf-false nil) (m-form nil)
		       (nograph nil) )
  "(defis <name> (<attribute>) [<Doc>] {[:T <cf-true>] | [:F <cf-false>]}*)"
  (declare (symbol name))
  (when (stringp (car keylist)) (setq doc (pop keylist)))
  (loop (when (endp keylist) (return))
	(case (pop keylist)
	      (:t (if (get-number (car keylist))
		      (setq cf-true (get-number (pop keylist)))
		    (error "defis: expected a number, got ~S" (car keylist))))
	      (:f (if (get-number (car keylist))
		      (setq cf-false (get-number (pop keylist)))
		    (error "defis: expected a number, got ~S" (car keylist))))
	      (:nograph (if (or (eq (car keylist) t) (eq (car keylist) nil))
			    (setq nograph (pop keylist))
			  (setq nograph t)))
	      (t (error "defis: unrecognized matrial ~S" keylist))))
  (unless (null cf-true)
	  (push (list cf-true (list (list (logical-true attribute)))) m-form))
  (unless (null cf-false)
	  (push (list cf-false (list (list (logical-false attribute))))
		m-form))
  `(defbel ,name (,(clean-attribute attribute)) ,doc
     ,@(if nograph '(:nograph t)) ,.m-form))




;;; defbelcond -- This function constructs belief functions using the
;;; conditional embedding method.  
(defmacro defbelcond (name &body bod &aux (form '()))
  (declare (type Symbol name)
	   (:returns (type Belief-Function)))
  "Define a new belief function
  (defbelcond <name> <frame> [<Doc-string>] [:nograph [t|nil]] {<component>}*)

  where <frame> is ( {<consequences>}* :given {<conditions>}*) and
<component> is (<condition ps> {<m-value>}*).  <condition ps> is a
<ps-set> defined over the <condition>s and <value> is multiple
<ps-val-pairs> where each <ps-set> is defined over the <consequences>
frame.

This uses the rather questionable conditional embedding method of
Smetts.

<Doc-String> is an optional documentation string.

The keyword :nograph t suppresses adding the relationship to
#?*model-graph* and #?*val-list*.
"
  (declare (symbol name))
  (let ((fullattlist (pop bod)) (components nil) (nograph nil) )
    (if (not (listp fullattlist))
	(error "defbel:~S is not an attribute list" fullattlist))
    (let ((attlist (remove :given fullattlist))
	  (dependents (ldiff fullattlist (member :given fullattlist)))
	  (independents (cdr (member :given fullattlist))))
      (when (stringp (car bod))
	    (push `(setf ,`(documentation (quote ,name) 'function) ,(pop bod))
		  form))
      (loop (unless (keywordp (car bod)) (return))
	    (case (pop bod)
		  (:nograph (if (or (null (car bod)) (eq t (car bod)))
				(setq nograph (pop bod))
			      (setq nograph t)))))
      (setq components
	    (mapcar
	     #'(lambda (#1=#:x)
		 (embed-belief-function dependents independents
					(car #1#) (cdr #1#)))
	     bod))
      (push `(setq ,name (normalize-bel-fun
			  (reduce #'@+2 (quote ,components)
				  :initial-value
				  (make-belief-function :frame (quote ,attlist)
							:ms vacuous)))) form)
      (unless nograph
	      (if (subsetp attlist (graph-nodes #!*model-graph*))
		  (push `(pushnew (quote ,attlist) (graph-edges ,#?*model-graph*)
				  :test #'equal) form)
		(error "defbel:~S contains undefined attributes" attlist))
	      (push `(set #?*val-list* (acons ',name ',attlist #!*val-list*)) form))))
  `(prog1 ,.form))


;; embed-belief-function -- takes a belief function and embeds it in a
;; larger frame with the given m-list when the condition is true and
;; vacuous when the condition is false.
(defun embed-belief-function (dependent independent condition m-list)
  (declare (type List dependent) (type List independent)
	   (type List m-list) (type PS-Set condition)
	   (:returns (type Belief-function)))
  "Creates a belief function by conditional embeding.  <m-list> is a
list of m-value conditional-ps-set pairs defined over the frame
<dependent>.  <condition> is a ps-set defined over the frame
<independent>.  The result is a belief function corresponding <m-list>
when conditioned on <condition>."
  (if (not (check-elm condition independent))
      (error "defbelcond: Bad condition ~S" condition))
  (let* ((unknown (mapcar #'logical-unknown dependent))
	 (dep-frame (list (mapcar #'logical-unknown independent)))
	 (cylander
	  (mapcar
	   #'(lambda (ps-el) (append unknown ps-el))
	   (ps-set-complement condition dep-frame)))
	 (ms-list (mapcar #'(lambda (#1=#:x)
			      (setup-embed-focal #1# dependent independent
						 condition cylander))
			  m-list)))
    (make-belief-function :frame (append dependent independent)
			  :ms (append ms-list
				      (list (make-m-value :element **frame**
							  :m (reduce #'-
								     (mapcar #'m-value-m
									     ms-list)
								     :initial-value 1.0)))))))


;; setup-embed-focal -- turns each focal element into a m-value structure and
;; checks for consistancy, and conditionally embeds the m-value.
;; Returns the m-value
(defun setup-embed-focal (m-pair dependent independent condition cylander)
  (declare (type List m-pair) (type List dependent) (type List independent)
	   (type PS-Set condition) (type PS-Set cylander)
	   (:returns (type M-Value)))
  "Turns <m-pair> which is a mass <value> <ps-set> pair into a focal
element over the frame (append <dependent> <independent>) by
conditional embeding.  <ps-set> is assumed to be defined over frame
<dependent> and <condition> over frame <independent>.  <cylander> is
a PS(TS)-set defined over the whole frame which corresponds to the
case when <condition> does not hold."
  (multiple-value-bind (m elm)
      (if (get-number (car m-pair)) (values (get-number (car m-pair)) (cadr m-pair))
	(if (get-number (cadr m-pair)) (values (get-number (cadr m-pair)) (car m-pair))
	  (error "defbel:Bad m-list ~S" m-pair)))
    (update-min-input m)
    (if (not (check-elm elm dependent))
	(error "defbel:Bad focal element ~S" elm))
    (make-m-value :m (long-float m)
		  :element (get-ps-set-sym
			    (append
			     (mapcan #'(lambda (#1=#:x)
					 (mapcar #'(lambda (#2=#:y)
						     (append #1# #2#))
						 condition))
				     elm)
			     cylander)))))




			       

;;; special function for dealing with minimum-inputs
(declaim (inline update-min-input))
(defun update-min-input (m)
  (declare (type Number m)
	   (:returns (type Long-Float #!*min-input*)))
  "Updates #?*min-input* to be the smallest value seen to date."
  (set #?*min-input* (min #!*min-input* m)))


;;; provide when loaded
(bel-provide :read)