;;; probread.lisp -- basic belief function defs.

;;; Copyright 1989 Russell G. Almond
;;; License is granted to copy this program for education or research
;;; purposes, with the restriction that no portion of this program may
;;; be copied without also copying this notice.  All other rights
;;; reserved. 

;;; 2/16/89 -- broke into two pieces to sevre  potentials as well as
;;; belief functions.  This is essentially an inherited version of read.lisp

;;; 7/19/89 -- Version 1.1
;;;	* Accept parameters in place of numbers [d]


;;; 8/4/89
;;;	* Now read macros accept :nograph and :sink keywords (with
;;; optional values).  If either of the keywords is given a value,
;;; then it must be either t or nil. [d]

;;; 8/9/89 
;;;	* Sink keyword now only defined on "defatt" statements.

;;; 2/26/92 --- Version 1.2 cleanup of documentation.

;(provide 'probread)
(in-package :prob)
(bel-require :lowread "lowread")
(bel-require :potentials "potentials")
;(use-package '(basic graphs sets))
;(export '(defatt reset-model ;inherited from lowread.lisp
;	  defpot defif defand defor defifor defis 
;	  defnand defnor defxor defiff defiffor defpotcond
;	  ))
;;; also inherited are *true-false-values* *model-graph* *val-list*

;;; These definitions are meant to be used in a rule data base.  Thus
;;; the defbel expands to a series of commands that define a new belief
;;; function, and the defatt defines a new attribute (variable).
;;; defatts and defbels can come in any order, but in general, the
;;; defatt defining the attributes used in a belief function must come
;;; before the defbel that uses them (Put the defatts first).  defif,
;;; defand, defor, etc are abbreviated modes that work for logical
;;; belief functions.


;;; Functions for defining potensials -- defpot
;;; (defpot <name> <frame> [<Doc-string>] { :array <array> | :vacuous
;;;                                         :ps-list {<ps-val-pair>}*})
;;; <name> is a name for the belief function
;;; <Doc-string> is the documentation string for the belief function
;;; <frame> is the list of attributes in the frame of the belief
;;; function
;;; Array is a lisp array whose dimensions should be the same as size
;;; of the frame.  (Use the #nA() format and make the elements
;;; long-float (use L for at least one exponent)).
;;; The ps-list format is meant to provide an interface which is
;;; similar to the ms-list format used by the belief function
;;; routines.  In particular, it creates the potenial by first
;;; building the array of the proper size, then for each ps-val-pair,
;;; adding to the elements of the array corresponding to the members of
;;; the ps-set the value given to that ps-set.  If :vacous is used,
;;; construct a uniform potential.  :nograph keyword is
;;; also accepted with an optional argument which must be literally t
;;; or nil.  :nograph causes defpot to ignore certain checks and not
;;; add information to #?*model-graph* or #?*val-list*.

;; defpot -- constructs the commands for making a belief function and
;; pushes them onto form.  Form is then executed as a progn
(defmacro defpot (name &body bod &aux (form '()) (otherpair nil)
		       array)
  (declare (type Symbol name) (type Array array)
	   (:returns (type Pot potential)))
  "Define a new potential
  (defpot <name> <frame> [<Doc-string>] [:nograph { nil | t}]
				{ :array <array> |
				  {:vacuous | :uniform} [<c>] |
                                  :ps-list {<ps-val-pair>}*})

  Defines a new probability or conditiondal probability distribution
<name> over frame <frame> in potential form.  <Doc-string> is an
optional documentation string.  The optional keyword :nograph
suppresses the addition of <frame> as an edge to #?*model-graph* and
the associated binding of <name> to <frame> in #?*val-list*.

The body of the definition can take one of three forms, the form is
indicated by a keyword:

   :array --- sole remaining argument should be an array whose
diminsions match the frame.  That is the following expression should
be true:
(equal (dimension <array>)
       (mapcar #'(lambda (att) (length (get att :values))) <frame>))


   :vacuous or :uniform --- creates a uniform potential accross all
values.  Value <c> at each outcome defaults to 1.0.

   :ps-list  builds the potential from focal elements as if it were a
Bayesian belief function.  In particular, it creates the potenial by first
building the array of the proper size, then for each ps-val-pair,
adding to the elements of the array corresponding to the members of
the ps-set the value given to that ps-set.  

  "
  (let ((attlist (pop bod)) (nograph nil) )
    (unless (listp attlist)
      (error "defpot:~S is not an attribute list" attlist))
    (when (stringp (car bod))
	  (push `(setf ,`(documentation (quote ,name) 'function) ,(pop bod))
		form))
    (loop 
     (case (car bod)
	   (:nograph (pop bod)
		     (if (or (eq (car bod) t) (eq (car bod) nil))
			 (setq nograph (pop bod))
		       (setq nograph t)))
	   ((:vacuous :uniform) (pop bod)
		     (let ((initial-el (if (and (not (endp bod))
						(get-number (car bod)))
					   (get-number (pop bod))
					 1.0L0)))
		       (setq array
			 (make-array (frame-size (attribute-to-frame attlist))
				     :element-type 'long-float
				     :initial-element initial-el))
		       (if (not (endp bod))
			   (cerror "Ignore unrecognized material"
				   "defpot: Found ~S when expecting end of list"
				   bod)
			 (return))))
	  (:array (pop bod)
		  (setq array (pop bod))
		  (unless (arrayp array) (eval array))
		  (unless (equalp (array-dimensions array)
				  (frame-size (attribute-to-frame attlist)))
			  (error "defpot: Array dimmensions ~S don't match attributes ~S"
				 (array-dimensions array) attlist))
		  (if (not (endp bod))
		      (cerror "Ignore unrecognized material"
			      "defpot: Found ~S when expecting end of list"
			      bod)
		    (return)))
	  (:ps-list (pop bod) (setq array (ps-list-to-array bod attlist))
		    (return))
	  (otherwise (error "defpot: Unrecognized keyword ~S" (car bod)))))
    (push `(setq ,name (make-pot :frame (quote ,attlist)
				 :array (quote ,(float-array array)))) form)
    (unless nograph
	    (if (subsetp attlist (graph-nodes #!*model-graph*))
		(push `(pushnew (quote ,attlist) (graph-edges ,#?*model-graph*)
				:test #'equal) form)
	      (error "defpot:~S contains undefined attributes" attlist))
	    (if (setq otherpair (assoc attlist #!*val-list* :test #'equal))
		(warn "defpot:This edge ~S (~S) shadows edge ~S" name attlist
		      (cdr otherpair)))
	    (push `(set #?*val-list* (acons ',name ',attlist #!*val-list*)) form)))
  `(prog1 ,.form))



;; ps-list-to-array -- converts the ps-list to an array
(defun ps-list-to-array (ps-list attlist)
  (declare (type List ps-list) (type List attlist)
	   (:returns (type (array Long-float) array)))
  "Turns a list of list of (<value>  <ps-set>) pairs (<ps-list>) into
an array by adding <value> to each element of array corresponding to
an outcome in <ps-set>.  <ps-set> is defined over frame <attlist>."

  (if (endp ps-list) (error "defpot: No ps-lists to construct potential."))
  (let ((array (make-array (frame-size (attribute-to-frame attlist))
			   :element-type 'long-float
			   :initial-element 0.0L0)))
    (map nil #'(lambda (pair) (add-ps-set pair attlist array))
	  ps-list)
    array))




;; add-ps-set -- turns each element of the ps-set into an index (using
;; attlist) and adds the weight value to the matrix at all points of
;; the index.
(defun add-ps-set (m-pair attlist array)
  (declare (type List m-pair) (type List attlist)
	   (type (Array Long-Float) array)
	   (:returns (type (Array Long-Float) array)))
  "Adds <m-pair> which should be (<value> <ps-set>) to <array> by
transforming <ps-set> into a list of indexes into <array> and adding
<value> to each corresponding element.  <ps-set> is defined over frame
<attlist>.

Destructively modifies <array> argument.
"
  (multiple-value-bind (m elm)
      (if (get-number (car m-pair))
	  (values (get-number (car m-pair)) (cadr m-pair))
	(if (get-number (cadr m-pair))
	    (values (get-number (cadr m-pair)) (car m-pair))
	  (error "defpot:Bad ps-value-list ~S" m-pair)))
    (update-min-input m)
    (if (check-elm elm attlist)
	(map nil #'(lambda (ind)
		     (incf (elt-array array ind) (long-float m)))
	      (ps-index elm (attribute-to-frame attlist)))
      (error "defpot:Bad ps-set ~S" elm)))
  array)



;; check-elm -- checks a focal element (ps-set) to see if it is valid.
;; returns t if it is valid, does not return (calls error) if it is not.
(defun check-elm (elm attlist)
  (declare (type List attlist)
	   (:returns (type (member T nil))))
  "Checks focal element <ps-set>, defined over frame <attlist> for
validity." 
  (if (listp elm) (every #'(lambda (#1=#:x) (check-atts #1# attlist)) elm)
    (error "defpot:~S is not a ps-set" elm)))

;;check-atts -- checks an single entry in a ps-set for validity
(defun check-atts (sub-elm attlist)
  (declare (type List sub-elm) (type List attlist)
	   (:returns (type (member T nil))))
  "Checks validity of a single tuple (sub-elm) in a PS(TS)-set, over
frame <attlist>."
  (cond ((and (endp sub-elm) (endp attlist)) t)
	((or (endp sub-elm) (endp attlist))
	 (error "defpot:Length mismatch ~S and ~S" sub-elm attlist))
	((listp (car sub-elm))
	 (if (subsetp (car sub-elm) (get (car attlist) :values))
	     (check-atts (cdr sub-elm) (cdr attlist))
	   (error "defpot:Unrecognized values ~S for ~S"
		  (car sub-elm) (car attlist))))
	(t (if (member (car sub-elm) (get (car attlist) :values))
	       (check-atts (cdr sub-elm) (cdr attlist))
	     (error "defpot:Unrecognized value ~S for ~S"
		    (car sub-elm) (car attlist))))))


;;; Logical macroes -- these macroes expand into defpot constructs.
;;; They are of the form:
;;; (def<fn> <name>  <attlist> [<Doc-string>] [<cf>])
;;; <fn> is one of if, and, or, ifor, nand, nor, xor
;;; <name> is the name given to the potential
;;; <Doc-string> documents the potential
;;; <attlist> is the list of attributes and '(not attribute)s effected 
;;; <cf> is an optional confidence factor [Default = 1.0]
;;; it is assumed that all attributes are of the :true-false type
;;; The result is a potential of which the elements corresponding to
;;; the focal element defined by <fn> and attlist have value <cf> and
;;; the complement of that ps-set has value 1-<cf>
;;; The focal element varies with <fn> as follows
;;;  if, ifor -- equivalent to the propositions
;;;   if (and (cdr <attlist>)) then (car attlist),
;;;   if (or (cdr <attlist>)) then (car attlist) 
;;; respectively
;;;  iff, iffor -- same with if and only if
;;; and, or, nand, nor, xor -- equivalent to the propositions
;;;   (<fn> <attlist>)
;;; in each case if (not <attribute>) is used instead of <attribute>
;;; then the logical values of true and false will be switch for that
;;; attribute.  It assumes that (car (get <attribute> :values)) is
;;; logical-true and the (cadr (get <attribute> :values)) is
;;; logical-false.

;; defif -- if (and (cdr attributes)) then (car attributes)
(defmacro defif (name (consequent . conditions)
		 &body bod &aux (doc "") 
				(cf 1.0L0) (nograph nil))
  (declare (type Symbol name) (type List conditions)
	   (:returns (type Pot potential)))
  "(defif <name>  (<consequent> {<condidion>}*) [<Doc>]
						[:nograph { t | nil}] [<cf>])

Defines a new potential representing an if-then relationship,
particularly, <consequent> is true if (and <condition>s) is true.  Each
element of the truth table corresponding to this relationship is
assigned value <cf> (default 1.0) and the complement of the truth
table is assigned value (1-<cf>).

<Doc> is an optional documentation string.

The keyword :nograph t suppresses adding the relationship to
#?*model-graph* and #?*val-list*.

<consequent> and <condition> are generalized attributes.  That is they
can either be (1) an attribute, (2) (<att> . <values>) where <att> is
an attribute and <values> is a list of values to be reguarded as true
or (3) (not <att>) where <att> is another generalized attribute.  
Attributes are passed to #'logical-true and #'logical-false.  Assumes
that (except for case 2) the car of the values list for attribute is
`true'. 

"
  (when (stringp (car bod)) (setq doc (pop bod)))
  (when (eq (car bod) :nograph) (pop bod)
	(if (or (eq (car bod) t) (eq (car bod) nil))
	    (setq nograph (pop bod))
	  (setq nograph t)))
  (when (get-number (car bod)) (setq cf (get-number (car bod))))
  (let ((ps-set (set-up-if consequent conditions))
	(attlist (clean-attribute-list (cons consequent conditions))))
    `(defpot ,name ,attlist ,doc ,@(if nograph '(:nograph t)) :ps-list
       ,(if (eql cf 1.0L0) (list cf ps-set)
	  (list (list cf ps-set)
		(list (- 1.0L0 cf)
		      (ps-set-complement ps-set (attribute-to-frame attlist))))))))



;; defifor -- if (or (cdr attributes)) then (car attributes)
(defmacro defifor (name (consequent . conditions)
		   &body bod &aux (doc "") 
				  (cf 1.0L0) (nograph nil))
  (declare (type Symbol name) (type List conditions)
	   (:returns (type Pot potential)))
  "(defifor <name>  (<consequent> {<condidion>}*) [<Doc>]
						[:nograph { t | nil}] [<cf>])

Defines a new potential representing an if-then relationship,
particularly, <consequent> is true if (or <condition>s) is true.  Each
element of the truth table corresponding to this relationship is
assigned value <cf> (default 1.0) and the complement of the truth
table is assigned value (1-<cf>).

<Doc> is an optional documentation string.

The keyword :nograph t suppresses adding the relationship to
#?*model-graph* and #?*val-list*.

<consequent> and <condition> are generalized attributes.  That is they
can either be (1) an attribute, (2) (<att> . <values>) where <att> is
an attribute and <values> is a list of values to be reguarded as true
or (3) (not <att>) where <att> is another generalized attribute.  
Attributes are passed to #'logical-true and #'logical-false.  Assumes
that (except for case 2) the car of the values list for attribute is
`true'. 

"
  (when (stringp (car bod)) (setq doc (pop bod)))
  (when (eq (car bod) :nograph) (pop bod)
	(if (or (eq (car bod) t) (eq (car bod) nil))
	    (setq nograph (pop bod))
	  (setq nograph t)))
  (when (get-number (car bod)) (setq cf (get-number (car bod))) )
  (let ((ps-set (set-up-ifor consequent conditions))
	(attlist (clean-attribute-list (cons consequent conditions))))
    `(defpot ,name ,attlist ,doc ,@(if nograph '(:nograph t)) :ps-list
       ,(if (eql cf 1.0L0) (list cf ps-set)
	  (list (list cf ps-set)
		(list (- 1.0L0 cf)
		      (ps-set-complement ps-set (attribute-to-frame attlist))))))))



;; defiff -- (and (cdr attributes)) if and only if (car attributes)
(defmacro defiff (name (consequent . conditions)
		  &body bod &aux (doc "") 
				 (cf 1.0L0) (nograph nil))
  (declare (type Symbol name) (type List conditions)
	   (:returns (type Pot potential)))
  "(defiff <name>  (<consequent> {<condidion>}*) [<Doc>]
						[:nograph { t | nil}] [<cf>])

Defines a new potential representing an if-then relationship,
particularly, <consequent> is true if and only if (and <condition>s)
is true.  Each element of the truth table corresponding to this
relationship is assigned value <cf> (default 1.0) and the complement
of the truth table is assigned value (1-<cf>).

<Doc> is an optional documentation string.

The keyword :nograph t suppresses adding the relationship to
#?*model-graph* and #?*val-list*.

<consequent> and <condition> are generalized attributes.  That is they
can either be (1) an attribute, (2) (<att> . <values>) where <att> is
an attribute and <values> is a list of values to be reguarded as true
or (3) (not <att>) where <att> is another generalized attribute.  
Attributes are passed to #'logical-true and #'logical-false.  Assumes
that (except for case 2) the car of the values list for attribute is
`true'. 

"
  (when (stringp (car bod)) (setq doc (pop bod)))
  (when (eq (car bod) :nograph) (pop bod)
	(if (or (eq (car bod) t) (eq (car bod) nil))
	    (setq nograph (pop bod))
	  (setq nograph t)))
  (when (get-number (car bod)) (setq cf (get-number (car bod))) )
  (let ((ps-set (set-up-iff consequent conditions))
	(attlist (clean-attribute-list (cons consequent conditions))))
    `(defpot ,name ,attlist ,doc ,@(if nograph '(:nograph t)) :ps-list
       ,(if (eql cf 1.0L0) (list cf ps-set)
	  (list (list cf ps-set)
		(list (- 1.0L0 cf)
		      (ps-set-complement ps-set (attribute-to-frame attlist))))))))




;; defiffor -- (or (cdr attributes)) if and only if (car attributes)
(defmacro defiffor (name (consequent . conditions)
		    &body bod &aux (doc "") 
				   (cf 1.0L0) (nograph nil))
  (declare (type Symbol name) (type List conditions)
	   (:returns (type Pot potential)))
  "(defiffor <name>  (<consequent> {<condidion>}*) [<Doc>]
						[:nograph { t | nil}] [<cf>])

Defines a new potential representing an if-then relationship,
particularly, <consequent> is true if and only if (or <condition>s) is
true.  Each element of the truth table corresponding to this
relationship is assigned value <cf> (default 1.0) and the complement
of the truth table is assigned value (1-<cf>).

<Doc> is an optional documentation string.

The keyword :nograph t suppresses adding the relationship to
#?*model-graph* and #?*val-list*.

<consequent> and <condition> are generalized attributes.  That is they
can either be (1) an attribute, (2) (<att> . <values>) where <att> is
an attribute and <values> is a list of values to be reguarded as true
or (3) (not <att>) where <att> is another generalized attribute.  
Attributes are passed to #'logical-true and #'logical-false.  Assumes
that (except for case 2) the car of the values list for attribute is
`true'. 

"
  (when (stringp (car bod)) (setq doc (pop bod)))
  (when (eq (car bod) :nograph) (pop bod)
	(if (or (eq (car bod) t) (eq (car bod) nil))
	    (setq nograph (pop bod))
	  (setq nograph t)))
  (when (get-number (car bod)) (setq cf (get-number (car bod))) )
  (let ((ps-set (set-up-iffor consequent conditions))
	(attlist (clean-attribute-list (cons consequent conditions))))
    `(defpot ,name ,attlist ,doc ,@(if nograph '(:nograph t)) :ps-list
       ,(if (eql cf 1.0L0) (list cf ps-set)
	  (list (list cf ps-set)
		(list (- 1.0L0 cf)
		      (ps-set-complement ps-set (attribute-to-frame attlist))))))))


;; defand -- (and attributes)
(defmacro defand (name conditions &body bod &aux (doc "") 
		       (cf 1.0L0) (nograph nil))
  (declare (type Symbol name) (type List conditions)
	   (:returns (type Pot potential)))
  "(defand <name>  ({<condidion>}*) [<Doc>] [:nograph { t | nil}] [<cf>])

Defines a new potential representing a logical assertion:
particularly, all <condition>s are  true.  Each
element of the truth table corresponding to this relationship is
assigned value <cf> (default 1.0) and the complement of the truth
table is assigned value (1-<cf>).

<Doc> is an optional documentation string.

The keyword :nograph t suppresses adding the relationship to
#?*model-graph* and #?*val-list*.

<Condition>s are generalized attributes.  That is they
can either be (1) an attribute, (2) (<att> . <values>) where <att> is
an attribute and <values> is a list of values to be reguarded as true
or (3) (not <att>) where <att> is another generalized attribute.  
Attributes are passed to #'logical-true and #'logical-false.  Assumes
that (except for case 2) the car of the values list for attribute is
`true'. 

"
  (when (stringp (car bod)) (setq doc (pop bod)))
  (when (eq (car bod) :nograph) (pop bod)
	(if (or (eq (car bod) t) (eq (car bod) nil))
	    (setq nograph (pop bod))
	  (setq nograph t)))
  (when (get-number (car bod)) (setq cf (get-number (car bod))) )
  (let ((ps-set (logical-and conditions))
	(attlist (clean-attribute-list  conditions)))
    `(defpot ,name ,attlist ,doc ,@(if nograph '(:nograph t)) :ps-list
       ,(if (eql cf 1.0L0) (list cf ps-set)
	  (list (list cf ps-set)
		(list (- 1.0L0 cf)
		      (ps-set-complement ps-set (attribute-to-frame attlist))))))))


;; defor -- (or attributes)
(defmacro defor (name conditions &body bod &aux (doc "") 
		       (cf 1.0L0) (nograph nil))
  (declare (type Symbol name) (type List conditions)
	   (:returns (type Pot potential)))
  "(defor <name>  ({<condidion>}*) [<Doc>] [:nograph { t | nil}] [<cf>])

Defines a new potential representing a logical assertion:
particularly, at least one <condition>s is true.  Each
element of the truth table corresponding to this relationship is
assigned value <cf> (default 1.0) and the complement of the truth
table is assigned value (1-<cf>).

<Doc> is an optional documentation string.

The keyword :nograph t suppresses adding the relationship to
#?*model-graph* and #?*val-list*.

<Condition>s are generalized attributes.  That is they
can either be (1) an attribute, (2) (<att> . <values>) where <att> is
an attribute and <values> is a list of values to be reguarded as true
or (3) (not <att>) where <att> is another generalized attribute.  
Attributes are passed to #'logical-true and #'logical-false.  Assumes
that (except for case 2) the car of the values list for attribute is
`true'. 

"
  (when (stringp (car bod)) (setq doc (pop bod)))
  (when (eq (car bod) :nograph) (pop bod)
	(if (or (eq (car bod) t) (eq (car bod) nil))
	    (setq nograph (pop bod))
	  (setq nograph t)))
  (when (get-number (car bod)) (setq cf (get-number (car bod))) )
  (let ((ps-set (logical-or conditions))
	(attlist (clean-attribute-list  conditions)))
    `(defpot ,name ,attlist ,doc ,@(if nograph '(:nograph t)) :ps-list
       ,(if (eql cf 1.0L0) (list cf ps-set)
	  (list (list cf ps-set)
		(list (- 1.0L0 cf)
		      (ps-set-complement ps-set (attribute-to-frame attlist))))))))

;; defnand -- (nand attributes)
(defmacro defnand (name conditions &body bod &aux (doc "") 
		       (cf 1.0L0) (nograph nil))
  (declare (type Symbol name) (type List conditions)
	   (:returns (type Pot potential)))
  "(defnand <name>  ({<condidion>}*) [<Doc>] [:nograph { t | nil}] [<cf>])

Defines a new potential representing a logical assertion:
particularly, at least one <condition> is false.  Each
element of the truth table corresponding to this relationship is
assigned value <cf> (default 1.0) and the complement of the truth
table is assigned value (1-<cf>).

<Doc> is an optional documentation string.

The keyword :nograph t suppresses adding the relationship to
#?*model-graph* and #?*val-list*.

<Condition>s are generalized attributes.  That is they
can either be (1) an attribute, (2) (<att> . <values>) where <att> is
an attribute and <values> is a list of values to be reguarded as true
or (3) (not <att>) where <att> is another generalized attribute.  
Attributes are passed to #'logical-true and #'logical-false.  Assumes
that (except for case 2) the car of the values list for attribute is
`true'. 

"
  (when (stringp (car bod)) (setq doc (pop bod)))
  (when (eq (car bod) :nograph) (pop bod)
	(if (or (eq (car bod) t) (eq (car bod) nil))
	    (setq nograph (pop bod))
	  (setq nograph t)))
  (when (get-number (car bod)) (setq cf (get-number (car bod))) )
  (let ((ps-set (logical-nand conditions))
	(attlist (clean-attribute-list  conditions)))
    `(defpot ,name ,attlist ,doc ,@(if nograph '(:nograph t)) :ps-list
       ,(if (eql cf 1.0L0) (list cf ps-set)
	  (list (list cf ps-set)
		(list (- 1.0L0 cf)
		      (ps-set-complement ps-set (attribute-to-frame attlist))))))))


;; defnor -- (nor attributes)
(defmacro defnor (name conditions &body bod &aux (doc "") 
						 (cf 1.0L0) (nograph nil))
  (declare (type Symbol name) (type List conditions)
	   (:returns (type Pot potential)))
  "(defnor <name>  ({<condidion>}*) [<Doc>] [:nograph { t | nil}] [<cf>])

Defines a new potential representing a logical assertion:
particularly, all <condition>s are false.  Each
element of the truth table corresponding to this relationship is
assigned value <cf> (default 1.0) and the complement of the truth
table is assigned value (1-<cf>).

<Doc> is an optional documentation string.

The keyword :nograph t suppresses adding the relationship to
#?*model-graph* and #?*val-list*.

<Condition>s are generalized attributes.  That is they
can either be (1) an attribute, (2) (<att> . <values>) where <att> is
an attribute and <values> is a list of values to be reguarded as true
or (3) (not <att>) where <att> is another generalized attribute.  
Attributes are passed to #'logical-true and #'logical-false.  Assumes
that (except for case 2) the car of the values list for attribute is
`true'. 

"
  (when (stringp (car bod)) (setq doc (pop bod)))
  (when (eq (car bod) :nograph) (pop bod)
	(if (or (eq (car bod) t) (eq (car bod) nil))
	    (setq nograph (pop bod))
	  (setq nograph t)))
  (when (get-number (car bod)) (setq cf (get-number (car bod))) )
  (let ((ps-set (logical-nor conditions))
	(attlist (clean-attribute-list  conditions)))
    `(defpot ,name ,attlist ,doc ,@(if nograph '(:nograph t)) :ps-list
       ,(if (eql cf 1.0L0) (list cf ps-set)
	  (list (list cf ps-set)
		(list (- 1.0L0 cf)
		      (ps-set-complement ps-set (attribute-to-frame attlist))))))))


;; defxor -- (xor attributes)
(defmacro defxor (name conditions &body bod &aux (doc "") 
						 (cf 1.0L0) (nograph nil))
  (declare (type Symbol name) (type List conditions)
	   (:returns (type Pot potential)))
    "(defxor <name>  ({<condidion>}*) [<Doc>] [:nograph { t | nil}] [<cf>])

Defines a new potential representing a logical assertion:
particularly, exactly one <condition> is true.  Each
element of the truth table corresponding to this relationship is
assigned value <cf> (default 1.0) and the complement of the truth
table is assigned value (1-<cf>).

<Doc> is an optional documentation string.

The keyword :nograph t suppresses adding the relationship to
#?*model-graph* and #?*val-list*.

<Condition>s are generalized attributes.  That is they
can either be (1) an attribute, (2) (<att> . <values>) where <att> is
an attribute and <values> is a list of values to be reguarded as true
or (3) (not <att>) where <att> is another generalized attribute.  
Attributes are passed to #'logical-true and #'logical-false.  Assumes
that (except for case 2) the car of the values list for attribute is
`true'. 

"

  (when (stringp (car bod)) (setq doc (pop bod)))
  (when (eq (car bod) :nograph) (pop bod)
	(if (or (eq (car bod) t) (eq (car bod) nil))
	    (setq nograph (pop bod))
	  (setq nograph t)))
  (when (get-number (car bod)) (setq cf (get-number (car bod))) )
  (let ((ps-set (logical-xor conditions))
	(attlist (clean-attribute-list  conditions)))
    `(defpot ,name ,attlist ,doc ,@(if nograph '(:nograph t)) :ps-list
       ,(if (eql cf 1.0L0) (list cf ps-set)
	  (list (list cf ps-set)
		(list (- 1.0L0 cf)
		      (ps-set-complement ps-set (attribute-to-frame attlist))))))))



;;; defis -- this defines a potential on a single attribute
;;; (defis <name> (<attribute>) [<Doc>] {[:T <cf-true>] | [:F <cf-false>]}*)
;;; The keyword :T indicates evidence in favour of the proposition <cf-true> 
;;; The keyword :F indicates evidence for the proposition attribute is
;;; false of weight <cf-false>.  If only one of :T or :F is supplied,
;;; the remaining one is computed from <cf-true> + <cf-false> = 1.  If
;;; neither is supplied, a uniform potential is created using
;;; <cf-true>=<cf-false>=1.  
(defmacro defis (name (attribute) &rest keylist &aux (doc "")
		       (cf-true nil) (cf-false nil) (m-form nil)
		       (nograph nil) )
  (declare (type Symbol name) (type Symbol attribute)
	   (:returns (type Pot potential)))
  "(defis <name> (<attribute>) [<Doc>]
			{[:T <cf-true>] | [:F <cf-false>] |
			 [:nograph {t | nil}}*)


   Defines a potential <name> over a binary attribute <attribute>.  
<cf-true> is the probability assigned to true (car (get <attribute> :values))
<cf-false> is the probablity assigned to false.  If only one of :T and
:F is supplied, then the remaining value is calculated from <cf-true>
+ <cf-false> = 1.  If neither is supplied, a uniform potential is
created using <cf-true>=<cf-false>=1.  

<Doc> is an optional documentation string.

The keyword :nograph t suppresses adding the relationship to
#?*model-graph* and #?*val-list*.

"
  (when (stringp (car keylist)) (setq doc (pop keylist)))
  (loop (when (endp keylist) (return))
	(case (pop keylist)
	      (:t (if (get-number (car keylist))
		      (setq cf-true (get-number (pop keylist)))
		    (error "defis: expected a number, got ~S" (car keylist))))
	      (:f (if (get-number (car keylist))
		      (setq cf-false (get-number (pop keylist)))
		    (error "defis: expected a number, got ~S" (car keylist))))
	      (:nograph (if (or (eq (car keylist) t) (eq (car keylist) nil))
			    (setq nograph (pop keylist))
			  (setq nograph t)))
	      (t (error "defis: unrecognized matrial ~S" keylist))))
  (cond ((and (null cf-true) (null cf-false))
	 (setq cf-true 1.0L0 cf-false 1.0L0))
	((null cf-true) (setq cf-true (- 1.0L0 cf-false)))
	((null cf-false) (setq cf-false (- 1.0L0 cf-true)))
	((not (eql 1.0L0 (+ cf-true cf-false)))
	 (warn "defis: :T (~S) and :F(~S)  do not sum to 1.0 in ~S"
	       cf-true cf-false name)))
  (push (list cf-true (list (list (logical-true attribute)))) m-form)
  (push (list cf-false (list (list (logical-false attribute))))	m-form)
  `(defpot ,name (,(clean-attribute attribute)) ,doc
     ,@(if nograph '(:nograph t)) :ps-list ,.m-form))



;;; defpotcond -- This function constructs potentials using the
;;; conditional embedding method.  
(defmacro defpotcond (name &body bod &aux (form '()) opt)
  (declare (type Symbol name)
	   (:returns (type Pot potential)))
  "Define a new potential representing a conditional probability
distribution. 

  (defpotcond <name> <frame> [<Doc-string>] [:nograph {t | nil}]
				<option> {<component>}*)

  where <frame> is ( {<consequences>}* :given {<conditions>}*)

  <option> is one of :array, :array-list (equivalent to :array) 
  or :ps-list <option>.

   and <component> is (<condition ps> {<value>}) 
  where <condition ps> is a <ps-set> defined over the <condition>s and
<value> is either an array (option :array) or multiple <ps-val-pairs>
(like a belief function specificiation) where each <ps-set> is defined
over the <consequences> frame.

<Doc-String> is an optional documentation string.

The keyword :nograph t suppresses adding the relationship to
#?*model-graph* and #?*val-list*.

" 
  (let ((fullattlist (pop bod)) (nograph nil) )
    (if (not (listp fullattlist))
	(error "defpot:~S is not an attribute list" fullattlist))
    (let* ((dependents (ldiff fullattlist (member :given fullattlist)))
	   (independents (cdr (member :given fullattlist)))
	   (attlist (append independents dependents)))
      (when (stringp (car bod))
	    (push `(setf ,`(documentation (quote ,name) 'function) ,(pop bod))
		  form))
      (loop (when (endp bod)
		  (error "defpotcond: ~S ran out of data before complete" 
			 name))
	    (case (setq opt (pop bod))
		  (:nograph (if (or (null (car bod)) (eq t (car bod)))
				(setq nograph (pop bod))
			      (setq nograph t)))
		  ((:array :array-list) (map nil #'(lambda (cond-arr)
						     (unless (arrayp (cadr cond-arr))
							     (setf (cadr cond-arr)
								   (eval (cadr cond-arr)))))
					     bod)
		   			(return))
		  (:ps-list (map nil #'(lambda (cond-ms)
					 (setf (cdr cond-ms)
					       (list (ps-list-to-array (cdr cond-ms) dependents))))
				 bod)
			    (return))
		  (otherwise (error "defpotcond: Unrecognized keyword ~S" opt))))
      (push `(setq ,name (make-pot :frame (quote ,attlist)
				   :array (quote ,(float-array
						   (embed-array
						    independents
						    dependents bod))))) form)
      (unless nograph 
	      (if (subsetp attlist (graph-nodes #!*model-graph*))
		  (push `(pushnew (quote ,attlist) (graph-edges ,#?*model-graph*)
				  :test #'equal) form)
		(error "defpot:~S contains undefined attributes" attlist))
	      (push `(set #?*val-list* (acons ',name ',attlist #!*val-list*)) form))))
  `(prog1 ,.form))


;;embed-array -- does error checking on list of ps-set (condition)
;;array pairs.  It then transforms this list of pairs into a (possibly
;;longer list) with one tuple of independents paired with one array.
;;It then transforms this into an array suitable for make-pot
(defun embed-array (independents dependents shortlist)
  (declare (type List dependents independents)
	   (type list shortlist)
	   (:returns (type (Array Long-Float) embeded-array)))
  "This builds an array over (append <dependents> <independents>)
from <shortlist> of (<ps-set> <conditional-array>)
defined where the <ps-set> is definded over <indpendents> and the
<conditional-array> is defined over <dependenets>."
  (let ((array-list
	 (mapcan #'(lambda (cond-array)
		     (check-elm (car cond-array) independents)
		     (mapcar #'(lambda (condx) (cons condx (cdr cond-array)))
			     (explode-ps-set (car cond-array))))
		 shortlist))) ; this creates a longer array with tuples
			    ; instead of ps-sets as the condition index
    (multiple-value-bind
	 (res errorval erroratts errtype)
	 (catch :embed (embed-array-aux independents dependents array-list))
	 (unless res
		 (ecase errtype
		       (:num (error "defpotcond: Wrong number of conditions for ~S of ~S"
				    errorval erroratts))
		       (:extra (error "defpotcond: Extra value for ~S in ~S"
				      errorval erroratts))
		       (:dims (error "defpotcond: Bad dimesions for array ~S in ~S"
				     errorval erroratts))))
	 (as-array
	  (frame-size (attribute-to-frame (append independents dependents)))
	  res
	  'long-float))))



;; embed-array-aux -- takes a list of condition array pairs, one for each
;; possible value of independents, and returns a large array which has
;; each of the  smaller arrays as the appropriately indexed sub array.
;; If an improper number of sub-arrays are found, nil is returned as
;; the result, as well as two values describing the index and the list
;; of indexies with which the problem occured.  Errors are thrown to
;; the :embed catch on the previous level
(defun embed-array-aux (independent dependent array-list)
  (declare (list dependent) (list independent) (list array-list))
  (when (endp independent)
	(let ((arr (rescale-array (cadar array-list))))
	  (unless (equal (array-dimensions arr)
			 (frame-size (attribute-to-frame dependent)))
		  (throw :embed (values nil nil nil :dims)))
	  (throw :embed (values (coerce (as-vector arr) '(vector long-float)) nil nil nil))))
  (let* ((first-var (car independent))
	 (by-first-var
	  (mapcar #'(lambda (xi) (remove xi array-list :test-not #'equal
					 :key #'caar))
		  (get first-var :values))))
    (unless (eql (length array-list)
		 (reduce #'+ (mapcar #'length by-first-var)))
	    (throw :embed (values nil nil first-var :extra)))
    (let ((mval (position-if #'null by-first-var)))
      (if mval
	  (throw :embed (values nil (elt (the list (get first-var :values)) mval)
				first-var :num)))
      (values
	(apply #'concatenate '(vector long-float)
	       (mapcar #'(lambda (sublist)
			   (multiple-value-bind
			    (res errvals erratts errtype)
			    (catch :embed
			      (embed-array-aux (cdr independent)
					       dependent
					       (mapcar #'(lambda (#1=#:x)
							   (cons (cdar #1#) 
								 (cdr #1#)))
						       sublist)))
			    (unless res
				    (throw :embed
					   (values nil
						   (cons (caaar sublist) errvals)
						   (cons first-var erratts)
						   errtype)))
			    (as-vector res)))
		       by-first-var))
	nil nil nil))))

			       

;;; special function for dealing with minimum-inputs
(declaim (inline update-min-input))
(defun update-min-input (m)
  (declare (type Number m)
	   (:returns (type Number #?*min-input*)))
  "Updates the value of #?*min-input* by taking minimum with <m>."
  (set #?*min-input* (min #!*min-input* m)))


;;; provide when loaded
(bel-provide :probread)