;;; potentials.lisp -- basic potential function arithimatic functions

;;; Copyright 1989 Russell G. Almond
;;; License is granted to copy this program for education or research
;;; purposes, with the restriction that no portion of this program may
;;; be copied without also copying this notice.  All other rights
;;; reserved. 

;;; 2/15/89 -- Created file from arithmatic.lisp
;;; 8/5/89 -- Version 1.1
;;;	* Maximizing version of projection[d]
;;;	  new keyword (:op #'+) added to @v & @-> (& @^) defaults to
;;;	  #'+ but can be set to #'max
;;;	* Mixture operator[d]
;;; 8/4/89 -- added copy-chage-frame utility to copy a val while
;;; changing its frame.

;;; 2/25/92 --- Version 1.2 cleaned up documentation.

;(provide 'potentials)
(in-package :prob)
(bel-require :structures "structures")
(bel-require :sets "sets")
(bel-require :lowread "lowread")
(bel-require :utils "utils")
;(use-package 'basic)
;(use-package 'sets)
;(use-package 'utils)
;(export '(@^ @v @-> @+ @+2 || -><- ppval make-vacuous
;	     direct-sum constant-pot normalize-pot
;	     approx-zerop pppot node-size reset-zero-tol
;	     copy-change-frame mix))
;;; the following have initial values defined here, the real constants
;;; are avaliable only in the *rules-package* (through #? and #!)
;;;  *min-input* *condition-ratio* *att-penalty* *zero-tol*

;;;; minimal extension

;;; @^ -- minimal extension -- extend pot so that its frame is
;;; new-frame.  (note: this also permutes the orders of the variables.)
(defun @^ (new-frame pot &key (op #'+))
  (declare (type List new-frame) (type Pot pot)
	   (type (Function (Number Number) Number) op)
	   (:returns (type Pot extended-pot)))
  "(prob:@^ <new-frame> <pot> &key (op #'+)) extends <pot> to cover
<new-frame>.   <New-frame> must contain the frame of the potential <pot>.
The operator <op> (defaults to #'+) is used to sumarize over margins.
In this case it is essentially ignored, but is included for compatabilitiy."
  (cond ((not (subsetp (pot-frame pot) new-frame))
	 (error "prob:@^: illeagal frame ~S.  Try using prob:@->." new-frame))
	((equalp new-frame (pot-frame pot)) pot)
	(t (make-pot
	    :frame new-frame
	    :array (make-array (frame-size (attribute-to-frame new-frame))
			       :element-type 'long-float
			       :initial-contents (array-constr-extend
						  new-frame 
						  (pot-frame pot)
						  pot))))))


;;; array-constr-extend -- 
(defun array-constr-extend (undone-frame old-frame-index pot)
  (declare (type List undone-frame) (type List old-frame-index)
	   (type Pot pot)
	   (:returns (type List extended-pot-array-contents)))

  "Creates and new array by extending and repeating the old arrays.  At
each level of decent, if the new attribute is in the old frame,
substitute each value in turn for the attribute in the
old-frame-index.  At the base case, old-frame-index will then be an
index into the array of potential, return that value.  If the
attribute is not on the list, then just repeat one value for each new
attribute. "

  (if (endp undone-frame)
      (elt-array (pot-array pot)
		 (car (ps-index (list old-frame-index)
			   (attribute-to-frame (pot-frame pot)))))
    (let* ((att (car undone-frame))
	   (ind (position att old-frame-index))
	   (new-frame-index (copy-seq old-frame-index)))
      (declare (list new-frame-index))
      (if ind
	  (mapcar #'(lambda (val)
		      (setf (elt new-frame-index ind) val)
		      (array-constr-extend (cdr undone-frame)
					   new-frame-index
					   pot))
		  (get att :values))
	(mapcar #'(lambda (val)
		    (array-constr-extend (cdr undone-frame)
					 new-frame-index
					 pot))
		(get att :values))))))


      



;;;; Marginalization

;;; @v -- marginalization.  Finds the margin of pot in the new frame.
;;; This routine is optimized for the case when new-frame is a subset
;;; of the frame of the potential.
(defun @v (new-frame pot &key (op #'+))
  (declare (type List new-frame) (type Pot pot)
	   (type (Function (Number Number) Number) op)
	   (:returns (type Pot marginal-pot)))
  "(prob:@v <new-frame> <pot> &key <op>) Margin of <pot> on
<new-frame>.   Assumes that <new-frame> is a subset of frame of <pot>.
<op> (default #'+) is used to sumarize over repeated elements.  #'max
gives poterior mode instead of margin as the effect."
  (cond ((not (subsetp new-frame (pot-frame pot)))
	 (error "prob:@v : Illegal new-frame ~S.  Use prob:@->"))
	(t (make-pot
	    :frame new-frame
	    :array (make-array (frame-size (attribute-to-frame new-frame))
			       :element-type 'long-float
			       :initial-contents (array-constr-marg
						  new-frame 
						  (pot-frame pot)
						  pot
						  :op op))))))



;;; array-constr-marg -- 
(defun array-constr-marg (undone-frame old-frame-index pot &key (op #'+))
  (declare (type List undone-frame) (type List old-frame-index)
	   (type (Function (Number Number) Number) op)
	   (type Pot pot)
	   (:returns (type List new-pot-array-contents)))

"Creates and new array by summing over the old arrays.  For each
attribute of the old frame which is not in the new frame, substitute
the list of possible values for the attribute.  Then pass the
resulting list to array-constr-marg-aux which will calculate the
result by recursive decent.  At each level of decent, each value of
the first attribute is substituted for the attribute name in the
old-frame-index.  At the base case, old-frame-index will then be a a
ps-set describing all the array elements which must be summed over to
get the element for that postition in the new array."

  (let ((new-frame-index (copy-seq old-frame-index)))
    (map nil #'(lambda (att)
		 (setq new-frame-index
		       (nsubstitute (get att :values) att new-frame-index)))
	  (set-difference old-frame-index undone-frame))
    (array-constr-marg-aux undone-frame new-frame-index pot :op op)))


(defun array-constr-marg-aux (undone-frame old-frame-index pot &key (op #'+))
  (declare (type List undone-frame) (type List old-frame-index)
	   (type (Function (Number Number) Number) op)
	   (type Pot pot)
	   (:returns (type List)))
  (if (endp undone-frame)
      (reduce op
	      (mapcar #'(lambda (index) (elt-array (pot-array pot) index))
		      (ps-index (list old-frame-index)
				(attribute-to-frame (pot-frame pot)))))
    (let* ((att (car undone-frame))
	   (ind (position att old-frame-index))
	   (new-frame-index (copy-seq old-frame-index)))
      (declare (list new-frame-index))
      (mapcar #'(lambda (val)
		  (setf (elt new-frame-index ind) val)
		  (array-constr-marg-aux (cdr undone-frame)
				       new-frame-index
				       pot :op op))
	      (get att :values)))))






;;;; projection

;;; @-> -- projection  -- this projects pot onto a new frame.  It
;;; is like (prob:@v new-frame
;;;                 (prob:@^ (union new-frame (b-f-frame pot))
;;;                         pot)) 
;;; but more efficient.
;;; added :op keyword so that maximizing projection (Pearls
;;; explanation stuff) could be used
(defun @-> (new-frame pot &key (op #'+))
  (declare (type List new-frame) (type Pot pot)
	   (type (Function (Number Number) Number) op)
	   (:returns (type Pot projected-pot))
	   )
  "(prob:@-> <new-frame> <pot> &key <op>) coerces <pot> to be over the
<new-frame> by uniform extension and marginalization.  Uses <op>
(default #'+) to summarize over values marginalized out.  Also known
to have an interesting effect is #'max."
  (make-pot
   :frame new-frame
   :array (make-array (frame-size (attribute-to-frame new-frame))
		      :element-type 'long-float
		      :initial-contents (array-constr-proj
					 new-frame 
					 (pot-frame pot)
					 pot :op op))))



;;; array-constr-proj -- 
(defun array-constr-proj (undone-frame old-frame-index pot
				       &key (op #'+))
  (declare (type List undone-frame) (type List old-frame-index)
	   (type (Function (Number Number) Number) op)
	   (type Pot pot)
	   (:returns (type List projected-array-contents)))

  "creates and new array by summing over
the old arrays.  For each attribute of the old frame which is not
in the new frame, substitute the list of possible values for the
attribute.  Then pass the resulting list to array-constr-proj-aux
which will calculate the result by recursive decent.
At each level of decent,  if the first attribute on the new
attribute list is in the old-attribute list, then each value of
the first attribute is substituted for the attribute name in the
old-frame-index, otherwise the value is repeated for each
instantiation of the new attribute.  At the base case, 
old-frame-index will then be a a ps-set describing all the array
elements which must be summed over to get the element for that
postition in the new array."

  (let ((new-frame-index (copy-seq old-frame-index)))
    (map nil #'(lambda (att)
		 (setq new-frame-index
		       (nsubstitute (get att :values) att new-frame-index)))
	    (set-difference old-frame-index undone-frame))
    (array-constr-proj-aux undone-frame new-frame-index pot :op op)))



(defun array-constr-proj-aux (undone-frame old-frame-index pot
					   &key (op #'+))
  (declare (type List undone-frame) (type List old-frame-index)
	   (type (Function (Number Number) Number) op)
	   (type Pot pot) (:returns (type List)))
  (if (endp undone-frame)
      (reduce op
	      (mapcar #'(lambda (index) (elt-array (pot-array pot) index))
		      (ps-index (list old-frame-index)
				(attribute-to-frame (pot-frame pot)))))
    (let* ((att (car undone-frame))
	   (ind (position att old-frame-index))
	   (new-frame-index (copy-seq old-frame-index)))
      (declare (list new-frame-index))
      (if ind 
	  (mapcar #'(lambda (val)
		      (setf (elt new-frame-index ind) val)
		      (array-constr-proj-aux (cdr undone-frame)
				       new-frame-index
				       pot :op op))
	      (get att :values))
	(mapcar #'(lambda (val)
		    (array-constr-proj-aux (cdr undone-frame)
					 new-frame-index
					 pot :op op))
		(get att :values))))))




	
;;;; Direct sums

;;; direct-sum -- takes the direct sum of two potentials over
;;; different frames.  The resulting frame is the union of the two
;;; frames. 
(defun direct-sum (prob1 prob2 &key (op #'+))
  (declare (type Pot prob1 prob2)
	   (type (Function (Number Number) Number) op)
	   (:returns (type Pot prob1*prob2))
	   )
  "Direct sum (convolution) of <prob1> and <prob2> which may be defined
over different frames." 
  (let ((new-nodes (union (pot-frame prob1)
			      (pot-frame prob2))))
    (@+ (@^ new-nodes prob1 :op op) (@^ new-nodes prob2 :op op) :op op)))

;;; direct-sum-operator-unnormalized -- takes the direct sum of two potentials
;;;  over the same frame, unnormalized
(defun @+2 (prob1 prob2 &key (op #'+))
  (declare (type Pot prob1 prob2)
	   (type (Function (Number Number) Number) op)
	   (ignore op)
	   (:returns (type Pot prob1*prob2))
	   )
  "Direct sum (convolution) of <prob1> and <prob2> over the same frame.
<op> is a summary operator, allowed with compatability with belief
function version."
  (if (not (equalp (pot-frame prob1)
		   (pot-frame prob2)))
      (error "prob:@+ : potentials do not have same frame.")
    (make-pot
	 :frame (pot-frame prob1)
	 :array (prod-array (pot-array prob1) (pot-array prob2)))))


;; prod-array -- this function find the products of two arrays. 
(defun prod-array (arr1 arr2)
  (declare (type (Array long-float) arr1 arr2)
	   (:returns (type (Array Long-Float) arr1*arr2)))
  "Elementwise multiplication of arrays <arr1> and <arr2> which are
assumed to be the same total length.  Result will have the same shape as
<arr1>.  

Programmers note: There are probably much faster implentation than this
one, however, to maintain compatability with VAX Common Lisp which
adheres to the letter rather than the spirit of the coercian rules for
mapping,  we have have this rather awkward form."

  (as-array (array-dimensions arr1)
	    (coerce			;Damn Dec Anyway!  (map
					;'(vector long-float) returns
					;a '(vector t) in VAX lisp,
					;this kludge avoids that.
	     (map 'list #'* (as-vector arr1) (as-vector arr2))
	     '(vector long-float))
	    'long-float))




;;; direct-sum-operator-normalized -- takes the direct sum of two 
;;; potentials over the same frame.
(defun @+ (prob1 prob2  &key (op #'+))
  (declare (type Pot prob1 prob2)
	   (type (Function (Number Number) Number) op)
	   (:returns (type Pot |prob1*prob2|))
	   )
  "Normalized direct sum (convolution) of <prob1> and <prob2> which are
defined over the same frame.  Uses <op> as summary operator." 
  (|| (@+2 prob1 prob2 :op op)))


;;; normalize-pot -- this normalizes a potential
(defun normalize-pot (prob1)
  (declare (type Pot prob1)
	   (:returns (type Pot |prob1|)))
  "Normalizes (divides by the sum) <prob1>."
  (make-pot
   :frame (pot-frame prob1)
   :array (rescale-array
	   (pot-array prob1))))

;;; rescale-array -- divides all elements of the array by the
;;; normalizing constant (sum of the elements)
(defun rescale-array (arr)
  (declare (type (Array Long-float) arr)
	   (:returns (type (Array Long-float) arr/c)))
  "Normalizes array <arr> by dividing all elements by the sum of all
elements. 

Programmers note:  Again the method is very crude and clunky to allow
for implementation difference between certain lisps.
"

  (let* ((vec (as-vector arr))
	 (k (reduce #'+ vec)))
    (declare (float k))
    (when (eql k 0)
	(error "rescale-array: Divide by zero"))
    (as-array (array-dimensions arr)
	      (coerce			;Damn DEC Anyway!
	       (map 'list #'(lambda (#1=#:x) (coerce (/ #1# k)
						     'long-float)) vec)
					;Damn Franz, Inc Anyway!
	       '(vector long-float))
	      'long-float)))

	 

;;; constant-pot -- returns the normalizing constant associated
;;; with an improper  potential.
(defun constant-pot (prob)
  (declare (type Pot prob)
	   (:returns (type Long-float c)))
  "Calculates nomalizing constand for potential <prob>."
  (reduce #'+ (as-vector (pot-array prob))))


;;; approx-zerop -- decides if the value is approximately equal to
;;; zero and if so returns t.
(defun approx-zerop (#1=#:x)
  (declare (long-float #1#)
	   (:returns (type (member T nil))))
  "Checks to see if <x> is sufficiently close to zero (less than
#!*zero-tol*) to be considered zero."
  (if (< (abs #1#) (the float #!*zero-tol*))
      t nil))				;Hi tippit!

      

;;; *zero-tol* -- number which anything smaller than this should be
;;; taken as zero.  
(defvar *zero-tol* long-float-epsilon  "Any number smaller than this
quantiy should be taken as zero.")

;;; *min-input* -- This variable is the minimum m-value input in any
;;; belief function
(defvar *min-input* .01  "Minimum value we have seen so far.")

;;; *condition-ratio* -- The condition ratio divided by the min-input
;;; is the smallest value that will be considered non-zero
(defvar *condition-ratio* 1.0E6
  "#?*zero-tol* = #!*min-input*/#!*condition-ratio*")

(defun reset-zero-tol ()
  (declare (:returns (type Long-Float #!*zero-tol*)))
  " Resets <#?*zero-tol*>.
#?*zero-tol* = #!*min-input*/#!*condition-ratio*"
  (set #?*zero-tol* (the long-float
		      (max (/ (the long-float #!*min-input*)
			      (the long-float #!*condition-ratio*))
			   long-float-epsilon))))


;;; Things moved over from computations because they really belong in
;;; package prob

;;; pppot -- prints a potential in a human readable form
(defun pppot (pot &optional (stream t))
  (declare (type Pot pot) (type Stream stream)
	   (:returns nil))
  "Pretty prints potential <pot> on <stream> (default t)."
  (format stream "~%Potential over frame:~S~%"
	  (pot-frame pot))
  (pp-labeled-array (attribute-to-frame (pot-frame pot))
		    (pot-array pot) stream 0 #\()
  (terpri stream))

;;; ugpot -- ugly prints potential
(defun ugpot (pot &optional (stream t))
  (declare (type Pot pot) (type Stream stream)
	   (:returns nil))
  "Ugly prints potential <pot> on <stream> (default t).  (faster and more
compact representation using default print method for arrays.)"
  (format stream "~%Potential over frame:~S~%"
	  (pot-frame pot))
  (map 'nil #'(lambda (v) (show-variable v stream)) (pot-frame pot))
  (print (pot-array pot) stream)
  (terpri stream))

;;; pp-labeled-array -- designed specificly as a recursive auxilary
;;; for printing potentials, but can be used to format/print arrays of
;;; number with labels for dimension.  labs is a list of labels for
;;; each dimension, its length must be equal to the rank of the array,
;;; and the length of each element must be equal to the corresponding
;;; dimension of the array.  tab gives the level of indentation the
;;; printing is currently at and pchar gives the opening parenthesis
;;; character which should be one of #\( or #\[.
(defun pp-labeled-array (labs arr &optional (stream t)
					    (tab 0) (pchar #\())
  (declare (type List labs) (type (Array Float) arr)
	   (type Fixnum tab) (type Character pchar)
	   (type Stream stream)
	   (:returns nil))
  "Pretty prints array <arr> with labels <labs>.  <labs> should be a
list of lists whose lengths match (dimension <arr>).  <stream> is
optional stream.  

<tab> and <pchar> control indentation and alternating
parenthesis characters (should be #\\( or #\\[) and are primarily for
recursive calls, but also could be used in other array formatting
commands."
  (if (> tab 70)
      (setq tab 0 pchar (if (eql pchar #\() #\[ #\())
      ;line to long, wrap and change brackets
    )
  (if (endp labs) (return-from pp-labeled-array nil))
  (let* ((level-labs (car labs))
	 (rest-labs (cdr labs))
	 (fstring (if (endp rest-labs)
		      "~21,1,1<~*~:;~S~;~16,9G~>"
		    "~*~4,2@<~S~> "))
	 (maxlen (apply #'max
			(mapcar #'(lambda (item)
				    (length (the simple-string
						 (format nil " ~4,2@<~S~> " item))))
				level-labs)))
	 (sub-dims (unless (endp rest-labs)
			   (frame-size rest-labs)))
	 (sub-tot-size (reduce #'* sub-dims))
	 (printer (if (endp rest-labs)
		      #'(lambda (num)
			  (format stream fstring tab
				  (elt level-labs num)
				  (elt arr num))
			  (setq fstring "~22,1,1<~%~v,1T~:; ~S~;~16,9G~>"))
		    #'(lambda (num)
			(format stream fstring tab (elt level-labs num))
			(setq fstring "~%~vT ~4,2@<~S~> ")
			(pp-labeled-array rest-labs
					  (make-array sub-dims
						      :element-type
						      (array-element-type arr)
						      :displaced-to arr
						      :displaced-index-offset
						      (* sub-tot-size num))
					  stream
					  (the fixnum (+ tab maxlen)) pchar)))))
    (declare (list level-labs) (fixnum maxlen sub-tot-size)
	     (list sub-dims))
    (format stream "~C" pchar)
    (mapcar printer (iota (the fixnum (1- (length level-labs)))))
    (format stream "~C" (if (eql pchar #\() #\) #\]))))



(defun ppval (pot &optional (stream t))
  (declare (type Pot pot) (type Stream stream)
	   (:returns nil))
  "Pretty (or ugly if Allegro CL) prints potential <pot> on <stream>
(default t)"
  #-excl(pppot pot stream)
  #+excl(ugpot pot stream)) 
					;use ugpot until I get a bug
					;fix from Franz
(defun || (pot)
  (declare (type Pot pot)
	   (:returns (type Pot |pot|)))
  "Normalizes (divide by sum of values) potential <pot>."
  (normalize-pot pot))

(defun -><- (pot)
  (declare (type Pot pot)
	   (:returns (type Long-Float c)))
  "Calculates normalizing constant for <pot>."
  (constant-pot pot))

;;; node-size -- This caluclates the size of a node, where a node is a
;;; list of attributes
(defun node-size (node)
  (declare (type List node)
	   (:returns (type Number size)))
  "Calculates the size (penalty function) of a frame of discernment.
Used in builing a tree of cliques."
  (reduce #'*
	  (mapcar #'(lambda (att) (+ (length (the list (get att :values))) #!*att-penalty*))
		  node)))

(defvar *att-penalty* 0 "Penalty for more attributes in a node")	  

(defun make-vacuous (frame)
  (declare (type List frame)
	   (:returns (type Pot uniform)))
  "Makes a uniform potential over <frame>."
  (make-pot :frame frame
	     :array (make-array (frame-size (attribute-to-frame frame))
				:initial-element 1.0L0
				:element-type 'long-float)))







;;; copy-change-frame -- copies a potential structure and chages
;;; its name at the same time.
(defun copy-change-frame (frame pott)
  (declare (type List frame) (type Pot pott)
	   (:returns (type Pot new-pott)))
  "Fast copy of potential <pott>, renaming frame to <frame>.

Warning:: this is meant to be a highly efficient way of cloning
essentially similar potentials.  It assumes that the frame of
<pott> and the new <frame> differ only in the names of the
attributes.  That is, there must be a one to one correspondence between
the attributes of <pott> and <frame> such that each differs only by
name and not in the possible values for the outcome space."
  (let ((res (copy-pot pott)))
    (setf (pot-frame res) frame)
    res))



;;; mix -- forms a weighted mixture of several potentials
;;; Note: as this uses mapcan to do its basic stuff, if the two lists
;;; are of unequal length, the excess elements of the longer one will
;;; be ignored.
(defun mix (list-pots list-weights)
  (declare (type (List Pot) list-pots)
	   (type (List Number) list-weights)
	   (:returns (type Pot average-pot)))
  "Produces a new potential by taking a weighted (using weights
<list-weights>) average over the values of the potentials in <list-pots>."
  (when (find (val-frame (car list-pots))
	      (cdr list-pots) :key #'val-frame
	      :test-not #'equal)
	(error "mix:All values must have the same frame"))
  (make-pot :frame (val-frame (car list-pots))
	    :array (linear-comb (mapcar #'pot-array list-pots)
				list-weights)))



;;; provide when loaded
(bel-provide :potentials)