;;; monte.lisp -- Monte Carlo Simulation Junk.

;;; Copyright 1989 Russell G. Almond
;;; License is granted to copy this program for education or research
;;; purposes, with the restriction that no portion of this program may
;;; be copied without also copying this notice.  All other rights
;;; reserved. 

;;; 7/18/89 Created (version 1.1)
;;;	* Allows for MC looping[d]
;;;	* Allows for drawing from inputs[d]
;;;	* Allows printing sink nodes during output[d]
;;;	* Allows drawing sample configuration in place of draws[todo]

;;; 2/25/92 Version 1.2 clean-up documentation.  Separated
;;; *save-messages* function into #!*save-messages* #!*save-values*
;;; and #!*save-sink-values* and made them mode dependent.

;(provide 'monte)
(in-package :shell )
(bel-require :computations "computations")
(bel-require :machine-utils "bymachine" )
;(use-package '(subshell utils))
;(export '(mc-loop mc-up-loop mc-sim-loop *sink-store-function*
;		  simulate-down draw-sources record-sinks 
;		  *sink-stream* open-sink-file mark-sink-file
;		  close-sink-file  *break-on-up* *break-on-down*
;		  *break-on-draw*  *break-on-cycle*
;		  *default-iterations* *last-iteration* mc-setup
;		  sink-node-handler restore-nominals gc-every-n-times
;		  *mc-gc-hook* *gc-every-n* *gc-counter*
;		  find-sink-sources record-sink-nodes 
;		  ))



;;;; Variables and constants

;;; redefinitions for old constants

;; with *explore-margins* t leaf nodes do not sink correctly for
;; one-way propagation.  *explore-margins* is an expensive option for
;; monte-carlo computations anyhow.
(setq *explore-margins* nil) 

;;; Sink related

;; *sink-store-function* -- controls whether sinked nodes store the
;; function (t) (belief/plaus or prob) or the drawn value (prob mode
;; only). 
(defvar *sink-store-function* t
  "store density to sink file (t) or value (nil)")

;; *sink-stream* -- output stream for sink file
(defvar *sink-stream* t "sink output stream")



;;; Monte-Carlo Shell control

;; breakpoint variables--these allow breaking at fixed places in the
;; loop
(defvar *break-on-draw* nil "Break after drawing from sources (children)")
(defvar *break-on-up* nil "Break after propagation upwards")
(defvar *break-on-down* nil "Break after propagation down")
(defvar *break-on-cycle* '(print #!*iteration-count*)
  "Break at end of cycle (after sink)")

(defvar sets::*save-values* nil "Save old values when looping")
(defvar sets::*save-sink-values* t "Save old sunk values when looping")
(setq sets::*save-messages* nil)	;change default if doing mc.

(setf will::*inherited-vars*
       (union '(sets::*save-messages* sets::*iteration-count*
		sets::*save-sink-values* sets::*save-values*)
	       will::*inherited-vars*))




;; iteration counter
(defvar sets::*iteration-count* 0 "Number of current iteration")
(defvar *last-iteration* 0 "Iteration at which to stop")
(defparameter *default-iterations* 100 "Default number of draws")
(proclaim '(fixnum sets::*iteration-count* *last-iteration*
		  *default-iterations*))

;; garbage-collection hook and control variables.  The variable
;; *gc-hook* allows for the control of garbage collection by mc-shell
;; The counters *gc-every-n* and *gc-counter* are for the
;; gc-every-n-times function which performs a gc every n times it is
;; called. 
(defvar *mc-gc-hook* nil "Expression to exaluate at end of every cycle")
(defvar *gc-every-n* 50 "Number of cycles per garbage collect")
(defvar *gc-counter* 0 "Number of cycles since last garbage collect")
(proclaim '(fixnum *gc-every-n* *gc-counter*))

;;;; Functions

;;; Monte Carlo Shells


;; mc-loop -- This assumes that set-it-up has been run and that
;; *margin-list* and *root* have been set to a sensible firing order.
;; It draws from the source distributions.  Propagates-up,
;; Propagates-down, and then writes the sync out to a file.  Running
;; it a second time allows for additional iterations.
(defun mc-loop (&optional (iterations *default-iterations*))
  (declare (type Fixnum iterations)
	   (:returns nil))
  "Cycles bidirectional monte carlo propagation.   The sequence of
operations is as follows:
	0) Inrement iteration count.
	1) Kill messages and drawn values.
	  ---> *mc-gc-hook*
	2) Draw random node quntities (draw-sources)
	  ---> *break-on-draw*
	3) Propagate values up
	  ---> *break-on-up*
	4) Propagate values down
	  ---> *break-on-down*
	5) Record sunk values to sink stream.
	  ---> *break-on-cycle*
	6) If *iteration-count* < *last-iteration* loop, else return

Note the presense of vairious hooks in this cycle:

<*mc-gc-hook*> is called at a time when the system has freed most of
its storage, and this would be a good time for garbage collection.
This form is evaled and should produce a sensible result (see
#'gc-every-n-times)

The <*break-on-draw*>, <*break-on-down*>, <*break-on-up*> and
<*break-on-cycle*> hooks all work in roughly the same way.  At the
appropriate place in the loop, their value is checked.  If the value
is nil, #'mc-loop proceeds with the calculation.  If the value
is a list whos car is a function, then that list is evaled.  If the
value is some other non-nil value, a break loop is entered.  This is
meant mainly to facilitate debugging.

Assumes that the tree of cliques, RNG and sinks have been initialized.
See #'mc-setup.
"
  (setq *last-iteration* (the fixnum (+ (the fixnum #!*iteration-count*)
					iterations)))
  (loop (incf #!*iteration-count*)
    (if #!*save-messages* (mark-all-messages) (kill-all-messages))
    (if #!*save-values* (progn (mark-child-values) (mark-parent-values))
      (progn  (kill-child-values) (kill-parent-values)))
    (if #!*save-sink-values* (mark-sink-values) (kill-sink-values))
					;Now have maximal garbage to
					;interesting storage, good
					;time to gc.
    (eval *mc-gc-hook*)
    (draw-sources)
    (when *break-on-draw*
      (if (and (listp *break-on-draw*) (fboundp (car *break-on-draw*)))
	  (eval *break-on-draw*)
	(break "Break on Draw cycle ~d>> " #!*iteration-count*)))
    (propagate-up)
    (when *break-on-up*
      (if (and (listp *break-on-up*) (fboundp (car *break-on-up*)))
	  (eval *break-on-up*)
	(break "Break on Propagation Up cycle ~d>> " #!*iteration-count*)))
    (propagate-down)
    (when *break-on-down*
      (if (and (listp *break-on-down*) (fboundp (car *break-on-down*)))
	  (eval *break-on-down*)
	(break "Break on Propagation Down cycle ~d>> " #!*iteration-count*)))
    (record-sinks)
    (when *break-on-cycle*
      (if (and (listp *break-on-cycle*)
	       (fboundp (car *break-on-cycle*)))
	  (eval *break-on-cycle*)
	(break "Break at end of cycle ~d>> " #!*iteration-count*)))
    (when (eql #!*iteration-count* *last-iteration*) (return))
    ))

;;mc-up-loop -- This one skips the propagation down step
(defun mc-up-loop (&optional (iterations *default-iterations*))
  "Cycles unidirectional monte carlo propagation.   In particular, it
omits the propagate down step from mc-loop.  The sequence of
operations is as follows:
	0) Inrement iteration count.
	1) Kill messages and drawn values.
	  ---> *mc-gc-hook*
	2) Draw random node quntities (draw-sources)
	  ---> *break-on-draw*
	3) Propagate values up
	  ---> *break-on-up*
	4) Record sunk values to sink stream.
	  ---> *break-on-cycle*
	5) If *iteration-count* < *last-iteration* loop, else return

Note the presense of vairious hooks in this cycle:

<*mc-gc-hook*> is called at a time when the system has freed most of
its storage, and this would be a good time for garbage collection.
This form is evaled and should produce a sensible result (see
#'gc-every-n-times)

The <*break-on-draw*>, <*break-on-up*> and <*break-on-cycle*> hooks
all work in roughly the same way.  At the appropriate place in the
loop, their value is checked.  If the value is nil, #'mc-up-loop proceeds
with the calculation.  If the value is a list whos car is a function,
then that list is evaled.  If the value is some other non-nil value, a
break loop is entered.  This is meant mainly to facilitate debugging.

Assumes that the tree of cliques, RNG and sinks have been initialized.
See #'mc-setup.
"
  (setq *last-iteration* (the fixnum (+ (the fixnum #!*iteration-count*)
					iterations)))
  (loop (incf #!*iteration-count*)
    (if #!*save-messages* (mark-all-messages) (kill-all-messages))
    (if #!*save-values* (progn (mark-child-values) (mark-parent-values))
      (progn  (kill-child-values) (kill-parent-values)))
    (if #!*save-sink-values* (mark-sink-values) (kill-sink-values))
					;Now have maximal garbage to
					;interesting storage, good
					;time to gc.
    (draw-sources)
    (when *break-on-draw*
      (if (and (listp *break-on-draw*)
	       (fboundp (car *break-on-draw*)))
	  (eval *break-on-draw*)
	(break "Break on Draw cycle ~d>> " #!*iteration-count*)))
    (propagate-up)
    (when *break-on-up*
      (if (and (listp *break-on-up*)
	       (fboundp (car *break-on-up*)))
	  (eval *break-on-up*)
	(break "Break on Propagation Up cycle ~d>> " #!*iteration-count*)))
    (record-sinks)
    (when *break-on-cycle*
      (if (and (listp *break-on-cycle*)
	       (fboundp (car *break-on-cycle*)))
	  (eval *break-on-cycle*)
	(break "Break at end of cycle ~d>> " #!*iteration-count*)))
    (when (eql #!*iteration-count* *last-iteration*) (return))
    ))




;;; mc-setup  -- sets up tree and initializes RNG routines
(defun mc-setup ()
  "Calls all setup routines for needed for monte carlo.  
In particular, calls #'set-it-up to set up tree, #'setup-all to set up
RNG and #'find-sink-sources to set up sink nodes."
  (set-it-up)				;set up tree
  (setup-all)				;set up RNG
  (find-sink-sources)			;set up sinks
  )




;;;; Drawing.  The concept of drawing is based on a list of parent
;;; nodes which contain information about the distribution of various
;;; component types. (As defined in my thesis).  The parents each have
;;; a list of children, which are components of that type.  Both
;;; parents and children are symbols with various properties.  

;;; Parent nodes have the following properties:
;;; :children -- list of children
;;; :model-type -- one of :binomial or :poisson
;;; :default-rate -- default value of rate property for poisson types
;;; :max-failures -- default value of max. failures for poisson types
;;; :distribution -- distribution of parameters (type distribution)
;;; :parameters-drawn -- parameters drawn for this cycle of iteration.
;;; :values -- list of values for the appropriate attribute (or dummy
;;; attribute)
;;; :frame -- dummy frame for parent
;;; value of parent nodes is a belief function or potential

;;; Children nodes have the following properties
;;; :parent -- parent node
;;; :model-type -- one of :binomial or :poisson
;;; :rate -- use rate property for poisson types
;;; :max-failures -- max. failures to model for poisson types
;;; :values -- list of values for the appropriate attribute (or dummy
;;; attribute)
;;; :frame -- frame of child node
;;; value is sample belief function or potential
;;; children have other attributes common to all nodes in tree model

;;; drawing routines

;; draw-sources -- this is called by the mc-looping routines.  It
;; draws all parent nodes and then maps their results to the children.
(defun draw-sources ()
  (declare (:returns nil))
  "Draws new random iterval/value for each parameter in
#!*parent-list* and propagates that value to all dependent (child)
nodes." 
  (map nil #'draw-parent-child #!*parent-list*))


;; draw-parent-child -- this handles a single parent.  It draws a set
;; of parameters, creates a val based on that drawing and then calls
;; draw-child for each child to copy that value over.
(defun draw-parent-child (parent)
  (declare (type Symbol parent))
  "Draws new random iterval/value for parameter <parent> and
propagates that value to all dependent (child) nodes." 
  (draw-parent parent)
  (dolist (child (get parent :children))
	  (draw-child child parent)))

;; draw-parent  -- draw parameters and construct default val for
;; parent. 
(defun draw-parent (node)
  (declare (type Symbol node)
	   (:returns (type Val drawn-value)))
  "Draws a new random interval/value for parent parameter at <node>
and sets up a dummy valuation for that parameter."
  (setf (get node :parameters-drawn) (draw (get node :distribution)))
  (set node (impute (get node :frame) (get node :parameters-drawn)
		    :param :model-type (get node :model-type)
		    :rate (get node :default-rate)
		    :max-events (get node :max-failures))))



;; draw-child -- if the default parameters of the parent match the
;; default parameters of the child we can simply copy the val and
;; change the frame.  If they do not match, then we need to
;; recalculate for this child (e.g., a poisson child with a different
;; time constant).
(defun draw-child (child parent)
  (declare (type Symbol child parent)
	   (:returns (type Val drawn-value)))
  "Propagates value of <parent> (recently constructed dummy node for a
parameter) to <child> adapting frame and (if necessary) ps-set."
  (set child (case (get child :model-type)
	       (:binomial (if (equal (get child :values) (get parent :values))
			      (copy-change-frame (get child :frame)
						 (eval parent))
			    (impute-binomial (get child :frame)
					     (get parent :parameters-drawn)
					     :model-type :binomial)))
	       (:poisson (if (and (eql (get child :rate)
				       (get parent :default-rate) )
				  (eql (get child :max-failures)
				       (get parent :max-falures)))
			     (copy-change-frame (get child :frame)
						(eval parent))
			   (impute-poisson (get child :frame)
					   (get parent :parameters-drawn)
					   :model-type :poisson
					   :rate (get child :rate)
					   :max-events (get child :max-failures))))
	       (t (error "draw-child: Bad child ~S" child)))))


;;;; Sinking values to nodes.

;;; The nodes on the sink-list are treated in a special way in the
;;; monte-carlo shell.  In each iteration it is assumed that after the
;;; propagation is finished, the current values of all sink nodes will
;;; be recorded.  If the value of *sink-store-function* is t (the
;;; default) this will happen.  If the value of *sink-store-function*
;;; is nil, then instead the value is assumed to be logical, and the
;;; name of the selected value drawn is sunk.

;;; sinks are actuall attributes which hang off of nodes.  It is
;;; possible to have a belief function, a sink and an attribute.

;;; record-sinks -- main interface from mc-shell.  It calls the
;;; sink-node-handler which returns a string for each node on the
;;; sink-list.  The results are then concatonated, with the maximum
;;; 80 characters per line.  Output is sent to
;;; *sink-stream*. 
(defun record-sinks ()
  (declare (:returns nil))
  "For all attributes on the #!*sink-list*  it applies the function
#'sink-node-handler to translate the value into a string.  These
strings are printed to *sink-stream*"
  (let ((sink-strings (mapcar #'sink-node-handler #!*sink-list*)))
    (format *sink-stream* "~&~{~<~%     ~5,80:;~A~>~^     ~}~%"
	    sink-strings)))

;;; open-sink-file -- opens the sink file.  Note! this is not done
;;; automaticly, the value of *sink-stream* defaults to t and the
;;; output is sent to the terminal.  This is not a wonderful format
;;; for capturing the results for later analysis.  Keyword arguments
;;; :if-exists and :if-does-not-exist are sent to open.
(defun open-sink-file (path &key (if-exists :overwrite)
			    (if-does-not-exist :create))
  (declare 
	   (type (or Pathname String Symbol) path)
	   (:returns (type Stream *sink-stream*)))
  "Opens <path> as new *sink-stream*.  Keyword args are passed to #'open."
  (setq *sink-stream*
	(open path :direction :output ;:element-type 'string-char
	      :if-exists if-exists :if-does-not-exist if-does-not-exist)))

;;; close-sink-file -- closes the sink file and resets sink to t
(defun close-sink-file ()
  (declare (:returns (type (member T) *sink-stream*)))
  "Closes *sink-stream* and redirects sink output to stream t
 (usually standard output)."
  (close *sink-stream*)
  (setq *sink-stream* t))

;;; mark-sink-file -- used to mark a special event (such as an
;;; interation count) into the sink file for future reference.  Takes
;;; two arguments, a name (which defaults to the name of the current
;;; rules package) and a number (which defaults to the current iteration
;;; count).
(defun mark-sink-file (&optional (name (package-name *rules-package*))
				 (number #!*iteration-count*))
  (declare (type (or String Symbol) name)
	   (type Fixnum number)
	   (:returns nil))
  "Places a mark in *sink-stream*, Mark consists of <name> (defaults
to current *rules-package* name) and <number> defaults to current
#!*iteration-count*."
  (format *sink-stream* "~%~%~A:  ~D~%~%" name number))

;;; record-sink-nodes -- prints a list of all sink nodes in order to
;;; the sink file so that later programs can reconstruct the data from
;;; the file.
(defun record-sink-nodes ()
  (declare (:returns nil))
  "Records the names of the nodes in #!*sink-list* on *sink-stream*"
  (format *sink-stream* "~&~{~<~%     ~5,80:;~S~>~^     ~}~%"
	  #!*sink-list*))



;;; sink-node-handler -- this is perhaps the most important and most
;;; likely to be changed part of the sink system.  This is the
;;; processing done to each node in order to sink that node.
(defun sink-node-handler (sink)
  (declare (type Symbol sink)
	   (:returns (type String sink-value)))
  "Gets a value in string form from sink-node <sink>."
  (let ((val (sink-to-sink sink)))
    (cond ((null *sink-store-function*) (sink-get-att-value val))
	  ((eql t *sink-store-function*) (sink-get-param-value val))
	  ((functionp *sink-store-function*) (apply *sink-store-function* val))
	  (t (sink-get-param-value val)))))
	   
;;; sink-to-sink -- computes the sink value for a sink
(defun sink-to-sink (sink)
  (declare (type Symbol sink)
	   (:returns (type Val sink-margin)))
  "Gets marginal valuation associated with <sink>."
  (let {(nodeval (marginal-node (get sink :sink-source)))
	(att (get sink :sink))}
    (setf (get sink :sink-value)
	  (if (equal (list att) (val-frame nodeval))
	      nodeval
	    (@@-> (list att) nodeval)))))

;;; find-sink-sources -- this function finds source nodes for all of
;;; the sinks.
(defun find-sink-sources ()
  "Finds nodes in the tree of cliques from which values in
#!*sink-list* can be read."
  (map nil #'find-sink-source #!*sink-list*))

;;; find-sink-source
(defun find-sink-source (sink)
  (declare (type Symbol sink)
	   (:returns (type Symbol source)))
  "Finds a node in the tree of cliques from which to read value of
<sink>."
  (let* [(att (get sink :sink))
	 (candidate-nodes (remove att #!*val-list*
				  :test-not #'member))]
    (if (null candidate-nodes)
	(error "find-sink-source: Orphaned sink ~S" sink))
    (setf (get sink :sink-source)
	  (car (find (reduce #'min (mapcar #'length candidate-nodes))
		     candidate-nodes :key #'length)))))


;;; kill-sink-values
(defun kill-sink-values ()
  (declare (:returns nil))
  "For each sink node, set the :sink-value to nil."
  (map nil #'(lambda (sink) (setf (get sink :sink-value) nil))
       #!*sink-list*))

(defun mark-sink-values ()
  (declare (:returns nil))
  "For each sink node, push the :sink-value on the :sink-val-list and
then set the :sink-value to nil."
  (map nil #'(lambda (sink)
	       (push (get sink :sink-value) (get sink :sink-val-list))
	       (setf (get sink :sink-value) nil))
       #!*sink-list*))



;;;; restore-nominals -- restores the nominal value to all of the
;;;; children nodes
(defun restore-nominals ()
  "Sets the value of each child node to the nominal (average) value." 
  (dolist (node #!*child-list*)
     (set node (get node :nominal-val))
     (setf (get node :received)
	   (remove-duplicates (get node :received) :key #'car))))



;;; kill-child-values -- sets the values of all child-nodes to nil.
;;; Warning: this will leave the system in a state unsuitable for
;;; computations until either, (a) draw-sources is executed to draw
;;; new mc-values or (b) restore-nominals is executed to restore those
;;; sources to the original values.
(defun kill-child-values ()
  "Sets the value of each child node to nil.  
Warning: this will leave the system in a state unsuitable for
computations until either, (a) draw-sources is executed to draw
new mc-values or (b) restore-nominals is executed to restore those
sources to the original values."
  (map nil #'(lambda (child) (set child nil)) #!*child-list*))

(defun mark-child-values ()
  "Sets the value of each child node to nil and pushes its old value
on the :orignal-val stack.
Warning: this will leave the system in a state unsuitable for
computations until either, (a) draw-sources is executed to draw
new mc-values or (b) restore-nominals is executed to restore those
sources to the original values."
  (map nil #'(lambda (child)
	       (push (eval child) (get child :original-val))
	       (set child nil))
       #!*child-list*))

;; kill-parent-values -- similar to above for parents.  See warning
;; above. 
(defun kill-parent-values ()
  "Sets the value of each parent node to nil.  
Warning: this will leave the system in a state unsuitable for
computations until either, (a) draw-sources is executed to draw
new mc-values or (b) restore-nominals is executed to restore those
sources to the original values."
  (map nil #'(lambda (parent)
	       (set parent nil)) #!*parent-list*))
(defun mark-parent-values ()
  "Sets the value of each parent node to nil and pushes its old value
on the :orignal-val stack.
Warning: this will leave the system in a state unsuitable for
computations until either, (a) draw-sources is executed to draw
new mc-values or (b) restore-nominals is executed to restore those
sources to the original values."
  (map nil #'(lambda (parent)
	       (push (eval parent) (get parent :original-val))
	       (set parent nil)) #!*parent-list*))


;;;; garbage collection -- In order to try an maximize the efficiency
;;;; of garbage collection, I have provided a hook *mc-gc-hook* whose
;;;; value gets called on every loop of the mc loop.  After messages
;;;; are killed and child and parent values are killed.  

;;; The following function is provided as an example of what might be
;;; called from *mc-gc-hook*.  Other things might be system dependent
;;; gc-features, such as LUCID lisp's ephemeral-gc

(defun gc-every-n-times ()
  "Collect garbage every *gc-every-n* times it is called.  This (or a
variation on the theme) would make a good value for *mc-gc-hook*"
  (when (<= (incf *gc-counter*) *gc-every-n*)
	(setq *gc-counter* 0)
	(gc)))



;;; provide when loaded
(bel-provide :monte)
