;;; lowread.lisp -- low level reading macros common to belief
;;; functions and potentials belief function defs.

;;; Copyright 1989 Russell G. Almond
;;; License is granted to copy this program for education or research
;;; purposes, with the restriction that no portion of this program may
;;; be copied without also copying this notice.  All other rights
;;; reserved. 

;;; 2/16/89 -- split off from read.lisp to sevre  potentials as well as
;;; belief functions

;;; 7/19/89 -- Version 1.1
;;; 	* Added facility to read parameters and constants in place of
;;; numbers. [d]
;;; 8/4/89
;;;	* Added :failure and :poisson max-event keywords to defatt.
;;;[d]
;;; 8/9/89 
;;;	* Added :sink keyword to defatt statement which makes a sink
;;;keyword of the same name

;;; 2/24/92 --- Version 1.2 --- Documentation and declaration cleanup.

;(provide 'lowread)
(in-package :sets )
(bel-require :structures  "structures")
(bel-require :sets  "sets")
(bel-require :graphs  "graphs")
(bel-require :utils  "utils")
;(use-package '(basic graphs utils))
;(export '(defatt *true-false-values* *model-graph* *val-list*
;	  set-up-if set-up-ifor set-up-iff set-up-iffor
;	  clean-attribute-list clean-attribute
;	  logical-true logical-false logical-unknown
;	  logical-nand logical-or logical-and logical-nor logical-xor
;	  print-ps-set reset-model attribute-to-frame get-number *sink-list*
;	  *failure-values*))




;;; First we define a few global variables of some use.

;; *model-graph* -- Build up a graph containing all belief functions here.
;; Each defatt defines a new node, each defbel defines a new edge.
(defvar *model-graph* (make-graph :nodes '() :edges '())
  "Graph with attributes as nodes, and belief functions as edges")

;; *val-list* -- Build up an association list containing all belief
;; functions (key) and their frames (datum).
(defvar *val-list* '() "List of belief functions and frames")

;; *true-false-values* -- A parameter giving the default values that a
;; true false attribute can take on, attribute given the :true-false
;; key. and similar for :failure

(defparameter *true-false-values* '(:T :F)
  "Values for a :true-false attribute")

(defparameter *failure-values* '(:F :NF)
  "Values for a :failure attribute")

;; *sink-list* -- list of monitored nodes for the Monte Carlo shell
(defvar sets::*sink-list* nil "List of monitored nodes.")



;; relax -- evaluate an expression or expressions with the value of
;; *break-on-warnings* set to nil (for deliberately ignoring
;; restrictions on overwriting edges.
(defmacro relax (&body bod)
  "Evaluate an expression or expressions with the value of
  *break-on-warnings* set to nil (for deliberately ignoring
  restrictions on overwriting edges."
  `(let ((*break-on-warnings* nil)) ,.bod))

;; reset-model -- resets *model-graph* and *val-list*
(defun reset-model ()
  "Reset the model descriptions for re-reads."
  (set #?*model-graph* (make-graph))
  (set #?*val-list* nil))



;;; Functions for defining attributes (variables) -- defatt
;;; (defatt <name> { <list-of-values> | :true-false } [<Doc-string>] )
;;; Sets the values property of the attribute and adds the attribute to
;;; the nodes of *model-graph*

;;defatt -- defatt works by constructing the commands necessary to
;;create a new attribute and pushing them into form, which is then
;;executed as a progn
;;
(defmacro defatt (name &body bod &aux (form nil) (nograph nil) (sink nil))
  "Define Attribute: 
  (defatt <name>  { <list-of-values> | <uniform-value-id> } [<Doc-string>] 
          &key nograph sink)
   Defines an attribute <name> which has either <list-of-values> or a
predefined set of values identified by <uniform-value-id>.
<Doc-string> is an optional documentation string.  Keyword <nograph>
suppresses the creation of a new node in #?*model-graph* and keyword
<sink> makes this value a sink.  

<uniform-value-id> should be one of 
	:true-false uses value of #!*true-false-values* as <list-of-values>
	:failure uses value of #!*failure-values* as <list-of-values>
	:poisson <n> uses {0 1 ... n+} as <list-of-values>

This is intended to resemple the defvar macro in style.  Note that it
evaluates its <list-of-values> argument.

Note:  intellegent destructuring of macro <bod> allows the program to
be flexible about things like proper handling of keyword args when
optional documentation string is missing.
"
  (declare (type Symbol name) (type (member T NIL nograph sink))
	   (:returns (type List outcomes)))
  (if (listp (car bod))
      (push `(setf ,`(get (quote ,name) :values)  ,(car bod)) form)
    (case (car bod)
	  (:true-false
	   (push `(setf ,`(get (quote ,name) :values) #!*true-false-values*) form))
	  (:failure
	   (push `(setf ,`(get (quote ,name) :values) #!*failure-values*) form))
	  (:poisson
	   (pop bod)
	   (let ((max-fails (get-number (car bod))))
	     (unless max-fails
		     (error "defatt:~S: :poisson keyword must be followed~
                             by number" name))
	     (push `(setf ,`(get (quote ,name) :values)
			  (iota ,max-fails 0 t)) form)
	     (push `(setf ,`(get ',name :poisson) ,max-fails)
		   form)))
	  (t 
	   (error "defatt: Bad values for attribute ~S" name))))
  (pop bod)
  (when (stringp (car bod))
	(push `(setf ,`(documentation (quote ,name) 'variable) ,(pop bod))
	      form))
  (loop (unless (keywordp (car bod)) (return))
	  (case (pop bod)
		(:nograph (if (and bod (or (null (car bod))
					   (eq t (car bod)))) 
			      (setq nograph (pop bod))
			    (setq nograph t)))
		(:sink (if (or (null (car bod)) (eq t (car bod)))
			      (setq sink (pop bod))
			    (setq sink t)))))
  (unless nograph
	  (push `(pushnew (quote ,name) (graph-nodes ,#?*model-graph*)
			  :test #'equal) form))
  (when sink
	(push `(setf (get ',name :sink) ',name) form)
	(push `(pushnew ',name ,#?*sink-list*) form))
  `(progn ,.form))


;; show-variable -- prints an attribute and its values.
(defun show-variable (var &optional (stream t))
  (declare (type Symbol var) (type Stream stream)
	   (:returns nil))
  "Prints a variable and its values."
  (format stream "~S:~S~%" var (get var :values)))



;;; functions for finding ps-sets from logical statements

;; set-up-if -- sets up the ps-set for a basic if-type rule
(defun set-up-if (consequent conditions)
  (declare (type (or Symbol List) consequent) (type List conditions)
	   (:returns (type Ps-Set if-rule)))
  "Sets up a PS (TS) set corresponding to logical rule if (and <conditions>)
then <consequent>.  Frame is (cons <consequent> <conditions>)."
  (cons (cons (logical-true consequent) (mapcar #'logical-true conditions))
	(mapcar #'(lambda (#1=#:x) (cons (logical-unknown consequent) #1#))
		(logical-nand conditions))))

;; set-up-ifor -- sets up the ps-set for a basic ifor-type rule
(defun set-up-ifor (consequent conditions)
  (declare (type (or Symbol List) consequent) (type List conditions)
	   (:returns (type Ps-Set if-rule)))
  "Sets up a PS (TS) set corresponding to logical rule if (or <conditions>)
then <consequent>.  Frame is (cons <consequent> <conditions>)."
  (cons (cons (logical-unknown consequent) (mapcar #'logical-false conditions))
	(mapcar #'(lambda (#1=#:x) (cons (logical-true consequent) #1#))
		(logical-or conditions))))

;; set-up-iff -- sets up the ps-set for a basic iff-type rule
(defun set-up-iff (consequent conditions)
  (declare (type (or Symbol List) consequent) (type List conditions)
	   (:returns (type Ps-Set if-rule)))
  "Sets up a PS (TS) set corresponding to logical rule (and <conditions>)
if and only if <consequent>.  Frame is (cons <consequent> <conditions>)."
  (cons (cons (logical-true consequent) (mapcar #'logical-true conditions))
	(mapcar #'(lambda (#1=#:x) (cons (logical-false consequent) #1#))
		(logical-nand conditions))))


;; set-up-iffor -- sets up the ps-set for a basic iffor-type rule
(defun set-up-iffor (consequent conditions)
  (declare (type (or Symbol List) consequent) (type List conditions)
	   (:returns (type Ps-Set if-rule)))
  "Sets up a PS (TS) set corresponding to logical rule (or <conditions>)
if and only if <consequent>.  Frame is (cons <consequent> <conditions>)."
  (cons (cons (logical-false consequent) (mapcar #'logical-false conditions))
	(mapcar #'(lambda (#1=#:x) (cons (logical-true consequent) #1#))
		(logical-or conditions))))



;;; Logical-condition generators --- These functions generate logical
;;; conditions from list of attributes and (not attribute)'s
;;; we will also allow conditions of the type (attribute {<value>}*) to
;;; handle some non true-false cases.

;;logical-true -- finds the logical true value of the attribute
(defun logical-true (attribute)
  (declare (type (or Symbol List) attribute)
	   (:returns (type T true-value)))
  "Gets the ``true'' value of <attribute>.  <attribute> can be
<attsym> or (not <att>) where <att> is an attribute or (<attsym> .
<vallist>)  where <attsym> is a symbol defined as an attribute (using
defatt) and <vallist> is a subset of values of <attsym>.  Unless the
attribute has the property :poisson, or is of the explicit value form,
the car of the value-list for that attribute is assumed to be
logical-true."
  (if (listp attribute)
      (if (equal (car attribute) 'not)
	  (logical-false (cadr attribute))
	(if (subsetp (cdr attribute) (get (car attribute) :values))
	    (cdr attribute)
	  (error "def<fn>:bad attribute ~S" attribute)))
    (if (get attribute :poisson)
	(iota (get attribute :poisson) 1 t) 
      (car (get attribute :values)))))



;;logical-false -- finds the logical false value of the attribute
(defun logical-false (attribute)
  (declare (type (or Symbol List) attribute)
	   (:returns (type T false-value)))
  "Gets the ``false'' value of <attribute>.  It returns the
set-complement of (logical-true <attsym>) with respect to values of
<attsym>.   <attribute> can be <attsym> or (not <att>) where <att> is
an attribute or (<attsym> . <vallist>)  where <attsym> is a symbol
defined as an attribute (using defatt) and <vallist> is a subset of
values of <attsym>.  Unless the attribute has the property :poisson,
or is of the explicit value form, the car of the value-list for that
attribute is assumed to be logical-true."
  (if (listp attribute)
      (if (equal (car attribute) 'not) (logical-true (cadr attribute))
 	(if (subsetp (cdr attribute) (get (car attribute) :values))
	    (set-difference (get (car attribute) :values) (cdr attribute))
	  (error "def<fn>:bad attribute ~S" attribute)))
    (if (get attribute :poisson) 0
      (cadr (get attribute :values)))))


;;logical-unknown -- finds the :values of the attribute
(defun logical-unknown (attribute)
  (declare (type (or Symbol List) attribute)
	   (:returns (type List all-values)))
  "Gets the ``false'' value of <attribute>.  It returns the
values of <attsym>.   <attribute> can be <attsym> or (not <att>) where
<att> is an attribute or (<attsym> . <vallist>)  where <attsym> is a symbol
defined as an attribute (using defatt) and <vallist> is a subset of
values of <attsym>.  Unless the attribute has the property :poisson,
or is of the explicit value form, the car of the value-list for that
attribute is assumed to be logical-true."
  (if (listp attribute)
      (if (equal (car attribute) 'not) (logical-unknown (cadr attribute))
	(if (subsetp (cdr attribute) (get (car attribute) :values))
	    (get (car attribute) :values)
	  (error "def<fn>:bad attribute ~S" attribute)))
    (get attribute :values)))

;; logical-nand -- sets up the truth table for nand in ps-set form
(defun logical-nand (conditions)
  (declare (type List conditions)
	   (:returns (type List nand-ps-set)))
  "Produces a ps-set corresponding to the event (nand <conditions>).
Conditions are a list of <attributes> (see logical-true)."
  (cond ((endp conditions) nil)
	((eql 1 (length conditions))
	 (list (list (logical-false (car conditions)))))
	(t (cons (cons (logical-false (car conditions))
		       (mapcar #'logical-unknown (cdr conditions)))
		 (mapcar #'(lambda (#1=#:x)
			     (cons (logical-true (car conditions)) #1#))
			 (logical-nand (cdr conditions)))))))


;; logical-or -- sets up the truth table for or in ps-set form
(defun logical-or (conditions)
  (declare (type List conditions)
	   (:returns (type List or-ps-set)))
  "Produces a ps-set corresponding to the event (or <conditions>).
Conditions are a list of <attributes> (see logical-true)."
  (cond ((endp conditions) nil)
	((eql 1 (length conditions))
	 (list (list (logical-true (car conditions)))))
	(t (cons (cons (logical-true (car conditions))
		       (mapcar #'logical-unknown (cdr conditions)))
		 (mapcar #'(lambda (#1=#:x)
			     (cons (logical-false (car conditions)) #1#))
			 (logical-or (cdr conditions)))))))

;; logical-and -- sets up the truth table for and in ps-set form
(defun logical-and (conditions)
  (declare (type List conditions)
	   (:returns (type List and-ps-set)))
  "Produces a ps-set corresponding to the event (and <conditions>).
Conditions are a list of <attributes> (see logical-true)."
  (list (mapcar #'logical-true conditions)))

;; logical-nor -- sets up the truth table for nor in ps-set form
(defun logical-nor (conditions)
  (declare (type List conditions)
	   (:returns (type List nor-ps-set)))
  "Produces a ps-set corresponding to the event (nor <conditions>).
Conditions are a list of <attributes> (see logical-true)."
  (list (mapcar #'logical-false conditions)))

;; logical-xor -- sets up the truth table for xor is ps-set form
(defun logical-xor (conditions)
  (declare (type List conditions)
	   (:returns (type List xor-ps-set)))
  "Produces a ps-set corresponding to the event (xor <conditions>).
Conditions are a list of <attributes> (see logical-true)."
  (cond ((endp conditions) nil)
	((eql 1 (length conditions))
	 (list (list (logical-true (car conditions)))))
	(t (cons (cons (logical-true (car conditions))
		       (mapcar #'logical-false (cdr conditions)))
		 (mapcar #'(lambda (#1=#:x)
			     (cons (logical-false (car conditions)) #1#))
			 (logical-xor (cdr conditions)))))))



;;; Functions dealing with attribute lists

;; clean-attribute-list -- this function cleans off the attribute list
;; of (not attribute) and (attribute value) constructions so that it is
;; suitable as the frame specification for a belief function
(defun clean-attribute-list (attlist)
  (declare (type List attlist)
	   (:returns (type List frame)))
  "this function cleans off the attribute list of (not attribute) and
(attribute value) constructions so that it is suitable as the frame
specification for a belief function."
  (cond ((endp attlist) '())
	(t (cons (clean-attribute (car attlist))
		 (clean-attribute-list (cdr attlist))))))

(defun clean-attribute (attribute)
  (declare (type (or Symbol List) attribute)
	   (:returns (type Symbol attsym)))
  "This function cleans off (not <attribute>) and (<attsym> .
<values>) constructions leaving just attribute symbol <attsym>."
  (if (listp attribute)
      (if (equal (car attribute) 'not) (clean-attribute (cadr attribute))
	(car attribute))
    attribute))

  

;; attribute-to-frame -- this converts an attribute list to a frame
;; expressed as a list of lists of values, used by many ps-set
;; functions 
(defun attribute-to-frame (alist)
  (declare (type List alist) (:returns (type List theta)))
  "Changes frame of attributes <alist> into frame of values <theta>."
  (mapcar #'(lambda (#1=#:x) (get #1# :values)) alist))





;;; print-ps-set -- This prints a ps-set using the {([ convention ])}
(defun print-ps-set (ps-set &optional (stream nil) &aux (space-flag nil))
  (declare (type List ps-set) (type (or Stream nil) stream)
	   (:returns nil))
  "Prints a PS(TS)-set using \\{([ ])\\} convention.  Prints list of
tuples in \\{ \\} braces and calls print-ps-tuple to print tuple."
  (terpri stream) (write-char #\{ stream)
  (map nil #'(lambda (ps-tuple)
	       (if space-flag (write-char #\space stream))
	       (print-ps-tuple ps-tuple stream)
	       (setq space-flag t))
	ps-set)
  (write-char #\} stream) (terpri stream))

(defun print-ps-tuple (ps-tuple &optional (stream nil) &aux (space-flag nil))
  (declare (type List ps-tuple) (type (or Stream nil) stream))
  "Prints one tuple of a TS-set.  Using ([]) convention.  Atoms are
printed using print-ps-atom."
  (write-char #\( stream)
  (map nil #'(lambda (ps-atom)
	       (if space-flag (write-char #\space stream))
	       (print-ps-atom ps-atom stream)
	       (setq space-flag t))
	ps-tuple)
  (write-char #\) stream))

(defun print-ps-atom (ps-atom &optional (stream nil) &aux (space-flag nil))
  (declare (type T ps-tuple) (type (or Stream nil) stream))
  "Prints one atom of a TS-set.  If single value, prints it; else
wraps [ ] around list."
  (if (atom ps-atom) (prin1 ps-atom stream)
    (progn 
      (write-char #\[ stream)
      (map nil #'(lambda (ps-at-el)
		   (if space-flag (write-char #\space stream))
		   (prin1 ps-at-el stream)
		   (setq space-flag t))
	    ps-atom)
      (write-char #\] stream))))



;;; get-number -- looks for number, or variable, parameter, constant,
;;; expression which will yield numeric result when evaled.  Returns
;;; nil if no number found, and the number if it is found
(defun get-number (form)
  (declare (type T form) (:returns (type (or Nil Number) eval-form)))
  "Checks <form> to see if it is a number, or variable, parameter, constant,
expression which will yield numeric result when evaled.  Returns
nil if no number found, and the number if it is found.  Will not
generate errors by evaluating s-exps which of which the car is not a
function.

Used to allow parameters, constants and arithmetic expressions in
place of number in definitions." 
  (cond ((numberp form) form)
	((and (symbolp form) (boundp form) (numberp (eval form)))
	 (eval form))
	((and (consp form) (or (fboundp (car form)) (eq (car form) 'lambda))
	      (numberp (eval form)))
	 (eval form))))



;;; provide when loaded
(bel-provide :lowread)