;;; lowdist -- This file has distribution function drawing information
;;; for belief functions in it.


;;; Copyright 1989 Russell G. Almond
;;; License is granted to copy this program for education or research
;;; purposes, with the restriction that no portion of this program may
;;; be copied without also copying this notice.  All other rights
;;; reserved. 

;;; 7/21/89 -- Version 1.1 Created File

;;; 8/5/89 -- Added nominal feature to distribtions.  Similar to setup
;;; and draw it creates a "nominal" value of the distribution for
;;; pre-montecarlo analysis.

;;; 2/24/92 -- Version 1.2 Updated documentation.



;;; The heart of the distribution feature is the creation of
;;; distribution function objects.  This file is meant to 
;;; handle common cases for belief functions and probabilities.

;(provide 'lowdist)
;(in-package 'sets :nicknames '(commonb));sets seems to have become the
;					;package for things which are
;					;common to bel and prob mode,
;					;maybe it needs to be rename
(bel-require :machine-utils "bymachine" )
(bel-require :lowread  "lowread")
;(use-package '(basic graphs sets utils))
;(export '(draw-default draw-mix draw setup-default setup-mix setup
;	  setup-all nominal nominal-default
;	  nominal-mix *parent-list* 
;	  *default-rate-default* *default-max-failures* *child-list*
;	  ))


;;; Variables

;; *rng-initialized* -- set when calls are made to setup-rng so that
;; cleanup-rng will be called at the proper time
(defvar *rng-initialized* nil "Count number of rng calls.")

;; *default-rate-default* -- default time constants for poisson
;; processes 
(defvar *default-rate-default* 1.0 "Default time constant for Poisson
processes.")

;; *default-max-failures* -- default maximum number of failures of
;; this type to consider.  If set to 1 makes poisson behave like
;; binomials.
(defvar *default-max-failures* 1 "Default number of failure events to
consider in Poisson processes.")


;;; Stuff for the sets package which is necessary for mc-constrol and
;;; is rule-package dependent
(defvar sets::*parent-list* nil "List of Parameter sources.")	;Distributions of sources
(defvar sets::*child-list* nil "List of Nodes dependent on Parameter
sources.")				;Source nodes 


;;; add to list of variables to be exported via inheritance
(setf  will::*inherited-vars*
       (union '(*parent-list* *default-rate-default*
		 *default-max-failures* *child-list*)
	       will::*inherited-vars*))

;;; A new version of reset-model is defined which will reset more
;;; model lists
;; reset-model -- resets *model-graph* and *val-list*
(defun reset-model ()
  (declare (:returns nil))
  "Resets model dependent values so they can be reloaded."
  (set #?*model-graph* (make-graph))
  (set #?*val-list* nil)
  (set #?*sink-list* nil)
  (set #?*parent-list* nil)
  (set #?*child-list* nil))




;;; Setup and drawing Drawing functions common to belief and probabilities
;;; Plus nominal functions


;; draw -- this is the basic drawing function.  It is applied to a
;; distribtuion and it in turn applies that distributions drawing
;; function to itself and returns the list of parameters generated.
(defun draw (dist)
  (declare (type Distribution dist)
	   (:returns (type (List Float) random-value-or-interval)))
  "Really a generic function.  Draws from <dist>.  Behavior will
depend on structure and mode."
  (funcall (dist-draw-function dist) dist))


;; setup -- this is the basic setup function, it is similar to draw.
(defun setup (dist)
  (declare (type Distribution dist)
	   (:returns (type (member T nil) status)))
  "Really a generic function.  Sets up RNG generator objects for
<dist>.  Objects allocated will depend on distribution type and mode."
  (funcall (dist-setup-function dist) dist))

;; nominal -- basic dispatching nominal function
(defun nominal (dist)
  (declare (type Distribution dist)
	   (:returns (values (type (List Float) nominal-value-or-interval)
			     (type (or List
				       (member :data :param)) data-type))))
  "Sets nominal (average) value for distribution.  Sometimes returns a
data object instead of an interval, second value flags this fact.
Meant to be used as imput to #'impute."
  (funcall (dist-nominal-function dist) dist))


;; draw-default -- generic drawing function does not exists,
;; prints error message instead.
(defun draw-default (dist)
  (declare (type Distribution dist)
	   (:returns nil))
  "Default method for drawing.  Produces an error."
  (cerror "supply parameters"
	  "draw: distribtuion type does not have defined draw function"))

;; setup-default -- simialr to draw-default
(defun setup-default (dist)
  (declare (type Distribution dist)
	   (:returns nil))
  "Default method for setup.  Produces an error."
  (cerror "supply parameters"
	  "setup: distribtuion type does not have defined setup function"))


;; nominal-default -- simialr to draw-default
(defun nominal-default (dist)
  (declare (type Distribution dist)
	   (:returns (values nil nil)))
  "Default method for nominal value.  Produces an error."
  (cerror "supply parameters"
	  "nominal: distribtuion type does not have defined setup function")
  (values nil nil))

  
;; draw-mix -- mixtures are the same in both belief and prob mode.
;; One of the components is selected according to the weight and then
;; that component is drawn from
(defun draw-mix (dist)
  (declare (type Mixture-Dist dist)
	   (:returns (type (List Float) values)))
  "Draws from Mixture Distribution <dist>.   mixtures are the same in
both belief and prob mode. One of the components is selected according
to the weight and then that component is drawn from in a recursive
call to #'draw."
  (do* ((uni (random (mix-total-weight dist))
	     (- uni (the float (caar mix-list))))
	(mix-list (mix-of dist) (cdr mix-list)))
       ((cond ((endp mix-list)
	      (error "draw-mix: ran out of distributions before random ~F is zero" 
		     uni ))
	     ((not (consp (car mix-list)))
	      (error "draw-mix: bad mixture list ~S" mix-list))
	     ((< uni (the float (caar mix-list)))
	      (return (draw (cdar mix-list))))))
       (declare (float uni))))


;; setup-mix -- recursively calls setup on each of the components.
(defun setup-mix (dist)
  (declare (type Mixture-Dist dist)
	   (:returns (type (member T NIL) status)))
  "Maps over set-up for all the components."
  (map nil #'(lambda (par) (setup (cdr par))) (mix-of dist))
  t)


;; nominal-mix -- creates a nominal value by mixing the nominal
;; values for all components.
(defun nominal-mix (dist &aux (data `()) (weights '()) (types `()))
  (declare (type Mixture-Dist dist)
	   (type List data weights types)
	   (:returns (values (type (List List) weights-data-list)
			     (type List types))))
  "Creates a nominal value for the mixture distribution <dist> by
producing three lists.  Returns two values:  This first value is a
list of <weights> and <data> returned by the smaller functions.  The
second value is a list of the <types> returned by applying the nominal
function to the components." 
  (dolist (adist (mix-of dist) (values (list weights data) types))
	  (multiple-value-bind (datum type)
			       (nominal (cdr adist))
	     (push datum data)
	     (push (/ (car adist) (mix-total-weight dist)) weights)
	     (push type types))))


	    

;;; setup-all --- initializes RNG package and sets up all parent
;;; nodes in the list parent-nodes.  
(defun setup-all ()
  (declare (:returns nil))
  "Maps setup-dist over all of the distributions associated with
#!*parent-list*.  Also produces a call to set-up-rng to initialize RNG
stuff and if previously initialized a call to clean-up-rng to reclaim
unused RNG objects." 
  (if *rng-initialized* (clean-up-rng))
  (let ((dist-list (mapcar #'(lambda (par) (get par :distribution))
			   #!*parent-list*)))
    (set-up-rng (reduce #'+ (mapcar #'dist-rng-calls-per-draw dist-list)))
    (map nil #'setup dist-list)))


  


;;; provide when loaded
(bel-provide :lowdist)
