;;; graphs.lisp -- function for manipulating graphs and hyper graphs
;;; 
;;; Theses are functions for manipulating graphs and hyper graphs in a
;;; way that is useful in belief functions.
;;;
;;; Copyright 1986 Russell G. Almond
;;; License is granted to copy this program for education or research
;;; purposes, with the restriction that no portion of this program may
;;; be copied without also copying this notice.  All other rights
;;; reserved. 

;;; 12/2/87 -- changed the way the program calculates cores by using
;;; the one-step lookahead smallest clique, ties fewest fill-in method
;;; ( the best found so far).  Previous version of layered-list is now
;;; old-layered-list

;;; 8/5/89 -- Version 1.1
;;;	* Switch to FF as basis for layered-list[debuged and undone]
;;;	* Allow easy test-list usage[d] 
;;;	* Faster FF code[d]
;;; layered-list in now essentially a hook function which calls
;;; fast-ff which calls the fast version of the fewest-fill-ins
;;; algorithm.  old-layered-list has been declare obsolete, use
;;; (test-list #'core-list ) instead.  Test-list is now supplied as
;;; standard without loading the search package.  

;;; 8/6/89 -- switched back to (test-list #'one-sc-tff) for
;;; layered-list as it is safer.  fast-ff doesn't seem to work as well
;;; in cases where there are different sized nodes.

;;; 10/15/90 --- added prop-graph to simple graph conversion for
;;; compatability with JAM's clay package for graph layout.

;;; 2/24/92 --- Version 1.2 commented to JAM's definitions
;;; specifications.  Note:  this file will soon be obsolete, replaced
;;; by the graf package.


;(provide 'graphs)
(in-package :graphs )
;(bel-require :structures "structures")
;(use-package 'basic)
;(eval-when (compile load eval) (import 'mode::node-size ))
;(export '(valid-graphp hyper-graphp graph-2-p skeleton reduce-edge
;		       neighbors neighborp closure s-delete s-2-delete
;		       r-delete r-2-delete equal-graph add-node add-edge
;		       equal-set layered-list core-list leaf-p s-delete-list
;		       one-sc-tff test-list fast-ff one-ff
;	  	       prop-graph-2-graph graph-2-prop-graph))
;;; the variable node-size should be imported from arithmatic or
;;; potentials (depending)



;;; valid-graphp -- returns t if candidate is a valid graph or hyper-graph.
(defun valid-graphp (candidate)
  (declare (type T candidate) (:returns (type (member T NIL) isit?)))
  "Returns T if <candidate> is a valid graph or hyper-graph."
  (and (graph-p candidate)
       (list-o-listsp (graph-edges candidate))
       (null (set-difference (apply #'append (graph-edges candidate))
			     (graph-nodes candidate)))))

;;; list-o-listsp -- returns t if its argument is a list of lists
(defun list-o-listsp (object)
  (declare (type T object) (:returns (type (member T NIL) isit?)))
  "Returns T if <object> is a list of lists."
  (cond ((null object) t)
	((not (listp object)) nil)
	((not (listp (car object))) nil)
	((list-o-listsp (cdr object)))))


;;; hyper-graphp -- returns t if candidate is a hyper-graph and not a 2-graph
(defun hyper-graphp (candidate)
  (declare (type T candidate) (:returns (type (member T NIL) isit?)))
  "Returns T if <candidate> is a hyper-graph and has at least on
non-binary edge."
  (and (valid-graphp candidate)
       (hyper-edges (graph-edges candidate))))


;;; hyper-edges -- takes a list of edges and returns thoses edges which are not
;;; of length 2.
(defun hyper-edges (edge-list)
  (declare (type List edge-list) (:returns (type List hyper-edges)))
  "Takes an <edge-list> and returns those edges which are not simple
 (of length 2)."
  (cond ((null edge-list) nil)		;base case
	((eql 2 (length (the list (car edge-list))))
	 (hyper-edges (cdr edge-list)))
	(t (cons (car edge-list) (hyper-edges (cdr edge-list))))))

;;; graph-2-p -- returns t if the candidate is a 2 graphe as opposed to a
;;; hyper-graph.
(defun graph-2-p (candidate)
  (declare (type T candidate) (:returns (type (member T NIL) isit?)))
  "Returns T if <candidate> is a valid simple graph (no hyper-edges)."
  (and (valid-graphp candidate)
       (null (hyper-edges (graph-edges candidate)))))



;;; skeleton -- takes a hyper graph as an argument and returns a
;;; two-section.
(defun skeleton (h-graph)
  (declare (type Graph h-graph) (:returns (type Graph 2-section)))
  "Turns <h-graph> into a simple graph with the same connectivity."
  (make-graph :nodes (graph-nodes h-graph)
	      :edges (interacton-edge (graph-edges h-graph))))

;;; interaction-edges --- converts a list of hyperedges into a list of
;;; edges.  
(defun interaction-edges (h-edges)
  (declare (type List h-edges) (:returns (type List s-edges)))
  "Turns a list of hyper edges into a longer list of simple edges
which preserve connectivitiy."
  (remove-duplicates (mapcan #'reduce-edge h-edges)
		     :test #'equal-set))

;;; reduce-edge -- this function takes a list of nodes and breakes it
;;; up into pairs.
(defun reduce-edge (edge)
  (declare (type List edge) (:returns (type List edge-list)))
  "Breaks a list of edges into simple edges."
  (case (length edge)
	(0 nil)
	(1 nil)				;ignore singletons
	(2 (list edge))			;needs no more
	(t (append
	     (mapcar #'(lambda (#1=#:x) (list (car edge) #1#)) (cdr edge))
	     (reduce-edge (cdr edge))))))

;;; neighbors -- this function takes two arguments, a node and a graph,
;;; and returns a list of all of the neighbors of that node. 
(defun neighbors (node graph)
  (declare (type T node) (type Graph graph)
	   (type Sequence neighbors))
  "Returns the neighborhood of <node> in <graph>."
  (remove-if-not #'(lambda (#1=#:x) (neighborp node #1# graph))
		 (graph-nodes graph)))

;;; neighborp -- this function takes two nodes and a graph as an
;;; argument.  If they are neighbors, it returns an edge containing
;;; both. If they are not, it returns nil.  NOTE: a node is not a
;;; neighbor of itself
(defun neighborp (node-a node-b graph)
  (declare (type T node-a node-b) (type Graph graph)
	   (:returns (type (or T Nil) arethey?)))
  "Returns non-nil if <node-a> and <node-b> are neighbors in <graph>."
  (cond ((not (member node-a (graph-nodes graph) :test #'equal)) nil)
	((not (member node-b (graph-nodes graph) :test #'equal)) nil)
	((equal node-a node-b) nil)
	(t (do* ((edge-list (graph-edges graph) (cdr edge-list))
		(edge1 (car edge-list) (car edge-list)))
		((null edge-list) nil)
		(if (and (member node-a edge1 :test #'equal)
			 (member node-b edge1 :test #'equal))
		    (return edge1) nil)))))

;;; closure -- this function takes two arguments, a node and a graph
;;; and returns the closure (neighbors(node) u (node)) of that node in
;;; that graph.
(defmacro closure (node graph)
  (declare (type T node) (type Graph graph)
	   (type Sequence neighbors))
  "Returns the closure of <node> in <graph> (neighborhood + <node>)"
  `(cons ,node (neighbors ,node ,graph)))


;;; s-delete -- this function takes a node and a graph as an argument.
;;; It then forms the maximal subgraph, not containing that node.
(defun s-delete (node graph)
  (declare (type T node) (type Graph graph)
	   (:returns (type Graph new-graph)))
  "Kong simple deletion of <node> from <graph>."
  (make-graph
    :nodes (remove node (graph-nodes graph) :test #'equal)
    :edges (remove-duplicates
	     (remove nil
	       (mapcar #'(lambda (edge) (remove node edge :test #'equal))
		       (graph-edges graph)))
	     :test #'equal-set)))


;;; s-2-delete -- this function takes a node and a 2-graph as an argument.
;;; It then forms the maximal 2-subgraph, not containing that node.
(defun s-2-delete (node graph)
  (declare (type T node) (type Graph graph)
	   (:returns (type Graph new-graph)))
  "Kong simple deletion of <node> from <graph>.  Assumes graph is
simple and does not create singleton edges."
  (make-graph
    :nodes (remove node (graph-nodes graph) :test #'equal)
    :edges (remove-duplicates
	     (remove-if-not #'(lambda (#1=#:x)
				(declare (list #1#))
				(eql (length #1#) 2))
			    (mapcar #'(lambda (edge) (remove node edge :test #'equal))
		       (graph-edges graph)))
	     :test #'equal-set )))

;;; r-delete -- this function takes a node and a graph and forms the
;;; minimal graph containing the s-deleted graph and preserving
;;; connectivity.  (See Kong, p 33).
(defun r-delete (node graph)
  (declare (type T node) (type Graph graph)
	   (:returns (type Graph new-graph)))
  "Kong r-deletion (elimination) of <node> from <graph>.  Elimination
preserves connectivity by introducing new edges linking the
<neighborhood> of graph."
  (make-graph
    :nodes (remove node (graph-nodes graph) :test #'equal)
    :edges (remove-if #'null
		      (remove-duplicates
		       (cons (neighbors node graph)
			     (remove-if #'(lambda (edge) (member node edge :test #'equal))
					(graph-edges graph)))
		       :test #'equal-set))))



	   
;;; r-2-delete -- this function takes a node and a 2-graph and forms the
;;; minimal graph containing the s-deleted graph and preserving
;;; connectivity.  (See Kong, p 33).
(defun r-2-delete (node graph)
  (declare (type T node) (type Graph graph)
	   (:returns (type Graph new-graph)))
  "Kong r-deletion (elimination) of <node> from <graph>.  Elimination
preserves connectivity by introducing new edges linking the
<neighborhood> of graph.  Assumes simple graph so only introduces
simple edges."
  (make-graph
    :nodes (remove node (graph-nodes graph) :test #'equal)
    :edges (remove-duplicates
	       (append (reduce-edge (neighbors node graph))
		       (remove-if #'(lambda (edge) (member node edge :test #'equal))
			       (graph-edges graph)))
	       :test #'equal-set )))



;;equal-graph -- This predicate takes two graphs as arguments and
;;returns t if they are equal and nil otherwise.  It does not worry
;;about the order of the nodes or edges.
(defun equal-graph (graph-1 graph-2)
  (declare (type Graph graph-1 graph-2)
	   (:returns (type (member T NIL) are-they=?)))
  "Tests two graphs for equality up to order of nodes and edges."
  (and (equal-set (graph-nodes graph-1) (graph-nodes graph-2))
       (equal-set (graph-edges graph-1) (graph-edges graph-2))))
  
     

;;equal-set -- This predicate takes two sets (lists) as arguments and
;;returns t if they are equal and nil otherwise.
(defun equal-set (set-1 set-2)
  (declare (type List set-1 set-2)
	   (:returns (type (member T NIL) are-they=?)))
  "Tests two sets for equality.  This turns out to be a very important
equality test in model graphs, where order seldom is important."
  (null (set-exclusive-or set-1 set-2)))

;; add-node -- Add a node to an existing graph, return new graph.
(defun add-node (graph node)
  (declare (type Graph graph) (type T node)
	   (:returns (type Graph new-graph)))
  "Adds <node> to <graph> creating <new-graph>."
  (make-graph :nodes (cons node (graph-nodes graph))
	      :edges (graph-edges graph)))

;; add-edge -- Add an edge to an existing graph, return new graph.
(defun add-edge (graph edge)
  (declare (type Graph graph) (type T edge)
	   (:returns (type Graph new-graph)))
  "Adds <edge> to <graph> creating <new-graph>."
  (make-graph :nodes (graph-nodes graph)
	      :edges (cons edge (graph-edges graph))))


;; leaf-p -- Checks to see whether the node is a leaf in the given
;; graph.  A node is a leaf if it's neighbors are all neighbors of each
;; other.  (As we are not dealing with directed graphs, we only need
;; to check neighbors in one direction).
(defun leaf-p (node graph)
  (declare (type Graph graph) (type T node)
	   (:returns (type (member T NIL) is-it?)))
  "Returns T if <node> is a leaf (all neighbors connected) in <graph>;
 nil otherwise."
  (cond ((not (member node (graph-nodes graph):test #'equal)) nil)
	(t (all-neighbors-p (neighbors node graph) graph))))


;; all-neighbors-p -- Checks to see whether the list of nodes are all
;; neighbors in the given graph.
(defun all-neighbors-p (nodelist graph)
  (declare (type List nodelist) (type Graph graph)
	   (:returns (type (member T NIL) are-they?)))
  "Checks to see if all the nodes in <nodelist> are neighbors in <graph>."
  (cond ((endp nodelist) t)
	((eql (length nodelist) 1) t)
	((every #'(lambda (#1=#:x) (neighborp (car nodelist) #1# graph))
		(cdr nodelist))
	 (all-neighbors-p (cdr nodelist) graph))
	(t nil)))

;; layered-list -- Creates a layered list for a given graph, with
;; leaves first, then level 1 branches and so forth.  Now a hook to
;; another function which does all the work. fast-ff be default but
;; other combinations such as (test-list #'one-ff-sc) are interesting.
;; More possibilities are available in the search.lisp file.
(defun layered-list (graph &key (heuristic #'one-sc-tff))
  (declare (type Graph graph)
	   (type (function (graph) list))
	   (:returns (type List pealing-order)))
  "Creates a pealing order for <graph>.  Uses <heuristic> to peal
non-leaf nodes."
  (test-list heuristic graph))


;;; This is a version of layered list especially designed for
;;; experimenting with deletion order programs.
(defun test-list (heuristic graph)
  (declare (function  heuristic (graph) list)
	   (type graph graph)
	   (:returns (type List pealing-order)))
  "This function peals <graph> using <heuristic> to determine the
pealing order of non-leaf nodes."
  (let ((leaves (remove-if-not #'(lambda (#1=#:x) (leaf-p #1# graph))
			      (graph-nodes graph))))
    (cond ((null (graph-nodes graph)) nil)
	  ((null leaves) (apply heuristic (list graph)))
	  (t (append leaves (test-list heuristic
			     (s-delete-list leaves graph)))))))


;; s-delete-list -- S-deletes a list of nodes from graph
(defun s-delete-list (node-list graph)
  (declare (type Graph graph) (type List node-list)
	   (:returns (type Graph new-graph)))
  "S-deletes multiple nodes from <graph>, returning a new graph."
  (make-graph
   :nodes (remove-if #'(lambda (#1=#:x) (member #1# node-list :test #'equal))
		     (graph-nodes graph))
   :edges (remove-duplicates
	   (remove nil
	    (mapcar #'(lambda (edge)
			(remove-if #'(lambda (#1#) (member #1# node-list
							 :test #'equal))
				   edge))
		    (graph-edges graph)))
	   :test #'equal-set)))


;; core-list -- this function returns a list of nodes (a candidate
;; optimal deletion order) for the core of a graph.
(defun core-list (graph)
  (declare (type Graph graph)
	   (:returns (type List order)))
  "This choses an arbitray deletion order for <graph> which is assumed
to have no leaves.  It sorts them according to size and number of
neighbors.  It is neither particularly fast nor good as far a
heuristics go."
  (sort
   (sort (graph-nodes graph)
	 #'> :key #'(lambda (node) (length (the list (get node :values)))))
   #'< :key #'(lambda (node) (length (the list (neighbors node graph))))))

;;; one-sc-tff -- finds core by one-step (smallest clique) method
;;; ties broken by fewest-fill-ins.
(defun one-sc-tff (graph)
  (declare (type Graph graph)
	   (:returns (type List deletion-order)))
  "Heuristic for choosing a deletion order of nodes from <graph>,
where it is assumed that there are no leaves in <graph>.  Heuristic
selects (1) nodes with smallest clique sizes, and (2) within those
with smallest cliques selects among those with the fewest fill-ins.
Among those, it finds the first.  It then recursively calls
#'test-list to remove any new leaves."
  (if (null (graph-nodes graph)) nil
    (let* ((smallest-clique
	    (reduce #'min (mapcar #'(lambda (node)
				      (node-size (closure node graph)))
				  (graph-nodes graph))))
	   (possible-deletions
	    (remove-if-not #'(lambda (#1=#:x)
			       (eql smallest-clique
				    (node-size (closure #1# graph))))
			   (graph-nodes graph)))
	   (fewest-fillins
	    (reduce #'min
		    (mapcar #'(lambda (nod)
				(length (the list (fill-ins nod graph))))
			    possible-deletions)))
	   (next-deletion
	    (find-if #'(lambda (nod)
			 (eql fewest-fillins
			      (length (the list (fill-ins nod graph)))))
		     possible-deletions))
	   )
;;      (print smallest-clique)
;;      (print possible-deletions)
;;      (print fewest-fillins)
;;      (print next-deletion)	   
;;      (terpri)
      (cons next-deletion (test-list #'one-sc-tff (r-delete next-deletion graph))))))


;;; fill-ins -- find the list of fill-ins corresponding to selecting a
;;; point in a graph.
(defun fill-ins (node graph)
  (declare (type Graph graph) (type T node)
	   (:returns (type List fill-in)))
  "Finds the edges which would be filled in if <node> was eliminated
from <graph>."
  (remove-if #'(lambda (#1=#:x) (member #1# (graph-edges graph) :test #'subsetp))
	     (reduce-edge (neighbors node graph))))

;;; fill-length -- length of a fill-in  defined as the sum of the
;;; product of the sizes of the attributes
(defun fill-length (fill-ins &aux (finum 0))
  (declare (type List fill-ins) (type Fixnum finum)
	   (:returns (type Fixnum finum)))
  "Calculates the ``length'' of fill-in <fill-ins> using the size of
the product space."
  (dolist (fi fill-ins finum)
     (incf finum (* (length (the list (get (car fi) :values)))
		    (length (the list (get (cadr fi) :values)))))))



;;; fast-ff -- fast implementaion of the fewest-fill-ins one step look
;;; ahead method which does the leaf checking as part of the basic
;;; strategy.
(defun fast-ff (graph &aux fill-in-num ff-so-far)
  (declare (type Graph graph) (type Fixnum fill-in-num)
	   (:returns (type List pealing-order)))
  "Fast implementation of the fewest-fill-ins algorithm which depends
on the fact that a fill-in number of zero means a leaf."
  (if (endp (graph-nodes graph)) `() 
    (let ((next-deletion 
	   (dolist (node (graph-nodes graph) ff-so-far)
		   (let ((node-fill-in-num (fill-length (fill-ins node graph))))
		     (declare (fixnum node-fill-in-num))
		     (cond ((eql 0 node-fill-in-num)
			    (return node))
			   ((or (null fill-in-num) (< node-fill-in-num fill-in-num))
			    (setq fill-in-num node-fill-in-num ff-so-far node)))))))
      (cons next-deletion (fast-ff (r-delete next-deletion graph))))))



;;; one-ff -- finds core by one-step (fewest fill-ins) method
(defun one-ff (graph)
  (declare (type Graph graph) 
	   (:returns (type List pealing-order)))
  "Heuristic for choosing a deletion order of nodes from <graph>,
where it is assumed that there are no leaves in <graph>.  Heuristic
selects (1) those with the fewest fill-ins.
Among those, it finds the first.  It then recursively calls
#'test-list to remove any new leaves."
  (if (null (graph-nodes graph)) nil
    (let* ((fewest-fillins
	    (reduce #'min
		    (mapcar #'(lambda (nod)
				(length (the list (fill-ins nod graph))))
			     (graph-nodes graph))))
	   (next-deletion
	    (find-if #'(lambda (nod)
			 (eql fewest-fillins
			      (length (fill-ins nod graph))))
		     (graph-nodes graph))))
;;      (print fewest-fillins)
;;      (print (remove-if-not
;;	      #'(lambda (nod)
;;		  (eql fewest-fillins (length (fill-ins nod graph))))
;;	      (graph-nodes graph)))
;;      (terpri)
      (cons next-deletion (test-list #'one-ff (r-delete next-deletion
							graph))))))

;;; Property graphs --- equivalent to the representation of a simple
;;; graph as a list of nodes and a list of edges, one can represent a
;;; simple graph as a list of nodes and a list of neighbors of that
;;; graph.  This representation is used for the tree-of-cliques.
;;; These routines provide a method for transfering back and forth
;;; from that representation.  Note that the conversion from graph to
;;; property graph reprsentation will work on hypergraphs except that
;;; information about hyperedges will be lost, that is
;;; (prop-graph-2-graph (graph-2-prop-graph *hypergraph*)) will be a
;;; simple (2) graph.

;; prop-graph-2-graph -- a property-graph is a list of symbols with the
;; :neighbors.  In order to avoid duplication, the procedure
;; prop-graph-find-edges, (which works recursively) eliminates nodes
;; edges which should have been already recorded.
(defun prop-graph-2-graph (node-list)
  (declare (type List node-list)
	   (:returns (type Graph graph)))
  "Converts a property graph, a list of nodes (<node-list>) with
:neighbors properties, into an ordinary graph object."
  (make-graph :nodes node-list
	      :edges (prop-graph-find-edges node-list)) )

(defun prop-graph-find-edges (node-list)
  (declare (type List node-list)
	   (:returns (type List edge-list)))
  "Finds edges from neighborhood structure of list of nodes."
  (if (endp node-list) '()
    (nconc (mapcar #'(lambda (#1=#:x) (list (car node-list) #1#))
		   (intersection (get (car node-list) :neighbors)
				 (cdr node-list)))
	   (prop-graph-find-edges (cdr node-list)))))
	   

;; graph-2-prop-graph -- does the inverse operation.  
(defun graph-2-prop-graph (graph)
  (declare (type Graph graph)
	   (:returns (type List prop-graph)))
  "Converts an ordinary <graph> into a list of nodes with :neighbors
property.  Note assumes nodes of <graph> are symbols."
  (mapc #'(lambda (node)
	    (setf (get node :neighbors)
	      (neighbors node graph)))
	(graph-nodes graph)))



;;; provide when loaded
(bel-provide :graphs)


