;;; distprob -- This file has distribution function drawing information
;;; for Bayesian in it.


;;; Copyright 1989 Russell G. Almond
;;; License is granted to copy this program for education or research
;;; purposes, with the restriction that no portion of this program may
;;; be copied without also copying this notice.  All other rights
;;; reserved. 

;;; 7/21/89 -- Version 1.1 Created File

;;; 8/5/89 -- Added nominal feature to distribtions.  Similar to setup
;;; and draw it creates a "nominal" value of the distribution for
;;; pre-montecarlo analysis.  The nominal function returns two values
;;; one of which is the actual value and the other is either :param or
;;; :data depending on what was returned.

;;; The heart of the distribution feature is the creation of
;;; distribution function objects.  This file is meant to parallel
;;; another one called distbel which will handle similar cases for
;;; belief functions.

;(provide 'distprob)
(in-package :prob )
(bel-require :potentials "potentials")
(bel-require :probread "probread")
(bel-require :lowdist "lowdist")
;(use-package '(basic graphs sets utils))
;(export '(draw draw-default draw-interval draw-uniform draw-beta
;	       draw-gamma draw-mix draw-lognorm
;	  setup setup-default setup-interval setup-uniform setup-beta
;	       setup-gamma setup-mix setup-lognorm setup-all
;	  impute-poisson impute-poisson-data impute-poisson-disc
;	  impute-binomial impute-binomial-data 
;	  sink-get-att-value sink-get-param-value
;	  nominal nominal-defaul nominal-mix nominal-interval
;	          nominal-uniform nominal-beta nominal-gamma
;		  nominal-lognorm
;	  impute impute-mix
;	  )) 
;; draw, draw-default and draw-mix are inherited from lowdist
;; as are setup, setup-default, setup-mix and  setup-all
;; *parent-list* is inherited from lowdist



;;; Variables

;; *rng-calls-per-draw* is usually 1 for probabilities
(defvar *rng-calls-per-draw* 1 "Number of random numbers needed to
draw a random object:  usually 1 for probabilities")


;; *error-factor-scale* is 1 if error factor represents sdv in log
;; scale and is 1.96... if it represents 95% confidence interval
(defvar *error-factor-scale* 1 "Do error factors represent sdv in log
scale or 95% confidnence interval (use 1.96)." )

;; *sink-param-format*, *sink-att-format* -- format used to write
;; sinked node by sink-get-param-value and sink-get-att-value
(defvar *sink-param-format* "~14,6,2G" "Format for printing
sink values.")
(defvar *sink-att-format* "~S" "Format for printing sink attributes.")

(setf  will::*inherited-vars*
       (union '(*sink-param-format* *sink-att-format* *error-factor-scale*
		 *rng-calls-per-draw*)
	       will::*inherited-vars*))

;;; Setup and Draw functions for mode dependent types.  Draw (the basic
;;; function), draw-default and draw-mix are inherited from
;;; lowdist. (similarly for setups).

;;; Setups are generally calls to setup-distribution which generates
;;; rng data for that particular type, and return indexes to data
;;; which are stored in rng-data-1 slot.  Calls to draw then
;;; mostly call draw-dist with rng-data as the argument.

;;; intervals 

;; setup-interval -- nothing to be done
(defun setup-interval (dist)
  (declare (type Interval dist)
	   (:returns t))
  "Setup random number generator for fixed interval distribution <dist>.
   In prob mode, this is a fixed point distribution and uses only
upper bound.
  (dummy function for consistency)."
  t)					
    
;; draw-interval -- returns upper and lower bounds on the interval.
(defun draw-interval (dist)
  (declare (type Interval dist)
	   (:returns (type (List Float) interval)))
  "Draws random interval from fixed interval distribution <dist>.
   In prob mode, this is a fixed point distribution and uses only
upper bound."
  (list (interval-upper dist) ))


;; nominal-interval -- returns upper and lower bounds on the interval.
(defun nominal-interval (dist)
  (declare (type Interval dist)
	   (:returns (values (type (List Float) interval)
			     (type (member :data :param) method))))
  "Returns nominal value (the interval) of fixed interval distribuiton
<dist>.  Nominal value is parameters.
   In prob mode, this is a fixed point distribution and uses only
upper bound.
"
  (values (list (interval-upper dist) ) :param))




;;; uniforms

;; setup-uniform -- initialize to draw uniform in range upper to
;; lower
(defun setup-uniform (dist)
  (declare (type Uniform dist)
	   (:returns (type (member t nil) status)))
  "Sets up RNG calls for uniform generator for distribuiton <dist>."
  (setf (dist-rng-data-1 dist)
	(set-up-dist #\U (uniform-upper dist) (uniform-lower dist)))
  t)

;; draw-uniform -- retuns an interval based on drawing two
;; uniforms in the range upper to lower
(defun draw-uniform (dist)
  (declare (type Uniform dist)
	   (:returns (type (List Float) interval)))
  "Draws a random value from uniform distribution <dist>."
  (list (draw-dist (dist-rng-data-1 dist))))


;; nominal-uniform -- returns the midpoint
(defun nominal-uniform (dist)
  (declare (type Uniform dist)
	   (:returns (values (type (List Float) interval)
			     (type (member :data :param) method))))
  "Returns nominal of uniform distirubtion <dist>.  Nominal is
(upper + lower)/2." 
  (values (list (/ (+ (uniform-upper dist) (uniform-lower dist)) 2.0))
	  :param))



;;; beta -- univariate beta with T events in N trials including prior.
;;; Draws from a beta (T,N-T+1).

;; setup-beta -- initializes beta-rngs
(defun setup-beta (dist)
  (declare (type Beta-Dist dist)
	   (:returns (type (member t nil) status)))
  "Sets up RNG calls for a univairiate beta distribution <dist>." 
  (setf (dist-rng-data-1 dist)
	(set-up-dist #\B (beta-events dist)
			    (1+ (- (beta-trials dist) (beta-events dist)))))
  t)



;; draw-beta -- draws from a bivariate beta distribution based on X
;; success in N trials
(defun draw-beta (dist)
  (declare (type Beta-Dist dist)
	   (:returns (type (List Float) interval)))
  "Draws a random number from beta distribution <dist>."
  (list (draw-dist (dist-rng-data-1 dist))))

;; nominal-beta -- returns the mean
(defun nominal-beta (dist)
  (declare (type Beta-Dist dist)
	   (:returns (values (type (List Float) interval)
			     (type (member :data :param) method))))
  "Returns mode of beta distribution <dist> (event/trials)."
  (values (list (/ (beta-events dist) (beta-trials dist))) :param))



;;; Gamma -- univariate gamma is easy.  If the data is X events in t
;;; observation time, then draw G(X,t) 

;; setup-gamma -- initializes gamma-rngs
(defun setup-gamma (dist)
  (declare (type Gamma-Dist dist)
	   (:returns (type (member T NIL) status)))
  "Sets up RNG calls for gamma distribution <dist>."
  (setf (dist-rng-data-1 dist)
	(set-up-dist #\G (gamma-events dist) (gamma-time dist)))
  t)


;; draw-gamma -- draws from a bivariate gamma distribution based on X
;; success in N trials
(defun draw-gamma (dist)
  (declare (type Gamma-Dist dist)
	   (:returns (type (List Float) interval)))
  "Draws a random gamma value from <dist>."
  (list (draw-dist (dist-rng-data-1 dist))))

;; nominal-gamma -- returns the expected value
(defun nominal-gamma (dist)
  (declare (type Gamma-Dist dist)
	   (:returns (values (type (List Float) interval)
			     (type (member :data :param) method))))
  "Returns mode of gamma distribution <dist>. (events/time)"
  (values (list (/ (gamma-events dist) (gamma-time dist))) :param))

  

;;; Mixture -- handled in lowdist

;;; Lognormal -- currently supported only in prob mode (here)

;; setup-lognormal
(defun setup-lognorm (dist)
  (declare (type Lognormal-Dist dist)
	   (:returns (type (member T NIL) status)))
  "Sets up RNG for lognormal-distribution <dist>"
  (let ((ef (/ (lognorm-error-factor dist) *error-factor-scale*)))
    (setf (dist-rng-data-1 dist)
	(set-up-dist #\L (lognorm-offset dist)
			    (- (lognorm-median dist) (lognorm-offset dist))
			    (log (* ef ef)))))
  t)


;; draw-lognormal
(defun draw-lognorm (dist)
  (declare (type Lognormal-Dist dist)
	   (:returns (type (List Float) interval)))
  "Draws the lognormal distribution <dist>."
  (list (draw-dist (dist-rng-data-1 dist))))

;; nominal-lognormal -- returns the median
(defun nominal-lognorm (dist)
  (declare (type Lognormal-Dist dist)
	   (:returns (values (type (List Float) interval)
			     (type (member :data :param) method))))
  "Returns a nominal value for the log-normal distribution <dist>."
  (values (list (lognorm-median dist)) :param))



;;;; imputeing functions for constructing binomial and poisson
;;;; potentials from draw data or nominal values of paramters from
;;;; data (with priors introduced as pseudo-data)

;; impute -- takes three arguments, frame, data, data-type (:data, :param or
;; list) plus a required :model-type which can be either :binomial or
;; :poisson and dispatches to the appropriate imputation function.
(defun impute (frame data data-type &rest key-pairs
		     &key (model-type nil) &allow-other-keys)
  (declare (type List frame) (type List data)
	   (type (or  List (member :param :data)) data-type)
	   (type (member :binomial :poisson) model-type)
	   (:returns (type Val imputed-value)))
  "Dispatchs to appropriate binomial or poisson imputation function
based on: (1) whether <model-type> keyword arg (required) is :binomial
or :poisson and (2) whether or not <data-type> is :param(eters) :data or a list
(indicating a mixture).  <frame> is the frame of the new value and
<data> is the appropriate data or parameters.  In prob mode, currently
<data-type> of :data is never generated."
  (if (listp data-type) (apply #'impute-mix frame data data-type key-pairs)
    (apply (case model-type
		 (:binomial (case data-type
				(:param #'impute-binomial)
				(:data #'impute-binomial-data)))
		 (:poisson (case data-type
				 (:param #'impute-poisson)
				 (:data #'impute-poisson-data)))
		 (t (error "impute: unrecognized or unsupplied model-type ~S" model-type)))
	   frame data key-pairs)))

;;; mixture distribution -- use mix on the results of applying impute
;;; to the arguments.
(defun impute-mix (frame data data-types &rest key-pairs &key
			 (model-type nil) &allow-other-keys)
  (declare (list Frame) (list Data) (list Data-Types)
	   (type (member :binomial :poisson) model-type)
	   (:returns (type Val imputed-value)))
  "Imputes mixture of models by (1) applying #'impute to each value in
the second half of the data list and (2) mixing them together using the
first half."
  (mix (mapcar #'(lambda (datum type)
		   (apply #'impute frame datum type key-pairs))
	       (cadr data) data-types)
       (car data)))

;;; binomial

;;; impute-binomial -- imputes a binomial potential given the (a)
;; frame and (b) list of the probability of failure.
(defun impute-binomial (frame paramlist &key &allow-other-keys)
  (declare (type List frame) (type List paramlist)
	   (:returns (type Val imputed-value)))
  "Imputes a probability potential over a binary outcome given the <frame> and
<paramlist> which is a list of the probability of the
event (failure)."
  (let ((prob (coerce (car paramlist) 'long-float))
	(att (car frame)))
    (make-pot :frame frame :array (make-array 2 :element-type 'long-float
					      :initial-contents
					      (list prob (- 1.0 prob))))))

;;; impute-binomial-data -- imputes a binomial potential given (a) the
;;; frame and (b) #events in #trials
(defun impute-binomial-data (frame paramlist &key &allow-other-keys)
  (declare (type List frame) (type List data-list)
	   (:returns (type Val imputed-value)))
  "Imputes a probability potential over a binary outcome using <frame> as
frame and data <data-list> of the form (N events (failures) in T
trials).  Assume Uniform prior and uses MLE." 
  (let ((events (car paramlist))
	(trials (cadr paramlist))
	(att (car frame)))
    (make-pot :frame frame
	      :array (make-array 2 :element-type 'long-float
				 :initial-contents
				 (list (coerce (/ events trials) 'long-float) 
				       (coerce (/ (- trials events) trials)
					       'long-float))))))


;;; poisson data type -- poisson potentials are indexed from
;;; max-events+ to 1 (to be compatable with binomial types).  

;; impute-poisson -- crease poisson potentials given single poisson
;; failure rate
(defun impute-poisson (frame opr-data
			&key ((:rate obs-time) #!*default-rate-default*)
			(max-events #!*default-max-failures*)
			&allow-other-keys
			&aux (ms-form nil))
  (declare (type List frame) (type List opr-data)
	   (type Long-Float obs-time) (fixnum max-events)
	   (:returns (type Val imputed-value)))
  "Imputes a probability potential over a counted outcome (Poisson) given the
<frame> and <opr-time> which is a list of the occurance rate of the
event (failure).  :rate key argument <obs-time> gives the demand time on
the process.  <max-events> provides a maximum on the number of events."
  (declare (list frame) (list opr-data) (long-float obs-time)
	   (integer max-events))
  (let ((name (car frame))
	(failure-rate (car opr-data)))
    (make-pot :frame frame
	      :array
	      (make-array (1+ max-events) :element-type 'long-float
			  :initial-contents
			  (oprate-lams failure-rate obs-time max-events)))))

  



;; impute-poisson-disc -- meaningless, map to impute-poisson
(defun impute-poisson-disc (frame opr-data &rest key-pairs
				  &key ((:rate obs-time) #!*default-rate-default*)
				  (max-events #!*default-max-failures*)
				  &allow-other-keys)
  (declare (type List frame) (type List opr-data)
	   (type Long-Float obs-time) (fixnum max-events)
	   (:returns (type Val imputed-value)))
  "Imputes a probability potential over a counted outcome (Poisson) given the
<frame> and <opr-data> which is a list of the occurance rate of the
event (failure), during the special case where the lower bound is zero.
:rate key argument <obs-time> gives the demand time on the process.
<max-events> provides a maximum on the number of events.  Is an alias
for #'impute-poisson for consistancy with Belief mode." 
  (apply #'impute-poisson frame opr-data key-pairs))


;; impute-poisson-data -- data is in the form X events in T
;; trial-time.  calculate falure rate and call impute-poisson
(defun impute-poisson-data (frame data-list &rest key-pairs
				  &key ((:rate obs-time) #!*default-rate-default*)
				  (max-events #!*default-max-failures*)
				  &allow-other-keys)
  (declare (type List frame) (type List data-list)
	   (type Long-Float obs-time) (fixnum max-events)
	   (:returns (type Val imputed-value)))
  "Imputes a probability potential over a counted outcome (Poisson) given the
<frame> and <data-list> which is a list of the  data (X events in T time
of trial).  :rate key argument <obs-time>
gives the demand time on the process.  <max-events> provides a maximum
on the number of events." 
  (let ((events (car data-list)) (trial-time (cadr trial-time)))
    (apply #'impute-poisson frame (list (/ events trial-time)) key-pairs)))



;;; sink output-functions
;; two output functions are defined, one for printing the potentials
;; themselves and one for printing the value selected in a random draw
;; called by sink-node-handler in the shell package.

;; sink-get-param-value  -- prints the probablility of being true
(defun sink-get-param-value (value)
  (declare (type Pot value)
	   (:returns nil))
  "Prints the probability of true associated with <value> for
parameter values. 
Uses #!*sink-param-format* to format result."
  (format nil #!*sink-param-format*
	  (elt (pot-array value) 0)))	;this depends rather
					;intamately on the convention
					;that the first value in the
					;list is considered "true"

;; sink-get-att-value -- prints the value associated with the logical
;; value in value.  
(defun sink-get-att-value (value &aux (foundatt nil))
  (declare (type Pot value)
	   (:returns nil))
  "Prints the  probability of event associated with potential
<value> as a way of monitoring node values.
Uses #!*sink-att-format* to format result."
  (format nil #!*sink-att-format*
	  (dotimes (i (length (get (car (pot-frame values)) :values))
		      foundatt)
		   (unless (eql 0.0 (elt (pot-array values) i))
			   (if foundatt
			       (error "sink-att:Nonlogical sink for sink-att")
			     (setq foundatt
				   (elt (get (car (pot-frame values)) :values)
					i)))))))



;;; poisson pdf functions.  oprate-lam is the p.d.f. of a poisson
;;; process.  oprate-lam+ is 1-F(x-1) where F(x) is the c.d.f.

;; oprate-lam -- takes three arguments, a rate, a time and a count and
;; returns the poisson process p.f.
(defun oprate-lam (rate time x)
  (declare ( type Float rate time) (type Integer x)
	   (:returns (type Float p.m.f.)))
  "Using parameter <rate>*<time>, calculates the Poisson p.m.f. at <x>."
  (let ((the Float (lam (* rate time))))
    (declare (type Float lam))
    (coerce (/ (* (exp (- 0.0 lam)) (expt lam x))
		  (! x)) 'long-float)))


;; oprate-lam+ -- returns the sum of oprate lam from x-min to
;; infinity.  (actually if work via 1-sum from 0 to x-min)
(defun oprate-lam+ (rate time x-min)
  (declare (type Float rate time) (type Integer x-min)
	   (:returns (type Float 1-p.f.)))
  "Calculates 1-F(<x-min>-1) for Poisson distribution with parameter
<rate>*<time>."
  (let* ((lam (the Float (* rate time))))
    (declare (type Float lam))
    (do* ((x 1 (1+ x))
	  (x-fac 1 (* x x-fac))
	  (sum 1.0 (+ sum (/ (expt lam x) x-fac))))
	 ((eql x x-min) (- 1.0 (* (exp (- 0.0 lam)) sum)))
       (declare (integer x x-fac) (float sum)))))


;; oprate-lams -- builds a list of the pf corresponding to (iota
;; x-max)
(defun oprate-lams (rate time x-max)
  (declare (type Float rate time) (type Integer x-max)
	   (:returns (type (List Float) p.m.f.)))
  "Returns p.m.f. at 0 to <x-max>+ as a list.  Poisson distribution
with parameter <rate>*<time>."
  (let* ((lam (* rate time))
	 (elam (exp (- lam))))
    (declare (float lam elam))
    (do* ((x 1 (1+ x))
	  (x-fac 1 (* x x-fac))
	  (f (* elam lam) (/ (* elam (expt lam x)) x-fac))
	  (f-list (list f elam) (cons f f-list))
	  (sum (+ elam f) (+ sum f)))
	 ((eql x x-max) (reverse (rplaca f-list (+ f (- 1.0 sum)))))
      (declare (integer x x-fac) (float sum f) (list f-list)))))


;;; provide when loaded
(bel-provide :distprob)