;;; distbel -- This file has distribution function drawing information
;;; for belief functions in it.


;;; Copyright 1989 Russell G. Almond
;;; License is granted to copy this program for education or research
;;; purposes, with the restriction that no portion of this program may
;;; be copied without also copying this notice.  All other rights
;;; reserved. 

;;; 7/21/89 -- Version 1.1 Created File

;;; 8/5/89 -- Added nominal feature to distribtions.  Similar to setup
;;; and draw it creates a "nominal" value of the distribution for
;;; pre-montecarlo analysis.  nominal returns two values the first of
;;; which is the actual nominal values of the parameters or data, the
;;; second of which is either :param or :data depending on what the
;;; first one is.  (mix actually returns a list as its second value).

;;; The heart of the distribution feature is the creation of
;;; distribution function objects.  This file is meant to parallel
;;; another one called distprob which will handle similar cases for
;;; probabilities.

;(provide 'distbel)
(in-package :belief )
(bel-require :arithmatic "arithmatic")
(bel-require :read "read")
(bel-require :lowdist "lowdist")
;;(use-package '(basic graphs sets utils))
;(export '(draw draw-default draw-interval draw-uniform draw-beta
;	       draw-gamma draw-mix draw-lognorm
;	  setup setup-default setup-interval setup-uniform setup-beta
;	       setup-gamma setup-mix setup-lognorm setup-all
;	  impute-poisson impute-poisson-data impute-poisson-disc
;	  impute-binomial impute-binomial-data sink-get-att-value
;	  sink-get-param-value 
;	  nominal nominal-default nominal-mix nominal-interval
;	          nominal-uniform nominal-beta nominal-gamma
;		  nominal-lognorm
;	  impute impute-mix
;	  )) 
;; draw, draw-default and draw-mix are inherited from lowdist
;; as are setup, setup-default, setup-mix and  setup-all
;; *parent-list* is inherited from lowdist



;;; Variables

;; *rng-calls-per-draw* is usually 2 for belief functions (upper and
;; lower bound are rv's)
(defvar *rng-calls-per-draw* 2 "Number of random numbers needed to
draw a random object:  usually 2 for belief functions (upper and lower
bounds are rv's)")

;; *error-factor-scale* is 1 if error factor represents sdv in log
;; scale and is 1.96... if it represents 95% confidence interval
;; (primarily for prob mode)
(defvar *error-factor-scale* 1 "Do error factors represent sdv in log
scale or 95% confidnence interval.")

;; *sink-param-format*, *sink-att-format* -- format used to write
;; sinked node by sink-get-param-value and sink-get-att-value
(defvar *sink-param-format* "~14,6,2G ~14,6,2G" "Format for printing
sink values.")
(defvar *sink-att-format* "~S" "Format for printing sink attributes.")

(setf  will::*inherited-vars*
       (union '(*sink-param-format* *sink-att-format* *error-factor-scale*
		 *rng-calls-per-draw*)
	       will::*inherited-vars*))

       


;;; Setup and Draw functions for mode dependent types.  Draw (the basic
;;; function), draw-default and draw-mix are inherited from
;;; lowdist. (similarly for setups).

;;; Setups are generally calls to setup-distribution which generates
;;; rng data for that particular type, and return indexes to data
;;; which are stored in rng-data-1 and rng-data-2.  Calls to draw then
;;; mostly call draw-dist with rng-data as the argument.

;;; intervals 

;; setup-interval -- nothing to be done
(defun setup-interval (dist)
  (declare (type Interval dist)
	   (:returns t))
  "Setup random number generator for fixed interval distribution <dist>.
  (dummy function for consistency)."
  t)					
    
;; draw-interval -- returns upper and lower bounds on the interval.
(defun draw-interval (dist)
  (declare (type Interval dist)
	   (:returns (type (List Float) interval)))
  "Draws random interval from fixed interval distribution <dist>."
  (list (interval-upper dist) (interval-lower dist)))


;; nominal-interval -- returns upper and lower bounds on the interval.
(defun nominal-interval (dist)
  (declare (type Interval dist)
	   (:returns (values (type (List Float) interval)
			     (type (member :data :param) method))))
  "Returns nominal value (the interval) of fixed interval distribuiton
<dist>.  Nominal value is parameters."
  (values (list (interval-upper dist) (interval-lower dist)) :param))



;;; uniforms

;; setup-uniform -- initialize to draw two uniforms in range upper to
;; lower
(defun setup-uniform (dist)
  (declare (type Uniform dist)
	   (:returns (type (member t nil) status)))
  "Sets up RNG calls for uniform generator for distribuiton <dist>."
  (setf (dist-rng-data-1 dist)
	(set-up-dist #\U (uniform-upper dist) (uniform-lower dist)))
  (setf (dist-rng-data-2 dist)
	(set-up-dist #\U (uniform-upper dist) (uniform-lower dist)))
  t)

;; draw-uniform -- retuns an interval based on drawing two
;; uniforms in the range upper to lower
(defun draw-uniform (dist)
  (declare (type Uniform dist)
	   (:returns (type (List Float) interval)))
  "Draws a random interval from uniform distribution <dist>.  Method
draws two uniform random numbers and sorts them."
  (let* ((uni1 (draw-dist (dist-rng-data-1 dist)))
	 (uni2 (draw-dist (dist-rng-data-2 dist))))
    (declare (float uni1 uni2))
    (if (> uni1 uni2) (list uni1 uni2) (list uni2 uni1))))

;; nominal-uniform -- returns the range
(defun nominal-uniform (dist)
  (declare (type Uniform dist)
	   (:returns (values (type (List Float) interval)
			     (type (member :data :param) method))))
  "Returns nominal value of uniform distirubtion <dist>.  Nominal is
parameters (upper lower)." 
  (values (list (uniform-upper dist) (uniform-lower dist)) :param))


;;; beta -- bivariate beta with T events in N trials is decomposed
;;; into a B(T,N-T+1) and a rescaled B(1,N-T)

;; setup-beta -- initializes beta-rngs
(defun setup-beta (dist)
  (declare (type Beta-Dist dist)
	   (:returns (type (member t nil) status)))
  "Sets up RNG calls for a bivairiate beta (Dempster) distribution <dist>." 
  (setf (dist-rng-data-1 dist)
	(if (eql (beta-events dist) 0.0) -1
	  (set-up-dist #\B (beta-events dist)
		       (1+ (- (beta-trials dist) (beta-events dist))))))
  (setf (dist-rng-data-2 dist)
	(if (eql (beta-events dist) (beta-trials dist)) -1
	  (set-up-dist #\B 1.0 (- (beta-trials dist) (beta-events dist)))))
  t)



;; draw-beta -- draws from a bivariate beta distribution based on X
;; success in N trials
(defun draw-beta (dist)
  (declare (type Beta-Dist dist)
	   (:returns (type (List Float) interval)))
  "Draws a random interval from a dempster bivariate beta
distribution <dist>."
  (let ((bet1 (if (minusp (dist-rng-data-1 dist)) 0.0
		(draw-dist (dist-rng-data-1 dist))))
	(bet2 (if (minusp (dist-rng-data-2 dist)) 1.0
		  (draw-dist (dist-rng-data-2 dist)))))
    (declare (float bet1 bet2))
    (list (the float (+ bet1 (*  bet2 (- 1.0 bet1)))) bet1 )))



;; nominal-beta -- returns the data.
(defun nominal-beta (dist)
  (declare (type Beta-Dist dist)
	   (:returns (values (type (List Float) interval)
			     (type (member :data :param) method))))
  "Returns data used to derive beta distribution <dist>."
  (values (list (beta-events dist) (beta-trials dist)) :data))


;;; Gamma -- bivariate gamma is easy.  If the data is X events in t
;;; observation time, then it is formed by drawing a G(X,t) for the
;;; lower bound, and an independent G(1,t).  The upper bound is formed
;;; by the sum of the two random variables.

;; setup-gamma -- initializes gamma-rngs
(defun setup-gamma (dist)
  (declare (type Gamma-Dist dist)
	   (:returns (type (member T NIL) status)))
  "Sets up RNG calls for bivariate gamma (Almond) distribution <dist>."
  (setf (dist-rng-data-1 dist)
	(if (= 0.0 (gamma-events dist)) -1
	  (set-up-dist #\G (gamma-events dist) (gamma-time dist))))
  (setf (dist-rng-data-2 dist)
	(set-up-dist #\G 1.0 (gamma-time dist)))
  t)



;; draw-gamma -- draws from a bivariate gamma distribution based on X
;; success in N trials
(defun draw-gamma (dist)
  (declare (type Gamma-Dist dist)
	   (:returns (type (List Float) interval)))
  "Draws a random bivariate gamma interval from <dist>."
  (let ((gam1 (if (minusp (dist-rng-data-1 dist)) 0.0
		(draw-dist (dist-rng-data-1 dist))))
	(gam2 (draw-dist (dist-rng-data-2 dist))))
    (declare (float gam1 gam2))
    (list (the float (+ gam1 gam2)) gam1 )))

;; nominal-gamma -- returns the data
(defun nominal-gamma (dist)
  (declare (type Gamma-Dist dist)
	   (:returns (values (type (List Float) interval)
			     (type (member :data :param) method))))
  "Returns data used to derive bivariate gamma distribution <dist>."
  (values (list (gamma-events dist) (gamma-time dist)) :data))


;;; Mixture -- handled in lowdist

;;; Lognormal --- currently unsupported, provide error message.
;; setup-lognormal
(defun setup-lognorm (dist)
  (declare (type Lognormal-Dist dist)
	   (:returns (type (member T NIL) status)))
  "Sets up RNG for lognormal-distribution <dist>"
  (let ((ef (/ (lognorm-error-factor dist) *error-factor-scale*)))
    (setf (dist-rng-data-1 dist)
	(set-up-dist #\L (lognorm-offset dist)
			    (- (lognorm-median dist) (lognorm-offset dist))
			    (log (* ef ef)))))
  t)

;; draw-lognormal
(defun draw-lognorm (dist)
  (declare (type Lognormal-Dist dist)
	   (:returns (type (List Float) interval)))
  "Draws the lognormal distribution <dist>."
  (let ((upper (draw-dist (dist-rng-data-1 dist))))
    (list upper (* upper (lower-to-upper dist)))))


;; nominal-lognormal -- returns the median
(defun nominal-lognorm (dist)
  (declare (type Lognormal-Dist dist)
	   (:returns (values (type (List Float) interval)
			     (type (member :data :param) method))))
  "Returns a nominal value for the log-normal distribution <dist>."
  (let ((upper (lognorm-median dist)))
    (values (list upper (* upper (lower-to-upper dist))) :param)))




;;; imputation functions -- takes a drawn value of the parameters and
;;; imputes a belief function (potential) of the given type.

;; impute -- takes three arguments, frame, data, data-type (:data, :param or
;; list) plus a required :model-type which can be either :binomial or
;; :poisson and dispatches to the appropriate imputation function.
(defun impute (frame data data-type &rest key-pairs
		     &key (model-type nil) &allow-other-keys)
  (declare (type List frame) (type List data)
	   (type (or  List (member :param :data)) data-type)
	   (type (member :binomial :poisson) model-type)
	   (:returns (type Val imputed-value)))
  "Dispatchs to appropriate binomial or poisson imputation function
based on: (1) whether <model-type> keyword arg (required) is :binomial
or :poisson and (2) whether or not <data-type> is :param(eters) :data or a list
(indicating a mixture).  <frame> is the frame of the new value and
<data> is the appropriate data or parameters."
  (if (listp data-type) (apply #'impute-mix frame data data-type key-pairs)
    (apply (case model-type
		 (:binomial (case data-type
				  (:param #'impute-binomial)
				  (:data #'impute-binomial-data)))
		 (:poisson (case data-type
				 (:param #'impute-poisson)
				 (:data #'impute-poisson-data)))
		 (t (error "impute: unrecognized or unsupplied model-type ~S" model-type)))
	 frame data key-pairs)))


;;; mixture distribution -- use mix on the results of applying impute
;;; to the arguments.
(defun impute-mix (frame data data-types &rest key-pairs &key
			 (model-type nil) &allow-other-keys)
  (declare (list Frame) (list Data) (list Data-Types)
	   (type (member :binomial :poisson) model-type)
	   (:returns (type Val imputed-value)))
  "Imputes mixture of models by (1) applying #'impute to each value in
the second half of the data list and (2) mixing them together using the
first half."
  (mix (mapcar #'(lambda (datum type)
		   (apply #'impute frame datum type key-pairs))
	       (cadr data) data-types)
       (car data)))



;;; binomial

;; impute-binomial -- imputes a binomial belief function given the (a)
;; frame and (b) list of lower and upper probabilites of failure.

(defun impute-binomial (frame paramlist &key &allow-other-keys)
  (declare (type List frame) (type List paramlist)
	   (:returns (type Val imputed-value)))
  "Imputes a belief function over a binary outcome given the <frame> and
<paramlist> which is a list of the upper and lower probabilities of the
event (failure)."
  (let ((down (cadr paramlist))
	(up (car paramlist))
	(att (car frame)))
    (declare (float up down))
    (make-belief-function
     :frame frame
     :ms (list (make-m-value
		     :m down
		     :element (get-ps-set-sym
			       {list (list (logical-true att))}))
		    (make-m-value
		     :m (the float (- 1.0 up))
		     :element (get-ps-set-sym
			       {list (list (logical-false att))}))
		    (make-m-value :m (the float (- up down))
				  :element **frame**)))))

;; impute-binomial-data gives nominal values when data comes in the
;; form of (T trials in N failures).  
(defun impute-binomial-data (frame data-list &key &allow-other-keys)
  (declare (type List frame) (type List data-list)
	   (:returns (type Val imputed-value)))
  "Imputes a belief function over a binary outcome using <frame> as
frame and data <data-list> of the form (N events (failures) in T
trials)." 
  (let ((att (car frame))
	(fails (coerce (car data-list) 'long-float))
	(trials (coerce (cadr data-list) 'long-float)))
    (declare (float fails trials))
    (make-belief-function
     :frame frame
     :ms (list (make-m-value
		     :m (the float (/ fails (+ trials 1.0)))
		     :element (get-ps-set-sym
			       {list (list (logical-true att))}))
		    (make-m-value
		     :m (the float (/ (- trials fails) (+ trials 1.0)))
		     :element (get-ps-set-sym
			       {list (list (logical-false att))}))
		    (make-m-value :m (the float (/ 1.0 (+ trials 1.0)))
				  :element **frame**)))))


;;; impute-poisson -- takes a frame data in the form of a list of
;; (lower upper) for bounds on lambda, the observation time and the
;; maximum number of events to mode and creates a poisson belief
;; function.
(defun impute-poisson (frame opr-data &rest keypairs
			&key ((:rate obs-time) #!*default-rate-default*)
			(max-events #!*default-max-failures*)
			&allow-other-keys
			&aux (ms-form nil))
  (declare (type List frame) (type List opr-data)
	   (type Long-Float obs-time) (fixnum max-events)
	   (:returns (type Val imputed-value)))
  "Imputes a belief function over a counted outcome (Poisson) given the
<frame> and <opr-data> which is a list of the upper and lower rates of the
event (failure).  :rate key argument <obs-time> gives the demand time on
the process.  <max-events> provides a maximum on the number of events."
  (if (eql (cadr opr-data) 0.0)
      (apply #'impute-poisson-disc frame opr-data keypairs)
    (let ((name (car frame)))
      (dotimes (i max-events)
	(dotimes (j (the fixnum (- max-events i)))
	  (push (make-m-value :m (oprate-m opr-data obs-time i j)
			      :element (get-ps-set-sym
					(list (list (iota (the fixnum (+ i j)) i)))))
		ms-form))
	(if (zerop i)
	    (push (make-m-value :m (oprate-m-edge opr-data obs-time i max-events)
				:element **frame**)
		  ms-form)
	  (push (make-m-value :m (oprate-m-edge opr-data obs-time i max-events)
			      :element (get-ps-set-sym (list (list (iota max-events i t)))))
		ms-form)))
      (push (make-m-value :m (oprate-m-rest opr-data obs-time max-events)
			  :element (get-ps-set-sym
				    (list (list (iota max-events max-events t)))))
	    ms-form)
      (make-belief-function :frame frame :ms (colapse-ms-list ms-form)))))




;; impute-poisson-disc -- imputes a poisson belief function from a
;; one-sided (upper bounded ) poisson distribution
(defun impute-poisson-disc (frame opr-data
				  &key ((:rate obs-time) #!*default-rate-default*)
				  (max-events #!*default-max-failures*)
				  &allow-other-keys
				  &aux (ms-form nil))
  (declare (type List frame) (type List opr-data)
	   (type Long-Float obs-time) (fixnum max-events)
	   (:returns (type Val imputed-value)))
  "Imputes a belief function over a counted outcome (Poisson) given the
<frame> and <opr-data> which is a list of the upper and lower rates of the
event (failure), during the special case where the lower bound is zero.
:rate key argument <obs-time> gives the demand time on the process.
<max-events> provides a maximum on the number of events." 
  (let ((name (car frame)))
    (dotimes (i max-events)
	     (push (make-m-value :m (oprate-m-vac opr-data obs-time 0 i)
				 :element (get-ps-set-sym (list (list (iota i 0)))))
		   ms-form))
    (push (make-m-value :m  (oprate-m-edge-vac opr-data obs-time 0 max-events)
			:element **frame**)
	  ms-form))
  (make-belief-function :frame frame :ms ms-form))

  




;; oprate-m -- computes m-values for the poisson process in the range
;; q to q+p, from opinion about that process
(defun oprate-m (opr-data s q p)
  (declare (type (List Long-Float) opr-data)
	   (type Long-Float s) (type Fixnum q p)
	   (:returns (type Long-Float m-value)))
  "Calculates m-value of focal element $\\{q, q+1, ... , q+p\\}$.  <s>
is demand time on system and <opr-data> is bounds on the event rate."
  (let {(upper (car opr-data)) (lower (cadr opr-data))}
    (declare (type Long-Float upper lower))
    (if (zerop q)
	(/ (* (expt s p) (exp (* -1 s upper))
	      (expt (- upper lower)  p))
	   (! p))
      (/ (* (expt s (the fixnum (+ q p)))
	    (expt lower q)
	    (exp (* -1 s upper))
	    (expt (- upper lower) p))
	 (! q) (! p)))))

;; oprate-m-vac -- computes m-values for the poisson process in the range
;; q to q+p, from opinion about that process
(defun oprate-m-vac (opr-data s q p)
  (declare (type (List Long-Float) opr-data)
	   (type Long-Float s) (type Fixnum q p)
	   (:returns (type Long-Float m-value)))
  "Calculates m-value of focal element $\\{q, q+1, ... , q+p\\}$.  <s>
is demand time on system and <opr-data> is bounds on the event rate.
Assumes lower bound is zero."
  (let {(upper (car opr-data))}
    (declare (type Long-Float upper))
    (if (zerop q)
	(/ (* (expt s p) (expt upper p)
	      (exp (* -1 s upper)))
	   (! p))
      0)))



;; oprate-m-edge -- computes m-values for the poisson process in the range
;; q to r+ (where r+ is max-events to infinity).
(defun oprate-m-edge (opr-data s q r &aux (sum 0.0L0))
  (declare (type (List Long-Float) opr-data)
	   (type Long-Float s) (type Fixnum q r)
	   (type Long-Float sum)
	   (:returns (type Long-Float m-value)))
  "Calculates m-value of focal element $\\{q, q+1, ... , r+\\}$, where
r+ is *max-events* to infinity.  <s> is demand time on system and
<opr-data> is bounds on the event rate." 
  (let {(upper (car opr-data)) (lower (cadr opr-data))}
    (declare (long-float upper lower))
    (let
	((res
	  (if (zerop q)
	      (- (exp (* -1 s lower))
		 (* (exp (* -1 s upper))
		    (dotimes (i (the fixnum (- r q)) sum)
		       (incf sum (/ (* (expt (- upper
						lower)
					     i)
				       (expt s i))
				    (! i))))))
	    (/ (* (expt lower q) (expt s q)
		  (- (exp (* -1 s lower))
		     (* (exp (* -1 s upper))
			(dotimes (i (the fixnum (- r q)) sum)
			   (incf sum (/ (* (expt (- upper
						    lower)
						 i)
					   (expt s i))
					(! i)))))))
	       (! q)))))
      (if (minusp res) 0 res))))



;; oprate-m-edge-vac -- computes m-values for the poisson process in the range
;; q to r+ (where r+ is max-events to infinity), for the vacuous bound
;; on the beleif function
(defun oprate-m-edge-vac (opr-data s q r &aux (sum 0.0L0))
  (declare (type (List Long-Float) opr-data)
	   (type Long-Float s) (type Fixnum q r)
	   (type Long-Float sum)
	   (:returns (type Long-Float m-value)))
  "Calculates m-value of focal element $\\{q, q+1, ... , r+\\}$, where
r+ is *max-events* to infinity  <s> is demand time on system and
<opr-data> is bounds on the event rate.  Assumes lower bound is zero." 
  (let {(upper (car opr-data))}
    (declare (long-float upper))
    (if (zerop q)
	(- 1
	   (* (exp (* -1 s upper))
	      (dotimes (i (the fixnum (- r q)) sum)
		 (incf sum (/ (* (expt upper i)
				 (expt s i)) (! i))))))
      0)))


;; oprate-m-rest -- computes m-values for the poisson process for the outcome
;; r+ (where r+ is max-events to infinity).
(defun oprate-m-rest (opr-data s r &aux (result 0L0))
  (declare (type (List Long-Float) opr-data)
	   (type Long-Float s) (type Fixnum r)
	   (type Long-Float result)
	   (:returns (type Long-Float m-value)))
  "Calculates m-value of focal element <r>+ where
r+ is *max-events* to infinity  <s> is demand time on system and
<opr-data> is bounds on the event rate." 
  (let {(upper (car opr-data)) (lower (cadr opr-data))}
    (declare (long-float upper lower))
    (- 1 (* (exp (* -1 s lower))
	    (dotimes (i r result)
	       (incf result
		     (/ (* (expt s i)
			   (expt lower i))
			(! i))))))))




;;; poisson given data

;; impute-poisson-data -- imputes a nominal poisson belief function
;; given data in the form X events in T time of trial.
(defun impute-poisson-data (frame poi-data 
			    &key ((:rate obs-time) #!*default-rate-default*)
			    (max-events #!*default-max-failures)
			    &allow-other-keys
			    &aux (ms-form nil))
  (declare (type List frame) (type List poi-data)
	   (type Long-Float obs-time) (fixnum max-events)
	   (:returns (type Val imputed-value)))
  "Imputes a belief function over a counted outcome (Poisson) given the
<frame> and <poidata> which is a list of the  data (X events in T time
of trial).  :rate key argument <obs-time>
gives the demand time on the process.  <max-events> provides a maximum
on the number of events." 
  (let ((name (car frame))
	(events (car poi-data))
	(trial-time (cadr poi-data)))
    (dotimes (i max-events)
      (dotimes (j (the fixnum (- max-events i)))
	(let ((m (poisson-m events trial-time obs-time i j)))
	  (unless (zerop m)
	     (push (make-m-value :m m
				 :element (get-ps-set-sym
					   (list (list
						  (iota (the fixnum (+ i j)) i)))))
		   ms-form))))
      (if (zerop i)
	  (let ((m (poisson-m-edge events trial-time obs-time i max-events)))
	    (unless (zerop m)
	       (push (make-m-value :m m :element **frame**) ms-form)))
	(let ((m (poisson-m-edge events trial-time obs-time i max-events)))
	  (unless (zerop m)
	     (push (make-m-value :m m
				 :element (get-ps-set-sym
					   (list (list (iota max-events i t)))))
		   ms-form)))))
    (let ((m (poisson-m-rest events trial-time obs-time max-events)))
      (unless (zerop m)
	 (push (make-m-value :m m
			     :element (get-ps-set-sym
				       (list (list (iota max-events max-events t)))))
	       ms-form))))
  (make-belief-function :frame frame :ms ms-form))


;; poisson-m -- computes m-values for the poisson process in the range
;; q to q+p.
(defun poisson-m (events trial-time s q p)
  (declare (type Number events) (type Float trial-time)
	   (type Long-Float s) (type Fixnum q p)
	   (:returns (type Long-Float m-value)))
  "Calculates m-value of focal element $\\{q, q+1, ... , q+p\\}$.  <s>
is demand time on system and <events> and <trial-time> are data from the
same process."
  (let ((evnt (truncate events)))
    (declare (fixnum evnt))
    (/ (* (expt s (the fixnum (+ q p)))
	  (expt trial-time (the fixnum (1+ evnt)))
	(binomial (the fixnum (+ evnt q -1)) q))
     (expt (+ s trial-time)
	   (the fixnum (+ q p evnt 1))))))

;; poisson-m-edge -- computes m-values for the poisson process in the range
;; q to r+ (where r+ is max-events to infinity).
(defun poisson-m-edge (events trial-time s q r)
  (declare (type Number events) (type Float trial-time)
	   (type Long-Float s) (type Fixnum q r)
	   (:returns (type Long-Float m-value)))
  "Calculates m-value of focal element $\\{q, q+1, ... , r+\\}$,  where
r+ is *max-events* to infinity.  <s> is demand time on system and
<events> and <trial-time> are data from the same process."
  (let ((evnt (truncate events)))
    (declare (fixnum evnt))
    (/ (* (expt s r)
	  (expt trial-time  evnt)
	  (binomial (the fixnum (+ evnt q -1)) q))
       (expt (+ s trial-time)
	     (the fixnum (+ r evnt))))))

;; poisson-m-rest -- computes m-values for the poisson process for the outcome
;; r+ (where r+ is max-events to infinity).
(defun poisson-m-rest (events trial-time s r &aux (result 1.0))
  (declare (type Number events) (type Float trial-time)
	   (type Long-Float s) (type Fixnum q r)
	   (type Float result)
	   (:returns (type Long-Float m-value)))
  "Calculates m-value of focal element r+,  where
r+ is *max-events* to infinity.  <s> is demand time on system and
<events> and <trial-time> are data from the same process."
  (let ((evnt (truncate events)))
    (declare (fixnum evnt))
    (dotimes (i r result)
       (setq result
	     (- result
		(/ (* (expt s i)
		      (expt trial-time  evnt)
		      (binomial (the fixnum (+ evnt i -1)) i))
		   (expt (+ s trial-time)
			 (the fixnum (+ i evnt)))))))))



	   
;;; sink output-functions
;; two output functions are defined, one for printing the belief functions
;; themselves and one for printing the value selected in a random draw
;; called by sink-node-handler in the shell package.

;; sink-get-param-value  -- prints the belief and plausibility of
;; being true.
(defun sink-get-param-value (value)
  (declare (type Belief-Function value)
	   (:returns nil))
  "Prints the belief and plausibility of <value> for parameter values.
Uses #!*sink-param-format* to format result."
  (let ((att (car (belief-function-frame value)))
	(m-true 0.0) (m-false 0.0) (m-frame 0.0))
    (declare (float m-true m-false m-frame))
    (dolist (m-val (belief-function-ms value))
	    (if (eq (m-value-element m-val) **frame**)
		(setq m-frame (m-value-m m-val))
	      (if (equal (ps-set-sym-val (m-value-element m-val))
			 {list (list (logical-false att))})
		  (setq m-false (m-value-m m-val))
		(setq m-true (m-value-m m-val)))))
    (format nil #!*sink-param-format*
	    m-true (the float (+ m-true m-frame)))))
  
;; sink-get-att-value -- logical belief function prints the only m-value
(defun sink-get-att-value (value)
  (declare (type Belief-Function value)
	   (:returns nil))
  "Prints the mass value for the first focal elment of belief function
<value> as a way of monitoring node values.
Uses #!*sink-att-format* to format result."
  (when (not (endp (cdr (belief-function-ms value))))
	(error "sink-att: Non logical belief function")
  (format nil #!*sink-att-format*
	  (ps-set-sym-val (m-value-element (car (belief-function-ms value)))))))


  

;;; provide when loaded
(bel-provide :distbel)
