;;; computations.lisp -- This series of programs is meant to compute
;;; the total belief functions for a marginal network.

;;; Copyright 1986 Russell G. Almond and Augustine Kong
;;; License is granted to copy this program for education or research
;;; purposes, with the restriction that no portion of this program may
;;; be copied without also copying this notice.  All other rights
;;; reserved. 

;;; 12/4/87 Modified to no longer give verbose output on do-it

;;; 7/18/89 Version 1.1 -- New Improved Shell:
;;; 	* Split the functionality of many parts into pieces[w]
;;;	* Allows alternative projection and combination rules[w]
;;;	* Allow re-loading of rule bases without redoing tree[w]
;;;	* Hook for Alternate projection and combination operations[w]

;;; 2/23/92 Version 1.2 -- Commented to definitions specifications.

;(provide 'computations)
(in-package :shell )
(bel-require :belief-init "belief-init")
(bel-require :graphs "graphs")
(bel-require :utils "utils")
(bel-require :machine-utils "bymachine")
;(use-package '(utils basic graphs inherited))
;(export '(process-node sum-over ppconflict do-it make-tree
;		       set-up-tree change-node update-tree
;		       augment-tree build-tree 
;		       conflict-node marginal-node redo-it
;		       undo-it undo-it! restore-node ppnode ppmessages
;		       ppall *explore-margins*
;		       set-it-up propagate-up propagate-down
;		       in-to-root *pick-root*
;		       find-new-values kill-all-messages
;		       mark-all-messages
;		       reload-tree @@+ @@+2 @@->
;		       ))



;;; These routines deal with computing the global belief function from
;;; the tree of cliques.  The algorithm works as follows.

;;; 1) The nodes of the tree are maded into a layed list.

;;; 2) Each node in the tree (from outer to inner layers):
;;;  a) checks to make sure that all nodes but one have passed
;;;  messages to it.
;;;  b) combines over all belief functions except that one.  
;;;  c) passes a message to that node with the message the belief
;;;  function projected onto the corresponding frame.
;;;  d) marks that message as being sent.

;;; 3) Each node in the tree (from inner to outer layers):
;;;  a) checks to make sure that all nodes have passed messages to it.
;;;  b) for each neighbor to which it has not sent a message, it
;;;  combines over all of the other belief functions and passes the
;;;  result to that node.

;;; In all cases, improper belief functions are used.  This allows us
;;; to asses the degree of conflict in any direction.

;;; A node is a symbol.
;;; Its value (required) is a belief-function (or potential) (could be vacuous).
;;; It has the neighbor property which gives a list of neighbors
;;; It has the sent property which is a list of neighbors to
;;; which it has sent messages.
;;; It has the received property which is an association list of
;;; neighbors and the belief functions they have sent.

;;; Control variables -Version 1.1

;; *save-messages* -- If non-nil old messages are retained, if nil old
;; messages are deleted and overwritten as soon as possible.  Should
;; be nil for simulations.
(defvar sets::*save-messages* t "Save changes to messages")

;; *pick-root* -- should root be chosen manually or automaticly
(defvar *pick-root* nil "Prompt for choice of root on do-it")


;;;; Functions


;;; process-node -- This does the basic operations for each node.
(defun process-node (node)
  (declare (type Symbol node)
	   (:returns nil))
  "Causes a node to compute and send messages to all of its
neighbors." 
  (map nil #'(lambda (neighbor) (compute-and-send node neighbor))
	(get node :neighbors)))

;;; compute-and-send -- This checks to see if the node is ready to
;;; send a message to its neighbor and then computes and sends that
;;; message. 
(defun compute-and-send (node neighbor)
  (declare (type Symbol node neighbor)
	   (:returns (type (or nil T) sent?)))
  "Checks to see if <node> is ready to send to <neighbor> and if so,
computes message and sends it."
  (when (and (not (member neighbor (get node :sent) ))
	     (ready-to-send node (remove neighbor (get node :neighbors))))
	(send-message neighbor node
		      (sum-over node
				(remove neighbor (get node :neighbors))))
	(push neighbor (get node :sent))))

		      

;;; ready-to-send -- Checks to see if we are ready to send a message
;;; by checking each neighbor in the list to see if we have received a
;;; message from it.
(defun ready-to-send (node neighbor-list)
  (declare (type Symbol node) (type List neighbor-list)
	   (:returns (type (member T NIL) ready?)))
  "Checks to see if <node> is ready to send by checking that a message
has been recieved from each node in <neighbor-list>."
  (cond ((endp neighbor-list) t)
	((assoc (car neighbor-list) (get node :received))
	 (ready-to-send node (cdr neighbor-list)))
	(t nil)))

;;; sum-over -- Sums over the belief functions received by the nodes
;;; in neighbor-list.  Result is an improper belief function.
;;; changed to ignore nil messages (for doing the right thing after
;;; partial propagation.)
(defun sum-over (node neighbor-list &aux mess)
  (declare (type Symbol node) (type List neighbor-list)
	   (:returns (type Val sum)))
  "Sums of the values received by <node> from the nodes in
<neighbor-list>, using the value of <node> as the base value."
  (cond ((endp neighbor-list) (symbol-value node))
	((setq mess (cdr (assoc (car neighbor-list)
				  (get node :received))))
	 (@@+2 mess
		(sum-over node (cdr neighbor-list))))
	(t (sum-over node (cdr neighbor-list)))))


;;; send-message--sends a message to the neighboring node
(defun send-message (neighbor node val)
  (declare (type Symbol neighbor node)
	   (type Val val))
  "Sends message <val> from <node> to <neighbor>."
  (push (cons node
	      (@@-> (val-frame (eval neighbor))
		   val))
	(get neighbor :received)))


;;; The updating problem. These routines deal with the problem of
;;; updating the tree of cliques after the node has changed.

;;; All we need to do, is mark all of the appropriate messages as
;;; unsent, then build a new layered list with the neighbors of the
;;; changed node at the top.

;;; change-node -- changes the belief function on a node.
(defun change-node (node new-val)
  (declare (type Symbol node) (type Val new-val)
	   (:returns (type Val new-val)))
  "Changes local value of <node> to <new-val>.  Pushes old value of
<node> on nodes :original-val stack."
  (push (eval node) (get node :original-val))
  (set node new-val))

;;; restore-node -- undoes change-node
(defun restore-node (node)
  (declare (type Symbol node))
  "Undoes change-node."
  (if (get node :original-val) (set node (pop (get node :original-val)))
    (eval node))
  (setf (get node :received) (pop-messages (get node :received))))


(defun pop-messages (message-list)
  (declare (type List message-list))
  "Pops messages off <message-list>. 
<message-list> should look like ((Name1 . value1) (Name2 . value2)
... (Namen . valuen) (Name1 . nil) (Name2 . nil) ... (NameN . nil)
 (Name1 . value1a) ... )  Want to peal off both values and nils. "
  (cond ((endp message-list) nil)
	((null (cdr (car message-list)))
	 (pop-messages-aux message-list)) 	; Now get the nils
	(t (pop-messages (cdr message-list)))))

(defun pop-messages-aux (message-list)
  (declare (type List message-list))
  "Pops the <nil> messages off <message-list>."
  (cond ((endp message-list) nil)
	((cdr (car message-list)) message-list) ; Done
	(t (pop-messages-aux (cdr message-list)))))


(defun restore-node! (node)
  (declare (type Symbol node))
  "Undoes all change-values." 
  (if (get node :original-val)
      (progn (set node (car (last (get node :original-val))))
	     (setf (get node :original-val) nil))
    (eval node))
  (setf (get node :received) (remove-duplicates (get node :received))))




;;; update-tree -- updates the tree of cliques starting from the given
;;; (just changed) node.
(defun update-tree (node)
  (declare (type Symbol node)
	   (:returns (type List update-list)))
  "Propagates changes from a just changed <node>.  As a side effect,
returns <update-list> of nodes."
  (setf (get node :sent) nil)
  (mapc #'process-node (cons node (build-update-list node))))

;;; build-update-list -- builds an update list for all of the
;;; neighbors of a given node, marking the neighbors as being not
;;; received from as well.
(defun build-update-list (node)
  (declare (type Symbol node)
	   (:returns (type List update-list)))
  "Constructs a sequence of nodes which must fire to globally
propagate change in <node>."
  (let ((neighbor-list 
	 (remove-if #'(lambda (#1=#:x) (member #1# (get node :sent)))
		    (get node :neighbors))))
    (append neighbor-list
	    (apply #'append (mapcar #'(lambda (#1#)
					(mark-and-build-update node #1#))
				    neighbor-list)))))


;;; mark-and-build-update -- marks child node appropriately, and updates
;;; and builds an update list for it.
(defun mark-and-build-update (parent-node child-node)
  (declare (type Symbol parent-node child-node)
	   (:returns (type List update-neighbors)))
  "Marks message from <child-node> to <parent-node> as invalid.
Return list of neighbors of <parent-node> which must have messages
invalidated." 
  (setf (get child-node :sent) (list parent-node))
  (if #!*save-messages*
      (push (cons parent-node nil) (get child-node :received))
    (setf (get child-node :received)
	  (remove parent-node (get child-node :received) :key #'car)))
  (build-update-list child-node))


;;; in-to-root -- creates a firing order for the nodes with a given
;;; root updates-value of margin-list
(defun in-to-root (root)
  (declare (type Symbol root)
	   (:returns (type List firing-list)))
  "Creates a firing order with <root> last."
  (unless (get root :neighbors)
	  (error "in-to-root: ~S is not a node" root))
  (set #?*root* root)
  (map nil #'(lambda (#1=#:x) (setf (get #1# :root-towards) nil))
       #!*margin-list*)
  (set #?*margin-list* (reverse (cons root (build-rooted-list root)))))
  
(defvar sets::*root* nil)

;; build-rooted-list -- builds an update list for all of the
;; neighbors of a given node, 
(defun build-rooted-list (node)
  (declare (type symbol node)
	   (:returns (type List update-list)))
  "Builds an update list for all of the neighbors of <node>.  Returns
a list of all nodes which must pass through <node> to get to root."
  (let* ((parent-node (get node :root-towards))
	 (neighbor-list 
	 (remove parent-node (get node :neighbors))))
    (append neighbor-list
	    (apply #'append (mapcar #'(lambda (child-node)
					(setf (get child-node :root-towards) node)
					(build-rooted-list child-node))
				    neighbor-list)))))

    
	


;;; build-tree -- builds a tree of cliques from the connectivity
;;; graph, the list of nodes should be in the proper deletion order
;;; (see layered-list in graphs.)
(defun build-tree (node-list graph)
  (declare (type List node-list) (type Graph graph)
	   (:returns (type Graph tree-of-cliques)))
  "Builds tree of cliques for <graph> by the Kong/Almond peeling
procedure, using <node-list> as the peeling order."
  (if (endp node-list) (make-graph)
    (let* ((node (car node-list))
	  (old-tree (build-tree (cdr node-list)
				 (r-delete node graph))))
      (if (null (graph-nodes old-tree))
	  (make-graph :nodes (list (closure node graph))
		      :edges nil)
	(if (member (neighbors node graph) (graph-nodes old-tree)
		    :test #'equal-set)
	    (make-graph :nodes (nsubstitute (closure node graph)
					    (neighbors node graph)
					    (graph-nodes old-tree)
					    :test #'equal-set)
			:edges (mapcar
				#'(lambda (edge)
				    (nsubstitute (closure node graph)
						 (neighbors node graph)
						 edge
						 :test #'equal-set))
				(graph-edges old-tree)))
	  ;else
	  (make-graph :nodes (cons (closure node graph)
				   (graph-nodes old-tree))
		      :edges (cons (list (closure node graph)
					 (find (neighbors node graph)
					       (graph-nodes old-tree)
					       :test #'subsetp))
				   (graph-edges old-tree))))))))

;;; augment-tree -- the tree of cliques is not guarenteeded to contain
;;; nodes for all of the nodes which have belief functions attached.
;;; augment-tree attaches all of the nodes in the node list to the
;;; tree.  It tries to find the smallest node to attach the new node
;;; to that contains a common element.  augment-tree modifies tree
(defun augment-tree (tree node-list)
  (declare (type Graph tree) (type List node-list)
	   (:returns (type Graph tree)))
  "Augments <tree> of cliques by adding nodes in <node-list>.
Destructively modifies <tree> argument."
  (cond ((null node-list) tree)
	((member (car node-list) (graph-nodes tree) :test #'equal-set)
	 (augment-tree tree (cdr node-list)))
	(t (let ((candidate-nodes (remove (car node-list) (graph-nodes tree)
					  :test-not #'subsetp)))
	     (if (null candidate-nodes)
		 (error "Node ~s is not adjacent to the tree" (car node-list)))
	     (push (car node-list) (graph-nodes tree))
	     (push (list (car node-list)
			 (find (reduce #'min (mapcar #'node-size candidate-nodes))
			       candidate-nodes :key #'node-size))
		   (graph-edges tree)))
	   (augment-tree tree (cdr node-list)))))




;;; set-up-tree -- this function transforms a tree from structure
;;; graph form to list of nodes form adding belief functions as it
;;; goes. 
(defun set-up-tree (tree)
  (declare (type Graph tree)
	   (:returns (type List #!*margin-list*)))
  "Transforms <tree> into list of nodes *margin-list* which form the
tree of cliques control structure."
  (if (null (graph-nodes tree)) nil
    (let* ((node (car (graph-nodes tree)))
	   (node-name (find-val node))
	   (node-list (set-up-tree (s-2-delete node tree)))
	   (neighbor-list (mapcar #'find-val
				  (neighbors node tree))))
      (setf (get node-name :neighbors) neighbor-list)
      (map nil #'(lambda (#1=#:x) (push node-name (get #1# :neighbors)))
	    neighbor-list)
      (setf (get node-name :sent) nil)
      (setf (get node-name :received) nil)
      (cons node-name node-list))))



;;; find-val -- this routine finds a belief function corresponding
;;; to the given list of nodes.  If there is none, then one is
;;; invented (the vacuous belief function over the given frame).
(defun find-val (frame)
  (declare (type List frame)
	   (:returns (type Val frame-value)))
  "Finds the value associated with <frame>.  If none is found, a unit
(vacuous) value is created over <frame> and returned."
  (cond ((car (rassoc frame #!*val-list* :test #'equal-set)))
	(t (let ((name (add-count (intern
				   (apply #'concatenate
					  (cons 'string
						(mapcar #'symbol-name frame)))
				   *rules-package*)
				  *rules-package*)))
	     (export name *rules-package*)
	     (set name (make-vacuous frame))
	     (set #?*val-list* (acons name frame #!*val-list*))
	     name))))


	 
;;; conflict-node -- 
(defun conflict-node (node)
  (declare (type Symbol node)
	   (:returns (type List conflict-alist)))
  "Calulate the amount of conflict in <node>.
Returns an alist of the form (neighbor . conflict)"
  (mapcar #'(lambda (neighbor &aux mess)
	      (cons neighbor
		    (if (setq mess (cdr (assoc neighbor
					  (get node :received))))
			(-><- mess)
		      nil)))
	  (get node :neighbors)))


;;; marginal-node -- calculate the marginal belief function (or potential)
;;; associated with a given node.
(defun marginal-node (node)
  (declare (type Symbol node)
	   (:returns (type Val marginal)))
  "Returns the marginal value over the frame associated with <node>."
  (|| (sum-over node (get node :neighbors))))

;;; include-nodes -- from *model-graph* builds up a list of nodes to
;;; include.   That is, it adds all edges of *model-graph* to the tree
;;; of cliques (if they are not already present).  If the variable,
;;; *explore-margins* is non-nil, then each attribute is also added to
;;; the tree of cliques as a singleton node.
(defun include-nodes (graph)
  (declare (type Graph graph)
	   (:returns (type List more-nodes)))
  "Returns a list of edges of <graph> to be added as nodes to the tree
of cliques.  If *explore-marginss* is non-nil, then nodes of <graph>
are added as singleton nodes as well."
  (if *explore-margins*
      (append (graph-edges graph) (mapcar #'list (graph-nodes graph)))
    (graph-edges graph)))

(defvar *explore-margins* t
  "If set to t the nodes for all attributes in model-graph are added
in tree of cliques" )

;;; the list of all of the nodes in the tree of cliques, this should
;;; be particular to each rule package, but is independent of mode,
;;; therefore it is placed as an external symbol in the sets package
;;; where it will be exported to both the belief and prob packages
(defvar sets::*margin-list* nil "Nodes in the tree of cliques.  Access
through #! and set through #?")
(export sets::*margin-list* 'sets)
		  
;;; do-it -- does the computation on *model-graph*      
(defun do-it ()
  (declare (:returns (type List #!*margin-list*)))
  "Sets up tree of cliques and propagates in both directions."
  (set-it-up)
  (propagate-up)
  (propagate-down)
  (format t "Global conflict: ~10,8F~%"
	  (-><-
	   (sum-over (car #!*margin-list*)
		     (get (car #!*margin-list*) :neighbors))))
  #!*margin-list*)


		  
;;; Subsets of the do-it functionality

;; set-it-up -- Builds and sets up the tree of cliques.
(defun set-it-up ()
  (declare (:returns (type Graph tree-o-cliques)))
  "Sets up tree of cliques and #?*margin-list*"
  (let* ((tree-o-cliques
	  (augment-tree
	   (build-tree (layered-list #!*model-graph*) #!*model-graph*)
	   (include-nodes #!*model-graph*))))
    (set #?*margin-list*
	  (mapcar #'find-val (graph-nodes tree-o-cliques)))
    (set-up-tree tree-o-cliques)
    (if *pick-root* (in-to-root (pick-root tree-o-cliques))
      (set #?*root* (car (last #!*margin-list*))))
    tree-o-cliques))


;;; Note: propagate from root on way down, not on way up!

;; propogate-up
(defun propagate-up (&optional (root nil))
  (declare (type Symbol root) (:returns nil))
  "Propagates forward to #!*root* node.  At the end of this 1/2
propagation, only #!*root* is guarenteed to have a consistant value." 
  (if root (in-to-root root))
  (map nil #'process-node (butlast #!*margin-list*))
  ;forward to root of tree
  )

;; propagate-down
(defun propagate-down ()
  (declare (:returns nil))
  "Finishes propagation back from root.  Assumes propagate-up has
recently been run."
  (map nil #'process-node (reverse #!*margin-list*)))


;;;ppall -- does what do used to do by default, pp all nodes
(defun ppall ()
  (declare (:returns (type List #!*margin-list*)))
  "Pretty prints all nodes."
  (mapc #'ppnode #!*margin-list*))


;;; redo-it -- change a single value for a node.
(defun redo-it (node val)
  (declare (type Symbol node) (type Val val)
	   (:returns (type List #!*margin-list*)))
  "Changes value of <node> to <val> and propagates changes."
  (change-node node val)
  (update-tree node)
  (format t "Global conflict: ~10,8F~%"
	  (-><-
	     (sum-over node (get node :neighbors))))
  #!*margin-list*)

;;; undo-it -- undoes changes
(defun undo-it ()
  "Undoes last changes and repropagates if necessary."
    (map nil #'restore-node #!*margin-list*)	;restore original values
    (unless #!*save-messages*
	    (propagate-up)
	    (propagate-down))
    )
  
(defun undo-it! ()
  "Undoes all changes and repropagates if necessary."
    (map nil #'restore-node! #!*margin-list*)	;restore original values
    (unless #!*save-messages*
	    (propagate-up)
	    (propagate-down))
    )



;;; Functions involved with reloading a new tree of cliques.
;;find-new-values -- loads all of the cliques
(defun find-new-values ()
  (declare (:returns nil))
  "Finds new values for each node in tree of cliques."
  (map nil #'find-val #!*margin-list*))

;;;kill-all-messages -- resets sent and received fields on all nodes
(defun kill-all-messages ()
  (declare (:returns nil))
  "Resets send and received fields on all nodes."
  (map nil #'(lambda (node) (setf (get node :sent) nil
				  (get node :received) nil))
       #!*margin-list*))

(defun mark-all-messages ()
  (declare (:returns nil))
  "Marks send and received fields as unsent on all nodes."
  (map nil #'(lambda (node)
	       (setf (get node :sent) nil)
	       (map nil #'(lambda (neighbor)
			    (push (list neighbor) (get node :received) ))
		    (get node :neighbors)))
       #!*margin-list*))


;;reload-model -- reload all of the nodes and resets them.
(defun reload-model ()
  (declare (:returns nil))
  "Kills all the messages and then reloads the values."
  (kill-all-messages)
  (find-new-values))

  
;;; ppnode -- prints out the information about a node
(defun ppnode (node)
  (declare (type Symbol node)
	   (:returns nil))
  "Pretty prints the conflict and value of <node>.  <node> should
evaluate to a symbol."
  (format t "~%Node: ~S ~%" node)
  (ppconflict (conflict-node node))
  (ppval (marginal-node node)))


;;; ppconflict -- prints out the conflict associated with node
(defun ppconflict (conflict-list)
  (declare (type List conflict-list)
	   (:returns nil))
  "Pretty prints the <conflict-list>."
  (format t "~%Source of Conflict ~30T Conflict~%")
  (map nil #'(lambda (conflict-pair)
	       (format t "~S ~30T ~10,8F~%"
		       (car conflict-pair) (cdr conflict-pair)))
	conflict-list))




;;; ppmessages -- prints the receieved message stack
(defun ppmessages (node)
  (declare (type Symbol node)
	   (:returns nil))
  "Pretty prints all messages received by <node>.  Note:  Form
defining <node> should evaluate to a symbol."
  (format t "~%Messages received by ~s~%" node)
  (format t "~& Local Information:~%")
  (ppval (eval node))
  (when (get node :original-val)
	(format t "~&Origianl Value:~%")
	(ppval (get node :original-val)))
  (map nil #'(lambda (acons)
	       (format t "~&Received from ~s:~%" (car acons))
	       (if (null (cdr acons)) (format t "~&------NIL-----~%")
		 (ppval (cdr acons))))
	(get node :received))
  (terpri))


;;;; Various shells associated with questions asked by the program.

;;; Root selection shell
(defun pick-root (tree-o-cliques )
  (declare (type Graph tree-o-cliques)
	   (:returns (type Symbol #!*root*)))
  "Creates a shell to pick a root node from <tree-o-cliques>."
  (prog (inputted outputted)
   loop (terpri)
        (format t "Choose a root node.  Choices are: ~% ~S ~%"
		#!*margin-list*)
	(format t "~% or :T to print the tree.")
   prompt (terpri)
        (format t "*ROOT*>")
	(setq inputted (read))
	(if (member inputted #!*margin-list*)
	    (return inputted))
	(when (eq :T inputted) (format t "~%~S" tree-o-cliques)
	      (go prompt))
	(let ((res (protected-eval inputted)))
	  (if (member res #!*margin-list*)
	      (return res))
	  (print res) (go loop))))



;;; Alternate projection and combination rules.  Default to @+2 and
;;; @-> to supply hooks for redefining these operations.


;; @@+ -- normalized combination
(defun @@+ (&rest args)
  (declare (:returns (type Val sum)))
  "Direct-sum-hook @@+ is used to allow redefinition of the direct sum
operation.  The function @@+ usually expands to @+ with the keyword
arg :op set to #'+ (ordinary direct combination.  However, :op #'max
also gives interesting results (most likely configuration).  This can
be used for experiementation  with the inference engine."
  (apply #'@+ (append args (list :op #'+))))

;; @@+2 -- unormalized combination
(defun @@+2 (&rest args)
  (declare (:returns (type Val sum)))
  "Unnormalized-direct-sum-hook @@+2 is used to allow redefinition of
the unormalized direct sum operation.  The function @@+2 usually
expands to @+2 with the keyword arg :op set to #'+ (ordinary direct
combination.  However, :op #'max also gives interesting results (most
likely configuration).  This can be used for experiementation  with
the inference engine.  

Note the similarity to @@+.  As @@+2 is used more often, both should
be changed for consistancy." 
  (apply #'@+2 (append args (list :op #'+))))

;; @@-> -- unormalized projection
(defun @@-> (&rest args)
    (declare (:returns (type Val project)))
  "Projection-hook @@-> is used to allow redefinition of the projection
operation.  The function @@-> usually expands to @-> with the keyword
arg :op set to #'+ (ordinary direct combination.  However, :op #'max
also gives interesting results (most likely configuration).  This can
be used for experiementation  with the inference engine."
  (apply #'@-> (append args (list :op #'+))))

;;; provide when loaded
(bel-provide :computations)
