;;; bymachine -- machine dependent code

;;; Copyright 1988 Russell G. Almond
;;; License is granted to copy this program for education or research
;;; purposes, with the restriction that no portion of this program may
;;; be copied without also copying this notice.  All other rights
;;; reserved. 


;;; provides palaces for machine dependent code

;;; 7/18/89 -- Added protected-eval <<**System Dependence**>> []
;;; 2/20/91 -- Separated by machine.  Added CMU-LISP support.


(bel-require :utils)
;(provide 'machine-utils)
(in-package :utils )
;(export '(protected-eval set-up-rng set-up-dist draw-dist clean-up-rng))


;;; <<**System Dependence**>>
;;; dispatches defintions accordig to the value of
;;; (lisp-implementation-type) currently known (to me) types are 
;;; "Lucid Common Lisp", "VAX LISP", and "Kyoto Common Lisp"  Code for those types
;;; "Allegro CL" added
;;; has been written but not necessarily debugged.



;;; **TO-DO** figure out versions of this for various machines.
;;; protected-eval -- This is a version of eval which ignores errors.
;;; There is not common lisp standard for this so I am afraid it will
;;; be machine dependent for now

(if (equal (lisp-implementation-type) "Allegro Cl")
    (defmacro protected-eval (form)
      `(errset ,form))
  (defmacro protected-eval (form)
  `(eval ,form)))

       


;;; Random number generators

;;; random number generators have been written in C for speed.  These
;;; routines call the interfaces, setup-rng (calls setup_rng),
;;; cleanup-rng (calls cleanup_rng), setup-distritbution (calls
;;; setup_distribution), and draw-distribution (calls draw_distribution).  
;;; Interfaces are set up so that all passing is done by well defined
;;; types, that is character, integers or double floating point
;;; numbers.


;;; Return codes from those functions (these should aggree with the
;;; values set in rng.h)
(defconstant *OK* 0 "OK Return code from RNG")
(defconstant *ERROR* -1 "Error return code from RNG")
(defconstant *PARAMETER-ERROR* -2 "Parameter specification error
return code from RNG")

;; set-up-rng -- takes one argument which is an integer and tells how
;; many distributions are to be created and allocates space for those
;; distributions.  It returns an integer which is OK if allocation
;; succeeded and ERROR if false.  
(defun set-up-rng (num-dists)
  (declare (type Fixnum num-dists))
  "Allocates C space for RNG objects.  <num-dists> is estimated number
of RNG objects needed."
  (if (eql *ERROR* (setup-rng num-dists))
      (error "set-up-rng: C error in setup-rng")))

;; clean-up-rng -- takes no arguments and de-allocates space allocated
;; by setup-rng and setup-distirubtion calls.
(defun clean-up-rng ()
  "Reclaims space for RNG objects."
  (if (eql *ERROR* (cleanup-rng))
      (error "clean-up-rng: C error in setup-rng")))


;; set-up-dist -- takes a distribution type argument which is
;; a character and three other optional arguments which are
;; parameters.  It returns an integer which is a table offset to the
;; data for that distribution stored by the C code.  This is used to
;; find a structure pointer in the C data space.  This distribution
;; number is then the sole argument to draw-distribution.
;;
;;
;; Currently supported types
;; Type		Par1		Par2		Par3
;;-----------------------------------------------------
;; U-niform	Upper(Lower)	Lower(Upper)	Unused
;; N-ormal	Mean		SDV		Unused
;; L-ognormal	Location(0.0)	Scale(Median)	Shape(var(log))
;; E-xponential	Scale		Unused		Unused
;; B-eta	Alpha		Beta		Unused
;; G-amma	Alpha		Scale		Unused


(defun set-up-dist (type &optional (par1 0.0) (par2 0.0) (par3 0.0)
			 &aux dist_num) 
  (declare (type Character type)
	   (type long-float par1 par2 par3)
	   (:returns (type Fixnum data-num)))
"Allocates and initializes RNG object in C-space.  
Takes a distribution <type> argument which is
a character and three other optional arguments which are
parameters.  It returns an integer which is a table offset to the
data for that distribution stored by the C code.  This is used to
find a structure pointer in the C data space.  This distribution
number is then the sole argument to draw-distribution.

\\begingroup\\tt\\obeylines\\obeyspaces

;; Currently supported types
;; Type		Par1		Par2		Par3
;;-----------------------------------------------------
;; U-niform	Upper(Lower)	Lower(Upper)	Unused
;; N-ormal	Mean		SDV		Unused
;; L-ognormal	Location(0.0)	Scale(Median)	Shape(var(log))
;; E-xponential	Scale		Unused		Unused
;; B-eta	Alpha		Beta		Unused
;; G-amma	Alpha		Scale		Unused
\\endgroup"
  (setq dist_num (setup-distribution type (coerce par1 'double-float)
				     (coerce par2 'double-float)
				     (coerce par3 'double-float)))
  (cond ((eql *ERROR* dist_num)
	 (error "set-up-distribution: C memory error"))
	((eql *PARAMETER-ERROR* dist_num)
	 (error "set-up-distribution: Bad Type or Parameters ~% ~C ~G ~G ~G~%"
		type par1 par2 par3))
	(t dist_num)))

;; draw-dist -- takes one argument which is an integer giving the
;; table index of its data in the C allocation space, and returns a
;; double precison random number with the specified distribution.  
(defun draw-dist (data-num)
  (declare (type Fixnum data-num)
	   (:returns (type Double random-number)))
  "Draws a random number using RNG and the RNG object identified by
<data-num>.   Distribution depends on what was set during previous
call to set-up-dist."
  (if (> 0 data-num) (error "draw-dist: Distribuiton not set up")
    (draw-distribution data-num)))


;;; C interface calls for Lucid Lisp (3.0)
(eval-when (eval compile load)
(when (equal "Lucid Common Lisp" (lisp-implementation-type))
      (bel-require :byLucid)))

;;; C interface calls for Allegro CL

(eval-when (eval compile load)
(when (equal "Allegro CL" (lisp-implementation-type))
      (bel-require :byAllegro "byAllegro")))


;;; C interface calls for Kyoto Common Lisp (3.0)

(eval-when (eval compile load)
(when (equal "Kyoto Common Lisp" (lisp-implementation-type))
      (bel-require :byKcl "byKcl")))


;;; C interface calls for CMU-LISP 15d

(eval-when (eval compile load)
(when (equal "CMU Common Lisp" (lisp-implementation-type))
      (bel-require :byCMU "byCMU")))

;;; VAX LISP
;;; As documentation is currently unavailable this is unsupported for
;;; now.
(eval-when (eval compile load)
(when (equal "VAX LISP" (lisp-implementation-type))
      (bel-require :byVAX "byVAX")))


;;; See also file lisp-init.lisp

;;; provide when loaded
(bel-provide :machine-utils)
