;;; bel's and q's -- translation and computations with bel's and q's.

;;; Copyright 1987 Russell G. Almond  & Mathis Thoma
;;; License is granted to copy this program for education or research
;;; purposes, with the restriction that no portion of this program may
;;; be copied without also copying this notice.  All other rights
;;; reserved. 

;;; 7/13/88 -- changed to update the way that **frame** is handled.


;(provide 'belq)
(in-package :belief)
(bel-require :structures "structures")
(bel-require :sets "sets")
(bel-require :arithmatic "arithmatic")
;(use-package 'basic)
;(use-package 'sets)
;(use-package 'belief)
;(export '(bel-m-frame bel-m-m-array make-bel-m bel-m-p
;	  bel-q-frame bel-q-q-array make-bel-q bel-q-p
;	  bel-bel-frame bel-bel-bel-array make-bel-bel bel-bel-p
;	  bel-pl-frame bel-pl-pl-array make-bel-pl bel-pl-p
;	  belfun-to-bsam bsam-to-belfun norm-m @-
;	  m->bel bel->m m->q q->m q+q q-q bel->pl pl->bel
;	  pl->q q->pl
;	  print-bel print-m print-q print-pl
;	  get-bel get-pl get-q get-m E^ Ev E^v
;	  sing-bp inc-bp dec-bp))

;;; structures for alternate representations.

;;; bel-m -- array of m-values representations
(defstruct (bel-m (:include val))
  "Belief function as array of mass function values."
  (m-array (make-array 0 :element-type 'long-float) :type (vector long-float)))

;;; bel-q -- array of q-values representations
(defstruct (bel-q (:include val))
  "Belief function as superpotential (array of commonalities)."
  (q-array (make-array 0 :element-type 'long-float) :type (vector long-float)))

;;; bel-bel -- array of bel-values representations
(defstruct (bel-bel (:include val))
  "Belief function as array of belief values."
  (bel-array (make-array 0 :element-type 'long-float) :type (vector long-float)))

;;; bel-pl -- array of pl-values representations
(defstruct (bel-pl (:include val))
  "Belief function as array of plausibility values."
  (pl-array (make-array 0 :element-type 'long-float) :type (vector long-float)))


;;; belief-function to bel-m translation

;;; belfun-to-bsam -- translates from belief function to bel-m notation
(defun belfun-to-bsam (bel)
  (declare (type Belief-Function bel)
	   (:returns (type Bel-M bel-m)))
  "Translates <bel> from Belief-Function to Bel-M representation."
  (let* ((theta (mapcar #'(lambda (att) (get att :values))
			(belief-function-frame bel)))
	 (size (apply #'* (mapcar #'length theta)))
	 (m-array (make-array (the fixnum (ash 1 size)) :element-type 'long-float
			      :initial-element 0.0l0)))
    (declare (fixnum size) (list theta))
    (map nil #'(lambda (mv)
		 (if (eq (m-value-element mv) **frame**)
		     (setf (aref m-array (1- (the fixnum (ash 1 size))))
			   (m-value-m mv))
		   (setf (aref m-array (ps-to-bs (ps-set-sym-val
						  (m-value-element mv))
						 theta))
			 (m-value-m mv))))
	 (belief-function-ms bel))
    (make-bel-m :frame (belief-function-frame bel) :m-array m-array)))
  

;;; ps-to-bs -- This translates from the ps-set notation to the
;;; bit-string notation.
;;; theta is (mapcar #'(lambda (att) (get att :values)))
(defun ps-to-bs (ps-set theta)
  (declare (type List ps-set) (type List theta)
	   (:returns (type Fixnum bs)))
  "Translates PS(TS)-set <ps-set> into bit-string notation.  <theta>
is frame expressed as list of (get att :values) lists."
  (apply #'logior
	 (mapcar #'(lambda (#1=#:x)
		     (the fixnum (ash 1 (the fixnum (tuple-to-bit #1# theta)))))
		 ps-set)))


;;; tuple-to-bit -- This takes a tuple (from a ps-set) and gives it a
;;; bit number for the appropriate bit string value
;;; theta is (mapcar #'(lambda (att) (get att :values)))
(defun tuple-to-bit (tuple theta)
  (declare (type List tuple) (type List theta)
	   (:returns (type Fixnum bit)))
  "Translates a <tuple> from a PS(TS)-set into bit to set for
bit-string notation.  <theta> is frame expressed as list of (get att
:values) lists." 
  (declare (list tuple) (list theta))
  (let ((value 0))
    (declare (fixnum value))
    (do ((tlist tuple (cdr tlist))
	 (flist theta (cdr flist)))
	((and (endp tlist) (endp flist)) value)
	(setq value  
	      (+ (position (car tlist) (car flist))
		 (the fixnum (* value (length (car flist)))))))))


;;; BEL-M TO BELIEF-FUNCTION TRANSLATION

;;; bsam-to-belfun -- this function transfers from a bit-string
;;; indexed array of m's to a belief function (list of m-value's)
(defun bsam-to-belfun (bsam)
  (declare (type Bel-M bsam)
	   (:returns (type Belief-Function bel)))
  "Translates <bsam> from Bel-M (bit-string indexed array) to
Belief-Function (list of focal elements) representation."
  (let ((theta (mapcar #'(lambda (att) (get att :values))
		       (bel-m-frame bsam)))
	(m-list (list (make-m-value :element **frame**
				    :m (elt (bel-m-m-array bsam)
					    (1- (array-total-size
						 (bel-m-m-array bsam)))))))
	(size-1 (1- (array-total-size (bel-m-m-array bsam)))))
    (do ((pos (position-if-not #'zerop (bel-m-m-array bsam) :start 0)
	      (position-if-not #'zerop (bel-m-m-array bsam) :start (1+ pos))))
	((or (null pos) (= pos size-1))	;don't include term for frame
	 (make-belief-function :frame (bel-m-frame bsam)
			       :ms m-list)) ;return with belief function
	(push (make-m-value :m (elt (bel-m-m-array bsam) pos)
			    :element (get-ps-set-sym
				      (bs-to-ps pos theta)))
	      m-list))))


;;; bs-to-ps -- This transfers the bit-string representation of a
;;; focal element into the ps-set representation.  Theta is the
;;; frame expressed as a list of lists.
(defun bs-to-ps (bs theta)
  (declare (type Integer bs) (type List theta)
	   (:returns (type List ps-set)))
  "Translates bit-string <bs> into PS(TS)-set <ps-set>.  <theta> is
frame expressed as list of (get att :values) lists." 
  (let ((focal nil))
    (dotimes (bit (integer-length bs) focal)
	     (if (logbitp bit bs)
		 (push (bit-to-tuple bit theta) focal)))))



;;; bit-to-tuple -- This takes the position of a bit and constructs
;;; the corresponding tuple.
(defun bit-to-tuple (bit theta)
  (declare (type Integer bit) (type List theta)
	   (:returns (type List tuple)))
  "Translates <bit> in bit-string to <tuple> in PS(TS)-set.  <theta>
is frame expressed as list of (get att :values) lists." 
  (let ((bv bit) 			;value of bit and remainder
	(br 0)                          ;(for divisions later on)
	(tuple nil))
    (dolist (|theta| (reverse theta) tuple)
	    (multiple-value-setq (bv br) (floor bv (length |theta|)))
	    (push (nth br |theta|) tuple))))



				 
	
	
;;; Moebius transformations from m->q->m->bel->m


;;; m->bel -- This transfers from bel-m to bel-bel notation

(defun m->bel (bsam &aux vertex plane-byte high-plane-byte)
  (declare (type Bel-M bsam) (type Integer vertex plane-bit)
	   (:returns (type Bel-Bel bsabel)))
  "Translates mass function <bsam> in Bel-M form to belief function
<bsabel> in Bel-Bel form.  Uses Thoma's fast Moebius transform."
  (let* ((bsabel-array (map (type-of (bel-m-m-array bsam))
			 #'identity (bel-m-m-array bsam)))
	 (size (array-total-size bsabel-array))
	 (lsize (1- (integer-length size))))
    (dotimes (plane  lsize
		    (make-bel-bel :frame (bel-m-frame bsam)
				  :bel-array bsabel-array))
	     (setq plane-byte (byte 1 plane)
		   high-plane-byte (byte (- lsize plane 1) (1+ plane)))
	     (dotimes (vnumber  (/ size 2))
		       (setq vertex (deposit-field (ash vnumber 1)
						   high-plane-byte vnumber))
		       (setf (aref bsabel-array (dpb 1 plane-byte vertex))
			     (+ (aref bsabel-array (dpb 1 plane-byte vertex))
				(aref bsabel-array (dpb 0 plane-byte vertex))))))))



;;; bel->m -- This transfers from bel-bel to bel-m notation

(defun bel->m (bsabel &aux vertex plane-byte high-plane-byte)
  (declare (type Bel-Bel bsabel) (type Integer vertex plane-bit)
	   (:returns (type Bel-M bsam)))
  "Translates belief function <bsabel> in Bel-Bel form to mass function
<bsam> in Bel-M form.  Uses Thoma's fast Moebius transform."
  (let* ((bsam-array (map (type-of (bel-bel-bel-array bsabel)) #'identity
			  (bel-bel-bel-array bsabel)))
	 (size (array-total-size bsam-array))
	 (lsize (1- (integer-length size))))
    (dotimes (plane  lsize
		    (make-bel-m :frame (bel-bel-frame bsabel)
				  :m-array bsam-array))
	     (setq plane-byte (byte 1 plane)
		   high-plane-byte (byte (- lsize plane 1) (1+ plane)))
	     (dotimes (vnumber  (/ size 2))
		       (setq vertex (deposit-field (ash vnumber 1)
						   high-plane-byte vnumber))
		       (setf (aref bsam-array (dpb 1 plane-byte vertex))
			     (- (aref bsam-array (dpb 1 plane-byte vertex))
				(aref bsam-array (dpb 0 plane-byte vertex))))))))



	     
;;; m->q -- This transfers from bel-m to bel-q notation

(defun m->q (bsam &aux vertex plane-byte high-plane-byte)
  (declare (type Bel-M bsam) (type Integer vertex plane-bit)
	   (:returns (type Bel-Q bsaq)))
  "Translates mass function <bsam> in Bel-M form to commonality 
<bsaq> in Bel-Q superpotential form.  Uses Thoma's fast Moebius transform."
  (let* ((bsaq-array (map (type-of (bel-m-m-array bsam)) #'identity (bel-m-m-array bsam)))
	 (size (array-total-size bsaq-array))
	 (lsize (1- (integer-length size))))
    (dotimes (plane  lsize
		    (make-bel-q :frame (bel-m-frame bsam)
				  :q-array bsaq-array))
	     (setq plane-byte (byte 1 plane)
		   high-plane-byte (byte (- lsize plane 1) (1+ plane)))
	     (dotimes (vnumber  (/ size 2))
		       (setq vertex (deposit-field (ash vnumber 1)
						   high-plane-byte vnumber))
		       (setf (aref bsaq-array (dpb 0 plane-byte vertex))
			     (+ (aref bsaq-array (dpb 0 plane-byte vertex))
				(aref bsaq-array (dpb 1 plane-byte vertex))))))))



;;; q->m -- This transfers from bel-q to bel-m notation

(defun q->m (bsaq &aux vertex plane-byte high-plane-byte)
  (declare (type Bel-Q bsaq) (type Integer vertex plane-bit)
	   (:returns (type Bel-M bsam)))
  "Translates commonality function <bsaq> in Bel-Q superpotential form
to mass function <bsam> in Bel-M array of masses form.  Uses Thoma's
fast Moebius transform." 
  (declare (type bel-q bsaq) (integer vertex) (integer plane-bit))
  (let* ((bsam-array (map (type-of (bel-q-q-array bsaq)) #'identity (bel-q-q-array bsaq)))
	 (size (array-total-size bsam-array))
	 (lsize (1- (integer-length size))))
    (dotimes (plane lsize
		    (make-bel-m :frame (bel-q-frame bsaq)
				  :m-array bsam-array))
	     (setq plane-byte (byte 1 plane)
		   high-plane-byte (byte (- lsize plane 1) (1+ plane)))
	     (dotimes (vnumber  (/ size 2))
		       (setq vertex (deposit-field (ash vnumber 1)
						   high-plane-byte vnumber))
		       (setf (aref bsam-array (dpb 0 plane-byte vertex))
			     (- (aref bsam-array (dpb 0 plane-byte vertex))
				(aref bsam-array (dpb 1 plane-byte vertex))))))))

;;; pl->q -- This transfers from bel-pl to bel-q notation

(defun pl->q (bsapl &aux vertex plane-byte high-plane-byte)
  (declare (type Bel-Pl bsapl) (type Integer vertex plane-bit)
	   (:returns (type Bel-Q bsaq)))
  "Translates plausibility function <bsapl> in Bel-PL form to commonality 
<bsaq> in Bel-Q superpotential form.  Uses Thoma's fast Moebius transform."
  (declare (type bel-pl bsapl) (integer vertex) (integer plane-bit))
  (let* ((bsaq-array (map (type-of (bel-pl-pl-array bsapl))
		       #'(lambda (#1=#:x) (- 1 #1#))
		       (bel-pl-pl-array bsapl)))
	 (size (array-total-size bsaq-array))
	 (lsize (1- (integer-length size))))
    (dotimes (plane  lsize
		    (make-bel-q :frame (bel-pl-frame bsapl)
				  :q-array bsaq-array))
	     (setq plane-byte (byte 1 plane)
		   high-plane-byte (byte (- lsize plane 1) (1+ plane)))
	     (dotimes (vnumber  (/ size 2))
		       (setq vertex (deposit-field (ash vnumber 1)
						   high-plane-byte vnumber))
		       (setf (aref bsaq-array (dpb 1 plane-byte vertex))
			     (- (aref bsaq-array (dpb 0 plane-byte vertex))
				(aref bsaq-array (dpb 1 plane-byte vertex))))))))



;;; q->pl -- This transfers from bel-q to bel-pl notation

(defun q->pl (bsaq &aux vertex plane-byte high-plane-byte)
  (declare (type Bel-Q bsaq) (type Integer vertex plane-bit)
	   (:returns (type Bel-Pl bsapl)))
  "Translates commonality <bsaq> in Bel-Q superpotential form to
plausibility function <bsapl> in Bel-Pl form.  Uses Thoma's fast
Moebius transform." 
  (declare (type bel-q bsaq) (integer vertex) (integer plane-bit))
  (let* ((bsapl-array (map (type-of (bel-q-q-array bsaq)) #'identity (bel-q-q-array bsaq)))
	 (size (array-total-size bsapl-array))
	 (lsize (1- (integer-length size))))
    (dotimes (plane lsize
		    (make-bel-pl :frame (bel-q-frame bsaq)
				 :pl-array (map (type-of bsapl-array)
						#'(lambda (#1=#:x) (- 1 #1#))
						bsapl-array)))
	     (setq plane-byte (byte 1 plane)
		   high-plane-byte (byte (- lsize plane 1) (1+ plane)))
	     (dotimes (vnumber (/ size 2))
		       (setq vertex (deposit-field (ash vnumber 1)
						   high-plane-byte vnumber))
		       (setf (aref bsapl-array (dpb 1 plane-byte vertex))
			     (- (aref bsapl-array (dpb 0 plane-byte vertex))
				(aref bsapl-array (dpb 1 plane-byte vertex))))))))



;;; arithmatic with q's and m's

;;; q+q -- This is the direct sum in the q mode.

(defun q+q (bsaq1 bsaq2)
  (declare (type Bel-Q bsaq1 bsaq2)
	   (:returns (type Bel-Q q1+q2)))
  "Direct sum  of commonalities <bsaq1> and <bsaq2> using
convolution." 
  (if (not (equal (bel-q-frame bsaq1) (bel-q-frame bsaq2)))
      (error "bel::q+q:incompatable frames ~s and ~s"
	     (bel-q-frame bsaq1) (bel-q-frame bsaq2))
    (make-bel-q :frame (bel-q-frame bsaq1)
		:q-array (map (type-of (bel-q-q-array bsaq1))
			      #'* (bel-q-q-array bsaq1)
			      (bel-q-q-array bsaq2)))))

;;; q-q -- This is the inverse direct sum in the q mode.

(defun q-q (bsaq1 bsaq2)
  (declare (type Bel-Q bsaq1 bsaq2)
	   (:returns (type Bel-Q q1-q2)))
  "Inverse direct sum  of commonalities <bsaq1> and <bsaq2> using
convolution.  Note:  does not always result in a proper belief
function.  See Thoma for theory." 
  (if (not (equal (bel-q-frame bsaq1) (bel-q-frame bsaq2)))
      (error "bel::q-q:incompatable frames ~s and ~s"
	     (bel-q-frame bsaq1) (bel-q-frame bsaq2))
    (make-bel-q :frame (bel-q-frame bsaq1)
		:q-array (map (type-of (bel-q-q-array bsaq1))
			      #'bel/ (bel-q-q-array bsaq1)
			      (bel-q-q-array bsaq2)))))


;;; bel/ -- special version of divide which sets 0/0=0
(defun bel/ (#1=#:x #2=#:y)
  (declare (type Long-Float #1# Long-Float #2#)
	   (:returns (type Long-Float x/y)))
  "Divides <x>/<y> but returns 0 when dividing 0/0.  Signals error if
x/0 (Augie says this can't happen)." 
  (cond ((= 0.0L0 #1# #2#) 0.0L0)		;0/0=0
	((= 0.0L0 #2#)
	 (cerror "What is x/0" "bel/: Augie says this can't happen")
	 0.0L0) ;divide by zero
	(t (/ #1# #2#))))



;;; norm-m --this function normalizes bel-m's
(defun norm-m (bsam)
  (declare (type Bel-M bsam)
	   (:returns (type Bel-M |bsam|)))
  "Normalizes a mass function in bit-string index array form."
  (let ((tot (reduce #'+ (bel-m-m-array bsam))))
    (make-bel-m :frame (bel-m-frame bsam)
		:m-array
		(map (type-of (bel-m-m-array bsam))
		     #'(lambda (#1=#:x) (/ #1# tot))
		     (bel-m-m-array bsam)))))



;;; @- -- direct subtraction
(defun @- (bel1 bel2)
  (declare (type Belief-Function bel1 bel2)
	   (:returns (type Belief-Function bel1-bel2)))
  "Inverse direct sum of two belief fucntions <bel1> and <bel2>.
Result may not be a proper belief function.  (Do you believe in
negative probability?)  See Thoma for explanation."
  (bsam-to-belfun
   (norm-m (q->m (q-q (m->q (belfun-to-bsam bel1))
		      (m->q (belfun-to-bsam bel2)))))))


;;; bel-bel to bel-pl

;; bel-to-pl -- beliefs to plausibilities
(defun bel->pl (bsabel)
  (declare (type Bel-Bel bsabel)
	   (:returns (type Bel-Pl bsapl)))
  "Converses belief function (Bel-Bel form) to plausibility function
(Bel-Pl from)"
  (make-bel-pl :frame (bel-bel-frame bsabel)
	       :pl-array (map (type-of (bel-bel-bel-array bsabel))
			      #'(lambda (#1=#:x) (- 1 #1#))
			      (reverse (bel-bel-bel-array bsabel)))))


;; pl-to-bel -- plausibilities to beliefs
(defun pl->bel (bsapl)
  (declare (type Bel-Pl bsapl)
	   (:returns (type Bel-Bel bsabel)))
  "Converses plausibility function (Bel-Pl form) to belief function
(Bel-Bel from)"
  (make-bel-bel :frame (bel-pl-frame bsapl)
		:bel-array (map (type-of (bel-pl-pl-array bsapl))
			      #'(lambda (#1=#:x) (- 1 #1#))
			      (reverse (bel-pl-pl-array bsapl)))))



;;; printing out belief functions


;;print-bel --prints out a belief function (all values of bel)
(defun print-bel (bsabel)
  (declare (type Bel-Bel bsabel)
	   (:returns nil))
  "Prints out a belief function in bit-string array (Bel-Bel) form."
  (format t "~%Belief function over frame:~S~%"
	  (bel-bel-frame bsabel))
  (format t "~&Belief~20Tfocal element~%")
  (let ((size (array-total-size (bel-bel-bel-array bsabel))))
    (dotimes
     (felm size nil)
     (format t "~&~16,9G~20T{ " (aref (bel-bel-bel-array bsabel) felm ))
     (format t "~{~#[~;~S ~; ~S ~S~; ~S ~S ~S ~; ~S ~S ~S ~S ~:;~S ~S ~S ~S~%~20T~]~} }~%"
	     (bs-to-ps felm
		       (mapcar #'(lambda (att) (get att :values)) (bel-bel-frame bsabel))))))
  nil)

;;print-m --prints out a belief function (all values of m)
(defun print-m (bsam)
  (declare (type Bel-m bsam)
	   (:returns nil))
  "Prints out a mass function in bit-string array (Bel-M) form."
  (format t "~%Belief function over frame:~S~%"
	  (bel-m-frame bsam))
  (format t "~&m-value~20Tfocal element~%")
  (let ((size (array-total-size (bel-m-m-array bsam))))
    (dotimes
     (felm size nil)
     (format t "~&~16,9G~20T{ " (aref (bel-m-m-array bsam) felm ))
     (format t "~{~#[~;~S ~; ~S ~S~; ~S ~S ~S ~; ~S ~S ~S ~S ~:;~S ~S ~S ~S~%~20T~]~} }~%"
	     (bs-to-ps felm
		       (mapcar #'(lambda (att) (get att :values)) (bel-m-frame bsam)))))))

;;print-q --prints out a belief function (all values of q)
(defun print-q (bsaq)
  (declare (type Bel-Q bsaq)
	   (:returns nil))
  "Prints out a commonality function in bit-string array (Bel-Q)
superpotential form."
  (declare (type bel-q bsaq))
  (format t "~%Belief function over frame:~S~%"
	  (bel-q-frame bsaq))
  (format t "~&Commonalities(q)~20Tfocal element~%")
  (let ((size (array-total-size (bel-q-q-array bsaq))))
    (dotimes
     (felm size nil)
     (format t "~&~16,9G~20T{ " (aref (bel-q-q-array bsaq) felm ))
     (format t "~{~#[~;~S ~; ~S ~S~; ~S ~S ~S ~; ~S ~S ~S ~S ~:;~S ~S ~S ~S~%~20T~]~} }~%"
	     (bs-to-ps felm
		       (mapcar #'(lambda (att) (get att :values)) (bel-q-frame bsaq)))))))

;;print-pl --prints out a belief function (all values of pl)
(defun print-pl (bsapl)
  (declare (type Bel-Pl bsapl)
	   (:returns nil))
  "Prints out a plausibility function in bit-string array (Bel-Pl) form."
  (format t "~%Belief function over frame:~S~%"
	  (bel-pl-frame bsapl))
  (format t "~&Plausibility~20Tfocal element~%")
  (let ((size (array-total-size (bel-pl-pl-array bsapl))))
    (dotimes
     (felm size nil)
     (format t "~&~16,9G~20T{ " (aref (bel-pl-pl-array bsapl) felm ))
     (format t "~{~#[~;~S ~; ~S ~S~; ~S ~S ~S ~; ~S ~S ~S ~S ~:;~S ~S ~S ~S~%~20T~]~} }~%"
	     (bs-to-ps felm
		       (mapcar #'(lambda (att) (get att :values)) (bel-pl-frame bsapl)))))))


;;; Get-x gets a single value from bsa notation, given a focal element
;;; in ps-set notation

;; get-bel
(defun get-bel (bsabel focal)
  (declare (type Bel-Bel bsabel) (type List focal)
	   (:returns (type Float bel-value)))
  "Returns belief value associated with <focal> element (in ps(ts)-set
form) from belief function <bsabel> in bit-string array belief form."
  (aref (bel-bel-bel-array bsabel)
	(ps-to-bs (explode-ps-set focal)
		  (mapcar #'(lambda (att) (get att :values))
			  (bel-bel-frame bsabel)))
	))

;; get-m
(defun get-m (bsam focal)
  (declare (type Bel-M bsam) (type List focal)
	   (:returns (type Float m-value)))
  "Returns mass value associated with <focal> element (in ps(ts)-set
form) from mass function <bsam> in bit-string array mass form."
  (aref (bel-m-m-array bsam)
	(ps-to-bs (explode-ps-set focal)
		  (mapcar #'(lambda (att) (get att :values))
			  (bel-m-frame bsam)))
	))

;; get-q
(defun get-q (bsaq focal)
  (declare (type Bel-Q bsaq) (type List focal)
	   (:returns (type Float q-value)))
  "Returns commonality value associated with <focal> element (in ps(ts)-set
form) from commonality function <bsaq> in bit-string array
superpotential form."   
  (aref (bel-q-q-array bsaq)
	(ps-to-bs (explode-ps-set focal)
		  (mapcar #'(lambda (att) (get att :values))
			  (bel-q-frame bsaq)))
	))

;; get-pl
(defun get-pl (bsapl focal)
  (declare (type Bel-Pl bsapl) (type List focal)
	   (:returns (type Float pl-value)))
  "Returns plausibility value associated with <focal> element (in ps(ts)-set
form) from plausibility function <bsapl> in bit-string array
plausibility form." 
  (aref (bel-pl-pl-array bsapl)
	(ps-to-bs (explode-ps-set focal)
		  (mapcar #'(lambda (att) (get att :values))
			  (bel-pl-frame bsapl)))
	))


		  
;;; Upper and lower expectations.

;; Upper expectation -- expects a univariate discrete numeric frame belief function
(defun E^ (bel-fun &key (utility #'identity))
  (declare (type Belief-Function bel-fun)
	   (type (function (t) float) utility)
	   (:returns (type Float utility)))
  "Calculates upper utility function.  <utility> defaults to
the identity function which only makes sense if the belief function is
numeric."
  (apply #'+
	 (mapcar #'(lambda (mv)
		     (* (apply #'max
			       (mapcar utility
				       (if (eq (m-value-element mv) **frame**)
					   (get (car
						 (belief-function-frame
						  bel-fun))
						:values)
					 (mapcar #'car
						 (ps-set-sym-val
						  (m-value-element mv))))))
			(m-value-m mv)))
		 (belief-function-ms bel-fun))))

;; Lower expectation -- expects a univariate discrete numeric frame
;; belief function 
(defun Ev (bel-fun &key (utility #'identity))
  (declare (type Belief-Function bel-fun)
	   (type (function (t) float) utility)
	   (:returns (type Float utility)))
  "Calculates lower utility function.  <utility> defaults to
the identity function which only makes sense if the belief function is
numeric."
  (apply #'+
	 (mapcar #'(lambda (mv)
		     (* (apply #'min
			       (mapcar utility
				       (if (eq (m-value-element mv) **frame**)
					   (get (car
						 (belief-function-frame
						  bel-fun))
						:values)
					 (mapcar #'car
						 (ps-set-sym-val
						  (m-value-element mv))))))
			(m-value-m mv)))
		 (belief-function-ms bel-fun))))



;;; E^v --- computes the upper and lower expectations of a numeric
;;; valued belief function.
(defun E^v (bel-fun &key (utility #'identity))
  (declare (type Belief-Function bel-fun)
	   (type (function (t) float) utility)
	   (:returns (type Float utility)))
  "Calculates upper and lower utility function.  <utility> defaults to
the identity function which only makes sense if the belief function is
numeric. "
  (apply #'pair+
	 (mapcar
	  #'(lambda (mv)
	      (list (* (apply #'min
			      (mapcar utility
				      (if (eq (m-value-element mv) **frame**)
					  (get (car
						(belief-function-frame bel-fun))
					       :values)
					(mapcar #'car
						(ps-set-sym-val
						 (m-value-element mv))))))
		       (m-value-m mv))
		    (* (apply #'max
			      (mapcar utility
				      (if (eq (m-value-element mv) **frame**)
					  (get (car
						(belief-function-frame bel-fun))
					       :values)
					(mapcar #'car
						(ps-set-sym-val
						 (m-value-element mv))))))
		       (m-value-m mv))))
	  (belief-function-ms bel-fun))))


;;; These functions are ment to deal with numeric belief functions,
;;; they print bels and pls for various interseting subsets.

;;; sing-bp -- prints the belief and plausibility for all of the
;;; singlton subsets.
(defun sing-bp (bel-fun)
  (declare (type Belief-Function bel-fun)
	   (:returns nil))
  "Prints the belief and plausibility for all of the singlton subsets."
  (let* ((bel-b (m->bel (belfun-to-bsam bel-fun)))
	 (bel-p (bel->pl bel-b))
	 (frame (belief-function-frame bel-fun))
	 (range (if (every #'numberp (get (car frame) :values))
		    (sort (copy-list (get (car frame) :values)) #'<)
		  (get (car frame) :values))))
    (unless (eql (length frame) 1)
	    (error "cannot find the singleton elemets of a multivariate belief function"))
    (map nil #'(lambda (#1=#:x) (get-print-bp #1# bel-b bel-p)) range)))

;;; inc-bp -- prints the belief and plausibility for all of the
;;; numeric subsets of increasing size
(defun inc-bp (bel-fun)
  (declare (type Belief-Function bel-fun)
	   (:returns (type List theta)))
  "Prints the belief and plausibility for an increasing collection of
subsets."
  (let* ((bel-b (m->bel (belfun-to-bsam bel-fun)))
	 (bel-p (bel->pl bel-b))
	 (frame (belief-function-frame bel-fun))
	 (range (if (every #'numberp (get (car frame) :values))
		    (sort (copy-list (get (car frame) :values)) #'<)
		  (get (car frame) :values))))
    (unless (eql (length frame) 1)
	    (error "cannot find the singleton elemets of a multivariate belief function"))
    (mapl #'(lambda (#1=#:x) (get-print-bp #1# bel-b bel-p)) range)))


;;; decc-bp -- prints the belief and plausibility for all of the
;;; numeric subsets of decreasing size
(defun dec-bp (bel-fun)
  (declare (type Belief-Function bel-fun)
	   (:returns (type List theta)))
  "Prints beliefs and plausibilities for a decreasing numeric subsets." 
  (let* ((bel-b (m->bel (belfun-to-bsam bel-fun)))
	 (bel-p (bel->pl bel-b))
	 (frame (belief-function-frame bel-fun))
	 (range (if (every #'numberp (get (car frame) :values))
		    (sort (copy-list (get (car frame) :values)) #'>)
		  (reverse (get (car frame) :values)))))
    (unless (eql (length frame) 1)
	    (error "cannot find the singleton elemets of a multivariate belief function"))
    (mapl #'(lambda (#1=#:x) (get-print-bp #1# bel-b bel-p)) range)))




;;; get-print-bp -- This takes a value or a list of values from a
;;; given frame and prints the belief and Plausibility.
(defun get-print-bp (elm bel pl)
  (declare (type List elm) (type Bel-Bel bel) (type Bel-Pl pl)
	   (:returns nil))
  "Prints belief and plausibility of <elm> using belief function <bel>
and plausibility function <pl>"
  (format t "[~16,9G  ~16,9G] {~S}" (get-bel bel (list (list elm)))
	  (get-pl pl (list (list elm))) elm)
  (terpri))


;;; provide when loaded
(bel-provide :belq)

