;;; belief-init -- startup definitions for the belief package.

;;; Copyright 1989 Russell G. Almond
;;; License is granted to copy this program for education or research
;;; purposes, with the restriction that no portion of this program may
;;; be copied without also copying this notice.  All other rights
;;; reserved. 


;;; 7/18/89 -Version 1.1 Added new inherited functions and vars
;;;	* Added additional mode dependent offsets as needed.

;;; 10/16/90 -- Twiddled with the prompt display for Allegro CL
;;; 



;(provide 'belief-init)
;(in-package :inherited) ;:nicknames '(mode will))
;(export '(*rules-package* *mode-package* 
;	  new-rules use-rules bel-mode prob-mode inherit-mode no-mode
;	  |#?-reader| |#!-reader|))
;(eval-when (compile load eval) (import 'user::*belief-format* )
;	   (if (equal "Allegro CL" (lisp-implementation-type))
;	       (import 'tpl::*prompt* )))

;;; create belief and prob packages by default
;(in-package 'belief :nicknames '(bel))
;(in-package 'prob :nicknames '(potentials))
;(in-package 'subshell)			;same trick, this gets
					;optionally loaded but
					;defineately used
(in-package :mode)

(defvar *mode-package* (find-package 'belief)
  "Which package to inherit symbols from.  
Caution:  examine but do not set.  Use function belief-mode,
prob-mode, inherit-mode and no-mode instead.")


;;; Mode functions.  
;;; Basicly the belief package has two modes, belief--where all values
;;; (nodes of tree of cliques, and edges of model-graph) are belief
;;; functions, and prob--where all values are potentials.  A package
;;; can have one of four different modes:  belief -- it directly uses
;;; the belief package, prob -- it directly uses the prob package,
;;; none-- it uses neither package, or inherited--it inherits certain
;;; functions which deal with generalized values from the current
;;; *rules-package*.  Generally, the user package and the shell
;;; package should be of type inherited in order for the use-rules
;;; mechanism to function correctly.  The system makes some attempt to
;;; change the mode of the user package from belief or prob to the
;;; mode of the new rules, but this is not always done  (in
;;; particular, if a new rules base of the opposite mode is loaded,
;;; the mode of the user package may not change.)

(defvar *rules-package* nil 
  "package which contains rules.
	Caution: do not set directly, use new-rules and use-rules."
)
(defvar *inherited-mode-packages* '(user shell)
  "packages to set to inherited mode.")
(defvar *stay-inherited-mode* '(user shell)
  "packages that should be returned to inherited mode when the rules
package switches (to avoid funny behavior)")

;; bel-mode -- changes the mode of one package (by default the current
;; one) to belief.  Primarily designed for rule-bases at the top of
;; the input file.
;; This should be wrapped in a (eval-when (compile load eval))
(defun bel-mode (&optional (pack *package*))
  (declare (type (or Package String Symbol) pack)
	   (:returns :belief))
  "Sets a package to belief mode.  special behavior if pack is the
current *rules-package*"
  (setq *inherited-mode-packages* (remove pack *inherited-mode-packages*))
  (unless (packagep pack) (setq pack (find-package pack)))
  (unuse-package :inherited pack)
  (unuse-package :prob pack)
  (use-package :belief pack)
  (when (eql pack *rules-package*)
	(set '*mode-package* (find-package 'bel))
	(set (intern "*MODE*" *rules-package*) :belief))
  :belief)

;; prob-mode -- changes the mode of one package (by default the current
;; one) to prob.  Primarily designed for rule-bases at the top of
;; the input file.
;; This should be wrapped in a (eval-when (compile load eval))
(defun prob-mode (&optional (pack *package*))
  (declare (type (or Package String Symbol) pack)
	   (:returns :prob))
  "Sets a package to prob mode.  special behavior if pack is the
current *rules-package*"
  (setq *inherited-mode-packages* (remove pack *inherited-mode-packages*))
  (unless (packagep pack) (setq pack (find-package pack)))
  (unuse-package :inherited pack)
  (unuse-package :belief pack)
  (use-package :prob pack)
  (when (eql pack *rules-package*)
	(set '*mode-package* (find-package 'prob))
	(set (intern "*MODE*" *rules-package*) :prob))
  :prob)



;; inherit-mode -- changes the mode of one package (by default the
;; current one) to direct inherited. (mode is inherited from the
;; *rules-package*) 
;; This should be wrapped in a (eval-when (compile load eval))
(defun inherit-mode (&optional (pack *package*))
  (declare (type (or Package String Symbol) pack)
	   (:returns :inheritied))
  "Puts a package into inherited mode:  mode is inherited from the
*rules-package*" 
  (pushnew pack *inherited-mode-packages*)
  (unless (packagep pack) (setq pack (find-package pack)))
  (unuse-package :belief pack)
  (unuse-package :prob pack)
  (use-package *rules-package* pack)
  (use-package :inherited pack)
  :inherited)

;; force-inherited-mode -- checks to make certain a package in
;; inherited mode, (but don't change the mode yet).  It is meant to be
;; called from new-rules
(defun force-inherited-mode (&optional (pack *package*))
  (declare (type (or Package String Symbol) pack)
	   (:returns :inherited))
  "Forces a package into inherited mode and onto the
*inherited-mode-packages* list.  But doesn't cause package to use
*rules-package*."
  (pushnew pack *inherited-mode-packages*)
  (unless (packagep pack) (setq pack (find-package pack)))
  (unuse-package :belief pack)
  (unuse-package :prob pack)
  (use-package :inherited pack)
  :inherited)



;; no-mode -- changes the mode of one package (by default the current
;; one) to none.  Could also be used to change the inheritance to
;; inherited indirect (with a call to use-package on some other
;; package of belief, prob or inherited mode).  
;; This should be wrapped in a (eval-when (compile load eval))
(defun no-mode (&optional (pack *package*))
  (declare (type (or Package Symbol String) pack)
	   (:returns nil))
  "removes dependence on mode and rules packages."
  (setq *inherited-mode-packages* (remove pack *inherited-mode-packages*))
  (unless (packagep pack) (setq pack (find-package pack)))
  (unuse-package :belief pack)
  (unuse-package :prob pack)
  (unuse-package :inherited pack)
  nil)


  
;;;; Rules Package Functions
;;; The current rules pacakge is defined by the value of the variable
;;; *rules-package*  The current rules package (when the system is in
;;; a proper state) should be used by both the user and the shell
;;; packages.  The value of the variable *belief-format* is used to
;;; determine whether belief or prob mode is the default for new
;;; packages.  


;;; There exist two classes of inherited objects, objects with
;;; function definitions which must be inherited, given in
;;; *inherited-funs* and objects with values which must be inherited,
;;; given in *inherited-vars*.  *inherited-funs* are separated into
;;; *inherited-funs* and *inherited-macs* which are treated differently.

(defparameter *inherited-funs* '(@^ @v @-> @+ @+2 || -><- ppval
				 make-vacuous direct-sum reset-zero-tol
				 node-size reset-model approx-zerop
				 conflict-bel-fun normalize-bel-fun 
				 constant-pot normalize-pot pppot ppbel
				 draw-default draw-interval
				 draw-mix draw-uniform draw-beta
				 draw-gamma draw-lognorm
				 setup-default setup-interval
				 setup-mix setup-uniform
				 setup-beta setup-gamma
				 setup-lognorm draw setup setup-all
				 copy-change-frame impute-binomial
				 impute-poisson impute-binomial-data
				 impute-poisson-data mix
				 impute-poisson-disc impute-mix
				 sink-get-att-value
				 sink-get-param-value
				 nominal nominal-default
				 nominal-interval nominal-mix
				 nominal-uniform nominal-beta
				 nominal-gamma nominal-lognorm
				 impute impute-mix)
  "These functions should dispatch according to the current mode to
get specialized behavior for belief functions and probabilities.")


(defparameter *inherited-macs* '(defatt defbel defif defand defor
				 defifor defis defnand defnor defxor
				 defiff defiffor defbelcond defpot
				 defpotcond)
    "These macros should dispatch according to the current mode to
get specialized behavior for belief functions and probabilities.") 

(defparameter *inherited-vars* '(*min-input* *condition-ratio*
				 *zero-tol* *att-penalty*
				 *true-false-values* *model-graph*
				 *val-list* *margin-list* *root*
				 *sink-list* *failure-values*
				 *save-messages* *save-sink-values*
				 *save-values*
				 )
  "These values live in the rules package, although their default
values are set according to the mode.  In function definitions they
should be set through the #? macro and accessed through the #! macro."
)

;;; Each inherited function (from *inherited-funs*) exists as a
;;; function in three places.  The inherited package, the belief
;;; package and the prob package.  The symbol in the inherited package
;;; is a dispatching macro which expands to the corresponding symbol
;;; in the belief or prob package depending on the current value of
;;; the *mode-package* variable.  



;; defun-mode-dispatched -- This creates a new dispatching macro
;; with name symbol, which according to the value of the
;; *mode-package* expands to either (belief::symbol args) or
;; (prob::symbol args) 
(defun defun-mode-dispatched (sym)
  (declare (type Symbol sym)
	   (:returns (type Function)))
  "Defines a function which according to the value of *mode-package*
expands into either (belief::symbol args) or (prob::symbol args)"
  (setf (symbol-function sym)
	#'(lambda (&rest args)
	    (let ((mode-sym (intern (symbol-name sym) *mode-package*)))
	      (when (macro-function mode-sym)
		    (error "~S is a macro, not a function" mode-sym))
	      (apply (symbol-function mode-sym) args)))))


;; defmacro-mode-dispatched takes care of macros.
(defun defmacro-mode-dispatched (sym)
  (declare (type Symbol sym) (:returns Function))
  "Defines a macro which according to the value of *mode-package*
uses either (belief::symbol args) or (prob::symbol args) to do its
expansion." 
  (setf (macro-function sym)
	#'(lambda (form env)
	    (let ((mode-sym (intern (symbol-name sym) *mode-package*)))
	      (unless (macro-function mode-sym)
		      (error "~S is not a macro" mode-sym))
	      (macroexpand (cons mode-sym (cdr form)) env)))))


;;; These functions set up all of the dispatching macros

(export *inherited-funs*)
(export *inherited-macs*)

(eval-when (compile load eval)
	   (map nil #'defun-mode-dispatched *inherited-funs*)
	   (map nil #'defmacro-mode-dispatched *inherited-macs*))
	   

;;; Inherited Values are slightly different.  Values are not external
;;; symbols of the inherited package, instead they are external
;;; symbols of the rules package.  Packages in inherited mode take
;;; their value from the current rules package.  When a new rules
;;; package is created, these values are initialized according to the
;;; current mode.  

;;; Programming with inherited values is slightly tricky.  The 
;;; the inherited symbols are present at execution time so the
;;; expression: (setq *example-val* <expression>) does what you expect
;;; when typed interactively, that is it changes the value of
;;; *example-val* in the current rules package.  However, because the
;;; value *example-val* is interned when the expression is read, the
;;; function (defun set-example-val (x) (setq *example-val* x)) will
;;; not do what you expect.  When it is invoked it will be equivalent
;;; to (setq <old-rules-package>:*example-val*) where
;;; <old-rules-package> will be the value of the rules package at the
;;; time the expression was read.  To get around that problem, the
;;; BELIEF package provides the read macro #?.  #?symbol becomes
;;; (find-symbol "FOO" inherited::*rules-package*) which is the
;;; desired version of foo.  Thus the proper way to do the above
;;; function is to use (defun set-example-val (x) (set #?*example-val*
;;; x))  Similarly, when the value of an inherited value is needed in
;;; a function, the construction #!name should be used.  

(defun |#?-reader| (stream subchar arg)
  (declare (type Stream stream) (ignore subchar)
	   (:returns (type List)))
  "Read macro which expands #?foo into (intern *rules-package* #:foo).
Used to set mode dispatched values."
  (list 'intern
	(symbol-name (funcall (get-dispatch-macro-character #\# #\:)
			      ; reader used for #: syntax (returns
			      ; unintern symbol)
			      stream #\: arg))
	'*rules-package*))


(defun |#!-reader| (stream subchar arg)
  (declare (ignore subchar) (type Stream stream)
	   (:returns (type List)))
  "Read macro which turns for #!foo into
(symbol-value (intern *rules-package* #:foo)).
Used to access mode-dispatched values."
  (list 'symbol-value (list 'intern
	(symbol-name (funcall (get-dispatch-macro-character #\# #\:)
			      ; reader used for #: syntax (returns
			      ; unintern symbol)
			      stream #\: arg))
	'*rules-package*)))



(eval-when (compile load eval)
	   (set-dispatch-macro-character #\# #\? #'|#?-reader|)
	   (set-dispatch-macro-character #\# #\! #'|#!-reader|))

;; initialize-inherited-var
;; This function initializes an inherited var when a new rule base is
;; created
(defun initialize-inherited-var (symbol)
  (declare (type Symbol symbol) (:returns (type T value)))
  "Defines new mode-dispatched value in *rules-package* and
initializes according to value in *mode-package*.  Value becomes
special. "
  (let ((new-var (intern (symbol-name symbol) *rules-package*))
	(par-var (find-symbol (symbol-name symbol) *mode-package*)))
    (export new-var *rules-package*)
    (proclaim (list 'special new-var))
    (set new-var (eval par-var))
    (setf (documentation new-var 'variable)
	  (documentation par-var 'variable))
    (eval new-var)))

;;; finially the special inherited variable *mode* behaves much like
;;; an inherited varaible, except that it is initialized at the outset
;;; instead of inbetween.  


;;; Functions for creating and switching between rule bases.

;; new-rules -- is designed to be used at the top of the rules file to
;; do many system dependent things which should be transparent
;; (mostly) to the user.  This constain a call to in-package.  It can
;; be invoked interactively, (to say interactively type in a rule
;; base), but then a (in-package 'user) should be done to return to
;; user mode.  Keywords define whether the new rules are created in
;; belief mode or prob mode.  It defaults to the current value of
;; *belief-format* 
;; This should be wrapped in a (eval-when (compile load eval)) if you
;; are compiling the rule base
(defmacro new-rules (rule-name &key belief prob nuke &aux obf rule-pack)
  (declare (type Symbol rule-name)
	   (type (member T NIL) belief prob nuke)
	   (type Package rule-pack) (type (member T NIL) obf)
	   (:returns (type Package rule-pack)))
  "Defines new rules package and if necessary load appropriate code
for needed options.
	:belief, :prob --- sets appropriate mode, otherwise it gets
its mode from *belief-format*
	:nuke --- requires additional code for monte carlo PRA."
  (unless (or belief prob) (setq belief *belief-format*))
  (if (and belief prob) (error "Must choose one mode for rules ~S"
			       rule-name))
  (setq obf *belief-format*)
  (user::load-belief-package :belief belief :prob prob :nuke nuke)
  (setq *belief-format* obf)
  (bel-provide :rules t)
  (unless (packagep rule-name)
    (unless (or (stringp rule-name) (boundp rule-name)
		(fboundp rule-name) (symbol-plist rule-name)) 
      (unintern rule-name 'user))	;this kludge gets rid of
					;symbols generated for the
					;sole purpose of making
					;package names as they will
					;frequently later cause a
					;confusing conflict between an
					;object with the same name as
					;the package in the package,
					;and the symbol generated
					;temporarily to name the
					;package, thus avoiding
					;confusing common lisp error
					;for novice users.
    (setf rule-pack (find-package rule-name))
    )
  (when (packagep rule-name)
	(setq rule-pack rule-name)
	(setq rule-name (package-name rule-name)))
  `(progn 
     (defpackage ,rule-name
       (:use :basic :graphs :inherited :subshell :common-lisp))
     (set (intern "*MODE*" (find-package ,rule-name))
	  ,(if belief :belief :prob)) 
     (use-rules ,rule-name)
     (user::franz-kludge)		;Macro-dispatch characters
     (map nil #'initialize-inherited-var *inherited-vars*)
     ,rule-pack))



;; use-rules -- switches the rules package to (an already existing)
;; package, which defaults to the current package.  Makes an attempt
;; to switch the mode of the user package if it is other than
;; inherited.  
;; This should be wrapped in a (eval-when (compile load eval) if you
;; are compiling the rule base
(defun use-rules (&optional (new-rules *package*))
  (declare (type (or Package Symbol String) new-rules)
	   (:returns (type Package *rules-package*)))
  "Switches rules package to new-rules (default current *package*)"
  (map nil #'force-inherited-mode *stay-inherited-mode*)
  (when *rules-package*
	(map nil #'(lambda (pack) (unuse-package *rules-package* pack))
	     *inherited-mode-packages*))
  (unless (packagep new-rules) (setq new-rules (find-package new-rules)))
  (setq *rules-package* new-rules)
  (setq *mode-package* (find-package #!*mode*))
  (setq *prompt* (concatenate 'string (package-name *rules-package*)
			      "> "))	;Hopefully harmless anyplace
					;but in LUCID lisp
  (map nil #'(lambda (pack) (use-package *rules-package* pack))
       *inherited-mode-packages*)
  *rules-package*)


;;; provide when loaded
(bel-provide :belief-init)