;;; arithmatic.lisp -- basic belief function arithimatic functions

;;; Copyright 1986 Russell G. Almond
;;; License is granted to copy this program for education or research
;;; purposes, with the restriction that no portion of this program may
;;; be copied without also copying this notice.  All other rights
;;; reserved. 

;;; 12/15/86 -- changed to refect new ps-set functions
;;; 7/13/88 -- changed to use explicit **frame** value for frame
;;; 2/15/89 -- changed to allow alternative potential value and long
;;; precision arithmatic
;;; 8/5/89 -- Version 1.1
;;;	* Maximizing version of projection[d]
;;;	  new keyword (:op #'+) added to @v & @-> (& @^) defaults to
;;;	  #'+ but can be set to #'max
;;;	* Mixture operator[d]
;;; 8/4/89 -- added copy-change-frame which copies and changes frames,
;;; works on both belief functions and potentials
;;; 2/22/92 -- commenting to conform to az specifications.


;(provide 'arithmatic)
(in-package :belief); :nicknames '(bel))
(bel-require :structures "structures")
(bel-require :sets "sets")
;(use-package 'basic)
;(use-package 'sets)
;(export '(@^ @v @-> @+ @+2 || -><- ppval make-vacuous
;	     direct-sum conflict-bel-fun normalize-bel-fun
;	     approx-zerop ppbel node-size reset-zero-tol mix
;	     copy-change-frame))
;; located in rules package, but initialized here are
;;  *min-input* *condition-ratio* *att-penalty* *zero-tol*

;;;; minimal extension

;;; @^ -- minimal extension -- extend bel-fun so that its frame is
;;; new-frame.  (note: this also permutes the orders of the variables.)
(defun @^ (new-frame bel-fun &key (op #'+))
  (declare (type List new-frame) (type Belief-Function bel-fun)
	   (ftype (function (Number Number) Number) opv op+) (ignore op)
	   (:returns (type Belief-Function minimal-extension)))
  "(bel:@^ new-frame bel-fun) extends <bel-fun> to cover <new-frame>.
  <New-frame> must contain the frame of the belief function.  <op> is
ignored but allowed for consistency."
  (cond ((not (subsetp (belief-function-frame bel-fun) new-frame))
	 (error "bel:@^: illeagal frame ~S.  Try using bel:@->." new-frame))
	((equalp new-frame (belief-function-frame bel-fun)) bel-fun)
	(t (make-belief-function
	    :frame new-frame
	    :ms (extend-ms-list new-frame (belief-function-frame bel-fun)
				(belief-function-ms bel-fun))))))

;;; extend-ms-list -- minimally extends the focal elements in an
;;; ms-list to cover the new-frame.  It also permutes the order of the
;;; elements. Focal elements in the ms-list cover variables given in
;;; the old-frame argument.
(defun extend-ms-list (new-frame old-frame ms-list)
  (declare (type List new-frame old-frame ms-list)
	   (:returns (type List extended-ms-list)))
  "Extends each focal element in the <ms-list> from <old-frame> to
<new-frame> and returns <ms-list> of updated focal elements.  Permutes
order of PS-set Tuple to aggree with <new-frame>."
  (cond ((endp ms-list) nil)
	(t (cons (make-m-value
		  :m (m-value-m (car ms-list))
		  :element (extend-ps-sym new-frame old-frame
					  (m-value-element (car ms-list))))
		 (extend-ms-list new-frame old-frame (cdr ms-list))))))

;;; extend-ps-sym -- extends a single focal element to cover new frame
;;; instead of old frame.  The focal element is assument to be a symbol
;;; rather than an actual ps-set
(defun extend-ps-sym (new-frame old-frame ps-sym)
  (declare (type List new-frame old-frame) (type Symbol ps-sym)
	   (:returns (type Symbol new-PS-set)))
  "Extends <ps-sym> from <old-frame> to <new-frame>, creating a  new
ps-set symbol if necessary."
  (if (eq ps-sym **frame**) **frame**
    (get-ps-set-sym (mapcar #'(lambda (ps-el)
				(copy-ps-el new-frame old-frame ps-el)) 
			    (ps-set-sym-val ps-sym)))))



;;; copy-ps-el -- copies one possible outcome.  As the outcome is
;;; copied, it is moved from the old frame to new frame.
(defun copy-ps-el (new-frame old-frame ps-el)
  (declare (type List new-frame old-frame) (type List ps-el)
	   (:returns (type List new-ps-el)))
  "Copies <ps-el> which corresponds to one outcome in the tuple of
sets list, defined over <old-frame> to <new-frame>."
  (if (null ps-el) nil
    (do ((new-el (make-list  (length new-frame)))
	 (i 0 (1+ i)))
	((eql i (length new-frame)) new-el)
	(declare (fixnum i))
	(let ((offset (position (elt new-frame i) old-frame)))
	  (setf (elt new-el i) (if offset (elt ps-el offset)
				 (get (elt new-frame i) :values)
				 ;; assume variable have :values property
				 ))))))

;;;; Marginalization

;;; @v -- marginalization.  Finds the margin of bel in the new frame.
;;; This routine is optimized for the case when new-frame is a subset
;;; of the frame of the belief function.
(defun @v (new-frame bel-fun &key (op #'+))
  (declare (type List new-frame) (type Belief-Function bel-fun)
	   (ftype (function (Number Number) Number) opv op+)
	   (:returns (type Belief-Function marginal-bel-fun)))
  "(bel:@v new-frame bel-fun) Margin of <bel-fun> on <new-frame>.
   Assumes that <new-frame> is a subset of frame of <bel-fun>.
   <op> is used to consolidate elements which become duplicated by
   marginalization. "
  (cond ((not (subsetp new-frame (belief-function-frame bel-fun)))
	 (error "bel:@v : Illegal new-frame ~S.  Use bel:@->"))
	(t (make-belief-function
	    :frame new-frame
	    :ms (remove-universe
		 new-frame
		 (colapse-ms-list
		  (margin-ms-list new-frame (belief-function-frame bel-fun)
				  (belief-function-ms bel-fun))
		  :op op))))))


;;; colapse-ms-list -- sums over duplicate entries in a list of
;;; m-values 
;; added op keyword which defines the reduction operation.  Currently
;; other interesting choice is #'max
(defun colapse-ms-list (ms-list &key (op #'+))
  (declare (type List ms-list)
	   (ftype (function (Number Number) Number) opv op+)
	   (:returns (type List shorter-ms-list)))
  "Colapses <ms-list> by consolodating duplicate elements.  <op> is
applied to reduce the masses."
  (if (endp ms-list) nil
    (if (and (approx-zerop (m-value-m (car ms-list)))
	     (not (eq (m-value-element (car ms-list)) **frame**)))
	(colapse-ms-list (cdr ms-list))
      (cons (make-m-value
	     :element (m-value-element (car ms-list))
	     :m (reduce op (mapcar #'m-value-m
				   (remove (m-value-element (car ms-list))
					   ms-list
					   :key #'m-value-element
					   :test-not #'eq))))
	    (colapse-ms-list (remove (m-value-element (car ms-list))
				     ms-list
				     :key #'m-value-element
				     :test #'eq)
			     :op op)))))



;;; margin-ms-list -- find the margins of each of the element of the
;;; ms-list in the new-frame
(defun margin-ms-list (new-frame old-frame ms-list)
  (declare (type List new-frame old-frame) (type List ms-list)
	   (:returns (type List marginalized-ms-list)))
  "Finds the margins over <new-frame> of each element of <ms-list>
which is a <mass> . <ps-set> pair defined over <old-frame>.  Permute
order of elements to match new frame."
  (cond ((endp ms-list) nil)
	(t (cons (make-m-value
		  :m (m-value-m (car ms-list))
		  :element (margin-ps-sym new-frame old-frame
					  (m-value-element
					   (car ms-list))))
		 (margin-ms-list new-frame old-frame (cdr ms-list))))))


;;; margin-ps-sym -- creates a new ps-sym that covers the
;;; marginalization of the ps-set to the new frame
(defun margin-ps-sym (new-frame old-frame ps-sym)
  (declare (type List new-frame old-frame)
	   (type Symbol ps-sym)
	   (:returns (type Symbol new-ps-set)))
  "Extends <ps-set> named <ps-sym> and defined over <old-frame> to
<new-frame>.  Permute order to match <new-frame>."
  (if (eq ps-sym **frame**) **frame**
    (fget-ps-set-sym
     (remove-duplicates
      (mapcar #'(lambda (ps-el) (copy-ps-el new-frame old-frame ps-el))
	      (ps-set-sym-val ps-sym))
      :test #'equalp))))



;;;; projection

;;; @-> -- projection  -- this projects bel-fun onto a new frame.  It
;;; is like (bel:@v new-frame
;;;                 (bel:@^ (union new-frame (b-f-frame bel-fun))
;;;                         bel-fun)) 
;;; but more efficient.
(defun @-> (new-frame bel-fun &key (op #'+))
  (declare (type List new-frame) (type Belief-Function bel-fun)
	   (ftype (function (Number Number) Number) opv op+)
	   (:returns (type Belief-Function projected-bel-fun)))
  "(bel:@-> new-frame bel-fun) coerces <bel-fun> to be over
   <new-frame> by minimal extension and marginalization.  <op> is used
to summarized the masses of duplicate elements."
  (make-belief-function
   :frame new-frame
   :ms (remove-universe
	new-frame
	(colapse-ms-list
	 (extend-ms-list new-frame (belief-function-frame bel-fun)
			 (belief-function-ms bel-fun))
	 :op op))))




	
;;;; Direct sums

;;; direct-sum -- takes the direct sum of two belief functions over
;;; different frames.  The resulting frame is the union of the two
;;; frames. 
(defun direct-sum (bel1 bel2 &key (opv #'+ opv?) (op+ #'+ op+?) (op #'+))
  (declare (type Belief-Function bel1) (type Belief-Function bel2)
	   (ftype (function (Number Number) Number) opv op+ op)
	   (:returns (type Belief-Function bel1+bel2)))
  "Direct sum (Dempster's rule) of <bel1> and <bel2>. Extends <bel1>
and <bel2> to the union of their frames first. <opv> is used to
summarize duplicate focal elements when marginalizing; <op+> is used
to summarize dupliate elements when combining.  <op> sets both <op+>
and <opv>."
  (unless opv? (setq opv op))
  (unless op+? (setq op+ op))
  (let ((new-nodes (union (belief-function-frame bel1)
			      (belief-function-frame bel2))))
    (@+ (@^ new-nodes bel1 :op opv) (@^ new-nodes bel2 :op opv) :op op+)))

;;; direct-sum-operator -- takes the direct sum of two belief
;;; functions over the same frame.
(defun @+ (bel1 bel2 &key (op #'+) )
  (declare (type Belief-Function bel1) (type Belief-Function bel2)
	   (function op (Number Number) Number)
	   (:returns (type Belief-Function bel1+bel2)))
  "Direct sum (Dempster's rule) of <bel1> and <bel2> which must be
over the same frame (or error is signaled).  <op> is used to summarize
duplicated focal elements."
  (if (not (equalp (belief-function-frame bel1)
		   (belief-function-frame bel2)))
      (error "bel:@+ : Belief functions do not have same frame.")
    (make-belief-function
     :frame (belief-function-frame bel1)
     :ms (rescale-ms-list
	  (colapse-ms-list
	   (ms-list-prod (belief-function-ms bel1) (belief-function-ms bel2))
	   :op op))
     )))

;;; direct-sum-operator-unnormalized -- takes the direct sum of two belief
;;; functions over the same frame.
(defun @+2 (bel1 bel2 &key (op #'+))
  (declare (type Belief-Function bel1) (type Belief-Function bel2)
	   (function op (Number Number) Number)
	   (:returns (type Belief-Function bel1*bel2)))
  "Unnormalized direct sum (convolution) of <bel1> and <bel2>.  <bel1>
and <bel2> must be over the same frame or else an error is signaled.
<op> is used to summarize duplicate focal elements."
  (if (not (equalp (belief-function-frame bel1)
		   (belief-function-frame bel2)))
      (error "bel:@+ : Belief functions do not have same frame.")
    (make-belief-function
     :frame (belief-function-frame bel1)
     :ms (colapse-ms-list
	  (ms-list-prod (belief-function-ms bel1) (belief-function-ms bel2))
	  :op op))
     ))


;;; normalize-bel-fun -- this normalizes a belief function
(defun normalize-bel-fun (bel1)
  (declare (type Belief-Function bel1)
	   (:returns (type Belief-Function |bel1|)))
  "Normalizes belief function <bel1> by removing mass assigned to
empty-ps-set and rescaling mass."
  (make-belief-function
   :frame (belief-function-frame bel1)
   :ms (rescale-ms-list
	(belief-function-ms bel1))))

;;; conflict-bel-fun -- returns the ammount of conflict associated
;;; with an improper  belief function.
(defun conflict-bel-fun (bel)
  (declare (type Belief-Function bel)
	   (:returns (type Float conflict)))
  "Calculates conflict (mass associated with empty-ps-set) of
unnormailzed belief function <bel>."
  (let ((null-value
	 (find 'mt-set (belief-function-ms bel) :key #'m-value-element)))
    (if (null null-value) 0.0 (m-value-m null-value))))



;;; ms-list-prod -- takes the cross product of two ms-lists.  It
;;; assumes that there is an implicit universe m-value at the end of
;;; the list with weight 1.0 - sum of the other weights.
;;; 7/13/88 -- changed to explicit **frame** value.
(defun ms-list-prod (ms1 ms2)
  (declare (type List ms1 ms2)
	   (:returns (type List ms1xms2)))
  "Applies product-intersection rule to ms-lists <ms1> and <ms2>."
  (mapcan #'(lambda (mv) (ms-prod-set (m-value-m mv)
				      (m-value-element mv)
				      ms2))
	  ms1))
  

;;; ms-prod-set -- takes the product of a single element m-value with
;;; m mm and element mel with all of the m-values on ms-list.  
(defun ms-prod-set (mm mel ms-list2)
  (declare (type Long-Float mm) (type Symbol mel)
	   (type List ms-list2)
	   (:returns (type List m1-x-ms-list2)))
  "Takes product-intersection of m-value (<mm> . <mel>) with <ms-list2>."
  (cond ((approx-zerop mm) nil)
	((eq mel **frame**) (ms-prod-universe mm ms-list2))
	(t
	 (mapcar #'(lambda (mv)
		     (make-m-value
		      :m (the long-float (* mm (m-value-m mv)))
		      :element (ps-sym-intersect mel (m-value-element mv))))
		 ms-list2))))




;;; ms-prod-universe -- takes the product of the last element (the
;;; universe) with the list of ms-values
(defun ms-prod-universe (mm ms-list)
  (declare (type Long-Float mm) (type List ms-list)
	   (:returns (type List mm-x-ms-list)))
  "Takes product of m-value (<mm> . **frame**) with <ms-list>."
  (if (approx-zerop mm) nil
    (mapcar #'(lambda (mv)
		(make-m-value
		 :m (the long-float (* mm (m-value-m mv)))
		 :element (m-value-element mv)))
	    ms-list)))
	       


;;; rescale-ms-list 
(defun rescale-ms-list (ms-list)
  (declare (type List ms-list)
	   (:returns (type List rescaled-ms-list)))
  " This divides all elements of the ms-list by the
 scaling factor and remove the mt-set elements.  The scaling factor
 is 1 over 1- the sum of the m-value of the mt-set factor element."
  (remove 0
	  (let ((mm (/ (reduce #'- 
			       (mapcar #'m-value-m
				       (remove 'mt-set ms-list
					       :test-not #'eq
					       :key #'m-value-element))
			       :initial-value 1.0))))
	    (declare (long-float mm))
	    (mapcar #'(lambda (mv)
			(make-m-value
			 :m (* mm (m-value-m mv))
			 :element (m-value-element mv)))
		    (remove 'mt-set ms-list
			    :key #'m-value-element
			    :test #'eq)))
	  :key #'m-value-m
	  :test #'eql))


;;; remove-universe -- removes the universe by testing for length eql
;;; to product of the lengths of the :values props of the frame
(defun remove-universe (frame ms-list)
  (declare (type List frame) (type List ms-list)
	   (:returns (type List new-ms-list)))
  "Replaces all representations of the frame with single **frame**
reference in <ms-list>, where <frame> is the frame over which focal
elements in <ms-list> are defined." 
  (let* ((uni-length (reduce #'* (mapcar #'(lambda (#1=#:x)
					     (length (the list (get #1# :values))))
			      frame)))
	 (frame-weight
	  (reduce #'+
		  (mapcar #'m-value-m
			  (remove-if-not #'(lambda (mv)
					     (is-frame mv uni-length)) ms-list ))
		  :initial-value 0.0))
	 (trim-msl 
	  (remove-if #'(lambda (mv)
			 (is-frame mv uni-length))  ms-list)))

     (append trim-msl (list (make-m-value :m frame-weight :element **frame**)))))

    




;;; is-frame -- used to test to see if a m-value-element represents
;;; the whole frame
(defun is-frame (#2=#:y frame-length)
  (declare (type M-Value #2#) (type Number frame-length)
	   (:returns (type (member T NIL) is-frame)))
  "Tests to see if a m-value-element represents the whoe frame."
  (cond ((eq **frame** (m-value-element #2#)))
	((eql (length (ps-set-sym-val (m-value-element #2#))) frame-length))
	(t nil)))


;;; approx-zerop -- decides if the value is approximately equal to
;;; zero and if so returns t.
(defun approx-zerop (#1=#:x)
  (declare (type Long-Float #1#)
	   (:returns (type (member T NIL))))
  "Tests to see if value is approximately zero by comparing to
<#!*zero-tol*>."
  (if (< (abs #1#) (the float #!*zero-tol*))
      t nil))				;Hi tippit!


;;; *zero-tol* -- number which anything smaller than this should be
;;; taken as zero.  
(defvar *zero-tol* long-float-epsilon "Anything smaller than this
value should be taken as zero.")

;;; *min-input* -- This variable is the minimum m-value input in any
;;; belief function
(defvar *min-input* .01 "Minimum m-value input in any belief function.
Is automatically updated to keep track of entries.")

;;; *condition-ratio* -- The condition ratio divided by the min-input
;;; is the smallest value that will be considered non-zero
(defvar *condition-ratio* 1.0E6 "<*condition-ratio*> is greater than
<*min-input*> / <*zero-tol*> ")


(defun reset-zero-tol ()
  (declare (:returns (type Long-Float *zero-tol*)))
  "Sets <#?*zero-tolerance*> to be <#!*min-input*>/<#!*condition-ratio*>.  
Note that this value will very from rules package to rules package."
  (set #?*zero-tol* (the long-float
		      (max (/ (the long-float #!*min-input*)
			      (the long-float #!*condition-ratio*))
			long-float-epsilon))))


;;; Things moved over from computations because they really belong in
;;; package belief

;;; ppbel -- prints a belief function in a human readable form
(defun ppbel (bel-fun &optional (stream t))
  (declare (type Belief-Function bel-fun)
	   (type Stream stream)
	   (:returns (type nil)))
  "``Pretty prints'' belief function <bel-fun>."
  (format stream "~%Belief function over frame:~S~%"
	  (belief-function-frame bel-fun))
  (format stream "~&m-value~20Tfocal element~%")
  (map nil
   #'(lambda (mv)
       (format stream "~&~16,9G~20T{ " (m-value-m mv))
       (if (eq (m-value-element mv) **frame**) (format t "**frame** }~%")
	 (format stream "~{~#[~;~S ~; ~S ~S~; ~S ~S ~S ~; ~S ~S ~S ~S ~:;~S ~S ~S ~S~%~20T~]~} }~%"
		 (ps-set-sym-val (m-value-element mv)))))
   (belief-function-ms bel-fun))
  (terpri stream))

(defun ppval (bel-fun &optional (stream t))
  (declare (type Belief-Function bel-fun)
	   (type Stream stream)
	   (:returns (type nil)))
  "``Pretty print'' value.  bel:ppval is an alias for bel:ppbel."
  (ppbel bel-fun stream))

(defun || (bel-fun)
  (declare (type Belief-Function bel-fun)
	   (:returns (type Belief-Function |bel-fun|)))
  "Normalize belief function (alias for bel:normalize-bel-fun)."
  (normalize-bel-fun bel-fun))

(defun -><- (bel-fun)
  (declare (type Belief-Function bel-fun)
	   (:returns (type Float conflict)))
  "Calculates conflict in a belief function (alias for
bel:conflict-bel-fun)."
  (conflict-bel-fun bel-fun))

;;; node-size -- This caluclates the size of a node, where a node is a
;;; list of attributes
(defun node-size (node)
  (declare (type List node)
	   (:returns (type Number size)))
  "Calculates the size (penalty function) of a frame of discernment.
Used in builing a tree of cliques."
  (reduce #'*
	  (mapcar #'(lambda (att)
		      (+ (length (the list (get att :values))) #!*att-penalty*))
		  node)))

(defvar *att-penalty* 0 "Penalty for more attributes in a node")	  

(defun make-vacuous (frame)
  (declare (type List frame)
	   (:returns (type Belief-Function vacuous)))
  "Makes a vacuous belief function over <frame>."
  (make-belief-function :frame frame :ms vacuous))



;;; copy-change-frame -- copies a belief function structure and chages
;;; its name at the same time.
(defun copy-change-frame (frame bel-fun)
  (declare (type List frame) (type Belief-Function bel-fun)
	   (:returns (type Belief-Function new-bel-fun)))
  "Fast copy of belief function <bel-fun>, renaming frame to <frame>.

Warning:: this is meant to be a highly efficient way of cloning
essentially similar belief functions.  It assumes that the frame of
<bel-fun> and the new <frame> differ only in the names of the
attributes.  That is, there must be a one to one correspondence between
the attributes of <bel-fun> and <frame> such that each differs only by
name and not in the possible values for the outcome space."
  (let ((res (copy-belief-function bel-fun)))
    (setf (belief-function-frame res) frame)
    res))


;;; mix -- forms a weighted mixture of several belief functions.
;;; Note: as this uses mapcan to do its basic stuff, if the two lists
;;; are of unequal length, the excess elements of the longer one will
;;; be ignored.
(defun mix (list-bels list-weights)
  (declare (type (List Belief-Function) list-bels)
	   (type (List Number) list-weights)
	   (:returns (type Belief-Function average-bel-fun)))
  "Produces a new belief function by taking a weighted (using weights
<list-weights>) average over the focal elements of the belief functions
in <list-bels>."
  (when (find (val-frame (car list-bels))
	      (cdr list-bels) :key #'val-frame
	      :test-not #'equal)
	(error "mix:All values must have the same frame"))
  (make-belief-function :frame (val-frame (car list-bels))
       :ms (colapse-ms-list
	    (mapcan #'rescale-ms list-bels list-weights)
	    :op #'+)))

;; rescale-ms -- takes a belief function and a weight and returns a
;; new-ms-list formed by copying and rescaling all of the elements of
;; the old list.
(defun rescale-ms (bel-fun weight)
  (declare (type Belief-Function bel-fun) (type Number weight)
	   (:returns (type Belief-Function weight-x-bel-fun)))
  "Rescales the ms-list of <bel-fun> by <weight>."
  (mapcar #'(lambda (ms-val &aux (nm-val (copy-m-value ms-val)))
	      (setf (m-value-m nm-val) (the long-float (* weight (m-value-m nm-val))))
	      nm-val)
	  (belief-function-ms bel-fun)))



;;; provide when loaded
(bel-provide :arithmatic)
