" (c) 1990, 1991, 1992 Copyright (c) University of Washington
  Written by Stephen Soderland, Tony Barrett and Daniel Weld.

  All rights reserved. Use of this software is permitted for non-commercial
  research purposes, and it may be copied only for that use.  All copies must
  include this copyright message.  This software is made available AS IS, and
  neither the authors nor the University of Washington make any warranty about
  the software or its performance.

  When you first acquire this software please send mail to 
  bug-ucpop@cs.washington.edu; the same address should be used for problems."

(in-package "UCPOP")

(use-package "VARIABLE")

(export '(define reset-domain record-vcr))

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;; 1. Variables

(defvar *Templates* nil)		; list of dummy steps
(defvar *scr-rules* nil)		; list of scrs
(defvar *search-limit* 2000)		; max number of plans created

;;; Statistics related variables

(defvar *nodes-visited* 0)		; # plans visited in search
(defvar *plans-created* 0)		; # plans created in search
(defvar *branch* 0)			; compute avg branch factor

;;; Variables for Stuart Russell's IE search routine

(defvar *ie-limit*)
(defvar *ie-branches*)
(defvar *recording* nil)

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;; 2. Interface functions

(defmacro DEFINE ((dtype name) &body body)
  (cond ((eq dtype 'operator)
	 `(apply #'defoper ',(cons name body)))
	((eq dtype 'scr)
	 `(apply #'defscr ',(cons name body)))))

;;;;;;;;;;;;;;;;;;;;;;;;
;;;  Purge a previously defined domain theory.
(defun RESET-DOMAIN ()
  (setf *templates* nil))

;;;;;;;;;;;;;;;;;;;;;;;;
;;;  Enable the vcr
(defun RECORD-VCR ()
  (setf *recording* t))

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;; 3. Defining operators

(defun DEFOPER (name &key parameters at-time context precondition
			  effects description)
  (declare (ignore description))
  (when at-time 
    (error "Explicit time points not implemented."))
  (when context
    (error "Contexts are not implemented."))
  (when (member name *templates* :key #'(lambda (a) (car (p-step-action a))))
    (error "Two actions with name ~a" name))
  (let* ((vars (parm-vars parameters))
	 (pre (remove nil `(,precondition ,@(parm-types parameters)))))
    (setf pre (if (> (length pre) 1) (cons :and pre) (car pre)))
    (when pre (test-wd pre vars))
    (push (make-p-step :action (cons name vars)
		       :parms (parm-vars parameters)
		       :precond pre
		       :add (mapcar #'(lambda (x) 
					(apply #'defeffect 
					       `(,vars ,@x)))
				    effects))
	  *templates*)))

(defun DEFEFFECT (vlst &key effect forall when likely)
  (when likely (error "Probability not implemented."))
  (dolist (f (parm-vars forall))
    (unless (variable? f) 
      (error "attempt to univerally quantify constant [~a]" f))
    (push f vlst))
  (when when (test-wd when vlst))
  (setf effect (if (eq :and (car effect)) (cdr effect) (list effect)))
  (dolist (e effect)
    (dolist (v (parm-vars forall))
      (unless (member v (if (eq :not (car e)) (cadr e) e))
	(error "Universal effect [~a] should mention :forall variable [~a]"
	       e v))))
  (when (parm-types forall)
    (if when (setf when `(:and ,@(parm-types forall) ,when))
      (setf when `(:and ,@(parm-types forall)))))
  (make-effect :forall (parm-vars forall)
	       :precond when
	       :add effect))

;;;;;;;;;;;;;;;;;;;;;;;;
;;;  Test to see if logical equation is syntacticly correct
(defun TEST-WD (wd vlst)
  (cond ((or (eq (car wd) :exists) (eq (car wd) :forall))
	 (unless (null (cdddr wd)) (cerror "" "illegal expression: ~a" wd))
	 (dolist (v (cadr wd))
	   (test-typed-var v)
	   (push (cadr v) vlst))
	 (test-wd (caddr wd) vlst))
	((or (eq (car wd) :eq) (eq (car wd) :neq))
	 (unless (null (cdddr wd)) (cerror "" "illegal expression: ~a" wd))
	 (test-arguments (cdr wd) vlst))
	((or (eq (car wd) :and) (eq (car wd) :or))
	 (dolist (e (cdr wd))
	   (test-wd e vlst)))
	((eq (car wd) :not)
	 (unless (null (cddr wd)) (cerror "" "illegal expression: ~a" wd))
	 (test-wd (cadr wd) vlst))
	(t (test-term wd vlst))))

(defun TEST-TERM (wd vlst)
  (unless (and (consp wd) (symbolp (car wd)))
    (cerror "" "illegal term: ~a" wd))
  (cond ((eq (car wd) :not)
	 (unless (null (cddr wd)) (cerror "" "illegal term: ~a" wd))
	 (test-term (cadr wd) vlst))
	(t (test-arguments (cdr wd) vlst))))

(defun TEST-ARGUMENTS (as vlst)
  (dolist (p as)
    (when (and (variable? p) (not (my-member p vlst)))
      (cerror "" "unbound variable ~a" p))
    (when (consp p) (cerror "" "illegal constant ~a" p))))

(defun TEST-TYPED-VAR (v)
  (unless (and (consp v)
	       (symbolp (car v)) (not (variable? (car v)))
	       (variable? (cadr v)) (null (cddr v)))
    (cerror "" "illegal typed variable ~a" v)))

(defun PARM-VARS (parms)
  (mapcar #'(lambda (p) (if (listp p) (cadr p) p)) parms))

(defun PARM-TYPES (parms)
  (remove-if #'symbolp parms))

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;; 4. Search routines

;;;;;;;;;;;;;;;;;;;;;;;;
;;;  A simple best first search strategy.  Returns 3 values: the found state,
;;;  the average branching factor, and the number of generated but unexplored 
;;;  states.  The search will only generate up to LIMIT states.  
(defun BESTF-SEARCH (initial-state daughters-fn goal-p rank-fn limit)
  (let ((branches nil))                         ; compute average branch factor
    (do* ((current-entry nil (car search-queue))
          (current-state initial-state (cdr current-entry))
          (search-queue nil (cdr search-queue)))
         ((or (null current-state)
	      (funcall goal-p current-state)
              (> 0 limit))
          (values current-state
                  (if (null branches) 0
                      (div* (apply #'+ branches) (length branches)))
                  (length search-queue)))
      (incf *nodes-visited*)
      (let ((children (funcall daughters-fn current-state)))
        (setf limit (- limit (length children)))
        (setf search-queue
              (merge
                  'list search-queue
                  (sort (mapcar #'(lambda (x) (cons (funcall rank-fn x) x))
                                children)
                        #'< :key #'car)
                  #'< :key #'car))
        (push (length children) branches)))))

;;;;;;;;;;;;;;;;;;;;;;;;
;;;  IE search function written by Stuart Russell 
;;;  (See "Efficient Memory-Bounded Search Methods" in ECAI-92)

(defun CALL-IE (node successors goalp rank-fn limit)
  (setf *ie-limit* limit)
  (setf *ie-branches* (cons 0 0))
  (let ((solution (ie (cons (funcall rank-fn node) node)
                      goalp successors rank-fn most-positive-single-float)))
    (values solution (if (zerop (car *ie-branches*)) 0 
                         (div* (cdr *ie-branches*) (car *ie-branches*))) 0)))

(defun IE (vnode goalp successors rank-fn bound &aux children)
  (cond ((or (funcall goalp (cdr vnode)) (> 0 *ie-limit*))
         (cdr vnode))
        ((null (setf children (mapcar #'(lambda (child)
                                          (cons (funcall rank-fn child) child))
                                      (funcall successors (cdr vnode)))))
         (setf (car vnode) most-positive-single-float)
         nil)
        (t (incf *nodes-visited*)
           (decf *ie-limit* (length children))
           (incf (car *ie-branches*)) (incf (cdr *ie-branches*) 
					    (length children))
           (dolist (vn children)    ;;; pathmax
             (setf (car vn) (max (car vn) (car vnode))))
           (do ()
               ((> (car vnode) bound))
             (setf children
                   (sort children #'< :key #'car))
             (let* ((best (car children))
                    (rest (cdr children))
		    ;; best sibling value
                    (new-bound (apply #'min (cons bound (mapcar #'car rest)))))
               (let ((v (ie best goalp successors rank-fn new-bound)))
                 (when v (return v)))
               (if (and rest (< (caar rest) (car best)))
                 (setf (car vnode) (caar rest))
                 (setf (car vnode) (car best))))))))

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;; 5. Miscelaneous other routines

;;;;;;;;;;;;;;;;;;;;;;;;
;;;  Record a single frame of the movie in the vcr.
(defun VCR-FRAME (parent reason child)
  (when *recording*
    (vcr-add-frame parent reason child))
  child)

;;;;;;;;;;;;;;;;;;;;;;;;
;;; Add new high step number to each variable id for every term in 
;;; step from template.
(defun INSTANTIATE-STEP (step num)
  (labels ((instantiate-effect (e)
	     (make-effect 
	      :id num
	      :forall (instantiate-term (effect-forall e) num)
	      :precond (instantiate-term (effect-precond e) num)
	      :add (instantiate-term (effect-add e) num))))
    (let ((s (find num (p-step-cache step) :key #'p-step-id)))
      (if s s
	(car (push 
	      (make-p-step
	       :id num
	       :action (instantiate-term (p-step-action step) num)
	       :precond (instantiate-term (p-step-precond step) num)
	       :add (mapcar #'instantiate-effect (p-step-add step)))
	      (p-step-cache step)))))))

;;;;;;;;;;;;;;;;;;;;;;;;
;;;  Get today's date
(defun TODAY ()
  (let ((d (multiple-value-list (get-decoded-time))))
    (format nil "~a/~a ~a at ~a:~a:~a"
            (nth 4 d) (nth 3 d) (nth 5 d) (nth 2 d) (nth 1 d) (nth 0 d))))

;;;;;;;;;;;;;;;;;;;;;;;;
;;;  Reset the global variables used for collecting statistics on the
;;;  planner.
(defun RESET-STAT-VARS ()
  (setf *nodes-visited* 0)
  (setf *unify-count* 0)
  (setf *compute-rank-unifies* 0)
  (setf *add-bind-count* 0)
  (setf *branch* 0)
  (setf *plans-created* 0))

;;;;;;;;;;;;;;;;;;;;;;;;
;;;  A divide routine that does not blow up on zero.
(defun div* (x y)
  (if (= y 0) (* 99999 99999 99999) (/ x y)))

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;; 6. Replace slow lisp utilities

(defun MY-MEMBER (elt lst &key (test #'eq))
  (dolist (l lst nil)
    (when (funcall test elt l) (return t))))

(defun REMOVE-1 (elt lst &key (test #'eq))
  (cond ((null lst) nil)
	((funcall test elt (car lst)) (cdr lst))
	(t (cons (car lst) (remove-1 elt (cdr lst) :test test)))))

(defun DELETE-1 (elt lst &key (test #'eq))
  (cond ((null lst) nil)
	((funcall test elt (car lst)) (cdr lst))
	(t (setf (cdr lst) (delete-1 elt (cdr lst) :test test))
	   lst)))

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;; 7. Print functions

;;;;;;;;;;;;;;;;;;;;;;;;;;
;;; Print function for Templates
(defun PRINT-TEMPLATES (&optional (templates *templates*))
  (format t "~&~%Templates:")
  (dolist (templ-n templates)
    (let ((action (p-step-action templ-n))
	  (pre-cond (p-step-precond templ-n))
	  (add (p-step-add templ-n)))
      (format t "~&~a~%  Pre  : ~a~%  Add  :~%"
	      action pre-cond)
      (dolist (a add)
	(format t "  <when: ~a  Add: ~a>~%" 
		(effect-precond a) (effect-add a))))))
		
