;;; TAQL Compiler, Type Declaration Module
;;;
;;; Gregg Yost, Erik Altmann
;;; School of Computer Science
;;; Carnegie Mellon University
;;;
;;; Working file: /afs/cs/user/altmann/soar/taql/types.lisp
;;; Created March 19, 1991
;;;
;;; This file contains the type declarations and associated routines for
;;; the lisp structures used in TAQL.  It also contains special variable
;;; proclamations for all special variables that are used in more than
;;; one file.
;;;
;;; EXPORTED ROUTINES:
;;;
;;;    - FILL THIS IN
;;;
;;; Known bugs/funnies:
;;;
;;; =======================================================================
;;; Modification history:
;;; =======================================================================
;;;
;;; 6-11-91 - gry -  Added exports for use-taql-library and duplicate-tc-warn.
;;;
;;; 5-21-91 - gry - Export symbols that TAQL supplies to users.  Frank
;;;   Ritter asked for this, and I'm sure I've missed some.
;;;
;;; 4-5-91 through 4-22-91 - gry - added space model support
;;;
;;; 3-19-91 - gry - Created.

;;; *** BEGIN CODE ***

(eval-when (compile load eval)
  (lispsyntax))

(eval-when (compile load eval)
   (if (find-package "COMMON-LISP-USER")
       (in-package "COMMON-LISP-USER")
       (in-package "USER")))

;;;; Below are the special variables and constants that are used in more than
;;;; one file:

(eval-when (compile load eval)

  (defconstant *lhs-relations* '(<> = > < >= <= <=>)
    "A list of the Soar LHS relations.")

  ;; 5-feb-90 -- GRY
  (defconstant *num-separator* '<>
    "The symbol use to separate the number at the end of a generated production name from the rest of the production name.")

  ;; 5-feb-90 -- GRY
  (defconstant *name-separator* '<>
    "The symbol use to separate components of a generated production name.")

  ;; This list must be ordered the same as *taql-context-variables*, below.
  ;;
  (defconstant *taql-context-objects*
    '(goal problem-space state operator object supergoal superspace
      superstate superoperator top-goal top-space top-state
      top-operator))

  ;; This list must be ordered the same as *taql-context-objects*.
  ;;
  (defconstant *taql-context-variables*
    '(=goal =problem-space =state =operator =object =object =superspace
      =superstate =superoperator =top-goal =top-space =top-state
      =top-operator))

  ;; ema, 11-21 - the eval-when is new:
  ;; Re-introduced(?) eval-when TFMcG 5-Sep-91
  (eval-when (compile eval load)
	     (proclaim (cons 'special *taql-context-variables*)))

  ;; 2-3:
  (defconstant *taql-types*
    '((propose-task-operator pto)
      (propose-task-state pts)
      (propose-space psp)
      (propose-initial-state pis)
      (propose-operator po)
      (compare c)
      (prefer p)
      (goal-test-group gtg)
      (apply-operator ao)
      (augment a)
      (result-superstate rs)
      (evaluation-properties ep)
      (evaluate-object eo)
      (propose-superobjects psu)
      (operator-control oc))
    "An a-list, where cars are valid TC types and cadrs are abbreviations.")

  ;; Enveloped in eval-when -- TFMcG 5-Sep-91
  (eval-when (compile eval load)
	     (proclaim
	      '(special
		*segment-name-most-recently-added-to*
		*current-segment-name*
		*taql-default-mode*
		*current-taql-name*
		*segment-table*
		*pseudo-sp-context*
		*prod-name-prefix*
		*proposed-operator-id*)))

  (export '(defprimitive
	    defenumeration
	    defobject
	    defobject-merge
	    defoperator
	    defoperator-merge
	    excise-data-model
	    print-data-model
	    untrace-load
	    trace-load
	    all-info-declared
	    all-types-declared
	    all-operators-declared
	    end-segment
	    begin-segment
	    segment-group
	    defspace
	    print-space-model
	    print-space-graph
	    excise-space-model
	    all-spaces-declared
	    check-spaces
	    taqlp
	    tpm
	    ptaql
	    taqln
	    taql-verbose
	    disable-in-taql
	    duplicate-tc-warn
	    use-taql-library
	    ))

  (export (mapcar #'car *taql-types*))

  ) ; End eval-when (compile load eval)

;; A segment-group table is a hash table of lists of segment names, keyed
;; by segment-group name.  A nil value returned by gethash indicates an
;; undefined segment-group name, since every segment group name must contain
;; at least on segment in its expansion.
;;
(deftype segment-group-table ()
  '(or null hash-table))

;; A segment table is a hash table of segment objects, keyed by segment name.
;;
(deftype segment-table ()
  '(or null hash-table))

(defstruct segment
  (name			nil	:type symbol)
  (tc-names		nil	:type list)
  (declared-data-model	nil	:type data-model)
  (inferred-data-model	nil	:type data-model))

;; A data model is a hash table of type-info objects, keyed by type name.
;;
(deftype data-model ()
  '(or null hash-table))

;; The global field is only meaningful for declared type information, for which
;; it will be t.
;;
(defstruct type-info
  (name			nil	:type symbol)
  (primitive		nil	:type (or symbol list))
  (global		nil	:type symbol)
  (attribute-info	nil	:type attribute-info-table))

;; An attribute info table is a hash table of attribute-info objects, keyed
;; by attribute name.
;;
(deftype attribute-info-table ()
  '(or null hash-table))

;; The 'required' field is only meaningful for operator types (the kinds
;; declared/printed using defoperator).  For such types, the default is that
;; an attribute is required upon proposal unless explicitly declared to be
;; optional.
;;
(defstruct attribute-info
  (name			nil	:type symbol)
  (value-types		nil	:type list)
  (used-values		nil	:type list)
  (required		t	:type symbol))

;; A space-info table is a hash table of space-info objects, keyed
;; by problem space name.
;;
(deftype space-info-table ()
  'hash-table)

(defstruct space-info
  (name			nil			:type symbol)
  (declared-space-model	(make-space-model)	:type space-model)
  (inferred-space-model	(make-space-model)	:type space-model))

;; These variables are declared elsewhere, but we need them here.
;;
(proclaim
 '(special
   *taql-default-mode*))

;; A nil home-segment-names field indicates an uninitialized (and hence
;; meaningless) space model.
;;
(defstruct space-model
  (name			nil
			:type symbol)
  (home-segment-names	nil
			:type list)
  (component-info-table	(make-hash-table :size 10)
			:type space-component-table)
  (propose-operator-table
			(make-hash-table :size 10)
			:type space-component-table)
  (apply-operator-table	(make-hash-table :size 10)
			:type space-component-table)
  (sticky		*taql-default-mode*
			:type symbol))

;; A space-component-table is a hash table of component-info objects, keyed
;; by problem component name.  For operators, the component name is the
;; operator name.
;;
(deftype space-component-table ()
  'hash-table)

(defstruct component-info
  (name			nil	:type symbol)
  (implementation-types	nil	:type list)
  (TC-alist		nil	:type list))

;; A space-function table is a hash table of lists of space-function objects,
;; keyed by 
;;
;;   1. (space . <space-name>), where <space-name> is the name of the space
;;      that the function is performed for.  The space name can be *unknown*
;;      when the name of the space can't be determined.
;;
;;   OR
;;
;;   2. (operator . <operator-name>), where <operator-name> is the name of the 
;;      operator that the function is performed for.  The operator name can
;;      be *unknown* when the name of the space can't be determined.
;;
(deftype space-function-table ()
  'hash-table)

(defstruct space-function
  (space-name		nil		:type symbol)
  (function		nil		:type list)
  (propose-tc-names	nil		:type list))

;;; Copier functions.

(defun segment-copier (segment)
  (make-segment
   :name (segment-name segment)
   :tc-names (copy-list (segment-tc-names segment))
   :declared-data-model (let ((model
			       (segment-declared-data-model segment)))
			  (if model
			    (data-model-copier model)))
   :inferred-data-model (let ((model
			       (segment-inferred-data-model segment)))
			  (if model
			    (data-model-copier model)))))

(defun data-model-copier (data-model)
  (copy-hash-table data-model #'type-info-copier))

(defun type-info-copier (type-info)
  (make-type-info
   :name (type-info-name type-info)
   :global (type-info-global type-info)
   :primitive (let ((prim (type-info-primitive type-info)))
		(if (consp prim)
		  (copy-list prim)
		  ;; ELSE
		  prim))
   :attribute-info (let ((att-info (type-info-attribute-info type-info)))
		     (if att-info
		       (attribute-info-table-copier att-info)))))

(defun attribute-info-table-copier (att-info-table)
  (copy-hash-table att-info-table #'attribute-info-copier))

(defun attribute-info-copier (att-info)
  (make-attribute-info
   :name (attribute-info-name att-info)
   :value-types (copy-list (attribute-info-value-types att-info))
   :used-values (copy-list (attribute-info-used-values att-info))
   :required (attribute-info-required att-info)))

;; This only sort of copies that hash table.  The result might not be
;; the same size as the original, and may not have the same :rehash-size,
;; :test, and :rehash-threshold.  But it works for my purposes here.
;; It WOULDN'T work if I ever specified :test in the hash tables I use.
;; But I always just use the default test (eql).
;;
;; The optional copier argument specifies a function that will make a copy
;; of a hash-table element.  If it is nil or not specified, elements are
;; not copied, but are tranferred as-is to the new table.
;;
(defun copy-hash-table (hash-table &optional (copier nil))
  (let ((result-table
	 (make-hash-table
	  :size (max 20
		     (round (* 1.5
			       (hash-table-count hash-table)))))))
    (maphash #'(lambda (key element)
		 (setf (gethash key result-table)
		       (if copier
			 (funcall copier element)
			 ;; ELSE
			 element)))
	     hash-table)

    result-table))
