;;; TAQL Compiler, Space-Model Module
;;;
;;; Gregg Yost, Erik Altmann
;;; School of Computer Science
;;; Carnegie Mellon University
;;;
;;; Working file: /afs/cs/user/altmann/soar/taql/space-model.lisp
;;; Created April 5, 1991
;;;
;;; This file implements the functions involved with TAQL's problem space
;;; model facility (space declarations and checking, ...).
;;;
;;; EXPORTED ROUTINES:
;;;
;;;    - FILL THIS IN
;;;
;;; Known bugs/funnies:
;;; 
;;;    - The global variables should be described where they are declared
;;;      special.
;;;    - There are other things we might want to check for in check-spaces,
;;;      though some of these are non-trivial
;;;       - result-return type matches space function
;;;       - disconnected space graph (you can see this when you do a
;;;         print-space-graph, but check-spaces doesn't check it)
;;;
;;; =======================================================================
;;; Modification history:
;;; =======================================================================
;;;
;;; 4-22-91 - gry - first version ready for internal use
;;;
;;; 4-5-91 - gry - Created.

;;; *** BEGIN CODE ***

(eval-when (compile load eval)
  (lispsyntax))

(eval-when (compile load eval)
   (if (find-package "COMMON-LISP-USER")
       (in-package "COMMON-LISP-USER")
       (in-package "USER")))

;; Enveloped in eval-when -- TFMcG 15-Aug-91
(eval-when (compile eval load)
	   (proclaim
	    '(special
	      *space-info-table*
	      *space-function-table*
	      *printed-space-keys*
	      *all-spaces-declared*
	      *sticky-all-spaces-declared*
	      *spaces-need-checking*
	      )))

(defmacro defspace (&body body)
  `(defspace-aux ',body))

(defun defspace-aux (body)
  (let ((error-detected nil)
	(space-name (car body)))

    (when (null body)
      (setq error-detected t)
      (taql-warn2 "Defspace: missing problem-space name."))

    (when (not (symbolp space-name))
      (setq error-detected t)
      (taql-warn2 "Defspace: problem space name must be a symbol, but found ~S"
		  space-name))

    ;; When there is an existing declaration for the same space, excise it,
    ;; and give the new declaration the same home segment as the old one.
    ;; This is the same thing that we do for TCs.  And, as for TCs, this
    ;; leaves you in the segment the space was originally defined in.
    ;;
    (let ((existing-model (get-space-model 'declared space-name)))
      (when existing-model
	(taql-warn-soft2
	 "Defspace: overwriting previous declaration of problem space ~S."
	 space-name)
	(setq *current-segment-name*
	      (car (space-model-home-segment-names existing-model)))
	(eval `(excise-space-model (declared ,space-name)))))

    (setq *segment-name-most-recently-added-to*
	  *current-segment-name*)

    ;; Binding *current-taql-name* to the name of the space is a bit of
    ;; a hack, but group-aruments uses taql-warn, so we have to.
    
    (let* ((*current-taql-name* (car body))
	   (keywords '(:initial-state-proposal
		       :entailment :goal-testing :result-returning
		       :selection :evaluation :default-operator-proposal
		       :default-operator-application :operators))
	   (args (group-arguments (cdr body) keywords 'defspace)))

      (dolist (keyword keywords)
	(let ((entry (assoc keyword (cdr args))))
	  (when (or (not entry)
		    (cddr entry))
	    (setq error-detected t)
	    (taql-warn2 "Defspace ~S: exactly one ~(~S~) keyword must appear."
			space-name keyword))))

      (when (car args)
	(setq error-detected t)
        (taql-warn2 "Defspace ~S: all arguments must be values of keywords."
		    space-name))

      (dolist (entry (cdr args))
	(when (and (member (car entry) keywords)
		   (not (eql (car entry) ':operators)))
	  (let* ((legal-values
		  (case (car entry)
		    ((:initial-state-proposal :selection
		      :default-operator-proposal :default-operator-application)
		     '(none direct subspace))
		    ((:entailment :goal-testing :result-returning :evaluation)
		     '(none direct))))
		 (actual-values (setf (cadr entry)
				      (if (consp (cadr entry))
					(remove-duplicates (cadr entry))
					;; ELSE
					(list (cadr entry)))))
		 (illegal-values
		  (set-difference actual-values legal-values)))
	    (when illegal-values
	      (setq error-detected t)
	      (taql-warn2 "Defspace ~S: the following are not legal values ~
                           for ~(~S~): ~S"
			  space-name (car entry) illegal-values))
	    (when (and (member 'none actual-values)
		       (member-if-not #'(lambda (x)
					  (eql x 'none))
				      actual-values))
	      (setq error-detected t)
	      (taql-warn2 "Defspace ~S: 'none' cannot appear along with ~
                             other values: ~S"
			  space-name (list (car entry) actual-values))))))

      (multiple-value-bind (parsed-operators error-detected-1)
	  (parse-defspace-operators space-name
				    (cadr (assoc :operators (cdr args)))
				    (cadr (assoc :default-operator-proposal
						 (cdr args)))
				    (cadr (assoc :default-operator-application
						 (cdr args))))

	(when error-detected-1
	  (setq error-detected t))

	(when (not error-detected)
	  (install-space-declaration space-name (cdr args) parsed-operators))))

    t))

;; Return two values:
;;
;;   1. A list containing entries of the form
;;        (op-name propose-implementation-types apply-implementation-types),
;;      one entry for each operator specified in op-arg (or nil if
;;      op-arg = none).
;;
;;   2. A value that is non-nil iff an error was detected.  The first 
;;      return value is undefined when an error has been detected.
;;
(defun parse-defspace-operators (space-name op-arg default-operator-proposal
				 default-operator-application)

  (let ((error-detected nil)
	(parsed-ops nil))

    (cond ((null op-arg)
	   (setq error-detected t)
	   (taql-warn2 "Defspace ~S: :operators list must contain at least ~
                        one operator."
		       space-name))

	  ((and (not (consp op-arg))
		(not (eql op-arg 'none)))
	   (setq error-detected t)
	   (taql-warn2 "Defspace ~S: :operators value must be 'none' or a ~
                        list, but found ~S"
		       space-name op-arg))

	  ((consp op-arg)
	   (dolist (single-op op-arg)
	     (multiple-value-bind (parsed-op error-detected-1)
		 (parse-defspace-operator space-name single-op
					  default-operator-proposal
					  default-operator-application)
	       (when error-detected-1
		 (setq error-detected t))
	       (push parsed-op parsed-ops)))))

    (values parsed-ops error-detected)))

(defun parse-defspace-operator (space-name op-arg default-propose
					   default-apply)

  (let ((error-detected nil)
	(parsed-op nil)
	(op-name (if (consp op-arg)
		   (car op-arg)
		   ;; ELSE
		   op-arg)))

    (when (or (not (symbolp op-name))
	      (keywordp op-name))
      (setq error-detected t)
      (taql-warn2 "Defspace ~S: expected symbolic operator name, but found ~S"
		  op-name))

    (cond ((consp op-arg)
	   (let* ((args (group-arguments (cdr op-arg)
					 '(:propose :apply)
					 'defspace))
		  (propose (assoc :propose (cdr args)))
		  (apply (assoc :apply (cdr args))))

	     (when (car args)
	       (setq error-detected t)
	       (taql-warn2 "Defspace ~S, operator ~S: all arguments must be ~
                            values of :propose/:apply, but found extra ~
                            arguments ~S"
			   space-name op-name (car args)))

	     (multiple-value-bind (propose-vals apply-vals error-detected-1)
		 (check-propose-apply-values space-name op-name propose apply)
	       (when error-detected-1
		 (setq error-detected t))
	       (setq parsed-op
		     (list op-name
			   (or propose-vals default-propose)
			   (or apply-vals default-apply))))))
	  (t
	   (setq parsed-op (list op-arg default-propose default-apply))))

    (values parsed-op error-detected)))

(defun check-propose-apply-values (space-name op-name propose apply)
  (let ((error-detected nil)
	(propose-vals (if (listp (cadr propose))
			(remove-duplicates (cadr propose))
			;; ELSE
			(list (cadr propose))))
	(apply-vals (if (listp (cadr apply))
		      (remove-duplicates (cadr apply))
		      ;; ELSE
		      (list (cadr apply)))))

    (when (cddr propose)
      (setq error-detected t)
      (taql-warn2 "Defspace ~S, operator ~S: at most one :propose keyword is ~
                   allowed."
		  space-name op-name))

    (when (cddr apply)
      (setq error-detected t)
      (taql-warn2 "Defspace ~S, operator ~S: at most one :apply keyword is ~
                   allowed."
		  space-name op-name))

    (let ((illegal-values
	   (set-difference propose-vals '(none direct))))
      (when illegal-values
	(setq error-detected t)
	(taql-warn2 "Defspace ~S, operator ~S: the following are not legal ~
                     values for :propose: ~S"
		    space-name op-name illegal-values))
      (when (and (member 'none propose-vals)
		 (member-if-not #'(lambda (x)
				    (eql x 'none))
				propose-vals))
	(setq error-detected t)
	(taql-warn2 "Defspace ~S, operator ~S: 'none' cannot appear along ~
                     with other values: ~S"
		    space-name op-name (list :propose propose-vals))))

    (let ((illegal-values
	   (set-difference apply-vals '(none direct subspace))))
      (when illegal-values
	(setq error-detected t)
	(taql-warn2 "Defspace ~S, operator ~S: the following are not legal ~
                     values for :apply: ~S"
		    space-name op-name illegal-values))
      (when (and (member 'none apply-vals)
		 (member-if-not #'(lambda (x)
				    (eql x 'none))
				apply-vals))
	(setq error-detected t)
	(taql-warn2 "Defspace ~S, operator ~S: 'none' cannot appear along ~
                     with other values: ~S"
		    space-name op-name (list :apply apply-vals))))

    (values propose-vals apply-vals error-detected)))

;; This assumes that no error was detected in defspace's arguments.
;;
(defun install-space-declaration (space-name keywords parsed-operators)
  (setq *spaces-need-checking* t)

  (let* ((space-info (add-space space-name))
	 (decl-model (make-space-model
		      :name space-name
		      :home-segment-names (list *current-segment-name*)))
	 (component-table (space-model-component-info-table decl-model))
	 (propose-table (space-model-propose-operator-table decl-model))
	 (apply-table (space-model-apply-operator-table decl-model)))

    (dolist (entry keywords)
      (when (not (eql (car entry) ':operators))
	(let ((component-name (intern (symbol-name (car entry))))
	      (impl-types (remove 'none (cadr entry))))
	  (setf (gethash component-name component-table)
		(make-component-info :name component-name
				     :implementation-types impl-types)))))

    (dolist (parsed-op parsed-operators)
      (let ((op-name (car parsed-op))
	    (propose-impl-types (remove 'none (cadr parsed-op)))
	    (apply-impl-types (remove 'none (caddr parsed-op))))
	(setf (gethash op-name propose-table)
	      (make-component-info :name op-name
				   :implementation-types propose-impl-types))
	(setf (gethash op-name apply-table)
	      (make-component-info :name op-name
				   :implementation-types apply-impl-types))))

    (setf (space-info-declared-space-model space-info) decl-model)))

;; EXPORTED FUNCTION:
;;
;; This must be called every for every TC that is loaded.  It builds up the
;; inferred space model.  The TC is registered as being part of every space
;; in space-names.  If space-names is nil, it is registered as being in
;; the *unknown* space.
;;
;; For tc-types other than propose-operator, propose-task-operator, and
;; apply-operator, the operator-names argument need not be anything meaningful.
;; But for these three TC types, the TC is registered as being part of each
;; of the named operators in each of the named spaces.  If one of the symbols
;; :any-name or :no-name appears as an operator name, it is treated as
;; *unknown*.
;;
;; For tc-types other than propose-space, the space-function argument need
;; need not be anything meaningful.  For propose-space TCs, it must be the
;; value returned by parse-space-function for the TC (see taql-compiler.lisp).
;;
(defun add-tc-to-spaces (space-names tc-name tc-type operator-names
				     space-function)
  (let ((component-name (tc-type-to-component-name tc-type))
	(space-model nil))

    (when (null space-names)
      (setq space-names '(*unknown*)))
    
    (dolist (space-name space-names)
      (setq *spaces-need-checking* t)

      (when (or (not (symbolp space-name))
		(variable-p space-name))
	(setq space-name '*unknown*))
      (when (not (eql component-name 'problem-space-proposal))
	(setq space-model
	      (space-info-inferred-space-model (add-space space-name)))
	(pushnew *current-segment-name*
		 (space-model-home-segment-names space-model)))
      (cond ((member component-name '(operator-proposal operator-application))
	     (when (null operator-names)
	       (setq operator-names '(*unknown*)))
	     (dolist (op-name operator-names)
	       (when (or (not (symbolp op-name))
			 (variable-p op-name)
			 (member op-name '(:no-name :any-name)))
		 (setq op-name '*unknown*))
	       (add-tc-to-space-aux space-model tc-name tc-type op-name
				    component-name 'direct)))
	    ((eql component-name 'problem-space-proposal)
	     (add-to-space-function-table space-name space-function tc-name
					  tc-type))
	    (t
	     (add-tc-to-space-aux space-model tc-name tc-type nil
				  component-name 'direct))))))

;; This is also called by update-space-models-for-propose-space.
;;
(defun add-tc-to-space-aux (space-model tc-name tc-type operator-name
					component-name impl-type)

  (let ((component nil))
    (setq *spaces-need-checking* t)

    (if (member component-name '(operator-proposal operator-application))
      (let ((propose-table (space-model-propose-operator-table space-model))
	    (apply-table (space-model-apply-operator-table space-model))
	    (component-table (space-model-component-info-table space-model)))

	(if (eql component-name 'operator-proposal)
	  (pushnew impl-type
		   (component-info-implementation-types
		    (add-component 'default-operator-proposal
				   component-table)))
	  ;; ELSE
	  (pushnew impl-type
		   (component-info-implementation-types
		    (add-component 'default-operator-application
				   component-table))))

	(when (not (or (gethash operator-name propose-table)
		       (gethash operator-name apply-table)))
	  ;; If we have neither propose nor apply entries for this operator,
	  ;; and we already know that the operator is applied in a subspace,
	  ;; then add info about the subspace application to the operator's
	  ;; entry in the apply table.
	  ;;
	  (let ((func-list (gethash (cons 'operator operator-name)
				    *space-function-table*)))
	    (when func-list
	      (let ((apply-entry (add-component operator-name apply-table))
		    (default-apply
		      (add-component
		       'default-operator-application component-table)))
		       
		(pushnew 'subspace
			 (component-info-implementation-types apply-entry))
		(pushnew 'subspace
			 (component-info-implementation-types default-apply))
		(setf (component-info-tc-alist apply-entry)
		      (add-alist-values
		       'propose-space
		       (apply #'append
			      (mapcar #'space-function-propose-tc-names
				      func-list))
		       (component-info-tc-alist apply-entry)))))))

	(setq component
	      (add-component operator-name
			     (if (eql component-name 'operator-proposal)
			       propose-table
			       ;; ELSE
			       apply-table))))

      ;; ELSE component-name is not operator-proposal or operator-application

      (setq component
	    (add-component
	     component-name
	     (space-model-component-info-table space-model))))

    (pushnew impl-type (component-info-implementation-types component))
    (setf (component-info-tc-alist component)
	    (add-alist-value tc-type tc-name
			     (component-info-tc-alist component)))))

(defun add-to-space-function-table (space-name space-function tc-name tc-type)
  (let* ((space-function (copy-list space-function))
	 ;; Thing-spec will be either (operator OPERATOR-COND) or
	 ;; (space SPACE-NAME).
	 ;;
	 (thing-spec (car (last space-function)))
	 (thing-type (car thing-spec))
	 (thing-item (cadr thing-spec)))

    (setq *spaces-need-checking* t)

    (case thing-type
      (space
       (when (or (not (symbolp thing-item))
		 (variablep thing-item))
	 (setq thing-item '*unknown*)))
      (operator
       (when (consp thing-item)
	 ;; It is the list form of an OPERATOR-COND, we just want its name.
	 (setq thing-item (car thing-item)))
       (when (or (not (symbolp thing-item))
		 (variablep thing-item)
		 (member thing-item '(:no-name :any-name)))
	 (setq thing-item '*unknown*)))
      (t
       (error "INTERNAL TAQL ERROR:  Case selector fell through: ~S"
	      thing-type)))

    ;; The form of the last element is a little different on the space-function
    ;; we store in the space function table.  It is a dotted pair rather than
    ;; a list, and the cdr of the pair is just the space or operator name,
    ;; rather than the space name or OPERATOR-COND.
    ;;
    (setq thing-spec (cons thing-type thing-item))
    (setf (car (last space-function)) thing-spec)

    (let ((existing-element
	   (find-if #'(lambda (item)
			(and (eql space-name (space-function-space-name item))
			     (equal space-function
				    (space-function-function item))))
		    (gethash thing-spec *space-function-table*))))
      (if existing-element
	(push tc-name (space-function-propose-tc-names existing-element))
	;; ELSE
	(push (make-space-function :space-name space-name
				   :function space-function
				   :propose-tc-names (list tc-name))
	      (gethash thing-spec *space-function-table*))))

    (update-space-models-for-propose-space space-function tc-name tc-type)))

(defun update-space-models-for-propose-space (space-function tc-name tc-type)
  (let* ((func-tail
	  ;; Bind func-tail to the last three elements of space-function
	  (do ((func space-function (cddr func)))
	      ((null (cdddr func))
	       func)))
	 (operation-type (first func-tail))
	 (operation-object (second func-tail))
	 (for-object-type (car (third func-tail)))
	 (for-object-name (cdr (third func-tail))))

    (setq *spaces-need-checking* t)

    ;; We deliberately don't set/add-to the inferred space model's
    ;; home-segment-names list here, since the models we are
    ;; manipulating aren't the ones corresponding to the :space
    ;; keyword in the TC (but, rather, to the space(s) the propose-operator
    ;; TC performs a function for, e.g. the superspace when the function
    ;; is (apply operator OPERATOR-COND)).  The effect is that if we never
    ;; infer anything else for one of these spaces, the home-segment-names
    ;; list will be nil, and so this bit of inferred information won't be
    ;; printed out.  Which seems fine, but we should do something that will
    ;; let us catch errors where we propose a space to perform a function
    ;; for a non-existent space or operator.
    ;;
    (case for-object-type
      (space
       (add-tc-to-space-aux
	(space-info-inferred-space-model (add-space for-object-name))
	tc-name
	tc-type
	nil
	(case operation-type
	  (select 'selection)
	  (propose
	   (case operation-object
	     (state 'initial-state-proposal)
	     (operator 'default-operator-proposal)
	     (t
	      (error "INTERNAL TAQL ERROR: illegal operation-object ~S"
		     operation-object))))
	  (t
	   (error "INTERNAL TAQL ERROR: illegal operation-type ~S"
		  operation-type)))
	'subspace))
      (operator
       ;; This is an apply-operator space function.  Hunt for every space the
       ;; named operator is either proposed or applied in, and (1) add
       ;; 'subspace to the list of implementation types for applying that
       ;; operator and (2) add tc-name to the tc-type entry on the tc-alist
       ;; for applying that operator.
       (maphash #'(lambda (space-name space-info)
		    (declare (ignore space-name))
		    (let* ((space-model
			    (space-info-inferred-space-model space-info))
			   (apply-entry
			    (gethash for-object-name
				     (space-model-apply-operator-table
				      space-model))))
		      (when (or (gethash for-object-name
					 (space-model-propose-operator-table
					  space-model))
				apply-entry)
			(when (not apply-entry)
			  (setq apply-entry
				(add-component
				 for-object-name
				 (space-model-apply-operator-table
				  space-model))))
			(pushnew 'subspace
				 (component-info-implementation-types
				  apply-entry))
			(pushnew 'subspace
				 (component-info-implementation-types
				  (add-component
				   'default-operator-application
				   (space-model-component-info-table
				    space-model))))
			(setf (component-info-tc-alist apply-entry)
			      (add-alist-value
			       tc-type tc-name
			       (component-info-tc-alist apply-entry))))))
		*space-info-table*))
      (t
       (error "INTERNAL TAQL ERROR: illegal for-object-type ~S"
	      for-object-type)))))

(defmacro print-space-model (&body body)
  `(print-space-model-aux ',body))

(defun print-space-model-aux (body)
  (multiple-value-bind (model-list print-tc-names)
      (process-space-model-spec-args body)

    ;; Sort the model list so that it is alphabetized by space name

    (setq model-list
	  (sort model-list
		#'string-lessp
		:key #'(lambda (x)
			 (symbol-name (space-model-name x)))))
    
    (dolist (space-model model-list)
      (do-print-space-model space-model print-tc-names))

    t))

(defun process-space-model-spec-args (args)
  (let ((print-tc-names (not (member :no-tc-names args)))
	(include-inferred nil)
	(include-declared nil)
	(exclude-inferred nil)
	(exclude-declared nil)
	(model-list nil))

    (setq args (remove :no-tc-names args))

    (multiple-value-bind (include-parsed-triples exclude-parsed-triples)
	(process-model-spec-args-common args 'space)

      (dolist (triple include-parsed-triples)
	(multiple-value-bind (inferred-spaces declared-spaces)
	    (expand-parsed-triple-to-spaces triple)
	  (setq include-inferred
		(union inferred-spaces include-inferred))
	  (setq include-declared
		(union declared-spaces include-declared))))

      (dolist (triple exclude-parsed-triples)
	(multiple-value-bind (inferred-spaces declared-spaces)
	    (expand-parsed-triple-to-spaces triple)
	  (setq exclude-inferred
		(union inferred-spaces exclude-inferred))
	  (setq exclude-declared
		(union declared-spaces exclude-declared))))

      (setq include-inferred
	    (set-difference include-inferred exclude-inferred))

      (setq include-declared
	    (set-difference include-declared exclude-declared))

      (dolist (space-name include-inferred)
	(let ((model (if (member space-name include-declared)
		       (merge-inferred-and-declared space-name)
		       ;; ELSE
		       (get-space-model 'inferred space-name))))
	  (when model
	    (push model model-list))))

      (dolist (space-name include-declared)
	(let ((model (if (member space-name include-inferred)
		       nil
		       ;; ELSE
		       (get-space-model 'declared space-name))))
	  (when model
	    (push model model-list))))
      
      (values model-list print-tc-names))))

;; See process-model-spec-args-common in data-model.lisp for a description
;; of the form of a parsed model spec.
;;
;; Return two values: a list of the names of spaces whose inferred models
;; were specified, and a list of the names of spaces whose declared models
;; were specified.
;;
(defun expand-parsed-triple-to-spaces (parsed-model-spec)
  (let ((space-restriction (car parsed-model-spec))
	(inferred-spaces nil)
	(declared-spaces nil))

    (dolist (model-spec (cdr parsed-model-spec))
      (let ((space-names
	     (expand-model-spec-to-space-names model-spec)))
	(if (eql (car model-spec) 'inferred)
	  (setq inferred-spaces (union space-names inferred-spaces))
	  ;; ELSE
	  (setq declared-spaces (union space-names declared-spaces)))))

    (when (not (eql space-restriction :all))
      (setq inferred-spaces
	    (intersection inferred-spaces space-restriction))
      (setq declared-spaces
	    (intersection declared-spaces space-restriction)))

    (values inferred-spaces declared-spaces)))

(defun do-print-space-model (space-model print-tc-names)
  (let ((space-name (space-model-name space-model))
	(home-segments
	 (setf (space-model-home-segment-names space-model)
	       (sort (space-model-home-segment-names space-model)
		     #'string-lessp
		     :key #'symbol-name)))
	(components '(initial-state-proposal
		      entailment
		      goal-testing
		      result-returning
		      selection
		      evaluation))
	(default-op-components '(default-operator-proposal
				  default-operator-application)))

    (format t "~%(defspace ~(~S~)~%  ;; This space appears in segment~:[~;s~] "
	    space-name
	    (cdr home-segments))
    (print-list-with-commas home-segments)
    (princ #\.)
    (terpri)

    (dolist (component-name components)
      (print-space-component component-name space-model print-tc-names))

    (terpri)

    (dolist (component-name default-op-components)
      (print-space-component component-name space-model print-tc-names))

    (print-space-operators space-model print-tc-names)

    (format t ")~%")

    t))

(defun get-print-implementation-types (component-name space-model)
  (or (get-component-implementation-types component-name space-model)
      '(none)))

(defun print-space-component (component-name space-model print-tc-names)
  (let ((info (gethash component-name
		       (space-model-component-info-table space-model)))
	(implementation-types
	 (get-print-implementation-types component-name space-model)))

    (format t "~%  :~(~S~)~32T~(~S~)"
	    component-name
	    (if (cdr implementation-types)
	      implementation-types
	      ;; ELSE
	      (car implementation-types)))

    (print-space-model-tcs info print-tc-names)))

(defun print-space-operators (space-model print-tc-names)
  (let* ((default-propose-types
	   (get-print-implementation-types 'default-operator-proposal
					   space-model))
	 (default-apply-types
	   (get-print-implementation-types 'default-operator-application
					   space-model))
	 (proposals (space-model-propose-operator-table space-model))
	 (applications (space-model-apply-operator-table space-model))
	 (op-names (sort (union (hash-table-key-list proposals)
				(hash-table-key-list applications))
			 #'string-lessp
			 :key #'symbol-name)))

    (format t "~%  :operators ")
    (if op-names
      (progn
	(princ #\()
	(dolist (op-name op-names)
	  (print-space-operator op-name
				(gethash op-name proposals)
				(gethash op-name applications)
				print-tc-names
				default-propose-types
				default-apply-types))
	(format t "~%    )"))
      ;; ELSE
      (format t "none"))))

(defun print-space-operator (op-name proposal-component application-component
				     print-tc-names default-propose-types
				     default-apply-types)
  (let* ((propose-types
	  (or 
	   (and proposal-component
		(component-info-implementation-types proposal-component))
	   '(none)))
	 (apply-types
	  (or
	   (and application-component
		(component-info-implementation-types application-component))
	   '(none)))
	 (print-propose (not (set-equal propose-types default-propose-types)))
	 (print-apply (not (set-equal apply-types default-apply-types))))

    (if (or print-propose print-apply)
      (progn
	(format t "~%     (~S" op-name)
	(when print-propose
	  (format t " :propose ~(~S~)"
		  (if (cdr propose-types)
		    propose-types
		    ;; ELSE
		    (car propose-types))))
	(when print-apply
	  (format t " :apply ~(~S~)"
		  (if (cdr apply-types)
		    apply-types
		    ;; ELSE
		    (car apply-types))))
	(princ #\)))
      ;; ELSE
      (format t "~%     ~S" op-name))

    (print-space-model-tcs proposal-component print-tc-names)

    (print-space-model-tcs application-component print-tc-names)))

(defun print-space-model-tcs (component-info print-tc-names)
  (when component-info
    (setf (component-info-tc-alist component-info)
	  (sort (component-info-tc-alist component-info)
		#'string-lessp
		:key #'(lambda (x)
			 (symbol-name (car x)))))
    (dolist (tc-type-entry (component-info-tc-alist component-info))
      (let ((num-tcs (length (cdr tc-type-entry))))
	(format t "~%       ; ~(~S~) (~S TC~:[s~;~])"
		(car tc-type-entry)
		num-tcs
		(eql num-tcs 1)))
      (when print-tc-names
	(dolist (tc-name (reverse (cdr tc-type-entry)))
	  (format t "~%       ;   ~S" tc-name))))))

(defun get-space-model (type space-name)
  (let ((space-info (gethash space-name *space-info-table*)))
    (when space-info
      (let ((model
	     (case type
	       (inferred
		(space-info-inferred-space-model space-info))
	       (declared
		(space-info-declared-space-model space-info))
	       (t
		(error "INTERNAL TAQL ERROR: Case selector fell through: ~S"
		       type)))))
	(when (space-model-home-segment-names model)
	  model)))))

(defun merge-inferred-and-declared (space-name)
  (let ((inferred (get-space-model 'inferred space-name))
	(declared (get-space-model 'declared space-name)))

    (cond ((and inferred declared)
	   (let ((result-model (copy-space-model declared)))
	     (setf (space-model-home-segment-names result-model)
		   (union
		    (space-model-home-segment-names inferred)
		    (space-model-home-segment-names declared)))
	     (setf (space-model-component-info-table result-model)
		   (merge-component-tables
		    (space-model-component-info-table inferred)
		    (space-model-component-info-table declared)))
	     (setf (space-model-propose-operator-table result-model)
		   (merge-component-tables
		    (space-model-propose-operator-table inferred)
		    (space-model-propose-operator-table declared)))
	     (setf (space-model-apply-operator-table result-model)
		   (merge-component-tables
		    (space-model-apply-operator-table inferred)
		    (space-model-apply-operator-table declared)))
	     result-model))

	  (inferred
	   inferred)

	  (t
	   declared))))

;; The result may share structure with one of the input tables, but
;; neither of the original tables will be destructively modified.
;;
(defun merge-component-tables (table1 table2)
  (let ((result (make-hash-table :size 10)))

    (maphash #'(lambda (component-name component-info)
		 (let ((table2-entry (gethash component-name table2)))
		   (setf (gethash component-name result)
			 (if table2-entry
			   (merge-component-info component-info table2-entry)
			   ;; ELSE
			   component-info))))
	     table1)
    (maphash #'(lambda (component-name component-info)
		 (let ((table1-entry (gethash component-name table1)))
		   (if (not table1-entry)
		     (setf (gethash component-name result)
			   component-info))))
	     table2)

    result))

;; The result may share structure with one of the input arguments, but
;; neither of the original arguments will be destructively modified.
;; We assume that both argument component-info structures have the same
;; value in their name field.  We also assume that at most one of the
;; arguments has a non-nil TC-alist field, which will be the case when
;; we are merging inferred and declared models for the same space.
;;
(defun merge-component-info (info1 info2)
  (make-component-info
   :name (component-info-name info1)
   :implementation-types (union (component-info-implementation-types info1)
				(component-info-implementation-types info2))
   :TC-alist (or (component-info-TC-alist info1)
		 (component-info-TC-alist info2))))

(defun add-space (space-name)
  (or (gethash space-name *space-info-table*)
      (let ((new-info
	     (setf (gethash space-name *space-info-table*)
		   (make-space-info :name space-name))))
	(setq *spaces-need-checking* t)
	(setf (space-model-name (space-info-inferred-space-model new-info))
	      space-name)
	(setf (space-model-name (space-info-declared-space-model new-info))
	      space-name)
	new-info)))

(defun add-component (component-name component-table)
  (or (gethash component-name component-table)
      (progn
	(setq *spaces-need-checking* t)
	(setf (gethash component-name component-table)
	      (make-component-info :name component-name)))))

(defun tc-type-to-component-name (tc-type)
  (case tc-type
    (propose-task-operator	'operator-proposal)
    (propose-task-state		'initial-state-proposal)
    (propose-space		'problem-space-proposal)
    (propose-initial-state	'initial-state-proposal)
    (propose-operator		'operator-proposal)
    (compare			'selection)
    (prefer			'selection)
    (goal-test-group		'goal-testing)
    (apply-operator		'operator-application)
    (augment			'entailment)
    (result-superstate		'result-returning)
    (evaluation-properties	'evaluation)
    (evaluate-object		'evaluation)
    (propose-superobjects	'result-returning)
    (operator-control		'selection)
    (otherwise
     (error "INTERNAL TAQL COMPILER ERROR.  Case selector ~S fell through."
	    tc-type))))

(defmacro print-space-graph (&body body)
  `(print-space-graph-aux ',body))

(defun print-space-graph-aux (body)
  (cond ((cdr body)
	 (taql-warn2 "Usage:  (print-space-graph [TOP-SPACE-NAME])"))
	((and body
	      (not (symbolp (car body))))
	 (taql-warn2
	  "Print-space-graph: argument must be a symbolic problem space ~
           name."))
	(t
	 (setq *printed-space-keys* nil)
	 (print-space-key-links
	  (cons 'space
		(if body
		  (car body)
		  ;; ELSE
		  'top-space))
	  0)
	 (do ((key (find-unprinted-space-key)
		   (find-unprinted-space-key)))
	     ((null key))
	   (terpri)
	   (print-space-key-links key 0))
	 (terpri) (terpri)
	 (check-unproposed-spaces)))
  t)

(defun find-unprinted-space-key ()
  (maphash #'(lambda (key func-list)
	       (declare (ignore func-list))
	       (when (not (member key *printed-space-keys* :test #'equal))
		 (return-from find-unprinted-space-key key)))
	   *space-function-table*)
  nil)

(defun print-space-key-links (key indent-level)
  (terpri)
  (print-spaces indent-level)
  (if (eql (car key) 'space)
    (princ "P: ")
    ;; ELSE
    (princ "O: "))
  (princ (cdr key))

  (if (member key *printed-space-keys* :test #'equal)
    (when (gethash key *space-function-table*)
      (princ " [...]"))
    ;; ELSE
    (progn
      (push key *printed-space-keys*)
      (case (car key)
	(space
	 (print-links-for-space key indent-level))
	(operator
	 (print-links-for-operator key indent-level))))))

(defun print-links-for-space (key indent-level)
  (let* ((func-graph (graphify-space-functions
		      (gethash key *space-function-table*)))
	 (propose-states (function-graph-child '(propose state) func-graph))
	 (select-states (function-graph-child '(select state) func-graph))
	 (propose-operators (function-graph-child '(propose operator)
						  func-graph))
	 (select-operators (function-graph-child '(select operator)
						 func-graph))
	 (all-operators
	  (sort (space-operators (cdr key))
		#'string-lessp
		:key #'symbol-name)))

    (print-space-function-graph propose-states indent-level)
    (print-space-function-graph select-states indent-level)

    (when (or propose-operators select-operators all-operators)
      (terpri)
      (print-spaces indent-level)
      (princ "S:"))

    (print-space-function-graph propose-operators indent-level)
    (print-space-function-graph select-operators indent-level)

    (dolist (op-name all-operators)
      (print-space-key-links (cons 'operator op-name) indent-level))))

(defun print-links-for-operator (key indent-level)
  (print-space-function-graph
   (graphify-space-functions (gethash key *space-function-table*))
   indent-level))

;; If func-graph has a child arc labeled child-arc-label, return a
;; function graph with nil node label whose single child is that arc.
;; Otherwise return nil.
;;
;; So this doesn't really return the child subgraph itself, but rather
;; a graph constructed from it.
;;
(defun function-graph-child (child-arc-label func-graph)
  (let ((child (assoc child-arc-label (cdr func-graph) :test #'equal)))
    (when child
      (list nil child))))

(defun graphify-space-functions (func-list)
  (graphify-space-functions-aux
   (mapcar #'(lambda (func-struct)
	       (cdr
		(reverse
		 (cons (space-function-space-name func-struct)
		       (space-function-function func-struct)))))
	   func-list)))

;; Each element in func-data-lists is formed by reversing a space function
;; list after removing the last elements, and placing the name of the
;; space that performs the function at the end of the resulting list.
;; For example, the func-data-list for a space named XX-SPACE that has
;; the function list
;;
;;    (propose space apply operator (operator . XX-OP))
;;
;; would be
;;
;;    (operator apply space propose XX-SPACE)
;;
;; An element of the func-data-lists can also have the form (SPACE-NAME),
;; in which case SPACE-NAME becomes on of the node labels of the returned
;; graph.
;;
;; This is a convenient form for building the graphified form of a set of
;; space functions.
;;
(defun graphify-space-functions-aux (func-data-lists)
  (when func-data-lists
    (multiple-value-bind (node-labels func-lists)
	(partition-func-data-lists func-data-lists)
      (let ((subgraph-arcs
	     (sort-function-graph-arcs
	      (form-space-function-subgraphs func-lists))))
	(cons node-labels subgraph-arcs)))))

(defun partition-func-data-lists (func-data-lists)
  (let ((labels nil)
	(func-lists nil))
    (dolist (func-data-list func-data-lists)
      (if (cdr func-data-list)
	(push func-data-list func-lists)
	;; ELSE
	(push (car func-data-list) labels)))
    (values (remove-duplicates labels) func-lists)))

(defun form-space-function-subgraphs (func-data-lists)
  (let ((result-alist nil))
    (dolist (func-list func-data-lists)
      (setq result-alist
	    (add-alist-value (list (cadr func-list) (car func-list))
			     (cddr func-list)
			     result-alist
			     :test #'equal)))
    (dolist (entry result-alist)
      (setf (cdr entry)
	    (list (graphify-space-functions-aux (cdr entry)))))
    result-alist))

(defun sort-function-graph-arcs (arcs)
  (let ((arc-label-order
	 '((propose problem-space)
	   (select problem-space)
	   (propose state)
	   (select state)
	   (propose operator)
	   (select operator)
	   (apply operator))))
    (sort arcs
	  #'(lambda (label-1 label-2)
	      (member label-2
		      (member label-1 arc-label-order :test #'equal)
		      :test #'equal))
	  :key #'car)))

(defun print-space-function-graph (func-graph indent-level)
  (let ((node-labels (car func-graph))
	(children (cdr func-graph)))

    (dolist (child children)
      (let ((arc-label (car child))
	    (child-subgraph (cadr child)))
	(print-arc-label-as-impasse arc-label indent-level)
	(print-space-function-graph child-subgraph (+ indent-level 3))))
    (dolist (space-name node-labels)
      (print-space-key-links (cons 'space space-name) indent-level))))

(defun print-arc-label-as-impasse (arc-label indent-level)
  (let* ((impasse
	  (case (car arc-label)
	    ((apply propose) 'no-change)
	    (select 'tie)))
	 (attribute
	  (cond ((eql impasse 'tie)
		 (cadr arc-label))
		((eql (car arc-label) 'apply)
		 'operator)
		(t ; (car arc-label) is propose
		 (case (cadr arc-label)
		   (operator 'state)
		   (state 'problem-space)
		   (problem-space 'goal))))))
    (terpri)
    (print-spaces indent-level)
    (format t "==>G: (~(~S ~S~))" attribute impasse)))

;; See the release notes for the syntax and semantics of excise-data-model.
;;
(defmacro excise-space-model (&body body)
  `(excise-space-model-aux ',body))

(defun excise-space-model-aux (body)
  (excise-space-models-in-segments
   (process-model-spec-args-common body 'space nil))

  t)

;; Excise space models specified by a list of parsed space model spec
;; triples.  See the parse-model-spec-triple function for a description of
;; the form of a parsed triple.
;;
;; Excise all specified space models whose home segment list includes the
;; specified segment.  Sticky declared models are not removed.
;; Stickiness is not meaningful for inferred models.
;;
;; This just removes the space models themselves, and not any TCs that they
;; contain.
;;
(defun excise-space-models-in-segments (parsed-model-specs)
  (dolist (parsed-spec parsed-model-specs)
    (excise-space-models-in-segments-aux parsed-spec)))

(defun excise-space-models-in-segments-aux (parsed-model-spec)
  (dolist (spec (cdr parsed-model-spec))
    (setq *spaces-need-checking* t)
    (if (eql (car spec) 'declared)
      (maphash #'(lambda (space-name space-info)
		   (let* ((model
			   (space-info-declared-space-model space-info))
			  (home-segments
			   (space-model-home-segment-names model)))
		     (when (and (or (eql (car parsed-model-spec) :all)
				    (member space-name
					    (car parsed-model-spec)))
				(member (cadr spec) home-segments)
				(not (space-model-sticky model)))
		       (if (space-model-home-segment-names
			    (space-info-inferred-space-model space-info))
			 ;; Just remove the declared model
			 (setf (space-info-declared-space-model space-info)
			       (make-space-model
				:name space-name
				:home-segment-names nil
				:sticky nil))
			 ;; ELSE remove the entire entry
			 (progn
			   (remove-space-function-table-entries space-name)
			   (remhash space-name *space-info-table*))))))
	       *space-info-table*)
      ;; ELSE inferred
      (maphash #'(lambda (space-name space-info)
		   (let* ((model
			   (space-info-inferred-space-model space-info))
			  (home-segments
			   (space-model-home-segment-names model)))
		     (when (and (or (eql (car parsed-model-spec) :all)
				    (member space-name
					    (car parsed-model-spec)))
				(member (cadr spec) home-segments))
		       (remove-space-function-table-entries space-name)
		       (if (space-model-home-segment-names
			    (space-info-declared-space-model space-info))
			 ;; Just remove the inferred model
			 (setf (space-info-inferred-space-model space-info)
			       (make-space-model
				:name space-name
				:home-segment-names nil))
			 ;; ELSE remove the entire entry
			 (remhash space-name *space-info-table*)))))
	   *space-info-table*))))

;; This function must be called when a segment is excised.
;;
;; Remove segment-name from all space model home-segment-names lists
;; for models that are either inferred or [declared and not sticky].
;; When a models's home-segment-names list becomes empty,
;; invalidate/reinitialize the model.  If its companion model is also
;; invalid, remove the whole entry for that space from the space-info
;; table.
;;
;; This may destructively modify the home-segment-names lists.
;;
(defun remove-segment-from-space-models (segment-name)
  (maphash #'(lambda (space-name space-info)
	       (let ((decl-model (space-info-declared-space-model space-info))
		     (inf-model (space-info-inferred-space-model space-info)))
		 (when (and (space-model-home-segment-names decl-model)
			    (not (space-model-sticky decl-model)))
		   (setq *spaces-need-checking* t)
		   (setf (space-model-home-segment-names decl-model)
			 (delete segment-name
				 (space-model-home-segment-names decl-model)))
		   (when (null (space-model-home-segment-names decl-model))
		     ;; Invalidate/reinitialize the declared model
		     (setf (space-info-declared-space-model space-info)
			   (make-space-model
			    :name space-name
			    :home-segment-names nil
			    :sticky nil))))
		 (when (space-model-home-segment-names inf-model)
		   (setq *spaces-need-checking* t)
		   (setf (space-model-home-segment-names inf-model)
			 (delete segment-name
				 (space-model-home-segment-names inf-model)))
		   (when (null (space-model-home-segment-names inf-model))
		     ;; Invalidate/reinitialize the inferred model
		     (remove-space-function-table-entries space-name)
		     (setf (space-info-declared-space-model space-info)
			   (make-space-model
			    :name space-name
			    :home-segment-names nil))))
		 (when (and (null (space-model-home-segment-names decl-model))
			    (null (space-model-home-segment-names inf-model)))
		   (setq *spaces-need-checking* t)
		   (remhash space-name *space-info-table*))))
	   *space-info-table*))

(defun remove-space-function-table-entries (space-name)
  (setq *spaces-need-checking* t)
  (remhash (cons 'space space-name) *space-function-table*)
  (maphash #'(lambda (key func-list)
	       (setf (gethash key *space-function-table*)
		     (delete-if #'(lambda (func-space-name)
				    (eql space-name func-space-name))
				func-list
				:key #'space-function-space-name))
	       (when (null (gethash key *space-function-table*))
		 (remhash key *space-function-table*)))
	   *space-function-table*))

(defun segment-has-space-models (segment-name)
  (maphash #'(lambda (space-name space-info)
	       (declare (ignore space-name))
	       (when (or (member
			  segment-name
			  (space-model-home-segment-names
			   (space-info-declared-space-model space-info)))
			 (member
			  segment-name
			  (space-model-home-segment-names
			   (space-info-inferred-space-model space-info))))
		 (return-from segment-has-space-models t)))
	   *space-info-table*)
  ;; If we get here, we didn't find one.
  nil)

;; Model-spec is a list of the form ({inferred | declared} SEGMENT-NAME).
;;
;; This returns a list of all space names (no duplicates) that have the
;; specified segment name as a home segment in their space-model of the
;; specified type.
;;
(defun expand-model-spec-to-space-names (model-spec)
  (let ((result nil))
    (maphash #'(lambda (space-name space-info)
		 (declare (ignore space-info))
		 (let ((model (get-space-model (car model-spec) space-name)))
		   (when (and model
			      (member (cadr model-spec)
				      (space-model-home-segment-names model)))
		     (push space-name result))))
	     *space-info-table*)
    result))

;; EXPORTED FUNCTION:
;;
;; We also remove the tc-name from the space-function-table.
;;
(defun remove-tc-from-space-models (tc-name)
  (when (not (default-tc-p tc-name))
    (remove-tc-from-space-function-table tc-name)
    (maphash #'(lambda (space-name space-info)
		 (declare (ignore space-name))
		 (let ((model (space-info-inferred-space-model space-info)))
		   (when (space-model-home-segment-names model)
		     (setq *spaces-need-checking* t)
		     (remove-tc-from-space-models-aux tc-name model))))
	     *space-info-table*)))

(defun remove-tc-from-space-function-table (tc-name)
  (maphash #'(lambda (key func-list)
	       (let ((new-func-list
		      (remove-tc-from-space-function-list tc-name func-list)))
		 (setq *spaces-need-checking* t)
		 (if new-func-list
		   (setf (gethash key *space-function-table*)
			 new-func-list)
		   ;; ELSE
		   (remhash key *space-function-table*))))
	   *space-function-table*))

(defun remove-tc-from-space-function-list (tc-name func-list)
  (setq *spaces-need-checking* t)
  (mapc #'(lambda (func-struct)
	    (setf (space-function-propose-tc-names func-struct)
		  (delete tc-name
			  (space-function-propose-tc-names func-struct))))
	func-list)
  (delete-if #'null func-list :key #'space-function-propose-tc-names))

;; We assume tc-name isn't a default TC.
;;
(defun remove-tc-from-space-models-aux (tc-name space-model)
  (remove-tc-from-space-component-table
   tc-name (space-model-component-info-table space-model))
  (remove-tc-from-space-component-table
   tc-name (space-model-propose-operator-table space-model))
  (remove-tc-from-space-component-table
   tc-name (space-model-apply-operator-table space-model)))

;; We assume tc-name isn't a default TC.
;;
(defun remove-tc-from-space-component-table (tc-name component-table)
  (maphash #'(lambda (component-name component-info)
	       (declare (ignore component-name))
	       (remove-tc-from-component-info tc-name component-info))
	   component-table))

;; We assume tc-name isn't a default TC.  This may destructively modify
;; the component-info and its tc-alist.  Updating the implementation-types
;; for default-operator-proposal and default-operator-application components
;; doesn't seem worth the effort, so I don't bother.
;;
(defun remove-tc-from-component-info (tc-name component-info)
  (setq *spaces-need-checking* t)
  (dolist (alist-entry (component-info-tc-alist component-info))
    (setf (cdr alist-entry)
	  (delete tc-name (cdr alist-entry))))
  (setf (component-info-tc-alist component-info)
	(delete-if-not #'cdr (component-info-tc-alist component-info)))
  (let ((tc-alist (component-info-tc-alist component-info)))
    (when (not (member (component-info-name component-info)
		       '(default-operator-proposal
			  default-operator-application)))
      (when (or (null tc-alist)
		(and (eql (caar tc-alist) 'propose-space)
		     (null (cdr tc-alist))))
	(setf (component-info-implementation-types component-info)
	      (delete 'direct
		      (component-info-implementation-types component-info))))
      (when (not (assoc 'propose-space tc-alist))
	(setf (component-info-implementation-types component-info)
	      (delete 'subspace
		      (component-info-implementation-types component-info))))))
  )

;; EXPORTED FUNCTION:
;;
(defun space-model-excise-task-extras ()
  (eval `(all-spaces-declared ,*sticky-all-spaces-declared*)))

(defmacro all-spaces-declared (&body body)
  `(all-spaces-declared-aux ',body))

(defun all-spaces-declared-aux (body)
  (if (or (null body)
	  (cddr body)
	  (not (member (car body) '(:yes :no))))
    (taql-warn2 "Usage: (all-spaces-declared {:yes | :no})")
    ;; ELSE
    (progn
      (when *taql-default-mode*
	(setq *sticky-all-spaces-declared* (car body)))
      (let ((new-value (eql (car body) :yes)))
	(when (not (eql *all-spaces-declared* new-value))
	  (setq *spaces-need-checking* t)
	  (setq *all-spaces-declared* new-value)))))
  t)

(defmacro check-spaces (&body body)
  `(check-spaces-aux ',body))

(defun check-spaces-aux (body)
  (cond
   (body
    (taql-warn2 "Check-spaces does not take any arguments."))
   (t
    (check-undeclared-spaces)
    (check-unproposed-spaces)
    (check-spaces-declared-inferred)
    (terpri)
    (setq *spaces-need-checking* nil)))

  t)

;; When *all-spaces-declared*, warn about all spaces that have a valid
;; inferred space model, but no valid declared space model.
;;
(defun check-undeclared-spaces ()
  (when *all-spaces-declared*
    (maphash
     #'(lambda (space-name space-info)
	 (when (and (not (eql space-name '*unknown*))
		    (space-model-home-segment-names
		     (space-info-inferred-space-model space-info))
		    (not (space-model-home-segment-names
			  (space-info-declared-space-model space-info))))
	   (taql-warn-soft2 "Problem space ~S was used but not declared."
			    space-name)))
     *space-info-table*)))

;; Warn about all known spaces that haven't been proposed with propose-space,
;; except for top-space.
;;
(defun check-unproposed-spaces ()
  (let ((unproposed-spaces
	 (set-difference (all-known-space-names)
			 (cons 'top-space (all-known-proposed-spaces)))))
    (when unproposed-spaces
      (taql-warn-soft2
       "~:[This space was~;These spaces were~] used, ~
        but ~:[was~;were~] never proposed by propose-space:"
       (cdr unproposed-spaces)
       (cdr unproposed-spaces)
       (cdr unproposed-spaces))
      (princ "    ")
      (print-list-with-commas unproposed-spaces))))

(defun check-spaces-declared-inferred ()
  (maphash #'(lambda (space-name space-info)
	       (let ((decl-model (space-info-declared-space-model space-info))
		     (inf-model (space-info-inferred-space-model space-info)))
		 (when (space-model-home-segment-names decl-model)
		   (check-spaces-declared-inferred-models
		    space-name decl-model inf-model))))
	   *space-info-table*))

(defun check-spaces-declared-inferred-models (space-name decl-model inf-model)
  (check-spaces-check-non-operator-components
   space-name
   (space-model-component-info-table decl-model)
   (space-model-component-info-table inf-model))
  (check-spaces-check-operators space-name decl-model inf-model))

;; This assumes that every component name appears in a declared component
;; info table, which currently it always does.
;;
(defun check-spaces-check-non-operator-components (space-name decl-table
							      inf-table)
  (maphash
   #'(lambda (component-name decl-info)
       (when (not (member component-name
			  '(default-operator-proposal
			     default-operator-application)))
	 (let ((inf-info (gethash component-name inf-table)))
	   (check-spaces-check-implementation-types
	    space-name
	    component-name
	    (component-info-implementation-types decl-info)
	    (if inf-info
	      (component-info-implementation-types inf-info)
	      ;; ELSE
	      nil)))))
   decl-table))

(defun check-spaces-check-implementation-types (space-name component-name
						decl-impl-types inf-impl-types
						&optional (opname nil
								  op-supplied))

  (let ((extra-decl (set-difference decl-impl-types inf-impl-types))
	(extra-inf (set-difference inf-impl-types decl-impl-types)))

    (dolist (extra-type extra-decl)
      (taql-warn-soft2
       "~:[~*~;Operator ~S ~]~
        :~(~S~) for space ~S was declared to be ~S, but no TCs perform this ~
        operation ~:[directly~;in a subspace~]."
       op-supplied opname
       component-name space-name (cond ((null decl-impl-types)
					'none)
				       ((cdr decl-impl-types)
					decl-impl-types)
				       (t
					(car decl-impl-types)))
       (eql extra-type 'subspace)))

    (dolist (extra-type extra-inf)
      (taql-warn-soft2
       "~:[~*~;Operator ~S ~]~
        :~(~S~) for space ~S was declared to be ~S, but there are TCs ~
        that perform this operation ~:[directly~;in a subspace~]."
       op-supplied opname
       component-name space-name (cond ((null decl-impl-types)
					'none)
				       ((cdr decl-impl-types)
					decl-impl-types)
				       (t
					(car decl-impl-types)))
       (eql extra-type 'subspace)))
    ))

(defun check-spaces-check-operators (space-name decl-model inf-model)
  (let ((decl-op-names (operators-in-space-model decl-model))
	(inf-op-names (operators-in-space-model inf-model)))

    (let ((decl-subspace-op-proposal
	   (member 'subspace
		   (get-component-implementation-types
		    'default-operator-proposal decl-model)))
	  (inf-subspace-op-proposal
	   (member 'subspace
		   (get-component-implementation-types
		    'default-operator-proposal inf-model))))

      (cond ((and decl-subspace-op-proposal
		  (not inf-subspace-op-proposal))
	     (taql-warn-soft2
	      "The declared :default-operator-proposal for space ~S indicates ~
               that its operators are proposed in a subspace, but there are ~
               no propose-space TCs to do that."
	      space-name))
	    ((and inf-subspace-op-proposal
		  (not decl-subspace-op-proposal))
	     (taql-warn-soft2
	      "The declared :default-operator-proposal for space ~S does not ~
               contain 'subspace', but there are propose-space TCs that ~
               indicate its operators are proposed in a subspace."
	      space-name))))

    (cond
     ((and (null decl-op-names)
	   inf-op-names)
      (taql-warn-soft2
       "No operators were declared for space ~S, but the following ~
        operator~:[ was~;s were~] used in the space:"
       space-name (cdr inf-op-names))
      (princ "    ")
      (print-list-with-commas inf-op-names))
     
     (t
      (setq inf-op-names (remove '*unknown* inf-op-names))
      (let ((decl-not-inf (set-difference decl-op-names inf-op-names))
	    (inf-not-decl (set-difference inf-op-names decl-op-names))
	    (decl-and-inf (intersection inf-op-names decl-op-names)))
	(when decl-not-inf
	  (taql-warn-soft2
	   "The following operator~:[ was~;s were~] declared for space ~S, ~
            but did not appear there:"
	   (cdr decl-not-inf) space-name)
	  (princ "    ")
	  (print-list-with-commas decl-not-inf))
	(when inf-not-decl
	  (taql-warn-soft2
	   "The following operator~:[~;s~] appeared in space ~S, ~
            but ~:[was~;were~] not declared for it:"
	   (cdr inf-not-decl) space-name (cdr inf-not-decl))
	  (princ "    ")
	  (print-list-with-commas inf-not-decl))
	(check-spaces-check-shared-operators
	 decl-and-inf space-name decl-model inf-model))))))

(defun check-spaces-check-shared-operators (decl-and-inf-op-names space-name
					    decl-model inf-model)
  (dolist (op-name decl-and-inf-op-names)
    (dolist (variant '(propose apply))
      (check-spaces-check-implementation-types
       space-name
       variant
       (get-operator-implementation-types op-name variant decl-model)
       (get-operator-implementation-types op-name variant inf-model)
       op-name))))

;; EXPORTED FUNCTION:
;;
;; Calling this function will call check-spaces if the space model has
;; changed since check-spaces was called last..
;; Otherwise it doesn't do anything.
;;
(defun check-spaces-if-necessary ()
  (when *spaces-need-checking*
    (check-spaces)))

(defun get-operator-implementation-types (op-name propose-or-apply space-model)
  (let* ((table (if (eql propose-or-apply 'propose)
		  (space-model-propose-operator-table space-model)
		  ;; ELSE
		  (space-model-apply-operator-table space-model)))
	 (info (gethash op-name table)))
    (and info
	 (component-info-implementation-types info))))

(defun get-component-implementation-types (component-name space-model)
  (let ((info (gethash component-name
		       (space-model-component-info-table space-model))))
    (and info
	 (component-info-implementation-types info))))

;; This returns a list of all spaces in the space info table, excluding
;; *unknown*.
;;
(defun all-known-space-names ()
  (let ((result nil))
    (maphash #'(lambda (space-name space-info)
		 (declare (ignore space-info))
		 (when (not (eql space-name '*unknown*))
		   (push space-name result)))
	     *space-info-table*)
    result))

(defun all-known-proposed-spaces ()
  (let ((result nil))
    (maphash #'(lambda (key func-list)
		 (declare (ignore key))
		 (dolist (func func-list)
		   (pushnew (space-function-space-name func) result)))
	     *space-function-table*)
    result))

;; Return a list of the names of all of the operators that are either
;; proposed or applied in the named space, combining information from
;; both the declared and inferred space models.
;;
(defun space-operators (space-name)
  (let ((info (gethash space-name *space-info-table*)))
    (when info
      (let ((declared-model (space-info-declared-space-model info))
	    (inferred-model (space-info-inferred-space-model info)))
	(union (operators-in-space-model declared-model)
	       (operators-in-space-model inferred-model))))))

;; If space-model is a valid model, return a list of the names of all of the
;; operators that are either proposed or applied in that space.
;;
(defun operators-in-space-model (space-model)
  (when (space-model-home-segment-names space-model)
    (union (hash-table-key-list
	    (space-model-propose-operator-table space-model))
	   (hash-table-key-list
	    (space-model-apply-operator-table space-model)))))

;; Destructively add value to the set of values for key on alist, or add
;; an entry for key if there isn't one already.  Returns the modified alist.
;; The values for a key are treated as a SET, not a list.  So the same value
;; won't be added twice.
;;
(defun add-alist-value (key value alist &key (test #'eql))
  (setq *spaces-need-checking* t)
  (let ((entry (assoc key alist :test test)))
    (if entry
      (progn
	(pushnew value (cdr entry) :test test)
	alist)
      ;; ELSE
      (cons (list key value) alist))))

;; This is list add-alist-value, but adds each of the values listed in values.
;;
(defun add-alist-values (key values alist &key (test #'eql))
  (setq *spaces-need-checking* t)
  (let ((entry (assoc key alist :test test)))
    (if entry
      (progn
	(setf (cdr entry)
	      (union (cdr entry) (remove-duplicates values :test test)
		     :test test))
	alist)
      ;; ELSE
      (cons (cons key (remove-duplicates values :test test)) alist))))

(defun set-equal (x y)
  (and (subsetp x y)
       (subsetp y x)))

(defun hash-table-key-list (table)
  (let ((result nil))
    (maphash #'(lambda (key val)
		 (declare (ignore val))
		 (push key result))
	     table)
    result))

(defun print-spaces (n)
  (dotimes (i n)
    (princ #\Space)))

(defun last-n-elements (n list)
  (let ((length (length list)))
    (if (>= n length)
      list
      ;; ELSE
      (nthcdr (- length n) list))))

;; Prints nil as ""
;; Prints (a) as "a"
;; Prints (a b) as "a and b"
;; Prints (a b c) as "a, b, and c"
;; Prints (a b c d) as "a, b, c, and d"
;; ...
;;
(defun print-list-with-commas (list)
  (format t "~{~#[~;~S~;~S and ~S~:;~@{~#[~;and ~]~S~^, ~}~]~}" list))

;; This is called by init-taql.
;;
(defun init-space-model-stuff ()
  (setq *spaces-need-checking* nil)

  (setq *space-info-table*
	(make-hash-table :size 30))

  ;; We can't copy a space function table using my copy-hash-table function,
  ;; because it has a non-default :test argument.  But currently we never
  ;; have occasion to copy it anyways.
  ;;
  (setq *space-function-table*
	(make-hash-table :size 30 :test #'equal))

  (setq *printed-space-keys* nil)

  (setq *sticky-all-spaces-declared* :no)

  ;; *all-spaces-declared* must have a value before the first time
  ;; all-spaces-declared is called.
  ;;
  (setq *all-spaces-declared* nil)
  (all-spaces-declared :no)

  t)

(eval-when (compile load eval)
  (soarsyntax))
