;;; TAQL Compiler, Operator-Control-Spec Module
;;;
;;; Gregg Yost, Erik Altmann
;;; School of Computer Science
;;; Carnegie Mellon University
;;;
;;; Working file: /afs/cs/user/altmann/soar/taql/operator-control.lisp
;;; Created September 11, 1990
;;;
;;; This file contains the code that implements the operator-control TC.
;;;
;;; IMPORTANT IMPORTANT IMPORTANT PLEASE READ
;;;
;;;   3-15-91 - gry
;;;   It is important for (goal ^taql-control-spec*) conditions to come first
;;;   in a production.  This provides much more constraint in some cases on
;;;   the number of partial production matches (at least, given the way Soar's
;;;   reorder currently works).  This is particularly important for productions
;;;   that make binary preferences, and have to test acceptable preferences
;;;   for two different operators.
;;;
;;; Known bugs/funnies:
;;; 
;;; =======================================================================
;;; Modification history:
;;; =======================================================================
;;;
;;; 9-29-90 - gry - changed operator-program to operator-control
;;;
;;; 9-19-90 - gry -
;;;    - Extended :exit-when to all control-spec types
;;;    - Added exit-loop and next-iteration loop operators
;;;
;;; 9-18-90 - gry -
;;;    - Added the :optional keyword
;;;    - Got rid of loop-exhaustion operator, instead now just make all
;;;      loops implicitly :optional.
;;;
;;; 9-17-90 - gry - Changed meaning of :when in OPERATOR-CONTROL-SPECs to only
;;;    constraint control-spec activation, and not its continued execution.
;;;
;;; 9-13-90 - gry - Created.

;;; The general strategy behind compiling an operator-control is to treat
;;; the control-spec as a regular expression on the sequences of operators that
;;; can be applied, and at run time simulate the NFA (Nondeterministic Finite
;;; Automata) that recognizes the regular expression.  As operators are
;;; applied, TAQL keeps track of the states the NFA could be in given the
;;; sequence of operators that have applied so far.  Operators that are not
;;; legal transitions out of any of the possible current NFA states are
;;; rejected.
;;;
;;; That's the basic idea.  The various control-spec modifiers (such as :ind,
;;; :ra, and :swp) complicate things a bit.  Also, :loop is not quite like
;;; a Kleene star (*) in a regular expression, in that a :loop will not
;;; exit until either it is exhausted, an :exit-when forces an exit, or a
;;; :swp following the loop is activated.

;;; *** BEGIN CODE ***

(eval-when (compile load eval)
  (lispsyntax))

(eval-when (compile load eval)
   (if (find-package "COMMON-LISP-USER")
       (in-package "COMMON-LISP-USER")
       (in-package "USER")))

;;; A list of the states in the control-spec NFA (Nondeterministic Finite
;;; Automata).
;;;
(defvar *nfa-states* nil
  "List of NFA state labels for operator-control")

(defmacro operator-control (&body body)
  `(operator-control-aux ',body))

(defun operator-control-aux (body)
  (reset-taql-warnings)
  (restart-genvar)
  (assign-context-vars)

  ;; Special initialization for operator-control TC.
  ;;
  (init-nfa)
  
  (let* ((*current-taql-name* (new-taql (car body)
					'operator-control))
	 (args (group-arguments (cdr body) '(:space :when :control-spec)
				'construct))
	 (space (assoc :space (cdr args)))
	 (control-spec (assoc :control-spec (cdr args)))
         (control-spec-id (makesym (get-internal-real-time) '*))
	 (*prod-name-prefix* (build-prefix (cadr space)))
	 (activation-conds
	  `(,.(if space
		`((goal ,=goal ^ problem-space ,=problem-space)
		  (problem-space ,=problem-space ^ name ,(cadr space))))
	    ,.(prepare-condition-sets (cdr (assoc :when (cdr
							 args)))))))

    (when (cddr space)
      (taql-warn "At most one :space argument may appear.")
      (setq space nil))

    (when (or (null control-spec) (cddr control-spec))
      (taql-warn "Exactly one :control-spec argument must appear.")
      (setq control-spec '(:control-spec (:seq junk))))
    
    (if (car args)
      (taql-warn "All arguments must be values of keywords."))
    
    (add-current-tc-to-spaces (cdr space) nil)

    (multiple-value-bind (initial-state final-state)
	(compile-operator-control-spec
	 `(:seq ,(cadr control-spec) exit-control-spec)
	 control-spec-id nil nil nil)

      (declare (ignore final-state))

      (generate-nfa-productions control-spec-id)
      (generate-do-optional-productions control-spec-id)

      (let ((spec-obj-id (genvar 'p)))

	(eval
	 `(taql-sp sticky
	   ,(newsym *prod-name-prefix* 'activate-control-spec *num-separator*)
	   ;; Put in this condition just so that the LHS will always
	   ;; have something in it.
	   ,@(when (not activation-conds)
	       `((goal ,=goal ^ problem-space ,=problem-space)))
	   ,@activation-conds
	   -->
	   ;; Create the control-spec-info object, and put the initial NFA
	   ;; states on it.
	   (goal ,=goal ^ taql-control-spec* ,spec-obj-id)
	   (control-spec-info*
	    ,spec-obj-id
	    ^ id ,control-spec-id
	    ,@(multify 'nfa-state (epsilon-closure initial-state))))))))
  t)

;;; ----------------------------------------------------------------------
;;; Miscellaneous utility routines
;;; ----------------------------------------------------------------------

;;; OPERATOR-CONTROL-SPEC classifier: either a complex control-spec,
;;; or just an OP-COND
;;;
(defun operator-spec-class (op-spec)
  (cond ((or (not (listp op-spec))
	     (not (keywordp (car op-spec)))
	     (eql (car op-spec) :any-name))
	 'op-cond)
	(t
	 'op-spec)))

;;; The *loop-names* variable and the push-loop-name, pop-loop-name,
;;; loop-name-p, and innermost-loop-name functions are used in
;;; compiling exit-loop and next-iteration operators.  These functions
;;; are the only ones that should use *loop-names* directly.

(defvar *loop-names* nil
  "Stack of names of lexically visible loops during compilation")

(defun loop-name-p (name)
  (member name *loop-names*))

;;; Returns the name of the innermost loop, if we are currently lexically
;;; within a loop, else return nil.
;;;
(defun innermost-loop-name ()
  (car *loop-names*))

(defun push-loop-name (loop-name)
  (if (loop-name-p loop-name)
    (taql-warn
     "Duplicate :loop-name ~S:  a loop cannot have the same name as a loop it is enclosed in."
     loop-name)
    ;; ELSE
    (push loop-name *loop-names*)))

(defun pop-loop-name ()
  (let ((name (pop *loop-names*)))
    (remprop name 'loop-exit-info)
    (remprop name 'loop-next-iteration-info)
    name))

;;; ----------------------------------------------------------------------
;;; General-purpose NFA (Nondeterministic Finite Automata) routines
;;; ----------------------------------------------------------------------

;;; Label generator.  The generator is used for both arc and state labels
;;; in the NFA.

(defun restart-genlabel ()
  (declare (special *taql-genlabel-counter*))
  (setq *taql-genlabel-counter* 1))

(defun genlabel ()
  (declare (special *taql-genlabel-counter*))
  (intern (makestring '* (incf *taql-genlabel-counter*) '*)))

;;; Initialize the NFA.  Call this at the beginning of operator-control-aux.
;;;
(defun init-nfa ()
  (restart-genlabel)
  (do ((name (pop-loop-name) (pop-loop-name)))
      ((null name)))
  (mapc #'(lambda (label)
	    (remprop label 'nfa-transitions)
	    (remprop label 'optional-submachine)
	    (remprop label 'select-when-proposed))
	*nfa-states*)
  (setq *nfa-states* nil))

;;; Use this to generate new nfa states, rather than calling genlabel directly.
;;;
(defun new-nfa-state ()
  (let ((label (genlabel)))
    (push label *nfa-states*)
    label))

;;; Add a transition to the NFA, from state-label to next-state-label on
;;; arc-label.
;;;
(defun add-transition (state-label arc-label next-state-label)
  (let ((existing-transitions
	 (assoc arc-label (get state-label 'nfa-transitions))))
    (if existing-transitions
      (pushnew next-state-label
	       (cdr existing-transitions))
      ;; ELSE
      (push (list arc-label next-state-label)
	    (get state-label 'nfa-transitions)))))

;;; Return the states in the epsilon closure of the states in state-labels.
;;; State-labels can be either a single label or a list of labels.  The
;;; epsilon closure is the set of states reachable from the argument states
;;; by following only epsilon transitions (a state is always in its own
;;; epsilon closure).
;;;
(defun epsilon-closure (state-labels)
  (when (atom state-labels)
    (setq state-labels (list state-labels)))

  (do ((closure state-labels)
       (stack state-labels))
      ((null stack)
       closure)

    (dolist (next-label (cdr (assoc '*epsilon*
				    (get (pop stack) 'nfa-transitions))))
      (when (not (member next-label closure))
	(push next-label closure)
	(push next-label stack)))))

;;; Return a list of the non-epsilon arc labels that transition out of the
;;; states in state-labels.
;;;
(defun non-epsilon-arc-labels (state-labels)
  (when (atom state-labels)
    (setq state-labels (list state-labels)))

  (let ((arc-labels nil))
    (dolist (state-label state-labels)
      (dolist (transition (get state-label 'nfa-transitions))
	(when (not (eql (car transition) '*epsilon*))
	  (push (car transition) arc-labels))))
    arc-labels))

;;; Print out the NFA.  This is a debugging routine.  For each state,
;;; it prints a line starting with the state label, followed by one
;;; indented line for each arc label that transitions from that state.
;;; The general form of each of these lines is
;;; "<arc-label> --> <next-state-label-1> ... <next-state-label-n>",
;;; (This is an NFA, so the same arc label may lead to several different
;;; next states.)
;;;
(defun print-nfa ()
  (dolist (nfa-state *nfa-states*)
    (format t "~%~S:~:{~%    ~S -->~@{ ~S~}~}"
	    nfa-state (get nfa-state 'nfa-transitions)))
  nil)

;;; ----------------------------------------------------------------------
;;; operator-control compilation routines
;;; ----------------------------------------------------------------------

;;; Generate the productions that update the NFA state as new operators are
;;; selected.  The NFA state is represented by a multi-attribute on the
;;; control-spec-info* object whose values are all the labels of states the NFA
;;; could be in at the moment (given the sequences of operators that has
;;; applied so far).
;;;
;;; The state is not updated on interrupt operators.  Also, an
;;; operator tagged select-when-proposed can only transition out of an
;;; NFA state that corresponds to a :swp segment of the operator
;;; control-spec.  For example, consider the control-spec
;;; (:or (:seq a b) (:seq :swp a c)).  After a is selected, we want it to be
;;; able to select only c, not b, because the :swp has forced it to
;;; choose a particulat path through the NFA.
;;;
(defun generate-nfa-productions (control-spec-id)
  (dolist (nfa-state *nfa-states*)
    (dolist (transition (get nfa-state 'nfa-transitions))
      (let ((arc-label (car transition)))
	(when (not (eql arc-label '*epsilon*))
	  (let* ((next-states (epsilon-closure (cdr transition)))
		 (c-id (genvar 'c))
		 (spec-obj-id (genvar 'p)))
	    (eval
	     `(taql-sp sticky
	       ,(newsym *prod-name-prefix* 'update-nfa-state *num-separator*)
	       (goal ,=goal ^ taql-control-spec* ,spec-obj-id
		     ^ operator ,=operator)
	       (control-spec-info* ,spec-obj-id
				  ^ id ,control-spec-id ^ nfa-state ,nfa-state)
	       (operator ,=operator
			 - ^ interrupt true ^ control-stuff* ,c-id)
	       (control-stuff* ,c-id ^ edit-enabled* true ^ label ,arc-label
			      ,@(if (not
				     (get nfa-state 'select-when-proposed))
				  `(- ^ select-when-proposed true)))
	       -->
	       (control-spec-info* ,spec-obj-id
				  ,@(multify 'new-nfa-state next-states))))))))))

;;; Generate the productions that make doing an :optional subpart better than
;;; not doing it.  To implement this, we make the operators that could be
;;; initial operators of the :optional part better than those that could be
;;; the next operator from the final state of its submachine.  Note that in
;;; order for this to work properly, the final state of a submachine cannot
;;; have any transitions back into itself.  Our construction of the machine
;;; guarantees this, but beware if you, the implementor, ever change how
;;; the NFA is constructed.
;;;
;;; We assume that the epsilon transition from the initial state to the final
;;; state is already in place.
;;; 
(defun generate-do-optional-productions (control-spec-id)
  (dolist (nfa-state *nfa-states*)
    (let ((submachine-final-state (get nfa-state 'optional-submachine)))
      (when submachine-final-state
      
	(let* ((labels-from-initial
		(non-epsilon-arc-labels (epsilon-closure nfa-state)))
	       (labels-from-final
		(non-epsilon-arc-labels
		 (epsilon-closure submachine-final-state)))
	       (initial-label-test
		(if (cdr labels-from-initial)
		  `(<< ,@labels-from-initial >>)
		  ;; ELSE
		  labels-from-initial))
	       (final-label-test
		(if (cdr labels-from-final)
		  `(<< ,@labels-from-final >>)
		  ;; ELSE
		  labels-from-final))
	       (c1-id (genvar 'c))
	       (c2-id (genvar 'c))
	       (op1-id (genvar 'o))
	       (op2-id (genvar 'o))
	       (spec-obj-id (genvar 'p)))
	  
	  (when (and labels-from-initial labels-from-final)
	    (eval
	     `(taql-sp unknown
		,(newsym *prod-name-prefix* 'do-optional-part-is-better
			 *num-separator*)
		(goal ,=goal
		      ^ taql-control-spec* ,spec-obj-id
		      ^ operator ,op1-id + { <> ,op1-id ,op2-id } +)
		(control-spec-info* ,spec-obj-id ^ id ,control-spec-id)
		(operator ,op1-id ^ control-stuff* ,c1-id)
		(control-stuff* ,c1-id
			       ^ label ,@initial-label-test
			       - ^ label ,@final-label-test)
		(operator ,op2-id ^ control-stuff* ,c2-id)
		(control-stuff* ,c2-id
			       ^ label ,@final-label-test
			       - ^ select-when-proposed true)
		-->
		(goal ,=goal ^ operator ,op1-id > ,op2-id)))))))))

;;; Compile an operator control-spec, building the corresponding NFA.
;;; There are two main cases: (1) the control-spec is a simple
;;; OP-COND, and (2) the control-spec has the form
;;; (MOD+ OPERATOR-CONTROL-SPEC+).  For OP-CONDSs, we generate productions
;;; that assign labels to proposed operators that match the OP-COND,
;;; and possibly additional productions for any modifiers (such as
;;; :oot (one-of-type)) that are in effect.  For control-specs of the
;;; form (MOD+ OPERATOR-CONTROL-SPEC+), we recursively compile the
;;; sub-control-specs, and construct the NFA for the whole control-spec
;;; from the NFAs for its sub-control-specs.  We use a minor variant of
;;; the standard regular-expression-to-NFA construction.
;;;
;;; Return two values, the labels of the initial and final NFA states for
;;; the control-spec.
;;;
(defun compile-operator-control-spec (control-spec control-spec-id
						   common-conds reject-after
						   one-of-type)
  (let* ((operator-spec-class (operator-spec-class control-spec)))

    (case operator-spec-class
      (op-cond
       (let* ((initial-state (new-nfa-state))
	      (final-state (new-nfa-state))
	      (op-cond-label (genlabel))
	      (prep-op (prepare-operator control-spec =operator))
	      (special-loop-op nil)
	      (c-id (genvar 'c))
	      (spec-obj-id (genvar 'p)))

	 (multiple-value-setq (prep-op special-loop-op)
	   (intercept-special-loop-operators
	    prep-op =operator initial-state op-cond-label control-spec))

	 (when (not special-loop-op)
	   (add-transition initial-state op-cond-label final-state))

	 (when (and special-loop-op
		    (or reject-after one-of-type))
	   (taql-warn ":Ra and :oot cannot be used with ~S" (caar prep-op))
	   (setq reject-after nil)
	   (setq one-of-type nil))

	 ;; We want the production that adds the label to the operator
	 ;; to be retractable, so that if the conditions become
	 ;; unsatisfied the label will retract, and we won't incorrectly
	 ;; transition on it. 
	 ;; A label will only be assigned when an operator is proposed in
	 ;; a NFA state where it is a transition.  So there is no need for
	 ;; a separate flag to indicate that an operator is a candidate,
	 ;; we can just test for the existence of a label.
	 ;;
	 (eval
	  `(taql-sp not-sticky
	       ,(newsym *prod-name-prefix* 'assign-label-to-candidate
			*num-separator*)
	     (goal ,=goal
		   ^ taql-control-spec* ,spec-obj-id
		   ^ operator ,=operator +)
	     (control-spec-info* ,spec-obj-id
				^ id ,control-spec-id
				^ nfa-state ,initial-state
				- ^ disable-label ,op-cond-label)
	     (operator ,=operator ^ control-stuff* ,c-id)
	     (control-stuff* ,c-id - ^ select-once-only* has-been)
	     ,@(cdar prep-op)
	     ,@(cdr prep-op)
	     ,@common-conds
	     -->
	     (control-stuff* ,c-id
			    ^ label ,op-cond-label + &
			    ,@(if reject-after
				`(^ select-once-only* not-yet)))))

	 (when one-of-type
	   (eval
	    `(taql-sp sticky
		 ,(newsym *prod-name-prefix* 'one-of-type*create-reject-label
			  *num-separator*)
	       (goal ,=goal
		     ^ taql-control-spec* ,spec-obj-id
		     ^ operator ,=operator)
	       (control-spec-info* ,spec-obj-id
				  ^ id ,control-spec-id
				  ^ nfa-state ,initial-state)
	       (operator ,=operator ^ control-stuff* ,c-id)
	       (control-stuff* ,c-id ^ label ,op-cond-label)
	       -->
	       (control-spec-info* ,spec-obj-id
				  ^ reject-label ,op-cond-label + &))))

	 (values initial-state final-state)))

      (op-spec ; Control-Spec is (MOD+ OPERATOR-CONTROL-SPEC+)
       (multiple-value-bind (spec-type sub-specs when exit-when indifferent
				       reject-after one-instance one-of-type
				       select-when-proposed optional loop-name)
	   (parse-op-spec-list control-spec)
	   (declare (ignore one-instance))

	 (let ((all-conds (append when common-conds)))
	   (multiple-value-bind (initial-state final-state)
	       (compile-op-spec-of-type spec-type sub-specs control-spec-id
					all-conds reject-after
					one-of-type loop-name)

	     ;; We have to do these now, because for the ones that compute
	     ;; epsilon closures, we only want the epsilon closure
	     ;; over the part of the machine that corresponds to this part
	     ;; of the control-spec, not the control-spec as a whole.  For
	     ;; example, if we have (:seq (:or :ind (:loop a) b) c), then the
	     ;; :ind should only apply to a and b (in the epsilon closure of
	     ;; the :or submachine), but not to c (which is in the epsilon
	     ;; closure of the machine as a whole).
	     ;;
	     (when indifferent
	       (compile-op-spec-indifferent initial-state control-spec-id))
	     (when select-when-proposed
	       (compile-op-spec-select-when-proposed initial-state
						     control-spec-id))
	     (when exit-when
	       (compile-op-spec-exit-when initial-state control-spec-id
					  exit-when))
	     (when optional
	       (setf (get initial-state 'optional-submachine)
		     final-state)
	       (add-transition initial-state '*epsilon* final-state))

	     (values initial-state final-state))))))))

;;; Parse an operator control-spec of the form (MOD+ OPERATOR-CONTROL-SPEC+).
;;;
;;; Return multiple values, in the following order:
;;;
;;;    spec-type:  one of sequence, loop, or.
;;;    sub-specs:  a list of the OPERATOR-CONTROL-SPECs appearing in the
;;;        control-spec
;;;    When:  a list of conditions that appeared in :when keywords, already
;;;        processed by prepare-condition-sets for your convenience.
;;;    Exit-when:  a list of conditions that appeared in :exit-when keywords,
;;;        already processed by prepare-condition-sets for your convenience.
;;;    Indifferent:  non-nil when :ind was given
;;;    Reject-after:  non-nil when :ra was given
;;;    One-instance:  non-nil when :oi was given
;;;        CURRENTLY :OI IS UNIMPLEMENTED, and will always be returned as nil.
;;;    One-of-type:  non-nil when :oot was given
;;;    Select-when-proposed:  non-nil when :swp was given
;;;    Optional:  non-nil when :optional was given
;;;    Loop-name:  value of any :loop-name keyword in a :loop
;;; 
;;; Although the syntax says the modifier keywords must appear at the
;;; beginning of the list, in fact they can appear anywhere in the list
;;; as long as at least one of them is at the beginning (lists that don't
;;; begin with keywords are assumed to be OP-CONDs).
;;;
(defun parse-op-spec-list (spec)
  (do ((spec-elements spec)
       (spec-type nil)
       (sub-specs nil)
       (when-conds nil)
       (exit-when nil)
       (indifferent nil)
       (reject-after nil)
       (one-instance nil)
       (one-of-type nil)
       (select-when-proposed nil)
       (optional nil)
       (loop-name nil))
      ((null spec-elements)

       (setq sub-specs (reverse sub-specs))

       ;; :when and :exit-when imply :optional
       ;;
       (when (or when-conds exit-when)
	 (setq optional t))

       ;; Default control-spec type is sequence
       ;;
       (when (not spec-type)
	 (setq spec-type 'sequence))

       ;; Every loop gets a name
       ;;
       (when (and (eql spec-type 'loop)
		  (not loop-name))
	 (setq loop-name (newsym 'loop)))

       (when (eq spec-elements spec)
	 (taql-warn "An OPERATOR-CONTROL-SPEC must begin with a keyword: ~S"
		    spec))
       (when (not sub-specs)
	 (taql-warn
	  "An OPERATOR-CONTROL-SPEC must have at least one sub-control-spec: ~S"
	  spec)
	 (setq sub-specs '(junk)))
       (when (and reject-after
		  (or (cdr sub-specs)
		      (not (eql (operator-spec-class (car sub-specs))
				'op-cond))))
	 (taql-warn
	  ":Ra can only be used in an OPERATOR-CONTROL-SPEC whose only element is an OP-COND: ~S"
	  spec)
	 (setq reject-after nil))
       (when (and one-of-type
		  (or (cdr sub-specs)
		      (not (eql (operator-spec-class (car sub-specs))
				'op-cond))))
	 (taql-warn
	  ":Oot can only be used in an OPERATOR-CONTROL-SPEC whose only element is an OP-COND: ~S"
	  spec)
	 (setq one-of-type nil))
       (when (and loop-name
		  (not (eql spec-type 'loop)))
	 (taql-warn ":Loop-name can only appear in a :loop: ~S" spec)
	 (setq loop-name nil))
       (when (and loop-name
		  (or (not (symbolp loop-name))
		      (variable-p loop-name)))
	 (taql-warn "The value of :loop-name must be a non-variable symbol: ~S"
		    spec))

       (values spec-type sub-specs (prepare-condition-sets when-conds)
	       (prepare-condition-sets exit-when) indifferent
	       reject-after one-instance one-of-type select-when-proposed
	       optional loop-name))

    (if (keywordp (car spec-elements))
      (case (car spec-elements)
	(:ind
	 (setq indifferent t))
	(:ra
	 (setq reject-after t))
	(:oi
	 (taql-warn ":Oi is not yet implemented: ~S" spec))
	(:oot
	 (setq one-of-type t))
	(:swp
	 (setq select-when-proposed t))
	(:optional
	 (setq optional t))
	(:loop-name
	 (when loop-name
	   (taql-warn "At most one :loop-name keyword can appear: ~S" spec))
	 (setq loop-name (cadr spec-elements))
	 (when (null loop-name)
	   (taql-warn "NIL is not a legal value for :loop-name: ~S" spec)))
	(:exit-when
	 (push (cadr spec-elements) exit-when))
	(:when
	 (push (cadr spec-elements) when-conds))
	(:seq
	 (when (and spec-type (not (eql spec-type 'sequence)))
	   (taql-warn "At most one of :seq, :loop, :or may appear: ~S" spec))
	 (setq spec-type 'sequence))
	(:loop
	 (when (and spec-type (not (eql spec-type 'loop)))
	   (taql-warn "At most one of :seq, :loop, :or may appear: ~S" spec))
	 (setq spec-type 'loop))
	(:or
	 (when (and spec-type (not (eql spec-type 'or)))
	   (taql-warn "At most one of :seq, :loop, :or may appear: ~S" spec))
	 (setq spec-type 'or))
	(otherwise
	 (taql-warn "~S is not a valid keyword in an OPERATOR-CONTROL-SPEC: ~S"
		    (car spec-elements) spec)))
      ;; ELSE not a keyword
      (push (car spec-elements) sub-specs))

    (case (car spec-elements)
      ((:when :exit-when :loop-name)
       (setq spec-elements (cddr spec-elements)))
      (otherwise
       (setq spec-elements (cdr spec-elements))))))

;;; Compile an operator control-spec of a specified type (sequence,
;;; loop, or).  sub-specs are the control-specs appearing within the
;;; control-spec being compiled, Common-conds are conditions from
;;; :when keyowrds in the control-spec, together with any inherited
;;; from higher levels of the control-spec.  Reject-after and
;;; one-of-type are non-nil if the control-spec has the :ra or :oot
;;; modifiers, respectively.  Loop-name is the value of any :loop-name
;;; keyword in a loop control-spec.
;;;
;;; Common-conds are propagated to all sub-control-specs for :loop and
;;; :or control-specs, but are only propagated to the initial
;;; sub-control-spec of :seq control-specs.  This is because we only
;;; want :when conditions to constrain the selection of the initial
;;; operator in a control-spec.  Return two values, the labels of the
;;; initial and final NFA states for the control-spec.
;;;
(defun compile-op-spec-of-type (spec-type sub-specs control-spec-id
					  common-conds
					  reject-after one-of-type loop-name)

  ;; Have to do this BEFORE compiling sub-control-specs.
  ;;
  (when loop-name
    (push-loop-name loop-name))

  (let ((sub-nfa-terminals
	 (cons (multiple-value-list
		(compile-operator-control-spec (car sub-specs) control-spec-id
					       common-conds reject-after
					       one-of-type))
	       (mapcar #'(lambda (sub-spec)
			   (multiple-value-list
			    (compile-operator-control-spec
			     sub-spec control-spec-id
			     (if (eql spec-type 'sequence)
			       nil
			       ;; ELSE
			       common-conds)
			     reject-after one-of-type)))
		       (cdr sub-specs)))))

    (case spec-type

      (sequence ; (:seq ...)
       (do ((terminals sub-nfa-terminals (cdr terminals)))
	   ((null (cdr terminals))
	    (values (caar sub-nfa-terminals) (cadar terminals)))
	 (add-transition (cadar terminals) '*epsilon* (caadr terminals))))

      (or ; (:or ...)
       (let ((initial-state (new-nfa-state))
	     (final-state (new-nfa-state)))
	 (dolist (terminal sub-nfa-terminals)
	   (add-transition initial-state '*epsilon* (car terminal))
	   (add-transition (cadr terminal) '*epsilon* final-state))
	 (values initial-state final-state)))

      (loop ; (:loop ...)
       (let ((extra-state (new-nfa-state))
	     (loop-final-state (new-nfa-state)))

	 (dolist (terminal sub-nfa-terminals)
	   (add-transition extra-state '*epsilon* (car terminal))
	   (add-transition (cadr terminal) '*epsilon* extra-state))

	 ;; We require that the final state of a submachine never have
	 ;; a transition back into the machine, so we need an extra
	 ;; state here to use as a final state.  This restriction is
	 ;; there in part so that the :optional keyword works properly.
	 ;;
	 (add-transition extra-state '*epsilon* loop-final-state)

	 ;; A loop is always implicitly :optional.
	 ;;
	 (setf (get extra-state 'optional-submachine) loop-final-state)

	 (do-loop-link-surgery loop-name extra-state loop-final-state)

	 (when loop-name
	   (pop-loop-name))

	 (values extra-state loop-final-state)))

      (otherwise
       (error "INTERNAL TAQL ERROR: Unknown spec-type ~S" spec-type)))))

;;; Generate the productions for an :ind modifier in a control-spec.  The
;;; initial-state argument is the label of the initial state of the NFA
;;; for the sub-control-spec the :ind keyword appeared in.
;;;
;;; This function must be called BEFORE the submachine is linked into the
;;; the machine for its parent control-spec, because :ind only applies to
;;; transitions in the submachine.
;;;
;;; Operators labeled with any arc label that transitions out of the
;;; epsilon closure of the initial state are indifferent.
;;;
(defun compile-op-spec-indifferent (initial-state control-spec-id)
  (let* ((arc-labels (non-epsilon-arc-labels (epsilon-closure initial-state)))
	 (arc-label-cond (if (cdr arc-labels)
			   `(<< ,@arc-labels >>)
			   ;; ELSE
			   arc-labels))
	 (op1-id (genvar 'o))
	 (op2-id (genvar 'o))
	 (c1-id (genvar 'c))
	 (c2-id (genvar 'c))
	 (spec-obj-id (genvar 'p)))

    (when arc-labels
      (eval
       `(taql-sp unknown
	    ,(newsym *prod-name-prefix* 'indifferent *num-separator*)
	  (goal ,=goal ^ taql-control-spec* ,spec-obj-id
		^ operator ,op1-id + { <> ,op1-id ,op2-id } +)
	  (control-spec-info* ,spec-obj-id
			     ^ id ,control-spec-id ^ nfa-state ,initial-state)
	  (operator ,op1-id ^ control-stuff* ,c1-id)
	  (control-stuff* ,c1-id ^ label ,@arc-label-cond)
	  (operator ,op2-id ^ control-stuff* ,c2-id)
	  (control-stuff* ,c2-id ^ label ,@arc-label-cond)
	  -->
	  (goal ,=goal ^ operator ,op1-id = ,op2-id))))))

;;; Generate the productions for an :swp modifier in a control-spec.  The
;;; initial-state argument is the label of the initial state of the NFA
;;; for the sub-control-spec the :swp keyword appeared in.
;;;
;;; This function must be called BEFORE the submachine is linked into the
;;; the machine for its parent control-spec, because :swp only applies to
;;; transitions in the submachine.
;;;
;;; Operators labeled with any arc label that transitions out of the
;;; epsilon closure of the initial state are tagged as select-when-proposed.
;;;
(defun compile-op-spec-select-when-proposed (initial-state control-spec-id)
  (let* ((closure (epsilon-closure initial-state))
	 (arc-labels (non-epsilon-arc-labels closure))
	 (arc-label-cond (if (cdr arc-labels)
			   `(<< ,@arc-labels >>)
			   ;; ELSE
			   arc-labels))
	 (c-id (genvar 'c))
	 (spec-obj-id (genvar 'p)))

    ;; All states in the epsilon-closure of the initial state of an swp
    ;; submachine are also swp states.
    ;;
    (dolist (state-label closure)
      (setf (get state-label 'select-when-proposed) t))

    (when arc-labels
      (eval
       `(taql-sp not-sticky
	    ,(newsym *prod-name-prefix* 'select-when-proposed *num-separator*)
	  (goal ,=goal
		^ taql-control-spec* ,spec-obj-id
		^ operator ,=operator +)
	  (control-spec-info* ,spec-obj-id
			     ^ id ,control-spec-id ^ nfa-state ,initial-state)
	  (operator ,=operator ^ control-stuff* ,c-id)
	  (control-stuff* ,c-id ^ label ,@arc-label-cond)
	  -->
	  (control-stuff* ,c-id ^ select-when-proposed true))))))

;;; Generate productions that disable the labels of arcs that lead out of
;;; the epsilon closure of the initial state of a submachine when its
;;; exit-when conditions are satisfied.
;;;
(defun compile-op-spec-exit-when (initial-state control-spec-id exit-when)
  (let* ((closure (epsilon-closure initial-state))
	 (arc-labels (non-epsilon-arc-labels closure))
	 (spec-obj-id (genvar 'p)))

    (when arc-labels
      (eval
       `(taql-sp not-sticky
	    ,(newsym *prod-name-prefix* 'disable-transitions-upon-exit-when
		     *num-separator*)
	  (goal ,=goal ^ taql-control-spec* ,spec-obj-id)
	  (control-spec-info* ,spec-obj-id
			     ^ id ,control-spec-id ^ nfa-state ,initial-state)
	  ,@exit-when
	  -->
	  (control-spec-info* ,spec-obj-id ^ disable-label ,@arc-labels))))))

;;; Edit NFA transitions so that exit-loop operators transition to the
;;; loop's final state, and next-iteration operators transition to the
;;; loop's initial state.
;;;
(defun do-loop-link-surgery (loop-name initial-state final-state)
  (dolist (transition (get loop-name 'loop-exit-info))
    (add-transition (car transition) (cadr transition) final-state))
  (dolist (transition (get loop-name 'loop-next-iteration-info))
    (add-transition (car transition) (cadr transition) initial-state)))

;;; When compiling an OP-COND, intercept the special loop operators
;;; exit-loop and next-iteration.  Parse their loop-name attribute
;;; (if any), mark the final state of the OP-COND submachine with
;;; either a loop-exit-info or a loop-next-iteration-info,
;;; and return the prep-op for just the operator name.
;;;
;;; If the operators is not a special loop operator, return the prep-op
;;; argument unchanged, and take no other action.
;;;
;;; We return two values, the first is the resulting prep-op, and the
;;; second is non-nil when the operator was a special loop operator.
;;;
(defun intercept-special-loop-operators (prep-op op-id op-initial-state
						 op-cond-label user-text)
  (let* ((op-name (caar prep-op))
	 (extra-conds (cdar prep-op))
	 (op-cond (cadr prep-op))
	 (extra-op-conds (cddr prep-op))
	 (inner-loop (innermost-loop-name)))
    (cond
     ((member op-name '(exit-loop next-iteration))
      (cond ((not inner-loop)
	     (taql-warn
	      "~S can only be used within a loop: ~S" op-name user-text))
	    ((or extra-conds extra-op-conds
		 (not (subsetp
		       (find-attribute-names-in-cond op-cond)
		       '(name loop-name))))
	     (taql-warn
	      "~S can only have a loop-name attribute: ~S" op-name user-text))
	    (t
	     (let* ((loop-name-att (member 'loop-name op-cond))
		    (loop-name (if loop-name-att
				 (cadr loop-name-att)
				 ;; ELSE
				 inner-loop)))
	       (when (not (loop-name-p loop-name))
		 (taql-warn "~S is not the name of a surrounding loop: ~S"
			    loop-name user-text)
		 (setq loop-name inner-loop))
	       (push (list op-initial-state op-cond-label)
		     (get loop-name
			  (if (eql op-name 'exit-loop)
			    'loop-exit-info
			    ;; ELSE
			    'loop-next-iteration-info))))))
      (values (prepare-operator op-name op-id) t))
     (t
      (values prep-op nil)))))

;;; Given a condition such as (class <id> ^ att val ^ att-2 val-2 val-3 ...),
;;; return a list of all the attribute names that appear in it.
;;;
;;; We assume that attribute names always immediately follow the ^, so that
;;; this won't work properly on something like
;;;    (class <id> ^ << att1 att2 >> val)
;;;
;;; This also will not work if cond has attribute path names (e.g. ^a.b.c).
;;; This should never be the case given the context in which this function
;;; is called (from intercept-special-loop-operators, to check that at most
;;; "name" and "loop-name" attributes are given in the OP-COND for a special
;;; loop operator).
;;;
(defun find-attribute-names-in-cond (cond)
  (do* ((current cond (cdr current))
	(result nil result))
      ((null current) result)
    (when (eql (car current) '^)
      (push (cadr current) result))))

(eval-when (compile load eval)
  (soarsyntax))
