;;; -*- Mode: Soar -*- 
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;;; 
;;;; File            : soarsim.lisp
;;;; Author          : Robert H Guttman
;;;; Created On      : Fri Apr 10 21:44:41 1992
;;;; Last Modified By: Scott Huffman
;;;; Last Modified On: Mon Jun 29 14:20:08 1992
;;;; Update Count    : 379
;;;; Soar Version    : 5.2.1
;;;; 
;;;; PURPOSE
;;;; 	SoarSIM is a simulation tool for the Soar general intelligence
;;;;  architecture.
;;;;
;;;; 2nd round of changes started 6/8/92 SBH:
;;;;    - structured objects
;;;;    - multiattribute fillers
;;;;
;;;; (C) Copyright 1992, University of Michigan, all rights reserved.
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

(lispsyntax)

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;;
;;;  SET-OUTPUT-MAPPINGS & SET-INPUT-FUNCTIONS
;;;
;;;  The Soar set-output-mappings function sets up the output-link mapping
;;;  between an output-link sent from Soar and SoarSIM's output-commands-
;;;  interface which eventually executes the associated output-command.
;;;  The Soar set-input-functions defines the SoarSIM input-systems-
;;;  interface as the function to handle the internalization of sensory
;;;  information from an external world.
;;;
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

(set-output-mappings ((output-commands-interface output-link)))
(set-input-functions (input-systems-interface))

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;;
;;;  Simulation Global Variables
;;;
;;;  *WORLD-OBJECTS*
;;;    This is a list of all world-objects defined using SoarSIM's
;;;    defobject function.  It can be used to check relationships between
;;;    world-objects in the external laws module or any of the other three
;;;    modules.
;;;
;;;  *CHANGED-WORLD-OBJECTS*
;;;    This is a list of defined world-objects similar to *world-objects*
;;;    except that *changed-world-objects* only contains those objects
;;;    that have changed in some way.  It is a useful function because it
;;;    enables an input-system to easily determine which world-objects
;;;    (of those that can be sensed) need to be internalized so as to update
;;;    one or more of its attribute-value pairs.
;;;
;;;    All newly defined world-objects are pushed onto the *changed-world-
;;;    objects* list.  Once they are popped off, they will be be pushed
;;;    back on if (1) they have a new attribute-value pair added to any
;;;    one of their four fields (public, private, output, or input) using
;;;    SoarSIMs attribute-value function, or (2) they have the value of
;;;    an existing attribute changed to a new value using SoarSIM's
;;;    attribute-value function.
;;;
;;;  *EXTERNAL-LAWS*
;;;    This is a list of external-laws as defined by SoarSIM's defexternal
;;;    function.  It is used by the external-laws-interface to call each
;;;    external-law when appropriate.
;;;
;;;  *INPUT-SYSTEMS*
;;;    This is a list of input-systems as defined by SoarSIM's definput
;;;    function.  It is used by the input-systems-interface to call each
;;;    input-system.
;;;
;;;  *OUTPUT-COMMANDS*
;;;    This is a list of output-commands sent by Soar.  It is recreated
;;;    each Elaboration Cycle in which one or more output-commands are
;;;    sent from Soar.  It is used by the output-commands-interface to
;;;    indicate which output-command-function to call.
;;;
;;;  *OUTPUT-COMMAND-FUNCTIONS*
;;;    This is a list of output-command-functions as defined by SoarSIM's
;;;    defoutput function.  It is used by the output-commands-interface to
;;;    help map output-commands onto their respective output-command-
;;;    function.
;;;
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

(setf *world-objects* NIL)
(setf *changed-world-objects* NIL)
(setf *external-laws* NIL)
(setf *input-systems* NIL)
(setf *output-commands* NIL)
(setf *output-command-functions* NIL)

(setf *update-pointer-list* nil)

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;;
;;;  Environment Global Variables
;;;
;;;  *SOARSIM-VERBOSE*
;;;    If the *text* global variable is set to NIL, no text will be
;;;    displayed when SoarSIM's display-text function is called.  SoarSIM's
;;;    set-text function sets the *text* global variable to either T or NIL
;;;    to display or not display text using display-text.
;;;
;;;  *SOARSIM-DEBUG*
;;;    If the *debug* global variable is set to NIL, no text will be
;;;    displayed when SoarSIM's display-debug function is called.  SoarSIM's
;;;    set-debug function sets the *debug* global variable to either T or NIL
;;;    to display or not display text using display-debug.
;;;
;;;  *SOARSIM-WARNINGS*
;;;    If the *soarsim-warnings* global variable is set to NIL, no warnings
;;;    will be displayed when SoarSIM's display-warnings function is called.
;;;    SoarSIM's set-warnings function sets the *soarsim-warnings* global
;;;    variable to either T or NIL to display or not display warnings using
;;;    display-warnings.
;;;
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

(setf *soarsim-verbose* T)
(setf *soarsim-debug* NIL)
(setf *soarsim-warnings* T)

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;;
;;;  World Objects' Structure
;;;
;;;  Each world-object has a public, private, output, and input field, and
;;;  initialization fields to store initial values.
;;;
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

(defstruct world-object
  (public  NIL)
  (private NIL)
  (public-init  NIL)
  (private-init NIL)
  (output  NIL)
  (input   NIL))


;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;;
;;;  OUTPUT-COMMANDS-INTERFACE
;;;
;;;  This function is called during each Elaboration Cycle in which SOAR
;;;  has sent one or more output-commands to an output mechanism.  Its
;;;  purpose is to execute output-commands by calling their associated
;;;  output-command-function as defined in an object's output field.
;;;
;;;  First, the *output-commands* global variable is built by retrieving
;;;  output-commands from SOAR's working memory.  Second, a loop begins,
;;;  iterating through each output-command (i.e. flip-switch) listed in
;;;  *output-commands*.  Next, the output-command-function associated with
;;;  the current output-command is found by traversing each world-object's
;;;  output field looking for an appropriate mapping. (i.e. flip-simulator-
;;;  switch).  An example output-command/output-command-function pair would
;;;  be (flip-switch flip-simulator-switch).  Next, if an output-command-
;;;  function is found, that function is called.  If one is not found, an
;;;  error message is displayed.
;;;
;;;  OUTPUTS
;;;    This is output-link passed by Soar.  It is used to retrieve the
;;;    output-commands from Soar's working memory.
;;;
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

(defun output-commands-interface (outputs)
  (let ((output-command NIL)
        (output-command-function NIL))
    (display-debug "~&~%==Output Commands Module==~%")
    (setf *outputs* outputs)
    (get-output-commands)
    (dolist (output-command *output-commands*)
      (setf output-command-function (get-output-command-function output-command))
      (if output-command-function
	  (if (defined-output-command-function-p output-command-function)
            (funcall output-command-function)
            (display-errors "~%; ERROR: Undefined output command function - ~a. ~%"
			      output-command-function))
	  (display-errors "~%; ERROR: Invalid output command - ~a. ~%"
			  output-command)))))


(defun get-output-commands ()
  (setf *output-commands* (get-output-values 'command *outputs*)))


(defun get-output-command-function (output-command)
  (let ((output-command-function NIL))
    (dolist (world-object *world-objects*)
      (setf output-command-function (car (att-value world-object output-command)))
      (if output-command-function
	  (return output-command-function)))))


(defun defined-output-command-function-p (output-command-function)
  (if (member output-command-function *output-command-functions* :test #'equal)
      T NIL))


;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;;
;;;  EXTERNAL-LAWS-INTERFACE
;;;
;;;  This function is called during each Soar Elaboration Cycle.
;;;  It matches & executes all applicable external laws by calling the
;;;  external law matching functions produced by defexternals.
;;;
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;


(defun external-laws-interface ()
    (display-debug "~&~%==External Laws Module==~%")
    (when *changed-world-objects*
      (dolist (el *external-laws*)
	(funcall el))))


;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;;
;;;  INPUT-SYSTEMS-INTERFACE
;;;
;;;  This function is called during each Soar Elaboration Cycle. Its
;;;  purpose is to internalize attributes and values of world-objects via
;;;  input-systems.  Each input-system defined using SoarSIM's definput
;;;  function gets called during each Soar Elaboration Cycle.
;;;
;;;  The external laws interface precedes the input systems interface
;;;  each cycle of the Interaction Model and is the first function called.
;;;  Next, a loop begins, iterating through each input-system (i.e.
;;;  vision-input-system) defined by SoarSIM's definput function.
;;;  Each defined input-system is then called, each internalizing world-
;;;  objects as appropriate.
;;;
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;


(defun input-systems-interface ()
  (let ((input-command NIL)
        (output-command-function NIL))
    (external-laws-interface)
    (display-debug "~&~%==Input Systems Module==~%")
    (dolist (input-system *input-systems*)
      (funcall input-system))
    ))


(defun get-world-object-of-input-system (input-system)
  (dolist (world-object *world-objects*)
    (if (member input-system (world-object-input world-object) :test #'equal)
        (return world-object))))


;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;;
;;;  DEFOBJECT (macro)
;;;
;;;  SoarSIM objects are defined using SoarSIM's defobject function.
;;;  (In reality, defobject is a macro.)  The purpose of this function
;;;  is to define each of the four possible world-object fields: public,
;;;  private, output, and input, relating to public & private attributes,
;;;  output-command mappings to associated output-command-functions, and
;;;  input-system listings respectively.
;;;
;;;  A skeleton world-object definition looks like this:
;;;
;;;    (defobject *world-object-1*
;;;      :public  '((pub-att-1 pub-val-1) (pub-att-2 pub-val-2) ... )
;;;      :private '((pri-att-1 pri-val-1) (pri-att-2 pri-val-2) ... )
;;;      :output  '((output-command-1 output-command-function-1)
;;;                 (output-command-2 output-command-function-2)
;;;                          .                   .
;;;                          .                   .
;;;      :input   '((input-system-1 input-system-2 ... ))
;;;
;;;  All or none fields need to be filled when a world-object is first
;;;  defined.  Attribute-value pairs for all fields can be dynamically
;;;  added at any time using SoarSIM's attribute-value function.
;;;
;;;  World-objects can also inherit attribute-value pairs from previously
;;;  defined world-objects.  This is done by defining a world-object with
;;;  a list of previously-defined world-objects such as this:
;;;
;;;    (defobject (*world-object-2* *world-object-1*)
;;;      :public  '((new-pub-att-x new-pub-val-x)
;;;                 (pub-att-1 new-pub-val-1) ... )
;;;
;;;  All of *world-object-1*'s attribute-value pairs will be inherited by
;;;  *world-object-2*.  Also, any newly defined attribute-value pairs will
;;;  be added to *world-object-2*.  Also, values of attributes of inherited
;;;  objects can be overridden be redefining them as is shown with
;;;  *world-object-1*'s pub-att-1 and *world-object-2*'s new value for this
;;;  attribute, new-pub-val-1.
;;;
;;;  Multiple world-objects may also be inherited such as this:
;;;
;;;    (defobject (*world-object-3* *world-object-2* *world-object-1*)
;;;      :public ... )
;;;
;;;  In this case, *world-object-1*'s attribute-value pairs are first
;;;  inherited, then *world-object-2*'s which can override anyone of
;;;  *world-object-1*'s attribute-value pairs.  Lastly, newly defined
;;;  attribute-value pairs can be defined for *world-object-3*.
;;;
;;;  WORLD-OBJECT-NAME
;;;    This is the name of the newly defined world-object.  If this
;;;    argument is a list (indicating a desire to inherit previously
;;;    defined world-objects), then the first element of this list
;;;    will be the name of the newly defined world-object.
;;;
;;;  ATTRIBUTE-AND-VALUES (optional)
;;;    This is the listing of attribute-value pairs for one or more
;;;    of the four world-object fields: public, private, output, and input.
;;;
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

;;; Fully rewritten SBH 5/21/92

(defmacro defobject (world-object-name &rest attribute-and-values)
  (let ((object-name world-object-name)
        (old-object NIL)
	(object (make-world-object))
	(public-list (second (member ':public attribute-and-values)))
	(private-list (second (member ':private attribute-and-values)))
	(input-list (second (member ':input attribute-and-values)))
	(output-list (second (member ':output attribute-and-values))))

    (pushnew object *world-objects*)

    (setf (world-object-private object) private-list)
    (setf (world-object-public object) public-list)
    (setf (world-object-private-init object) (full-copy private-list))
    (setf (world-object-public-init object) (full-copy public-list))
    (setf (world-object-input object) input-list)
    (setf (world-object-output object) output-list)


    ;; Inheritance of attributes of other objects.
    (when (listp world-object-name)
      (setf object-name (first world-object-name))
      (setf old-objects (reverse (rest world-object-name)))
      (dolist (old-object old-objects)
        (inherit-object object (eval old-object))))

    (att-value object 'global-var-name object-name 'private)
    (att-value object 'global-var-name object-name 'private-init)


    ;;; Set the global variable that names the object to be the object.
    (list 'setf object-name (list 'quote object))
    ))
    

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;;
;;;  INHERIT-OBJECT
;;;
;;;  This function adds all of the attribute-value pairs of the second
;;;  world-object passed to the first world-object passed.  It does this
;;;  by simply traversing through each of the four world-objects fields
;;;  and calling SoarSIM's attribute-value function to add each
;;;  attribute-value pair in each field.  SBH: If the new-object already has
;;;  a value for the attribute, then it is left unchanged.
;;;
;;;  NEW-OBJECT
;;;    This is the object defined by SoarSIM's defobject function that
;;;    inherits attribute-values from the old-object.
;;;
;;;  OLD-OBJECT
;;;    This is the object defined by SoarSIM's defobject function that
;;;    has the attribute-value pairs in its four fields copied to the
;;;    new-object.
;;;
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;


(defun inherit-object (new-object old-object)
    (dolist (att-val (world-object-public old-object))
      (inherit-attribute-value new-object att-val 'public))

    (dolist (att-val (world-object-public-init old-object))
      (inherit-attribute-value new-object att-val 'public-init))
    
    (dolist (att-val (world-object-private old-object))
      (inherit-attribute-value new-object att-val 'private))
    
    (dolist (att-val (world-object-private-init old-object))
      (inherit-attribute-value new-object att-val 'private-init))
    
    (dolist (att-val (world-object-output old-object))
      (inherit-attribute-value new-object att-val 'output))
    
    (dolist (inp-sys (world-object-input old-object))
      (pushnew inp-sys (world-object-input new-object)))
    
    new-object)


(defun inherit-attribute-value (new-obj att-val field-type)
  (let ((att (first att-val))
	(val (second att-val)))
    (if (not (has-value new-obj att field-type))
	(att-value new-obj att val field-type))))



(defun has-value (obj att field)
  (case field
    ('public (second (assoc att (world-object-public obj))))
    ('private (second (assoc att (world-object-private obj))))
    ('public-init (second (assoc att (world-object-public-init obj))))
    ('private-init (second (assoc att (world-object-private-init obj))))
    ('input (second (assoc att (world-object-input obj))))
    ('output (second (assoc att (world-object-output obj))))
    ))

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;;
;;; NUMBER-OF-OBJECTS
;;;
;;; This function returns the number of world-objects defined using
;;; SoarSIM's defobject function.
;;;
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

(defun number-of-objects () (length *world-objects*))

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;;
;;; DEFOUTPUT (macro)
;;;
;;; This SoarSIM function is used to define output-command-functions.
;;; Defining output-command-functions is similar to defining a regular
;;; Common Lisp function.  A skeleton input-system looks like this:
;;;
;;; (defoutput output-command-function-name ()
;;; ( ... lisp code ... ))
;;;
;;; Care should be taken to make sure that output-command-functions are
;;; the proper mapping of Soar's output-commands as defined by world-
;;; objects' output fields.  No arguments are passed to output-command-
;;; functions.
;;;
;;; FUNCTION-NAME
;;; This is the name of the newly defined output-command-function.
;;;
;;; FUNCTION-BODY
;;; This is the body of the output-command-function and is similar
;;; to regular Common Lisp code.
;;;
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;


(defmacro defoutput (function-name &rest function-body)
  (pushnew function-name *output-command-functions*)
  `(defun ,function-name ,@function-body))


;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;;
;;; DEFEXTERNAL (macro)
;;;
;;; This SoarSIM function is used to define external-laws.  Defining
;;; external-laws is similar to defining a regular Common Lisp function
;;; except that a test-list is included which specifies the criteria
;;; for deciding which world-objects should get passed to this external-
;;; law.
;;;
;;;
;;; REWRITTEN SBH 5-22
;;;
;;; An external law builds a function (with a gensymed name) that, when
;;; called, matches all world objects against the law's tests, and for every
;;; match, calls the law's body of code.
;;;
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;


(defmacro defexternal (function-name &rest law &aux test-list function-body var-list)
  ;; Find test-list & function-body.
  (do* ((law law (cdr law))
	(piece (car law)(car law)))
       ((eq piece '-->)(setf function-body (rest law)))
    (push piece test-list))
  (setf test-list (reverse test-list))

  ;; Find variables to be passed.
  (dolist (cnd test-list)
    (if (soarvar-p (first cnd))
	(pushnew (first cnd) var-list))
    (if (soarvar-p (third cnd))
	(pushnew (third cnd) var-list)))

  (pushnew (build-external-law-caller function-name test-list var-list) *external-laws*)
  `(defun ,function-name ,var-list ,@function-body))


(defun soarvar-p (v)
  (let* ((sv (prin1-to-string v))
 	 (sv1st (elt sv 0))
	 (svlast (elt sv (- (length sv) 1))))
    (and (eq sv1st #\<)
	 (eq svlast #\>))))

(defun sim-varnamep (v)
  (let* ((sv (prin1-to-string v))
 	 (sv1st (elt sv 0))
	 (svlast (elt sv (- (length sv) 1))))
    (and (eq sv1st #\*)
	 (eq svlast #\*))))



(defun build-external-law-caller (fn tests vars &aux result name)
  (setf result `(if (ext-law-match (quote ,tests)
		                   (make-binding-list (quote ,vars) ,@vars))
		    (funcall (quote ,fn) ,@vars)))
  (dolist (var (reverse vars) result)
    (setf result (list 'dolist `(,var *world-objects*) result)))

  (setf result (list 'defun (setf name (intern (gensym "F"))) '() result))
  (eval result)
  )

(defun make-binding-list (varlist &rest vals)
  (mapcar #'cons varlist vals))
  

;;; The matcher for a set of tests and bindings.

(defun ext-law-match (tests binding-list)
  (dolist (test tests)
    (let ((obj (cdr (assoc (car test) binding-list)))
	  (property (second test))
	  (value (if (soarvar-p (third test))
		     (eval (car (att-value (cdr (assoc (third test) binding-list))
					   'global-var-name)))
		     (third test)))
	  (negation (if (eq (fourth test) '-) T NIL)))
      (if negation
	  (cond ((member value (att-value obj property))
		 (return-from ext-law-match nil)))
	  (cond ((or (null (field-of-attribute obj property))
		     (not (member value (att-value obj property))))
;;		 (format t "Obj ~a~%    Property: ~a, Value: ~a~%" obj property value)
		 (return-from ext-law-match nil))))))
  T
  )


;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;;
;;;  DEFINPUT (macro)
;;;
;;;  This SoarSIM function is used to define input-systems.  Defining
;;;  input-systems is similar to defining a regular Common Lisp function.
;;;  A skeleton input-system looks like this:
;;;
;;;    (definput input-system-name ()
;;;      ( ... lisp code ... ))
;;;
;;;  No arguments are passed to input-systems.
;;;
;;;  FUNCTION-NAME
;;;    This is the name of the newly defined input-system.
;;;
;;;  FUNCTION-BODY
;;;    This is the body of the input-system and is regular
;;;    Common Lisp code.
;;;
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;


(defmacro definput (function-name &rest function-body)
  (pushnew function-name *input-systems*)
  `(defun ,function-name ,@function-body))


;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;;
;;; ATT-VALUE
;;;
;;; This function accesses the attributes and values of a specified
;;; world-object.  It can either simply return a value for a specified
;;; attribute of a world-object or set the value of an attribute of a
;;; world-object.  New attribute-value pairs can also be added to a
;;; world-object using this function with the optional ability to
;;; specify under which field to place the new pair.  How this function
;;; operates is best explained using the following chart:
;;;
;;;    Are just a world-object and an attribute passed?
;;;    * YES
;;;      --> If the public field has this attribute, return its value
;;;      --> If the private field has this attribute, return its value
;;;      --> If the output field has this attribute, return its value
;;;      --> Otherwise, return NIL
;;;    * NO, a value is also passed
;;;      --> If the public field has this attribute,
;;;          change its value ONLY if it differs
;;;          then push this world-object it onto *changed-world-objects*
;;;      --> If the private field has this attribute,
;;;          change its value ONLY if it differs
;;;          then push this world-object it onto *changed-world-objects*
;;;      --> If the output field has this attribute,
;;;          change its value ONLY if it differs
;;;          then push this world-object it onto *changed-world-objects*
;;;      --> Otherwise, is a field also specified?
;;;          * YES
;;;            --> Add the new attribute-value pair to this field
;;;          * NO, just a value is specified
;;;            --> Add the new attribute-value pair to the public field
;;;
;;;  It should be noted that when adding a new attribute-value pair
;;;  without specifying which field it should be put in, the pair
;;;  will be put in the public field.
;;;
;;;  WORLD-OBJECT
;;;    This argument is a world-object as defined by the defobject SoarSIM
;;;    function.
;;;
;;;  ATTRIBUTE
;;;    This is an attribute of the world-object passed to this function.
;;;
;;;  VALUE (optional)
;;;    This is the value that the attribute of the world-object should
;;;    be set to (if it doesn't already have that value).
;;;
;;;  FIELD (optional)
;;;    This is the field that a new attribute-value pair should be put in
;;;    of the world-object (if that attribute doesn't already exist for
;;;    the world-object).
;;;
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

(defun att-value (world-object attribute &optional (value '*unbound*)
				     (field 'public))
  (cond ((not (equal value '*unbound*))
	 (if (atom value) (setf value (list value)))
	 (setf value (fix-world-object-references value))
	 (cond ((and (eq field 'public)
		     (assoc attribute (world-object-public world-object)))
		(unless (equal (rest (assoc attribute (world-object-public world-object)))
			       value)
		  (setf (rest (assoc attribute (world-object-public world-object))) value)
		  (pushnew world-object *changed-world-objects*))
		value)
	       ((and (eq field 'private)
		     (assoc attribute (world-object-private world-object)))
		(unless (equal (rest (assoc attribute (world-object-private world-object)))
			       value)
		  (setf (rest (assoc attribute (world-object-private world-object))) value)
		  (pushnew world-object *changed-world-objects*))
		value)
	       ((and (eq field 'public-init)
		     (assoc attribute (world-object-public-init world-object)))
		(unless (equal (rest (assoc attribute (world-object-public-init world-object)))
			       value)
		  (setf (rest (assoc attribute (world-object-public-init world-object))) value)
		  (pushnew world-object *changed-world-objects*))
		value)
	       ((and (eq field 'private-init)
		     (assoc attribute (world-object-private-init world-object)))
		(unless (equal (rest (assoc attribute (world-object-private-init world-object)))
			       value)
		  (setf (rest (assoc attribute (world-object-private-init world-object))) value)
		  (pushnew world-object *changed-world-objects*))
		value)
	       ((and (eq field 'output)
		     (assoc attribute (world-object-output world-object)))
		(unless (equal (rest (assoc attribute (world-object-output world-object)))
			       value)
		  (setf (rest (assoc attribute (world-object-output world-object))) value)
		  (pushnew world-object *changed-world-objects*))
		value)
	       ((equal field 'public)
		(pushnew (cons attribute value) (world-object-public world-object))
		(pushnew world-object *changed-world-objects*)
		value)
	       ((equal field 'private)
		(pushnew (cons attribute value) (world-object-private world-object))
		(pushnew world-object *changed-world-objects*)
		value)
	       ((equal field 'output)
		(pushnew (cons attribute value) (world-object-output world-object))
		(pushnew world-object *changed-world-objects*)
		value)
	       ((equal field 'public-init)
		(pushnew (cons attribute value) (world-object-public-init world-object))
		(pushnew world-object *changed-world-objects*)
		value)
	       ((equal field 'private-init)
		(pushnew (cons attribute value) (world-object-private-init world-object))
		(pushnew world-object *changed-world-objects*)
		value)
	       (T
		(display-error "~%; ERROR: ~a is not a valid field to set the attribute"
			       field)
		(display-error "~%;        ~a and value ~a to." attribute value)
		(display-error "~%;        Valid fields are PUBLIC, PRIVATE, and OUTPUT.~%"))))
	(t (cond ((assoc attribute (world-object-public world-object))
		  (refix-world-object-references
		   (rest (assoc attribute (world-object-public world-object)))))
		 ((assoc attribute (world-object-private world-object))
		  (if (eq attribute 'global-var-name)
		      (rest (assoc attribute (world-object-private world-object)))
		      (refix-world-object-references
		       (rest (assoc attribute (world-object-private world-object))))))
		 ((assoc attribute (world-object-output world-object))
		  (refix-world-object-references
		   (rest (assoc attribute (world-object-output world-object)))))))))



(defun fix-world-object-references (vlist &aux result)
  (dolist (v vlist result)
    (cond ((member v *world-objects*)
	   (push (second (assoc 'global-var-name (world-object-private v))) result))
	  (t (push v result)))))


(defun refix-world-object-references (vlist &aux result)
  (dolist (v vlist result)
    (cond ((sim-varnamep v)
	   (push (get-object-from-global-var-name v) result))
	  (t (push v result)))))


(defun get-object-from-global-var-name (name)
  (dolist (w *world-objects*)
    (if (eq name (second (assoc 'global-var-name (world-object-private w))))
	(return w))))


;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;;
;;;  FIELD-OF-ATTRIBUTE
;;;
;;;  This function returns which of the three fields (public, private,
;;;  output) an attribute is in of a world-object.  If the
;;;  attribute does not exist in the world-object, NIL is returned.
;;;
;;;  WORLD-OBJECT
;;;    This argument is a world-object as defined by the defobject SoarSIM
;;;    function.
;;;
;;;  ATTRIBUTE
;;;    This is an attribute of the world-object passed to this function.
;;;
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

(defun field-of-attribute (world-object attribute)
  (cond ((assoc attribute (world-object-public world-object)) 'public)
        ((assoc attribute (world-object-private world-object)) 'private)
        ((assoc attribute (world-object-output world-object)) 'output)))


;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;;
;;;  INTERNALIZE-WORLD-OBJECT
;;;
;;;  SBH 5/21/92: added io-tags for objects.
;;;
;;;  The purpose of this function is to internalize world-objects into
;;;  Soar's working memory.  Generally, it should be called only within
;;;  input-systems defined by SoarSIM's definput function.
;;;
;;;    Previously internalized world-object?
;;;    * YES
;;;      --> Put object-id on input-link if not already there
;;;      --> Has any attribute's values changed?
;;;          * YES
;;;            --> internalize changed attribute values
;;;            --> change the link-id
;;;            --> change the object's io-tag.
;;;          * NO
;;;            --> do nothing
;;;    * NO
;;;      --> create new object-id for world-object
;;;      --> add object-id to input-link
;;;      --> internalize all attribute values
;;;      --> change the link-id
;;;      --> change the object's io-tag.
;;;
;;;
;;;
;;;  WORLD-OBJECT
;;;    This argument is a world-object as defined by the defobject SoarSIM
;;;    function.
;;;
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;


(defun internalize-world-object (world-object)
  (let* ((input-link  (get-input-link 'input-link-owner 'input-link))
         (input-id (wme-value input-link))
         (object-id NIL)
         (world-object-attribute-list NIL)
         (world-object-attribute NIL)
         (world-object-attribute-value NIL))
    (add-object-to-input-link world-object)
    (setf object-id (car (att-value world-object 'object-id)))
    (cond ((internalized-p world-object)
	   (setf internal-attribute-value-list
		 (get-internal-attribute-value-list world-object))
	   (setf external-attribute-value-list (world-object-public world-object))
	   (dolist (external-attribute-value external-attribute-value-list)
	     (setf attribute      (first external-attribute-value))
	     (setf external-value (att-value world-object attribute))
	     (setf internal-value (rest (assoc attribute internal-attribute-value-list)))
	     (cond ((not (unordered-equal external-value internal-value))
		    (update-internal-object-attribute-value world-object attribute)
		    (change-link-id)
		    (change-object-tag world-object)))))
	  (T
	   (change-link-id)
	   (change-object-tag world-object)
	   (display-text "~%~%; (INP) Object ~a " object-id)
	   (display-text "is being internalized.  Attributes:")

	   (dolist (this-att-val (world-object-public world-object))
	     (setf world-object-attribute (first  this-att-val))
	     (setf world-object-attribute-value (rest this-att-val))
	     (multi-add-input 'input-link-owner 'object object-id
			world-object-attribute world-object-attribute-value)
	     (display-text "~%;         ~a: ~a"
			   world-object-attribute world-object-attribute-value))))))


;;; Helper that tests if two lists have the same elements.
(defun unordered-equal (l1 l2)
  (if (and (null (set-difference l1 l2))
	   (null (set-difference l2 l1)))
      t nil))



;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;; MULTI-ADD-INPUT, MULTI-DELETE-INPUT
;;;
;;; These functions add/delete an attr/value set to WM, where the value is a
;;; multiattribute.
;;;
;;; If a value in the value list is the name of a world object, we insert its
;;; soar object-id, if it has one yet.
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;


(defun multi-add-input 	(owner obj obj-id att val)
  (dolist (v val)
    (let ((vobj (get-object-from-global-var-name v)))
      (cond (vobj
	     ;; (format t "vobj-id ~a~~%~%" (att-value vobj 'object-id))
	     (cond ((att-value vobj 'object-id)
		    (add-input owner obj obj-id att (car (att-value vobj 'object-id))))
		   (t
		    (add-input owner obj obj-id att '*unknown*)
		    (pushnew (list vobj
				   (get-world-object-from-object-id obj-id)
				   att) *update-pointer-list*)
		    )))
	    (t (add-input owner obj obj-id att v))))))


(defun multi-delete-input (owner obj obj-id att val)
  (dolist (v val)
    (delete-input owner obj obj-id att v)))


;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;;
;;;  ADD-OBJECT-TO-INPUT-LINK
;;;
;;;  This function adds a world-object's id to the input-link.  This allows
;;;  Soar to know what external world objects it can perceive.
;;;
;;;  First, if the world-object passed to this function does not yet have
;;;  an id, one is created for it.  Also, this id is then put on the input-
;;;  link.  If, however, the world-object passed to this function already
;;;  has an id, a check is made to see if it is already on the input-link.
;;;  If the id is not yet on the input-link, it is put there.  If it is
;;;  already on the input-link, it is left there and nothing new is added
;;;  to the input-link.
;;;
;;;  The id of the world object is returned in all cases.
;;;
;;;  WORLD-OBJECT
;;;    This argument is a world-object as defined by the defobject SoarSIM
;;;    function.
;;;
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

(defun add-object-to-input-link (world-object)
  (let* ((input-link  (get-input-link 'input-link-owner 'input-link))
         (input-id    (wme-value input-link))
         (object-id   (car (att-value world-object 'object-id))))
    (cond ((null object-id)
           (setf object-id
		(wme-value (add-input 'input-link-owner 'input-link input-id 'object)))
           (att-value world-object 'object-id object-id 'private)
	   (update-pointers-to-object world-object))
          ((not (on-input-link-p world-object))
           (wme-value (add-input 'input-link-owner 'input-link input-id 'object object-id))
           (display-text "~%; (INP) A world-object with ID ~a "
                    (car (att-value world-object 'object-id)))
           (display-text "is now perceivable.~%")
           object-id))))



(defun update-pointers-to-object (obj)
  (cond ((assoc obj *update-pointer-list*)
	 (let* ((entry (assoc obj *update-pointer-list*))
		(aobj (second entry))
		(attr (third entry))
		(aobj-id (car (att-value aobj 'object-id)))
		(obj-id  (car (att-value obj 'object-id))))
	   ;; (format t "assoc. id ~a; this id: ~a~%~%" aobj-id obj-id)
	   (add-input 'input-link-owner 'object aobj-id attr obj-id)
	   (delete-input 'input-link-owner 'object aobj-id attr '*unknown*))
	 (setf *update-pointer-list*
	       (remove (assoc obj *update-pointer-list*) *update-pointer-list*))
	 (update-pointers-to-object obj))
	(t nil)))


;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;;
;;;  REMOVE-OBJECT-FROM-INPUT-LINK
;;;
;;;  This function removes a world-object's id from the input-link.  This
;;;  allows Soar to know that an external world object is no longer
;;;  perceivable.
;;;
;;;  If the world-object passed to this function has an id and is already
;;;  on the input-link, it gets removed from the input-link. The id of the
;;;  world object is returned.
;;;
;;;  WORLD-OBJECT
;;;    This argument is a world-object as defined by the defobject SoarSIM
;;;    function.
;;;
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

(defun remove-object-from-input-link (world-object)
  (let* ((input-link  (get-input-link 'input-link-owner 'input-link))
         (input-id    (wme-value input-link))
         (object-id   (car (att-value world-object 'object-id))))
    (when (and object-id (on-input-link-p world-object))
      (wme-value (delete-input 'input-link-owner 'input-link input-id 'object object-id))
      (display-text "~%; (INP) A world-object with ID ~a "
                    (car (att-value world-object 'object-id)))
      (display-text "can no longer be perceived.~%"))
    object-id))


;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;;
;;;  ON-INPUT-LINK-P
;;;
;;;  This function checks to see if a world-object is currently on the
;;;  input-link.  If the world-object passed to this function is on the
;;;  input-link, T is returned.  Otherwise, NIL is returned.
;;;
;;;  WORLD-OBJECT
;;;    This argument is a world-object as defined by the defobject SoarSIM
;;;    function.
;;;
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

(defun on-input-link-p (world-object)
  (let* ((object-id (car (att-value world-object 'object-id)))
         (input-link (get-input-link 'input-link-owner 'input-link))
         (input-id (wme-value input-link)))
    (if (match-input :owner 'input-link-owner :class 'input-link
                     :id input-id :attribute 'object :value object-id)
        T NIL)))


;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;;
;;;  INTERNALIZED-P
;;;
;;;  This function checks to see if a world-object has been internalized.
;;;  This can be true even if the world-object's id is no longer on the
;;;  input-link.  If the world-object passed to this function has been
;;;  internalized at any prior point, T is returned.  Otherwise, NIL is
;;;  returned.
;;;
;;;  WORLD-OBJECT
;;;    This argument is a world-object as defined by the defobject SoarSIM
;;;    function.
;;;
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

(defun internalized-p (world-object)
  (let ((object-id (car (att-value world-object 'object-id))))
    (if (and object-id
	       (match-input :owner 'input-link-owner :class 'object :id object-id))
        T NIL)))


;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;;
;;;  CHANGE-LINK-ID
;;;
;;;  This function changes the id of the link-id.  Its purpose is to signal
;;;  Soar that new information is on the input-link, either that a new
;;;  world-object has been internalized, an already internalized world-
;;;  object has had a new attribute-value pair added, and/or that an already
;;;  internalized world-object has just had the value changed of one or more
;;;  of its attributes.
;;;
;;;  If there already exists a link-id, it is removed.  In either case,
;;;  a new link-id is added to Soar's working memory.  This id is returned.
;;;
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

(defun change-link-id ()
  (let* ((input-link (get-input-link 'input-link-owner 'input-link))
         (input-id (wme-value input-link))
         (link-id (wme-value (first (match-input :owner     'input-link-owner
						   :class     'input-link
						   :id        input-id
						   :attribute 'link-id)))))
    (if link-id
        (delete-input 'input-link-owner 'input-link input-id 'link-id link-id))
    (wme-value (add-input 'input-link-owner 'input-link input-id 'link-id))))


;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;;
;;;  CHANGE-OBJECT-TAG
;;;
;;;  This function changes the io-tag of an object.  Its purpose is to signal
;;;  Soar that new information is on the object.
;;;
;;;  If there already exists a io-tag, it is removed.  In either case,
;;;  a new io-tag is added to Soar's working memory.  It is returned.
;;;
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

(defun change-object-tag (obj)
  (let* ((input-link (get-input-link 'input-link-owner 'input-link))
         (object-id (car (att-value obj 'object-id)))
         (io-tag (wme-value (first (match-input :owner     'input-link-owner
						:class     'object
						:id        object-id
						:attribute 'io-tag)))))
    (if io-tag
        (delete-input 'input-link-owner 'object object-id 'io-tag io-tag))
    (wme-value (add-input 'input-link-owner 'object object-id 'io-tag))))


;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;;
;;;  GET-INTERNAL-ATTRIBUTE-VALUE-LIST
;;;
;;;  This function gets Soar's internal the attribute-value list for a
;;;  specified world-object.  This list may vary from the external world-
;;;  object's attribute-value list since things can occur in an external
;;;  world that has not been perceived yet by Soar.
;;;
;;;  This function operates by first, retrieving all attribute-value pairs
;;;  from Soar's working memory.  It then iterates through each item in
;;;  this list and extracts the necessary attribute-value information to
;;;  put it in a convenient format.  This format is similar to the format
;;;  used when defining world-object attributes and associated values:
;;;
;;;    ((attribute-1 value-1) (attribute-2 value-2) ... )
;;;
;;;  A list of Soar's internal attribute-value pairs for a specified
;;;  world-object is returned in the above format.  If the world-object
;;;  has not yet been internalized, NIL is returned.
;;;
;;;  WORLD-OBJECT
;;;    This argument is a world-object as defined by the defobject SoarSIM
;;;    function.
;;;
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

(defun get-internal-attribute-value-list (world-object)
  (let* ((object-id (car (att-value world-object 'object-id)))
         (av-list (match-input :owner 'input-link-owner :class 'object :id object-id))
         (attval NIL)
         (internal-a-v-list NIL))
    (dolist (a-v av-list internal-a-v-list)
      (setf attval (list (wme-attribute a-v)
			 (wme-value a-v)))
      (setf internal-a-v-list (merge-in attval internal-a-v-list)))))

;;; Merge-in is a helping function that takes an att-value pair and adds it
;;; to the appropriate "entry" in the av-list.

(defun merge-in (attval av-list)
  (let* ((a (car attval))
	 (v (second attval))
	 (entry (full-copy (assoc a av-list))))
    (cond (entry
	   (setf (rest (assoc a av-list)) (cons v (rest entry)))
	   av-list)
	  (t (push attval av-list)))))



;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;;
;;;  UPDATE-INTERNAL-OBJECT-ATTRIBUTE-VALUE
;;;
;;;  This function updates Soar's internal representation of a specific
;;;  attribute of the specified world-object.  Its purpose is for Soar to
;;;  perceive a change of an external world-object to more closely match
;;;  the attributes and values of that object.
;;;
;;;  First, Soar's internal value of the specific attribute is retrieved.
;;;  Next, the specified world-object's attribute-value pair is removed
;;;  from Soar's working memory.  Next, an attribute-value pair for the
;;;  specified world-object is added to Soar's working memory with
;;;  the specific attribute as passed to this function, but with the
;;;  external value of this attribute as opposed to the recently removed
;;;  internal one.
;;;
;;;  An error occurs if the specified object does not have the specified
;;;  attribute and nothing is removed or added from Soar's working memory.
;;;
;;;  WORLD-OBJECT
;;;    This argument is a world-object as defined by the defobject SoarSIM
;;;    function.
;;;
;;;  ATTRIBUTE
;;;    This is an attribute of the world-object passed to this function.
;;;
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

(defun update-internal-object-attribute-value (obj attribute)
  (let* ((object-id (car (att-value obj 'object-id)))
         (sim-val (fix-world-object-references (att-value obj attribute)))
         (soar-vallist
          (get-internal-attribute-value-list obj))
         (soar-val (rest (assoc attribute soar-vallist))))
    (cond ((and (att-value obj 'object-id)
                (field-of-attribute obj attribute))
	   ;; (format t "id ~a ev ~a iv ~a~%" object-id sim-val soar-val)
           (multi-delete-input 'input-link-owner 'object object-id attribute soar-val)
           (multi-add-input  'input-link-owner 'object object-id attribute
			     sim-val)
	   (cond ((not (equal soar-val '(*unknown*)))
		  (display-text "~%; (INP) Internal object with ID ~a has public attribute"
				object-id)
		  (display-text "~%;       ~a changed from ~a to ~a. ~%"
				attribute soar-val
				(rest (assoc attribute
					     (get-internal-attribute-value-list obj)))))))
          ((att-value obj 'object-id)
           (display-error "~%; ERROR: Failed to update attribute-value of internal object")
           (display-error "~%;        because this world-object hasn't been internalized yet:")
           (display-error "~%;        ~a~%" obj))
          (T
           (display-error "~%; ERROR: Failed to update attribute-value of internal object")
           (display-error "~%;        because world-object with ID ~a doesn't have the attribute ~a.~%"
                    (att-value obj 'object-id) attribute)))))


;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;;
;;;  GET-WORLD-OBJECT-FROM-OBJECT-ID
;;;
;;;  Upon being internalized, a world-object gets an id that is auto-
;;;  matically added to its private field: (object-id id).  This
;;;  is the id that gets put on Soar's input-link and also the id of the
;;;  object class of the world-object when it gets internalized.  Its
;;;  purpose is to maintain the relationship between an external world-
;;;  object and its internal representation within Soar.
;;;
;;;  This function returns the world-object of a specified id that was
;;;  created when that world-object was first internalized.  It does
;;;  this by iterating through each world object in the *world-objects*
;;;  global list until a world-object with that id is found.  NIL is
;;;  returned if the specified id does not belong to any world-object.
;;;
;;;  A NIL would be returned if an id other than one that has been auto-
;;;  matically created by SoarSIM and attached to a world-object is passed
;;;  to this function.  System designers must make sure that only id's
;;;  created by SoarSIM should be passed to this function.
;;;
;;;  OBJECT-ID
;;;    This argument is the id of a world-object.  This id is created by
;;;    SoarSIM when the world-object is first internalized within Soar.
;;;
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

(defun get-world-object-from-object-id (object-id)
  (dolist (world-object *world-objects*)
    (if (equal (car (att-value world-object 'object-id)) object-id)
        (return world-object))))


;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;;
;;;  GET-ARGx
;;;
;;;  Each of the nine (9) get-arg functions retrieves an argument passed
;;;  by Soar along with an output-command.  For example, the SoarSIM
;;;  function get-arg1 will retrieve the value of the argument ^arg1
;;;  passed by Soar.  If one of the below functions is called when Soar
;;;  has not passed an associated argument, NIL is returned.
;;;
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

(defun get-arg1 ()
  (first (get-output-values 'arg1 *outputs*)))

(defun get-arg2 ()
  (first (get-output-values 'arg2 *outputs*)))

(defun get-arg3 ()
  (first (get-output-values 'arg3 *outputs*)))

(defun get-arg4 ()
  (first (get-output-values 'arg4 *outputs*)))

(defun get-arg5 ()
  (first (get-output-values 'arg5 *outputs*)))

(defun get-arg6 ()
  (first (get-output-values 'arg6 *outputs*)))

(defun get-arg7 ()
  (first (get-output-values 'arg7 *outputs*)))

(defun get-arg8 ()
  (first (get-output-values 'arg8 *outputs*)))

(defun get-arg9 ()
  (first (get-output-values 'arg9 *outputs*)))

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;;
;;;  DISPLAY-x
;;;
;;;  The display-text, display-debug, display-warnings, and display-error
;;;  functions display text to either help show the interactions between
;;;  world-objects, display useful SoarSIM debug information, display
;;;  SoarSIM  warnings, or display SoarSIM errors respectively.  They
;;;  are similar in format to the Common Lisp format statement except
;;;  that specified text will only be displayed when the functions
;;;  respective global variable is set to T (except display-error which
;;;  always displays error messages when called).
;;;
;;;  Here is the list of the four functions and their respective global
;;;  variables which dictate whether text passed to these functions will
;;;  get displayed or not.  The set-x functions can be used to set or
;;;  toggle the value of their respective global variables:
;;;
;;;    SoarSIM function     Global Variable     Setting function
;;;    ----------------     ---------------     ----------------------
;;;    display-text         *soarsim-verbose*              set-text [T | NIL]
;;;    display-debug        *soarsim-debug*             set-debug [T | NIL]
;;;    display-warnings     *soarsim-warnings*          set-warnings [T | NIL]
;;;    display-errors       n/a                 n/a
;;;
;;;  For example, (display-text "~%; My name is ~a.~%" 'ROB) will
;;;  display:
;;;
;;;  ; My name is ROB
;;;
;;;  if the *soarsim-verbose* global variable is set to T.  If the *soarsim-verbose* global
;;;  variable is set to NIL, the specified text will NOT be displayed.
;;;  Also, the set-text SoarSIM function can be used to set or toggle
;;;  the *soarsim-verbose* global variable.
;;;
;;;  TEXT
;;;    This is the text that should be displayed, formatted just as it
;;;    would for the Common Lisp format statement.
;;;
;;;  VARIABLES
;;;    As with the Common Lisp format statement, variables may be passed
;;;    to be displayed within the text.  Up to ten variables may be
;;;    passed to SoarSIM's display-text function.
;;;
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

(defun display-text (text &rest variables)
  (if *soarsim-verbose*
      (format T text (first variables) (second variables) (third variables)
                     (fourth variables) (fifth variables) (sixth variables)
                     (seventh variables) (eighth variables) (ninth variables)
                     (tenth variables))))

(defun display-debug (text &rest variables)
  (if *soarsim-debug*
      (format T text (first variables) (second variables) (third variables)
                     (fourth variables) (fifth variables) (sixth variables)
                     (seventh variables) (eighth variables) (ninth variables)
                     (tenth variables))))

(defun display-warnings (text &rest variables)
  (if *soarsim-warnings*
      (format T text (first variables) (second variables) (third variables)
                     (fourth variables) (fifth variables) (sixth variables)
                     (seventh variables) (eighth variables) (ninth variables)
                     (tenth variables))))

(defun display-errors (text &rest variables)
  (format T text (first variables) (second variables) (third variables)
                 (fourth variables) (fifth variables) (sixth variables)
                 (seventh variables) (eighth variables) (ninth variables)
                 (tenth variables)))

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;;
;;;  SOARSIM-SETTINGS
;;;
;;;  This function displays SoarSIM's environment settings.  This consists
;;;  of the three global variables: *soarsim-verbose*, *soarsim-debug*, *soarsim-warnings*.
;;;
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

(defun soarsim-settings ()
  (format T "~% VERBOSE is set to ~16a ... (soarsim-set-verbose [T | NIL])"
    *soarsim-verbose*)
  (format T "~% DEBUG is set to ~18a ... (soarsim-set-debug [T | NIL])"
    *soarsim-debug*)
  (format T "~% WARNINGS is set to ~15a ... (soarsim-set-warnings [T | NIL])"
    *soarsim-warnings*)
)

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;;
;;;  soarsim-SET-x
;;;
;;;  Each of SoarSIM's set-x functions sets its respective global variable.
;;;  For example, (set-text T) sets the *soarsim-verbose* global variable to T.  Now,
;;;  when SoarSIM's display-text function is called, its text will be
;;;  displayed.  (Any value other than NIL will be considered as if T was
;;;  passed as a setting.  Thus, the respective global variable will get
;;;  set to T and not whatever non-NIL setting was passed.)
;;;
;;;  These set-x functions can also toggle between T and NIL for their
;;;  respective global variables by not including a setting value.  For
;;;  example, (set-text) will set the *soarsim-verbose* global variable to T if it is
;;;  currently NIL, or NIL if it is currently T.
;;;
;;;  SETTING (optional)
;;;    This is either T or NIL to set the respective global variable to
;;;    that setting.  If no setting is passed, the respective global
;;;    variable toggles between T and NIL.
;;;
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

(defun soarsim-set-verbose (&optional (setting 'unbound))
  (if (equal setting 'unbound)
      (setf *soarsim-verbose* (not *soarsim-verbose*))
      (if setting
          (setf *soarsim-verbose* T)
          (setf *soarsim-verbose* NIL)))
  (format T "~% TEXT is set to ~a~%" (if *soarsim-verbose* 'DISPLAY 'DO-NOT-DISPLAY)))

(defun soarsim-set-debug (&optional (setting 'unbound))
  (if (equal setting 'unbound)
      (setf *soarsim-debug* (not *soarsim-debug*))
      (if setting
          (setf *soarsim-debug* T)
          (setf *soarsim-debug* NIL)))
  (format T "~% DEBUG is set to ~a~%" (if *soarsim-debug* 'DISPLAY 'DO-NOT-DISPLAY)))

(defun soarsim-set-warnings (&optional (setting 'unbound))
  (if (equal setting 'unbound)
      (setf *soarsim-warnings* (not *soarsim-warnings*))
      (if setting
          (setf *soarsim-warnings* T)
          (setf *soarsim-warnings* NIL)))
  (format T "~% WARNINGS is set to ~a~%" (if *soarsim-warnings* 'DISPLAY 'DO-NOT-DISPLAY)))


;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;;
;;;  INIT-SOARSIM
;;;
;;;
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

(defun init-soarsim ()
  (setf *changed-world-objects* *world-objects*)
  (setf *update-pointer-list* nil)

  ;; Re-initializes world-objects.
  (dolist (obj *world-objects*)
	(setf (world-object-private obj)(full-copy (world-object-private-init obj)))
	(setf (world-object-public obj)(full-copy (world-object-public-init obj))))
  )


(defun reset-soarsim ()
  (dolist (w *world-objects*)
    (eval `(setf ,(car (att-value w 'global-var-name)) nil)))
  (setf *world-objects* NIL)
  (setf *changed-world-objects* NIL)
  (setf *external-laws* NIL)
  (setf *input-systems* NIL)
  (setf *output-commands* NIL)
  (setf *update-pointer-list* nil)
  (setf *output-command-functions* NIL)
  (format t "~%; SoarSIM reset.~%")
  )



;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;; Helping fns.
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

(defun full-copy (s)
  (cond ((null s) nil)
	((atom s) s)
	((listp s)
	 (cons (full-copy (car s))(full-copy (cdr s))))))

    

(format T "~%~%SoarSIM v1.0, Environment Settings:~%")
(soarsim-settings)
(soarsyntax)


;;; This production kills off an output link once one appears.

(operator-applications '(soarsim*default*eliminate*output-link))

(sp soarsim*default*eliminate*output-link
  (goal <g> ^object nil ^state <s>)
  (state <s> ^output-link <o>)
  -->
  (state <s> ^output-link <o> -))


