
/****************************************************************************
 *
 * MODULE:  wmemory.c
 *
 ****************************************************************************
 *
 * Abstract:
 *    Routines to handle working memory.
 *
 ****************************************************************************
 *
 * CParaOPS5
 * Change Log:
 *    10 Aug 89 V4.0  Dirk Kalp
 *                    Merged with ParaOPS5 4.3.
 *    12 May 89 V2.0  Dirk Kalp
 *                    Create CParaOPS5 from ParaOPS5 4.2.
 *
 ****************************************************************************
 *
 * ParaOPS5
 * Change Log:
 *    25 Jun 89 V4.3  Dirk Kalp
 *                    Change global routine names to have "ops_" prefix. This
 *                    is to prevent conflicts with system and user defined
 *                    names at link time.
 *    14 Feb 89 V4.1  Dirk Kalp
 *                    Added support for standard OPS5 printing of wmes as
 *                    attribute followed by value. Put hooks into compiler, cops5,
 *                    in ../lhs/ to add attribute id information for literalized
 *                    classnames to the end of the compiler's assemler output file
 *                    and use that information here to print the wme.
 *                    Added WM_Tail to keep a pointer to the end of WM so we can
 *                    use it to get WM printed in increasing order of timetag
 *                    (i.e., reverse of way it's printed out now for top-level
 *                    "wm" and "ppwm" cmds).
 *    24 Oct 88 V4.0  Dirk Kalp
 *                    Release of ParaOPS5 Version 4.0.
 *    23 Oct 88 V3.3  Dirk Kalp
 *                    Modified form of printout in "ops_write_wme" routine.
 *    24 Sep 88 V3.2  Dirk Kalp
 *                    Added routine "ops_do_remove" to handle top level user
 *                    interface cmd "remove".
 *    21 Sep 88 V3.1  Milind Tambe
 *                    Added routines "ops_print_wm_timetag" and "ops_print_wm_class"
 *                    to handle top level user interface cmds "wm" and "ppwm".
 *    13 Aug 88 V3.0  Dirk Kalp
 *                    Use fprintf instead of printf. Added routine
 *                    "ops_reinit_wmemory" to handle system reinitialization.
 *    25 May 88 V2.0  Dirk Kalp
 *                    Updated to consolidate Vax and Encore versions.
 *    31 Jul 86 V1.0  Dirk Kalp
 *
 * Copyright (c) 1986, 1987, 1988, 1989 Carnegie-Mellon University
 * All rights reserved.  The CMU software License Agreement
 * specifies the terms and conditions for use and redistribution.
 *
 ****************************************************************************/


#include "global.h"



/* Exported routines:
 *    void    ops_init_wmemory()
 *    void    ops_reinit_wmemory()
 *    wmeptr  ops_newwme()
 *    void    ops_addtarget(pWme)
 *    void    ops_deltarget(pWme)
 *    boolean ops_inwm(pWme)
 *    void    ops_goto_hell()
 *    void    ops_do_remove(ttag)
 *    int     ops_get_timetag(pWme)
 *    void    ops_print_wm_timetag(ttag)
 *    void    ops_print_wm_pattern(target)
 *    void    ops_print_wm_class(class_name)
 *    void    ops_write_atom(fp_x, atom)
 *    void    ops_write_wme(fp_x, pWme)
 *    wmeptr  ops_get_WM_head()
 *    wmeptr  ops_get_WM_tail()
 *
 */



/* Imported Routines:
 *    From utility.c:
 *       ops_malloc
 *
 *    From gensymbol.c:
 *       ops_pname
 *       ops_symname_lookup
 *       ops_symid_lookup
 *
 *    From match.c:
 *       ops_match
 */



/* External Routines:
 *    These routines from other modules return values other than the
 *    standard integer and so their return types are declared here
 *    for routines in this module that call them.
 */
extern string ops_pname();            /* Imported from gensymbol.c. */
extern symptr ops_symname_lookup();   /* Imported from gensymbol.c. */
extern symptr ops_symid_lookup();     /* Imported from gensymbol.c. */
extern char   *ops_malloc();          /* Imported from utility.c. */
extern void   ops_fatal();            /* Imported from utility.c. */
extern void   ops_warn();             /* Imported from utility.c. */
extern void   ops_match();            /* Imported from match.c. */


/* Forward Declarations:
 *    These routines return values other than the standard integer and
 *    their return types are given here for other routines in this module
 *    that call them before they are defined.
 */
void  ops_goto_hell();
void  ops_write_atom();
void  ops_write_wme();


   

static wmeptr      WorkingMemory;   /* Points to working memory. */
static wmeptr      WM_Tail;         /* Points to end of WorkingMemory. */
static wmeptr      WmePurgatory;    /* Deleted wmes go here temporarily. */
static wmeptr      WmeFreeList;     /* Free list for wme record structures. */
static int         NextTimeTag;     /* Time tag to assigned to next wme. */



#define PRINTALL	 -1     /* Print all wm */

extern boolean  RefArray[WMESIZE];  /* Used for "ppwm" by rhsrtn.c. */
extern boolean  StandardWmePrintFormat;   /* Defined in rhsrtn.c to select wme print format. */





void
ops_init_wmemory()
/*---------------------------------------------------------------------------
 *
 * Abstract:
 *    Initialize variables used to maintain working memory. Called once
 *    when the system starts up.
 *
 * Parameters:
 *    None.
 *
 * Environment:
 *    Nothing special.
 *
 * Calls:
 *    No one.
 *
 * Called by:
 *    "ops_rt_init" in "rhsrtn.c".
 *
 *-------------------------------------------------------------------------*/
{
   WorkingMemory = NULL;
   WM_Tail       = NULL;
   WmePurgatory  = NULL;
   WmeFreeList   = NULL;
   NextTimeTag   = MINTIMETAG;
}



void
ops_reinit_wmemory()
/*---------------------------------------------------------------------------
 *
 * Abstract:
 *    Initialize working memory in response to a system reinitialization.
 *
 * Parameters:
 *    None.
 *
 * Environment:
 *    Nothing special.
 *
 * Calls:
 *    No one.
 *
 * Called by:
 *    "ops_reinit" in "rhsrtn.c".
 *
 *-------------------------------------------------------------------------*/
{
   wmeptr ptr, lastptr;
   int i;

   ops_goto_hell();         /* Put WmePurgatory onto the WmeFreeList. */

   if (WorkingMemory)
     {
      ptr = WorkingMemory;
      while (ptr)  { lastptr = ptr; ptr = ptr->flink; }
      lastptr->flink = WmeFreeList;
      WmeFreeList = WorkingMemory;
      WorkingMemory = NULL;
     }

   WM_Tail       = NULL;

   NextTimeTag   = MINTIMETAG;
}




wmeptr
ops_newwme()
/*---------------------------------------------------------------------------
 *
 * Abstract:
 *    Allocate a new record to hold a working memory element.
 *
 * Parameters:
 *    None.
 *
 * Environment:
 *    Nothing special.
 *
 * Returns:
 *    A pointer to the wme.
 *
 * Calls:
 *    "ops_malloc" in "utility.c".
 *
 * Called by:
 *    "ops_bmake" and "ops_bmodify" in "rhsrtn.c".
 *
 *-------------------------------------------------------------------------*/
{
   wmeptr new;
   
   if (WmeFreeList)
     {
      new = WmeFreeList;
      WmeFreeList = new->flink;
     }
   else
      new = (wmerec *) ops_malloc(sizeof(wmerec));

   /* Init just the time tag and the links. */
   new->wme[WMETIMETAG] = NULLTIMETAG;
   new->wme[WMEDELNDX]  = DELWM;
   new->flink           = NULL;
   new->blink           = NULL;
     
   return(new);
}




void
ops_addtarget(pWme)
   wmeptr pWme;
/*---------------------------------------------------------------------------
 *
 * Abstract:
 *    Add an element to working memory. First set the wme's time tag and
 *    then invoke the Rete match.
 *
 * Parameters:
 *    pWme - a pointer to the new wme.
 *
 * Environment:
 *    Nothing special.
 *
 * Calls:
 *    "ops_match" in "match.c".
 *
 * Called by:
 *    "ops_emake" and "ops_emodify" in "rhsrtn.c".
 *
 *-------------------------------------------------------------------------*/
{
   /* Set the time tag for the new wme.
    */
   pWme->wme[WMETIMETAG] = NextTimeTag++;
   pWme->wme[WMEDELNDX]  = INWM;
   
   /* Add it to working memory.
    */
   pWme->flink = WorkingMemory;
   pWme->blink = NULL;
   if (WorkingMemory != NULL)  WorkingMemory->blink = pWme;
   WorkingMemory = pWme;
   if (WM_Tail == NULL)  WM_Tail = pWme;   /* 2/14/89: So we can print WM in reverse. */
   
   /* Perform the match.
    */
   ops_match(pWme, DIRIN);
}




void
ops_deltarget(pWme)
   wmeptr pWme;
/*---------------------------------------------------------------------------
 *
 * Abstract:
 *    Remove an element from working memory. First invoke the Rete match
 *    and then remove the wme from working memory. We actually must not
 *    delete the storage for the wme until all actions in the current
 *    rule's RHS are performed since later actions in the RHS may refer
 *    to this wme.
 *
 * Parameters:
 *    pWme - a pointer to the wme.
 *
 * Environment:
 *    Nothing special.
 *
 * Calls:
 *    "ops_match" in "match.c".
 *
 * Called by:
 *    "ops_remove" and "ops_bmodify" in "rhsrtn.c".
 *
 *-------------------------------------------------------------------------*/
{
   wmeptr pred;
   wmeptr succ;
   
   /* Perform the match.
    */
   ops_match(pWme, DIROUT);
   
   /* Remove it from working memory.
    */
   pred = pWme->blink;
   succ = pWme->flink;
   if (pred)
      pred->flink = succ;
   else
      WorkingMemory = succ;
   if (succ)  succ->blink = pred;
   if (WM_Tail == pWme)  WM_Tail = pred;   /* 2/14/89: So we can print WM in reverse. */
   
   /* But keep the wme around yet for a while.
    */
   pWme->flink = WmePurgatory;   /* The flink chains the list together. */
   if (WmePurgatory == NULL)
      pWme->blink = pWme;        /* The blink always points to the list end. */
   else
      pWme->blink = WmePurgatory->blink;
   WmePurgatory = pWme;
   pWme->wme[WMEDELNDX]  = DELWM;    /* Marks it as removed. */
}



boolean
ops_inwm(pWme)
   wmeptr pWme;
{
   if (pWme->wme[WMEDELNDX] == INWM)
      return(TRUE);
   else
      return(FALSE);
}



void
ops_goto_hell()
/*---------------------------------------------------------------------------
 *
 * Abstract:
 *    Move all wme's in purgatory to hell. Called when the RHS actions of
 *    the current rule are completed.
 *
 * Parameters:
 *    None.
 *
 * Environment:
 *    Nothing special.
 *
 * Calls:
 *    No one.
 *
 * Called by:
 *    ops_"fire" in "rhsrtn.c".
 *
 *-------------------------------------------------------------------------*/
{
   if (WmePurgatory)
     {
      WmePurgatory->blink->flink = WmeFreeList;   /* axg: 8/3/86: switched order
      						   * of blink and flink.
						   */
      WmeFreeList = WmePurgatory;
      WmePurgatory = NULL;
     }
}

void
ops_do_remove(ttag)
   int ttag;
/*---------------------------------------------------------------------------
 *
 * Abstract:
 *    Remove an element(s) from working memory. First see if the timetag is
 *    valid and if the wme is still present in working memory. Next invoke
 *    the Rete match and then remove the wme from working memory.
 *
 * Parameters:
 *    ttag - the timetag of the wme to remove, NULLTIMETAG implies remove
 *           all wmes.
 *
 * Environment:
 *    Interacting with user at top level cmd interface. Caller has already
 *    checked that ttag >= NULLTIMETAG.
 *
 * Calls:
 *    "ops_match" in "match.c".
 *
 * Called by:
 *    "ops_main" in "rhsrtn.c".
 *
 *-------------------------------------------------------------------------*/
{
   wmeptr pred;
   wmeptr succ;
   wmeptr pWme;
   wmeptr last;

   pWme = WorkingMemory;
   
   if (ttag == NULLTIMETAG)
     {
      /* Remove all of working memory.
       */
      while (pWme)
        {
	 ops_match(pWme, DIROUT);
         pWme->wme[WMEDELNDX] = DELWM;    /* Marks it as removed. */
         last = pWme;
	 pWme = pWme->flink;
	}
      if (WorkingMemory)
        {
	 last->flink = WmeFreeList;
	 WmeFreeList = WorkingMemory;
	 WorkingMemory = NULL;
	}
      WM_Tail = NULL;
     }
   else
     {
      /* Remove the specified wme.
       */
      if ((ttag >= NextTimeTag) || (WorkingMemory == NULL))
         fprintf(fp_err, "** WME with timetag = %d not found.\n", ttag);
      else
        {
	 while (pWme)
	    if (pWme->wme[WMETIMETAG] == ttag)  break;  else  pWme = pWme->flink;
	 if (pWme)
	   {
            /* Perform the match.
             */
            ops_match(pWme, DIROUT);
            
            /* Remove it from working memory.
             */
            pred = pWme->blink;
            succ = pWme->flink;
            if (pred)
               pred->flink = succ;
            else
               WorkingMemory = succ;
            if (succ)  succ->blink = pred;
            if (WM_Tail == pWme)  WM_Tail = pred;   /* 2/14/89: So we can print WM in reverse. */
            
            /* Put it on the free list.
             */
            pWme->flink = WmeFreeList;
	    WmeFreeList = pWme;
            pWme->wme[WMEDELNDX]  = DELWM;    /* Marks it as removed. */
	   }
	 else
            fprintf(fp_err, "** WME with timetag = %d not found.\n", ttag);
	}
     }
}





int
ops_get_timetag(pWme)
   wmeptr pWme;
/*---------------------------------------------------------------------------
 *
 * Abstract:
 *    Extract the time tag from a working memory element.
 *
 * Parameters:
 *    pWme - pointer to a wme; it masy be NULL.
 *
 * Environment:
 *    Nothing special.
 *
 * Returns:
 *    The time tag or an invalid time tag if the parameter is empty.
 *
 * Calls:
 *    No one.
 *
 * Called by:
 *    "ops_modifycs" in "conres.c".
 *
 *-------------------------------------------------------------------------*/
{
   if (pWme)
      return(pWme->wme[WMETIMETAG]);
   else
      return(NULLTIMETAG);
}


 


/* Print working memory given the timetag   *
 * If timetag = -1 then print all the wmes  *
 * else search the wme list and print the
 * appropriate wme
 */

void
ops_print_wm_timetag(ttag)
   int ttag;
{
   wmeptr pWme;

   pWme = WM_Tail;   /* 2/14/89: Changed from WorkingMemory so we can print WM in reverse. */

   if (ttag == PRINTALL)
     {
      if (pWme)
         dumpwm(fp_out);
      else
         fprintf(fp_out, "** Working Memory is empty.\n");
     }
   else
     {
      if (ttag >= NextTimeTag)
         fprintf(fp_out, "** WME with timetag = %d not found.\n", ttag);
      else
        {
         while (pWme)
	    if (pWme->wme[WMETIMETAG] == ttag)
	       break;
	    else
	       pWme = pWme->blink;   /* 2/14/89: Changed from flink so we can print WM in reverse. */
         if (pWme)
            ops_write_wme(fp_out, pWme);
         else
            fprintf(fp_out, "** WME with timetag = %d not found.\n", ttag);
	}
     }
}



/* Implements "ppwm" top level cmd.
 */

void
ops_print_wm_pattern(target)  /* Print all WMEs that match target wme pattern. */
   wmeptr target;         /* target == Null -> print all of WM. */
{
   wmeptr  pWme;
   boolean found_any;
   boolean ItMatches;
   int     i, len;

   pWme = WM_Tail;   /* 2/14/89: Changed from WorkingMemory so we can print WM in reverse. */

   if (pWme == NULL)
     {
      fprintf(fp_out, "** Working Memory is empty.\n");
      return;
     }

   if (target == NULL)
     {
      dumpwm(fp_out);
      return;
     }

   len = target->wme[WMELENGTH];
   found_any = FALSE;
   while (pWme)
     {
      ItMatches = TRUE; /* Assume at first it will match. */
      for (i = 1; i <= len; i++)
        {
	 if (RefArray[i] == TRUE)    /* Marks which fields must compare. */
	   {
	    if (target->wme[i] != pWme->wme[i])  { ItMatches = FALSE; break;}
           }
        }
      if (ItMatches)
        {
	 found_any = TRUE;
	 ops_write_wme(fp_out, pWme);
	}
      pWme = pWme->blink;   /* 2/14/89: Changed from flink so we can print WM in reverse. */
     }     

   if (!found_any)  fprintf(fp_out, "** No WMEs found to match the pattern given.\n");
}



/* Restricted version of "ppwm". Takes only class name as arg.
 */

void
ops_print_wm_class(class_name)
   string class_name;   /* Null class_name -> print all of WM. */
{
   wmeptr pWme;
   symptr psym;
   OpsVal val;
   boolean found;

   pWme = WM_Tail;   /* 2/14/89: Changed from WorkingMemory so we can print WM in reverse. */

   if (pWme == NULL)
     {
      fprintf(fp_out, "** Working Memory is empty.\n");
      return;
     }

   if (class_name == NULL)
         dumpwm(fp_out);
   else
     {
      psym = ops_symname_lookup(class_name);
      if (psym == NULL)
        {
         fprintf(fp_out, "** No WMEs found with class name %s\n", class_name);
	 return;
	}
      val = sym2val(psym->SymId);
      found = FALSE;
      while (pWme)
        {
         if (pWme->wme[WMEMINNDX] == val)
	   {
	    found = TRUE;
            ops_write_wme(fp_out, pWme);
	   }
	 pWme = pWme->blink;   /* 2/14/89: Changed from flink so we can print WM in reverse. */
        }
      if (!found)  fprintf(fp_out, "** No WMEs found with class name %s\n", class_name);
     }
}



static
dumpwm(fp_x)
   FILE *fp_x;
{
   wmeptr pWme;
   
   pWme = WM_Tail;   /* 2/14/89: Changed from WorkingMemory so we can print WM in reverse. */
   while (pWme)
     {
      ops_write_wme(fp_x, pWme);
      pWme = pWme->blink;   /* 2/14/89: Changed from flink so we can print WM in reverse. */
     }
}



void
ops_write_wme(fp_x, pWme)
   FILE   *fp_x;
   wmeptr pWme;
{
   int i;

   if (StandardWmePrintFormat)
     {
      std_write_wme(fp_x, pWme);
     }
   else
     {
      fprintf(fp_x, "WmeTimeTag = %d  WmeLength = %d  WME: ", pWme->wme[WMETIMETAG],
             pWme->wme[WMELENGTH]);
   
      for (i = WMEMINNDX; i <= pWme->wme[WMELENGTH]; i++)
         ops_write_atom(fp_x, pWme->wme[i]);

      fprintf(fp_x, "\n");
     }
}


void
ops_write_atom(fp_x, atom)
   FILE   *fp_x;
   OpsVal atom;
{
   if (symbolp(atom))
      fprintf(fp_x, "%s ", ops_pname(atom));
   else
      fprintf(fp_x, "%d ", val2int(atom));
}



static
std_write_atom(fp_x, atom)
   FILE   *fp_x;
   OpsVal atom;
{
   if (symbolp(atom))
      fprintf(fp_x, "%s", ops_pname(atom));
   else
      fprintf(fp_x, "%d", val2int(atom));
}



static
std_write_vector(fp_x, pWme, beg)
   FILE   *fp_x;
   wmeptr pWme;
   int    beg;
{
   int i, end;

   end = pWme->wme[WMELENGTH];
   for (i = beg; i <= end; i++)
     {
      std_write_atom(fp_x, pWme->wme[i]);
      if (i != end)  fprintf(fp_x, "  ");
     }
}



static
std_write_wme(fp_x, pWme)
   FILE   *fp_x;
   wmeptr pWme;
/*---------------------------------------------------------------------------
 *
 * Abstract:
 *    This routine prints a wme in the standard OPS5 format (or close to it
 *    based on output from Lisp OPS5 program). The basic format is:
 *         WmeTimetag:  (classname  ^att val  ^att val  ......  ^att val)
 *    Wme fields that are nil are not printed unless they occur in a vector
 *    attribute or unless the classname field has a value which is not a
 *    declared name (from a  Literalize). In this latter case, the whole wme
 *    is just printed as a vector. The vector is simply the list of values
 *    in the consecutive wme fields. For the basic format, the (att, val)
 *    pairs are printed out in the order in which they occur in the wme
 *    data strucure (which is just an array of 1..127 fields) as determined
 *    by the bindings assigned by the compiler. They are not displayed in the
 *    order that they appear in the Literalize declaration for the class.
 *    A field in a wme that has a non-nil value but is not bound to an
 *    attribute of its class is printed in the form "^n val" where "n" is the
 *    array index of the field. However, the "^n" is not printed if the
 *    preceding field (i.e., n-1) of the wme was printed.
 *
 * Parameters:
 *    fp_x - file to write to.
 *    pWme - pointer to a wme.
 *
 * Environment:
 *    Nothing special.
 *
 * Returns:
 *    Nothing
 *
` * Calls:
 *    "std_write_vector" and "std_write_atom" in this module.
 *    "ops_symid_lookup" in "gensymbol.c".
 *
 * Called by:
 *    "ops_write_wme" in this module.
 *
 *-------------------------------------------------------------------------*/
{
   int    len, i, last_idx, att_idx, *intptr;
   symptr psym;
   OpsVal atom;

   fprintf(fp_x, "%d:  (", pWme->wme[WMETIMETAG]);

   atom = pWme->wme[WMEMINNDX];   /* the classname field */
   if (!symbolp(atom))
     {  /* Not a classname so just print wme in vector form (including nil slots). */
      std_write_vector(fp_x, pWme, WMEMINNDX);
     }
   else
     {
      psym = ops_symid_lookup(val2sym(atom));
      intptr = psym->AttPtrList;
      if (intptr == 0)
        {  /* Not a declared classname. */
         std_write_vector(fp_x, pWme, WMEMINNDX);
	}
      else
        {
         if (*intptr == 0)
	   {  /* Classname has no declared attributes. */
            std_write_vector(fp_x, pWme, WMEMINNDX);
	   }
	 else
	   {  /* Declared classname has at least 1 declared attribute. */
	    len = pWme->wme[WMELENGTH];
	    fprintf(fp_x, "%s", psym->SymName);   /* classname */
	    psym = ops_symid_lookup(val2sym(*(int *)(*intptr)));   /* 1st attribute */
	    att_idx = psym->OpsBind;
	    last_idx = WMEMINNDX;
	    i = last_idx + 1;
	    while (i <= len)
	      {
 	       atom = pWme->wme[i];
               if (i == att_idx)
	         {
                  if (psym->is_vecatt)
		    {
		     /* print ^vatt val ... val */
		     fprintf(fp_x, "  ^%s ", psym->SymName);
		     std_write_vector(fp_x, pWme, i);
		     break;
		    }
		  else if (atom != symnil)
		    {
		     /* print ^att val */
		     fprintf(fp_x, "  ^%s ", psym->SymName);
		     std_write_atom(fp_x, atom);
		     last_idx = i;
		    }
		  else  /* atom == symnil */
		     /* null stmt */;   /* Don't print nil atoms unless part of vector. */
		  if (*++intptr)
		    {
          	     psym = ops_symid_lookup(val2sym(*(int *)(*intptr)));   /* 1st attribute */
	             att_idx = psym->OpsBind;
		    }
		 }
	       else  /* i != att_idx */
	         {
		  if (i == last_idx + 1)
		    {
		     if (atom != symnil)
		       {
		        /* print val */
		        fprintf(fp_x, "  ");
		        std_write_atom(fp_x, atom);
			last_idx = i;
		       }
		    }
		  else
		    {
		     if (atom != symnil)
		       {
		        /* print ^n val */
		        fprintf(fp_x, "  ^%d ", i);
		        std_write_atom(fp_x, atom);
			last_idx = i;
		       }
		    }
		 }

               i++;   /* advance to next field in wme */
	      } /* endwhile */

	   }
	}
     }

   fprintf(fp_x, ")\n");
}


/*  These 2 routines are for the SPAM guys (who wanted to get at the
 *  data structure from the user-defined routines) but anyone else is
 *  free to use them.
 */
wmeptr
ops_get_WM_head()
{
   return(WorkingMemory);
}

wmeptr
ops_get_WM_tail()
{
   return(WM_Tail);
}
