
/****************************************************************************
 *
 * MODULE:  rhsrtn.c
 *
 ****************************************************************************
 *
 * Abstract:
 *    This module provides the entry point, the "main" routine, for the
 *    PSM control control process. It contains routines to initialize the
 *    system, fork match processes, and perform the OPS5 recognize-act cycle.
 *    It provides the top level OPS5 user interface and routines to perform
 *    RHS actions.
 *
 ****************************************************************************
 *
 * CParaOPS5
 * Change Log:
 *    29 Sep 89 V5.3  Dirk Kalp
 *                    Fixed synchonization bug in "ops_PopTaskQueueN" in module
 *                    "matchasm.c". Added some documentation to the code.
 *                    Replaced NON_SHARED_MEMORY_VERSION with UNIPROC_VERSION
 *                    for conditional compilation option.
 *                    Added ParaOPS5 4.4 bug fix.
 *                    Update version to "CParaOPS5 Version 5.3 P4.4" for release.
 *    30 Aug 89 V5.1  Anurag Acharya
 *                    Fixed external routine bug in compiler in ../lhs.
 *    16 Aug 89 V5.0  Anurag Acharya 
 *                    Integrated the uniprocessor version
 *    10 Aug 89 V4.0  Dirk Kalp
 *                    Merged with ParaOPS5 4.3.
 *    15 May 89 V3.0  Dirk Kalp
 *                    Use new lock macros from psm_locks.h. Update version
 *                    number to 3.0 P4.2.
 *    12 May 89 V2.0  Dirk Kalp
 *                    Create CParaOPS5 from ParaOPS5 4.2.
 *
 ****************************************************************************
 *
 * ParaOPS5
 * Change Log:
 *    29 Sep 89 V4.4  Dirk Kalp
 *                    Fixed bug related to "ops_user_init". The "uargv" array
 *                    was declared size 10 but should have been 100.
 *    25 Jun 89 V4.3  Dirk Kalp
 *                    Change global routine names to have "ops_" prefix. This
 *                    is to prevent conflicts with system and user defined
 *                    names at link time.
 *                    Allow user to specify the size for the symbol table
 *                    with -y cmd line switch.
 *                    Fixed to always call "ops_user_init" and "ops_user_exit"
 *                    and limit number of args to 100.
 *    19 Apr 89 V4.2  Dirk Kalp
 *                    Fixed bug in "pbreak" cmd. Fixed bugs in "ops_accept" and
 *                    "ops_acceptline" RHS functions. Added call to "ops_user_exit"
 *                    (in module "user.c") routine from "ops_main". "Ops_user_exit"
 *                    is an empty routine that the user can fill in with his
 *                    own code to perform clean-up activities before the
 *                    production system program exits.
 *    14 Feb 89 V4.1  Dirk Kalp
 *                    Added support for standard OPS5 printing of wmes as
 *                    attribute followed by value. Put hooks into compiler, cops5,
 *                    in ../lhs/ to add attribute id information for literalized
 *                    classnames to the end of the compiler's assemler output file.
 *                    Added switch, -z, to select the mode for printing wmes in either
 *                    the previous format (as a vector of values including the
 *                    nil fields) or the new standard format. The previous format
 *                    is useful in some debugging situations.
 *                    Added top-level cmds:
 *                         switches   - show available cmd line switches
 *                         wmefmt     - select print format for wmes
 *                         litbind    - shows binding for an attribute
 *                         classbind  - shows bindings for all attributes of a class
 *
 *                    Wilson Harvey and Dirk Kalp
 *                    Added support for switch to signal user-specified cmdline
 *                    args. The switch "--" causes the rest of the cmdline to be
 *                    passed to the routine "ops_user_init" which can then interpret
 *                    the rest of the cmdline as user specific switches or args.
 *                    The "ops_user_init" routine in module "user.c" is an empty
 *                    procedure which the user can fill in with his own code to
 *                    process the rest of the the cmd line. 
 *    24 Oct 88 V4.0  Dirk Kalp
 *                    Release of ParaOPS5 Version 4.0.
 *    23 Oct 88 V3.6  Dirk Kalp
 *                    Added use of Num_WM_Changes and Num_RHS_Actions counters.
 *                    Added routine "ops_run_stats" to print some more statistics.
 *                    Modified format of watch output and of "matches" output.
 *                    Fixed up "cs" cmd to also print the dominant instantiation.
 *                    Added routine "special_getstring" to handle parsing of
 *                    user input from top level cmd interface. This routine
 *                    is used for cmds like "make" which require an rhs
 *                    pattern or "pbreak" which require a production name.
 *                    We have to handle quoted strings for both kinds and the
 *                    "^" operator for cmds like "make". Some day when we have
 *                    time, we should just write lex and yacc specs to handle
 *                    parsing the entire top level cmd interface.
 *     7 Oct 88 V3.5  Dirk Kalp
 *                    Removed call to "ops_wakeup_matchers" in "mra_cycle" routine
 *                    and moved it to "match" routine in "match.c" - this is
 *                    the correct place for it since we want to wake them up
 *                    only after the first change to working memory has been
 *                    made on a given cycle. Fixed a synchronization bug between
 *                    control process and matcher processes by modifying the
 *                    "wait_for_rete" routine. Control process must leave its
 *                    bit (bit0) in BitVector turned on until it's sure that
 *                    the matchers have noticed all the tasks queued by the
 *                    control process. Added cmd line switches:
 *                         -s to set conflict resolution strategy
 *                         -h to print help msg
 *                         -? to print help msg
 *                         -v to print ParaOPS5 version number
 *     2 Oct 88 V3.4  Dirk Kalp
 *                    Added user interface cmds: pbreak, call, openfile, closefile
 *                    default, ask, version. Made "ppwm" fully functional. Added
 *                    "dollar_ifile" and "dollar_ofile".
 *     1 Oct 88 V3.3  Dirk Kalp
 *                    Added support for OPS5 top level user interface cmds
 *                    "matches" and "call". Put hooks into compiler, cops5,
 *                    in ../lhs/ to add nodeid information for productions
 *                    to the end of the compiler's assemler output file.
 *                    Also added info for external function names.
 *    21 Sep 88 V3.2  Dirk Kalp
 *                    Change name of user interface cmd from "quit" to "exit"
 *                    for OPS5 standard. Added cmd line switches:
 *                         -m  for size of shared memory
 *                         -d  print debugging info
 *                         -i  to specify input file of makes on cmd line
 *                         -f  to specify cmd line switches in a file
 *                         -a  ask user after this many firings to continue
 *                    Added user interface cmds for top "make" and "remove".
 *                    Make sure match time is not accumulated for top level
 *                    make and remove as well as for loading WM from a file.
 *                    Added user interface cmd "resolve" to show the dominant
 *                    instantiation in the conflict set.
 *    20 Sep 88 V3.1  Milind Tambe
 *                    Added "ppwm" and "wm" user interface cmds.
 *    13 Aug 88 V3.0  Dirk Kalp
 *                    Use fprintf instead of printf. Change max number of CEs
 *                    from 64 to 256. Add routines "wait_for_rete" and
 *                    "ops_wakeup_matchers" and modify "mra_cycle" to use them.
 *                    Added "ops_reinit" routine to handle reinitialization
 *                    of system.
 *     9 Aug 88 V2.1  Dirk Kalp
 *                    Added routine "ops_ucall" for OPS5 user defined functions.
 *                    Implemented standard OPS5 routines needed in order to
 *                    permit user defined functions and call actions to
 *                    communicate with the OPS5 interpreter.
 *    25 May 88 V2.0  Dirk Kalp
 *                    Updated to consolidate Vax and Encore versions.
 *    17 Sep 86       Dirk Kalp
 *    16 Sep 86       Dirk Kalp
 *    10 Sep 86       Dirk Kalp
 *    28 Aug 86       Dirk Kalp
 *    22 Aug 86       Anoop Gupta
 *    22 Jul 86 V1.0  Lanny Forgy
 *
 * Copyright (c) 1986, 1987, 1988, 1989 Carnegie-Mellon University
 * All rights reserved.  The CMU software License Agreement
 * specifies the terms and conditions for use and redistribution.
 *
 ****************************************************************************/


#ifdef MACH_SHARED_MEMORY_VERSION
#include <mach.h>
#endif

#ifdef UMAX_SHARED_MEMORY_VERSION
#include <parallel.h>
#endif

#include "global.h"


string  ParaOPS5_Version = "CParaOPS5 Version 5.3 P4.4";


/* #define SYS_DEBUG  1     Define here or in "version.h" for system debugging use only. */



/* Exported routines:
 *    void     ops_value(v)
 *    void     ops_settab(ndx)
 *    boolean  ops_emptytarget()
 *    void     ops_rval() 
 *    void     ops_symcon()
 *    void     ops_tab()
 *    void     ops_variabletab()
 *    void     ops_variable()
 *    void     ops_fixcon()
 *    void     ops_bmake()
 *    void     ops_emake()
 *    void     ops_remove()
 *    void     ops_bmodify()
 *    void     ops_emodify()
 *    void     ops_reset()
 *    void     ops_bind()
 *    void     ops_cbind()
 *    void     ops_call()
 *    void     ops_fcall()
 *    void     ops_litval(argc)
 *    void     ops_substr(argc)
 *    void     ops_genatom(argc)
 *    void     ops_crlf(argc)
 *    void     ops_rjust(argc)
 *    void     ops_tabto(argc)
 *    void     ops_ucall()
 *    OpsVal   dollar_intern(str)
 *    int      dollar_symbol(atom)
 *    int      dollar_eql(atom1, atom2)
 *    OpsVal   dollar_cvna(num)
 *    int      dollar_cvan(atom)
 *    char*    dollar_cvas(atom)
 *    int      dollar_litbind(atom)
 *    int      dollar_parametercount()
 *    OpsVal   dollar_parameter(findex)
 *    void     dollar_value(atom)
 *    void     dollar_tab(atom)
 *    void     dollar_assert()
 *    void     dollar_reset()
 *    FILE    *dollar_ifile(fileatom)
 *    FILE    *dollar_ofile(fileatom)
 *    void     ops_accept(argc)
 *    void     ops_acceptline(argc)
 *    void     ops_write()
 *    void     ops_openfile()
 *    void     ops_closefile()
 *    void     ops_default()
 *    void     ops_halt()
 *    void     ops_add()
 *    void     ops_sub()
 *    void     ops_mult()
 *    void     ops_div()
 *    void     ops_mod()
 *    void     opsret()
 *    void     ops_rt_init(argc,argv)
 *    void     ops_start_processes()
 *    void     ops_fire(strt)
 *    void     ops_main()
 *    void     ops_wakeup_matchers()
 *    void     ops_run_stats(fp_x)
 *
 */



/* Imported Routines:
 *    From utility.c:
 *       ops_malloc
 *       ops_compute_utime
 *       ops_compute_stime
 *       ops_fatal
 *       ops_warn
 *       ops_init_profile
 *       ops_get_profile
 *       ops_equiv_strings
 *       ops_init_time
 *       ops_print_timing_info
 *    From match.c:
 *       ops_init_task_queue
 *       ops_init_tok_mems
 *       ops_reinit_tok_mems
 *    From conres.c.c:
 *       ops_dumpallcs
 *       ops_print_cs
 *       ops_resolve
 *       ops_init_conflict_set
 *       ops_reinit_conflict_set
 *    From wminput.c:
 *       ops_do_loadwm
 *    From shmem.c:
 *       ops_touch_pages
 *       ops_InitShMem
 *    From user.c:
 *       ops_user_init
 *       ops_user_exit
 *    From gensymbol.c:
 *       ops_init_symbols
 *       ops_symid_lookup
 *       ops_symname_lookup
 *       ops_new_symbol
 *       ops_intern
 *       ops_pname
 *       ops_sym_binding
 *       ops_make_symbol
 *       ops_get_litval
 *    From wmemory.c:
 *       ops_newwme
 *       ops_inwm
 *       ops_print_wm_timetag
 *       ops_print_wm_pattern
 *       ops_goto_hell
 *       ops_init_wmemory
 *       ops_reinit_wmemory
 *       ops_add_target
 *       ops_del_target
 *       ops_do_remove
 */



/* External Routines:
 *    These routines from other modules return values other than the
 *    standard integer and so their return types are declared here
 *    for routines in this module that call them.
 */
extern char    *ops_malloc();          /* Imported from utility.c. */
extern void     ops_fatal();           /* Imported from utility.c. */
extern void     ops_warn();            /* Imported from utility.c. */
extern boolean  ops_equiv_strings();   /* Imported from utility.c. */
extern double   ops_compute_utime();   /* Imported from utility.c. */
extern double   ops_compute_stime();   /* Imported from utility.c. */
extern void     ops_init_profile();    /* Imported from utility.c. */
extern void     ops_get_profile();     /* Imported from utility.c. */
extern void     ops_init_time();       /* Imported from utility.c. */
extern void     ops_print_timing_info();  /* Imported from utility.c. */
extern symptr   ops_init_symbols();    /* Imported from gensymbol.c. */
extern symptr   ops_symid_lookup();    /* Imported from gensymbol.c. */
extern symptr   ops_symname_lookup();  /* Imported from gensymbol.c. */
extern symptr   ops_new_symbol();      /* Imported from gensymbol.c. */
extern symptr   ops_make_symbol();     /* Imported from gensymbol.c. */
extern OpsVal   ops_intern();          /* Imported from gensymbol.c. */
extern string   ops_pname();           /* Imported from gensymbol.c. */
extern OpsVal  *ops_newwme();          /* Imported from wmemory.c. */
extern boolean  ops_inwm();            /* Imported from wmemory.c. */
extern void     ops_init_wmemory();    /* Imported from wmemory.c. */
extern void     ops_reinit_wmemory();  /* Imported from wmemory.c. */
extern void     ops_addtarget();       /* Imported from wmemory.c. */
extern void     ops_deltarget();       /* Imported from wmemory.c. */
extern void     ops_goto_hell();       /* Imported from wmemory.c. */
extern void     ops_do_remove();       /* Imported from wmemory.c. */
extern void     ops_print_wm_timetag();/* Imported from wmemory.c. */
extern void     ops_print_wm_pattern();/* Imported from wmemory.c. */
extern void     ops_do_loadwm();       /* Imported from wminput.c. */
extern void     ops_touch_pages();     /* Imported from shmem.c. */
extern void     ops_InitShMem();       /* Imported from shmem.c. */
extern void     ops_init_task_queue(); /* Imported from match.c. */
extern void     ops_init_tok_mems();   /* Imported from match.c. */
extern void     ops_reinit_tok_mems(); /* Imported from match.c. */
extern void     ops_resolve();         /* Imported from conres.c. */
extern void     ops_dumpallcs();       /* Imported from conres.c. */
extern void     ops_print_cs();        /* Imported from conres.c. */
extern void     ops_init_conflict_set();   /* Imported from conres.c. */
extern void     ops_reinit_conflict_set(); /* Imported from conres.c. */
extern void     ops_user_init();        /* Imported from user.c. */
extern void     ops_user_exit();        /* Imported from user.c. */




/* Forward Declarations:
 *    These routines return values other than the standard integer and
 *    their return types are given here for other routines in this module
 *    that call them before they are defined.
 */
symptr   test_pbreak();
string   strip_quotes();
boolean  ops_emptytarget();
void     store_atom();
void     remove_port();
void     add_port();
void     print_switch_help();
void     actinit();
void     kill_processes();
void     mra_cycle();
void     wait_for_rete();
void     clear_input();
void     eof_exit();
void     getstring();
void     special_getstring();
void     init_cmds();
void     do_help();
void     find_nodeid_lists();
void     find_ext_addr();
void     find_lit_lists();
void     do_matches();
void     print_rmem();
void     print_lmem();
void     fill_result_wme();
void     top_tab();
void     top_tab_att();
void     top_tab_num();
void     top_wme();
void     top_wme_num();
void     top_wme_atom();
void     top_openfile();
void     top_add_port();
void     top_closefile();
void     top_default();
void     find_vecatt_list();
void     show_pbreaks();
void     pbreak_toggle();
void     ops_rt_init();
void     ops_start_processes();
void     ops_fire();
void     ops_main();





/*
**	Declarations
*/

typedef int (*pRtn)();

typedef struct port_rec_struct
        {
	 FILE   *FilePtr;       /* The system file pointer. */
	 string  FileName;      /* Valid system file name. */
	 int     AccessType;    /* How the file is used. */
	 OpsVal  OpsName;       /* Symbolic atom associated with file. */
	 int     Position;      /* Position on output line. */
	}
	port_rec;

#define MAXFILES          32	/* How many files can be open at once */

#define STDIN_NDX         0     /* Index for stdin. */
#define STDOUT_NDX        1     /* Index for stdout. */

#define READ_ACCESS       0     /* File opened for input. */
#define WRITE_ACCESS      1     /* File opened for output. */
#define ANY_ACCESS        2     /* File opened for input/output. */
#define NO_ACCESS        -1     /* File not opened. */

#define NOT_A_PORT       -1     /* Symbol not associated with a file. */
#define WRONG_PORT_TYPE  -2     /* File not opened for desired access. */ 

#define EMPTY_NAME        0     /* Impossible OpsVal for a symbolic atom. */

/*
**	Global variables
*/


static boolean TouchPages;             /* Tells if user wants shared memory pages touched. */
int *TouchCount;                       /* Counter to tell when all processes have touched pages. */

#define MINSTACKNDX 0
#define STACKSIZE 64
#define MAXSTACKNDX (STACKSIZE - 1)

static int StackNdx;
static OpsVal Stack[STACKSIZE];


static port_rec PortFile[MAXFILES];	/* File structures. */
static int      PortsLeft;              /* Ports available for file associations. */
static int      PortDefaultRead;        /* The default port to use for input. */
static int      PortDefaultWrite;       /* The default port to use for output. */

static long *ppc;			/* pseudo program counter */

static int flg_run;
int	*next;
static boolean HaltPerformed;           /* Set when HALT action executed. */

static OpsVal TmpWme[WMESIZE];		/* dummy wme for write, call, etc. */

static OpsVal *Target;			/* current element to process */

static OpsVal *LastAddWme;		/* last wme added by make or modify */

static int NxtNdx;			/* next slot to change in *Target */

#define BINDCOUNT 64

static OpsVal BindList[BINDCOUNT];	/* the bindings of normal vars */

typedef OpsVal *pWme;

static pWme CBindList[BINDCOUNT];	/* the bindings of ce vars */

#define LHSSIZE 256
static pWme Instance[LHSSIZE];



#define  FIRST_CMD     0
#define  CMD_Q         FIRST_CMD
#define  CMD_HELP      CMD_Q + 1
#define  CMD_EXIT      CMD_HELP + 1
#define  CMD_RUN       CMD_EXIT + 1
#define  CMD_WATCH     CMD_RUN + 1
#define  CMD_LOADWM    CMD_WATCH + 1
#define  CMD_MAKE      CMD_LOADWM + 1
#define  CMD_DUMPCS    CMD_MAKE + 1
#define  CMD_STRATEGY  CMD_DUMPCS + 1
#define  CMD_RESOLVE   CMD_STRATEGY + 1
#define  CMD_ZEROTIME  CMD_RESOLVE + 1
#define  CMD_SHOWTIME  CMD_ZEROTIME + 1
#define  CMD_WM	       CMD_SHOWTIME + 1
#define  CMD_PPWM      CMD_WM + 1
#define  CMD_REMOVE    CMD_PPWM + 1
#define  CMD_MATCHES   CMD_REMOVE + 1
#define  CMD_CALL      CMD_MATCHES + 1
#define  CMD_OPENFILE  CMD_CALL + 1
#define  CMD_CLOSEFILE CMD_OPENFILE + 1
#define  CMD_DEFAULT   CMD_CLOSEFILE + 1
#define  CMD_VERSION   CMD_DEFAULT + 1
#define  CMD_ASK       CMD_VERSION + 1
#define  CMD_PBREAK    CMD_ASK + 1
#define  CMD_SWITCHES  CMD_PBREAK + 1
#define  CMD_LITBIND   CMD_SWITCHES + 1
#define  CMD_CLASSBIND CMD_LITBIND + 1
#define  CMD_WMFORMAT  CMD_CLASSBIND + 1
#define  LAST_CMD      CMD_WMFORMAT

#define  NUMBER_CMDS   LAST_CMD + 1

static string          CmdTable[NUMBER_CMDS];

static int             CurrentCmd;
static boolean         FoundCmdError;

static char            LastInputChar;

#define  MAX_STRING_SIZE  128

#define  INFINITY      -1

#define   MIN_WATCH      0
#define   MAX_WATCH      3

static int BeginWatch;

static unsigned int BitVectorMask;    /* A bit for each active process. */

boolean RotateQueue;

static string128 InputMakeFile;    /* File of make actions (to init wm). */

static int SharedMemSize;   /* Specifies how much memory to allocate for wmes, tokens, etc. */

static int AskFireCnt;      /* Ask user, after this many firings, if he wants to continue firing. */

static int StartStrategy;   /* The conflict resolution strategy selected initially. */


#define PRINTALL	 -1     /* Print all wm */

boolean RefArray[WMESIZE];    /* Tells how to match WMEs for ppwm pattern. */

/* These locations are defined in the assembler coded output file produced
 * by the compiler, "cops5", as a result of compiling an OPS5 program. They
 * provide the support to implement the OPS5 top level user interface cmds
 * "matches" and "call".
 */
extern int MatchesFlag;      /* Set to 1, not used currently. */
extern int NumProductions;   /* How many productions were compiled. */
extern int *NodeIdLists[];      /* A table of beta nodeids associated with each production. */
extern int *ExternalList[];     /* A table of external function names and addresses. */
extern int NumExternals;



typedef struct pbreak_struct
        {
	 symptr               psym;    /* Symbol table entry of a production. */
	 struct pbreak_struct *next;
	}
	pbreak_rec, *pbreak_ptr;

static pbreak_ptr  PBreakList;        /* List of prod with breakpoints set. */
static pbreak_ptr  PBreakFreeList;    /* Return pbreak_rec storage here. */
static int         PBreakCount;       /* Tells how many on PBreakList. */
static symptr      PBreakFlag;        /* Identifies prod we break on, NULL otherwise. */



int Num_WM_Changes;           /* Counts adds/deletes to working memory. */
static int Num_RHS_Actions;   /* Counts RHS actions performed. */



/* These locations are defined in the assembler coded output file produced
 * by the compiler, "cops5", as a result of compiling an OPS5 program. They
 * provide the information needed to support the printing of wmes in standard
 * OPS5 format: ^attribute_name attribute_value.
 */
extern int *LiteralizeLists[];  /* A table of attributes associated with each literalized classname. */
extern int *VectorAttList[];    /* A table of the vector attributes declared. */

boolean StandardWmePrintFormat;   /* Indicates which format is used to print wmes. */


extern ThreadedCodeElement *RHScode[];   /* Data structure generated by compiler to hold RHS code. */


extern int  Default_MaxSymbols;   /* Size to allocate for SymbolTable. */




/*************************************************************************

		   	READING FROM PROGRAM STREAM

*************************************************************************/

/*
 *   nextint returns the next integer in the rhs code
 */

static
int nextint()
{
    return *(ppc++);
}

/*
 *	nextptr return the next pointer in the rhs code
 */

static
char *nextptr()
{
    return (char *) *(ppc++);
}



/*************************************************************************

		    HANDLING THE STACK OF VALUES

*************************************************************************/

/*
 *	resetstack() clears out the stack
 */

static void resetstack()
  {
    StackNdx = MINSTACKNDX - 1;
  }

/*
 *	push() pushes a value onto the stack
 */

static void push(v)
  OpsVal v;
  {
    if (++StackNdx > MAXSTACKNDX)
	ops_fatal("value stack overflow");
    Stack[StackNdx] = v;
  }


/*
 *	pop() pops a value from the stack
 */

static OpsVal pop()
  {
    if (StackNdx < MINSTACKNDX)
	ops_fatal("value stack underflow");
    return Stack[StackNdx--];
  }




/*************************************************************************

		   MANIPULATING THE CURRENT TARGET

*************************************************************************/

/*
 *	cleartarget() clears out a wme and prepares it to receive values
 */

static void cleartarget()
  {
    register OpsVal *cpy;
    register OpsVal *last;

    last = Target+WMEMAXNDX;
    for (cpy = Target+WMEMINNDX; cpy <= last; ++cpy)
	*cpy = symnil;
    Target[WMELENGTH] = 0;

    NxtNdx = WMEMINNDX;
  }


/*
 *	copywme() copies all the values of one wme into another one
 *
 */

static void copywme(dest, src)
  OpsVal *dest, *src;
  {
    register OpsVal *d, *s;
    register int z;

    d = dest;
    s = src;
    d[WMELENGTH] = s[WMELENGTH];
    for (z = s[WMELENGTH]; z >= WMEMINNDX; --z) {
 	++d;
	++s;
	*d = *s;
    };
  }
    

/*
 *	settarget() makes the current target be a particular wme
 */

static void settarget(ptr)
  OpsVal *ptr;
  {
    Target = ptr;
  }


/*
 *	ops_value() places a value into a wme
 */

void
ops_value(v)
  OpsVal v;
  {
    if ((NxtNdx > WMEMAXNDX) || (NxtNdx < WMEMINNDX))
	ops_fatal("working memory subelement index out of bounds");
    Target[NxtNdx] = v;
    if (NxtNdx > Target[WMELENGTH])
	Target[WMELENGTH] = NxtNdx;
    NxtNdx += 1;
  }

/*
 *	ops_settab() does the "^" operation
 */

void
ops_settab(ndx)
  int ndx;
  {
    if ((ndx<WMEMINNDX) || (ndx>WMEMAXNDX))
	ops_warn("illegal value after '^' -- ignoring it"); 
    else
	NxtNdx = ndx;
  }
  


boolean
ops_emptytarget()
{
   return(Target[WMELENGTH] == 0);
}




/*************************************************************************

		   MANIPULATING THE CURRENT TARGET

*************************************************************************/

void
ops_rval() 
  {
    ops_value(pop());
  }

void
ops_symcon()
  {
    OpsVal *p;

    p = (OpsVal *) nextptr();
    push(*p);
  }


void
ops_tab()
  {
    ops_settab(nextint());
  }


void
ops_variabletab()
{
    ops_settab(ops_get_litval(pop()));
}


void
ops_variable()
{
  int ce, fld;
  pWme wme;

    ce = nextint();
    
    if (ce < 0)
	push(BindList[-ce]);
    else {
	fld = nextint();
    	wme = Instance[ce-1];
	push(wme[fld]);
    };
}

void
ops_fixcon()
  {
    push(int2val(nextint()));
  }


/*
 *	make
 */

void
ops_bmake()
  {
    Num_RHS_Actions++;
    settarget(ops_newwme());
    cleartarget();
    actinit();
  }

void
ops_emake()
  {
    ops_addtarget(Target);
    LastAddWme = Target;
  }

/*
 * remove
 */

void
ops_remove()
  {
    int t;

    Num_RHS_Actions++;
    t = nextint();
    if (t < 0) 
	settarget(CBindList[-t]);
    else
	settarget(Instance[t-1]);
    if (ops_inwm(Target)) 
	ops_deltarget(Target); 
  }


/*
 * modify
 */

void
ops_bmodify()
  {
    register OpsVal *cpy;
    register OpsVal *last;

    int t;
    pWme ptr;

    Num_RHS_Actions++;
    /* find the old element */
    t = nextint();
    if (t < 0) 
	settarget(CBindList[-t]);
    else
	settarget(Instance[t-1]);

    /* copy values into new element */
    ptr = ops_newwme();

    last = ptr+WMEMAXNDX;
    for (cpy = ptr+WMEMINNDX; cpy <= last; ++cpy)
	*cpy = symnil;
    copywme(ptr, Target);

    /* delete the old element */
    if (ops_inwm(Target)) 
	ops_deltarget(Target);

    /*  prepare to process the new one */
    settarget(ptr);
    actinit();
  }


void
ops_emodify()
  {
    ops_addtarget(Target);
    LastAddWme = Target;
  }


/*
 *	ops_reset() is called to initialize the result array for all
 *	actions except make, modify, and remove
 */

void
ops_reset()
  {
    settarget(TmpWme);
    cleartarget();
    actinit();
  }



/*
 *	bind and cbind
 */

void
ops_bind()
  {
    int t;

    Num_RHS_Actions++;
    t = - nextint();
    BindList[t] = TmpWme[WMEMINNDX];
  }

void
ops_cbind()
  {
    int t;

    Num_RHS_Actions++;
    if (LastAddWme == NULL)
    	ops_fatal("no make or modify before cbind");
    t = nextint();
    CBindList[t] = LastAddWme;
  }


/*
 *	the call action
 */

void
ops_call()
  {
    pRtn ptr;			/* the action to call */

    Num_RHS_Actions++;
    ptr = (pRtn) nextptr();
    (*ptr)();
  }


/*************************************************************************

		   	      RHS FUNCTIONS

*************************************************************************/

/*
 *	ops_fcall() is the basic routine to call functions
 *
 */

void
ops_fcall()
  {
    int argc;			/* how many arguments are on stack */
    pRtn ptr;			/* the function to call */

    argc = nextint();
    ptr = (pRtn) nextptr();
    (*ptr)(argc);
    resetstack();
  }


/*
 *	ckargc() checks the number of arguments passed to a function
 *
 */

static void
ckargc(got, want)
  int got, want;
  {
    if (got < want)
    	ops_fatal("too few arguments supplied to function");
    if (got > want)
        ops_warn("too many arguments supplied to function -- ignoring extras");
  }


/*
 *	param() returns the nth argument to a function, assuming that
 *	the first argument is at the bottom of the stack
 */

static
OpsVal param(narg)
  int narg;
  {
    int off;

    off = narg - 1 + MINSTACKNDX;
    if ((off < MINSTACKNDX) || (off > StackNdx))
	ops_fatal("illegal parameter number");
    return Stack[off];
  }


void
ops_litval(argc)
  int argc;
  {
    ckargc(argc,1);
    ops_value(int2val(ops_get_litval(param(1))));
  }


void
ops_substr(argc)
  int argc;
  {
    register OpsVal *ptr;
    register int ndx;
    int wme, v2, v3;
    OpsVal a2, a3;

    ckargc(argc,3);
    wme = val2int(param(1));
    if (wme < 0) 
	ptr = CBindList[-wme];
    else
	ptr = Instance[wme-1];

    a2 = param(2);
    if (numberp(a2))
    	v2 = val2int(a2);
    else
      {
        if (a2 == syminf)
	   v2 = *(ptr + WMELENGTH);
	else
	  {
	   v2 = ops_sym_binding(val2sym(a2));
	   if (v2 < 0)
	      ops_fatal("SUBSTR: Arg2 is a symbolic atom but not an attribute name.");
	  }
      }

    a3 = param(3);
    if (numberp(a3))
    	v3 = val2int(a3);
    else
      {
        if (a3 == syminf)
	   v3 = *(ptr + WMELENGTH);
	else
	  {
           v3 = ops_sym_binding(val2sym(a3));
	   if (v3 < 0)
	      ops_fatal("SUBSTR: Arg3 is a symbolic atom but not an attribute name.");
	  }
      }

    if (v2 > v3)  ops_warn("SUBSTR: Arg2 > arg3.");
    if (v2 < WMEMINNDX)  ops_fatal("SUBSTR: Arg2 is out of range.");
    if (v3 > WMEMAXNDX)  ops_fatal("SUBSTR: Arg3 is out of range.");
    if (v3 > *(ptr+WMELENGTH))
    	{ if (*watch > 2) ops_warn("SUBSTR: Arg3 is beyond actual length of wme."); }

    for(ndx=v2; ndx<=v3; ++ndx)  ops_value(ptr[ndx]);
  }

void
ops_genatom(argc)
  int argc;
  {
    ckargc(argc,0);
    ops_value(ops_make_symbol());
  }


void
ops_crlf(argc)
  int argc;
  { 
    ckargc(argc,0);
    ops_value(symcrlf);
  }

void
ops_rjust(argc)
  int argc;
  {
    ckargc(argc,1);
    ops_value(symrjust);
    ops_value(param(1));
  }

void
ops_tabto(argc)
  int argc;
  {
    ckargc(argc,1);
    ops_value(symtabto);
    ops_value(param(1));
  }


/*************************************************************************
 *
 *	      RHS USER DEFINED FUNCTIONS AND CALL ACTIONS
 *
 *************************************************************************
 *
 * Implementation Details:
 *    In the current C based OPS5 interpreter, the scalar values (i.e.,
 *    symbolic and numeric atoms) are represented as the type OpsVal which
 *    is a 32 bit quantity. The low bit of an OpsVal indicates if the atom
 *    is a number or symbol (i.e., string). For a numeric atom, the top 31
 *    bits of an OpsVal hold the value of the number. For a symbolic atom,
 *    they hold the value of the symbol ID which is an index into the global
 *    symbol table where the actual string symbol is stored.
 *
 *    To accomodate C syntax requirements, the names of the interface
 *    routines are changed from "$parameter, $value, $assert, ...." to
 *    "dollar_parameter, dollar_value, dollar_assert, ....".
 *
 *    User defined functions are limited to a maximum of 50 parameters.
 *
 *************************************************************************/



void
ops_ucall()
/*---------------------------------------------------------------------------
 *
 * Abstract:
 *    This is the basic routine used by the interpreter to call user defined
 *    functions. 
 *
 * Parameters:
 *    None although implicit parameters have been set up both by the OPS5
 *    compiler and at run time by the interpreter as it executes the threaded
 *    code for the RHS.
 *
 * Environment:
 *    The parameters to the user defined function are held in the interpreter's
 *    stack and must be retrieved from there and placed into the argument list
 *    of the user function call. The identity of the user function along with
 *    the count of its arguments are to be found in the next 2 elements of the
 *    threaded code stream of the RHS.
 *
 * Returns:
 *    Nothing although the user defined function may return values through
 *    the result element wme and assert wmes into working memory.
 *
 * Calls:
 *    nextint, nextptr, and param in this module.
 *    the user defined function.
 *
 * Called by:
 *    the OPS5 interpreter as it executes the threaded code of the RHS.
 *
 *-------------------------------------------------------------------------*/
{
    int argc;          /* How many arguments are on stack. */
    pRtn ptr;	       /* The user C function to call. */
    OpsVal arg[50];    /* HACK: Assume limit of 50 parameters for */
                       /*       a user function.                  */
    int i;

    argc = nextint();
    ptr = (pRtn) nextptr();
    for (i = 0; i < argc; i++) arg[i] = param(i+1);

    /*  HACK: Call the user function. C does not care about the number
     *        of parameters, so we just need to make sure there are enough.
     *        Handle the most common cases correctly and just be inefficient
     *        and push extra dummy parameters on the stack for the  other cases.
     *        Perhaps there is a better way to do this.
     */
    switch(argc)
    {
     case 0:  (*ptr)();
              break;

     case 1:  (*ptr)(arg[0]);
              break;

     case 2:  (*ptr)(arg[0],
                     arg[1]);
              break;

     case 3:  (*ptr)(arg[0],
                     arg[1],
                     arg[2]);
              break;

     case 4:  (*ptr)(arg[0],
                     arg[1],
                     arg[2],
                     arg[3]);
              break;

     case 5:  (*ptr)(arg[0],
                     arg[1],
                     arg[2],
                     arg[3],
                     arg[4]);
              break;

     case 6: case 7: case 8: case9:
     case 10: (*ptr)(arg[0],
                     arg[1],
                     arg[2],
                     arg[3],
                     arg[4],
                     arg[5],
                     arg[6],
                     arg[7],
                     arg[8],
                     arg[9]);
              break;

     case 11: case 12: case 13: case 14: case 15: case 16: case 17: case 18: case 19:
     case 20: (*ptr)(arg[0],
                     arg[1],
                     arg[2],
                     arg[3],
                     arg[4],
                     arg[5],
                     arg[6],
                     arg[7],
                     arg[8],
                     arg[9],
                     arg[10],
                     arg[11],
                     arg[12],
                     arg[13],
                     arg[14],
                     arg[15],
                     arg[16],
                     arg[17],
                     arg[18],
                     arg[19]);
              break;

     case 21: case 22: case 23: case 24: case 25: case 26: case 27: case 28: case 29:
     case 30: (*ptr)(arg[0],
                     arg[1],
                     arg[2],
                     arg[3],
                     arg[4],
                     arg[5],
                     arg[6],
                     arg[7],
                     arg[8],
                     arg[9],
                     arg[10],
                     arg[11],
                     arg[12],
                     arg[13],
                     arg[14],
                     arg[15],
                     arg[16],
                     arg[17],
                     arg[18],
                     arg[19],
                     arg[20],
                     arg[21],
                     arg[22],
                     arg[23],
                     arg[24],
                     arg[25],
                     arg[26],
                     arg[27],
                     arg[28],
                     arg[29]);
              break;

     case 31: case 32: case 33: case 34: case 35: case 36: case 37: case 38: case 39:
     case 40: (*ptr)(arg[0],
                     arg[1],
                     arg[2],
                     arg[3],
                     arg[4],
                     arg[5],
                     arg[6],
                     arg[7],
                     arg[8],
                     arg[9],
                     arg[10],
                     arg[11],
                     arg[12],
                     arg[13],
                     arg[14],
                     arg[15],
                     arg[16],
                     arg[17],
                     arg[18],
                     arg[19],
                     arg[20],
                     arg[21],
                     arg[22],
                     arg[23],
                     arg[24],
                     arg[25],
                     arg[26],
                     arg[27],
                     arg[28],
                     arg[29],
                     arg[30],
                     arg[31],
                     arg[32],
                     arg[33],
                     arg[34],
                     arg[35],
                     arg[36],
                     arg[37],
                     arg[38],
                     arg[39]);
              break;

     case 41: case 42: case 43: case 44: case 45: case 46: case 47: case 48: case 49:
     case 50: (*ptr)(arg[0],
                     arg[1],
                     arg[2],
                     arg[3],
                     arg[4],
                     arg[5],
                     arg[6],
                     arg[7],
                     arg[8],
                     arg[9],
                     arg[10],
                     arg[11],
                     arg[12],
                     arg[13],
                     arg[14],
                     arg[15],
                     arg[16],
                     arg[17],
                     arg[18],
                     arg[19],
                     arg[20],
                     arg[21],
                     arg[22],
                     arg[23],
                     arg[24],
                     arg[25],
                     arg[26],
                     arg[27],
                     arg[28],
                     arg[29],
                     arg[30],
                     arg[31],
                     arg[32],
                     arg[33],
                     arg[34],
                     arg[35],
                     arg[36],
                     arg[37],
                     arg[38],
                     arg[39],
                     arg[40],
                     arg[41],
                     arg[42],
                     arg[43],
                     arg[44],
                     arg[45],
                     arg[46],
                     arg[47],
                     arg[48],
                     arg[49]);
              break;

     default: ops_fatal("User defined function call has more than 50 parameters.\n");
              break;

    } /* end switch */

    resetstack();
  }


OpsVal
dollar_intern(str)
   char *str;
/*---------------------------------------------------------------------------
 *
 * Abstract:
 *    Convert a string to a symbolic atom and return the symbolic atom.
 *    Also enters it into the global symbol table if it is not already
 *    there.
 *
 * Parameters:
 *    str - the string to convert.
 *
 * Environment:
 *    This routine is used by user defined functions and call actions to
 *    communicate with the OPS5 interpreter.
 *
 * Returns:
 *    The symbolic atom representing the string symbol.
 *
 * Calls:
 *    ops_symname_lookup and ops_new_symbol in gensymbol.c.
 *    ops_malloc in utility.c.
 *
 * Called by:
 *    User defined routines.
 *
 *-------------------------------------------------------------------------*/
{
   string s;
   symptr psym;

   psym = ops_symname_lookup(str);

   if (psym == NULL)
     {
      /* Allocate permanent storage for the string in shared memory
       * and add it to the symbol table.
       */
      s = (string) ops_malloc(strlen(str) + 1);
      strcpy(s, str);
      psym = ops_new_symbol(s);
     }

   return(sym2val(psym->SymId));
}

   


int
dollar_symbol(atom)
   OpsVal atom;
/*---------------------------------------------------------------------------
 *
 * Abstract:
 *    Test if the atom is a symbolic atom or numeric atom.
 *
 * Parameters:
 *    atom - the atom to test.
 *
 * Environment:
 *    This routine is used by user defined functions and call actions to
 *    communicate with the OPS5 interpreter.
 *
 * Returns:
 *    TRUE if a symbolic atom and FALSE otherwise.
 *
 * Calls:
 *    symbolp macro in global.h.
 *
 * Called by:
 *    User defined routines.
 *
 *-------------------------------------------------------------------------*/
{
   return(symbolp(atom));  /* Return TRUE if atom is a symbol. */
}




int
dollar_eql(atom1, atom2)
   OpsVal atom1, atom2;
/*---------------------------------------------------------------------------
 *
 * Abstract:
 *    Test if 2 atoms are the same.
 *
 * Parameters:
 *    atom1, atom2 - the atoms to test.
 *
 * Environment:
 *    This routine is used by user defined functions and call actions to
 *    communicate with the OPS5 interpreter.
 *
 * Returns:
 *    TRUE if the same, FALSE otherwise.
 *
 * Calls:
 *    Nothing.
 *
 * Called by:
 *    User defined routines.
 *
 *-------------------------------------------------------------------------*/
{
   return(atom1 == atom2);  /* Return TRUE if atoms are the same. */
}


OpsVal
dollar_cvna(num)
   int num;
/*---------------------------------------------------------------------------
 *
 * Abstract:
 *    Convert a regular integer number to an OPS5 numeric atom.
 *
 * Parameters:
 *    num  - the number to convert.
 *
 * Environment:
 *    This routine is used by user defined functions and call actions to
 *    communicate with the OPS5 interpreter.
 *
 * Returns:
 *    The numeric atom representing the number.
 *
 * Calls:
 *    int2val macro in global.h.
 *
 * Called by:
 *    User defined routines.
 *
 *-------------------------------------------------------------------------*/
{
   return(int2val(num));
}


int
dollar_cvan(atom)
   OpsVal atom;
/*---------------------------------------------------------------------------
 *
 * Abstract:
 *    Convert a numeric atom to a regular number.
 *
 * Parameters:
 *    atom - the numeric atom to convert.
 *
 * Environment:
 *    This routine is used by user defined functions and call actions to
 *    communicate with the OPS5 interpreter.
 *
 * Returns:
 *    The number represented by the numeric atom.
 *
 * Calls:
 *    val2int macro in global.h.
 *
 * Called by:
 *    User defined routines.
 *
 *-------------------------------------------------------------------------*/
{
   return(val2int(atom));
}



char*
dollar_cvas(atom)
   OpsVal atom;
/*---------------------------------------------------------------------------
 *
 * Abstract:
 *    Convert a symbolic atom into a string.
 *
 * Parameters:
 *    atom - the symbolic atom to convert.
 *
 * Environment:
 *    This routine is used by user defined functions and call actions to
 *    communicate with the OPS5 interpreter.
 *
 * Returns:
 *    A pointer to the string represented by the symbolic atom.
 *
 * NOTE:
 *    This is not a standard OPS5 routine but seems like it might be
 *    useful to have available.
 *
 * Calls:
 *    ops_symid_lookup in gensymbol.c.
 *    malloc and strcpy from system libraries.
 *
 * Called by:
 *    User defined routines.
 *
 *-------------------------------------------------------------------------*/
{
   string s;
   symptr psym;

   psym = ops_symid_lookup(val2sym(atom));
   if (psym == NULL)  return(NULL);
   s =(string) malloc(strlen(psym->SymName) + 1);
   strcpy(s, psym->SymName);
   return(s);
}


int
dollar_litbind(atom)
   OpsVal atom;
/*---------------------------------------------------------------------------
 *
 * Abstract:
 *    Check if a symbolic atom has received (via a Literalize or Literal
 *    declaration) a numeric binding (which represents an attribute index
 *    to a field in a wme). 
 *
 * Parameters:
 *    atom - the symbolic atom to check.
 *
 * Environment:
 *    This routine is used by user defined functions and call actions to
 *    communicate with the OPS5 interpreter.
 *
 * Returns:
 *    The numeric binding assigned to the symbol if it's an attribute.
 *    Returns 0 otherwise.
 *
 * NOTE:
 *    This functions differs from the one defined in the standard OPS5
 *    manual in that it returns 0 if the string symbol has no binding.
 *
 * Calls:
 *    ops_symid_lookup in gensymbol.c.
 *    symbolp and val2sym macros in global.h
 *
 * Called by:
 *    User defined routines.
 *
 *-------------------------------------------------------------------------*/
{
   symptr psym;

   if (!symbolp(atom))  return(0);

   psym = ops_symid_lookup(val2sym(atom));
   if (psym == NULL) return(0);

   if (psym->OpsBind == INVALIDBINDING)
      return(0);
   else
      return(psym->OpsBind);
}


int
dollar_parametercount()
/*---------------------------------------------------------------------------
 *
 * Abstract:
 *    Determine the number of the last field in the result element wme that
 *    received a value. For call actions, this corresponds to the number of
 *    parameters supplied to the called user function.
 *
 * Parameters:
 *    None.
 *
 * Environment:
 *    This routine is used by user defined functions and call actions to
 *    communicate with the OPS5 interpreter.
 *
 * Returns:
 *    The index of the last field assigned a value.
 *
 * Calls:
 *    Nothing.
 *
 * Called by:
 *    User defined routines.
 *
 *-------------------------------------------------------------------------*/
{
   return(Target[WMELENGTH]);
}


OpsVal
dollar_parameter(findex)
   int findex;
/*---------------------------------------------------------------------------
 *
 * Abstract:
 *    Get the value (i.e., atom) held in the indicated field of the result
 *    element wme.
 *
 * Parameters:
 *    findex - the index of the required field of the result element.
 *
 * Environment:
 *    This routine is used by user defined functions and call actions to
 *    communicate with the OPS5 interpreter.
 *
 * Returns:
 *    The atom from the indicated field. If the field was never given a
 *    value, the symbolic atom for "nil" is returned.
 *
 * Calls:
 *    ops_fatal in utility.c.
 *
 * Called by:
 *    User defined routines.
 *
 *-------------------------------------------------------------------------*/
{
   if ((findex < WMEMINNDX) || (findex > WMEMAXNDX))
      ops_fatal("dollar_parameter: wme field index out of range.\n");

   return(Target[findex]);
}


void
dollar_value(atom)
   OpsVal atom;
/*---------------------------------------------------------------------------
 *
 * Abstract:
 *    Insert a symbolic or numeric atom into the result element wme.
 *
 * Parameters:
 *    atom - the atom to insert.
 *
 * Environment:
 *    This routine is used by user defined functions and call actions to
 *    communicate with the OPS5 interpreter.
 *
 * Returns:
 *    Nothing.
 *
 * Calls:
 *    ops_value in this module.
 *
 * Called by:
 *    User defined routines.
 *
 *-------------------------------------------------------------------------*/
{
   ops_value(atom);
}


void
dollar_tab(atom)
   OpsVal atom;
/*---------------------------------------------------------------------------
 *
 * Abstract:
 *    Advance the pointer into the result element wme to the field indicated.
 *    This is where the next value will be inserted into the result element
 *    when "dollar_value" is next called.
 *
 * Parameters:
 *    atom - a numeric or symbolic atom that specifies the field index; a
 *           symbolic atom here must represent an attribute symbol that
 *           received a numeric binding via a Literalize or Literal
 *           declaration.
 *
 * Environment:
 *    This routine is used by user defined functions and call actions to
 *    communicate with the OPS5 interpreter.
 *
 * Returns:
 *    Nothing.
 *
 * Calls:
 *    ops_symid_lookup in gensymbol.c.
 *    symbolp, val2sym, and val2int macros in global.h.
 *    ops_fatal in utility.c.
 *
 * Called by:
 *    User defined routines.
 *
 *-------------------------------------------------------------------------*/
{
   symptr psym;
   int    index;

   if (symbolp(atom))
     {
      psym = ops_symid_lookup(val2sym(atom));
      if (psym == NULL) ops_fatal("dollar_tab: arg is not a valid symbol.\n");
      if (psym->OpsBind == INVALIDBINDING)  ops_fatal("dollar_tab: arg has no attribute binding\n");
      NxtNdx = psym->OpsBind;
     }
   else
     {
      index = val2int(atom);
      if ((index < WMEMINNDX) || (index > WMEMAXNDX))
         ops_fatal("dollar_tab: arg is out of range for wme field index.\n");
      NxtNdx = index;
     }
}


void
dollar_assert()
/*---------------------------------------------------------------------------
 *
 * Abstract:
 *    Copy the result element wme into working memory.
 *
 * Parameters:
 *    None.
 *
 * Environment:
 *    This routine is used by user defined functions and call actions to
 *    communicate with the OPS5 interpreter.
 *
 * Returns:
 *    Nothing.
 *
 * Calls:
 *    ops_newwme and ops_addtarget in wmemory.c.
 *    copywme in this module.
 *
 * Called by:
 *    User defined routines.
 *
 *-------------------------------------------------------------------------*/
{
   register OpsVal *cpy;
   register OpsVal *last;
   pWme            ptr;

   ptr = ops_newwme();
   last = ptr + WMEMAXNDX;
   for (cpy = ptr + WMEMINNDX; cpy <= last; cpy++)  *cpy = symnil;
   copywme(ptr, Target);

   ops_addtarget(ptr);
   LastAddWme = ptr;
}


void
dollar_reset()
/*---------------------------------------------------------------------------
 *
 * Abstract:
 *    Clears out all values in the result element wme.
 *
 * Parameters:
 *    None.
 *
 * Environment:
 *    This routine is used by user defined functions and call actions to
 *    communicate with the OPS5 interpreter.
 *
 * Returns:
 *    Nothing.
 *
 * Calls:
 *    cleartarget in this module.
 *
 * Called by:
 *    User defined routines.
 *
 *-------------------------------------------------------------------------*/
{
   cleartarget();
}



FILE *
dollar_ifile(fileatom)
   OpsVal fileatom;
/*---------------------------------------------------------------------------
 *
 * Abstract:
 *    Get a file descriptor to access a file previously opened by an
 *    "openfile" RHS action. Access is for reading.
 *
 * Parameters:
 *    fileatom - a symbolic atom that is associated with an open file,
 *               the symbolic atom was associated by a prior "openfile".
 *
 * Environment:
 *    This routine is used by user defined functions and call actions to
 *    communicate with the OPS5 interpreter.
 *
 * Returns:
 *    The file descriptor or NULL if the fileatom is not associated with
 *    a file open for input.
 *
 * Calls:
 *    lookup_port in this module.
 *
 * Called by:
 *    User defined routines.
 *
 *-------------------------------------------------------------------------*/
{
   int port;

   port = lookup_port(fileatom, READ_ACCESS);
   if ((port == NOT_A_PORT) || (port == WRONG_PORT_TYPE))
      return(NULL);
   else
      return(PortFile[port].FilePtr);
}

   


FILE *
dollar_ofile(fileatom)
   OpsVal fileatom;
/*---------------------------------------------------------------------------
 *
 * Abstract:
 *    Get a file descriptor to access a file previously opened by an
 *    "openfile" RHS action. Access is for writing.
 *
 * Parameters:
 *    fileatom - a symbolic atom that is associated with an open file,
 *               the symbolic atom was associated by a prior "openfile".
 *
 * Environment:
 *    This routine is used by user defined functions and call actions to
 *    communicate with the OPS5 interpreter.
 *
 * Returns:
 *    The file descriptor or NULL if the fileatom is not associated with
 *    a file open for output.
 *
 * Calls:
 *    lookup_port in this module.
 *
 * Called by:
 *    User defined routines.
 *
 *-------------------------------------------------------------------------*/
{
   int port;

   port = lookup_port(fileatom, WRITE_ACCESS);
   if ((port == NOT_A_PORT) || (port == WRONG_PORT_TYPE))
      return(NULL);
   else
      return(PortFile[port].FilePtr);
}

   


/*************************************************************************

		   	     INPUT-OUTPUT

*************************************************************************/



static
boolean
non_white(ch)
   char ch;
{
   if ((ch == '\t') || (ch == '\n') || (ch == ' '))
      return(FALSE);
   else
      return(TRUE);
}


static
boolean
is_white(ch)
   char ch;
{
   if ((ch == '\t') || (ch == '\n') || (ch == ' '))
      return(TRUE);
   else
      return(FALSE);
}


static
boolean
is_digit(ch)
{
	if ((ch < '0' ) || (ch > '9')) 
		return(FALSE);
	else
		return(TRUE);
}


static
boolean
is_int(str)
   char str[];
{
   int i;

   if ((str[0] == '+') || (str[0] == '-'))
     {
      if (str[1] != '\0')
         i = 1; 
      else
         return(FALSE);
     }
   else
      i = 0;

   while (str[i] != '\0')
      if ((str[i] < '0') || (str[i] > '9'))
         return(FALSE);
      else
         i++;

   return(TRUE);
}



static
int
atoi(str)
   char str[];
{
   int sign, val, i;

   i = 0;
   if (str[i] == '-')
     {
      sign = -1;
      i++;
     }
   else if (str[i] == '+')
     {
      sign = 1;
      i++;
     }
   else
      sign = 1;

   val = 0;
   while (str[i])  val = 10*val + str[i++] -'0';

   val = val*sign;

   return(val);
}



void
ops_accept(argc)
   int argc;
{
   OpsVal  arg;
   int     port, res, idx;
   char    ch, str[128];

   if (argc != 0)  ckargc(argc, 1);

   if (argc == 0)
      port = PortDefaultRead;
   else
     {
      arg = param(1);
      if (!symbolp(arg))
         ops_fatal("ACCEPT: Argument must be symbolic atom.");
      else
        {
	 /* Is the atom a file opened for input? */
	 port = lookup_port(arg, READ_ACCESS);
	 if (port == NOT_A_PORT)
	    ops_fatal("ACCEPT: Argument is not bound to filename.");
	 else if (port == WRONG_PORT_TYPE)
	    ops_fatal("ACCEPT: File not opened for reading.");
	}
     }

   res = get_nonwhite_char(port, &ch);
   if (res == EOF)
     {
      ops_value(symeof);
      return;
     }

   if (ch == '(')
     {
      /* List of atoms to input. */
      idx = 0; 
      while ((res = fscanf(PortFile[port].FilePtr, "%c", &ch)) != EOF)
        {
	 if ((ch == '"') || (ch == '|'))
	   {
	    res = scan_quoted_str(port, str, ch, 0);
	    idx = 0;
	   }
	 else if ((ch == ' ') || (ch == '\t') || (ch == '\n'))
	   {
	    if (idx > 0)
	      {
	       store_atom(str, idx);
	       idx = 0;
	      }
	    else
	       continue;    /* Just for emphasis. */
	   }
	 else if (ch == ')')
	   {
	    if (idx > 0)  store_atom(str, idx);
	    return;
	   }
	 else
	   {
	    str[idx++] = ch;
	    if (idx > 127)  ops_fatal("ACCEPT: Input string too big, limit is 127 chars.");
	   }
	} /* end while */

      if (res == EOF)  ops_fatal("ACCEPT: End of file encountered inputting list.");
     }
   else if ((ch == '"') || (ch == '|'))
     {
      /* Only 1 quoted atom to input. */
      res = scan_quoted_str(port, str, ch, 0);
      if (res == EOF)  ops_fatal("ACCEPT: End of file encountered inputting quoted atom.");
     }
   else
     {
      /* Only 1 atom to input. */
      str[0] = ch;
      idx = 1;
      while ((res = fscanf(PortFile[port].FilePtr, "%c", &ch)) != EOF)
         if (non_white(ch))
	   {
	    str[idx++] = ch;
	    if (idx > 127)  ops_fatal("ACCEPT: Input string too big, limit is 127 chars.");
	   }
	 else
	    break;
      if (res == EOF)
         ops_value(symeof);
      else
         store_atom(str, idx);
     }
}


static
int
scan_quoted_str(port, str, qch, hack)
   int    port;
   string str;
   char   qch;    /* The quote char. */
   int    hack;   /* 0 => ACCEPT, 1 => ACCEPTLINE */
{
   string s;
   char   ch;
   int    idx, res;

   idx = 0; 
   while ((res = fscanf(PortFile[port].FilePtr, "%c", &ch)) != EOF)
     {
      if (ch == qch) 
        {
         str[idx] = '\0';
         s = (string) ops_malloc(idx+1);
         strcpy(s, str);
         ops_value(ops_intern(s));
	 return(res);
	}
      else if (ch == '\n')
         /* Skip over newlines. */ ;
      else
	{
	 str[idx++] = ch;
	 if (idx > 127)
	   {
	    if (hack)
	       ops_fatal("ACCEPTLINE: Input string too big, limit is 127 chars.");
            else
	       ops_fatal("ACCEPT: Input string too big, limit is 127 chars.");
	   }
	}
     } /* end while */

   /* If we get here, then we must have hit EOF before we
    * encountered the enclosing quote char. So let the
    * caller know about it.
    */
   str[idx] = '\0';
   return(res);
}

static void
store_atom(str, idx)
   char str[];
   int  idx;
{
   string s;

   str[idx] = '\0';
   if (is_int(str))
      ops_value(int2val(atoi(str)));
   else
    {
     s = (string) ops_malloc(idx+1);
     strcpy(s, str);
     ops_value(ops_intern(s));
    }
}



static
int
get_nonwhite_char(port, ch)
   int port;
   char *ch;
{
   int res;

   while ((res = fscanf(PortFile[port].FilePtr, "%c", ch)) != EOF)
      if (non_white(*ch))  break;
   
   return(res);
}



void
ops_acceptline(argc)
   int argc;
{
   OpsVal  arg;
   int     args_used, port, res, idx;
   char    ch, str[128];
   boolean got_input;
   int l_parencnt;

   if (argc < 0)  ckargc(argc, 0);

   args_used = 0;
   if (argc == 0)
      port = PortDefaultRead;
   else
     {
      arg = param(1);
      if (!symbolp(arg))
         port = PortDefaultRead;
      else
        {
	 /* Is the atom a file opened for input? */
	 port = lookup_port(arg, READ_ACCESS);
	 if ((port == NOT_A_PORT) || (port == WRONG_PORT_TYPE))
	    port = PortDefaultRead;
	 else
	    args_used++;
	}
     }

   got_input = FALSE;
   l_parencnt = 0;   /* So we can count matching parens in input to parse lists. */
   idx = 0;
   while ((res = fscanf(PortFile[port].FilePtr, "%c", &ch)) != EOF)
     {
      if ((ch == '"') || (ch == '|'))
        {
         res = scan_quoted_str(port, str, ch, 1);
         got_input = TRUE;     /* Even if above was a quoted null string, it's considered input. */
         if (res == EOF)  break;
        }
      else if (ch == '\n')
        {
	 if (idx > 0)
	   {
            store_atom(str, idx);
	    got_input = TRUE;
	   }
         break;
	}
      else if ((ch == '\t') || (ch == ' '))
        {
	 if (idx > 0)
	     {
	      store_atom(str, idx);
	      got_input = TRUE;
	      idx = 0;
	     }
	 else
	      continue;       /* Just for emphasis. */
	}
      else if (ch == '(')
         l_parencnt++;       /* And just continue. */
      else if ((ch == ')') && (l_parencnt))
        {
         l_parencnt--;
	 if (idx > 0)
	     {
	      store_atom(str, idx);
	      got_input = TRUE;
	      idx = 0;
	     }
	 else
	      continue;       /* Just for emphasis. */
	}
      else
        {
         str[idx++] = ch;
         if (idx > 127)  ops_fatal("ACCEPTLINE: Input string too big, limit is 127 chars.");
        }
     } /* end while */


   if ((res == EOF) || (!got_input))
     {
      /* Put args into result element. */
      for (idx = args_used+1; idx <= argc; idx++)
	 ops_value(param(idx));
     }

   if (l_parencnt)  ops_warn("ACCEPTLINE: Unbalanced parens encountered in input - ignoring.");
}


static void
docrlf(port)
  int port;
  {
    fprintf(PortFile[port].FilePtr, "\n");
    PortFile[port].Position = 0;
  }


static void
dotabto(port, col)
  int port, col;
  {
    register int z;

    if (col < PortFile[port].Position)
    	docrlf(port);
    for (z=PortFile[port].Position; z<col-1; ++z)
        fprintf(PortFile[port].FilePtr, " ");
    PortFile[port].Position = col-1;
  }


static void
dorjust(port, fld, val)
  int port;
  int fld;
  OpsVal val;
  {
    register char *ptr;
    int len;
    int x;
    char obuf[32];

    if (symbolp(val))
        ptr = ops_pname(val);
    else {
        for (ptr=obuf+31; ptr>=obuf; --ptr) *ptr='\0';
	ptr=obuf;
        sprintf(ptr, "%d", val2int(val));
    };
    len = strlen(ptr);
    for (x = fld-len; x>0; --x)
	fprintf(PortFile[port].FilePtr, " ");
    fprintf(PortFile[port].FilePtr, "%s", ptr);
    if (fld > len)
    	PortFile[port].Position += fld;
    else
    	PortFile[port].Position += len;
  }



static void
wratom(port, val, need)
  int port;
  OpsVal val;
  int need;
  {
    register char *ptr;
    char obuf[32];

    if (symbolp(val))
        ptr = ops_pname(val);
    else {
        for (ptr=obuf+31; ptr>=obuf; --ptr) *ptr='\0';
	ptr=obuf;
        sprintf(ptr, "%d", val2int(val));
    };
    if (need) {
    	fprintf(PortFile[port].FilePtr, " ");
	PortFile[port].Position += 1;
    };
    fprintf(PortFile[port].FilePtr, "%s", ptr);
    PortFile[port].Position += strlen(ptr);
  }



void
ops_write()
  {
    int needspace, max, k;
    int port;
    OpsVal x;

    Num_RHS_Actions++;
    max = TmpWme[WMELENGTH];
    if (max < 1)
    	{ops_warn("WRITE: Nothing to write."); return;};

    k = 1; x = TmpWme[k];
    if (!symbolp(x))
       port = PortDefaultWrite;
    else
      {
       /* Is the atom a file opened for output? */
       port = lookup_port(x, WRITE_ACCESS);
       if ((port == NOT_A_PORT) || (port == WRONG_PORT_TYPE))
          port = PortDefaultWrite;
       else
          k++;
      }

    needspace = 1;
    for ( ; k <= max; ++k) {
    	x = TmpWme[k];
	if (x == symcrlf) {
	    docrlf(port);
	    needspace = 0;
	} else if (x == symrjust) {
	    dorjust(port, val2int(TmpWme[k+1]), TmpWme[k+2]);
	    k += 2;
	    needspace = 1;
	} else if (x == symtabto) {
	    dotabto(port, val2int(TmpWme[++k]));
	    needspace = 0;
	} else {
	    wratom(port, x, needspace);
	    needspace = 1;
	};
    };
  }


void
ops_openfile()
{
   OpsVal fatom, dir, fname;
   string filename, filetype;
   FILE   *fileptr;
   int    access;

   Num_RHS_Actions++;
   if (TmpWme[WMELENGTH] < 3)
      ops_fatal("OPENFILE: Action requires 3 args.");
   else if (TmpWme[WMELENGTH] > 3)
      ops_warn("OPENFILE: Too many args; ignoring extras.");

   fatom = TmpWme[1];
   fname = TmpWme[2];
   dir   = TmpWme[3];

   if ((!symbolp(fatom)) || (!symbolp(fname)) || (!symbolp(dir)))
      ops_fatal("OPENFILE: Args must be symbolic atoms.");

   if (fatom == symnil)  ops_fatal("OPENFILE: Nil is an invalid arg.");

   if (dir ==symin)
     {
      filetype = "r";
      access = READ_ACCESS;
     }
   else if (dir == symout)
     {
      filetype = "w";
      access = WRITE_ACCESS;
     }
   else
      ops_fatal("OPENFILE: Filetype arg must be in or out.");

   filename = ops_pname(fname);
   if ((fileptr = fopen(filename, filetype)) == NULL)
      ops_fatal("OPENFILE: System Error: File could not be accessed.");

   add_port(fileptr, filename, access, fatom);
}



void
ops_closefile()
{
   OpsVal fatom;
   int    numfiles, i, port;

   Num_RHS_Actions++;
   numfiles = TmpWme[WMELENGTH];
   if (numfiles < 1)  { ops_warn("CLOSEFILE: No files to close."); return; }

   for (i = numfiles; i > 0; i--)
     {
      fatom = TmpWme[i];
      if (!symbolp(fatom))
         ops_fatal("CLOSEFILE: Arg must be a symbolic atom");
      else
        {
	 if (fatom == symnil)  ops_fatal("CLOSEFILE: Nil is an invalid arg.");

	 port = lookup_port(fatom, ANY_ACCESS);
	 if (fclose(PortFile[port].FilePtr) == EOF)
	    ops_fatal("CLOSEFILE: System Error: Stream could not be closed.");

	 remove_port(port);
	}
     }
}



void
ops_default()
{
   OpsVal fatom, WhichDefault;
   int    access, port;

   Num_RHS_Actions++;
   if (TmpWme[WMELENGTH] < 2)
      ops_fatal("DEFAULT: Action requires 2 args.");
   else if (TmpWme[WMELENGTH] > 2)
      ops_warn("DEFAULT: Too many args; ignoring extras.");

   fatom        = TmpWme[1];
   WhichDefault = TmpWme[2];

   if (WhichDefault == symaccept)
      access = READ_ACCESS;
   else if ((WhichDefault == symwrite) || (WhichDefault == symtrace))
      access = WRITE_ACCESS;
   else
      ops_fatal("DEFAULT: Default type must be accept, write, or trace.");

   if (!symbolp(fatom))
      ops_fatal("DEFAULT: File arg must be a symbolic atom.");
   else
     {
      port = lookup_port(fatom, access);
      if (port == NOT_A_PORT)
         ops_fatal("DEFAULT: Arg not associated with a port.");
      else if (port == WRONG_PORT_TYPE)
         ops_fatal("DEFAULT: File not opened for this type of access.");
      else
        {
	 if (WhichDefault == symaccept)
	    PortDefaultRead = port;
	 else
	    PortDefaultWrite = port;
	}
     }
}



static void
remove_port(port)
   int port;
{
   if (port == PortDefaultRead)   PortDefaultRead = STDIN_NDX;
   if (port == PortDefaultWrite)  PortDefaultWrite = STDOUT_NDX;

   PortFile[port].FilePtr    = NULL;
   PortFile[port].FileName   = NULL;
   PortFile[port].AccessType = NO_ACCESS;   
   PortFile[port].OpsName    = EMPTY_NAME;   
   PortFile[port].Position   = 0;

   PortsLeft++;
}



static
int
lookup_port(name, access)
   OpsVal name;
   int    access;
{
   int i;

   for (i = 0; i < MAXFILES; i++)
      if (PortFile[i].OpsName == name)
        {
         if ((access == ANY_ACCESS) || (access == PortFile[i].AccessType))
	    return(i);
	 else
	    return(WRONG_PORT_TYPE);
	}

   return(NOT_A_PORT);
}



static void
add_port(fileptr, filename, access, name)
   FILE   *fileptr;
   string  filename;
   int     access;
   OpsVal  name;
{
   int port, i;

   /* See if the name is already entered with a file association.
    * If it is, then the user neglected to close the file
    * associated with this symbolic name. If so, we'll print a
    * warning below and just reuse the entry.
    */
   port = lookup_port(name, ANY_ACCESS);

   if (port == NOT_A_PORT)
     {
      if (PortsLeft)
        {
	 i = 0;
	 while (PortFile[i].FilePtr) i++;

	 PortFile[i].FilePtr    = fileptr;
	 PortFile[i].FileName   = filename;
	 PortFile[i].AccessType = access;
	 PortFile[i].OpsName    = name;
	 PortFile[i].Position   = 0;

	 PortsLeft--;
	}
      else
         ops_fatal("OPENFILE: No ports left to allocate.");
     }
   else
     {
      ops_warn("OPENFILE: File previously associated with name was not closed.");

      PortFile[port].FilePtr    = fileptr;
      PortFile[port].FileName   = filename;
      PortFile[port].AccessType = access;
      PortFile[port].Position   = 0;
     }
} 



static void
init_io()
{
   int i;

   for (i = 0; i < MAXFILES; i++)
     {
      PortFile[i].FilePtr    = NULL;
      PortFile[i].FileName   = NULL;
      PortFile[i].AccessType = NO_ACCESS;
      PortFile[i].OpsName    = EMPTY_NAME;
      PortFile[i].Position   = 0;
     }

   /* Set up default input/output. */
   PortFile[STDIN_NDX].FilePtr    = stdin;
   PortFile[STDIN_NDX].FileName   = "stdin";
   PortFile[STDIN_NDX].AccessType = READ_ACCESS;
   PortFile[STDIN_NDX].OpsName    = symnil;       /* In OPS5, nil -> terminal. */
   PortFile[STDIN_NDX].Position   = 0;

   PortFile[STDOUT_NDX].FilePtr    = stdout;
   PortFile[STDOUT_NDX].FileName   = "stdout";
   PortFile[STDOUT_NDX].AccessType = WRITE_ACCESS;
   PortFile[STDOUT_NDX].OpsName    = symnil;       /* In OPS5, nil -> terminal. */
   PortFile[STDOUT_NDX].Position   = 0;

   PortsLeft = MAXFILES - 2;

   PortDefaultRead  = STDIN_NDX;
   PortDefaultWrite = STDOUT_NDX;
}
      
	 


void
ops_halt()
{
   Num_RHS_Actions++;
   HaltPerformed = TRUE;
}


/*************************************************************************

		   	 ARITHMETIC OPERATIONS


*************************************************************************/

void
ops_add()
{
  int x, y;

   x = val2int(pop());
   y = val2int(pop());
   push(int2val(y+x));
}

void
ops_sub()
{
  int x, y;

   x = val2int(pop());
   y = val2int(pop());
   push(int2val(y-x));
}

void
ops_mult()
{
  int x, y;

   x = val2int(pop());
   y = val2int(pop());
   push(int2val(y*x));
}

void
ops_div()
{
  int x, y;

   x = val2int(pop());
   y = val2int(pop());
   push(int2val(y/x));
}

void
ops_mod()
{
  int x, y;

   x = val2int(pop());
   y = val2int(pop());
   push(int2val(y%x));
}

void
opsret()
{
  flg_run = 0;
}







main(argc, argv)
   int  argc;
   char *argv[];
{
   ops_rt_init(argc, argv);

if (show_debug)  printf("calling syminstall\n");
   ops_syminstall();

if (show_debug)  printf("calling bind_names\n");
   ops_bind_names();

if (show_debug)  printf("calling start_processes\n");
   ops_start_processes();

if (show_debug)  printf("calling fire makestart\n");
   flg_ruleid = symnil;
   ops_fire((pRtn)RHScode[0]);    /* we may want to fire this after init in ops_main ??? */

if (show_debug)  printf("calling ops_main\n");
   ops_main();
}


#ifdef SYS_DEBUG
/* Used only for system debugging.
 */
FILE *fireout, *firein;
int   ReportFire, TrackFire, BreakOnTrackFire;
int   FireID[1000], NumFireIDs;

get_fireids()   /* Reads in the trace file, FIREIN, that shows the expected */
{               /* production fired in order.                               */
   int num;
   string128  str;

   firein = fopen("FIREIN", "r");

   NumFireIDs = 0;
   while ((fscanf(firein, "%d %d %s", &num, &FireID[NumFireIDs], str) != EOF))
     {
      printf("%d  %d  %s\n", NumFireIDs+1, FireID[NumFireIDs], str);
      NumFireIDs++;
      if (NumFireIDs > 999)  ops_fatal("get_fireids: too many firings");
     }
}
#endif   /* SYS_DEBUG */



/*************************************************************************

		   	     INITIALIZATION

*************************************************************************/

/*
 *	ops_rt_init() is called once when the system starts up
 */

static   int    uargc;
static   char  *uargv[100];   /* We will limit number of args for ops_user_init to 100. */


void
ops_rt_init(argc,argv)
   int argc;
   char *argv[];
{

    /* Set up file descriptors.
     */
    fp_watch   = stdout; /* fopen("PSMWATCH", "w"); */
    fp_dbug    = stdout; /* fopen("DBUG", "w"); */
    fp_err     = stderr; /* fopen("ERRORFILE","w"); */
    fp_out     = stdout;

    /* Set flg_ruleid to indicate that we have not yet fired a production. */
    flg_ruleid = 0;    /* i.e., an illegal OpsVal */
    Num_WM_Changes = 0;
    Num_RHS_Actions = 0;
    
    /* interpret flags */
    BeginWatch = 0;
    NumProc = 1;
    NumQueues = 1;
    NumQueues_Minus1 = 0;
    QueueSelect = 0;
    RotateQueue = FALSE;
    WantCmdInterpreter = FALSE;
    TouchPages = FALSE;
    InputMakeFile[0] = (char) NULL;
    SharedMemSize = 1023*2*4;   /* Number of Kbytes. */
    AskFireCnt = 0;  /* Ask user, after this many firings, if he wants to continue firing. */
    StartStrategy = MEA;
    StandardWmePrintFormat = TRUE;
    PBreakList     = NULL;
    PBreakFreeList = NULL;
    PBreakCount    = 0;
    PBreakFlag     = NULL;
    Default_MaxSymbols = 4096;   /* Size to allocate for SymbolTable. */

#ifdef SYS_DEBUG
BreakOnTrackFire = ReportFire = TrackFire = 0;
#endif
    
    /* Process cmd line args. If we see a "--" switch, we will pass
     * rest of cmd line args to ops_user_init.
     */
    uargc = 0;
    while (--argc > 0)
       if (cmd_line_arg(*++argv) == 1)
          while (--argc > 0)
	     if (uargc < 100)
	        uargv[uargc++] = *++argv;
	     else
	       {
                fprintf(fp_err, "Can't pass more than 100 args to ops_user_init,\n");
                ops_fatal("Goodbye\n");
	       }
    ops_user_init(uargc, uargv);


#ifdef UNIPROC_VERSION
   if (NumProc != 1)
      ops_fatal("Number of processes must be 1 unless using shared memory version.\n");
#endif


    ops_InitShMem(SharedMemSize);  /* Allocation in Kbytes. */

    end_of_run_flag = (int *) ops_malloc(PointerSize);
    *end_of_run_flag = 0;
    next	= (int *)ops_malloc(PointerSize);
    *next	= 0;
    watch = (int *) ops_malloc(PointerSize);
    *watch = BeginWatch;

    TouchCount  = (int *) ops_malloc(PointerSize);
    *TouchCount = NumProc;

    /* init other modules */
    ops_init_symbols();
    ops_init_wmemory();
    ops_init_conflict_set();   *CR_Strategy = StartStrategy;
    ops_init_task_queue();
    ops_init_tok_mems();

#ifndef UNIPROC_VERSION
    ops_init_profile();
#endif

    HaltPerformed = FALSE;

    /* Set up input/output. */
    init_io();
    LastInputChar = '\0';
    
    
    /* Set up for getting some timing results */
    g_time_match_beg = (struct rusage *) malloc(sizeof(struct rusage));
    g_time_match_end = (struct rusage *) malloc(sizeof(struct rusage));
    g_time_rhs_beg = (struct rusage *) malloc(sizeof(struct rusage));
    g_time_rhs_end = (struct rusage *) malloc(sizeof(struct rusage));
    g_time_init = (struct rusage *) malloc(sizeof(struct rusage));
    g_time_load_beg = (struct rusage *) malloc(sizeof(struct rusage));
    g_time_load_end = (struct rusage *) malloc(sizeof(struct rusage));

}


static void
get_switches_from_file(filename)
   char *filename;
{
   FILE      *fp_switches;
   string256 fcmd_arg;
   string    tmparg;

   if ((fp_switches = fopen(filename, "r")) == NULL)
     {
      fprintf(fp_err, "Can't open file %s for cmd line args.\n", filename);
      ops_fatal("Goodbye\n");
     }

   while (fscanf(fp_switches, "%s", fcmd_arg) != EOF)
     if (cmd_line_arg(fcmd_arg) == 1)   /* Upon "--" switch, rest of args go to ops_user_init. */
        while (fscanf(fp_switches, "%s", fcmd_arg) != EOF)
           if (uargc < 100)
	     {
	      tmparg = (char *) malloc(strlen(fcmd_arg));
	      strcpy(tmparg, fcmd_arg);
              uargv[uargc++] = tmparg;
	     }
	   else
	     {
              fprintf(fp_err, "Can't pass more than 100 args to ops_user_init,\n");
	      fprintf(fp_err, "file is  %s.\n", filename);
              ops_fatal("Goodbye\n");
	     }
}


static
int
cmd_line_arg(cmd_arg)
   char *cmd_arg;
{
   int       return_val;       /* The value returned to caller, 1 => saw "--" switch. */

    return_val = 0;   /* Return 0 to caller as long as "--" switch not specified. */
   
    if (cmd_arg[0] != '-')
      {
       fprintf(fp_err, "Bad cmd line arg: %s\n", cmd_arg);
       ops_fatal("Cmd line arg must begin with a '-'.\n");
      }
    else
      {
	switch(cmd_arg[1])
	{
	  case 'w': if (is_int(&(cmd_arg[2])))
		       BeginWatch = atoi(&(cmd_arg[2]));
		    else
		      {
	               fprintf(fp_err, "Watch flag w requires integer arg between %d and %d. Setting watch to %d.\n",
		               MIN_WATCH, MAX_WATCH, MIN_WATCH);
		       BeginWatch = MIN_WATCH;
		      }
	  	    if ((BeginWatch < MIN_WATCH) || (BeginWatch > MAX_WATCH))
	  	      {
	     		fprintf(fp_err, "Value of watch flag should be between %d and %d.  Setting watch to %d.\n",
			       MIN_WATCH, MAX_WATCH, MIN_WATCH);
	     		BeginWatch = MIN_WATCH;
		      }
		    break;
	  case 'p': if (!is_int(&(cmd_arg[2])))
	               ops_fatal("p switch requires integer arg between 1 and 32.\n");
		    else
		       NumProc = atoi(&(cmd_arg[2]));
	  	    if ((NumProc < 1) || (NumProc > MAX_PROCESSES))
	  	      {
	     		fprintf(fp_err, "Number of processors should be between 1 and %d.  Setting NumProc to 1.\n",
                               MAX_PROCESSES);
	     		NumProc = 1;
		      }
		    break;
	  case 'q': if (!is_int(&(cmd_arg[2])))
	               ops_fatal("q switch requires integer arg between 1 and 32.\n");
		    else
		       NumQueues = atoi(&(cmd_arg[2]));
	  	    if ((NumQueues < 1) || (NumQueues > MAX_QUEUES))
	  	      {
	     		fprintf(fp_err, "Number of queues should be between 1 and %d.  Setting NumQueues to 1.\n",
                               MAX_QUEUES);
	     		NumQueues = 1;
		      }
		    else
		      { int i, j, k;
		        j = k = 0;
		        for (i=32; i>0; i=i/2)
			   if (i & NumQueues)  { if (j != 0) { k = i; break; } else j = i; }
			if (k != 0)
			  {
	     		   fprintf(fp_err, "Number of queues must be a power of 2.  Setting NumQueues to %d.\n", j);
  	     		   NumQueues = j;
			  }
		      }
		    NumQueues_Minus1 = NumQueues - 1;
		    break;
	  case 'm': if (!is_int(&(cmd_arg[2])))
	               ops_fatal("m switch requires integer arg to specify Kbytes of allocated memory.\n");
		    else
		       SharedMemSize = atoi(&(cmd_arg[2]));
		    break;
	  case 'c': WantCmdInterpreter = TRUE;
	            break;
	  case 'd': show_debug = TRUE;
	            break;
	  case 'r': RotateQueue = TRUE;
	            break;
	  case 'i': strcpy(InputMakeFile, &(cmd_arg[2]));
	  	    break;
	  case 'f': get_switches_from_file(&(cmd_arg[2]));
	  	    break;
	  case 's': if (ops_equiv_strings(&(cmd_arg[2]), "mea"))
	  	       StartStrategy = MEA;
                    else if (ops_equiv_strings(&(cmd_arg[2]), "lex"))
	  	       StartStrategy = LEX;
		    else
		       ops_fatal("strategy switch should specify mea or lex.\n");
	            break;
	  case 'v': fprintf(fp_out, "%s\n", ParaOPS5_Version);
	            break;
	  case 't': TouchPages = TRUE;
	            break;
	  case 'a': if (!is_int(&(cmd_arg[2])))
	               ops_fatal("a switch requires integer arg.\n");
		    else
		       AskFireCnt = atoi(&(cmd_arg[2]));
		    if (AskFireCnt < 0)  AskFireCnt = 0;
	            break;
	  case 'z': if (!is_int(&(cmd_arg[2])))
	               ops_fatal("z switch requires integer arg.\n");
		    else
		      {
                       if (atoi(&(cmd_arg[2])) == 0)
		          StandardWmePrintFormat = FALSE;
                       else
			  StandardWmePrintFormat = TRUE;
		      }
	            break;
	  case 'y': if (!is_int(&(cmd_arg[2])))
	               ops_fatal("y switch requires integer arg to specify size of symbol table.\n");
		    else
  	               Default_MaxSymbols = atoi(&(cmd_arg[2]));
		    break;
	  case '-': return_val = 1;   /* Signal caller that he should pass rest of */
	            break;            /* args to ops_user_init routine for processing. */
	  case '?':
	  case 'h': print_switch_help();
	            exit(0);
	            break;
#ifdef SYS_DEBUG
	  case 'R': ReportFire = 1;    /* Produces a production firing trace file, FIREOUT. */
	            fireout = fopen("FIREOUT", "w");
	            break;
	  case 'T': TrackFire = 1;     /* Turns on tracking of productions fired against trace. */
		    get_fireids();     /* Input the trace file, renamed FIREIN, from a previous run. */
	            break;
#endif
	  default : fprintf(fp_err, "%c is not a valid switch.\n", cmd_arg[1]);
                    print_switch_help();
	            ops_fatal("Goodbye\n");
	            break;
		    		    
	}
      }

    return(return_val);
}


static void
print_switch_help()
{
   fprintf(fp_out, "Valid Switches are:\n");
   fprintf(fp_out, "	-an		Ask to continue every n firings (n=0 means don't ask).\n");
   fprintf(fp_out, "	-c		Invoke user command interface.\n");
   fprintf(fp_out, "	-d		Show some debugging infomation.\n");
   fprintf(fp_out, "	-ffile		Take program switches from a file.\n");
   fprintf(fp_out, "	-h		Print this help message.\n");
   fprintf(fp_out, "	-ifile		Load working memory from an input file.\n");
   fprintf(fp_out, "	-mn		Set memory allocation size to n kilobytes.\n");
   fprintf(fp_out, "	-pn		Run with n processes.\n");
   fprintf(fp_out, "	-qn		Run with n task queues.\n");
   fprintf(fp_out, "	-r		Distribute root node tokens round robin among queues.\n");
   fprintf(fp_out, "	-sstr		Set strategy, str is mea or lex.\n");
   fprintf(fp_out, "	-t		Touch all the allocated memory pages.\n");
   fprintf(fp_out, "	-v		Print the ParaOPS5 version number.\n");
   fprintf(fp_out, "	-w		Set watch level to n (0, 1, 2, 3).\n");
   fprintf(fp_out, "	-yn		Set size of symbol table to n symbols.\n");
   fprintf(fp_out, "	-zn		Select wme print format, n=0 for vector format, n=1 for standard format.\n");
   fprintf(fp_out, "	--		Pass remaining switch args to ops_user_init routine.\n");
   fprintf(fp_out, "	-?		Print this help message.\n");
   fprintf(fp_out, "Switch Defaults:\n");
   fprintf(fp_out, "	-a0 -m8184 -p1 -q1 -smea -w0 -y4096 -z1\n");
}



static void
ops_reinit()
{
   struct rusage   time_init;
   struct rusage  *ptr_time_init;
   ptr_time_init = &time_init;

    /* Get timing statistics for last run. */
    ops_print_timing_info();

    getrusage(RUSAGE_SELF, ptr_time_init);

    /* Set flg_ruleid to indicate that we have not yet fired a production. */
    flg_ruleid = 0;    /* i.e., an illegal OpsVal */
    
    *next = 0;  



    /* init other modules */
    ops_reinit_wmemory();
    ops_reinit_conflict_set();
    ops_reinit_tok_mems();
    ops_init_time();
    HaltPerformed = FALSE;

    /* Set up input/output. */
    init_io();
    LastInputChar = '\0';

    getrusage(RUSAGE_SELF, g_time_init);
    g_init_utime = ops_compute_utime(ptr_time_init, g_time_init);
    g_init_stime = ops_compute_stime(ptr_time_init, g_time_init);
}


/*
 *	actinit() should be called before the execution of
 *	an rhs action begins
 *
 */

static void
actinit()
  {
    resetstack();
    NxtNdx = 1;
  }

impdel() {}


/* extern etext(); */

void
ops_start_processes()
{
   int proc_cnt;
   int pid;

   ProcessID = 0;
   ProcessIDBit = 1;
   MyPrimaryQueue = ProcessID & NumQueues_Minus1;

#ifndef UNIPROC_VERSION
   if (show_debug)  {fprintf(fp_dbug, "Process %d forking other processes.\n", ProcessID);  fflush(fp_dbug);}

   BitVectorMask = 1;

   for (proc_cnt = 1; proc_cnt < NumProc; proc_cnt++)
     {
      BitVectorMask = (BitVectorMask << 1) | 1;   /* Set a bit for each process. */
      pid = fork();
      if (pid == -1)
        {
	 kill_processes();
	 ops_fatal("Fork failed ... Goodbye.\n");
	}
      if (pid == 0)  { ProcessID = proc_cnt;  break; }
      if (show_debug)  fprintf(fp_dbug, "PID=%d\n", pid);
     }

   if (ProcessID != 0)
     {  /* Child process */
      if (show_debug)  {fprintf(fp_dbug, "Process %d starting up...entering MatcherCode...\n", ProcessID); fflush(fp_dbug);}
      ProcessIDBit = (1 << ProcessID);
      MyPrimaryQueue = ProcessID & NumQueues_Minus1;
/*      monstartup((int) 2, etext);  */
      if (TouchPages)
        {
	 ops_touch_pages();
         if (show_debug) {fprintf(fp_dbug, "Process %d done touching pages.\n", ProcessID);  fflush(fp_dbug);}
        }
      ops_MatcherCode();
/*      ops_get_profile(); */
      if (show_debug)  {fprintf(fp_dbug, "Process %d terminating.\n", ProcessID);  fflush(fp_dbug);}
      exit(0);
     }
   else
     {  /* Process0, the parent */
        /* Just return and finish with the startup. */
       if (show_debug)  {fprintf(fp_dbug, "Process %d done forking other processes.\n", ProcessID);  fflush(fp_dbug);}
       BitVectorMask = BitVectorMask ^ 1;   /* Clear bit for P0 only. */
       if (TouchPages)
         {
	  ops_touch_pages();
          printf("Process %d done touching pages.\n", ProcessID);
          while (*TouchCount);   /* Wait till everyone has touched pages. */
	 }
/*       monstartup((int) 2, etext); */
     }
#else
   /* For uniprocessor version, just see if touch pages wanted.
    */
   if (TouchPages)
     {
      ops_touch_pages();
      if (show_debug) {fprintf(fp_dbug, "Process %d done touching pages.\n", ProcessID);  fflush(fp_dbug);}
     }
/*       monstartup((int) 2, etext); */
#endif
}



static void
kill_processes()
{

   *end_of_run_flag = 1;

   Test_Then_Lock(tb_lock);
   *BitVector = 0;   /* Clear BitVector to allow other processes to terminate. */
   Release(tb_lock);

   if (show_debug) fprintf(fp_dbug, "BitVectorMask = 0x%x\n", BitVectorMask);
/*    ops_get_profile(); */
   if (show_debug)  {fprintf(fp_dbug, "Process Zero terminating.\n");  fflush(fp_dbug);}

   ops_user_exit();
}





/*************************************************************************

		   	  CONTROLLING ROUTINES

*************************************************************************/

static  int   fire_cnt = 0;

void
ops_fire(strt)
  pRtn strt;
{
    register pRtn ptr;
    int i;
    symptr psym;

#ifdef SYS_DEBUG
if (show_debug) printf("enter fire\n");
#endif

    fire_cnt++;

    if (*watch >= 1)
      {
	psym = ops_symid_lookup(val2sym(flg_ruleid));
	fprintf(fp_watch, "\n%d. fire(%s): ", fire_cnt, psym->SymName);
	for (i=0; i < flg_len_instance; ++i)
	  if (Instance[i] != NULL) fprintf(fp_watch, "%d ", *(Instance[i] + WMETIMETAG));
	  else fprintf(fp_watch, "nil ");
	fprintf(fp_watch, "\n");
	fflush(fp_watch);
      }

	
#ifdef SYS_DEBUG
/* Used only for system debugging.
 */
if (ReportFire)
      {
	psym = ops_symid_lookup(val2sym(flg_ruleid));
	fprintf(fireout, "%d %d %s\n", fire_cnt, flg_ruleid, psym->SymName);
	fflush(fireout);
      }
if (TrackFire)
  {
   if (flg_ruleid != FireID[fire_cnt-1])
      {
	psym = ops_symid_lookup(val2sym(flg_ruleid));
	fprintf(fp_watch, "\n%d. fire(%s): ", fire_cnt, psym->SymName);
	for (i=0; i < flg_len_instance; ++i)
	  if (Instance[i] != NULL) fprintf(fp_watch, "%d ", *(Instance[i] + WMETIMETAG));
	  else fprintf(fp_watch, "nil ");
	fprintf(fp_watch, "\n");
	fprintf(fp_watch, "This rule should not be firing. The correct rule is:   ");
	psym = ops_symid_lookup(val2sym(FireID[fire_cnt-1]));
	fprintf(fp_watch, "%d. fire(%s): \n", fire_cnt, psym->SymName);
	fflush(fp_watch);
	BreakOnTrackFire = 1;
        printf("Suspending mra_cycle\n");
	return;
      }
  }
#endif   /* SYS_DEBUG */
	
    LastAddWme = NULL;
    ppc = (long *) strt;
    flg_run = 1;
    while (flg_run)
      {
       ptr = (pRtn) nextptr();
       (*ptr)();
      }

#ifdef SYS_DEBUG
if (show_debug) printf("fire: before goto_hell\n");
#endif

    ops_goto_hell();

#ifdef SYS_DEBUG
if (show_debug) printf("exit fire\n");
#endif
}


void
ops_main()   /* called from assembly code interface */
{

if (show_debug) printf("rhsrtn.c: enter ops_main\n");

   /* These next 2 stmts set up information to support top level "matches"
    * and "call" user interface cmds.
    */
   find_nodeid_lists();  /* Process the table of production beta nodeids at end of assembly file. */
   find_ext_addr();      /* Process the table of external routine names and addresses. */

   /* These next 2 stmts set up information to support printing wmes in standard
    * OPS5 format: ^attribute_name attribute_value.
    */
   find_lit_lists();     /* Process the table of literalized classes at end of assembly file. */
   find_vecatt_list();   /* Process the table of vector attributes at end of assembly file. */

if (show_debug) printf("rhsrtn.c: ops_main: before init_time\n");
   ops_init_time();   /* Set globals for accumulating time. */



   if (InputMakeFile[0] != (char) NULL)
     {
      /* NOTE: When loading working memory from a file, we do not accumulate the match
       *       time since, for NumProc > 1, the match overlaps with parsing of the
       *       load file.
       */
      getrusage(RUSAGE_SELF, g_time_load_beg);
      /* ops_wakeup_matchers() called in match.c if NumProc > 1 */
      ops_do_loadwm(InputMakeFile);
      ops_MatcherCode();  /* Since not timing the match on loads, I do it even if NumProc > 1. */      
      g_match_started = FALSE;   /* Reset this so match time for the load does not get accumulated. */
      /* if (NumProc > 1)  sleep_matchers();    Not needed since call to ops_MatcherCode has this effect. */
      getrusage(RUSAGE_SELF, g_time_load_end);
      if (show_debug) fprintf(fp_out, "MAKE file %s loaded to Working Memory.\n", InputMakeFile);
      g_load_utime += ops_compute_utime(g_time_load_beg, g_time_load_end);
      g_load_stime += ops_compute_stime(g_time_load_beg, g_time_load_end);
     }



   if (!WantCmdInterpreter)
     {
if (show_debug) printf("rhsrtn.c: ops_main: before mra_cycle\n");
      mra_cycle(INFINITY);
      ops_print_timing_info();
#ifdef SYS_DEBUG
if (BreakOnTrackFire)  goto Track;
#endif
      kill_processes();
      return;  /* Back to "main" routine. */
     }
#ifdef SYS_DEBUG
Track:
#endif

   /* Otherwise run the cmd interpreter. */
   init_cmds();
   while (TRUE)
     {
      int       n;
      string128 str;
      symptr    psym;
      pRtn      funcptr;

      CurrentCmd = getcmd();
      FoundCmdError = FALSE;
      switch(CurrentCmd)
        {
	 case CMD_RUN:
                  if (LastInputChar == '\n')
		    {
		     mra_cycle(INFINITY);
		     ops_print_timing_info();
		     break;
		    }
	          getstring(str);
		  if (strlen(str) == 0)
		     n = INFINITY;
                  else
		    {
		     if (is_int(str))
		       {
			n = atoi(str);
			if (n < 1)
			  {
			   fprintf(fp_err, "** Run parameter must be a positive number.\n");
			   break;
		          }
		       }
		     else
		       {
		        fprintf(fp_err, "** Error in run parameter. Type ? for help.\n");
		        break;
		       }
		    }
		  mra_cycle(n);
		  ops_print_timing_info();
	          break;
	 case CMD_WATCH:
                  if (LastInputChar == '\n')
		    {
		     fprintf(fp_out, "Watch value is %d.\n", *watch);
		     break;
		    }
	          getstring(str);
		  if (strlen(str) == 0)
		     fprintf(fp_out, "Watch value is %d.\n", *watch);
                  else
		    {
		     if (is_int(str))
		       {
			n = atoi(str);
			if ((n < MIN_WATCH) || (n > MAX_WATCH))
			   fprintf(fp_err, "** Watch value must be in range %d to %d, leaving it at %d\n",
			          MIN_WATCH, MAX_WATCH, *watch);
		        else
			   *watch = n;
		       }
		     else
		        fprintf(fp_err, "** Error in watch parameter. Type ? for help.\n");
		    }
	          break;
	 case CMD_WMFORMAT:
                  if (LastInputChar == '\n')
		    {
		     fprintf(fp_err, "** Must specify integer arg: 0 => vector format, 1 => ^att val format.\n");
		     break;
		    }
	          getstring(str);
		  if (strlen(str) == 0)
		     fprintf(fp_err, "** Must specify integer arg: 0 => vector format, 1 => ^att val format.\n");
                  else
		    {
		     if (is_int(str))
		       {
			n = atoi(str);
			if (n == 0)
			   StandardWmePrintFormat = FALSE;
		        else
			   StandardWmePrintFormat = TRUE;
		       }
		     else
		        fprintf(fp_err, "** Must specify integer arg: 0 => vector format, 1 => ^att val format.\n");
		    }
	          break;
	 case CMD_ASK:
                  if (LastInputChar == '\n')
		    {
		     fprintf(fp_out, "Ask value is currently %d.\n", AskFireCnt);
		     break;
		    }
	          getstring(str);
		  if (strlen(str) == 0)
		     fprintf(fp_out, "Ask value is currently %d.\n", AskFireCnt);
                  else
		    {
		     if (is_int(str))
		       {
			n = atoi(str);
			if (n < 0)
			   fprintf(fp_err, "** Ask value must be in specified >= 0, leaving it at %d.\n", AskFireCnt);
		        else
			   AskFireCnt = n;
		       }
		     else
		        fprintf(fp_err, "** Error in ask parameter. Type ? for help.\n");
		    }
	          break;
	 case CMD_LOADWM:
                  if (LastInputChar == '\n')
		    {
		     fprintf(fp_err, "** Must specify input file for loadwm.\n");
		     break;
		    }
	          getstring(str);
		  if (strlen(str) == 0)
		    {
		     fprintf(fp_err, "** Must specify input file for loadwm.\n");
		     break;
		    }
                  /* NOTE: When loading working memory from a file, we do not accumulate the match
		   *       time since, for NumProc > 1, the match overlaps with parsing of the
		   *       load file.
		   */
                  getrusage(RUSAGE_SELF, g_time_load_beg);
                  /* ops_wakeup_matchers() called in match.c if NumProc > 1 */
		  ops_do_loadwm(str);
		  ops_MatcherCode();  /* Since not timing the match on loads, I do it even if NumProc > 1. */      
                  g_match_started = FALSE;   /* Reset this so match time for the load does not get accumulated. */
                  /* if (NumProc > 1)  sleep_matchers();    Not needed since call to ops_MatcherCode has this effect. */
                  getrusage(RUSAGE_SELF, g_time_load_end);
                  if (show_debug) fprintf(fp_out, "MAKE file %s loaded to Working Memory.\n", InputMakeFile);
                  g_load_utime += ops_compute_utime(g_time_load_beg, g_time_load_end);
                  g_load_stime += ops_compute_stime(g_time_load_beg, g_time_load_end);
	          break;
	 case CMD_MAKE:
                  if (LastInputChar == '\n')
		    {
		     fprintf(fp_err, "** Must specify a pattern argument to make.\n");
		     break;
		    }
                  ops_bmake();
		  fill_result_wme();
                  if (FoundCmdError)  break;
		  if (ops_emptytarget())
		    {
		     fprintf(fp_err, "** Must specify a pattern argument to make.\n");
		     break;
		    }
                  /* ops_wakeup_matchers() called in match.c if NumProc > 1 */
                  ops_emake();
		  ops_MatcherCode();  /* Since not timing the match on top make, I do it even if NumProc > 1. */      
                  g_match_started = FALSE;   /* Reset this so match time for the make does not get accumulated. */
                  /* if (NumProc > 1)  sleep_matchers();    Not needed since call to ops_MatcherCode has this effect. */
	          break;
	 case CMD_OPENFILE:
                  if (LastInputChar == '\n')
		    {
		     fprintf(fp_err, "** Must specify a pattern argument to openfile.\n");
		     break;
		    }
                  ops_reset();
		  fill_result_wme();
                  if (FoundCmdError)  break;
		  if (ops_emptytarget())
		    {
		     fprintf(fp_err, "** Must specify a pattern argument to openfile.\n");
		     break;
		    }
                  top_openfile();
	          break;
	 case CMD_CLOSEFILE:
                  if (LastInputChar == '\n')
		    {
		     fprintf(fp_err, "** Must specify a pattern argument to closefile.\n");
		     break;
		    }
                  ops_reset();
		  fill_result_wme();
                  if (FoundCmdError)  break;
		  if (ops_emptytarget())
		    {
		     fprintf(fp_err, "** Must specify a pattern argument to closefile.\n");
		     break;
		    }
                  top_closefile();
	          break;
	 case CMD_DEFAULT:
                  if (LastInputChar == '\n')
		    {
		     fprintf(fp_err, "** Must specify a pattern argument to default.\n");
		     break;
		    }
                  ops_reset();
		  fill_result_wme();
                  if (FoundCmdError)  break;
		  if (ops_emptytarget())
		    {
		     fprintf(fp_err, "** Must specify a pattern argument to default.\n");
		     break;
		    }
                  top_default();
	          break;
	 case CMD_CALL:
                  if (LastInputChar == '\n')
		    {
		     fprintf(fp_err, "** Must specify an external user function name to call action.\n");
		     break;
		    }
                  getstring(str);
		  if (strlen(str) == 0)
		    {
		     fprintf(fp_err, "** Must specify an external user function name to call action.\n");
		     break;
		    }
                  psym = ops_symname_lookup(str);
		  if (psym == NULL)
		    {
		     fprintf(fp_err, "** %s is not an external function name.\n", str);
		     break;
		    }
                  funcptr = (pRtn) psym->Func_addr;
		  if (funcptr == NULL)
		    {
		     fprintf(fp_err, "** %s is not an external function name.\n", str);
		     break;
		    }
                  ops_reset();
		  fill_result_wme();
                  if (FoundCmdError)  break;
                  (*funcptr)();
                  ops_MatcherCode();   /* Make sure match occurs for any "dollar_assert" executed by ext function. */
                  g_match_started = FALSE;   /* Reset this so match time for the asserts does not get accumulated. */
	          break;
         case CMD_REMOVE:
                  if (LastInputChar == '\n')
		    {
		     fprintf(fp_err, "** Must specify timetags of wmes to remove.\n");
		     break;
		    }
	          getstring(str);
		  if (strlen(str) == 0)
		    {
		     fprintf(fp_err, "** Must specify timetags of wmes to remove.\n");
		     break;
		    }
                  /* ops_wakeup_matchers() called in match.c if NumProc > 1 */
                  if ((str[0] == '*') && (str[1] == '\0'))
		    {
		     ops_do_remove(NULLTIMETAG);   /* Remove all wmes. */
		    }
                  else
		    {
   		     for (;;)
   		       {
                        if (!is_int(str))
   		          {
			   fprintf(fp_err, "** Error in specifying timetag. Type ? for help.\n");
			   break;
		          }
		        else
                          {
			   n = atoi(str);
			   if (n < MINTIMETAG)
			     {
		              fprintf(fp_err, "** Wme timetag must be a positive number.\n");
			      break;
			     }
			   else
                              ops_do_remove(n);
                          }
		        if (LastInputChar != '\n')  getstring(str);  else  break;
		        if (strlen(str) == 0)  break;
		       }
                    }
		  ops_MatcherCode();  /* Since not timing the match on top remove, I do it even if NumProc > 1. */      
                  g_match_started = FALSE;   /* Reset this so match time for the removes does not get accumulated. */
                  /* if (NumProc > 1)  sleep_matchers();    Not needed since call to ops_MatcherCode has this effect. */
	          break;
	 case CMD_DUMPCS:
                  ops_MatcherCode();   /* Make sure the match is idle. */
	          ops_dumpallcs(fp_out);
                  fprintf(fp_out, "\nDominant Instantiation:\n");
                  ops_resolve(FALSE);
	          break;
	 case CMD_WM:
		  if (LastInputChar == '\n')
		    {
		     ops_print_wm_timetag(PRINTALL);
		     break;
		    }
		  getstring(str);
		  if (str[0] == '\0')  { ops_print_wm_timetag(PRINTALL);  break; }
		  for (;;)
		    {
		     if (str[0] == '\0')  break;
		     if (is_int(str))
		       {
		        n = atoi(str);
		        if (n < 1)
		          {
			   fprintf(fp_err, "** WME timetag must be a positive number.\n");
			   break;
			  }
		        else
		          {
			   ops_print_wm_timetag(n);
			   if (LastInputChar != '\n')  getstring(str);  else  break;
			  }
		       }
		     else
		       {
		        fprintf(fp_err, "** Error in specifying timetag. Type ? for help.\n");
		        break;
		       }
        	    }
		  break;
	 case CMD_PPWM:
                  if (LastInputChar == '\n')
                    {
		     ops_print_wm_pattern(NULL);  /* NULL -> print all of WM. */
		     break;
		    }
		  ops_reset();
		  for (n = 0; n < WMESIZE; n++)  RefArray[n] = FALSE;
		  fill_result_wme();
		  if (FoundCmdError)  break;
		  if (ops_emptytarget())
		    {
	             ops_print_wm_pattern(NULL);
		     break;
		    }
		 ops_print_wm_pattern(Target);
		 break;
         case CMD_MATCHES:
                  if (LastInputChar == '\n')
		    {
		     fprintf(fp_err, "** Must specify a production name.\n");
		     break;
		    }
	          special_getstring(str);
                  if (FoundCmdError)  break;
		  if (strlen(str) == 0)
		    {
		     fprintf(fp_err, "** Must specify a production name.\n");
		     break;
		    }
		  if (str[0] == '^')
		    {
		     fprintf(fp_err, "** Invalid use of '^' operator for production name, it must be quoted.\n");
		     break;
		    }
		  for (;;)
   		    {
                     int *nidlist;
		     int nodeids[LHSSIZE];
		     int nodecnt;

		     psym = ops_symname_lookup(strip_quotes(str));
		     if (psym == NULL)
		       {
			fprintf(fp_err, "** %s is not a production name.\n", str);
			break;
		       }
                     nidlist = psym->NodeidList;
		     if (nidlist == NULL)
		       {
			fprintf(fp_err, "** %s is not a production name.\n", psym->SymName);
			break;
		       }
                     nodecnt = 0;
		     while (*nidlist > 0)  nodeids[nodecnt++] = *nidlist++;
		     do_matches(psym, nodeids, nodecnt);
		     if (LastInputChar != '\n')  special_getstring(str);  else  break;
		     if (strlen(str) == 0)  break;
		    }
	          break;
         case CMD_PBREAK:
                  if (LastInputChar == '\n')
		    {
		     show_pbreaks();
		     break;
		    }
	          special_getstring(str);
                  if (FoundCmdError)  break;
		  if (strlen(str) == 0)
		    {
		     show_pbreaks();
		     break;
		    }
		  if (str[0] == '^')
		    {
		     fprintf(fp_err, "** Invalid use of '^' operator for production name, it must be quoted.\n");
		     break;
		    }
		  for (;;)
   		    {
		     psym = ops_symname_lookup(strip_quotes(str));
		     if (psym == NULL)
		       {
			fprintf(fp_err, "** %s is not a production name.\n", str);
			break;
		       }
		     if (psym->NodeidList == NULL)
		       {
			fprintf(fp_err, "** %s is not a production name.\n", psym->SymName);
			break;
		       }
		     /* Now toggle it and adjust global count. */
		     pbreak_toggle(psym);
		     if (LastInputChar != '\n')  special_getstring(str);  else  break;
		     if (strlen(str) == 0)  break;
		    }
	          break;
         case CMD_LITBIND:
                  if (LastInputChar == '\n')
		    {
		     fprintf(fp_err, "** Must specify an attribute name as an arg.\n");
		     break;
		    }
	          special_getstring(str);
                  if (FoundCmdError)  break;
		  if (strlen(str) == 0)
		    {
		     fprintf(fp_err, "** Must specify an attribute name as an arg.\n");
		     break;
		    }
		  if (str[0] == '^')
		    {
		     fprintf(fp_err, "** Invalid use of '^' operator for attribute name, it must be quoted.\n");
		     break;
		    }
	          psym = ops_symname_lookup(strip_quotes(str));
		  if (psym == NULL)
		    {
		     fprintf(fp_err, "** %s is not an attribute name.\n", str);
		     break;
		    }
		  if (psym->OpsBind == INVALIDBINDING)
		     fprintf(fp_err, "** %s is not an attribute name.\n", psym->SymName);
                  else
		     fprintf(fp_out, "%s = %d\n", psym->SymName, psym->OpsBind);
	          break;
         case CMD_CLASSBIND:
                  if (LastInputChar == '\n')
		    {
		     fprintf(fp_err, "** Must specify a class name as an arg.\n");
		     break;
		    }
	          special_getstring(str);
                  if (FoundCmdError)  break;
		  if (strlen(str) == 0)
		    {
		     fprintf(fp_err, "** Must specify a class name as an arg.\n");
		     break;
		    }
		  if (str[0] == '^')
		    {
		     fprintf(fp_err, "** Invalid use of '^' operator for class name, it must be quoted.\n");
		     break;
		    }
	          psym = ops_symname_lookup(strip_quotes(str));
		  if (psym == NULL)
		    {
		     fprintf(fp_err, "** %s is not a class name.\n", str);
		     break;
		    }
		  if (psym->AttPtrList == NULL)
                    {
		     fprintf(fp_err, "** %s is not a declared class name.\n", psym->SymName);
		     break;
	            }
		  else if (*(psym->AttPtrList) == NULL)
                    {
		     fprintf(fp_err, "** %s class does not have any declared attributes.\n", psym->SymName);
		     break;
	            }
                  else
		    {
		     int *intptr;
		     symptr asym;

		     fprintf(fp_out, "Bindings for attributes of class %s:\n", psym->SymName);
                     intptr = psym->AttPtrList;
                     while (*intptr)
		       {
                        asym = ops_symid_lookup(val2sym(*(int *)(*intptr)));
		        fprintf(fp_out, "\t\t%s = %d\n", asym->SymName, asym->OpsBind);
			intptr++;
		       }
		    }
	          break;
	 case CMD_STRATEGY:
                  if (LastInputChar == '\n')
		    {
		     fprintf(fp_out, "Strategy is currently");
		     if (*CR_Strategy == MEA) fprintf(fp_out, " mea\n"); else fprintf(fp_out, " lex\n");
		     break;
		    }
	          getstring(str);
		  if (strlen(str) == 0)
		    {
		     fprintf(fp_out, "Strategy is currently");
		     if (*CR_Strategy == MEA) fprintf(fp_out, " mea\n"); else fprintf(fp_out, " lex\n");
		    }
                  else
		    {
		     if (ops_equiv_strings(str,"mea"))
		        *CR_Strategy = MEA;
	             else if (ops_equiv_strings(str, "lex"))
		        *CR_Strategy = LEX;
		     else
		        fprintf(fp_err, "** Error in strategy parameter. Type ? for help.\n");
		    }
	          break;
         case CMD_RESOLVE:
                  ops_MatcherCode();   /* Make sure the match is idle. */
	          ops_resolve(FALSE);  /* Just make the inquiry and print the instantiation. */
	          break;
	 case CMD_ZEROTIME:
                  ops_init_time();
	          break;
	 case CMD_SHOWTIME:
                  ops_print_timing_info();
	          break;
	 case CMD_VERSION:
                  fprintf(fp_out, "%s\n", ParaOPS5_Version);
	          break;
	 case CMD_SWITCHES:
	          print_switch_help();
		  break;
	 case CMD_Q:
	 case CMD_HELP:
	          do_help();
	          break;
	 case CMD_EXIT:
                  kill_processes();
		  return;  /* Back to assembly code */
                  break;
	} /* endswitch */

      clear_input();

     } /* endwhile */
}


static void
mra_cycle(num_firings)    /* match-recognize-act cycle */
   int num_firings;

{
   register int z;
   symptr psym;

#ifdef SYS_DEBUG
if (show_debug) printf("rhsrtn.c: ops_main: enter mra_cycle\n");
#endif

   for (;;)
     {
      wait_for_rete();
	
      getrusage(RUSAGE_SELF, g_time_rhs_beg);
      ops_resolve(TRUE);

      if (HaltPerformed)
        {
	 psym = ops_symid_lookup(val2sym(flg_ruleid));
	 fprintf(fp_out, "Halt in Production %s\n", psym->SymName);
         fflush(fp_out);
	 return;
	}

      if (flgpfire == NULL)
        {
         if (show_debug)  fprintf(fp_dbug, "\n$$$$$$$$$$ NO PRODUCTIONS LEFT IN CONFLICT SET $$$$$$$$$$\n");
	 return;
	}

      g_match_started = FALSE;   /* Set to TRUE when 1st change to wm occurs */
	                         /* due to the production about to be fired. */

      for (z = 0; z < flg_len_instance; ++z)
	 Instance[z] = (pWme) (*flg_instance)[z];

      /* if (NumProc > 1)  ops_wakeup_matchers();  This gets done in match.c, instead. */

      ops_fire(RHScode[flgpfire]);
#ifdef SYS_DEBUG
if (BreakOnTrackFire)  return;
#endif
      
      if (PBreakCount)  PBreakFlag = test_pbreak(flg_ruleid);  else  PBreakFlag = NULL;

#ifdef ENCORE_VERSION_DEBUG
           Test_Then_Lock(tb_lock);
	   *BitVector = BitVectorMask;   /* Clear process 0 bit and set everyone else's. */
           Release(tb_lock);
#endif

      /* If fire did not produce an wm changes, then time spent in rhs
       * will be all the time taken by "ops_fire".
       */
      if ((g_match_started == FALSE) || (NumProc == 1))
         getrusage(RUSAGE_SELF, g_time_rhs_end);

      /* Now we can compute the time taken by the rhs.
       */
      g_rhs_utime += ops_compute_utime(g_time_rhs_beg, g_time_rhs_end);
      g_rhs_stime += ops_compute_stime(g_time_rhs_beg, g_time_rhs_end);

      if (PBreakFlag)
        {
	 wait_for_rete();
	 fprintf(fp_out, "\n$$$$$$ PBREAK AT PRODUCTION %s. $$$$$$\n", PBreakFlag->SymName);
	 return;   /* Return to cmd interpreter. */
	}

      if ((num_firings == INFINITY) && (AskFireCnt > 0))
	{
	 /* Just ask periodically whether to continue. */
	 if ((fire_cnt % AskFireCnt) == 0)
           {
            fprintf(fp_out, "\nfire_cnt is %d, do you want to halt? :[no] ", fire_cnt);
            scanf("%c", &LastInputChar);
            if ((LastInputChar == 'y') || (LastInputChar == 'Y'))
	      {
               /* show_hashtable(); */
               wait_for_rete();
	       return;
	      }
            clear_input();  /* Clear stdin */
           }
        }
      else
	{
	 num_firings--;
	 if (num_firings == 0)  { wait_for_rete(); return; }  /* Return to cmd interpreter. */
	}

     } /* end for */

} /* end mra_cycle() */



void
ops_wakeup_matchers()
{
   Test_Then_Lock(tb_lock);
   *BitVector = *BitVector | 1;   /* Set process 0 bit to wakeup other processes. */
   Release(tb_lock);
}



static void
sleep_matchers()
{
   Test_Then_Lock(tb_lock);
   *BitVector = *BitVector & ~1;   /* Clear process 0 bit so other processes can go to sleep. */
   Release(tb_lock);
}


static
boolean
queues_active()
{
   int i;

   /* Check to see if all tasks queued by the control process have been
    * noticed by the matchers. The only way to tell this is to see if
    * the queue used by the control process is empty at some point during
    * match. During the match, that queue may indeed contain tasks queued
    * by the matchers as tasks are removed and added. However, if that queue
    * ever makes a transition to empty, we are sure that all the tasks
    * queued by the control process have been picked up for processing.
    *
    * If RotateQueue is being used, then we must check over all the queues.
    *
    * NOTE: Finding the queue(s) empty here does not mean the match is
    *       complete. We're not doing any synchronized checking on the
    *       state of the matchers and the queues. We're only looking for
    *       an empty transition which is what we need to guarantee that
    *       the initial wme add/delete tasks from the control process
    *       have been picked up by the matchers.
    */

   if (!RotateQueue)
     {
      if (*tq_index[0] == -1)  return(FALSE);  else  return(TRUE);
     }
   else
     {
      for (i = 0; i < NumQueues, i < NumProc; i++)
        {
	 if (*tq_index[i] != -1)  return(TRUE);
	}
      return(FALSE);
     }
}



static void
wait_for_rete()
{
   if (NumProc == 1)
     {
      getrusage(RUSAGE_SELF, g_time_match_beg);
/*           printf("At ops_MatcherCode call for NumProc=1, BitVector= 0x%x\n", *BitVector); */
/*	   dodebug(); */
      ops_MatcherCode();
     }
   else 
     {
      while (queues_active())  ;    /* Loop here until sure all tasks added by control process are noticed. */
      
      Test_Then_Lock(tb_lock);
      *BitVector = *BitVector & ~1;   /* Clear process 0 bit so matchers can eventually go to idle loop. */
      Release(tb_lock);

      while (*BitVector) /*dodebug()*/  ;   /* Process 0 does not participate in the match. */
     }

   if (g_match_started)
     {
      getrusage(RUSAGE_SELF, g_time_match_end);
      g_match_utime += ops_compute_utime(g_time_match_beg, g_time_match_end);
      g_match_stime += ops_compute_stime(g_time_match_beg, g_time_match_end);
      g_match_started = FALSE;   /* Must reset for next time. */
     }
}


static void
show_hashtable()
{
	int jj;
	FILE	*ha;
	alpha_cell *a; 
	beta_cell *b;

		ha = fopen("hash.table","w");
		for(jj = 0;jj<4096;jj++){
		printf("%d\n",jj);
		b = ltokHT[jj];
		while(b)
		{
		fprintf(ha,"%d ",b->nodeid);
		b= b->next;
		}
		fprintf(ha," -1 \n");
		a = rtokHT[jj];
		while(a)
		{
		fprintf(ha,"%d ",a->nodeid);
		a = a->next;
		}
		fprintf(ha," -2 \n");
		}
}



static void
clear_input()
{
   char ch;

   while (LastInputChar != '\n')                                  /* Clear stdin */
      if (scanf("%c", &LastInputChar) == EOF)  eof_exit();
}


static void
eof_exit()
{
   kill_processes();
   fprintf(fp_err, "\nEOF Exit.\n");
   exit(0);
}


static void
getstring(str)
   string128 str;
{
   int idx;

   idx = 0;
   if (scanf("%c", &LastInputChar) == EOF)  eof_exit();
   while (LastInputChar != '\n')
     {
      if (non_white(LastInputChar))
        {
	 if (idx == MAX_STRING_SIZE-2)  /* allow for null byte at end */
	   {
	    fprintf(fp_err, "** Error: input string too long.\n");
	    clear_input();
	    idx = 0;  /* make string empty */
	    break;
	   }
	 else
	    str[idx++] = LastInputChar;
	}
      else   /* white space char */
        {
	 if (idx > 0)  break;
	}

      if (scanf("%c", &LastInputChar) == EOF)  eof_exit();
     }

   str[idx] = '\0';
}



static void
special_getstring(str)
   string128 str;
/*-------------------------------------------------------------------
 *
 *  Abstract:
 *    This routine is used in place of getstring whenever we want to
 *    input rhs patterns for cmds like "make" or production names
 *    for cmds like "pbreak". In both cases, we may have to process
 *    quoted strings. In case of rhs patterns, we have to deal with
 *    the '^' operator as well. If the characters, '^', '|', or '"',
 *    are to be part of a string symbol, then they must be quoted,
 *    eg., |n^2| or "n^2" for the string symbol n^2, "a|b" for the
 *    symbol a|b, etc.
 *
 *    Therefore this routine works as follows:
 *       1. Strip off leading white-space characters.
 *       2. If the 1st non-white character is '^', return it as
 *          a single character string.
 *       3. If the 1st non-white character is a quote ('"' or '|'),
 *          scan in all characters until a matching quote is found
 *          or end of line is found. Return the quoted string.
 *       4. If the 1st non-white character is none of the above,
 *          scan in all characters until white space or end of line
 *          is reached. Return the string.
 *       5. Signal an error and return the NULL string if an unquoted
 *          occurrence of '^', '|', or '"' is found in the middle of
 *          a string or if the max size for a string is exceeded.
 *
 *    For quoted strings, the quotes are included as part of the string
 *    so that the caller can determine whether it's a symbolic or
 *    numeric atom, eg., |874| is the string "874" and not the number
 *    874. The caller must strip off the quotes when doing symbol table
 *    lookups.
 *
 *    The caller must determine whether or not the string returned is
 *    valid in the given context, eg., if pbreak needs a production name
 *    and gets back the single character string "^", it must report the
 *    error.
 *
 *  Parameters:
 *    str - a character array in which the string is placed.
 *
 *  Returns:
 *    A string as specified above.
 *
 *  Environment:
 *    The top level cmd interface is processing the user's input for
 *    one of the cmds.
 *
 *---------------------------------------------------------------------*/
{
   int idx;
   char quote;

   idx = 0;
   if (scanf("%c", &LastInputChar) == EOF)  eof_exit();
   while (LastInputChar != '\n')
     {
      if ((idx == 0) && (LastInputChar == '^'))
        {
	 str[0] = '^';
	 str[1] = '\0';
         return;
	}
      else if ((idx == 0) && ((LastInputChar == '"') || (LastInputChar == '|')))
        {
	 quote = LastInputChar;
	 do
	   {
	    if (idx == MAX_STRING_SIZE-2)  /* allow for null byte at end */
	      {
               str[idx] = '\0';
	       fprintf(fp_err, "** Error: input string %s%c is too long.\n", str, LastInputChar);
	       FoundCmdError = TRUE;
	       clear_input();
	       str[0] = '\0';  /* make string empty */
	       return;
	      }
	    else
	       str[idx++] = LastInputChar;
            if (scanf("%c", &LastInputChar) == EOF)  eof_exit();
  	   }
	 while ((LastInputChar != quote) && (LastInputChar != '\n'));
	 str[idx++] = quote;
	 str[idx] = '\0';
	 return;
	}
      else if (non_white(LastInputChar))
        {
	 if (idx == MAX_STRING_SIZE-2)  /* allow for null byte at end */
	   {
            str[idx] = '\0';
	    fprintf(fp_err, "** Error: input string %s%c is too long.\n", str, LastInputChar);
	    FoundCmdError = TRUE;
	    clear_input();
	    str[0] = '\0';  /* make string empty */
	    return;
	   }
	 else if ((LastInputChar == '^') || (LastInputChar == '"') || (LastInputChar == '|'))
	   {
            str[idx++] = LastInputChar;
            str[idx] = '\0';
	    fprintf(fp_err, "** Error: The '%c' character must be quoted when part of a symbol.\n", LastInputChar);
            fprintf(fp_err, "          Input string is %s.\n", str);
	    FoundCmdError = TRUE;
	    clear_input();
	    str[0] = '\0';  /* make string empty */
	    return;
	   }
	 else
	    str[idx++] = LastInputChar;
	}
      else   /* white space char */
        {
	 if (idx > 0)  break;
	}

      if (scanf("%c", &LastInputChar) == EOF)  eof_exit();
     } /* endwhile */

   str[idx] = '\0';
}


static
int
getcmd()
{
   string128 cmd;
   int       i;

   while (TRUE)
     {
      fprintf(fp_out, "\nenter cmd > ");
      getstring(cmd); /* scanf("%s", cmd); */

      if (strlen(cmd) == 0)  continue;

      for (i = FIRST_CMD; i <= LAST_CMD; i++)
         if (ops_equiv_strings(cmd, CmdTable[i]))  return(i);

      fprintf(fp_out, "** %s is not a valid cmd, type ? for help.", cmd);
      clear_input();
     } /* endwhile */
}


static void
init_cmds()
{
   CmdTable[CMD_Q]        = "?";
   CmdTable[CMD_HELP]     = "help";
   CmdTable[CMD_EXIT]     = "exit";
   CmdTable[CMD_RUN]      = "run";
   CmdTable[CMD_WATCH]    = "watch";
   CmdTable[CMD_LOADWM]   = "loadwm";
   CmdTable[CMD_MAKE]     = "make";
   CmdTable[CMD_REMOVE]   = "remove";
   CmdTable[CMD_DUMPCS]   = "cs";
   CmdTable[CMD_STRATEGY] = "strategy";
   CmdTable[CMD_RESOLVE]  = "resolve";
   CmdTable[CMD_ZEROTIME] = "zerotime";
   CmdTable[CMD_SHOWTIME] = "showtime";
   CmdTable[CMD_WM]	  = "wm";
   CmdTable[CMD_PPWM]	  = "ppwm";
   CmdTable[CMD_MATCHES]  = "matches";
   CmdTable[CMD_VERSION]  = "version";
   CmdTable[CMD_ASK]	  = "ask";
   CmdTable[CMD_OPENFILE] = "openfile";
   CmdTable[CMD_CLOSEFILE] = "closefile";
   CmdTable[CMD_DEFAULT]  = "default";
   CmdTable[CMD_CALL]	  = "call";
   CmdTable[CMD_PBREAK]	  = "pbreak";
   CmdTable[CMD_WMFORMAT] = "wmefmt";
   CmdTable[CMD_SWITCHES] = "switches";
   CmdTable[CMD_LITBIND]  = "litbind";
   CmdTable[CMD_CLASSBIND] = "classbind";

}


static void
not_impl()
{
   fprintf(fp_out, "** Cmd not implemented yet.");
}


static void
do_help()
{
   fprintf(fp_out, "Valid cmds are:\n");
   fprintf(fp_out, "\t\n");
   fprintf(fp_out, "\trun  [n]\n");
   fprintf(fp_out, "\t\t- execute for n production firings or, if n not given, until conflict set empty\n");
   fprintf(fp_out, "\twatch  [val]\n");
   fprintf(fp_out, "\t\t- set watch to val or, if val not given, show current watch val\n");
   fprintf(fp_out, "\twmefmt  val\n");
   fprintf(fp_out, "\t\t- select the format for printing wmes, 0 => vector format, 1 => ^att value format\n");
   fprintf(fp_out, "\task  [val]\n");
   fprintf(fp_out, "\t\t- set ask to val or, if val not given, show current ask val\n");
   fprintf(fp_out, "\tloadwm  inputfile\n");
   fprintf(fp_out, "\t\t- load working memory from input file\n");
   fprintf(fp_out, "\tmake  rhs_term_list\n");
   fprintf(fp_out, "\t\t- make the wme specified by the pattern in the rhs_term_list\n");
   fprintf(fp_out, "\tremove  ttag1 ttag2 ..... ttagn\n");
   fprintf(fp_out, "\t\t- remove the wmes specified by their timetags, 'remove *' means all wmes\n");
   fprintf(fp_out, "\tcs\n");
   fprintf(fp_out, "\t\t- display the conflict set\n");
   fprintf(fp_out, "\twm  [ttag1 ttag2 ..... ttagn]\n");
   fprintf(fp_out, "\t\t- print the wmes specified by the timetags, 'wm' with no args means print all wmes \n");
   fprintf(fp_out, "\tppwm  rhs_term_list\n");
   fprintf(fp_out, "\t\t- print the wmes that match the pattern, 'ppwm' with no args means print all wmes \n");
   fprintf(fp_out, "\tmatches  prod1 prod2 ..... prodn\n");
   fprintf(fp_out, "\t\t- print the partial matches for the given productions\n");
   fprintf(fp_out, "\tpbreak  prod1 prod2 ..... prodn\n");
   fprintf(fp_out, "\t\t- toggle breakpoints for given productions, 'pbreak' with no args shows active ones\n");
   fprintf(fp_out, "\tstrategy  [mea | lex]\n");
   fprintf(fp_out, "\t\t- set strategy to mea or lex, or else show current setting\n");
   fprintf(fp_out, "\tresolve\n");
   fprintf(fp_out, "\t\t- show the next production that will fire\n");
   fprintf(fp_out, "\topenfile  rhs_term_list\n");
   fprintf(fp_out, "\t\t- open the file specified by the pattern in the rhs_term_list\n");
   fprintf(fp_out, "\tclosefile  rhs_term_list\n");
   fprintf(fp_out, "\t\t- close the files specified by the pattern in the rhs_term_list\n");
   fprintf(fp_out, "\tdefault  rhs_term_list\n");
   fprintf(fp_out, "\t\t- sets defaults for write, trace, accept according to pattern in the rhs_term_list\n");
   fprintf(fp_out, "\tcall  external_functions  rhs_term_list\n");
   fprintf(fp_out, "\t\t- invoke an external function with args supplied through pattern in the rhs_term_list\n");
   fprintf(fp_out, "\tlitbind attribute_name\n");
   fprintf(fp_out, "\t\t- show the numeric binding assigned to an attribute\n");
   fprintf(fp_out, "\tclassbind class_name\n");
   fprintf(fp_out, "\t\t- show the numeric bindings assigned to each of the attributes of a class\n");
   fprintf(fp_out, "\tzerotime\n");
   fprintf(fp_out, "\t\t- initialize the time gathering statistics to zero\n");
   fprintf(fp_out, "\tshowtime\n");
   fprintf(fp_out, "\t\t- display the timing statistics\n");
   fprintf(fp_out, "\tversion\n");
   fprintf(fp_out, "\t\t- show the version number of the system.\n");
   fprintf(fp_out, "\texit\n");
   fprintf(fp_out, "\t\t- stop firing productions and exit the program\n");
   fprintf(fp_out, "\tswitches\n");
   fprintf(fp_out, "\t\t- show switches that may be specified on cmd line when program invoked\n");
   fprintf(fp_out, "\t?\n");
   fprintf(fp_out, "\t\t- print this help msg\n");
   fprintf(fp_out, "\thelp\n");
   fprintf(fp_out, "\t\t- print this help msg\n");
   fprintf(fp_out, "\t\n");

}




static void
find_nodeid_lists()
{
   register int    **intptr;
   symptr psym;

   /* The table of nodeid lists is arranged as follows:
    *
    *       ProductionID1, nodeid1, nodeid2, .... nodeidn1, -1
    *       ProductionID2, nodeid1, nodeid2, .... nodeidn2, -1
    *          .                                      .
    *          .                                      .
    *          .                                      .
    *       ProductionIDk, nodeid1, nodeid2, .... nodeidnk, -1
    *       NULL
    *
    * where each is a 32-bit quantity.  The ProductionIDs are
    * pointers into the ops_symbols table and give us the link
    * to the production name in the symbol table where we will
    * store a pointer to its list of nodeids.
    */

   intptr = NodeIdLists;
   while (*intptr)
     {
      psym = ops_symid_lookup(val2sym(*(*intptr)));
      if (psym == NULL)
        {
	 fprintf(fp_err, "find_nodeid_lists: bad NodeIdList table.\n");
	 ops_fatal("Production name in table not found.\n");
	}

      psym->NodeidList = (int *)++intptr;  /* Store away the head of the nodeid list in   */
                                           /* symbol table entry for the production name. */
      while (((int)*intptr++) >= 0) ;      /* Find end of this nodeid list (marked by -1). */

      /* That leaves us pointing to the next production, unless
       * we're at the end of the table. In that case we will be
       * looking at a NULL.
       */
     }
}


static void
find_ext_addr()
{
   register int    **intptr;
   symptr psym;

   /* The table of External names is arranged as follows:
    *
    *       ExternalNameID1, RoutinePtr1
    *       ExternalNameID2, RoutinePtr2
    *          .                  .
    *          .                  .
    *          .                  .
    *       ExternalNameIDn, RoutinePtrn
    *       NULL
    *
    * where each is a 32-bit quantity.  The ExternalNameIDs are
    * pointers into the ops_symbols table and give us the link
    * to the external name in the symbol table where we will
    * store the address of the external routine. The Unix linker
    * has filled in the RoutinePtrs for us.
    */


   intptr = ExternalList;
   while (*intptr)
     {
      psym = ops_symid_lookup(val2sym(**(intptr)));
      if (psym == NULL)
        {
	 fprintf(fp_err, "find_ext_lists: bad External Name List table.\n");
	 ops_fatal("External name in table not found.\n");
	}

      psym->Func_addr = (int)*++intptr;  /* Store away the address of the external routine in */
                                         /* the symbol table entry for the external name.     */

      intptr++;   /* Advance to next name in table. */
     }
}



static void
find_lit_lists()
{
   register int    **intptr;
   symptr psym;

   /* The table of literalized classname lists is arranged as follows:
    *
    *       ClassnameID1, AttributeID1, AttributeID2, .... AttributeIDn1, 0
    *       ClassnameID1, AttributeID1, AttributeID2, .... AttributeIDn2, 0
    *          .                                      .
    *          .                                      .
    *          .                                      .
    *       ClassnameID1, AttributeID1, AttributeID2, .... AttributeIDnk, 0
    *       NULL
    *
    * where each is a 32-bit quantity.  Each of the ClassnameIDs are
    * pointers into the ops_symbols table and give us the link
    * to the corresponding name in the symbol table where we will
    * store a pointer to its list of AttributeIDs. The AttributeIDs are
    * also pointers into the ops_symbols table that similarly give access
    * to their symbol table entries. We use these lists in order to print
    * wmes in the standard OPS5 format of "^attribute_name attribute_value".
    * The compiler has put the attributes in increasing order based on
    * binding index with any vector attribute last in the list for a class.
    */

   intptr = LiteralizeLists;
   while (*intptr)
     {
      psym = ops_symid_lookup(val2sym(*(*intptr)));
      if (psym == NULL)
        {
	 fprintf(fp_err, "find_lit_lists: bad LiteralizeLists table.\n");
	 ops_fatal("Class name in table not found in symbol table.\n");
	}

      if (show_debug)  fprintf(fp_dbug, "find_lit_lists: %s is a class name.\n", psym->SymName);

      psym->AttPtrList = (int *)++intptr;  /* Store away the head of the att ptr list in  */
                                           /* the symbol table entry for the class name. */

      while (*intptr++ != NULL)  ;     /* Find end of this att ptr list (marked by 0). */

      /* That leaves us pointing to the next class name, unless
       * we're at the end of the table. In that case we will be
       * looking at a NULL.
       */
     }
}






static void
find_vecatt_list()
{
   int    **intptr;
   symptr psym;

   /* The table of vector attributes is arranged as follows:
    *
    *       VectorAttID1, VectorAttID1, VectorAttID2, .... VectorAttIDn, 0
    *
    * where each is a 32-bit quantity.  Each of the VectorAttIDs are
    * pointers into the ops_symbols table and give us the link
    * to the corresponding name in the symbol table where we will
    * mark the name as a vector attribute.
    */

   intptr = VectorAttList;
   while (*intptr)
     {
      psym = ops_symid_lookup(val2sym(*(*intptr)));
      if (psym == NULL)
        {
	 fprintf(fp_err, "find_vecatt_list: bad VectorAttList table.\n");
	 ops_fatal("Vector attribute name in table not found in symbol table.\n");
	}

      if (show_debug)  fprintf(fp_dbug, "find_vecatt_list: %s is a vector attribute.\n", psym->SymName);

      psym->is_vecatt = TRUE;  /* Mark it as a vector attribute. */
      intptr++;
     }
}




static void
do_matches(psym, nodeids, nodecnt)
   symptr psym;
   int    nodeids[];
   int    nodecnt;
{
   int i;

   fprintf(fp_out, "MATCHES for Production: %s\n", psym->SymName);

   if (nodecnt > 0)
     {
      /* Print WMEs that match each CE.
       */
      for (i = 0; i < nodecnt; i++)
        {
         if (i == 0)
           {
            print_lmem(nodeids[0], 1);
	   }
         fprintf(fp_out, "\n(CE_%d):\n", i+2);
         print_rmem(nodeids[i]);
        }
     }

   if (nodecnt > 1)
     {
      for (i = 1; i < nodecnt; i++)  print_lmem(nodeids[i], i+1);
     }

   fprintf(fp_out, "\nConflict Set (refracted instantiations marked):\n");
   ops_print_cs(fp_out, psym);

   fprintf(fp_out, "\n");
}



static void
print_rmem(nid)
   register int nid;
{
   register alpha_cell *aptr;
   register int i, j;


   for (i=0; i < MEM_HASHTABLE_SIZE; i++)
     {
      aptr = rtokHT[i];
      while (aptr)
	{
         if (aptr->nodeid == nid)
            fprintf(fp_out, "   (%d)\n", aptr->pWme->wme[WMETIMETAG]);
	 aptr = aptr->next;
	}
     }
}



static void
print_lmem(nid, tokensize)
   register int nid, tokensize;
{
   register beta_cell *bptr;
   register int i, j;

   fprintf(fp_out, "\n(CE_1");
   for (i=2; i<=tokensize; i++)  fprintf(fp_out, " CE_%d", i);
   fprintf(fp_out, "):\n");

   for (i=0; i < MEM_HASHTABLE_SIZE; i++)
     {
      bptr = ltokHT[i];
      while (bptr)
	{
         if (bptr->nodeid == nid)
	   {
	    fprintf(fp_out, "   (");
	    for (j=0; j<tokensize; j++)
	      {
	       if (j>0)  fprintf(fp_out, " ");
	       if (bptr->token[j])
	          fprintf(fp_out, "%d", bptr->token[j]->wme[WMETIMETAG]);
	       else
		  fprintf(fp_out, "nil");
	      }
	    fprintf(fp_out, ")\n");
	   }
	 bptr = bptr->next;
	}
     }
}




static void
fill_result_wme()
{
   string128 str;
   int       len;

   special_getstring(str);
   if ((len = strlen(str)) == 0)  return;

   for (;;)
     {
      if (str[0] == '^')
        {
         if (LastInputChar != '\n')  special_getstring(str);  else  return;
	 if ((len = strlen(str)) == 0)  return;
         if (str[0] == '^')
           {
            fprintf(fp_err, "** Bad input in rhs pattern: %s is not a valid attribute name.\n", str);
            FoundCmdError = TRUE;
            return;
	   }
	 top_tab(str);
	}
      else
         top_wme(str);

      if (FoundCmdError)  return;
      if (LastInputChar != '\n')  special_getstring(str);  else  return;
      if ((len = strlen(str)) == 0)  return;
     }
}



static void
top_tab(str)
   string str;
{
   int n;

   if (is_int(str))
     {
      n = atoi(str);
      top_tab_num(n);
     }
   else
      top_tab_att(strip_quotes(str));
}



static void
top_tab_att(str)
   string str;
{
   symptr psym;

   psym = ops_symname_lookup(str);
   if (psym != NULL)
     {
      if (psym->OpsBind != INVALIDBINDING)
        {
	 ops_settab(psym->OpsBind);
	 return;
	}
     }

   fprintf(fp_err, "** %s is not a valid attribute name.\n", str);
   FoundCmdError = TRUE;
}


static void
top_tab_num(n)
   int n;
{
   if ((n < WMEMINNDX) || (n > WMEMAXNDX))
     {
      fprintf(fp_err, "** WME attribute index = %d is out of range.\n", n);
      FoundCmdError = TRUE;
     }
   else
      ops_settab(n);
}


static void
top_wme(str)
   string str;
{
   int n;

   if (is_int(str))
     {
      n = atoi(str);
      top_wme_num(n);
     }
   else
      top_wme_atom(strip_quotes(str));
}


static void
top_wme_num(n)
   int n;
{
   /* HACK for "ppwm". */
   if (CurrentCmd == CMD_PPWM)  RefArray[NxtNdx] = TRUE;

   ops_value(int2val(n));
}


static void
top_wme_atom(str)
   string str;
{
   symptr psym;
   string s;

   psym = ops_symname_lookup(str);

   if (psym == NULL)
     {
      /* Allocate permanent storage for the symbol name and
       * copy the str over. Then enter the symbol in the symbol table.
       */
      s = (string) ops_malloc(strlen(str) + 1);
      strcpy(s, str);

      psym = ops_new_symbol(s);
     }

   /* HACK for "ppwm". */
   if (CurrentCmd == CMD_PPWM)  RefArray[NxtNdx] = TRUE;

   ops_value(sym2val(psym->SymId));
}

static
string
strip_quotes(str)   /* Modifies the string in place. Should call only once for a given string */
   string str;      /* since, for example, |"Hello"| intends the string to be "Hello".        */
{
   char q;
   int  len, i;

   q = str[0];
   if ((q == '"') || (q == '|'))
     {
      len = strlen(str);
      if (str[len-1] == q)
        {
         len = len - 2;    /* Size after quotes are stripped. */
	 for (i=0; i<len; i++)  str[i] = str[i+1];
	 str[len] = '\0';
	}
     }

   return(str);
}




static void
top_openfile()
{
   OpsVal fatom, dir, fname;
   string filename, filetype;
   FILE   *fileptr;
   int    access;

   if (TmpWme[WMELENGTH] < 3)
     {
      fprintf(fp_err, "** Openfile action requires 3 args.\n");
      FoundCmdError = TRUE;
      return;
     }

   fatom = TmpWme[1];
   fname = TmpWme[2];
   dir   = TmpWme[3];

   if ((!symbolp(fatom)) || (!symbolp(fname)) || (!symbolp(dir)))
     {
      fprintf(fp_err, "** Openfile args must be symbolic atoms.\n");
      FoundCmdError = TRUE;
      return;
     }

   if (fatom == symnil)
     {
      fprintf(fp_err, "** First arg to openfile must not be nil.\n");
      FoundCmdError = TRUE;
      return;
     }

   if (dir ==symin)
     {
      filetype = "r";
      access = READ_ACCESS;
     }
   else if (dir == symout)
     {
      filetype = "w";
      access = WRITE_ACCESS;
     }
   else
     {
      fprintf(fp_err, "** Filetype arg to openfile must be in or out.\n");
      FoundCmdError = TRUE;
      return;
     }

   filename = ops_pname(fname);
   if ((fileptr = fopen(filename, filetype)) == NULL)
     {
      fprintf(fp_err, "** File specified could not be accessed.\n");
      FoundCmdError = TRUE;
      return;
     }

   top_add_port(fileptr, filename, access, fatom);
}



static void
top_add_port(fileptr, filename, access, name)
   FILE   *fileptr;
   string  filename;
   int     access;
   OpsVal  name;
{
   int port, i;

   /* See if the name is already entered with a file association.
    * If it is, then the user neglected to close the file
    * associated with this symbolic name. If so, we'll print a
    * warning below and just reuse the entry.
    */
   port = lookup_port(name, ANY_ACCESS);

   if (port == NOT_A_PORT)
     {
      if (PortsLeft)
        {
	 i = 0;
	 while (PortFile[i].FilePtr) i++;

	 PortFile[i].FilePtr    = fileptr;
	 PortFile[i].FileName   = filename;
	 PortFile[i].AccessType = access;
	 PortFile[i].OpsName    = name;
	 PortFile[i].Position   = 0;

	 PortsLeft--;
	}
      else
        {
         fprintf(fp_err, "** No ports left to allocate.\n");
	 FoundCmdError = TRUE;
	 return;
        }
     }
   else
     {
      fprintf(fp_err, "** Warning: File previously associated with name was not closed.\n");

      PortFile[port].FilePtr    = fileptr;
      PortFile[port].FileName   = filename;
      PortFile[port].AccessType = access;
      PortFile[port].Position   = 0;
     }
} 



static void
top_closefile()
{
   OpsVal fatom;
   int    numfiles, i, port;

   numfiles = TmpWme[WMELENGTH];
   if (numfiles < 1)
     {
      fprintf(fp_err, "** No file args specified to closefile.\n");
      return;
     }

   for (i = numfiles; i > 0; i--)
     {
      fatom = TmpWme[i];
      if (!symbolp(fatom))
        {
         fprintf(fp_err, "** Args to closefile must be symbolic atoms.\n");
         FoundCmdError = TRUE;
         return;
        }
      else
        {
	 if (fatom == symnil)  continue;  /* Just skip over nil args. */

	 port = lookup_port(fatom, ANY_ACCESS);
	 if (fclose(PortFile[port].FilePtr) == EOF)
           {
            fprintf(fp_err, "** System error: stream could not be closed.\n");
            FoundCmdError = TRUE;
            return;
           }

	 remove_port(port);
	}
     }
}



static void
top_default()
{
   OpsVal fatom, WhichDefault;
   int    access, port;

   if (TmpWme[WMELENGTH] < 2)
     {
      fprintf(fp_err, "** Default action requires 2 args.\n");
      FoundCmdError = TRUE;
      return;
     }

   fatom        = TmpWme[1];
   WhichDefault = TmpWme[2];

   if (WhichDefault == symaccept)
      access = READ_ACCESS;
   else if ((WhichDefault == symwrite) || (WhichDefault == symtrace))
      access = WRITE_ACCESS;
   else
     {
      fprintf(fp_err, "** Default type must be accept, write, or trace.\n");
      FoundCmdError = TRUE;
      return;
     }

   if (!symbolp(fatom))
     {
      fprintf(fp_err, "** Default file arg must be a symbolic atom.\n");
      FoundCmdError = TRUE;
      return;
     }
   else
     {
      port = lookup_port(fatom, access);
      if (port == NOT_A_PORT)
        {
         fprintf(fp_err, "** Default file arg not asssociated with a port.\n");
         FoundCmdError = TRUE;
         return;
        }
      else if (port == WRONG_PORT_TYPE)
        {
         fprintf(fp_err, "** Default file not not opened for this type of access.\n");
         FoundCmdError = TRUE;
         return;
        }
      else
        {
	 if (WhichDefault == symaccept)
	    PortDefaultRead = port;
	 else
	    PortDefaultWrite = port;
	}
     }
}




static void
show_pbreaks()
{
   pbreak_ptr pbptr;

   if ((pbptr = PBreakList) == NULL)
     {
      fprintf(fp_out, "No production breakpoints set.\n");
      return;
     }

   while (pbptr)
     {
      fprintf(fp_out, "%s\n", pbptr->psym->SymName);
      pbptr = pbptr->next;
     }
}



static void
pbreak_toggle(sym)
   symptr sym;
{
   pbreak_ptr ptr, prev;

   if (sym->pbreak)
     {
      sym->pbreak = FALSE;   /* toggle it off */
      PBreakCount--;
      prev = NULL;
      ptr = PBreakList;
      while (ptr)
        {
	 if (ptr->psym == sym)
	    break;
	 else
	   {
	    prev = ptr;
	    ptr = ptr->next;
	   }
	}
      if (ptr)
        {
	 if (prev)  prev->next = ptr->next;  else  PBreakList = ptr->next;
	 ptr->next = PBreakFreeList;
	 PBreakFreeList = ptr;
	}
      else
        {
	 /* This should not occur. */
         fprintf(fp_err, "** System error in maintaining pbreak list, ignoring it.\n");
	}
     }
   else
     {
      sym->pbreak = TRUE;  /* toggle it on */
      PBreakCount++;
      if (PBreakFreeList)
        {
	 ptr = PBreakFreeList;
	 PBreakFreeList = ptr->next;
	}
      else
        {
	 if ((ptr = (pbreak_rec *) malloc(sizeof(pbreak_rec))) == NULL)
	    ops_fatal("rhsrtn.c: pbreak_toggle: Malloc failed, system out of memory.\n");
	}
      ptr->psym = sym;
      ptr->next = PBreakList;
      PBreakList = ptr;
     }
}



static
symptr
test_pbreak(ruleid)
   OpsVal  ruleid;
{
   pbreak_ptr pbptr;
   symptr     sym;

   sym = ops_symid_lookup(val2sym(ruleid));

   pbptr = PBreakList;
   while (pbptr)
     {
      if (pbptr->psym == sym)
         return(sym);
      else
         pbptr = pbptr->next;
     }

   return(NULL);
}
 


void
ops_run_stats(fp_x)
   FILE *fp_x;
{
   fprintf(fp_x, "Productions Fired: %d   RHS Actions Performed: %d   Working Memory Changes: %d\n",
                  fire_cnt, Num_RHS_Actions, Num_WM_Changes);
}
