
/****************************************************************************
 *
 * MODULE:  psm_locks.c
 *
 ****************************************************************************
 *
 * Abstract:
 *    Routines to allocate and initialize locks. Lock operations are defined
 *    in the module "psm_locks.h" as machine dependent macros. The macros
 *    define procedure calls which invoke hand-coded assembly routines to
 *    implement the locks. These routines are in modules of the form
 *    <machine-type>_locks.s.
 *
 *    This module also hides the definition and implementation of the locks
 *    associated with the hash table based token memories (ltokHT, rtokHT,
 *    ltokHTconj, rtokHTconj). The definitions of these locks (formerly
 *    defined globally as tokHTlock and ManyRead_1Write_Cnt) are held
 *    private to this module. Callers seeking to lock hash table buckets
 *    will simply invoke the macros defined in module "psm_locks.h", supplying
 *    the bucket number as a parameter.
 *
 ****************************************************************************
 *
 * CParaOPS5
 * Change Log:
 *    29 Sep 89 V5.3  Dirk Kalp
 *                    Provided conditional compilation variable to allow
 *                    selection of simple or complex stlye hash table locks.
 *                    Complex locks are the default.
 *                    Also added some documentation.
 *    16 Aug 89 V5.0  Anurag Acharya
 *                    Integrated the uniprocessor version
 *    15 Aug 89       Anurag Acharya
 *                    Enclosed the entire file in a conditional compilation
 *                    block (#ifndef UNIPROC_VERSION)
 *    10 Aug 89 V4.0  Dirk Kalp
 *                    Update for CParaOPS5 Version 4.0 P4.3.
 *                    Prefix exported routines with "ops_".
 *    15 May 89 V3.0  Dirk Kalp
 *                    Create for CParaOPS5 Version 3.0 P4.2.
 *
 * Copyright (c) 1986, 1987, 1988, 1989 Carnegie-Mellon University
 * All rights reserved.  The CMU software License Agreement
 * specifies the terms and conditions for use and redistribution.
 *
 ****************************************************************************/

/* The entire file is enclosed within this conditional compilation
 * so that none of it is included when the uniprocessor version is
 * created. The uniprocessor version is completely machine independent
 * since it does not use locks.
 */
#ifndef UNIPROC_VERSION

#include "psm_locks.h"
#include "version.h"



/* Exported routines:
 *    LockPtr ops_alloc_and_init_lock()
 *    void    ops_alloc_and_init_hashtable_locks(HT_size)
 */



/* Imported Routines:
 *    From utility.c:
 *       ops_malloc
 */



/* External Routines:
 *    These routines from other modules return values other than the
 *    standard integer and so their return types are declared here
 *    for routines in this module that call them.
 */
extern char   *ops_malloc();      /* Imported from utility.c. */




LockPtr
ops_alloc_and_init_lock()
/*---------------------------------------------------------------------------
 *
 * Abstract:
 *    Allocate a lock in shared memory and initialize it to open.
 *
 * Parameters:
 *    None.
 *
 * Environment:
 *    All locks should be allocated and initialized at system startup before
 *    any match processes are forked.
 *
 * Calls:
 *    "ops_malloc" in "utility.c".
 *
 * Returns:
 *    A pointer to the lock.
 *
 * Called by:
 *    "ops_init_conflict_set" in "conres.c".
 *    "ops_init_task_queue" and "ops_init_tok_mems" in "match.c".
 *    "ops_init_profile" in "utility.c".
 *
 *-------------------------------------------------------------------------*/
{
   LockPtr lockptr;

   lockptr = (LockPtr) ops_malloc(LOCKSIZE);
   *lockptr = 0;
   return(lockptr);
}



#ifdef COMPLEX_HASH_TABLE_LOCKS
int     *LeftRightCounter;          /* Points to an array of counters, 1 per       */
#endif                              /* each token hash table bucket line. A single */
				    /* counter is a 32 bit quantity. The left      */
				    /* most 16 bits are used to control access     */
				    /* to the left hash bucket, the right most     */
				    /* 16 control the right token memory bucket.   */

 char    *tokHTlock;                /* Points to an array of locks used to               */
                                    /* control access to the hash table buckets.         */
				    /* For complex hash table locks, the locks           */
                                    /* are used in conjucntion with the LeftRightCounter */
				    /* array to control access. For simple style         */
				    /* locks, the locks directly provide exclusive       */
				    /* access to a line in the hash tables.              */


void
ops_alloc_and_init_hashtable_locks(HT_size)
   int HT_size;
/*---------------------------------------------------------------------------
 *
 * Abstract:
 *    Allocate an array of locks in shared memory and initialize them to open.
 *    These locks are used to control access to the hash table buckets that
 *    store tokens from the Rete network. Each lock is associated with a
 *    single hash table "line". A "line" is a pair of corresponding buckets
 *    (buckets with the same hash index) from the left and right hash tables.
 *    For complex hash table locks that implement a multiple-reader-single-
 *    writer access to a given line, the locks are used in conjunction with
 *    the LeftRightCounter array to implement the protocol. For simple hash
 *    table locks that provide exclusive access to a given line, the locks are
 *    just the same simple spin-locks which are used in the rest of the
 *    system for accessing shared data and synchronization. The 2 style of
 *    locks are described in more detail with the LOCKING ROUTINES in module
 *    "psm_locks.h".
 *
 * Parameters:
 *    HT_size - the number of hash table buckets
 *
 * Environment:
 *    All locks should be allocated and initialized at system startup before
 *    any match processes are forked.
 *
 * Calls:
 *    "ops_malloc" in "utility.c".
 *
 * Returns:
 *    Nothing.
 *
 * Called by:
 *    "ops_init_tok_mems" in "match.c".
 *
 *-------------------------------------------------------------------------*/
{
   int i;

   tokHTlock        = (char *) ops_malloc(HT_size * sizeof(char));
#ifdef COMPLEX_HASH_TABLE_LOCKS
   LeftRightCounter = (int *)  ops_malloc(HT_size * sizeof(int));
#endif

   for (i=0; i<HT_size; i++)
     {
      tokHTlock[i]        = 0;
#ifdef COMPLEX_HASH_TABLE_LOCKS
      LeftRightCounter[i] = 0;
#endif
     }
}


#endif   /* not UNIPROC_VERSION */
