
/****************************************************************************
 *
 * MODULE:  match.c
 *
 ****************************************************************************
 *
 * Abstract:
 *    Routines needed by the Rete match assembly code to add/delete elements
 *    from node memories, step through node memories, build new tokens, and
 *    add/delete production instantiations to/from the conflict set.
 *
 ****************************************************************************
 *
 * CParaOPS5
 * Change Log:
 *    16 Aug 89 V5.0  Anurag Acharya
 *                    Integrated the uniprocessor version
 *    15 Aug 89       Anurag Acharya
 *                    Enclosed all the lock initializations in conditional
 *                    compilation blocks (#ifndef UNIPROC_VERSION)
 *    10 Aug 89 V4.0  Dirk Kalp
 *                    Merged with ParaOPS5 4.3.
 *    15 May 89 V3.0  Dirk Kalp
 *                    Use new lock macros from psm_locks.h.
 *    12 May 89 V2.0  Dirk Kalp
 *                    Create CParaOPS5 from ParaOPS5 4.2.
 *
 ****************************************************************************
 *
 * ParaOPS5
 * Change Log:
 *    25 Jun 89 V4.3  Dirk Kalp
 *                    Change global routine names to have "ops_" prefix. This
 *                    is to prevent conflicts with system and user defined
 *                    names at link time.
 *    24 Oct 88 V4.0  Dirk Kalp
 *                    Release of ParaOPS5 Version 4.0.
 *    23 Oct 88 V3.2  Dirk Kalp
 *                    Modified format of watch output and printing of wmes.
 *                    Add use of Num_WM_Changes to count changes to working memory.
 *     7 Oct 88 V3.1  Dirk Kalp
 *                    Added "ops_wakeup_matchers" call to "ops_match" routine. This is
 *                    where it should be called, not in "mra_cycle".
 *    13 Aug 88 V3.0  Dirk Kalp
 *                    Use fprintf instead of printf. Use global free lists
 *                    for alpha and beta cells. Add routines "ops_reinit_tok_mems",
 *                    "bfree", and "afree" to handle system reinitialization
 *                    and reclaim storage for alpha and beta cells in hash
 *                    tables. Convert "pnameid" parameter of "ops_term" routine
 *                    to its dereferenced form (corresponding change to
 *                    ../lhs/gencode.c for OPS5 compiler).
 *    25 May 88 V2.0  Dirk Kalp
 *                    Updated to consolidate Vax and Encore versions.
 *    31 Jul 86
 *    27 Jul 86
 *    23 Jul 86
 *    18 Jul 86
 *    15 Jul 86 V1.0  Dirk Kalp
 *
 * Copyright (c) 1986, 1987, 1988, 1989 Carnegie-Mellon University
 * All rights reserved.  The CMU software License Agreement
 * specifies the terms and conditions for use and redistribution.
 *
 ****************************************************************************/


#include "global.h"



/* Exported routines:
 *    void  ops_init_task_queue()
 *    void  ops_init_tok_mems()  
 *    void  ops_reinit_tok_mems()
 *    void  ops_match(elem, dir)
 *    void  ops_term(cecount, spcount, pnameId, rhsaddr, lpart, rpart)
 *    void  ops_TaskQueueOverflow(n)
 *
 */



/* Imported Routines:
 *    From utility.c:
 *       ops_malloc
 *    From gensymbol.c:
 *       ops_symid_lookup
 *    From wmemory.c:
 *       ops_write_atom
 *       ops_write_wme
 *    From rhsrtn.c:
 *       ops_wakeup_matchers
 *    From conres.c:
 *       ops_modifycs
 *    From assembly module:
 *       ops_eval_rete
 *    From psm_locks.c:
 *       ops_alloc_and_init_lock
 *       ops_alloc_and_init_hashtable_locks
 */



/* External Routines:
 *    These routines from other modules return values other than the
 *    standard integer and so their return types are declared here
 *    for routines in this module that call them.
 */
extern char   *ops_malloc();          /* Imported from utility.c. */
extern symptr  ops_symid_lookup();    /* Imported from gensymbol.c. */
extern void   *ops_write_wme();       /* Imported from wmemory.c. */
extern void   *ops_write_atom();      /* Imported from wmemory.c. */
extern void   *ops_modifycs();        /* Imported from conres.c. */
extern void    ops_wakeup_matchers(); /* Imported from rhsrtn.c. */

#ifndef UNIPROC_VERSION
extern LockPtr ops_alloc_and_init_lock();            /* Imported from psm_locks.c. */
extern void    ops_alloc_and_init_hashtable_locks(); /* Imported from psm_locks.c. */
#endif


/* Forward Declarations:
 *    These routines return values other than the standard integer and
 *    their return types are given here for other routines in this module
 *    that call them before they are defined.
 */
/* There are none. */


extern boolean RotateQueue;

extern int Num_WM_Changes;           /* Counts adds/deletes to working memory. */



void
ops_init_task_queue()
{
   int i;

   BitVector  = (int *)      ops_malloc(PointerSize);
  *BitVector  = 0;

#ifndef UNIPROC_VERSION
   tb_lock    = ops_alloc_and_init_lock();
   if (show_debug) printf("MATCH.C: *tb_lock = 0x%x\n", *tb_lock);
#endif

   QueueSize = TASK_QUEUE_SIZE / NumQueues;

   for (i=0; i<NumQueues; i++)
     {
      tq_index[i]   = (int *)      ops_malloc(PointerSize);
      task_queue[i] = (tq_cell **) ops_malloc(QueueSize * PointerSize);

#ifndef UNIPROC_VERSION
      tq_lock[i]    = ops_alloc_and_init_lock();
      if (show_debug) printf("MATCH.C: tq_lock[%d] = 0x%x *tq_lock[%d] = 0x%x\n", i, tq_lock[i], i, *tq_lock[i]);
#endif

      *tq_index[i] = -1;
     }

   Global_Free_TQcell  = (tq_cell **) ops_malloc(PointerSize);

#ifndef UNIPROC_VERSION
   Free_TQcell_Lock    = ops_alloc_and_init_lock();
#endif

   *Global_Free_TQcell = NULL;
}


void
ops_init_tok_mems()  
{
   register i;

   ltokHT     = (beta_cell **)  ops_malloc(MEM_HASHTABLE_SIZE * PointerSize);
   ltokHTconj = (beta_cell **)  ops_malloc(MEM_HASHTABLE_SIZE * PointerSize);
   rtokHT     = (alpha_cell **) ops_malloc(MEM_HASHTABLE_SIZE * PointerSize);
   rtokHTconj = (alpha_cell **) ops_malloc(MEM_HASHTABLE_SIZE * PointerSize);

#ifndef UNIPROC_VERSION
   ops_alloc_and_init_hashtable_locks(MEM_HASHTABLE_SIZE);
#endif

   Global_Free_4_Bcell   = (beta_cell **)  ops_malloc(PointerSize);
   Global_Free_32_Bcell  = (beta_cell **)  ops_malloc(PointerSize);
   Global_Free_128_Bcell = (beta_cell **)  ops_malloc(PointerSize);
   Global_Free_256_Bcell = (beta_cell **)  ops_malloc(PointerSize);
   Global_Free_Acell     = (alpha_cell **) ops_malloc(PointerSize);

#ifndef UNIPROC_VERSION
   Free_4_Bcell_Lock     = ops_alloc_and_init_lock();
   Free_32_Bcell_Lock    = ops_alloc_and_init_lock();
   Free_128_Bcell_Lock   = ops_alloc_and_init_lock();
   Free_256_Bcell_Lock   = ops_alloc_and_init_lock();
   Free_Acell_Lock       = ops_alloc_and_init_lock();
#endif

   *Global_Free_4_Bcell   = NULL;
   *Global_Free_32_Bcell  = NULL;
   *Global_Free_128_Bcell = NULL;
   *Global_Free_256_Bcell = NULL;
   *Global_Free_Acell     = NULL;

   for(i= MEM_HASHTABLE_SIZE - 1; i >= 0; i--)
     {
      ltokHT[i]     = NULL;
      ltokHTconj[i] = NULL;
      rtokHT[i]     = NULL;
      rtokHTconj[i] = NULL;
     }
}



void
ops_reinit_tok_mems()
{
   register i;

   /* Put tokens onto the alpha and beta global free lists.
    */
   for(i= MEM_HASHTABLE_SIZE - 1; i >= 0; i--)
     {
      bfree(&ltokHT[i]);
      bfree(&ltokHTconj[i]);
      afree(&rtokHT[i]);
      afree(&rtokHTconj[i]);
     }
}


static
bfree(bptr)
   beta_cell **bptr;
{
   register beta_cell *ptr, *nptr;

   ptr = *bptr;  *bptr = NULL;
   while (ptr)
     {
      nptr = ptr->next;

      if (ptr->toksize <= 4)
        {
	 ptr->next = *Global_Free_4_Bcell;
         *Global_Free_4_Bcell = ptr;
	}
      else if (ptr->toksize <= 32)
        {
	 ptr->next = *Global_Free_32_Bcell;
         *Global_Free_32_Bcell = ptr;
	}
      else if (ptr->toksize <= 128)
        {
	 ptr->next = *Global_Free_128_Bcell;
         *Global_Free_128_Bcell = ptr;
	}
      else
        {
	 ptr->next = *Global_Free_256_Bcell;
         *Global_Free_256_Bcell = ptr;
	}

      ptr = nptr;
     }

}


static
afree(aptr)
   alpha_cell **aptr;
{
   register alpha_cell *ptr;

   if (*aptr == NULL)  return;

   ptr = *aptr;
   while (ptr->next)  ptr = ptr->next;

   ptr->next = *Global_Free_Acell;
   *Global_Free_Acell = *aptr;

   *aptr = NULL;
}




void
ops_match(elem, dir)
   wmeptr elem;
   int    dir;
/*---------------------------------------------------------------------------
 *
 * Abstract:
 *    Invokes the Rete matcher.
 *
 * Parameters:
 *    elem - the wme to be added or deleted as a result of a MAKE, MODIFY,
 *           or REMOVE RHS action.
 *    dir  - DIRIN or DIROUT for the wme 'elem'.
 *
 * Environment:
 *    The Rete match assembly code will need to access the variables
 *    'ops_target', 'curdir', 'succdir', .....
 *
 * Calls:
 *    "ops_write_wme" in "wmemory.c".
 *    "ops_wakeup_matchers" in "rhsrtn.c".
 *    "ops_eval_rete" in the Rete match assembly code.
 *
 * Called by:
 *    "ops_deltarget" and "ops_addtarget" in "wmemory.c".
 *
 *-------------------------------------------------------------------------*/
{
   curwme     = elem;
   curdir     = dir;
   succdir    = dir;
   ops_target = elem;  /* Name Rete knows for getting at curwme. */

   Num_WM_Changes++;   /* Count another change to working memory. */

   if (*watch >= 2)
     {
      if (dir == DIRIN)
         fprintf(fp_watch, "=>WM:  ");
      else
         fprintf(fp_watch, "<=WM:  ");
      ops_write_wme(fp_watch, elem);
      fflush(fp_watch);
     }


   /* Call Rete at the assembly code interface. The assembly code knows
    * about 'ops_target' and will make register R10 point to it.
    */
   if (! RotateQueue) QueueSelect = 0;    /* For now, queue all changes to task_queue[0] */
   ops_eval_rete();
/*   if (*watch >= 3) {fprintf(fp_watch, "RETURN FROM OPS_EVAL_RETE\n"); fflush(fp_watch);} */
/*           fprintf(fp_watch, "After call for ops_eval_rete, BitVector= 0x%x\n", *BitVector); */
/*	   dodebug(); */
   
   /* Right after the first real change to working memory is made, take
    * a time reading when we begin the match for the production being fired.
    * Also it is here that we wakeup the match processes to start processing
    * tasks from the task queue(s).
    */
   if ((!g_match_started) && (NumProc > 1))
     {
      getrusage(RUSAGE_SELF, g_time_match_beg);
      ops_wakeup_matchers();    /* This activates the match processes from their busy wait. */
      getrusage(RUSAGE_SELF, g_time_rhs_end);
     }
   g_match_started = TRUE;

}


void
ops_term(cecount, spcount, pnameId, rhsaddr, lpart, rpart)
   int    cecount;
   int    spcount;
   OpsVal pnameId;  /* Changed from *pnameId; gencode.c also changed */
   int    rhsaddr;  /* Changed from char *rhsaddr for CParaOps5. */
   beta_cell  *lpart;
   alpha_cell *rpart;
/*---------------------------------------------------------------------------
 *
 * Abstract:
 *    Add or delete an instantiation of a production to/from the conflict
 *    set.
 *
 * Parameters:
 *    cecount - the number of condition elements in the production.
 *    spcount - the specificity count for the production.
 *    pnameId - OpsVal representation for a symbol name of a production.
 *    rhsaddr - an index into the threaded code array, RHScode, of the start
 *              of the right-hand side actions for this production.
 *
 * Environment:
 *    Nothing special.
 *
 * Calls:
 *    "ops_symid_lookup" in "gensymbol.c".
 *    "ops_modifycs" in "conres.c".
 *
 * Called by:
 *    the Rete match assembly code.
 *
 *-------------------------------------------------------------------------*/
{
   symptr psym;
   string pname;
   int i;
   wmeptrvec  prod_ins;

   if (cecount == 1)
     {
/*	 asm("	movl	r8, _curwme");  */
	 prod_ins[0] = curwme;
     }
   else
     {
	 if (rpart == NULL) /* the last condition element was negated */
	   { prod_ins[cecount-1] = NULL;}
	 else
	   { prod_ins[cecount-1] = rpart->pWme;}
	 for (i = cecount-2; i >= 0;  i--)
	    prod_ins[i] = lpart->token[i];
     }
   
   if (*watch >= 3)
     {
      psym = ops_symid_lookup(val2sym(pnameId));
      pname = psym->SymName;
      if (curdir == DIRIN)
         fprintf(fp_watch, "=>CS:  ");
      else
         fprintf(fp_watch, "<=CS:  ");
      fprintf(fp_watch, "ProcessID=%d   %s: ", ProcessID, pname);
      for (i=0; i < cecount; i++)
         if (prod_ins[i] != NULL)
	    fprintf(fp_watch, "%d ", (prod_ins[i])->wme[WMETIMETAG]);
	 else
	    fprintf(fp_watch, "nil ");
      fprintf(fp_watch, "\n");
      fflush(fp_watch);
}
   
   ops_modifycs(pnameId,
            cecount,
            spcount,
            rhsaddr,
            prod_ins,
            curdir
           );
}




void
ops_TaskQueueOverflow(n)
   int n;
{
/* Routine called from assembly code routine PushTaskQueue to indicate
 * that the maximum limit of task queue, that is, TASK_QUEUE_SIZE is being
 * exceeded.
 */

   fprintf(fp_err, "Fatal Error:  Task Queue Number %d Overflow. \n", n);
   exit(1);

}




static
void
show_node_mems()
{
   int mcnt;
   int i, j;
   beta_cell *bptr;
   alpha_cell *aptr;
   wmeptr wptr;


   for (mcnt=0; mcnt < MEM_HASHTABLE_SIZE; mcnt++)
     {
      bptr = ltokHT[mcnt];
      if (bptr)
        {
	 fprintf(fp_dbug, "\t\tLMEM%d:\n", mcnt);
	 while (bptr)
	   {
	    fprintf(fp_dbug, "\t\t\trefcnt=%d  toksiz=%d  nodeid=%d\n", bptr->refcount, bptr->toksize, bptr->nodeid);
	    for (i=0; i<bptr->toksize; i++)
	      {
	       wptr = bptr->token[i];
	       fprintf(fp_dbug, "\t\t\t\tTok%d tag=%d len=%d wme: ", i, wptr->wme[WMETIMETAG],
	              wptr->wme[WMELENGTH]);
	       for (j=WMEMINNDX; j<=wptr->wme[WMELENGTH]; j++)  ops_write_atom(fp_dbug, wptr->wme[j]);
	       fprintf(fp_dbug, "\n");
	      }
	    bptr = bptr->next;
	   }
	}

      bptr = ltokHTconj[mcnt];
      if (bptr)
        {
	 fprintf(fp_dbug, "\t\tLMEM_CONJ%d:\n", mcnt);
	 while (bptr)
	   {
	    fprintf(fp_dbug, "\t\t\trefcnt=%d  toksiz=%d  nodeid=%d\n", bptr->refcount, bptr->toksize, bptr->nodeid);
	    for (i=0; i<bptr->toksize; i++)
	      {
	       wptr = bptr->token[i];
	       fprintf(fp_dbug, "\t\t\t\tTok%d tag=%d len=%d wme: ", i, wptr->wme[WMETIMETAG],
	              wptr->wme[WMELENGTH]);
	       for (j=WMEMINNDX; j<=wptr->wme[WMELENGTH]; j++)  ops_write_atom(fp_dbug, wptr->wme[j]);
	       fprintf(fp_dbug, "\n");
	      }
	    bptr = bptr->next;
	   }
	}



      aptr = rtokHT[mcnt];
      if (aptr)
        {
	 fprintf(fp_dbug, "\t\tRMEM%d:\n", mcnt);
	 while (aptr)
	   {
	       fprintf(fp_dbug, "\t\t\tnodeid=%d\n", aptr->nodeid);
	       wptr = aptr->pWme;
	       fprintf(fp_dbug, "\t\t\t\tWME tag=%d len=%d wme: ", wptr->wme[WMETIMETAG],
	              wptr->wme[WMELENGTH]);
	       for (j=WMEMINNDX; j<=wptr->wme[WMELENGTH]; j++)  ops_write_atom(fp_dbug, wptr->wme[j]);
	       fprintf(fp_dbug, "\n");
	       aptr = aptr->next;
	   }
	}

      aptr = rtokHTconj[mcnt];
      if (aptr)
        {
	 fprintf(fp_dbug, "\t\tRMEM_CONJ%d:\n", mcnt);
	 while (aptr)
	   {
	       fprintf(fp_dbug, "\t\t\tnodeid=%d\n", aptr->nodeid);
	       wptr = aptr->pWme;
	       fprintf(fp_dbug, "\t\t\t\tWME tag=%d len=%d wme: ", wptr->wme[WMETIMETAG],
	              wptr->wme[WMELENGTH]);
	       for (j=WMEMINNDX; j<=wptr->wme[WMELENGTH]; j++)  ops_write_atom(fp_dbug, wptr->wme[j]);
	       fprintf(fp_dbug, "\n");
	       aptr = aptr->next;
	   }
	}


     }
}

        

	      
static
void
show_node_bmem(bptr)
beta_cell *bptr;
{
   int i, j;
   alpha_cell *aptr;
   wmeptr wptr;


	 fprintf(fp_dbug, "\n ******* Printing Contents of LMEM starting address 0x%x  ****** \n", bptr);
	 while (bptr)
	   {
	    fprintf(fp_dbug, "\trefcnt=%d  toksiz=%d  nodeid=%d\n", bptr->refcount, bptr->toksize, bptr->nodeid);
	    for (i=0; i<bptr->toksize; i++)
	      {
	       wptr = bptr->token[i];
	       if (wptr != NULL)
	         {
	           fprintf(fp_dbug, "\t\tWME%d tag=%d len=%d wme: ", i, wptr->wme[WMETIMETAG],
	              wptr->wme[WMELENGTH]);
	           for (j=WMEMINNDX; j<=wptr->wme[WMELENGTH]; j++)  ops_write_atom(fp_dbug, wptr->wme[j]);
	           fprintf(fp_dbug, "\n");
		 }
	       else fprintf(fp_dbug, "\t\tWME%d,  nil \n", i);
	      }
	    bptr = bptr->next;
	   }

}

static
void
show_node_amem(aptr)
alpha_cell *aptr;
{
   int i, j;
   wmeptr wptr;

	 fprintf(fp_dbug, "\n ******* Printing Contents of RMEM starting address 0x%x  ****** \n", aptr);
	 while (aptr)
	   {
	       fprintf(fp_dbug, "\tnodeid=%d", aptr->nodeid);
	       wptr = aptr->pWme;
	       fprintf(fp_dbug, "\tWME tag=%d len=%d wme: ", wptr->wme[WMETIMETAG],
	              wptr->wme[WMELENGTH]);
	       for (j=WMEMINNDX; j<=wptr->wme[WMELENGTH]; j++)  ops_write_atom(fp_dbug, wptr->wme[j]);
	       fprintf(fp_dbug, "\n");
	       aptr = aptr->next;
	   }

}
