
/****************************************************************************
 *
 * MODULE:  gencode.c
 *
 ****************************************************************************
 *
 * Abstract:
 *    The routines in this module are used primarily to generate C
 *    code for the Rete network and the interface routine to the
 *    Rete code for "C" callers for the given production system. Also
 *    appended to the end of the code file here
 *    will be several data areas used to hold some information from the
 *    compile-time symbol table or to hold pointers to dynamically allocated
 *    data structures for the node memories of the Rete network.
 *
 *    Prior to generating the C code here, the code file will contain the
 *    "threaded" code array produced during the compilation of the productions.
 *    This "threaded" code is a form of intermediate code for the right-hand
 *    sides (RHS) of the productions and is executed in an interpreted
 *    manner to perform the RHS actions of a production selected for firing.
 *    Some of the low level routines in this module are also used in this
 *    compilation process to write the "threaded" code to the output file.
 *
 ****************************************************************************
 *
 * CParaOPS5
 * Change Log:
 *    29 Sep 89 V5.2 Anurag Acharya
 *                   modified write_ext_list() so that it only writes the
 *                   list of external routine names and does not 
 *                   generate the extern declarations. This was required as
 *                   the extern declarations are needed for every file and
 *                   need to be at the top while the list of routine names
 *                   needs to be at the end and msut be generated only once.
 *                   The functionality of generating the extern declarations
 *                   is taken over by "write_ext_funct_list()" in cmprhs.c.
 *    24 Sep 89      Anurag Acharya
 *                   fixed bug in "end_rete()". V5.1 had the the call to 
 *                   create_new_ofile() commented out. This resulted in 
 *                   -n switch becoming ineffective. now -n works again.
 *    30 Aug 89 V5.1 Anurag Acharya
 *                   Fixed a bug with external routines. Now the 
 *                   external routines are declared at the top of the file
 *                   and the routines are available for array intializations.
 *    25 Jun 89 V4.0 Dirk Kalp
 *                   Update with ParaOPS5 4.3.
 *    12 May 89 V2.0 Dirk Kalp
 *                   Create CParaOPS5 from ParaOPS5 4.2.
 *
 ****************************************************************************
 *
 * ParaOPS5
 * Change Log:
 *    25 Jun 89 V4.3 Dirk Kalp
 *                   Change all compiler generated routine names to have
 *                   "ops_" prefix. This is to ensure no conflicts with
 *                   system or user routine names when user programs are
 *                   linked.
 *    19 Apr 89 V4.2 Dirk Kalp
 *                   Added use of "codeoutsym" to pay attention to newline char
 *                   in quoted symbols.
 *    14 Feb 89 V4.1 Dirk Kalp
 *                   Fix bug in writing out ExternalList. Added routines
 *                   "write_lit_lists" and "write_vecatt_list" to support
 *                   standard OPS5 format for printing wmes.
 *    24 Oct 88 V4.0 Dirk Kalp
 *                   Release of ParaOPS5 Version 4.0.
 *    21 Oct 88 V3.3 Dirk Kalp
 *                   Fixed up use of data and text section pseudo ops.
 *     7 Oct 88 V3.2 Dirk Kalp
 *                   Improved MatcherCode routine at QueueLoop label.
 *     1 Oct 88 V3.1 Dirk Kalp
 *                   Added routines "write_nodeid_lists" and "write_ext_list"
 *                   to support OPS5 top level cmds "matches" and "call".
 *    13 Aug 88 V3.0 Dirk Kalp
 *                   Updated change to alpha and beta cell definitions
 *                   from ../rhs/global.h. Required changes to routines
 *                   gen_lhash, gen_rhash, left_join, gentest, gen_rfixcount,
 *                   gen_lfixcount, and gen_next_tok.
 *                   Updated pnode to provide term routine with the SymId
 *                   of the pname instead of a ptr to it.
 *                   Improved code generated for rfixcount and ltestcount routines.
 *    25 May 88 V2.0 Dirk Kalp
 *                   Updated to consolidate Vax and Encore versions.
 *                   WARNING: The assembler/linker on the Vax apparently does not
 *                            handle conditional branches to external labels properly.
 *                            See the comment below that precedes the declaration for
 *                            WriteLocalJump.
 *    17 Sep 86      Dirk Kalp
 *    12 Sep 86      Dirk Kalp
 *    23 Aug 86      Anoop Gupta
 *    22 Aug 86      Dirk Kalp
 *    13 Aug 86
 *     8 Aug 86
 *     7 Aug 86
 *     2 Aug 86      Anoop Gupta
 *    31 Jul 86      Dirk Kalp
 *    30 Jul 86
 *    10 Jul 86
 *    10 Jul 86
 *     9 Jul 86
 *     1 May 86
 *    29 Apr 86
 *    24 Apr 86
 *    21 Apr 86
 *    16 Apr 86
 *     4 Apr 86
 *     3 Apr 86
 *     1 Apr 86
 *    31 Mar 86
 *    26 Mar 86 V1.0  Dirk Kalp
 *                    Put together from previous version created by
 *                    Quei-Len Lee.
 *
 * Copyright (c) 1986, 1987, 1988, 1989 Carnegie-Mellon University
 * All rights reserved.  The CMU software License Agreement
 * specifies the terms and conditions for use and redistribution.
 *
 ****************************************************************************/

 
   
#include "defs.h"


/* Imported Routines:
 *    From ops5.c:
 *       opserror
 *       opswarn
 *       putstring
 *       puttab
 *       puteol
 *       bomb_out
 *
 *    From literal.c:
 *       lookup_sym_addr
 *
 *    From cmplhs.c:
 *       new_attlist_rec
 *
 *    From system:
 *       malloc
 */



/* External Routines:
 *    These routines from other modules return values other than the
 *    standard integer and so their return types are declared here
 *    for routines in this module that call them.
 */
extern sym_attptr new_attlist_rec();   /* Imported from cmplhs.c. */
extern symptr     lookup_sym_addr();   /* Imported from literal.c. */



/* Forward Declarations:
 *    These routines return values other than the standard integer and
 *    their return types are given here for other routines in this module
 *    that call them before they are defined.
 */
nodeptr      nextnode();
stkptr       newstkptr();
nodeptr      popts();
string       opp_test();


static boolean have_not_node;



init_gen_rete()
{

   flag_skip_son  = FALSE;
   flag_ri        = FALSE;
   have_not_node  = FALSE;
   
   freestack      = NULL;
   stacktree      = NULL;
   
   g_label        = 11;        /* First label will be 'L11'. */
   nodeid_cnt     = 0;
   g_num_rete_roots = 0;

}



gen_rete(from_cmp_prod)
boolean from_cmp_prod;
/*---------------------------------------------------------------------------
 *
 * Abstract:
 *    This routine generates code for the Rete network according to the
 *    type of node. Beginning at the ROOT_NODE, the network is traversed
 *    in a depth-first manner as described in the routine 'nextnode' which
 *    is called to select the next node we visit. 
 *
 * Parameters:
 *    None.
 *
 * Environment:
 *    The global 'stacktree' points to the stack used for backtracking.
 *
 * Calls:
 *    "nextnode", "tyya', "txxa", "tyyn", "txxn", "tyys", "txxs", "any",
 *    "beta", "ri", "root", "pnode", "end_rete", and "outeol" in this
 *    module.
 *    "opserror" and "putstring" in "ops5.c".
 *
 * Called by:
 *    "cmp_production" in "cmplhs.c" and "leave_main" in this module.
 *
 *-------------------------------------------------------------------------*/
{
   nodeptr node;


   outeol();
   node = g_root;

   while (node)
     {
      switch (node->nodename)
        {
         case TEQA: tyya(node->nodeinfo.tnode, TEST_EQ);    break;
         case TNEA: tyya(node->nodeinfo.tnode, TEST_NE);    break;
         case TXXA: txxa(node->nodeinfo.tnode);             break;
         case TEQN: tyyn(node->nodeinfo.tnode, TEST_EQ);    break;
         case TNEN: tyyn(node->nodeinfo.tnode, TEST_NE);    break;
         case TGTN: tyyn(node->nodeinfo.tnode, TEST_GT);    break;
         case TGEN: tyyn(node->nodeinfo.tnode, TEST_GE);    break;
         case TLTN: tyyn(node->nodeinfo.tnode, TEST_LT);    break;
         case TLEN: tyyn(node->nodeinfo.tnode, TEST_LE);    break;
         case TXXN: txxn(node->nodeinfo.tnode);             break;
         case TEQS: tyys(node->nodeinfo.tnode, TEST_EQ);    break;
         case TNES: tyys(node->nodeinfo.tnode, TEST_NE);    break;
         case TGTS: tyys(node->nodeinfo.tnode, TEST_GT);    break;
         case TGES: tyys(node->nodeinfo.tnode, TEST_GE);    break;
         case TLTS: tyys(node->nodeinfo.tnode, TEST_LT);    break;
         case TLES: tyys(node->nodeinfo.tnode, TEST_LE);    break;
         case TXXS: txxs(node->nodeinfo.tnode);             break;
         case ANY_NODE:  any(node->nodeinfo.anynode);       break;
         case AND_NODE:  beta(node);                        break;
         case NOT_NODE:  have_not_node = TRUE; beta(node);  break;
         case RI_NODE:   ri();                              break;
         case ROOT_NODE: root();                            break;
         case P_NODE:    pnode(node);                       break;
         default:   opserror("gen_rete: internal error ");
                    putstring("no such node name ");        break;
        }
      node = nextnode(node);
     }
   if (stacktree != NULL)
      printf("Stacktree is not empty, stacktree->lab %d\n", stacktree->lab);

   end_rete(from_cmp_prod);
}




nodeptr
nextnode(node)
   nodeptr node;
/*---------------------------------------------------------------------------
 *
 * Abstract:
 *    This routine selects the next node we will visit in the network. We
 *    traverse the network in a depth-first manner except whenever we visit
 *    a beta node via its right input. In this case, code will be generated
 *    for the right input of the beta node and all its right-brother beta
 *    nodes and we will skip the sons (and the nodes below them) of the
 *    beta nodes until we later visit the beta nodes via their left inputs.
 *    At that time, we will generate code for the left input to a beta node
 *    along with the code to perform variable binding tests for the beta
 *    node and proceed then in the normal depth-first manner to handle the
 *    nodes below the beta node.
 *
 * Parameters:
 *    node - the node that we last visited.
 *
 * Environment:
 *    Global flags 'flag_skip_son' and 'flag_ri' tell us how to pick the
 *    the next node and whether to push a brother of the next node onto
 *    the stack for backtracking.
 *
 * Returns:
 *    A pointer to the next node we will visit.
 *
 * Calls:
 *    "wrllabel", "popts", and "pushts" in this module.
 *
 * Called by:
 *    "gen_rete" in this module.
 *
 *-------------------------------------------------------------------------*/
{
   nodeptr n;

   /* Select the next node to visit in our 'depth-first' traverse of
    * the network.
    */
   if ((node->son != NULL) && (flag_skip_son == FALSE))
     {
      /* Normal depth-first traverse visits son of node next.
       */
      n = node->son;
     }
   else
     {
      /* If node has no son (i.e., we have a terminal P_NODE in the
       * network), then we backtrack as usual in the depth-first
       * traverse. Also we will backtrack if the 'flag_skip_son' flag
       * was set. This indicates that 'node' is a beta node which we
       * just visited via its right input and we've handled it in a
       * special way (re: 'beta' and 'right_beta' routines). In this
       * case, we've generated code for the right input of the beta
       * node along with all its right-brother beta nodes. Instead of
       * continuing on in the depth-first path from the beta node to
       * its son, we backtrack instead. Later we will again visit the
       * beta node (along with its right brothers) but via its left
       * input instead. At that time, we will generate code for the
       * left input along with the variable binding tests for the beta
       * node and then proceed onward in the normal depth-first manner
       * from the beta node.
       *
       * In both cases here we backtrack simply by popping the stack
       * holding the node to which we backtrack. But first we write the
       * backtrack label into the code stream and reset the 'flag_skip_son'
       * flag.
       */
      wrllabel(fp_root, stacktree->lab);
      flag_skip_son = FALSE;
      n = popts();
     }

   
   /* If the new node 'n' has a brother, we push the brother onto the
    * backtrack stack. But beta nodes, having 2 inputs and thus 2
    * brother chains, are handled a little differently than the normal
    * depth-first method. If we visit the beta node over its right input,
    * we will visit and process all the beta nodes in the right-brother
    * chain and short circuit the depth-first traverse as mentioned
    * above. Thus we don't ever push a right-brother node to the backtrack
    * stack. If we are visiting the beta node over its left input, then we
    * handle it like any other node pushing its left-brother (if one exists)
    * to the backtrack stack. The flag, 'flag_ri', tells us how we
    * arrived at the beta node. If it is set, then 'node' was an RI_NODE
    * and thus 'n' is being visited via its right input.
    */
   if (flag_ri == TRUE)
     {
      /* 'n' must be a beta node being visited over its right input so
       * don't push its right_brother to the backtrack stack.
       */
      /* Do nothing! */
     }
   else
     {
      /* 'n' is either a beta node being visited via its left input or else
       * some non-beta node so, acording to the usual depth-first method,
       * push its left-brother plus a new label for backtracking to the
       * stack.
       */
      if (n != NULL) 
         if (n->brothl)  pushts(n->brothl, nextlab());
     }

   return(n);
}




stkptr
newstkptr()
/*---------------------------------------------------------------------------
 *
 * Abstract:
 *    Allocate and return a new stack record.
 *
 * Parameters:
 *    None.
 *
 * Environment:
 *    Nothing special.
 *
 * Returns:
 *    A pointer to the new stack record.
 *
 * Calls:
 *    "malloc' system call.
 *    "opserror" and "puteol" in "ops5.c".
 *
 * Called by:
 *    "pushts" in this module.
 *
 *-------------------------------------------------------------------------*/
{
   stkptr ptr;

   if (freestack)
     {
      ptr = freestack;
      freestack = freestack->next;
     }
   else
     {
      if ((ptr = (stkrec *)malloc(sizeof(stkrec))) == NULL)
        {
         opserror("newstkptr: no more memory...");  puteol();
         bomb_out();
        }
      }

   ptr->node = NULL;
   ptr->lab  = 0;
   ptr->next = NULL;

   return(ptr);
}




pushts(node, label)
   nodeptr node;
   int     label;
/*---------------------------------------------------------------------------
 *
 * Abstract:
 *    Push a node from the network onto the stack used for backtracking in
 *    the depth-first traverse of the Rete network during code generation.
 *    Also associated with the node is the label assigned to the entry point
 *    of the code to be generated for the node when this node is later popped
 *    from the stack in a backtracking move. 
 *
 * Parameters:
 *    node  - the node to be pushed to the stack.
 *    label - the associated label.
 *
 * Environment:
 *    The global 'stacktree' points to the top of the stack.
 *
 * Calls:
 *    "newstkptr" in this module.
 *
 * Called by:
 *    "nextnode" and "root" in this module.
 *
 *-------------------------------------------------------------------------*/
{
   stkptr sp;
  
   /* Get a new stack record and fill it in.
    */
   sp = newstkptr();
   sp->node = node;
   sp->lab  = label;

   /* Push the new record onto the backtracking stack.
    */
   sp->next = stacktree;
   stacktree = sp;

   /*** showstk(sp, 1); ***/
}



nodeptr
popts()
/*---------------------------------------------------------------------------
 *
 * Abstract:
 *    Pop the top element from the backtracking stack to obtain the next
 *    node to be visited in the depth-first traverse of the Rete network.
 *
 * Parameters:
 *    None.
 *
 * Environment:
 *    The global 'stacktree' points to the top of the stack.
 *
 * Returns:
 *    A pointer to the node contained in the record popped from the stack.
 *
 * Calls:
 *    No one.
 *
 * Called by:
 *    "nextnode" in this module.
 *
 *-------------------------------------------------------------------------*/
{
   stkptr tmp;
  
   if (stacktree == NULL)
     {
      printf("Node stack underflow!!!\n");
      return(NULL);    /* Later change this to bomb_out() */
     }


   tmp = stacktree;
   /*** showstk(tmp, 0); ***/
   stacktree = stacktree->next;

   tmp->next = freestack;
   freestack = tmp;

   return(tmp->node);
} 



showstk(ptr, dir)
   stkptr ptr;
   int dir;
{
   int sadr;
   int nadr, xadr;

   sadr = (int) ptr;
   nadr = (int) ptr->node;
   xadr = (int) ptr->next;
   printf("%d Dir, ptr = %d, nodeptr = %d, label = %d, next = %d\n",
          dir, sadr, nadr, ptr->lab, xadr);
}




txxa(node)
   t_nodeptr node;
/*---------------------------------------------------------------------------
 *
 * Abstract:
 *    T_NODE tests if a field of a wme is of type STRING (i.e., is there
 *    an address stored at the field instead of an integer number.
 *
 *    The value stored in a field of a wme is always an integer or a pointer
 *    to a string constant. The value is left shifted by one bit and, if it's
 *    a pointer, the low order bit is set to distinguish it from an integer.
 *
 * Parameters:
 *    node - the T_NODE for which we're generating code.
 *
 * Environment:
 *    Vax register R8 holds the address of the current working memory
 *    element which the node will examine. Backtrack stack holds the code
 *    label to which the matcher backtracks if the node test fails.
 *
 * Calls:
 *    "codeouts", "codeoutl", "llabelname", and "outeol" in this module.
 *
 * Called by:
 *    "gen_rete" in this module.
 *
 *-------------------------------------------------------------------------*/
{
   fprintf(fp_root, "\tif (!symbolp(R8.WmePtr->wme[%d]))  ", node->pos);
   goto_fprintf(fp_root, stacktree->lab);   /* Backtrack jump on failure. */
}



txxn(node)
   t_nodeptr node;
/*---------------------------------------------------------------------------
 *
 * Abstract:
 *    T_NODE tests if a field of a wme is of type integer number.
 *
 *    The value stored in a field of a wme is always an integer or a pointer
 *    to a string constant. The value is left shifted by one bit and, if it's
 *    a pointer, the low order bit is set to distinguish it from an integer.
 *
 * Parameters:
 *    node - the T_NODE for which we're generating code.
 *
 * Environment:
 *    Vax register R8 holds the address of the current working memory
 *    element which the node will examine. Backtrack stack holds the code
 *    label to which the matcher backtracks if the node test fails.
 *
 * Calls:
 *    "codeouts", "codeoutl", "llabelname", and "outeol" in this module.
 *
 * Called by:
 *    "gen_rete" in this module.
 *
 *-------------------------------------------------------------------------*/
{
   fprintf(fp_root, "\tif (!numberp(R8.WmePtr->wme[%d]))  ", node->pos);
   goto_fprintf(fp_root, stacktree->lab);   /* Backtrack jump on failure. */
}



txxs(node)
   t_nodeptr node;
/*---------------------------------------------------------------------------
 *
 * Abstract:
 *    T_NODE tests if 2 fields of a wme are of the same type (i.e., integer
 *    or string). The fields are of the same type if the low order bit of
 *    each field is the same.
 *
 *    The value stored in a field of a wme is always an integer or a pointer
 *    to a string constant. The value is left shifted by one bit and, if it's
 *    a pointer, the low order bit is set to distinguish it from an integer.
 *
 * Parameters:
 *    node - the T_NODE for which we're generating code.
 *
 * Environment:
 *    Vax register R8 holds the address of the current working memory
 *    element which the node will examine. Backtrack stack holds the code
 *    label to which the matcher backtracks if the node test fails.
 *
 * Calls:
 *    "codeouts", "codeoutl", "llabelname", and "outeol" in this module.
 *
 * Called by:
 *    "gen_rete" in this module.
 *
 *-------------------------------------------------------------------------*/
{
   fprintf(fp_root, "\tif (difftypes(R8.WmePtr->wme[%d], R8.WmePtr->wme[%d]))  ",
                    node->pos, node->value.snum);
   goto_fprintf(fp_root, stacktree->lab);   /* Backtrack jump on failure. */
}



tyya(node, test)
   t_nodeptr node;
   int       test;
/*---------------------------------------------------------------------------
 *
 * Abstract:
 *    T_NODE tests a field of a wme against a string lhs-value. The test is
 *    either TEST_EQ or TEST_NE. The field of the wme is compared against an
 *    element of the 'ops_symbols' table which holds the modified form (i.e.,
 *    pointer left-shifted 1 with low bit set) of a pointer to a string.
 *
 *    The value stored in a field of a wme is always an integer or a pointer
 *    to a string constant. The value is left shifted by one bit and, if it's
 *    a pointer, the low order bit is set to distinguish it from an integer.
 *
 * Parameters:
 *    node    - the T_NODE for which we're generating code.
 *    testnot - the Vax conditional jump instruction that does a jump if the
 *              test fails (i.e., it has the opposite sense of the T_NODE
 *              test).
 *
 * Environment:
 *    Vax register R8 holds the address of the current working memory
 *    element which the node will examine. Backtrack stack holds the code
 *    label to which the matcher backtracks if the node test fails.
 *
 * Calls:
 *    "codeouts", "codeoutl", "llabelname", "outeol", "gen_atmoffset",
 *    and "jump" in this module.
 *
 * Called by:
 *    "gen_rete" in this module.
 *
 *-------------------------------------------------------------------------*/
{
   /* Code to compare for 2
    * identical string pointers.
    */
   fprintf(fp_root, "\tif (R8.WmePtr->wme[%d] %s OpsSymbols[%d] /* %s */)  ",
                    node->pos, opp_test(test), gen_atmoffset(node->value.str), node->value.str);
   goto_fprintf(fp_root, stacktree->lab);   /* Backtrack jump on failure. */
}



tyys(node, test)
   t_nodeptr node;
   int       test;
/*---------------------------------------------------------------------------
 *
 * Abstract:
 *    T_NODE tests if 2 fields of a wme that were bound by the same internal
 *    variable satisfy the predicate represented by the T_NODE. The code
 *    generated here will depend upon the given predicate.
 *
 *    The value stored in a field of a wme is always an integer or a pointer
 *    to a string constant. The value is left shifted by one bit and, if it's
 *    a pointer, the low order bit is set to distinguish it from an integer.
 *
 * Parameters:
 *    node    - the T_NODE for which we're generating code.
 *    testnot - the Vax conditional jump instruction that does a jump if the
 *              test fails (i.e., it has the opposite sense of the T_NODE
 *              test).
 *
 * Environment:
 *    Vax register R8 holds the address of the current working memory
 *    element which the node will examine. Backtrack stack holds the code
 *    label to which the matcher backtracks if the node test fails.
 *
 * Calls:
 *    "codeouts", "codeoutl", "llabelname", "outeol", and "jump" in this
 *    module.
 *
 * Called by:
 *    "gen_rete" in this module.
 *
 *-------------------------------------------------------------------------*/
{
   /* If predicate here is not
    * TEST_EQ or TEST_NE, then
    * generate code to jump to
    * backtrack label if fields
    * are not both of type integer.
    * (Predicates other than EQ
    * and NE do not make sense
    * for strings and we treat
    * them as failures during
    * the match phase.)
    */
   if ((test != TEST_EQ) && (test != TEST_NE))
     {
      fprintf(fp_root, "\tif (symbolp(R8.WmePtr->wme[%d]) || symbolp(R8.WmePtr->wme[%d]))  ",
                       node->pos, node->value.snum);
      goto_fprintf(fp_root, stacktree->lab);   /* Backtrack jump on failure. */
     }

   /* In all other cases, we simply
    * compare the 2 fields and jump
    * to the backtrack label if the
    * test fails.
    */
   fprintf(fp_root, "\tif (R8.WmePtr->wme[%d] %s R8.WmePtr->wme[%d])  ",
                    node->pos, opp_test(test), node->value.snum);
   goto_fprintf(fp_root, stacktree->lab);   /* Backtrack jump on failure. */
}



tyyn(node, test)
  t_nodeptr node;
  int test;
/*---------------------------------------------------------------------------
 *
 * Abstract:
 *    T_NODE tests if a field of a wme is an integer and compares it to the
 *    specified value stored in the node by the compilation.
 *
 *    The value stored in a field of a wme is always an integer or a pointer
 *    to a string constant. The value is left shifted by one bit and, if it's
 *    a pointer, the low order bit is set to distinguish it from an integer.
 *
 * Parameters:
 *    node    - the T_NODE for which we're generating code.
 *    testnot - the Vax conditional jump instruction that does a jump if the
 *              test fails (i.e., it has the opposite sense of the T_NODE
 *              test).
 *
 * Environment:
 *    Vax register R8 holds the address of the current working memory
 *    element which the node will examine. Backtrack stack holds the code
 *    label to which the matcher backtracks if the node test fails.
 *
 * Calls:
 *    "codeouts", "codeoutl", "llabelname", "outeol", and "jump" in this
 *    module.
 *
 * Called by:
 *    "gen_rete" in this module.
 *
 *-------------------------------------------------------------------------*/
{
   /* If predicate here is not
    * TEST_EQ or TEST_NE, then
    * generate code to jump to
    * backtrack label if field
    * is not of type integer.
    * For EQ and NE, even if the
    * field is a string, the
    * compare instruction next will
    * result in a failure in the
    * match and thus the integer test
    * instruction is not required.
    */
   if ((test != TEST_EQ) && (test != TEST_NE))
     {
      fprintf(fp_root, "\tif (!numberp(R8.WmePtr->wme[%d]))  ",
                       node->pos, node->value.snum);
      goto_fprintf(fp_root, stacktree->lab);   /* Backtrack jump on failure. */
     }

   /* Generate code to compare
    * the field value against
    * the value stored in the
    * node. Remember to shift
    * the node value left.
    */
   fprintf(fp_root, "\tif (R8.WmePtr->wme[%d] %s int2val(%d))  ",
                    node->pos, opp_test(test), node->value.num);
   goto_fprintf(fp_root, stacktree->lab);   /* Backtrack jump on failure. */
}




any(node)
   any_nodeptr node;
/*---------------------------------------------------------------------------
 *
 * Abstract:
 *    Generate code for an ANY_NODE to test if a field of a wme is equal to
 *    one of the constants in the list for a disjunction. The constants in
 *    the list are either integers or strings. (Note that in << red 10 >>,
 *    '10' is an integer while in << red // 10 >>, '10' is a string.)
 *
 *    The value stored in a field of a wme is always an integer or a pointer
 *    to a string constant. The value is left shifted by one bit and, if it's
 *    a pointer, the low order bit is set to distinguish it from an integer.
 *
 * Parameters:
 *    node - the ANY_NODE for which we're generating code.
 *
 * Environment:
 *    Vax register R8 holds the address of the current working memory
 *    element which the node will examine. Backtrack stack holds the code
 *    label to which the matcher backtracks if the node test fails.
 *
 * Calls:
 *    "codeouts", "codeoutl", "llabelname", "outeol", "nextlab", and "jump"
 *    in this module.
 *
 * Called by:
 *    "gen_rete" in this module.
 *
 *-------------------------------------------------------------------------*/
{
   int        succlink;
   anyatm_ptr atm;


   /* Code to put the wme field into a tmp.
    */
   fprintf(fp_root, "\tTmp = R8.WmePtr->wme[%d];  /* Tmp is an OpsVal */\n", node->snum);

   /* Code to compare the field
    * to each element in the
    * disjunction list until we
    * find a match or exhaust
    * the list.
    */
   succlink = nextlab();
   atm = node->anyatm_list;
   while (atm)
     {
      if (atm->is_int)
        {
         fprintf(fp_root, "\tif (Tmp == int2val(%d))  goto L%d;\n", atm->aval.any_int, succlink);
        }
      else
        {
         /* 'Ops_symbols' will already
          * hold a modified pointer.
          */
         fprintf(fp_root, "\tif (Tmp == OpsSymbols[%d] /* %s */)  goto L%d;\n",
                           symoffset(atm->aval.any_ptr), atm->aval.any_ptr->symname, succlink);
        }
          
     atm = atm->next;
     }

   /* Code to backtrack on failure
    * to match any constant in the
    * disjunction list..
    */      
   goto_fprintf(fp_root, stacktree->lab);   /* Backtrack jump on failure. */

   /* Code label to jump to upon
    * a successful match.
    */ 
   wrllabel(fp_root, succlink);            /*  succlink:                          */
}




beta(node)
   nodeptr node;
/*---------------------------------------------------------------------------
 *
 * Abstract:
 *    Generate code for a beta node. The code generated depends upon whether
 *    we entered the beta node from its left or right input (i.e., at
 *    execution time, the matcher is adding/deleting a token to/from either
 *    the left or right memory associated with the node and the code must
 *    examine the opposite memory to see if any new tokens must be propagated
 *    down the network).
 *
 * Parameters:
 *    node - the beta node for which we're generating code.
 *
 * Environment:
 *    The global 'flag_ri' tells us how the beta node was visited in the
 *    'depth-first' processing of the network to generate the code.
 *
 * Calls:
 *    "right_beta"and "left_beta" in this module.
 *
 * Called by:
 *    "gen_rete" in this module.
 *
 *-------------------------------------------------------------------------*/
{
   if (flag_ri)
      right_beta(node);   /* Visiting beta node via right input. */
   else
      left_beta(node);    /* Visiting beta node via left input. */
}



right_beta(node)
   nodeptr node;
/*---------------------------------------------------------------------------
 *
 * Abstract:
 *    We are visiting a beta node via its right input and must generate code
 *    for the matcher to add/delete a token to/from the node's right memory
 *    and to scan the left memory and build any new tokens to pass down the
 *    network as the result of consistent variable bindings between left
 *    memory tokens and the new right memory token.
 *
 * Parameters:
 *    node - the beta node for which we're generating code.
 *
 * Environment:
 *    The left and right memories for a beta node are found by indexing
 *    into the global hash tables 'ltokHT' and 'rtokHT' to obtain
 *    pointers to the node's left and right memories.
 *
 * Calls:
 *    "getnid", "rightand_join", and "rightnot_join" in this module.
 *
 * Called by:
 *    "beta" in this module.
 *
 *-------------------------------------------------------------------------*/
{  
   nodeptr      nextbeta;  /* Ptr to access the right-brother chain. */

   /* For the beta node and each beta node in it's right-brother chain,
    * generate code to handle the right input to the beta node. 
    */   
   nextbeta = node;
   while (nextbeta)
     {
      /* First get/assign the node_id.  Also get beta level into 'cur_betalev'. */
      getnid(nextbeta);
      
      /* Generate code for the right input to select the beta node's right
       * memory and add/delete the current wme to/from the right memory.
       * Generate code to join in turn each token from the node's left memory
       * with the current wme from the right and perform the variable binding
       * tests on the joined tokens. The code generated here also determines
       * if a joined token should be passed down the network to the node's
       * successor. This determination is based upon the results of the tests
       * and, for NOT_NODEs, the reference counts for the left tokens.
       */
      if (nextbeta->nodename == AND_NODE)
         rightand_join(nextbeta);
      else
         rightnot_join(nextbeta);
      
      nextbeta = nextbeta->brothr;
     }

   /* Reset the flag that marked the entry to the beta node via its right
    * input.
    */
   flag_ri = FALSE;
   
   /* Set a flag to tell "nextnode" to skip the code generation for the
    * sons of the beta nodes until later when the nodes are again visited
    * via their left inputs. Instead, code will be generated next for the
    * node held in the backtracking stack.
    */
   flag_skip_son = TRUE;
}



left_beta(node)
   nodeptr node;
/*---------------------------------------------------------------------------
 *
 * Abstract:
 *    We are visiting a beta node via its left input and must generate code
 *    for the matcher to add/delete a token to/from the node's left memory
 *    and to scan the right memory and build any new tokens to pass down the
 *    network as the result of consistent variable bindings between right
 *    memory tokens and the new left memory token.
 *
 * Parameters:
 *    node - the beta node for which we're generating code.
 *
 * Environment:
 *    The left and right memories for a beta node are found by indexing
 *    into the global hash tables 'ltokHT' and 'rtokHT' to obtain
 *    pointers to the node's left and right memories.
 *
 * Calls:
 *    "getnid", "leftand_join", and "leftnot_join" in this module.
 *
 * Called by:
 *    "beta" in this module.
 *
 *-------------------------------------------------------------------------*/
{  
   /* First get/assign nodeid.  Also get the beta level into 'cur_betalev'. */
   getnid(node);
   
   /* Generate code for the left input to select the beta node's left
    * memory and add/delete the current token to/from the left memory.
    * Generate code to join in turn each wme from the node's right memory
    * with the current token from the left and perform the variable binding
    * tests on the joined tokens. The code generated here also determines
    * if a joined token should be passed down the network to the node's
    * successor. This determination is based upon the results of the tests
    * and, for NOT_NODEs, the reference counts for the left tokens.
    */
   if (node->nodename == AND_NODE)
      leftand_join(node);
   else
      leftnot_join(node);
}




getnid(node)
   nodeptr node;
/*---------------------------------------------------------------------------
 *
 * Abstract:
 *    This routine obtains/assigns the nodeid for a beta
 *    node and asigns it along with the node's 'betalev' to global
 *    variables.
 *
 * Parameters:
 *    node - the beta node for which we're retrieving (or allocating) nid.
 *
 * Environment:
 *    Global nodeid_cnt keeps tract of next one to be allocated.
 *
 * Calls:
 *    No one.
 *
 * Called by:
 *    "left_beta" and "right_beta" in this module.
 *
 *-------------------------------------------------------------------------*/
{
   beta_nodeptr bptr;
   
   bptr = node->nodeinfo.betanode;

   assign_nid_plus_rtns(node);   
   
   /* Assign the globals. */
   cur_nodeid  = bptr->nodeid;
   cur_betalev = bptr->betalev;
}



string
make_rtn_name(prefix, id)
   string prefix;
   int    id;
{
   int n, ilen, plen;
   string name;

   ilen = 1;
   n = id;
   while (n = n / 10)  ilen++;

   plen = strlen(prefix);

   name = (string) malloc(plen+ilen+1);

   for (n=0; n<plen; n++)  name[n] = prefix[n];

   for (n=ilen-1; n>=0; n--)  {name[plen+n] = '0' + (id % 10);  id = id / 10;}

   name[plen+ilen] = 0;

   return(name);
}

   
assign_nid_plus_rtns(node)
   nodeptr node;
{
   p_nodeptr    pn;
   beta_nodeptr bptr;

   if (node->nodename == P_NODE)
     {
      pn = node->nodeinfo.pnode;
      pn->rtn = make_rtn_name("pnode_p", pn->pindex);
/*      fprintf(fp_incl, "int %s();\n", pn->rtn); */
      return;
     }
   
   bptr = node->nodeinfo.betanode;
   
   if (bptr->nodeid == -1)
     {
      bptr->nodeid = nodeid_cnt++;
      if (node->nodename == AND_NODE)
        {
	 bptr->right_rtn = make_rtn_name("rightand_", bptr->nodeid);
	 bptr->left_rtn  = make_rtn_name("leftand_", bptr->nodeid);
	 bptr->testpass_rtn = make_rtn_name("and_testpass_", bptr->nodeid);
/*	 fprintf(fp_incl, "int %s(), %s(), %s();\n",                            */
/*	                  bptr->right_rtn, bptr->left_rtn, bptr->testpass_rtn); */
	}
      else
        {
	 bptr->right_rtn = make_rtn_name("rightnot_", bptr->nodeid);
	 bptr->left_rtn  = make_rtn_name("leftnot_", bptr->nodeid);
	 bptr->pass_rtn  = make_rtn_name("not_pass_", bptr->nodeid);
	 bptr->test_rtn  = (bptr->testlist) ? make_rtn_name("not_test_", bptr->nodeid) : "NULL";
/*	 fprintf(fp_incl, "int %s(), %s(), %s();  ",			     */
/*	                  bptr->right_rtn, bptr->left_rtn, bptr->pass_rtn);  */
/*         if (bptr->testlist)						     */
/*	    fprintf(fp_incl, "boolean %s();\n", bptr->test_rtn);	     */
/*	 else								     */
/*	    fprintf(fp_incl, "\n");                                          */
	}
     }
}



gen_lhash(node)
   nodeptr node;
{
   beta_nodeptr bptr;
   tests_ptr    tests;
   int          test;

   bptr = node->nodeinfo.betanode;

   fprintf(fp_sub, "\tHKey = %d;\n", (cur_nodeid*137)%MEM_HASHTABLE_SIZE);

   tests = bptr->testlist;
   while (tests)
     {
      test = tests->test % 10;
   
      if (test == TEST_EQ)
        {
         if (bptr->betalev == 1)
	   {
            /* r8 == curwme so just use offset
	     * to get to the wme field.
	     */
            fprintf(fp_sub, "\tHKey = HKey ^ R8.WmePtr->wme[%d];\n", tests->l_snum);
	   }
	 else
	   {
	    /* r9 == beta cell, r8 == alpha cell
	     */
	    if (tests->l_wme == bptr->betalev)
	      {
               /* The wmeptr field in the alpha cell
	        * is at byte offset 8 from beginning
		* of cell. (Modified 7/31/87 by Dirk)
		*/
               fprintf(fp_sub, "\tHKey = HKey ^ R8.AlphaPtr->pWme->wme[%d];\n", tests->l_snum);
	      }
	    else
	      {
               /* The token field in the beta cell
	        * is at byte offset 16 from beginning
		* of cell. (Modified 7/31/87 by Dirk)
		*/
               fprintf(fp_sub, "\tHKey = HKey ^ R9.BetaPtr->token[%d]->wme[%d];\n", (tests->l_wme - 1), tests->l_snum);
	      }
           }
        }
        
      tests = tests->next;
      
     } /* end while */

   fprintf(fp_sub, "\tHKey = HKey & HKEY_MASK;\n");
}



gen_rhash(node)
   nodeptr node;
{
   beta_nodeptr bptr;
   tests_ptr    tests;
   int          test;

   bptr = node->nodeinfo.betanode;

   fprintf(fp_sub, "\tHKey = %d;\n", (cur_nodeid*137)%MEM_HASHTABLE_SIZE);
   
   tests = bptr->testlist;
   while (tests)
     {
      test = tests->test % 10;
   
      if (test == TEST_EQ)
        {
         /* r8 == curwme so just use offset
          * to get to the wme field.
	  */
         fprintf(fp_sub, "\tHKey = HKey ^ R8.WmePtr->wme[%d];\n", tests->r_snum);
	}
        
      tests = tests->next;
      
     } /* end while */

   fprintf(fp_sub, "\tHKey = HKey & HKEY_MASK;\n");
}



rightand_join(node)
   nodeptr node;
/*---------------------------------------------------------------------------
 *
 * Abstract:
 *    This routine generates the code for the matcher to perform the
 *    variable binding tests between the token just added to the right
 *    memory of the current beta node and each token in its left memory.
 *    A loop is constructed which iterates through the left memory tokens
 *    and  calls another piece of generated code that performs the variable
 *    binding tests and determines if the joined token is passed to the node's
 *    successor. This determination is made based upon the results of the
 *    tests and, for NOT_NODEs, the reference count of the left token.
 *
 * Parameters:
 *    node - the beta node for which we're generating code.
 *
 * Environment:
 *
 * Calls:
 *    "add_link", "nextlab", "jump", "codeouts", "codeoutl", "llabelname",
 *    and "outeol" in this module.
 *
 * Called by:
 *    "right_beta" in this module.
 *
 * Change Log:
 *	8/3/86	axg	Changed so that it first checks if the testlink and
 *			betalink have already been assigned values.
 *
 *-------------------------------------------------------------------------*/
{
   beta_nodeptr bptr;

   bptr = node->nodeinfo.betanode;

   fprintf(fp_root, "\tR0 = %s;  ops_PushTaskQueue();\n", bptr->right_rtn);   

/*   if (node->brothr == NULL)		       */
/*      goto_fprintf(fp_root, stacktree->lab); */

   fprintf(fp_sub, "%s()\n{\n", bptr->right_rtn);   /* rightand_nodeid() */
   fprintf(fp_sub, "\tint %s();\n", bptr->testpass_rtn);
   gen_rhash(node);
   fprintf(fp_sub, "\tops_rightand_input(%d, %s);\n", cur_nodeid, bptr->testpass_rtn);
   fprintf(fp_sub, "\treturn(RIGHT_LEAVE_BETA_TASK);\n}\n\n");
}



leftand_join(node)
   nodeptr node;
/*---------------------------------------------------------------------------
 *
 * Abstract:
 *    This routine generates the code for the matcher to perform the
 *    variable binding tests between the token just added/deleted to the left
 *    memory of the current beta node and each token in its right memory.
 *    A loop is constructed which iterates through the right memory tokens
 *    and  calls another piece of generated code that performs the variable
 *    binding tests and determines if the joined token is passed to the node's
 *    successor. This determination is made based upon the results of the
 *    tests and, for NOT_NODEs, the reference count of the left token.
 *
 * Parameters:
 *    testlink - the entry point label to the code which performs the
 *               variable binding tests for the beta node. This label
 *               will be the target of a JSB (jump to subroutine call)
 *               in the generated code.
 *
 * Environment:
 *    Information about the current beta node is held in the globals
 *    'cur_rmem' and 'cur_betalev'. 
 *
 * Calls:
 *    "codeouts", "codeoutl", "llabelname", and "outeol" in this module.
 *
 * Called by:
 *    "left_beta" in this module.
 *
 *
 * Change Log:
 *      8/7/86  dlk     Changed to output code for either jsb or rsb upon
 *                      exiting from node. Depends on left brother and
 *                      beta level.
 *	8/3/86	axg	Changed so that it first checks if the testlink and
 *			betalink have already been assigned values.  If not,
 *			in case the visit is first from the left side, it
 *			allocates new labels.
 *
 *-------------------------------------------------------------------------*/
{
   beta_nodeptr bptr;

   bptr = node->nodeinfo.betanode;

   if (cur_betalev <= 1)
     {
      fprintf(fp_root, "\tR0 = %s;  ops_PushTaskQueue();\n", bptr->left_rtn);
/*      goto_fprintf(fp_root, stacktree->lab); */
     }

   fprintf(fp_sub, "%s()\n{\n", bptr->left_rtn);   /* leftand_nodeid() */
   fprintf(fp_sub, "\tint %s();\n", bptr->testpass_rtn);
   gen_lhash(node);
   fprintf(fp_sub, "\tops_leftand_input(%d, %d, %s);\n", cur_nodeid, cur_betalev, bptr->testpass_rtn);
   fprintf(fp_sub, "\treturn(LEFT_LEAVE_BETA_TASK);\n}\n\n");

   gen_and_testpass(node);   /* and_testpass_nodeid() */
}



rightnot_join(node)
   nodeptr node;
/*---------------------------------------------------------------------------
 *
 * Abstract:
 *    This routine generates the code for the matcher to perform the
 *    variable binding tests between the token just added to the right
 *    memory of the current beta node and each token in its left memory.
 *    A loop is constructed which iterates through the left memory tokens
 *    and  calls another piece of generated code that performs the variable
 *    binding tests and determines if the joined token is passed to the node's
 *    successor. This determination is made based upon the results of the
 *    tests and, for NOT_NODEs, the reference count of the left token.
 *
 * Parameters:
 *    node - the beta node for which we're generating code.
 *
 * Environment:
 *
 * Calls:
 *    "add_link", "nextlab", "jump", "codeouts", "codeoutl", "llabelname",
 *    and "outeol" in this module.
 *
 * Called by:
 *    "right_beta" in this module.
 *
 * Change Log:
 *	8/3/86	axg	Changed so that it first checks if the testlink and
 *			betalink have already been assigned values.
 *
 *-------------------------------------------------------------------------*/
{
   beta_nodeptr bptr;

   bptr = node->nodeinfo.betanode;

   fprintf(fp_root, "\tR0 = %s;  ops_PushTaskQueue();\n", bptr->right_rtn);   

/*   if (node->brothr == NULL)		       */
/*      goto_fprintf(fp_root, stacktree->lab); */

   fprintf(fp_sub, "%s()\n{\n", bptr->right_rtn);   /* rightnot_nodeid() */
   if (bptr->testlist)  fprintf(fp_sub, "\tboolean %s();\n", bptr->test_rtn);
   fprintf(fp_sub, "\tint %s();\n", bptr->pass_rtn);
   gen_rhash(node);
   fprintf(fp_sub, "\tif (ops_rightnot_input(%d, %s, %s) == EXIT_NODE) return(RIGHT_LEAVE_BETA_TASK);\n",
                    cur_nodeid, bptr->test_rtn, bptr->pass_rtn);
   fprintf(fp_sub, "\treturn(RIGHT_LEAVE_BETA_TASK);\n}\n\n");
}





leftnot_join(node)
   nodeptr node;
/*---------------------------------------------------------------------------
 *
 * Abstract:
 *    This routine generates the code for the matcher to perform the
 *    variable binding tests between the token just added/deleted to the left
 *    memory of the current beta node and each token in its right memory.
 *    A loop is constructed which iterates through the right memory tokens
 *    and  calls another piece of generated code that performs the variable
 *    binding tests and determines if the joined token is passed to the node's
 *    successor. This determination is made based upon the results of the
 *    tests and, for NOT_NODEs, the reference count of the left token.
 *
 * Parameters:
 *    testlink - the entry point label to the code which performs the
 *               variable binding tests for the beta node. This label
 *               will be the target of a JSB (jump to subroutine call)
 *               in the generated code.
 *
 * Environment:
 *    Information about the current beta node is held in the globals
 *    'cur_rmem' and 'cur_betalev'. 
 *
 * Calls:
 *    "codeouts", "codeoutl", "llabelname", and "outeol" in this module.
 *
 * Called by:
 *    "left_beta" in this module.
 *
 *
 * Change Log:
 *      8/7/86  dlk     Changed to output code for either jsb or rsb upon
 *                      exiting from node. Depends on left brother and
 *                      beta level.
 *	8/3/86	axg	Changed so that it first checks if the testlink and
 *			betalink have already been assigned values.  If not,
 *			in case the visit is first from the left side, it
 *			allocates new labels.
 *
 *-------------------------------------------------------------------------*/
{
   beta_nodeptr bptr;

   bptr = node->nodeinfo.betanode;

   if (cur_betalev <= 1)
     {
      fprintf(fp_root, "\tR0 = %s;  ops_PushTaskQueue();\n", bptr->left_rtn);
/*      goto_fprintf(fp_root, stacktree->lab); */
     }

   fprintf(fp_sub, "%s()\n{\n", bptr->left_rtn);   /* leftnot_nodeid() */
   if (bptr->testlist)  fprintf(fp_sub, "\tboolean %s();\n", bptr->test_rtn);
   fprintf(fp_sub, "\tint %s();\n", bptr->pass_rtn);
   gen_lhash(node);
   fprintf(fp_sub, "\tif (ops_leftnot_input(%d, %d, %s) == EXIT_NODE)  return(LEFT_LEAVE_BETA_TASK);\n",
                    cur_nodeid, cur_betalev, bptr->test_rtn);

   fprintf(fp_sub, "testrefcnt:\n");
   fprintf(fp_sub, "\tif (R7.BetaPtr->refcount == 0)\n");
   fprintf(fp_sub, "\t  {\n");
   fprintf(fp_sub, "\t   R6.AlphaPtr = NULL;\n");
   fprintf(fp_sub, "\t   (*%s)();\n", bptr->pass_rtn);
   fprintf(fp_sub, "\t  }\n");
   fprintf(fp_sub, "\treturn(LEFT_LEAVE_BETA_TASK);\n}\n\n");

   if (bptr->testlist)  gen_not_test(node);   /* not_test_nodeid() */
   gen_not_pass(node);   /* not_pass_nodeid() */
}








gen_and_testpass(node)
   nodeptr node;
/*---------------------------------------------------------------------------
 *
 * Abstract:
 *    Using the "testlist' of a beta node, generate code that will perform
 *    the tests of externally bound variables in the condition elements
 *    joined by the beta node. At execution time, the code generated here
 *    will be entered as a subroutine call via a JSB instruction. If all of
 *    the binding tests pass, the token formed by the join of the left and
 *    right tokens is passed down in the network.
 *
 *    The value stored in a field of a wme is always an integer or a pointer
 *    to a string constant. The value is left shifted by one bit and, if it's
 *    a pointer, the low order bit is set to distinguish it from an integer.
 *
 * Parameters:
 *    node - the beta node for which we're generating code.
 *
 * Environment:
 *    The global 'ops_cevars' will hold a token formed by the join of a left
 *    memory token with a right memory token which will be tested by the
 *    code generated here. More specifically, 'ops_cevars' is an array of
 *    pointers to wme's where each wme is an array of 127 attribute value
 *    fields. The runtime stack will hold the code address to which the
 *    matcher backtracks (via an RSB instruction) if one of the variable
 *    binding tests for the node fails.
 *
 * Calls:
 *    "codeouts", "codeoutl", "llabelname", "outeol", "nextlab", "jump",
 *    "wrllabel", and "wrtest" in this module.
 *
 * Called by:
 *    "left_beta" in this module.
 *
 *-------------------------------------------------------------------------*/
{
   beta_nodeptr bptr;
   tests_ptr    tests;
   int          test;
   nodeptr      nextson;

   bptr = node->nodeinfo.betanode;

   fprintf(fp_sub, "%s()\n{\n", bptr->testpass_rtn);   /* and_testpass_nodeid() */   
    
   nextson = node->son;
   while (nextson)
     {
      assign_nid_plus_rtns(nextson);
      fprintf(fp_sub, "\tint %s();\n", ((nextson->nodename == P_NODE) ?
                       nextson->nodeinfo.pnode->rtn : nextson->nodeinfo.betanode->left_rtn));
      nextson = nextson->brothl;
     }

   /* Generate code to perform each of
    * the variable binding tests.
    */
   tests = bptr->testlist;
   while (tests)
     {
      /* Code generated depends upon
       * the test we must perform.
       */
      test = tests->test % 10;
      if (test == TEST_XX)
        {
         /* For TEST_XX, all we need do
          * is check that both fields are
	  * the same type.
          */
         fprintf(fp_sub, "\tif (difftypes(R7.BetaPtr->token[%d]->wme[%d], R6.AlphaPtr->pWme->wme[%d])) return;\n",
	                 (tests->l_wme - 1), tests->l_snum, tests->r_snum);
        }
      else
        {
         /* For other predicates, code
          * depends on the predicate and
          * the type of the fields.
          */
         if ((test != TEST_EQ) &&
             (test != TEST_NE))
           {
            /* For tests other than EQ
             * and NE, we must generate
             * code to check that both
             * fields are of type integer
             * and do exit return if not.
                         */
            fprintf(fp_sub, "\tif (symbolp(R7.BetaPtr->token[%d]->wme[%d]) || symbolp(R6.AlphaPtr->pWme->wme[%d])) return;\n",
	                     (tests->l_wme - 1), tests->l_snum, tests->r_snum);
           }
           
         /* In all other cases, we simply
          * compare the 2 fields and
          * perform the test indicated.
          */  
         fprintf(fp_sub, "\tif (R6.AlphaPtr->pWme->wme[%d] %s R7.BetaPtr->token[%d]->wme[%d]) return;\n",
	                 tests->r_snum, opp_test(test), (tests->l_wme - 1), tests->l_snum);
	}
      tests = tests->next;
     }

   nextson = node->son;
   while (nextson)
     {
      fprintf(fp_sub, "\tR0 = %s;  ops_PushTaskQueue();\n", ((nextson->nodename == P_NODE) ?
                       nextson->nodeinfo.pnode->rtn : nextson->nodeinfo.betanode->left_rtn));
      nextson = nextson->brothl;
     }

   fprintf(fp_sub, "}\n\n");
}



gen_not_test(node)
   nodeptr node;
/*---------------------------------------------------------------------------
 *
 * Abstract:
 *    Using the "testlist' of a beta node, generate code that will perform
 *    the tests of externally bound variables in the condition elements
 *    joined by the beta node. At execution time, the code generated here
 *    will be entered as a subroutine call via a JSB instruction. If all of
 *    the binding tests pass, the token formed by the join of the left and
 *    right tokens is passed down in the network.
 *
 *    The value stored in a field of a wme is always an integer or a pointer
 *    to a string constant. The value is left shifted by one bit and, if it's
 *    a pointer, the low order bit is set to distinguish it from an integer.
 *
 * Parameters:
 *    node - the beta node for which we're generating code.
 *
 * Environment:
 *    The global 'ops_cevars' will hold a token formed by the join of a left
 *    memory token with a right memory token which will be tested by the
 *    code generated here. More specifically, 'ops_cevars' is an array of
 *    pointers to wme's where each wme is an array of 127 attribute value
 *    fields. The runtime stack will hold the code address to which the
 *    matcher backtracks (via an RSB instruction) if one of the variable
 *    binding tests for the node fails.
 *
 * Calls:
 *    "codeouts", "codeoutl", "llabelname", "outeol", "nextlab", "jump",
 *    "wrllabel", and "wrtest" in this module.
 *
 * Called by:
 *    "left_beta" in this module.
 *
 *-------------------------------------------------------------------------*/
{
   beta_nodeptr bptr;
   tests_ptr    tests;
   int          test;

   bptr = node->nodeinfo.betanode;

   fprintf(fp_sub, "boolean\n%s()\n{\n", bptr->test_rtn);   /* not_test_nodeid() */   
    
   /* Generate code to perform each of
    * the variable binding tests.
    */
   tests = bptr->testlist;
   while (tests)
     {
      /* Code generated depends upon
       * the test we must perform.
       */
      test = tests->test % 10;
      if (test == TEST_XX)
        {
         /* For TEST_XX, all we need do
          * is check that both fields are
	  * the same type.
          */
         fprintf(fp_sub, "\tif (difftypes(R7.BetaPtr->token[%d]->wme[%d], R6.AlphaPtr->pWme->wme[%d])) return(TRUE);\n",
	                 (tests->l_wme - 1), tests->l_snum, tests->r_snum);
        }
      else
        {
         /* For other predicates, code
          * depends on the predicate and
          * the type of the fields.
          */
         if ((test != TEST_EQ) &&
             (test != TEST_NE))
           {
            /* For tests other than EQ
             * and NE, we must generate
             * code to check that both
             * fields are of type integer
             * and do exit return if not.
                         */
            fprintf(fp_sub, "\tif (symbolp(R7.BetaPtr->token[%d]->wme[%d]) || symbolp(R6.AlphaPtr->pWme->wme[%d])) return(FALSE);\n",
	                     (tests->l_wme - 1), tests->l_snum, tests->r_snum);
           }
           
         /* In all other cases, we simply
          * compare the 2 fields and
          * perform the test indicated.
          */  
         fprintf(fp_sub, "\tif (R6.AlphaPtr->pWme->wme[%d] %s R7.BetaPtr->token[%d]->wme[%d]) return(FALSE);\n",
	                 tests->r_snum, opp_test(test), (tests->l_wme - 1), tests->l_snum);
	}
      tests = tests->next;
     }

   fprintf(fp_sub, "return(TRUE);\n}\n\n");
}

gen_not_pass(node)
   nodeptr node;
{
   nodeptr      nextson;

   fprintf(fp_sub, "%s()\n{\n", node->nodeinfo.betanode->pass_rtn);   /* not_pass_nodeid() */   
    
   /* Fill in nodes with ids and rtn names and output forward declarations. */
   nextson = node->son;
   while (nextson)
     {
      assign_nid_plus_rtns(nextson);
      fprintf(fp_sub, "\tint %s();\n", ((nextson->nodename == P_NODE) ?
                       nextson->nodeinfo.pnode->rtn : nextson->nodeinfo.betanode->left_rtn));
      nextson = nextson->brothl;
     }

   nextson = node->son;
   while (nextson)
     {
      fprintf(fp_sub, "\tR0 = %s;  ops_PushTaskQueue();\n", ((nextson->nodename == P_NODE) ?
                       nextson->nodeinfo.pnode->rtn : nextson->nodeinfo.betanode->left_rtn));
      nextson = nextson->brothl;
     }

   fprintf(fp_sub, "}\n\n");
 }

string
opp_test(test)
   int test;
/*---------------------------------------------------------------------------
 *
 * Abstract:
 *    This routine writes a conditional jump instruction to the code stream
 *    for a failed variable binding test. The jump will have the opposite
 *    sense of the test.
 *
 * Parameters:
 *    test - the predicate from an element of a 'testlist' of a beta node.
 *    link - the code label to jump to if the test fails.
 *
 * Environment:
 *    Nothing special.
 *
 * Calls:
 *    "jump" and "codeouts" in this module.
 *    "opserror" in "ops5.c".
 *
 * Called by:
 *    "gentest" in this module.
 *
 *-------------------------------------------------------------------------*/
{
   switch (test)
     {
      case TEST_EQ:   return("!=");  break;
      case TEST_NE:   return("==");  break;
      case TEST_GT:   return("<=");  break;
      case TEST_GE:   return("<");   break;
      case TEST_LT:   return(">=");  break;
      case TEST_LE:   return(">");   break;
      default:        opserror("wrtest internal error");
                      codeouts("\tXXX\t .....OPSERROR\n");
                      bomb_out();
     }
}




ri()
/*---------------------------------------------------------------------------
 *
 * Abstract:
 *    This routine is called when an RI_NODE is visited. It tells us that we
 *    are about to visit a beta node via its right input. Here we set a flag
 *    to signal this so that we can generate the correct code for visiting a
 *    beta node via its right input.
 *
 * Parameters:
 *    None.
 *
 * Environment:
 *    Nothing special.
 *
 * Calls:
 *    No one.
 *
 * Called by:
 *    "gen_rete" in this module.
 *
 *-------------------------------------------------------------------------*/
{
   flag_ri = TRUE;
}



root()
/*---------------------------------------------------------------------------
 *
 * Abstract:
 *    This routine is called to process the ROOT_NODE, the first node in the
 *    Rete network built by the compilation of the productions. Here we
 *    prepare for the code to be generated for the network by setting up a
 *    Text segment, writing a start label to the code stream, and initializing
 *    the stack to be used for backtracking in the network during code
 *    generation.
 *
 * Parameters:
 *    None.
 *
 * Environment:
 *    The "gen_rete" routine has just begun to generate code for the Rete
 *    network built by the compilation of the productions.
 *
 * Calls:
 *    "codeouts", "wrlabel", and "pushts" in this module.
 *
 * Called by:
 *    "genrete" in this module.
 *
 *-------------------------------------------------------------------------*/
{
   fprintf(fp_root, "\n\nReteCodeRoot%d()\n{\n", g_num_rete_roots);
   fprintf(fp_root, "\tOpsVal Tmp;  /* used by any_nodes */\n");
   fprintf(fp_root, "ops_rete_root%d:   /* Label it for old time sake. */\n", g_num_rete_roots);
   fprintf(fp_root, "\tR6.WmePtr = R8.WmePtr;   /* So ops_PushTaskQueue works OK. */\n");
   fprintf(fp_root, "\tsuccdir = curdir;   /* Preset direction of succ node for ops_PushTaskQueue. */\n\n");

   pushts(NULL, OPS_LEAVE); 

   g_num_rete_roots++;
}



pnode(pnode)
   nodeptr pnode;
/*---------------------------------------------------------------------------
 *
 * Abstract:
 *    Generate the code for a P_NODE terminal node.
 *
 * Parameters:
 *    pn - a pointer to a P_NODE nodeinfo record.
 *
 * Environment:
 *    Nothing special.
 *
 * Calls:
 *    "codeouts", "codeoutl", "outeol", and "gen_atmoffset" in this module.
 *
 * Called by:
 *    "gen_rete" in this module.
 *
 * Change Log:
 *      8/22/86 dlk     Changed label in threaded code to be "p1:", "p2:",
 *                      etc. for productions instead of the actual name
 *                      from the source file.
 *      8/13/86 dlk     Changed to output only an rsb instruction if the pnode
 *                      is the successor of a beta node and has no left
 *                      brother.
 *      8/7/86  dlk     Changed to output a jsb instruction if left brother.
 *	8/2/86	axg	Changed the routine to output a rsb instruction if the
 *			pnode is a successor of a beta-node.
 *
 *-------------------------------------------------------------------------*/
{
   p_nodeptr pn;

   pn = pnode->nodeinfo.pnode;

   if (pn->cecount <= 1)
     {
      assign_nid_plus_rtns(pnode);
      fprintf(fp_root, "\tR0 = %s;  ops_PushTaskQueue();\n", pn->rtn);
/*      goto_fprintf(fp_root, stacktree->lab); */
     }

   fprintf(fp_sub, "%s()\n{\n", pn->rtn);   /* pnode_pindex() */
   fprintf(fp_sub, "\tops_term(%d, %d, OpsSymbols[%d], %d, R9.BetaPtr, R8.AlphaPtr);  /* %s */\n",
                    pn->cecount, pn->testcnt, gen_atmoffset(pn->pname), pn->pindex, pn->pname);
   fprintf(fp_sub, "\treturn(LEAVE_PNODE_TASK);\n}\n\n");
}




leave_main()
/*---------------------------------------------------------------------------
 *
 * Abstract:
 *    This routine completes the construction of the Vax assembly code
 *    output file by writing a final piece of "threaded" code to the
 *    output file; and then, generating Vax instructions for the Rete
 *    network, the "C" interface routine to the Rete code, and the "C"
 *    "main" routine that begin the production system program. Added to
 *    the output file following this are several data areas that hold
 *    information from the compile-time symbol table along with space for
 *    dynamically allocated data that will access symbolic string constants
 *    and node memories. These data areas consist of 'ops_bound_names',
 *    'ops_bindings', 'ops_names', 'ops_symbols'.
 *
 * Parameters:
 *    None.
 *
 * Environment:
 *    The Rete network has just been built by the compilation of the
 *    productions and we must now generate Vax code for the network.
 *
 * Calls:
 *    "write_code" and "write_data" in this module.
 *
 * Called by:
 *    "main" in "ops5.c".
 *
 *-------------------------------------------------------------------------*/
{

   write_code();
   
   write_data();
}



write_code()
/*---------------------------------------------------------------------------
 *
 * Abstract:
 *    This routine adds to the code output file a final piece of "threaded"
 *    code; and then, generates Vax instructions for the Rete network, the
 *    "C" interface routine to the Rete code, and the "C" "main" routine
 *    that begins the production system program.
 *
 * Parameters:
 *    None.
 *
 * Environment:
 *    The Rete network has just been built by the compilation of the
 *    productions and we must now generate Vax code for the network.
 *
 * Calls:
 *    "make_start", "gen_rete", "ops_eval_rete", and "write_main" in
 *    this module.
 *
 * Called by:
 *    "leave_main" in this module.
 *
 *-------------------------------------------------------------------------*/
{
   make_start();        /* Make the wme (start). */
   ops_eval_rete();     /* Write "C" entry routine to Rete code. */
/*   write_main(); */       /* Write "main" "C" bootstrap routine. */

   gen_rete(FALSE);     /* Write code for Rete network. */
   			/* FALSE indicates not being called from cmp_production */
}



make_start()
/*---------------------------------------------------------------------------
 *
 * Abstract:
 *    This routine writes the last piece of "threaded" code labelled with
 *    'ops_makestart' to the output file before we begin to write the Vax
 *    assembly code for the Rete network. The piece of "threaded" code
 *    generated here will, when executed, initialize the working memory of
 *    the production system by creating the 'start' symbol - a workng
 *    memory element whose first field holds the value 'start'. This piece
 *    of "threaded" code is called as part of the initialization sequence
 *    when execution of the production system is begun. Having the 'start'
 *    symbol in the working memory instantiates the production in the system
 *    that will set up the initial contents of the working memory for the
 *    system.
 *
 * Parameters:
 *    None.
 *
 * Environment:
 *    The Rete network has just been built by the compilation of the
 *    productions and the "threaded" code for the RHSs of the productions
 *    has been written to the output file.
 *
 * Calls:
 *    "outeol", "wrlabel", "newinst", "codeouts", and "gen_atmoffset"
 *    in this module.
 *
 * Called by:
 *    "write_code" in this module.
 *
 *-------------------------------------------------------------------------*/
{
   int i;

   /* Newline and then write the production name as a label.
    */  
   fprintf(fp_sub, "\n/* ops_makestart */\nstatic ThreadedCodeElement p0[] = {ops_bmake, ops_symcon, &OpsSymbols[%d] /* start */, ops_rval, ops_emake, opsret};\n", gen_atmoffset("start"));

   fprintf(fp_sub, "\n\nextern ThreadedCodeElement ");
   for (i = 0; i <= g_pcount; i++)
     {
      if (i>0)  fprintf(fp_sub, ", ");
      fprintf(fp_sub, "p%d[]", i);
     }
   fprintf(fp_sub, ";\n\n");

   fprintf(fp_sub, "\n\nThreadedCodeElement *RHScode[] = {");
   for (i = 0; i <= g_pcount; i++)
     {
      if (i>0)  fprintf(fp_sub, ", ");
      fprintf(fp_sub, "p%d", i);
     }
   fprintf(fp_sub, "};\n\n");
}



ops_eval_rete()
/*---------------------------------------------------------------------------
 *
 * Abstract:
 *    This routine writes Vax assembly code to the output file. The code
 *    produced is the "ops_eval_rete" function which serves as the "C"
 *    interface to the compiled Rete code. The code does the initial
 *    push corresponding to a wme-change into the task_queue.  The
 *    wme to be added/deleted is passed to ops_PushTaskQueue in r6, the
 *    direction is assumed to be appropriately set in succdir in match
 *    procedure in match.c.
 *    The "C" environment has the address of the new
 *    working memory element in the "C" variable 'ops_target'.
 *
 * Parameters:
 *    None
 *
 * Environment:
 *    The compiled code for the Rete network has just been written to the
 *    output file by the "gen_rete" routine.
 *
 * Calls:
 *    "begin_rtn" and "codeouts" in this module.
 *
 * Called by:
 *    "write_code" in this module.
 *
 *-------------------------------------------------------------------------*/
{
   int i;

   for (i=0; i <= g_num_rete_roots; i++)
     {
      fprintf(fp_sub, "int ReteCodeRoot%d();\n", i);
     }


      fprintf(fp_sub, "\nops_eval_rete()\n");
      fprintf(fp_sub, "{\n");
      fprintf(fp_sub, "   R6.WmePtr = ops_target;\n");
      fprintf(fp_sub, "   R7.Ptr = NULL;\n");
      fprintf(fp_sub, "   /* curdir is set by caller in match.c */\n");
   for (i=0; i <= g_num_rete_roots; i++)
     {
      fprintf(fp_sub, "   R0 = ReteCodeRoot%d;\n", i);
      fprintf(fp_sub, "   ops_PushTaskQueueN(QueueSelect);\n");
      fprintf(fp_sub, "   QueueSelect = (QueueSelect + 1) %% NumQueues;\n");
     }
   fprintf(fp_sub, "}\n\n");
}







write_data()
/*---------------------------------------------------------------------------
 *
 * Abstract:
 *    This routine completes the construction of the Vax assembly code
 *    output file by writing a final piece of "threaded" code to the
 *    output file; and then, generating Vax instructions for the Rete
 *    network, the "C" interface routine to the Rete code, and the "C"
 *    "main" routine that begin the production system program. Added to
 *    the output file following this are several data areas that hold
 *    information from the compile-time symbol table along with space for
 *    dynamically allocated data that will access symbolic string constants
 *    and node memories. These data areas consist of 'ops_bound_names',
 *    'ops_bindings', 'ops_names', 'ops_symbols'.
 *
 * Parameters:
 *    None.
 *
 * Environment:
 *    The  Vax code for the Rete network has just been written to the code
 *    output file along with the entry routines for "C".
 *
 * Calls:
 *    "write_opsbound_names", "write_opsbindings", "write_opsname",
 *    "write_opssym", and "codeouts" in this module.
 *
 * Called by:
 *    "leave_main" in this module.
 *
 *-------------------------------------------------------------------------*/
{

   write_ext_list();        /* write table of external routine names */
   write_lit_lists();      /* Write table of classnames and their attrs. */
   write_vecatt_list();    /* Write table of vector attributes. */
   write_opsbound_names(); /* Write pointers to attribute names. */
   write_opsbindings();    /* Write assigned bindings for attribute names. */
   write_opsname();        /* Write symbolic string names at 'ops_names'. */
   write_opssym();         /* Allocate space for 'ops_symbols' table. */
   write_nodeid_lists();   /* Write list of nodeids associated with each production. */
   cat_ofiles();
}



write_opsbound_names()
/*---------------------------------------------------------------------------
 *
 * Abstract:
 *    This routine writes the 'ops_bound_names' table of data to the output
 *    file. Each element of the table is a pointer to an entry in the
 *    'ops_symbols' table that corresponds to the symbolic string constant
 *    for an attribute name.
 *
 * Parameters:
 *    None
 *
 * Environment:
 *    All the Vax code has been written to the output file.
 *
 * Calls:
 *    "outeol", "codeouts", "newinst", "codeoutl", "outtab", and "symoffset"
 *    in this module.
 *
 * Called by:
 *    "write_data" in this module.
 *
 *-------------------------------------------------------------------------*/
{
   sym_attptr aptr;
   int        i, num;

   fprintf(fp_sub, "int OpsBoundNames[] = {\n");

   num = 0;   
   i = FIRST_OPSBIND;
   while (i <= g_last_buc)
     {
      aptr = bucket[i];
      while (aptr)
        {
         fprintf(fp_sub, "                       %d, /* %s */\n",
	                  symoffset(aptr->attsym), aptr->attsym->symname);
         num++;
         aptr = aptr->next;
        }
      ++i;
     }

   fprintf(fp_sub, "                       -1};\n\n");
   fprintf(fp_sub, "int NumBindings = %d;\n\n", num);
}



write_opsbindings()
/*---------------------------------------------------------------------------
 *
 * Abstract:
 *    This routine writes the 'ops_bindings' table of data to the output
 *    file. Each element of the table holds the binding for an attribute
 *    name represented in the 'ops_bound_names' table previously written
 *    by the "write_opsbound_name" routine. The elements of these 2 tables
 *    are in a 1-1 correspondence based on order and the 2 tables are used
 *    at execution time to construct the run-time symbol table.
 *
 * Parameters:
 *    None
 *
 * Environment:
 *    The 'ops_bound_names' table of data has just been written to the
 *    output file..
 *
 * Calls:
 *    "outeol", "wrlabel", "codeouts", and "codeouti" in this module.
 *
 * Called by:
 *    "write_data" in this module.
 *
 *-------------------------------------------------------------------------*/
{
   sym_attptr aptr;
   int        i, bind;
   boolean    first;

   fprintf(fp_sub, "int OpsBindings[] = {");

   first = TRUE;   
   i = FIRST_OPSBIND;
   while (i <= g_last_buc)
     {
      aptr = bucket[i];
      while (aptr)
        {
	 if (first)  first = FALSE;  else  fprintf(fp_sub,", ");
         fprintf(fp_sub, "%d", aptr->attsym->opsbind);
         aptr = aptr->next;
        }
      ++i;
     }
   fprintf(fp_sub, "};\n");
}



write_opsname()
/*---------------------------------------------------------------------------
 *
 * Abstract:
 *    This routine writes the 'ops_names' table of data to the output file.
 *    The table holds all the symbolic strin constants that were encountered
 *    during the compilation of the declarations and productions in the
 *    system. The representation for each string name is in ASCIIZ format -
 *    ASCII byte codes terminated by a 0 byte.
 *
 * Parameters:
 *    None
 *
 * Environment:
 *    The list headed by 'g_opssym' and built earlier in the compilation
 *    holds the string names.
 *
 * Calls:
 *    "outeol", "codeouts", and "put_asciiz" in this module.
 *
 * Called by:
 *    "write_data" in this module.
 *
 *-------------------------------------------------------------------------*/
{
   sym_attptr sym;
   boolean    first;
   string ptr;

   fprintf(fp_sub, "\nchar *OpsNames[%d] = {\n", g_offset);

   first = TRUE;
   sym = g_opssym;
   while (sym)
     {
      if (first)  first = FALSE;  else  fprintf(fp_sub, ",\n");
      fprintf(fp_sub, "                      %c", '"');
      ptr = sym->attsym->symname;
      while (*ptr)
        {
         if (*ptr == '"')
	    fprintf(fp_sub, "%c%c", '\\', '"');
	 else if (*ptr == '\n')
	    fprintf(fp_sub, "%c%c", '\\', 'n');
	 else
	    fprintf(fp_sub, "%c", *ptr);
         ptr++;
	}
      fprintf(fp_sub, "%c", '"');
      sym = sym->next;
      }
   fprintf(fp_sub, "\n                     };\n");

}





write_opssym()
/*---------------------------------------------------------------------------
 *
 * Abstract:
 *    Allocate space for the 'ops_symbol' table in the code output file.
 *    This table will be initialized at run-time with the starting addresses
 *    of the ASCIIZ representation for each string name stored in the
 *    'ops_names' table (re: "write_opsnames"). The pointers stored in this
 *    table will actually be left shifted one bit and have their low bit set.
 *    The reason these run-time pointers to string names are represented in
 *    this manner in the table is to facilitate the Rete match. A field of a
 *    working memory element will be either a symbolic string constant
 *    represented in the manner just described or an integer number
 *    represented in the same way except that the low bit is clear. Thus, at
 *    run-time, the Rete match can simply compare against the field directly
 *    without explicitly interrogating the the type of lhs-value stored in
 *    the attribute field. 
 *
 * Parameters:
 *    None
 *
 * Environment:
 *    The global 'g_offset' tells how many bytes are required for the
 *    'ops_symbols' table.
 *
 * Calls:
 *    "codeouts", "codeoutl", and "outeol" in this module.
 *
 * Called by:
 *    "write_data" in this module.
 *
 *-------------------------------------------------------------------------*/
{
   fprintf(fp_sub, "\nint SymCount = %d;\n", g_offset);
   fprintf(fp_sub, "OpsVal OpsSymbols[%d];\n", g_offset);
}


write_nodeid_lists()
/*---------------------------------------------------------------------------
 *
 * Abstract:
 *    Write the list of nodeids for the beta nodes associated with each
 *    production. This is to support the OPS5 top level cmd "matches".
 *
 * Parameters:
 *    None
 *
 * Environment:
 *    The list of productions and their nodes are accessed through
 *    ProductionList.
 *
 * Calls:
 *    "symoffset", "newinst", "codeouts", "codeoutl", and "outeol" in this module.
 *
 * Called by:
 *    "write_data" in this module.
 *
 *-------------------------------------------------------------------------*/
{
   plist_ptr prod;
   blist_ptr bptr;

   fprintf(fp_sub, "\nint MatchesFlag = 1;\n");
   fprintf(fp_sub, "int NumProductions = %d;\n", g_pcount);
   fprintf(fp_sub, "int *NodeIdLists[] = {\n");

   prod = ProductionList;
   while (prod)
     {
      fprintf(fp_sub, "                      &OpsSymbols[%d],  /* %s */\n",
                       symoffset(prod->prodsym), prod->prodsym->symname);
      fprintf(fp_sub, "                       ");
      bptr = prod->blist_head;
      while (bptr)
        {
         fprintf(fp_sub, "%d,",bptr->bnode->nodeinfo.betanode->nodeid);
	 bptr = bptr->next;
	}
      fprintf(fp_sub, "-1,\n");  /* Valid nodeids are >= 0, so end each list with -1 */
      prod = prod->next;
     }

   fprintf(fp_sub, "                      0};\n\n");

}



write_ext_list()
/*---------------------------------------------------------------------------
 *
 * Abstract:
 *    Write the list of names declared external at the end of the assembly
 *    output file in a way that will get the linker to tell us the routine
 *    addresses at run-time. This is to support the OPS5 top level cmd "call".
 *
 * Parameters:
 *    None
 *
 * Environment:
 *    The list of external names is in ExternalList.
 *
 * Calls:
 *    "symoffset", "newinst", "codeouts", "codeoutl", and "outeol" in this 
 *     module.
 *
 * Called by:
 *    "write_data" in this module.
 *
 * Log:
 *    9/29/89 : Anurag Acharya
 *       removed code to generate the extern declarations for the routines.
 *       that functionality has been taken over by "write_ext_func_list()"
 *       in cmprhs.c
 *
 *-------------------------------------------------------------------------*/
{
   elist_ptr eptr;
   int       cnt;

  /* print the declaration of the list of external routine names */
   fprintf(fp_sub, "\n\nint *ExternalList[] = {\n");

  /* cycle thru the list of external routines and print the name and the 
   * the index of the corresponding OpsSymbol for each of the entries.
   */
  for (eptr = ExternalList, cnt = 0; eptr != NULL; eptr = eptr->next, cnt++)
      fprintf(fp_sub, "                       &OpsSymbols[%d], %s,\n",
                       symoffset(eptr->ext_fname), eptr->ext_fname->symname);
   fprintf(fp_sub, "                       0};\n");

   /* print the number of external routines declared in the program */
   fprintf(fp_sub, "int NumExternals = %d;\n\n", cnt);
}



write_lit_lists()
/*---------------------------------------------------------------------------
 *
 * Abstract:
 *    Write the list of classnames and their attributes at the end of the
 *    assembly output file. This is to support standard OPS5 format for
 *    printing wmes.
 *
 * Parameters:
 *    None
 *
 * Environment:
 *    The list of classnames with their attributes is in b_cname_ptr.
 *
 * Calls:
 *    "symoffset", "newinst", and "codeouts" in this module.
 *
 * Called by:
 *    "write_data" in this module.
 *
 *-------------------------------------------------------------------------*/
{
   cname_ptr  cptr;
   sym_attptr aptr;
   symptr     ref[128];
   int        i, last;

   for (i=0; i<128; i++)  ref[i] = NULL;

   fprintf(fp_sub, "int *LiteralizeLists[] = {\n");

   cptr = b_cname_ptr;
   while (cptr)
     {
      fprintf(fp_sub, "                          &OpsSymbols[%d],  /* %s */\n",
                       symoffset(cptr->cname), cptr->cname->symname);

      last = 0;
      aptr = cptr->attlist;
      while (aptr)
        {
	 i = aptr->attsym->opsbind;
	 if (i > last)  last = i;
	 ref[i] = aptr->attsym;
         aptr = aptr->next;
	}
      for (i=1; i<=last; i++)
        {
	 if (ref[i] != NULL)
           {
            fprintf(fp_sub, "                            &OpsSymbols[%d],  /* %s */\n",
                             symoffset(ref[i]), ref[i]->symname);
	    ref[i] = NULL;
	   }
	}

      fprintf(fp_sub, "                            0,\n\n");
      cptr = cptr->next;
     }

   fprintf(fp_sub, "                          0};\n\n");
}



write_vecatt_list()
/*---------------------------------------------------------------------------
 *
 * Abstract:
 *    Write the list of names declared as vector attributes at the end of the
 *    assembly output file. This is to support standard OPS5 format for
 *    printing wmes.
 *
 * Parameters:
 *    None
 *
 * Environment:
 *    The list of vector attributes is in g_vecatt_list
 *
 * Calls:
 *    "symoffset", "newinst", and "codeouts" in this module.
 *
 * Called by:
 *    "write_data" in this module.
 *
 *-------------------------------------------------------------------------*/
{
   sym_attptr vptr;

   fprintf(fp_sub, "int *VectorAttList[] = {\n");

   vptr = g_vecatt_list;
   while (vptr)
     {
      fprintf(fp_sub, "                        &OpsSymbols[%d], /* %s */\n",
                       symoffset(vptr->attsym), vptr->attsym->symname);
      vptr = vptr->next;
     }

   fprintf(fp_sub, "                        0};\n\n");
}




static
int
gen_atmoffset(atm)
   string atm;
/*---------------------------------------------------------------------------
 *
 * Abstract:
 *    Used only in this module to fetch the symbol table entry corresponding
 *    to a string that was entered earlier into the symbol table during the
 *    compilation of the productions that produced the node network. The
 *    string should be found in the symbol table although it may not as yet
 *    have been added to the 'g_opssym' list and assigned an offset in
 *    'ops_symbols'. Here we get (or assign) the offset and generate code
 *    for 'ops_symbols + offset'.
 *
 * Parameters:
 *    atm - a string to look up in the symbol table.
 *
 * Environment:
 *    Nothing special.
 *
 * Calls:
 *    "symoffset" in this module.
 *    "lookup_sym_addr" in "literal.c".
 *    "opserror", "puttab", "putstring", and "puteol" in "ops5.c".
 *
 * Called by:
 *    "tyya", "pnode", and "make_start" in this module.
 *
 *-------------------------------------------------------------------------*/
{
   symptr sym;
   
   sym = lookup_sym_addr(atm);
   
   if (sym == NULL)
     {
      opserror("Compile error: did not enter a symbol into  symbol table.");
      puttab();
      putstring(atm);
      puteol();
      
      codeouts("\tXXX\t....OPSERROR\n");
      return(-1);  /* Return a bogus offset. */
     }
   else
      {
       /* Get the offset of the symbol in 'ops_symbols' and return it.
        */
       return(symoffset(sym));
      }
}



int
symoffset(sym)
   symptr sym;
/*---------------------------------------------------------------------------
 *
 * Abstract:
 *    This routine is used to generate code to reference an element in the
 *    table 'ops_symbols'. At runtime an element of 'ops_symbols' will hold
 *    a pointer to a symbolic string constant stored in the data section
 *    'ops_names' built as a result of compiling the declarations and
 *    productions.
 *
 * Parameters:
 *    sym - pointer to the symbol table entry holding a symbolic string
 *          constant. Caller ensures that pointer is not nil.
 *
 * Environment:
 *    Globals 'g_opssym' and 'g_offset' maintain a list of symbolic
 *    constants and an offset into 'ops_symbols' respectively.
 *
 * Calls:
 *    "new_opssym", "nextopssym" "codeouts", "codeoutc", and "codeoutl"
 *    in this module.
 *
 * Called by:
 *    "gen_atmoffset", "any", and "write_opsbound_names" in this module.
 *    "cmp_constant" in "cmplhs.c".
 *
 *-------------------------------------------------------------------------*/
{
   /* If the symbolic constant has not been assigned an offset into the
    * 'ops_symbols' table, then give it the next slot, build a record for
    * it and attach it to the 'g_opssym' list, and advance the offset into
    * 'ops_symbols'.
    */ 
   if (sym->offset < 0) 
     {
      sym->offset = g_offset;   /* Assign it an offset. */
      new_opssym(sym);          /* Add to the 'g_opssym' list. */
      nextopssym();             /* Advance 'g_offset'. */
     }
 
   return(sym->offset);
}



new_opssym(sym)
   symptr sym;
/*---------------------------------------------------------------------------
 *
 * Abstract:
 *    This routine is used to add a symbolic string constant to the global
 *    list headed by 'g_opssym'. 
 *
 * Parameters:
 *    sym - a symbol table entry containing a symbolic constant.
 *
 * Environment:
 *    The list of constants is maintained through 'g_opssym' and 'g_lastsym'.
 *
 * Calls:
 *    "new_attlist_rec" in "literal.c".
 *
 * Called by:
 *    "symoffset" in this module.
 *
 *-------------------------------------------------------------------------*/
{
   sym_attptr rec;

   /* Build the record for the symbol to be added to the 'g_opssym' list.
    */
   rec = new_attlist_rec(sym);

   /* Put new record at end of list.
    */
   if (g_lastsym)  g_lastsym->next = rec;
   g_lastsym = rec;
   if (g_opssym == NULL)  g_opssym = rec;
}



nextopssym()
/*---------------------------------------------------------------------------
 *
 * Abstract:
 *    This routine advances the offset into the 'ops_symbols' table. An
 *    element in the table is a 4 byte pointer to a symbolic string constant
 *    stored in the data section labelled 'ops_names'. The pointer is
 *    actually left shifted 1 bit and the low bit set.
 *
 * Parameters:
 *    None
 *
 * Environment:
 *    Nothing special.
 *
 * Calls:
 *    No one.
 *
 * Called by:
 *    "symoffset" in this module.
 *
 *-------------------------------------------------------------------------*/
{
   g_offset = g_offset + 1;
}




begin_rtn(rtn)
   string rtn;
/*---------------------------------------------------------------------------
 *
 * Abstract:
 *    Write out starting addr at the beginning of a routine that will be
 *    called from the "C" code.
 *
 * Parameters:
 *    rtn - the string name of the routine.
 *
 * Environment:
 *    Nothing special.
 *
 * Calls:
 *    "codeouts", "outeol", and "wrlabel" in this module.
 *
 * Called by:
 *    "ops_eval_rete" and "write_main" in this module.
 *
 *-------------------------------------------------------------------------*/
{  
}



end_rete(from_cmp_prod)
boolean from_cmp_prod;
/*---------------------------------------------------------------------------
 *
 * Abstract:
 *    Write out boiler plate needed at end of a routine. Also write out
 *    the subroutines 'rfixcount' and 'ltestcount' if the network has
 *    NOT_NODEs.
 *
 * Parameters:
 *    None.
 *
 * Environment:
 *    Nothing special.
 *
 * Calls:
 *    "gen_rfixcount","gen_ltestcount", "codeouts" and "outeol" in
 *    this module.
 *
 * Called by:
 *    "gen_rete" in this module.
 *
 * Log: 
 *     9/24/89 : Anurag Acharya
 *               moved the comments so that the call to "create_new_ofile()"
 *               was no longer commented out.
 *-------------------------------------------------------------------------*/
{
   /* wrllabel() puts out the end of the ReteRootCode routine */

   if (! from_cmp_prod)
   {
    /****************************
     * Code commented out to avoid closing the include file prematurely.
     * Concatenation of the files is done at the end of write_data() instead.

    cat_ofiles();
    if (have_not_node)
     {
      gen_rfixcount();
      gen_ltestcount();
     }
    gen_next_tok();
    gen_task_queue();
    outeol();
    ****************************/
   }
  else
   {
     create_new_ofile();
   }
 
}





/*---------------------------------------------------------------------------
 *
 * Abstract:
 *    The next four routines are used to generate labels in the code.
 *
 * Parameters:
 *    Depends on the routine.
 *
 * Environment:
 *    For 'L' labels, 'g_label' gives the next index to use.
 *
 * Calls:
 *    "codeouts" and "codeoutl" in this module.
 *
 * Called by:
 *    Many routines in this module.
 *
 *-------------------------------------------------------------------------*/



nextlab()
{
   int l;

   l = g_label++;
   return(l);
}



goto_fprintf(fp_x, l)
   FILE *fp_x;
   int l;
{
   if (l == OPS_LEAVE)
      fprintf(fp_x, "goto LeaveTask;\n");
   else
      fprintf(fp_x, "goto L%d;\n", l);
}



wrllabel(fp_x, l)
   FILE *fp_x;
   int l;
{
   if (l == OPS_LEAVE)
      fprintf(fp_x, "LeaveTask:  return(LEAVE_ROOT_TASK);\n}\n");
   else
      fprintf(fp_x, "L%d:\n", l);
}







/*---------------------------------------------------------------------------
 *
 * Abstract:
 *    The next set of routines are primitives used to write generated code
 *    and data to the output file.
 *
 * Parameters:
 *    Depends on the routine.
 *
 * Environment:
 *    The global 'codeout' holds the ID of the file to which the generated
 *    Vax code will be written.
 *
 * Calls:
 *    No one.
 *
 * Called by:
 *    many routines in this module and other modules.
 *
 *-------------------------------------------------------------------------*/



newinst()
{
   if (first_inst)  first_inst = FALSE;  else  fprintf(fp_sub, ", ");
}



outeol()
{
   fprintf(fp_sub, "\n");
}



outtab()
{
   fprintf(fp_sub, "\t");
}


codeoutopssym(sym)
   symptr sym;
{
   fprintf(fp_sub, "&OpsSymbols[%d]", symoffset(sym));
}


codeouts(str)
   char *str;
{
   fprintf(fp_sub, &str[1]);   /* str[0] is the '_' used for asm code version, eg. "_bmake". */
}


codeoutsx(str)   /* I think only used for ext func names. */
   char *str;
{
   fprintf(fp_sub, str);
}


/********************
codeouts4(fmt, a1, a2, a3, a4)
   char *fmt, *a1, *a2, *a3, *a4;
{
   fprintf(codeout, fmt, a1, a2, a3, a4);
}
*********************/

codeoutl(num)
   int num;
{
   fprintf(fp_sub, "%d", num);
}



codeouti(itg)
   opsnum itg;
{
   fprintf(fp_sub, "%d", itg);
}


