;;; TAQL Compiler, Pseudo-SP Expansion Module
;;;
;;; Gregg Yost, Erik Altmann
;;; School of Computer Science
;;; Carnegie Mellon University
;;;
;;; Working file: /afs/cs/user/altmann/soar/taql/pseudo-sp.lisp
;;; Created February 20, 1991
;;;
;;; This file implements the functions that expand TAQL's pseudo-SP into
;;; genuine Soar productions.  Most of the expansion routines are in this
;;; file.  However, routines specific to data-type processing (for example,
;;; data-macro and directive expansion) are in data-types.lisp.
;;;
;;; EXPORTED ROUTINES:
;;;
;;;    - structured-value-spec-p
;;;    - attribute-path-p
;;;    - expand-pseudo-sp
;;;    - split-production
;;;
;;; Known bugs/funnies:
;;; 
;;;    - Negated attribute paths and structure value specs after a negated
;;;      attribute are not compiled right.  I don't see a particularly easy
;;;      way to change it, though.  You'd have to twist things around into
;;;      conjunctive negations.
;;;
;;; =======================================================================
;;; Modification history:
;;; =======================================================================
;;;
;;; 2-20-91 - gry - Created (many of these functions were previously in
;;;   data-types.lisp).

;;; *** BEGIN CODE ***

(eval-when (compile load eval)
  (lispsyntax))

(eval-when (compile load eval)
   (if (find-package "COMMON-LISP-USER")
       (in-package "COMMON-LISP-USER")
       (in-package "USER")))

(defvar *pseudo-sp-context* nil
  "LHS or RHS, influences the way certain things are expanded.")

(defvar *pseudo-sp-warning-occurred* nil
  "Non-nil if an error is detected during pseudo-sp expansion.")

(defvar *currently-expanding-pseudo-sp* nil
  "Bound to non-nil only during pseudo-sp expansion.")

(defvar *extra-conds-at-outer-level* nil
  "Conditions resulting from data-macro expansion that belong outside of any conjunction the data-macro call appears in.")

;; EXPORTED ROUTINE:
;;
;; This is the the single exported entry point to the pseudo-sp expansion
;; code.  It takes an almost-Soar-SP production and expands data
;; macros and directives in it.  It also expands attribute paths and
;; structured value specifications.  It returns three values: the name
;; of the resulting production, its LHS, and its RHS.
;;
;; We do quite a bit of destructive editing during expansion.  For safety,
;; we do a copy-tree on the sp here.  Probably excessive, but let's not
;; worry about efficiency until we have to.
;;
(defun expand-pseudo-sp (sp)
  (setq sp (copy-tree sp))

  (let ((rhs-extra nil)
	(lhs-extra nil)
	(*currently-expanding-pseudo-sp* t))
    
    (setq *pseudo-sp-warning-occurred* nil)
    (setq *extra-conds-at-outer-level* nil)
    (reset-per-production-data-model-info)
    
    (multiple-value-bind (name lhs rhs)
	(split-production sp)
      
      (let ((*pseudo-sp-context* 'LHS))
	(multiple-value-setq (lhs rhs-extra)
	  (expand-pseudo-sp-lhs-with-directives
	   (listify-lhs-conjunctions lhs))))

      (let ((*pseudo-sp-context* 'RHS))
	(multiple-value-setq (lhs-extra rhs)
	  (expand-pseudo-sp-rhs-with-directives rhs)))
      
      (check-saved-operators)
      (check-structured-value-types)

      (values
       name
       (append lhs lhs-extra *extra-conds-at-outer-level*)
       (append rhs rhs-extra)))))

;; Take an SP-format production and return three values:  the production's
;; name, its LHS, and its RHS, respectively.
;;
(defun split-production (sp)
  (let ((tail (member '--> sp)))
    (values
       (cadr sp)
       (cddr (ldiff sp tail))
       (cdr tail))))

(defun listify-lhs-conjunctions (lhs)
  (multiple-value-bind (top-conds conjunctions remaining-lhs)
      (listify-lhs-conjunctions-aux lhs nil)
    (declare (ignore remaining-lhs))
      ;; Remaining-lhs will always be nil.
    (cons 'top-level (cons top-conds conjunctions))))

(defun listify-lhs-conjunctions-aux (lhs awaiting-close-brace)
  (prog ((top-conds nil)
         (conjunctions nil)
         (current lhs))
   loop
    (cond ((null current)
           (when awaiting-close-brace
             (taql-warn "Unmatched open brace in conjunction."))
           (return
             (values (reverse top-conds)
                     (reverse conjunctions)
                     nil)))
          ((eql (car current) '})
           (if (not awaiting-close-brace)
             (progn
               (taql-warn "Unmatched close brace in conjunction.")
               ; Skip the extrace brace.
               (pop current))
             ; ELSE
             (return
              (values (reverse top-conds)
                      (reverse conjunctions)
                      (cdr current)))))
          ((eql (car current) '{)
           (multiple-value-bind (sub-top-conds sub-conjunctions remaining-lhs)
               (listify-lhs-conjunctions-aux (cdr current) t)
             (push
                (cons 'non-negated
                      (cons sub-top-conds sub-conjunctions))
                conjunctions)
             (setq current remaining-lhs)))
          ((and (eql (car current) '-)
                (eql (cadr current) '{))
           (multiple-value-bind (sub-top-conds sub-conjunctions remaining-lhs)
               (listify-lhs-conjunctions-aux (cddr current) t)
             (push
                (cons 'negated
                      (cons sub-top-conds sub-conjunctions))
                conjunctions)
             (setq current remaining-lhs)))
          (t
           (push (car current) top-conds)
           (pop current)))
    (go loop)))

(defun unlistify-lhs (lhs)
  (let ((tag (car lhs))
        (top-conds (cadr lhs))
        (conjunctions (cddr lhs)))
    (let ((innards
           (append top-conds
                   (apply #'append
                          (mapcar #'unlistify-lhs conjunctions)))))
      (case tag
        (top-level innards)
        (non-negated `({ ,@innards }))
        (negated `(- { ,@innards }))
        (t
         (error "INTERNAL TAQL ERROR: case selector ~S fell through" tag))))))

(defun expand-pseudo-sp-lhs-conjunctions (conjunctions)
  (let ((expanded-conjuncts
         (mapcar #'(lambda (conj)
                     (multiple-value-list
                       (expand-pseudo-sp-lhs-with-directives conj)))
                 conjunctions)))
    (values
      (apply #'append (mapcar #'car expanded-conjuncts))
      (apply #'append (mapcar #'cadr expanded-conjuncts)))))

;;; ----------------------------------------------------------------------
;;; ----------------------------------------------------------------------

;;; This is a very simple production parser that expands data-macros,
;;; attribute paths, and structured value specs.  We assume the data
;;; type directives have already been extracted from the pseudo production
;;; parts and expanded.
;;; The entry point to the parser is expand-pseudo-sp*production-part.
;;; (This routine is NOT an externally exported entry point!  See the
;;; expand-pseudo-sp function.)

(defvar *current-part* nil
  "Used to give context in error messages.")

;; This is the parser entry point (non-exported! -- this is an
;; internal entry point).  It takes part of a production (a list of
;; either conditions or actions) and returns two values, the expanded
;; things that belong on the LHS and RHS, respectively.  The second
;; argument to the function must be either LHS or RHS, specifying
;; whether the first argument is from the LHS or RHS of the
;; productions.
;;
;; For the LHS case, this routine will not always be called with the
;; complete list of LHS conditions at the same time.  For example,
;; conjunctive negations are processed separately.  See
;; expand-pseudo-sp, expand-pseudo-sp-lhs-with-directives, and
;; expand-pseudo-sp-lhs-conjunctions.
;;
(defun expand-pseudo-sp*production-part (part side)
  (multiple-value-bind (modified-part extra-lhs extra-rhs)
      (case side
	(LHS
	 (expand-pseudo-sp*LHS part))
	(RHS
	 (expand-pseudo-sp*RHS part))
	(t
	 (error "INTERNAL TAQL ERROR: case selector ~S fell through" side)))
    (case side
      (LHS (values (append modified-part extra-lhs) extra-rhs))
      (RHS (values extra-lhs (append modified-part extra-rhs))))))

;;; ----------------------------------------------------------------------
;;; The LHS pseudo-sp expander begins here.
;;; ----------------------------------------------------------------------

(defun expand-pseudo-sp*LHS (lhs)
  (cond (lhs
         (multiple-value-bind (ce extra-lhs extra-rhs)
             (expand-pseudo-sp*ce (car lhs))
           (multiple-value-bind (ces extra-lhs-2 extra-rhs-2)
               (expand-pseudo-sp*LHS (cdr lhs))
             (values
              (cons ce ces)
              (append extra-lhs extra-lhs-2)
              (append extra-rhs extra-rhs-2)))))
        (t
         (values nil nil nil))))

;; CE = ( CONJUNCTIVE-TEST CONJUNCTIVE-TEST TEST* )
;;
;; I also accept atoms as CEs, so that I can call this function on
;; every element of a list like (CE - CE - { CE CE CE } CE CE).  Also,
;; Soar productions can contain doc strings and production type specifiers
;; after the production name, and I don't want to choke on these.  This
;; will result in Soar catching some errors that I don't.
;;
(defun expand-pseudo-sp*ce (ce)
  (when (not (consp ce))
    (return-from expand-pseudo-sp*ce (values ce nil nil)))
  
  (setq *current-part* ce)
  (setq ce (copy-list ce))
  
  (let ((extra-lhs nil)
        (extra-rhs nil)
        (ce-start ce))

    ;; Parse class
    (multiple-value-bind (lhs1 rhs1 resume positive-values)
        (expand-pseudo-sp*conjunctive-test ce)
      (setq extra-lhs (append lhs1 extra-lhs))
      (setq extra-rhs (append rhs1 extra-rhs))
      (new-class positive-values)
      (setq ce resume))

    ;; Parse identifier
    (multiple-value-bind (lhs1 rhs1 resume positive-values)
	(expand-pseudo-sp*conjunctive-test ce)
      (setq extra-lhs (append lhs1 extra-lhs))
      (setq extra-rhs (append rhs1 extra-rhs))
      (new-identifier positive-values)
      (setq ce resume))

    ;; Parse TEST*
    (multiple-value-bind (lhs1 rhs1)
	(expand-pseudo-sp*test* ce)
      (setq extra-lhs (append lhs1 extra-lhs))
      (setq extra-rhs (append rhs1 extra-rhs)))

    (values ce-start extra-lhs extra-rhs)))

;; CONJUNCTIVE-TEST ::= { DISJUNCTIVE-TEST* } | DISJUNCTIVE-TEST
;;
;; If the optional disallow-att-path argument is non-nil, then the conjunctive
;; test is not allowed to contain symbols that satify attribute-path-p.
;; This is used to warn about attribute path names that appear inside
;; conjunctive attribute name tests, where they are illegal.  An attribute
;; path name can only appear immediately following ^.
;;
(defun expand-pseudo-sp*conjunctive-test (start
					    &optional (disallow-att-path nil))
  (let ((extra-lhs nil)
        (extra-rhs nil)
        (start0 start)
	(positive-items nil))

    (cond ((eql (car start) '{)
           (setq start (cdr start))
           ;; Parse disjunctive-test*
           ;; loop until find close brace
           (prog ()
             loop
              (when (null start)
                (taql-warn "Missing closing brace in conjunction: ~S~%  ~S"
                           start0 *current-part*)
                (return))

              (when (eql (car start) '})
                (setq start (cdr start))
                (return))

              (multiple-value-bind (lhs1 rhs1 resume positive-disjunct-items)
                  (expand-pseudo-sp*disjunctive-test start disallow-att-path)
		(setq positive-items
		      (append positive-disjunct-items positive-items))
                (setq extra-lhs (append lhs1 extra-lhs))
                (setq extra-rhs (append rhs1 extra-rhs))
                (setq start resume))

              (go loop)))
          (t
           ;; Parse disjunctive-test
           (multiple-value-bind (lhs1 rhs1 resume positive-disjunct-items)
               (expand-pseudo-sp*disjunctive-test start disallow-att-path)
	     (setq positive-items positive-disjunct-items)
             (setq extra-lhs (append lhs1 extra-lhs))
             (setq extra-rhs (append rhs1 extra-rhs))
             (setq start resume))))

    (values extra-lhs extra-rhs start positive-items)))

;; DISJUNCTIVE-TEST ::= << CONSTANT* >> | RELATIONAL-TEST
;;
(defun expand-pseudo-sp*disjunctive-test (start
					    &optional (disallow-att-path nil))
  (let ((extra-lhs nil)
        (extra-rhs nil)
        (start0 start)
	(positive-items nil))

    (cond
     ((eql (car start) '<<)
      (setq start
	    (do ((ptr (cdr start) (cdr ptr)))
		((or (if (null ptr)
		       (progn
			 (taql-warn
			  "Missing closing >> in disjunction: ~S~%  ~S"
			  start0 *current-part*)
			 t))
		     (eql (car ptr) '>>))
		 (cdr ptr))
	      
	      (if (expand-pseudo-sp*check-constant (car ptr))
		(if (and disallow-att-path (attribute-path-p (car ptr)))
		  (taql-warn
		   "Attribute path names may not appear in disjunctions, ~
                    but found ~S~%  ~S"
		   (car ptr) *current-part*)
		  ;; ELSE
		  (push (car ptr) positive-items))
		;; ELSE
		(taql-warn
		 "A disjunction must contain only constants, ~
                  but found ~S~%  ~S"
		 (car ptr) *current-part*)))))

     (t
      ;; Parse relational-test
      (multiple-value-bind (lhs1 rhs1 resume positive-relational-items)
	  (expand-pseudo-sp*relational-test start disallow-att-path)
	(setq positive-items positive-relational-items)
	(setq extra-lhs (append lhs1 extra-lhs))
	(setq extra-rhs (append rhs1 extra-rhs))
	(setq start resume))))
    
    (values extra-lhs extra-rhs start positive-items)))

;; RELATIONAL-TEST ::= RELATION SINGLE-TEST | SINGLE-TEST
;; RELATION ::= <> | < | > | <= | >= | = | <=>
;; SINGLE-TEST ::= DATA-MACRO-CALL | VARIABLE | CONSTANT
;;
;; There will actually only ever be at most one positive-item returned, but
;; we return a list so that we can distinguish that cases where no
;; positive items are found from the case where the single positive item
;; NIL is found.
;;
(defun expand-pseudo-sp*relational-test (start
					   &optional (disallow-att-path nil))

  (let ((extra-lhs nil)
        (extra-rhs nil)
        (relation nil)
        (single-test nil)
        (test-place nil)
	(positive-items nil))

    (cond ((null start)
	   (taql-warn "Expected relational test, found nothing.~%  ~S"
		      *current-part*))
	  ((member (car start) *lhs-relations*)
           (setq relation (car start))
           (when (null (cdr start))
             (taql-warn
                "Relation symbol ~S must be followed by an argument.~%  ~S"
                        relation *current-part*))
           (setq single-test (cadr start))
           (setq test-place (cdr start))
           (setq start (cddr start)))
          (t
           (setq single-test (car start))
           (setq test-place start)
           (setq start (cdr start))))

    (when (and disallow-att-path
	       (attribute-path-p single-test))
      (taql-warn
       "Attribute path names are only permitted immediately after ^, but ~
        found ~S in~%  ~S"
       single-test *current-part*)
      (setq test-place nil))

    (cond ((or (variable-p single-test)
	       (expand-pseudo-sp*check-constant single-test))
	   (when (or (null relation)
		     (eq relation '=))
	     (setq positive-items (list single-test))))

	  ((data-macro-call-p single-test)

	   (when (and relation
		      (not (member relation
				   (data-macro-compatible (car single-test)))))
	     (taql-warn "Call to ~S cannot appear after ~S~%  ~S"
			(car single-test) relation *current-part*)
	     (setq test-place nil))

	   (when test-place
	     ;; Only process the value if we actually found one.
	     (multiple-value-bind (value lhs1 rhs1)
		 (expand-if-data-macro-and-inspect-expansion single-test)
	       (setf (car test-place) value)
	       (setq extra-lhs (append lhs1 extra-lhs))
	       (setq extra-rhs (append rhs1 extra-rhs))

	       (when (or (null relation)
			 (eq relation '=))
		 ;; We use (car test-place) instead of single-test so that we
		 ;; get the root-id in the case where single-test was a
		 ;; data-macro call.
		 (setq positive-items (list (car test-place)))))))

	  (t
	   (taql-warn
	    "Expected constant, variable, or data macro call, ~
             but found ~S~%  ~S"
	    single-test *current-part*)
	   (when (structured-value-spec-p single-test)
	     (format
	      *error-output*
	      "Possibly you have used a structured value specification, ~
               which is illegal in this context."))
	   (when (member single-test '(^ -))
	     ;; Someone probably left out a class, identifier, attribute, or
	     ;; value, causing use to hit the next attribute test early.  Put
	     ;; it back on the front of start to try to avoid cascading errors.
	     (push single-test start))
	   (setq test-place nil)))

    (values extra-lhs extra-rhs start positive-items)))

(defun expand-pseudo-sp*test* (start)
  (let ((extra-lhs nil)
	(extra-rhs nil))

    ;; Parse test*
    ;; (while ce <parse another test>)
    ;;
    (prog ()
      loop
       (when (null start) (return))

       (multiple-value-bind (lhs1 rhs1 resume)
           (expand-pseudo-sp*test start)
         (setq extra-lhs (append lhs1 extra-lhs))
         (setq extra-rhs (append rhs1 extra-rhs))
         (setq start resume))

       (go loop))

    (values extra-lhs extra-rhs)))

;; TEST ::= - +TEST | +TEST
;; +TEST ::= ^ ATTRIBUTE-TEST VALUE-TEST*
;; ATTRIBUTE-TEST ::= ATTRIBUTE-PATH | CONJUNCTIVE-TEST
;;
;; ATTRIBUTE-PATH is an attribute pathname in the new syntax, for
;; example att1.att2.att3.
;;
(defun expand-pseudo-sp*test (start)
  (when (eql (car start) '-)
    (setq start (cdr start)))
  
  (if (eql (car start) '^)
    (setq start (cdr start))
    ;; ELSE
    (progn
      (taql-warn "Missing ^ in condition:~%  ~S" *current-part*)
      (setq start (cdr (member '^ start)))))
  
  (cond ((attribute-path-p (car start))
	 (expand-pseudo-sp*attribute-path-plus-values start))
	(t
	 (multiple-value-bind (lhs1 rhs1 resume1 positive-values)
	     (expand-pseudo-sp*conjunctive-test
	      start 'disallow-attribute-path-symbols)
	   (new-attribute positive-values)
	   (multiple-value-bind (lhs2 rhs2 resume2)
	       (expand-pseudo-sp*value-test* resume1)
	     (values (append lhs1 lhs2)
		     (append rhs1 rhs2)
		     resume2))))))

;; VALUE-TEST ::= { STRUCTURED-VALUE-TEST | CONJUNCTIVE-TEST } {+ | epsilon}
;;
;; STRUCTURED-VALUE-TEST is a tree-structured value specification in the
;; new syntax.
;;
(defun expand-pseudo-sp*value-test* (start)
  (let ((extra-lhs nil)
        (extra-rhs nil))

    ;; Parse value-test*
    ;; loop until find ^ or - or end of list
    (prog ()
      loop
       (when (or (null start)
                 (eql (car start) '^)
                 (eql (car start) '-))
         (return))
      
       (if (structured-value-spec-p (car start))
	 (multiple-value-bind (link-id lhs1 rhs1)
	     (expand-pseudo-sp*structured-value-test (car start))
	   (setq extra-lhs (append lhs1 extra-lhs))
	   (setq extra-rhs (append rhs1 extra-rhs))
	   (setf (car start) link-id)
	   (new-value (list link-id))
	   (setq start (cdr start)))
	 
	 ;; ELSE parse conjunctive-test

	 (multiple-value-bind (lhs1 rhs1 resume positive-values)
	     (expand-pseudo-sp*conjunctive-test start)
	   (new-value positive-values)
	   (setq extra-lhs (append lhs1 extra-lhs))
	   (setq extra-rhs (append rhs1 extra-rhs))
	   (setq start resume)))

       ;; If the value test is followed by a + (a test for an acceptable
       ;; preference), skip over the +.
       (when (eql (car start) '+)
	 (setq start (cdr start)))

       (go loop))

    (values extra-lhs extra-rhs start)))

;; STRUCTURED-VALUE-TEST ::= (CONJUNCTIVE-TEST? CONJUNCTIVE-TEST? TEST+)
;;
;; If only one conjunctive test is present, we have to figure out if it
;; is a class or id.  We assume it is an id if it only contains variables
;; and special symbols (such as { } << >> <> < > <= >= = <=>).
;;
;; Return three values:
;;
;;   1. The identifier variable that will link to the expanded structured
;;      value specification.
;;   2. Extra LHS conditions resulting from expansion.
;;   3. Extra RHS actions resulting from expansion.
;;
(defun expand-pseudo-sp*structured-value-test (spec)
  (let ((extra-lhs nil)
	(extra-rhs nil)
	(spec-start spec)
	(class nil)
	(positive-class-symbols nil)
	(id nil)
	(positive-id-symbols nil)
	(first-conjunction nil))

    ;; Parse CONJUNCTIVE-TEST? CONJUNCTIVE-TEST?

    (if (member (car spec) '(^ -))
      ;; Neither class nor id is present.  Get class from declarations.
      (setq class (list (get-structured-value-type-name))
	    positive-class-symbols class
	    id (list (genvar2 (car positive-class-symbols)))
	    positive-id-symbols id)
      ;; ELSE At least one of class, id is present.
      (multiple-value-bind (lhs1 rhs1 resume1 positive-symbols-1)
	  (expand-pseudo-sp*conjunctive-test spec)
	(setq extra-lhs (append lhs1 extra-lhs))
	(setq extra-rhs (append rhs1 extra-rhs))
	(setq first-conjunction (ldiff spec resume1))
	(setq spec resume1)

	(if (member (car spec) '(^ -))
	  ;; Only one of class, id is present.  Must figure out which.
	  (if (every #'(lambda (x)
			 (or (variable-p x)
			     (member x '({ } << >> <> < > <= >= = <=>))))
		     first-conjunction)
	    ;; No non-variables, assume it is an id.  Get class from
	    ;; declarations.
	    (setq id first-conjunction
		  positive-id-symbols positive-symbols-1
		  class (list (get-structured-value-type-name))
		  positive-class-symbols class)
	    ;; ELSE assume it is a class
	    (setq class first-conjunction
		  positive-class-symbols positive-symbols-1
		  id (list (genvar2 (car positive-class-symbols)))
		  positive-id-symbols id))

	  ;; ELSE both class and id are present.

	  (multiple-value-bind (lhs2 rhs2 resume2 positive-symbols-2)
	      (expand-pseudo-sp*conjunctive-test spec)
	    (setq id (ldiff spec resume2)
		  positive-id-symbols positive-symbols-2)
	    (setq class first-conjunction
		  positive-class-symbols positive-symbols-1)
	    (setq extra-lhs (append lhs2 extra-lhs))
	    (setq extra-rhs (append rhs2 extra-rhs))
	    (setq spec resume2)))))

    (when (null positive-id-symbols)
      (taql-warn "Illegal identifier field in ~S" spec-start)
      (setq positive-id-symbols (list (genvar 'junk))))

    (when (null spec)
      (taql-warn
       "A structured value spec must have at least one attribute test: ~S"
       spec-start))

    (cond ((member '*unknown* positive-class-symbols)
	   (taql-warn "Cannot determine object type for structured value: ~S"
		      spec-start)
	   (values (genvar 'junk) nil nil))
	  (t
	   (save-current-object-components)

	   (new-class positive-class-symbols)
	   (new-identifier positive-id-symbols)
	   
	   ;; Parse TEST*
	   (multiple-value-bind (lhs1 rhs1)
	       (expand-pseudo-sp*test* spec)
	     (setq extra-lhs (append lhs1 extra-lhs))
	     (setq extra-rhs (append rhs1 extra-rhs)))
	   
	   (restore-current-object-components)
	   
	   (values (car positive-id-symbols)
		   (cons `(,@class ,@id ,@spec)
			 extra-lhs)
		   extra-rhs)))))

;;; ----------------------------------------------------------------------
;;; The RHS pseudo-sp expander begins here.
;;; ----------------------------------------------------------------------

(defun expand-pseudo-sp*RHS (rhs)
  (cond (rhs
         (multiple-value-bind (action extra-lhs extra-rhs)
             (expand-pseudo-sp*rhs-action (car rhs))
           (multiple-value-bind (actions extra-lhs-2 extra-rhs-2)
               (expand-pseudo-sp*RHS (cdr rhs))
             (values
              (cons action actions)
              (append extra-lhs extra-lhs-2)
              (append extra-rhs extra-rhs-2)))))
        (t
         (values nil nil nil))))

;; The RHS-ACTION grammer below is a simplification of the actual Soar grammer.
;; So many syntax errors will be caught by Soar, not by TAQL.
;;
;; RHS-ACTION ::= ( NON-MAKE | MAKE )
;; NON-MAKE ::= ANYTHING-BUT-^*
;; MAKE ::= VAR-OR-SYM VAR-OR-SYM ATTRIBUTE-MAKE+
;;
;; VAR-OR-SYM is a variable or a non-special symbol.  Special symbols are
;; { } << >> ^ -.
;;
(defun expand-pseudo-sp*rhs-action (action)
  (when (not (listp action))
    (taql-warn "An action must be a list:~%  ~S" action)
    (return-from expand-pseudo-sp*rhs-action (values nil nil nil)))

  (setq *current-part* action)
  (setq action (copy-list action))

  (let ((extra-lhs nil)
        (extra-rhs nil))

    (cond ((member '^ action) ; A make action
	   (let ((class (car action))
		 (id (cadr action))
		 (att-start (cddr action)))

	     (if (expand-pseudo-sp*check-var-or-sym class)
	       (new-class (list class))
	       ;; ELSE
	       (if (eql class '^)
		 (progn
		   (taql-warn "Missing class and identifier in action:~%  ~S"
			      *current-part*)
		   (setq att-start action))
		 ;; ELSE
		 (taql-warn "Illegal class specifier ~S in action:~%  ~S"
			    class *current-part*)))

	     (when (not (eql class '^))
	       (if (expand-pseudo-sp*check-var-or-sym id)
		 (new-identifier (list id))
		 ;; ELSE
		 (if (eql id '^)
		   (progn
		     (taql-warn "Missing identifier in action:~%  ~S"
				*current-part*)
		     (setq att-start (cdr action)))
		   ;; ELSE
		   (taql-warn "Illegal identifier ~S in action:~%  ~S"
			      id *current-part*))))
	     
	     (multiple-value-setq (extra-lhs extra-rhs)
	       (expand-pseudo-sp*attribute-make+ att-start))))

	  (t                  ; A non-make action
	   ;; For non-make actions, just expand any data macros in the
	   ;; action, and do no further parsing/error-checking.
	   ;;
	   (multiple-value-setq (action extra-lhs extra-rhs)
	     (expand-data-macros-in-list action))))

    (values action extra-lhs extra-rhs)))

(defun expand-pseudo-sp*attribute-make+ (start)
  (let ((extra-lhs nil)
	(extra-rhs nil))

    (prog ()
      (when (null start)
	(taql-warn "A make action must specify at least one attribute~%  ~S"
		   *current-part*))

      loop
       (when (null start) (return))

       (multiple-value-bind (lhs1 rhs1 resume)
           (expand-pseudo-sp*attribute-make start)
         (setq extra-lhs (append lhs1 extra-lhs))
         (setq extra-rhs (append rhs1 extra-rhs))
         (setq start resume))

       (go loop))

    (values extra-lhs extra-rhs)))

;; The grammer below is a simplification of the actual Soar grammer.
;; So many syntax errors will be caught by Soar, not by TAQL.
;;
;; ATTRIBUTE-MAKE ::= ^ VAR-OR-SYM PREFERENCE*
;;    The VAR-OR-SYM here may be an attribute pathname (e.g. att1.att2.att3).
;;
;; VAR-OR-SYM is a variable or a non-special symbol.  Special symbols are
;; { } << >> ^ -.
;;
(defun expand-pseudo-sp*attribute-make (start)
  (if (eql (car start) '^)
    (setq start (cdr start))
    ;; ELSE
    (progn
      (taql-warn "Missing ^ in action:~%  ~S" *current-part*)
      (setq start (cdr (member '^ start)))))
  
  (cond ((attribute-path-p (car start))
	 (expand-pseudo-sp*attribute-path-plus-values start))
	(t
	 (let ((att (car start)))
	   (when (not (expand-pseudo-sp*check-var-or-sym att))
	     (taql-warn "Illegal attribute name ~S in action:~%  ~S"
			(car start) *current-part*)
	     (setq att 'junk))

	   (new-attribute (list att))

	   (multiple-value-bind (lhs1 rhs1 resume)
	       (expand-pseudo-sp*preference* (cdr start))
	     (values lhs1
		     rhs1
		     resume))))))

;; The grammer below is a simplification of the actual Soar grammer.
;; So many syntax errors will be caught by Soar, not by TAQL.
;;
;; PREFERENCE ::= VALUE-MAKE PREFERENCE-SPECIFIER*
;;
(defun expand-pseudo-sp*preference* (start)
  (let ((extra-lhs nil)
        (extra-rhs nil))

    ;; Parse preference*
    ;; loop until find ^ or end of list
    (prog ()
      loop
       (when (or (null start)
                 (eql (car start) '^))
         (return))
      
       (multiple-value-bind (lhs1 rhs1 resume)
	   (expand-pseudo-sp*value-make start)
	 (setq extra-lhs (append lhs1 extra-lhs))
	 (setq extra-rhs (append rhs1 extra-rhs))
	 (setq start resume))
      
       (multiple-value-bind (lhs1 rhs1 resume)
	   (expand-pseudo-sp*preference-specifier* start)
	 (setq extra-lhs (append lhs1 extra-lhs))
	 (setq extra-rhs (append rhs1 extra-rhs))
	 (setq start resume))

       (go loop))

    (values extra-lhs extra-rhs start)))

;; The grammer below is a simplification of the actual Soar grammer.
;; So many syntax errors will be caught by Soar, not by TAQL.
;;
;; VALUE-MAKE ::= DATA-MACRO | STRUCTURED-VALUE-MAKE
;;                           | VARIABLE-OR-CONSTANT-OR-FUNCTION-CALL
;;
(defun expand-pseudo-sp*value-make (start)
  (let ((extra-lhs nil)
        (extra-rhs nil))

    (cond ((data-macro-call-p (car start))
	   (multiple-value-bind (value lhs1 rhs1)
	       (expand-if-data-macro-and-inspect-expansion (car start))
	     (setf (car start) value)
	     (setq extra-lhs (append lhs1 extra-lhs))
	     (setq extra-rhs (append rhs1 extra-rhs))))

	  ((structured-value-spec-p (car start))
	   (multiple-value-bind (link-id lhs1 rhs1)
	       (expand-pseudo-sp*structured-value-make (car start))
	     (setf (car start) link-id)
	     (setq extra-lhs (append lhs1 extra-lhs))
	     (setq extra-rhs (append rhs1 extra-rhs))))

	  ((or (variable-p (car start))
	       (expand-pseudo-sp*check-constant (car start))
	       (consp (car start)))
	   ;; A legal value, but we don't need to do any further processing.
	   ;; We assume that if (car start) is still a list after data-macro
	   ;; calls and structured value specs have been expanded, then it
	   ;; is a function call.
	   )

	  (t
	   (taql-warn
	    "Expected variable, symbol, number, string, data-macro or ~
             function call, or a structured value specification, but ~
             found ~S used as an attribute value."
	    (car start))))

    (new-value (list (car start)))
    (values extra-lhs extra-rhs (cdr start))))

(defun expand-pseudo-sp*preference-specifier* (start)
  (let ((extra-lhs nil)
        (extra-rhs nil)
	(done nil))

    ;; Parse preference-specifier*
    ;; loop until find something that is not a preference specifier,
    ;; or reach the end of the list
    (prog ()
      loop
       (when (or done (null start))
         (return))
      
       (multiple-value-bind (lhs1 rhs1 resume)
	   (expand-pseudo-sp*preference-specifier start)
	 (setq extra-lhs (append lhs1 extra-lhs))
	 (setq extra-rhs (append rhs1 extra-rhs))
	 (setq done (eq start resume))
	 (setq start resume))

       (go loop))

    (values extra-lhs extra-rhs start)))

;; The grammer below is a simplification of the actual Soar grammer.
;; So many syntax errors will be caught by Soar, not by TAQL.
;;
;; PREFERENCE-SPECIFIER ::= epsilon | UNARY-PREF | FORCED-UNARY-PREF
;;                                  | BINARY-PREF VALUE-MAKE
;; UNARY-PREF ::= + | - | ! | ~ | @
;; FORCED-UNARY-PREF ::= BINARY-PREF {, | ) | ^}
;; BINARY-PREF ::= > | = | < | &
;;
(defun expand-pseudo-sp*preference-specifier (start)
  (let ((extra-lhs nil)
        (extra-rhs nil))

    (case (car start)
      ((+ - ! ~ @)                ; unary preference
       (setq start (cdr start)))
      ((> = < &)                  ; maybe unary, maybe binary
       (if (or (null (cdr start))
	       (member (cadr start) '(^ soar::|,|)))
	 ;; Forced unary preference
	 (progn
	   (setq start (cdr start))
	   (when (eql (car start) 'soar::|,|)
	     (setq start (cdr start))))
	 ;; ELSE a binary preference
	 (multiple-value-bind (lhs1 rhs1 resume)
	     (expand-pseudo-sp*value-make (cdr start))
	   (setq extra-lhs (append lhs1 extra-lhs))
	   (setq extra-rhs (append rhs1 extra-rhs))
	   (setq start resume))))
      (t
       ;; Do nothing -- an epsilon preference specifier
       ))

    (values extra-lhs extra-rhs start)))

;; STRUCTURED-VALUE-MAKE ::= ( VAR-OR-SYM? VAR-OR-SYM? ATTRIBUTE-MAKE+ )
;;
;; VAR-OR-SYM is a variable or a non-special symbol.  Special symbols are
;; { } << >> ^ -.
;;
;; If only one VAR-OR-SYM is present, we have to figure out if it
;; is a class or id.  We assume it is an id if it is a variable, and that
;; otherwise it is a class.
;;
;; Return three values:
;;
;;   1. The identifier variable that will link to the expanded structured
;;      value specification.
;;   2. Extra LHS conditions resulting from expansion.
;;   3. Extra RHS actions resulting from expansion.
;;
(defun expand-pseudo-sp*structured-value-make (spec)
  (let ((extra-lhs nil)
	(extra-rhs nil)
	(class nil)
	(id nil)
	(spec-start spec)
	(first-var-or-sym nil))

    ;; Parse VAR-OR-SYM? VAR-OR-SYM?

    (if (eql (car spec) '^)
      ;; Neither class nor id is specified.  Get class from declarations.
      (setq class (get-structured-value-type-name)
	    id (genvar2 class))
      ;; ELSE At least one of class, id is present.
      (progn
	(setq first-var-or-sym (car spec))
	(setq spec (cdr spec))
	
	(if (eql (car spec) '^)
	  ;; Only one of class, id is present.  Must figure out which.
	  (if (variable-p first-var-or-sym)
	    ;; A variable, assume it is an id.  Get class from declarations.
	    (setq id first-var-or-sym
		  class (get-structured-value-type-name))
	    ;; ELSE assume it is a class
	    (setq class first-var-or-sym
		  id (genvar2 first-var-or-sym)))
	  
	  ;; ELSE both class and id are present.
	  
	  (progn
	    (setq id (car spec))
	    (setq class first-var-or-sym)
	    (setq spec (cdr spec))))))

    (when (not (expand-pseudo-sp*check-var-or-sym class))
      (taql-warn "Illegal class ~S in structured value spec:~%  ~S"
		 class spec-start)
      (setq class 'junk-class))

    (when (not (expand-pseudo-sp*check-var-or-sym id))
      (taql-warn "Illegal identifier ~S in structured value spec:~%  ~S"
		 id spec-start)
      (setq id (genvar 'junk)))

    (cond ((eql class '*unknown*)
	   (taql-warn "Cannot determine object type for structured value: ~S"
		      spec-start)
	   (values (genvar 'junk) nil nil))
	  (t
	   (save-current-object-components)

	   (new-class (list class))
	   (new-identifier (list id))

	   ;; Parse ATTRIBUTE-MAKE+
	   (multiple-value-bind (lhs1 rhs1)
	       (expand-pseudo-sp*attribute-make+ spec)
	     (setq extra-lhs (append lhs1 extra-lhs))
	     (setq extra-rhs (append rhs1 extra-rhs)))

	   (restore-current-object-components)

	   (values id
		   extra-lhs
		   (cons `(,class ,id ,@spec)
			 extra-rhs))))))

;;; ----------------------------------------------------------------------
;;; Pseudo-sp expansion routines common both LHS and RHS begin here.
;;; ----------------------------------------------------------------------

;;; The following two predicates are exported routines.

;; EXPORTED ROUTINE:
;; We assume that anything that appears in the value position of a condition
;; or action is a tree-structured value spec if it is a list that contains
;; an attribute (i.e. contains the symbol ^).
;;
(defun structured-value-spec-p (arg)
  (and (consp arg)
       (member '^ arg)))

;; EXPORTED ROUTINE:
;; An attribute pathname is a symbol that contains periods separating the
;; path components.
;;
(defun attribute-path-p (arg)
  (if (symbolp arg)
    (find #\. (the string (symbol-name (the symbol arg))))))

(defun expand-pseudo-sp*check-constant (item)
  (or (and (symbolp item)
           (not (variable-p item))
           (not (member item '({ } << >> ^ -))))
      (numberp item)
      (stringp item)))

;; A variable is always a non-special symbol, so we don't have to do a
;; separate variable-p test below.
;;
(defun expand-pseudo-sp*check-var-or-sym (item)
  (or (and (symbolp item)
           (not (member item '({ } << >> ^ -))))))

;; Expands an attribute pathname and its values.  It takes one
;; argument, a pointer to the attribute pathname in the
;; condition/action.  We assume that this list does begin with an
;; attribute pathname (e.g. a.b.c).
;;
;; It returns three values:
;;
;;   1. Any extra lhs conditions resulting from expansion.
;;   2. Any extra rhs actions resulting from expansion.
;;   3. The location in the condition/action to resume parsing.
;;      This will point to the first thing that appears after the attribute
;;      pathname and its values.
;;
;; The argument list is destructively modified to reflect the top-level
;; effects of the expansion.
;;
;; This works in both conditions and actions.
;;
(defun expand-pseudo-sp*attribute-path-plus-values (start)
  (do* ((pathname (symbol-name (car start)))
	(values-start (cdr start))
	(first-period-pos (position #\. pathname))
	(first-att (intern (subseq pathname 0 first-period-pos)))

	(last-period-pos first-period-pos period-pos)
	(period-pos (position #\. pathname :start (+ 1 first-period-pos))
		    (position #\. pathname :start (+ 1 period-pos)))
	(last-att first-att att)
	(att (intern (subseq pathname (+ 1 last-period-pos) period-pos))
	     (intern (subseq pathname (+ 1 last-period-pos) period-pos)))

	(class (progn (new-attribute (list first-att))
		      (save-current-object-components)
		      (get-structured-value-type-name))
	       next-class)
	(next-class)
	(first-id (genvar2 class))
	(id)
	(last-id first-id id)
	(result nil))

      ((or (null period-pos)
	   (eql class '*unknown*))
       ;; Both (null period-pos) and (eql class '*unknown*) can be true at
       ;; the same time.  We want the (eql class '*unknown*) case to take
       ;; precedence.
       ;;
       (if (not (eql class '*unknown*))
	 ;; Compile successfully so far
	 (progn
	   (new-class (list class))
	   (new-identifier (list last-id))
	   (new-attribute (list att))
	   (multiple-value-bind (lhs1 rhs1 resume)
	       (if (eql *pseudo-sp-context* 'LHS)
		 (expand-pseudo-sp*value-test* values-start)
		 ;; ELSE
		 (expand-pseudo-sp*preference* values-start))
	     (setq result
		   (nreverse
		    (cons
		     `(,class ,last-id ^ ,att ,@(ldiff values-start resume))
		     result)))
	     (if (eql *pseudo-sp-context* 'LHS)
	       (setq lhs1 (append result lhs1))
	       ;; ELSE
	       (setq rhs1 (append result rhs1)))
	     (restore-current-object-components)
	     (new-value (list first-id))
	     (setf (car start) first-att)
	     (setf (cdr start) (cons first-id resume))
	     (values lhs1 rhs1 resume)))
	 ;; ELSE could not determine value class or value class primitive
	 (progn
	   ;; Skip the values, since processing them at this point could
	   ;; lead to spurious type error messages.
	   (let ((resume
		  (if (eql *pseudo-sp-context* 'LHS)
		    (member-if #'(lambda (x) (member x '(^ -))) (cdr start))
		    ;; ELSE
		    (member '^ (cdr start)))))
	     (taql-warn "Unknown or primitive value type for component ~
                         ~S of attribute path ~S"
			last-att
			(car start))
	     (values nil nil resume)))))

    (new-class (list class))
    (new-identifier (list last-id))
    (new-attribute (list att))
    (setq next-class (get-structured-value-type-name))
    (setq id (genvar2 next-class))

    (push `(,class ,last-id ^ ,att ,id)
	  result)))

;; This routine is like expand-if-data-macro, but ensures that a data-macro's
;; expansion is parsed as well (and thus passed through the data-model code,
;; for instance).
;;
(defun expand-if-data-macro-and-inspect-expansion (arg)
  (cond ((data-macro-call-p arg)
	 (save-current-object-components)
	 (multiple-value-bind (value lhs1 rhs1)
	     (expand-data-macro arg)
	   (let ((extra-lhs nil)
		 (extra-rhs nil))
	     (when lhs1
	       (let ((*pseudo-sp-context* 'LHS))
		 (multiple-value-bind (modified-lhs rhs2)
		     (expand-pseudo-sp*production-part lhs1 'LHS)
		   (setq lhs1 modified-lhs)
		   (setq extra-rhs rhs2))))
	     (when rhs1
	       (let ((*pseudo-sp-context* 'RHS))
		 (multiple-value-bind (lhs2 modified-rhs)
		     (expand-pseudo-sp*production-part rhs1 'RHS)
		   (setq rhs1 modified-rhs)
		   (setq extra-lhs lhs2))))
	     (restore-current-object-components)
	     (values value
		     (append lhs1 extra-lhs)
		     (append rhs1 extra-rhs)))))
	(t
	 (values arg nil nil))))

;;; ----------------------------------------------------------------------
;;; This is the end of the production parser
;;; ----------------------------------------------------------------------

;; If the optional argument is supplied (regardless of its value),
;; *pseudo-sp-warning-occurred* is set to non-nil.
;;
;; In any case, the current value of *pseudo-sp-warning-occurred* is
;; returned.
;;
(defun pseudo-sp-warning-occurred (&optional (set nil set-supplied))
  (declare (ignore set))
  (when set-supplied
    (setq *pseudo-sp-warning-occurred* t))
  *pseudo-sp-warning-occurred*)

(defun currently-expanding-pseudo-sp ()
  *currently-expanding-pseudo-sp*)

(eval-when (compile load eval)
  (soarsyntax))
