;;; TAQL Compiler, Data-Model Module
;;;
;;; Gregg Yost, Erik Altmann
;;; School of Computer Science
;;; Carnegie Mellon University
;;;
;;; Working file: /afs/cs/user/altmann/soar/taql/data-model.lisp
;;; Created March 1, 1991
;;;
;;; This file implements the functions involved with TAQL's data-model
;;; facility (type declarations, type checking, ...).
;;;
;;; EXPORTED ROUTINES:
;;;
;;;    - FILL THIS IN
;;;
;;; Known bugs/funnies:
;;; 
;;;    - The global variables should be described where they are declared
;;;      special.
;;;    - It would be nice if TAQL could figure out what operator you were
;;;      talking about in (superoperator ...) conditions, so that it could
;;;      check the attributes.  You can detect these if you think to print
;;;      out the inferred data model an check operator *unknown*, but it is
;;;      easy to forget to do that, and easy to not realize that one of the
;;;      attributes it lists shouldn't be there.
;;;    - Unrestrained use of the new syntax requires some degreee of
;;;      declarations.  If you are operating in the mode of
;;;      code-->print-inferred-model-->check-modify-and-install, this
;;;      discourages using the new syntax initially.
;;;    - For attributes that take values from a limited set, maybe the printer
;;;      should print out defenumerations for the limited sets (which users
;;;      could then edit, or whatever).  Right now, it is too tempting
;;;      to just leave it as :type primitive, with the resulting loss of
;;;      error-checking ability.
;;;    - See /afs/cs/user/doorenbs/soar/taql/db/dm-ideas.txt for Bob's ideas.
;;;
;;; =======================================================================
;;; Modification history:
;;; =======================================================================
;;;
;;; 6-14-91 - gry - Added use-type-defs command.
;;;
;;; 5-22-91 - gry - Finished coding to create distinction between local
;;;   and global types.
;;;
;;; 4-5-91 through 4-22-91 - gry - added space model support (mostly
;;;   just changes to process-model-spec-args-common).
;;;
;;; 3-1-91 - gry - Created.

;;; *** BEGIN CODE ***

(eval-when (compile load eval)
  (lispsyntax))

(eval-when (compile load eval)
   (if (find-package "COMMON-LISP-USER")
       (in-package "COMMON-LISP-USER")
       (in-package "USER")))

;; This shouldn't be reset by restart-soar, so we initialize it here instead
;; of in init-data-model-stuff.  It can only be reset by untrace-load.
;;
(defvar *trace-load-list* nil)

;; These are initialized by init-data-model-stuff.
;;
;; Enveloped in eval-when -- TFMcG 5-Sep-91
(eval-when (compile eval load)
	   (proclaim
	    '(special
	      *current-class*
	      *current-identifiers*
	      *current-attribute*
	      *current-object-components-stack*
	      *saved-operator-augs*
	      *id-class-mapping*
	      *class-attribute-value-set*
	      *all-info-not-declared-types*
	      *sticky-all-info-not-declared-types*
	      *all-types-declared*
	      *sticky-all-types-declared*
	      *all-operators-declared*
	      *sticky-all-operators-declared*
	      *last-aug*-class*
	      *last-aug*-id*
	      *extra-trace-classes*
	      *proposed-operator-id*
	      *inhibit-global-inheritance*

	      *sticky-declarations*
	      ;; *Sticky-declarations* is a list of declarations that appeared
	      ;; in start-default/stop-default sections.  It is stored in reverse
	      ;; order of the order in which the declarations were received.  It
	      ;; only contains declarations that didn't have errors.  These
	      ;; declarations must be reinstated after an excise-task.
	      ;; Each entry has the form (segment-name . declaration), where
	      ;; segment-name is the name of the segment that the declaration appeared
	      ;; in.
	      )))

;; Primitive-p recognizes constant attribute values.
;;
(defun primitive-p (x)
  (or (and (symbolp x)
	   (not (variable-p x)))
      (numberp x)
      (stringp x)))

;; This function tests whether two objects that satisfy primitive-p
;; are equal.  Symbols and numbers are compared with eql (which treats
;; numbers as equal only if they have the same type and value),
;; strings with string= (which is case-sensitive).
;;
(defun primitive-eql (x y)
  (or (eql x y)
      (and (stringp x)
	   (stringp y)
	   (string= x y))))

;; This assumes both of its arguments satisfy primitive-p.  This is not
;; case sensitive.  Numbers are numerically ordered among themselves,
;; and are made less than any string or symbol.  Strings and symbols
;; are ordered alphabetically (using the print name, for symbols).
;;
(defun primitive-lessp (x y)
  (etypecase x
    (symbol (etypecase y
	      (symbol (string-lessp (symbol-name x) (symbol-name y)))
	      (number nil)
	      (string (string-lessp (symbol-name x) y))))
    (number (etypecase y
	      ((or symbol string) t)
	      (number (< x y))))
    (string (etypecase y
	      (symbol (string-lessp x (symbol-name y)))
	      (number nil)
	      (string (string-lessp x y))))))

(defun special-type-name-p (name)
  (or (eql name '*unknown*)
      (operator-type-name-p name)))

(defun operator-type-name-p (name)
  (and (symbolp name)
       (eql 0 (search "OPERATOR*" (symbol-name name)))))

;; This assumes its argument satisfies operator-type-name-p.  It returns the
;; name of the associated operator, as a symbol.
;;
(defun operator-type-operator-name (type-name)
  (intern (subseq (symbol-name type-name) 9)))

(defmacro defprimitive (&body body)
  `(defprimitive-aux ',body 'defprimitive))

(defmacro defenumeration (&body body)
  `(defprimitive-aux ',body 'defenumeration))

(defun defprimitive-aux (body variant)
  (let ((type-name (car body)))

    (setq *segment-name-most-recently-added-to*
	  *current-segment-name*)

    (cond ((null body)
	   (taql-warn2 "~@(~S~):  Missing type name:  ~S"
		       variant (cons variant body)))
	  ((not (symbolp type-name))
	   (taql-warn2 "~@(~S~):  Type name must be a symbol.  ~S"
		       variant (cons variant body)))
	  ((special-type-name-p type-name)
	   (taql-warn2 "~@(~S~):  Type ~S cannot be declared."
		       variant type-name))
	  ((gethash type-name (get-data-model 'declared '*global*))
	   (taql-warn2 "~@(~S~):  Type ~S has already been declared."
		       variant type-name))
	  ((and (eql variant 'defprimitive)
		(or (not (cdr body))
		    (cddr body)))
	   (taql-warn2 "Defprimitive ~S:  Usage:  (defprimitive TYPENAME ~
                        PREDICATE-NAME)"
		       type-name))
	  ((and (eql variant 'defprimitive)
		(or (not (symbolp (cadr body)))
		    (not (fboundp (cadr body)))))
	   (taql-warn2
	    "Defprimitive ~S:  ~S is not the name of a defined function."
	    type-name (cadr body)))
	  ((and (eql variant 'defenumeration)
		(not (cdr body)))
	   (taql-warn2
	    "Defenumeration ~S:  An enumeration must have at least one value."
	    type-name))
	  ((and (eql variant 'defenumeration)
		(member-if-not #'primitive-p (cdr body)))
	   (taql-warn2
	    "Defenumeration ~S:  All enumerated values must be non-variable ~
             symbols, numbers, or strings."
	    type-name))
	  (t
	   (when *taql-default-mode*
	     (push (cons *current-segment-name* (cons variant body))
		   *sticky-declarations*))
	   (install-primitive type-name
			      (if (eql variant 'defprimitive)
				(cadr body)
				;; ELSE
				(cdr body))))))
  t)

;; This assumes defprimitive-aux didn't detect any errors in the
;; arguments.
;;
(defun install-primitive (type-name primitive)
  (let* ((install-in-segments
	  (if (eql *current-segment-name* '*global*)
	    '(*global*)
	    ;; ELSE
	    (list '*global* *current-segment-name*)))
	 (install-in-models
	  (mapcar #'(lambda (segment-name)
		      (get-data-model 'declared segment-name))
		  install-in-segments)))

    (dolist (data-model install-in-models)
      (add-type-to-model data-model type-name primitive))))

(defmacro defobject (&body body)
  `(defobject-aux ',body 'defobject))

(defmacro defobject-merge (&body body)
  `(defobject-aux ',body 'defobject-merge))

(defmacro defoperator (&body body)
  `(defobject-aux ',body 'defoperator))

(defmacro defoperator-merge (&body body)
  `(defobject-aux ',body 'defoperator-merge))

(defun defobject-aux (body variant)
  (let* ((type-name (car body))
	 (original-type-name type-name)
	 (error-found-1 nil)
	 (local-model (get-data-model 'declared *current-segment-name*))
	 (global-model (get-data-model 'declared '*global*))
	 (global nil))

    (setq *segment-name-most-recently-added-to*
	  *current-segment-name*)

    (cond ((null body)
	   (setq error-found-1 t)
	   (taql-warn2 "~@(~S~):  Missing type name:  ~S"
		       variant (cons variant body)))
	  ((not (symbolp type-name))
	   (setq error-found-1 t)
	   (taql-warn2 "~@(~S~):  Type name must be a symbol.  ~S"
		       variant (cons variant body)))
	  ((null type-name)
	   (setq error-found-1 t)
	   (taql-warn2 "~@(~S~):  Nil is not a legal type name."
		       variant))
	  ((special-type-name-p type-name)
	   (setq error-found-1 t)
	   (taql-warn2 "~@(~S~):  Type ~S cannot be declared."
		       variant type-name))
	  ((member type-name '(:all-info-declared :global))
	   (setq error-found-1 t)
	   (taql-warn2 "~@(~S~):  Missing type name (found ~(~S~) instead)."
		       variant type-name)))
  
    (when (not error-found-1)
      (multiple-value-setq (body global error-found-1)
	(find-global-in-decl variant body)))
      
    (when (not error-found-1)
      (when (member variant '(defoperator defoperator-merge))
	(setq type-name (makesym 'operator* type-name)))
      
      (cond
       ((and (member variant '(defobject defoperator))
	     (gethash type-name (if global global-model local-model)))
	(setq error-found-1 t)
	(taql-warn2 "~@(~S~):  ~@(~S~) ~S has already been declared."
		    variant
		    (if (eql variant 'defobject) 'type 'operator)
		    original-type-name))
       
       ((and (gethash type-name local-model)
	     (not (eql global
		       (type-info-global (gethash type-name local-model)))))
	(setq error-found-1 t)
	(taql-warn2 "~@(~S~) ~S:  Global and local declarations for a ~
                     type cannot appear in the same segment."
		    variant type-name))
       
       ((and global
	     (get-primitive type-name))
	(setq error-found-1 t)
	(taql-warn2 "~@(~S~):  Type ~S is currently declared as ~
                     a primitive type."
		    variant type-name))
       (t
	(multiple-value-bind (modified-body all-info-declared error-found)
	    (find-all-info-declared-in-decl variant body)
	  (when (not error-found)
	    (setq body
		  `(,(car modified-body)
		    :global ,(if global 'yes 'no)
		    ,@(cdr modified-body)))
	    (multiple-value-bind (att-decls error-detected)
		(check-attribute-decls original-type-name (cdddr body) variant)
	      (when (not error-detected)
		(let* ((inherit-globals
			(and (not *inhibit-global-inheritance*)
			     (not global)
			     (not (eql *current-segment-name* '*global*))
			     (not (get-primitive type-name))
			     (get-type-info type-name 'declared '*global*)))
		       (type-info
			(att-decls-to-type-info type-name att-decls
						inherit-globals global)))
		  (when (and global
			     (not (eql *current-segment-name* '*global*)))
		    (merge-type-into-data-model type-info global-model))
		  (merge-type-into-data-model type-info local-model)
		  (when *taql-default-mode*
		    (push (cons *current-segment-name*
				(if inherit-globals
				  (type-info-to-decl-list type-info variant)
				  ;; ELSE
				  (cons variant body)))
			  *sticky-declarations*))
		  (when all-info-declared
		    (eval `(all-info-declared
			    ,all-info-declared
			    ,(if (or global
				     (eql *current-segment-name* '*global*))
			       type-name
			       ;; ELSE
			       (list *current-segment-name*
				     type-name))))))))))))))
  t)

(defmacro use-type-defs (&body body)
  `(use-type-defs-aux ',body))

(defun use-type-defs-aux (body)
  (let* ((args (group-arguments body '(:global :all-info-declared)
				'use-type-defs))
	 (globals (assoc :global (cdr args)))
	 (all-info-declareds (assoc :all-info-declared (cdr args)))
	 (type-args (car args))
	 (error-detected nil))

    (when (null type-args)
      (setq error-detected t)
      (taql-warn2 "Use-type-defs requires at least one type argument."))

    (when (cddr globals)
      (setq error-detected t)
      (taql-warn2 "Use-type-defs takes at most one :global keyword."))

    (when (cddr all-info-declareds)
      (setq error-detected t)
      (taql-warn2
       "Use-type-defs takes at most one :all-info-declared keyword."))

    (when (and globals
	       (not (member (cadr globals) '(yes no))))
      (setq error-detected t)
      (taql-warn2
       "Use-type-defs:  Value of :global must be yes or no, not ~S"
       (cadr globals)))

    (when (and all-info-declareds
	       (not (member (cadr all-info-declareds) '(yes no))))
      (setq error-detected t)
      (taql-warn2
       "Use-type-defs:  Value of :all-info-declared must be yes or no, not ~S"
       (cadr all-info-declareds)))

    (dolist (arg type-args)
      (cond
       ((and (not (symbolp arg))
	     (not (and (consp arg)
		       (cdr arg)
		       (every #'symbolp arg))))
	(taql-warn2 "Use-type-defs:  Expected TYPE-NAME or ~
                     (SEGMENT-NAME TYPE-NAME+), but found ~S"
		    arg))
       (t
	(let ((segment-name (if (symbolp arg) '*global* (car arg)))
	      (type-names (if (symbolp arg) (list arg) (cdr arg))))
	  (dolist (type-name type-names)
	    (let ((type-info
		   (gethash type-name
			    (get-data-model 'declared segment-name))))
	      (when (and type-info
			 (type-info-global type-info)
			 (not (eql segment-name '*global*)))
		;; We want to ignore global declarations stored in local
		;; segments.
		(setq type-info nil))
	      (cond ((not type-info)
		     (if (eql segment-name '*global*)
		       (taql-warn2 "Use-type-defs:  Type ~S is not ~
                                    declared globally."
				   type-name)
		       ;; ELSE
		       (taql-warn2 "Use-type-defs:  Type ~S is not ~
                                    declared locally in segment ~S."
				   type-name segment-name)))
		    ((type-info-primitive type-info)
		     (taql-warn2
		      "Use-type-defs takes only non-primitive type ~
                       names as arguments, but ~S was declared primitive."
		      arg))
		    ((not error-detected)
		     (install-use-type-def type-info (cadr globals)
					   (cadr all-info-declareds)))))))))))
    t)

(defun install-use-type-def (type-info global all-info-declared)
  (let* ((variant (if (operator-type-name-p (type-info-name type-info))
		    'defoperator-merge
		    ;; ELSE
		    'defobject-merge))
	 ;; The find-global-in-decl strips out the :global keyword.
	 ;; We don't need its other return values here.
	 (decl (find-global-in-decl
		variant
		(type-info-to-decl-list type-info variant))))

    (setq decl
	  `(,(car decl) ,(cadr decl)
	    ,@(if global
		(list :global global))
	    ,@(if all-info-declared
		(list :all-info-declared all-info-declared))
	    ,@(cddr decl)))

    (eval decl)))

;; We assume type-info is non-primitive.  We convert it to a list in the
;; form of a declaration.  This list will NOT have an :all-info-declared
;; keyword in it.
;;
(defun type-info-to-decl-list (type-info variant)
  (with-input-from-string
     (stream (with-output-to-string (*standard-output*)
	        (print-structured-data-type type-info
					    nil
					    :decl-command variant
					    :print-all-info-declared nil)))
   (read stream)))

;; We want to remove any :all-info-declared keyword from a declaration
;; argument list so that (1) we know all the rest of the arguments are
;; attribute declarations and (2) when we re-evaluate a sticky declaration,
;; we don't want to evaluate the :all-info-declared part again.
;;
(defun find-all-info-declared-in-decl (variant args)
  (let ((type-name (car args))
	(keyword (member :all-info-declared (cdr args))))
    (cond ((not keyword)
	   (values args nil nil))
	  ((not (member (cadr keyword) '(yes no)))
	   (taql-warn2 "~@(~S~) ~S:  The value of :all-info-declared must be ~
                        yes or no, not ~S"
		       variant type-name (cadr keyword))
	   (values args nil t))
	  ((member :all-info-declared (cdr keyword))
	   (taql-warn2 "~@(~S~) ~S:  The :all-info-declared keyword can ~
                        appear at most once."
		       variant type-name)
	   (values args nil t))
	  (t
	   (values (append (ldiff args keyword)
			   (cddr keyword))
		   (if (eql (cadr keyword) 'yes)
		     :yes
		     ;; ELSE
		     :no)
		   nil)))))

;; We want to remove any :global keyword from a declaration
;; argument list so that we know all the rest of the arguments are
;; attribute declarations.
;;
(defun find-global-in-decl (variant args)
  (let ((type-name (car args))
	(keyword (member :global (cdr args))))
    (cond ((not keyword)
	   (values args nil nil))
	  ((not (member (cadr keyword) '(yes no)))
	   (taql-warn2 "~@(~S~) ~S:  The value of :global must be ~
                        yes or no, not ~S"
		       variant type-name (cadr keyword))
	   (values args nil t))
	  ((member :global (cdr keyword))
	   (taql-warn2 "~@(~S~) ~S:  The :global keyword can ~
                        appear at most once."
		       variant type-name)
	   (values args nil t))
	  (t
	   (values (append (ldiff args keyword)
			   (cddr keyword))
		   (eql (cadr keyword) 'yes)
		   nil)))))

(defun check-attribute-decls (type-name att-decls variant)
  (let ((result nil)
	(error-detected nil)
	(att-name nil)
	(value-type-names nil)
	(required nil)
	(attribute-required-default
	 (member variant '(defoperator defoperator-merge))))

    (dolist (decl att-decls)
      (setq att-name nil)
      (setq value-type-names nil)
      (setq required attribute-required-default)
      (typecase decl
	(symbol (setq att-name decl))
	(list (multiple-value-setq (att-name required value-type-names
					     error-detected)
		(check-attribute-decl-list-form decl type-name variant)))
	(t
	 (taql-warn2 "~@(~S~) ~S:  Malformed attribute declaration ~S"
		     variant
		     type-name
		     decl)
	 (setq error-detected t)))

      (cond ((eql att-name '*unknown*)
	     (taql-warn2
	      "~@(~S~) ~S:  The *unknown* attribute cannot be declared."
	      variant type-name)
	     (setq error-detected t))
	    ((member '*unknown* value-type-names)
	     (taql-warn2 "~@(~S~) ~S:  An attribute's value type cannot be ~
                          declared to be *unknown*."
			 variant type-name)
	     (setq error-detected t))
	    ((member att-name result :key #'car)
	     (taql-warn2
	      "~@(~S~) ~S:  Attribute ~S was declared more than once."
	      variant type-name att-name)
	     (setq error-detected t))
	    ((not error-detected)
	     (push (cons att-name (cons required value-type-names)) result))))

    (values result error-detected)))

(defun check-attribute-decl-list-form (decl type-name variant)
  (let* ((att-name nil)
	 (value-type-names nil)
	 (error-detected nil)
	 (doing-operator-decl (member variant
				      '(defoperator defoperator-merge)))
	 (attribute-required
	  ;; The default is required for attributes of specific operators
	  doing-operator-decl))

    (if (symbolp (car decl))
      (setq att-name (car decl))
      ;; ELSE
      (progn
	(taql-warn2 "~@(~S~) ~S:  Attribute name must be a symbol: ~S"
	 variant
	 type-name
	 decl)
	(setq error-detected t)))

    (prog ((decl-ptr (cdr decl)))
     loop
      (when (null decl-ptr)
	(return))

      (case (car decl-ptr)
	(:optional (if doing-operator-decl
		     (setq attribute-required nil)
		     ;; ELSE
		     (progn
		       (taql-warn2
			"~@(~S~) ~S:  The :optional keyword is not meaningful ~
                         in ~(~S~): ~S"
			variant type-name variant decl)
		       (setq error-detected t))))
	(:type (if (and (cdr decl-ptr)
			(or (symbolp (cadr decl-ptr))
			    (consp (cadr decl-ptr))))
		 (let ((temp-value-type-names nil))
		   (setq temp-value-type-names (cadr decl-ptr))
		   (when (null temp-value-type-names)
		     (taql-warn2
		      "~@(~S~) ~S:  Nil is not a legal value type:  ~S"
		      variant type-name decl)
		     (setq error-detected t))
		   (when (symbolp temp-value-type-names)
		     (setq temp-value-type-names (list temp-value-type-names)))
		   (let ((error-element
			  (find-if-not #'symbolp temp-value-type-names)))
		     (if error-element
		       (taql-warn2
			"~@(~S~) ~S:  Non-symbolic type name ~S in ~
                             value type:  ~S"
			variant type-name error-element decl)
		       ;; ELSE
		       (when (setq error-element
				   (find-if #'special-type-name-p
					    temp-value-type-names))
			 (taql-warn2
			  "~@(~S~) ~S:  Value type name cannot be ~
                               declared to be ~S:  ~S"
			  variant type-name error-element decl)))
		     (when error-element
		       (setq error-detected t)))
		   (when (not error-detected)
		     (setq value-type-names
			   (append temp-value-type-names value-type-names))))
		 ;; ELSE
		 (progn
		   (taql-warn2
		    "~@(~S~) ~S:  The value of :type must be a symbol or a ~
                     list of symbols: ~S"
		    variant type-name decl)
		   (setq error-detected t)))
	       (setq decl-ptr (cdr decl-ptr)))
	(t
	 (taql-warn2
	  "~@(~S~) ~S:  Expected ~S, but found ~S: ~S"
	  variant
	  type-name
	  (if doing-operator-decl
	    ":type or :optional"
	    ;; ELSE
	    ":type")
	  (car decl-ptr)
	  decl)
	 (setq error-detected t)))

      (setq decl-ptr (cdr decl-ptr))
      (go loop))

  (values att-name attribute-required value-type-names error-detected)))

;; This expects an att-decls argument of the form returned by
;; check-attribute-decls, and assumes that check-attribute-decls has
;; already be called and did not detect any errors.
;;
;; If inherit-globals is non-nil, then it is a type-info object that should
;; be merged into the type info for type-name.
;;
;; Return the resulting type-info structure.
;;
(defun att-decls-to-type-info (type-name att-decls inherit-globals global)
  (let ((type-info (make-type-info :name type-name
				   :primitive nil
				   :global global)))

    (when inherit-globals
      (setq type-info
	    (merge-type-infos inherit-globals type-info)))

  (dolist (decl att-decls)
    (let ((att-info (add-attribute-to-type-info type-info (car decl))))

      ;; We only set required to nil, never to T (its default value).
      ;; This is so that if an attribute is declared :optional in one
      ;; declaration, and something is added to that attribute's declaration
      ;; later by defoperator-merge, it will remain optional even if it isn't
      ;; specified the second time.
      ;;
      (when (not (cadr decl))
	(setf (attribute-info-required att-info) nil))

      (dolist (value-type-name (cddr decl))
	(pushnew value-type-name (attribute-info-value-types att-info)))))

  type-info))

;; See the release notes for a description of the arguments print-data-model
;; takes.  THIS IS NOW OBSOLETE CODE, PRINT-DATA-MODEL HAS CHANGED.  I'M
;; KEEPING THIS AROUND FOR A WHILE IN CASE I WANT TO CHANGE THINGS BACK.
;;
(defun process-data-model-spec-args (args)
  (multiple-value-bind (include-parsed-triples exclude-parsed-triples)
      (process-model-spec-args-common args 'type)

    (data-model-difference
     (data-model-union (mapcar #'form-data-model include-parsed-triples))
     (data-model-union (mapcar #'form-data-model exclude-parsed-triples)))))

;; See the parse-model-spec-triple function for a description of the form
;; of the parsed-data-model-spec argument.
;;
;; This returns a data model formed by taking the union of all the specified
;; data models and, if the type sepcifier is not :all, including only
;; the named data types (otherwise all types in the union are included in
;; the result).
;;
(defun form-data-model (parsed-data-model-spec)
  (let ((model-union
	 (data-model-union
	  (mapcar #'(lambda (model-spec)
		      (get-data-model (car model-spec) (cadr model-spec)))
		  (cdr parsed-data-model-spec))))
	(type-spec (car parsed-data-model-spec)))

    (cond ((eql type-spec :all)
	   model-union)
	  ((null type-spec)
	   ;; Return an empty data model
	   (new-data-model))
	  (t
	   (include-exclude-types model-union type-spec nil)))))

;; This function expands arguments for either {print,excise}-data-model or
;; {print,excise}-space-model.  It returns two values:
;;
;;    (1) A list of parsed model spec triples that appeared before the
;;        first :exclude in the argument list.
;;
;;    (2) A list of parsed model spec triples that appeared after the
;;        first :exclude in the argument list.
;;
;; The format of a parsed model spec triple is given in the documentation
;; for the parse-model-spec-triple function.
;;
;; Any default argument values are returned as part of the return values.
;;
;; The variant argument must be on of the symbols 'type or 'space, indicating
;; whether the argument list being processed is for data models or
;; space models, respectively.
;;
;; When an error is detected, all return values are nil.
;;
;; When the optional allow-exclude-args argument is nil (the default is T),
;; it is an error for the argument list to contain :exclude,
;; :dont-exclude-taql, or :dont-exclude-soar.
;;
(defun process-model-spec-args-common (args variant
					    &optional (allow-exclude-args t))

  (let ((include-parsed-triples nil)
	(exclude-parsed-triples nil)
	(including t)
	(exclude-taql t)
	(exclude-soar t)
	(error-detected nil))

    (dolist (arg args)
      (case arg
	(:exclude
	 (if allow-exclude-args
	   (setq including nil)
	   ;; ELSE
	   (progn
	     (setq error-detected t)
	     (taql-warn2 ":Exclude is not a valid argument here."))))
	
	(:dont-exclude-taql
	 (if allow-exclude-args
	   (setq exclude-taql nil)
	   ;; ELSE
	   (progn
	     (setq error-detected t)
	     (taql-warn2 ":Dont-exclude-taql is not a valid argument here."))))
	
	(:dont-exclude-soar
	 (if allow-exclude-args
	   (setq exclude-soar nil)
	   ;; ELSE
	   (progn
	     (setq error-detected t)
	     (taql-warn2 ":Dont-exclude-soar is not a valid argument here."))))

	(otherwise
	 (multiple-value-bind (parsed-triple error-detected-1)
	     (parse-model-spec-triple arg variant)
	   (when error-detected-1
	     (setq error-detected t))
	   (when (not error-detected)
	     (if including
	       (pushnew parsed-triple include-parsed-triples :test #'equal)
	       ;; ELSE
	       (pushnew parsed-triple exclude-parsed-triples
			:test #'equal)))))))

    (cond
     (error-detected
      (values nil nil))
     (t
      (when (null include-parsed-triples)
	(setq include-parsed-triples
	      (list (parse-model-spec-triple '(all :all :all) variant))))
      (when exclude-taql
	(when (eql variant 'type)
	  (pushnew
	   (parse-model-spec-triple
	    '(all
	      (aug* control-stuff* link-index* taql-stuff*)
	      :all)
	    variant)
	   exclude-parsed-triples
	   :test #'equal))
	(pushnew
	 (parse-model-spec-triple '(declared :all taql) variant)
	 exclude-parsed-triples
	 :test #'equal))
      (when exclude-soar
	(pushnew
	 (parse-model-spec-triple '(declared :all soar) variant)
	 exclude-parsed-triples
	 :test #'equal))
      
      (values include-parsed-triples exclude-parsed-triples)))))

;; Parse a single argument of either print-data-model or print-space-model,
;; depending on the value or the variant parameter (whose value must be
;; either 'space or 'type).
;;
;; The form of an argument to either of these commands is
;;
;;   MODEL-ITEMS-SPEC ::= ( [ MODEL-TYPE [ ITEM-PART [ SEGMENT-PART ] ] ] )
;;
;; where
;;
;;   MODEL-TYPE ::= inferred | declared | all
;;     The default is all.
;;
;;   ITEM-PART ::= :all | ITEM-NAME | ( ITEM-NAME+ )
;;     where ITEM name is either a type name or a problem space name, depending
;;     on the value of Variant.  The default is :all.
;;
;;   SEGMENT-PART ::= SEGMENT-SPEC | ( SEGMENT-SPEC+ )
;;     The default is :all.
;;     (:all is a SEGMENT-SPEC, so we don't have to explicitly list it here)
;;
;; Returns two values:
;;
;;   1. A parsed specification of the form ( ITEM-PART . MODEL-SPEC* ) where
;;        MODEL-SPEC ::= ( {inferred | declared} SEGMENT-NAME )
;;      and the MODEL-SPEC is expanded from the MODEL-TYPE and SEGMENT-PART
;;      parts of the argument.  ITEM-PART is either :all or a list of items
;;      (it will never be a single symbolic item).
;;
;;   2. A flag that is non-nil iff an error was detected.  The first
;;      return value is undefined when an error is detected.
;;
(defun parse-model-spec-triple (spec variant)
  (when (not (member variant '(type space)))
    (error "INTERNAL TAQL ERROR:  Illegal variant argument ~S" variant))

  (cond
   ((listp spec)
    (let ((model-type (if spec (car spec) 'all))
	  (item-part (if (cdr spec) (cadr spec) :all))
	  (segment-part (if (cddr spec) (caddr spec) :all))
	  (error-detected nil))

      (when (cdddr spec)
	(setq error-detected t)
	(taql-warn2
	 "A MODEL-ITEMS-SPEC can have at most three element:  ~S"
	 spec))

      (multiple-value-bind (model-specs error-detected-1)
	  (expand-multi-model-spec (cons model-type
					 (if (consp segment-part)
					   segment-part
					   ;; ELSE
					   (list segment-part))))
	(when error-detected-1
	  (setq error-detected t))

	(cond ((eql item-part :all)
	       ;; do nothing
	       )
	      ((symbolp item-part)
	       ;; Put the item in a list, so that we'll have more uniform
	       ;; return values (always a list or :all).
	       (setq item-part (list item-part)))
	      ((listp item-part)
	       (when (member-if-not #'symbolp item-part)
		 (taql-warn2
		  "A ~:@(~S~) list must contain only symbolic ~(~S~) ~
                   names:  ~S"
		  variant variant spec)
		 (setq error-detected t)))
	      (t
	       (taql-warn2
		"Expected :all, a ~(~S~) name, or a list of ~(~S~) names, but ~
                 found ~S"
		variant variant item-part)
	       (setq error-detected t)))

	(values (cons item-part model-specs) error-detected))))

   (t ; spec is not a list
    (taql-warn2 "Expected a MODEL-ITEMS-SPEC but found ~S" spec)
    (values nil t))))

;; A legal spec argument has the form
;;
;;    SEGMENT-SPEC | ({inferred | declared | all} . SEGMENT-SPEC*)
;;
;; It returns two values:
;;
;;  1. A list each of whose elements has the form (MODEL-TYPE SEGMENT-NAME),
;;     where MODEL-TYPE is either inferred or declared (not 'all' -- it is
;;     expanded to separate inferred and declared entries) and the
;;     SEGMENT-NAMEs are expanded from SEGMENT-SPEC.  The simple SEGMENT-SPEC
;;     form is equivalent to (all SEGMENT-SPEC).  When no SEGMENT-SPECs
;;     are given in the list form, it defaults to :all.
;;
;;  2. An flag that is non-nil iff an error was detected in the argument.
;;     When an error is detected, the first return value is undefined.
;;
(defun expand-multi-model-spec (spec)
  (cond ((atom spec)
	 (let ((result nil)
	       (segment-names (expand-segment-spec spec)))
	   (dolist (segment-name segment-names)
	     (setq result
		   (cons (list 'inferred segment-name)
			 (cons (list 'declared segment-name)
			       result))))
	   (values result (not segment-names))))

	((member (car spec) '(inferred declared all))
	 (when (not (cdr spec))
	   (setq spec (list (car spec) :all)))

	 (let ((types (if (eql (car spec) 'all)
			'(inferred declared)
			;; ELSE
			(list (car spec))))
	       (error-detected nil)
	       (result nil))

	   (dolist (segment-spec (cdr spec))
	     (let ((segment-names (expand-segment-spec segment-spec)))
	       (if segment-names
		 (dolist (segment-name segment-names)
		   (dolist (type types)
		     (pushnew (list type segment-name) result :test #'equal)))
		 ;; ELSE
		 (setq error-detected t))))

	   (values result error-detected)))

	(t
	 (taql-warn2 "Expected one of Inferred, Declared, or All, ~
                      but found ~S"
		     (car spec))
	 (values nil t))))

;; Data-model must be either a data model or nil.  If it is nil, return nil.
;; Otherwise return a data model formed from data-model by
;; including only types whose names appear in include-types if include-types
;; is non-nil, and excluding any types whose names appear in exclude-types.
;; 
;; If the same type name appears in both include-types and exclude-types,
;; the exclude takes precedence.
;;
;; The resulting data-model may share structure with data-model (and they may
;; in fact be eq in some cases).  The argument data-model is not
;; destructively modified.
;;
(defun include-exclude-types (data-model include-types exclude-types)
  (when data-model
    (cond (include-types
	   (let ((result-model (new-data-model
				(round (* 1.5 (length include-types))))))
	     (dolist (type-name include-types)
	       (when (not (member type-name exclude-types))
		 (let ((info (gethash type-name data-model)))
		   (when info
		     (setf (gethash type-name result-model) info)))))
	     result-model))

	  (exclude-types
	   (let ((result-model (copy-hash-table data-model)))
	     (dolist (type-name exclude-types)
	       (remhash type-name result-model))
	     result-model))

	  (t
	   data-model))))

;; Every item in model-list is assumed to be either a data model or nil.
;;
;; This will NOT destructively modify any of its argument models.  Also,
;; the model returned will not share structure with any of the argument
;; models.
;;
(defun data-model-union (model-list)
  (cond ((null model-list)
	 nil)
	((null (car model-list))
	 ;; This case avoids calling data-model-copier on nil in the next
	 ;; case.
	 (data-model-union (cdr model-list)))
	(t
	 (data-model-union-aux (data-model-copier (car model-list))
			       (data-model-union (cdr model-list))))))

;; This may destructively modify one or both of its argument models.
;;
(defun data-model-union-aux (model-1 model-2)
  (cond ((null model-1)
	 model-2)
	((null model-2)
	 model-1)
	(t
	 ;; Merge model-1 into model-2, rather than vice-versa, since
	 ;; because of the way data-model-union is defined model-2 is
	 ;; likely to be larger.
	 (maphash #'(lambda (type-name type-info)
		      (declare (ignore type-name))
		      (merge-type-into-data-model type-info model-2))
		  model-1)
	 model-2)))

;; Destructively merges type-info into data-model.  The type info merged
;; into data-model will not share structure with the type-info
;; argument.  Return the modified data-model.
;;
(defun merge-type-into-data-model (type-info data-model)
  (let ((type-name (type-info-name type-info)))
    (setf (gethash type-name data-model)
	  (merge-type-infos type-info (gethash type-name data-model)))
    data-model))

;; Merge two type-info structures, returning the new type-info structure.
;; The second type-info argument may be destructively modified, but the
;; result will not share structure with type-info-1.  Either of the
;; arguments may be nil.  We assume that both type-info arguments, if non-nil,
;; are for types with the same name.  The value of the global field in the
;; result is the value from type-info-2 if type-info-2 is non-nil, otherwise
;; it is the value from type-info-1.
;;
;; If both arguments are nil, we return nil, since we wouldn't known what name
;; to give a resulting type-info structure.
;;
(defun merge-type-infos (type-info-1 type-info-2)
  (when type-info-1
    (setq type-info-1 (type-info-copier type-info-1)))
  (cond
   ((null type-info-1)
    type-info-2)
   ((null type-info-2)
    type-info-1)
   (t
    (let ((type-name (type-info-name type-info-1))
	  (prim-1 (type-info-primitive type-info-1))
	  (prim-2 (type-info-primitive type-info-2)))

      (cond ((and prim-1
		  prim-2
		  ;; Use equal here for the enumeration case.
		  (not (equal prim-1 prim-2)))
	     (taql-warn2 "Attempted to merge two different definitions for ~
                          primitive type ~S.  This cannot be done.  The two ~
                          definitions are ~S and ~S"
			 type-name prim-1 prim-2))
	    ((or (and prim-1
		      (not prim-2))
		 (and (not prim-1)
		      prim-2))
	     (taql-warn2 "Attempted to merge primitive and non-primitive ~
                          definitions for type ~S.  This cannot be done."
			 type-name))
	    (t
	     ;; Neither is primitive.
	     (setf (type-info-attribute-info type-info-2)
		   (merge-attribute-info-tables
		    (type-info-attribute-info type-info-1)
		    (type-info-attribute-info type-info-2)))))

      type-info-2))))

;; Non-destructively merge two type info structures.  Either argument may be
;; nil.  The result will not share structure with either argument.  Return nil
;; if both arguments are nil.
;;
(defun type-info-union (type-info-1 type-info-2)
  (if type-info-2
    (merge-type-infos type-info-1 (type-info-copier type-info-2))
    ;; ELSE
    type-info-1))

;; Merges att-info-1 into att-info-2 (either of which may be nil).
;; The merge may destructively modify att-info-2, the merged att-info
;; may or may not share structure with either of the arguments.
;; Returns the merged attribute-info structure, or nil if both of its
;; arguments are nil.
;;
(defun merge-attribute-info-tables (att-info-1 att-info-2)
  (cond ((null att-info-1)
	 att-info-2)
	((null att-info-2)
	 att-info-1)
	(t
	 (maphash #'(lambda (att-name att-info)
		      (declare (ignore att-name))
		      (merge-att-info-into-table att-info att-info-2))
		  att-info-1)
	 att-info-2)))

;; Destructively merges att-info into att-info-table.  The info merged
;; into the table may or may not share structure with the att-info
;; argument.  Return the modified att-info-table.
;;
(defun merge-att-info-into-table (att-info att-info-table)
  (let ((att-name (attribute-info-name att-info)))

    (let ((old-info (gethash att-name att-info-table)))
      (if (not old-info)
	(setf (gethash att-name att-info-table) att-info)
	;; ELSE
	(let ((new-value-types (attribute-info-value-types att-info)))
	  (when new-value-types
	    (setf (attribute-info-value-types old-info)
		  (union new-value-types
			 (attribute-info-value-types old-info))))
	  (dolist (value (attribute-info-used-values att-info))
	    (pushnew value
		     (attribute-info-used-values old-info)
		     :test #'primitive-eql))))))

  att-info-table)

;; Each of model-1, model-2 is assumed to be either a data model or nil.
;;
;; If model-1 is nil, return nil.  Otherwise return a data model that
;; is a copy of model-1 with all information about a type/attribute
;; pair excluded if model-2 contains any information about that type/attribute
;; pair.  If every attribute for a type in model-1 also appears in model-2,
;; the entire type is excluded from model-1.  Primitive types are never
;; excluded from the result, even if they also appear in model-2.
;;
;; This will NOT destructively modify any of its argument models.  Also,
;; the model returned will not share structure with either of the argument
;; models.
;;
(defun data-model-difference (model-1 model-2)
  (if (null model-1)
    nil
    ;; ELSE
    (let ((result-model (data-model-copier model-1)))
      (when model-2
	(maphash #'(lambda (type-name type-info)
		     (declare (ignore type-name))
		     (data-model-difference-aux type-info result-model
						model-2))
		 result-model))

      result-model)))

;; Exclude information from type-info (an element of result-model) that
;; also appears in other-model, as described in the documentation for
;; data-model-difference.  This is a destructive operation.
;;
;; This returns the modified result-table.
;;
(defun data-model-difference-aux (type-info result-model other-model)
  (let* ((type-name (type-info-name type-info))
	 (other-def (gethash type-name other-model)))

    (setq type-info
	  (subtract-type-infos type-info other-def))

    ;; If we aren't left with any attributes, get rid of the whole type.
    ;;
    (when (empty-type-info type-info)
      (remhash type-name result-model)))

  result-model)

;; Exclude information from type-info-1 that also appears in type-info-2.
;; Either argument may be nil.  If either argument is a primitive type,
;; type-info-1 is returned unchanged.
;;
;; This is a potentially destructive operation (it may modify type-info-1).
;;
(defun subtract-type-infos (type-info-1 type-info-2)
  (when (and type-info-1
	     type-info-2
	     (not (type-info-primitive type-info-1))
	     (not (type-info-primitive type-info-2)))

    (let ((att-info-table (type-info-attribute-info type-info-1))
	  (other-att-info-table (type-info-attribute-info type-info-2)))
      
      ;; Get rid of attributes that are mentioned type-info-2
      ;;
      (when (and att-info-table other-att-info-table)
	(maphash #'(lambda (att-name att-info)
		     (declare (ignore att-info))
		     (when (gethash att-name other-att-info-table)
		       (remhash att-name att-info-table)))
		 att-info-table))))

  type-info-1)

;; This is a non-destructive version of subtract-type-infos.
;;
(defun type-info-difference (type-info-1 type-info-2)
  (when type-info-1
    (subtract-type-infos (type-info-copier type-info-1) type-info-2)))

;; See the release notes for syntax and semantics of excise-data-model.
;;
(defmacro excise-data-model (&body body)
  `(excise-data-model-aux ',body))

(defun excise-data-model-aux (body)
  (excise-type-info-for-data-models
   (process-model-spec-args-common body 'type nil))

  t)

(defmacro print-data-model (&body body)
  `(print-data-model-aux ',body))

(defun print-data-model-aux (body)
  (multiple-value-bind (include-parsed-triples exclude-parsed-triples
					       infer-globals)
      (process-print-data-model-args body)

    (let ((include-types (expand-to-simple-triples include-parsed-triples))
	  (exclude-types (expand-to-simple-triples exclude-parsed-triples))
	  (last-segment-printed '(not-a-segment-name))
	  (printed-begin-segment nil)
	  (global-types nil)
	  (items-to-print nil)
	  (global-items-to-print nil))

      (setq include-types
	    (set-difference include-types exclude-types :test #'equal))
      (multiple-value-setq (global-types include-types)
	(split-out-globals include-types infer-globals))
      (setq include-types
	    (collapse-inferred-and-declared-to-all include-types))

      (dolist (include-spec include-types)
	(let ((type-info
	       (exclude-parts-from-type-info include-spec exclude-types)))
	  (when type-info
	    (push (list (caddr include-spec) type-info)
		  items-to-print))))

      (dolist (include-spec global-types)
	(let ((type-info
	       (exclude-parts-from-type-info include-spec exclude-types)))
	  (when type-info
	    (push type-info global-items-to-print))))

      (setq items-to-print
	    (sort-data-type-items-for-printing items-to-print :key #'cadr))
      (setq items-to-print
	    ;; Sort non-global items by segment
	    (sort items-to-print
		#'string-lessp
		:key #'(lambda (x)
			 (symbol-name (car x)))))

      (mapc #'(lambda (t-info)
		(setq t-info (type-info-copier t-info))
		(setf (type-info-global t-info) t)
		(print-data-type t-info '*global*))
	    (sort-data-type-items-for-printing
	     (merge-items-for-same-type global-items-to-print)))

      (dolist (item items-to-print)
	(let ((item-segment (car item))
	      (item-type-info (cadr item)))
	  (when (not (eql last-segment-printed item-segment))
	    (setq last-segment-printed item-segment)
	    (setq printed-begin-segment t)
	    (format t "~%(begin-segment ~(~S~))~%" item-segment))
	  (print-data-type item-type-info item-segment)))
    
      (when printed-begin-segment
	(format t "~%(end-segment)~%"))))

  t)
      
(defun process-print-data-model-args (args)
  (let ((error-detected nil))
    (multiple-value-bind (new-args infer-global-values)
	(remove-keyword-and-values ':infer-globals args)

      (when (member-if-not #'(lambda (x)
			       (member x '(yes no)))
			   infer-global-values)
	(setq error-detected t)
	(taql-warn2 "Print-data-model:  Expected 'yes' or 'no' as ~
                     value of :infer-globals."))

      (when (and (member 'yes infer-global-values)
		 (member 'no infer-global-values))
	(setq error-detected t)
	(taql-warn2 "Print-data-model:  Both 'yes' and 'no' given as ~
                     values of :infer-globals."))

      (if error-detected
	(values nil nil nil)
	;; ELSE
	(multiple-value-bind (include-parsed-triples exclude-parsed-triples)
	    (process-model-spec-args-common new-args 'type)
	  (values include-parsed-triples
		  exclude-parsed-triples
		  (or (null infer-global-values)
		      (eql (car infer-global-values) 'yes))))))))

;; This would fail on lists where keyword appears as the value of another
;; keyword.  We also assume that keyword, if it appears, is always followed
;; by exactly one value.  If keyword appears at the end of list, it will
;; be treated as having the value nil.
;;
(defun remove-keyword-and-values (keyword list)
  (if (member keyword list)
    (prog ((modified-list nil)
	   (values nil))
      loop
        (when (null list)
	  (return (values (reverse modified-list) values)))

	(if (eql (car list) keyword)
	  (progn
	    (push (cadr list) values)
	    (setq list (cddr list)))
	  ;; ELSE
	  (progn
	    (push (car list) modified-list)
	    (setq list (cdr list))))

	(go loop))
    ;; ELSE
    (values list nil)))

;; Sort the type list so that primitive types come before non-primitive
;; ones, and non-primitive types come before operators.  Types are
;; alphabetized within each of these groups.
;;
;; The optional key-func accesses the type-info structure of each element
;; of the items list.  Sorting is done based on this type-info structures.
;;
(defun sort-data-type-items-for-printing (items &key (key #'identity))
  (setq items
	(sort items
	      #'string-lessp
	      :key #'(lambda (x)
		       (symbol-name (type-info-name (funcall key x))))))
  (setq items
	(stable-sort items
		     #'(lambda (x y)
			 (and x (not y)))
		     :key #'(lambda (x)
			      (type-info-primitive (funcall key x)))))
  (setq items
	(stable-sort items
		     #'(lambda (x y)
			 (and (not (operator-type-name-p x))
			      (operator-type-name-p y)))
		     :key #'(lambda (x)
			      (type-info-name (funcall key x)))))
  items)

(defun expand-to-simple-triples (parsed-triples)
  (let ((result nil))
    (dolist (parsed-triple parsed-triples)
      (setq result
	    (union (expand-to-simple-triples-aux parsed-triple) result
		   :test #'equal)))
    result))

;; The form of a parsed-triple is described in parse-model-spec-triple.
;; This expands such a parsed triple to a list of triples of the form
;;    ({declared | inferred} TYPE-NAME SEGMENT-NAME)
;; The expanded triple never contains :all -- they are expanded.  Also,
;; the result list never contains a triple specifying type information
;; that does not exist.  It will, however, return (declared TYPE SEG)
;; when TYPE has a global definition that appears in SEG.
;;
(defun expand-to-simple-triples-aux (parsed-triple)
  (let ((item-part (car parsed-triple))
	(model-specs (cdr parsed-triple))
	(result-triples nil))

    (dolist (model-spec model-specs)
      (let ((model-types
	     (type-names-in-data-model (car model-spec) (cadr model-spec))))
	(dolist (type-name (if (eql item-part :all)
			     model-types
			     ;; ELSE
			     (intersection item-part model-types)))
	  (push (list (car model-spec) type-name (cadr model-spec))
		result-triples))))

    result-triples))

;; This partitions simple-tripl-list into two lists, one of triples that
;; should be printed as globals, and the other of triples that should be
;; printed as locals.
;;
;; A triple ({inferred | declared} TYPE-NAME SEGMENT-NAME) should be treated
;; as global if any of the following conditions hold:
;;
;;  1. It is a declared triple and TYPE-NAME has a global declaration in
;;     SEGMENT-NAME.
;;  2. It is an inferred triple and TYPE-NAME is not declared locally in
;;     SEGMENT-NAME and TYPE-NAME has a global declaration (which might
;;     come from a segment other than SEGMENT-NAME).
;;  3. It is an inferred triple and TYPE-NAME is not declared locally in
;;     SEGMENT-NAME and there is some *different* segment that has inferred
;;     information for TYPE-NAME that also does not have a local declaration
;;     for type-name.
;;
;; Rule 3 is a heuristic to try to get inferred type information printed out
;; right even before any declarations are present to tell it what is and is
;; not local.  Essentially, it is an assumption that any type that appears
;; in more than one segment without being declared local in those segments is
;; global.
;;
;; If the infer-globals argument is nil, only Rule 1 will be used to identify
;; globals.
;;
(defun split-out-globals (simple-triple-list infer-globals)
  (let ((global-triples nil)
	(other-triples nil))

    (dolist (triple simple-triple-list)
      (let ((model-type (car triple))
	    (type-name (cadr triple))
	    (segment-name (caddr triple)))
	(if (or (and (eql model-type 'declared)
		     (declared-global-p type-name segment-name))
		(and (eql model-type 'inferred)
		     infer-globals
		     (not (declared-local-p type-name segment-name))
		     (or (get-type-info type-name 'declared '*global*)
			 (has-inferred-maybe-global-info-elsewhere
			  type-name segment-name))))
	  (push triple global-triples)
	  ;; ELSE
	  (push triple other-triples))))

    (values global-triples other-triples)))

;; Return non-nil iff there exists a segment other than segment-name that
;; has inferred type info for type-name but no local declaration for type-name.
;;
(defun has-inferred-maybe-global-info-elsewhere (type-name segment-name)
  (dolist (segment-name-2 (all-non-global-segment-names))
    (when (and (not (eql segment-name-2 segment-name))
	       (get-type-info type-name 'inferred segment-name-2)
	       (not (declared-local-p type-name segment-name-2)))
      (return-from has-inferred-maybe-global-info-elsewhere t)))
  nil)

;; If both (declared T S) and (inferred T S) appear in the argument list,
;; then they will be replaced by (all T S) in the result list.
;;
(defun collapse-inferred-and-declared-to-all (simple-triple-list)
  (let ((result nil))
    (prog ((remaining-items (copy-list simple-triple-list)))
      loop
        (when (null remaining-items)
	  (return))

	(let ((opposite-triple
	       (cons (if (eql (caar remaining-items) 'declared)
		       'inferred
		       ;; ELSE
		       'declared)
		     (cdar remaining-items))))
	  (if (member opposite-triple (cdr remaining-items) :test #'equal)
	    (progn
	      (push (cons 'all (cdar remaining-items)) result)
	      (setq remaining-items
		    (delete opposite-triple remaining-items :test #'equal)))
	    ;; ELSE
	    (push (car remaining-items) result)))

	(setq remaining-items (cdr remaining-items))
	(go loop))
    result))

;; This gets the type-info specified by include-triple, and excludes from it
;; attributes that are also defined in matching items in exclude-triples.
;; An exclude-triple matches if either is is for the same type and segment,
;; or it is a global declaration for the same type.
;;
;; Return the resulting type-info structure if the exclusions haven't made it
;; empty.  If the exclusions have made it empty, return nil.  (If the type
;; info was initially empty, and there was nothing to exclude from it, we
;; return the empty type info, not nil.)
;;
(defun exclude-parts-from-type-info (include-triple exclude-triples)
  (let* ((model-type (car include-triple))
	 (type-name (cadr include-triple))
	 (segment (caddr include-triple))
	 (found-item-to-exclude nil)
	 (type-info
	  (case model-type
	    ((inferred declared)
	     (get-type-info type-name model-type segment))
	    (all
	     (type-info-union
	      (get-type-info type-name 'inferred segment)
	      (get-type-info type-name 'declared segment))))))

    (dolist (matching-item
	     (collect #'(lambda (triple)
			  (and (eql (cadr triple) type-name)
			       (or (eql (caddr triple) segment)
				   (and (eql (car triple) 'declared)
					(type-info-global
					 (get-type-info-2 triple))))))
		      exclude-triples))
      (setq found-item-to-exclude t)
      (setq type-info
	    (type-info-difference type-info (get-type-info-2 matching-item))))

    (if (and found-item-to-exclude
	     (empty-type-info type-info))
      nil
      ;; ELSE
      type-info)))

(defun merge-items-for-same-type (type-info-list)
  (let ((data-model (new-data-model)))
    (dolist (type-info type-info-list)
      (merge-type-into-data-model type-info data-model))
    (hash-table-to-list data-model)))

(defun print-data-type (type-info home-segment)
  (if (type-info-primitive type-info)
    (print-primitive-data-type type-info)
    ;; ELSE
    (print-structured-data-type type-info home-segment))
  t)

;; Assumes that type-info is a primitive type.
;;
(defun print-primitive-data-type (type-info)
  (let ((primitive (type-info-primitive type-info)))
    (if (listp primitive)
      (format t "~%(defenumeration ~(~S~)~%  ~{~( ~S~)~})~%"
	      (type-info-name type-info)
	      primitive)
      ;; ELSE
      (format t "~%(defprimitive ~(~S ~S~))~%"
	      (type-info-name type-info)
	      primitive))))

;; The home-segment argument is only used to determine the value to print
;; for :all-info-declared.  So if print-all-info-declared is nil, the
;; home-segment needn't have any particular value.
;;
(defun print-structured-data-type (type-info
				   home-segment
				   &key (decl-command nil command-supplied)
				        (print-all-info-declared t))

  (let* ((att-info (type-info-attribute-info type-info))
	 (type-name (type-info-name type-info))
	 (operator-type-p (operator-type-name-p type-name))
	 (type-name-to-print
	  (if operator-type-p
	    (operator-type-operator-name type-name)
	    ;; ELSE
	    type-name)))

    (format t "~%(~(~S ~S~)~%  :global ~(~S~)"
	    (cond (command-supplied
		   decl-command)
		  ((and (member type-name '(goal problem-space state operator))
			(type-info-global type-info))
		   'defobject-merge)
		  (operator-type-p
		   'defoperator)
		  (t
		   'defobject))
	    type-name-to-print
	    (if (type-info-global type-info)
	      'yes
	      ;; ELSE
	      'no))

    (when print-all-info-declared
      (format t "~%  :all-info-declared ~(~S~)"
	      (if (member (list home-segment type-name)
			  *all-info-not-declared-types* :test #'equal)
		'no
		;; ELSE
		'yes)))

    (when att-info
      (mapc #'(lambda (info)
		(print-attribute-info info operator-type-p))
	    (sort (hash-table-to-list att-info)
		  #'string-lessp
		  :key #'(lambda (x)
			   (symbol-name (attribute-info-name x))))))
    (format t ")~%")))

(defun print-attribute-info (att-info operator-type-p)
  (let* ((value-types
	  (setf (attribute-info-value-types att-info)
		(sort (attribute-info-value-types att-info)
		      #'primitive-lessp)))
	 (used-values
	  (setf (attribute-info-used-values att-info)
		(sort (attribute-info-used-values att-info)
		      #'primitive-lessp)))
	 (optional (and operator-type-p
			(not (attribute-info-required att-info))))
	 (need-list-form (or value-types optional)))

    (terpri) (write-string "  ")
    (when need-list-form
      (write-char #\())
    (format t "~(~S~)" (attribute-info-name att-info))
    (when optional
      (format t " :optional"))
    (when value-types
      (format t " :type ~(~S~)"
	      (if (cdr value-types)
		value-types
		;; ELSE
		(car value-types))))
    (when used-values
      (format t "~%     ;; Values used: ~{~( ~S~)~}~%"
	      used-values))
    (when need-list-form
      (when used-values
	(write-string "     "))
      (write-char #\)))))

(defun hash-table-to-list (hash-table)
  (let ((result nil))
    (when hash-table
      (maphash #'(lambda (key entry)
		   (declare (ignore key))
		   (push entry result))
	       hash-table))
    result))

(defun type-names-in-data-model (model-type segment-name)
  (let ((result nil))
    (maphash #'(lambda (type-name type-info)
		   (declare (ignore type-info))
		   (push type-name result))
	     (get-data-model model-type segment-name))
    result))

(defun save-current-object-components ()
  (push
    (list *current-class*
	  *current-identifiers*
	  *current-attribute*)
    *current-object-components-stack*)
  t)

(defun restore-current-object-components ()
  (when (null *current-object-components-stack*)
    (error "INTERNAL TAQL COMPILER ERROR.  ~
            *current-object-components-stack* is null."))
  (let ((values (pop *current-object-components-stack*)))
    (setq *current-class* (car values))
    (setq *current-identifiers* (cadr values))
    (setq *current-attribute* (caddr values)))
  t)

;; New-class takes as its argument a list of positive symbols that appeared
;; in the class position of a condition or action.  By positive symbol,
;; I mean something that didn't follow one of these relations: <>, <, >,
;; >=, <=, <=>.
;;
(defun new-class (classes)
  (cond ((or (null classes)   ; no elements
	     (cddr classes))  ; more than one element
	 (setq *current-class* '*unknown*))
	(t
	 (let ((class (car classes)))
	   (if (or (not (symbolp class))
		   (variable-p class))
	     (setq *current-class* '*unknown*)
	     ;; ELSE
	     (setq *current-class* class)))))

  (when (eql *current-class* 'init-stuff*)
    (setq *last-aug*-id* nil))

  (do-class-traces *current-class*)

  (when (not (member *current-class* '(*unknown* operator**unknown*)))

    (let ((type-info (get-declared-type-info *current-class*)))

      (cond (type-info
	     (when (type-info-primitive type-info)
	       (taql-warn-soft
		"Type ~S was used as a structured type, but was declared ~
                 primitive."
		*current-class*)))

	    ((and *all-types-declared*
		  (not (operator-type-name-p *current-class*)))
	     (taql-warn-soft
	      "A structured object of type ~S was used, but that type ~
               was not declared."
	      *current-class*))

	    ((and *all-operators-declared*
		  (operator-type-name-p *current-class*))
	     (taql-warn-soft
	      "Operator ~S was used, but was not declared."
	      (operator-type-operator-name *current-class*))))))

  (add-type-to-model (get-data-model 'inferred *current-segment-name*)
		     *current-class* nil)

  t)

;; New-identifier takes as its argument a list of positive symbols
;; that appeared in the identifier position of a condition or action.
;; By positive symbol, I mean something that didn't follow one of
;; these relations: <>, <, >, >=, <=, <=>.
;;
;; It is an error if the argument is not nil or a list of variables.
;;
(defun new-identifier (ids)
  (let ((non-variable (member-if-not #'variable-p ids)))

    (if non-variable
      (progn
	(taql-warn "Only variables can be used as identifiers, but found ~S"
		   (car non-variable))
	(setq *current-identifiers* nil))
      ;; ELSE
      (progn
	(setq *current-identifiers* ids)
	(mapc #'(lambda (id)
		  (record-variable-used-as-id *current-class* id))
	      ids)))))

;; New-attribute takes as its argument a list of positive symbols that appeared
;; in the attribute position of a condition or action.  By positive symbol,
;; I mean something that didn't follow one of these relations: <>, <, >,
;; >=, <=, <=>.
;;
(defun new-attribute (attributes &optional (save-operator-augmentations t))
  (cond ((or (null attributes)   ; no elements
	     (cddr attributes))  ; more than one element
	 (setq *current-attribute* '*unknown*))
	(t
	 (let ((attribute (car attributes)))
	   (if (or (not (symbolp attribute))
		   (variable-p attribute))
	     (setq *current-attribute* '*unknown*)
	     ;; ELSE
	     (setq *current-attribute* attribute)))))

  (when (and save-operator-augmentations
	     (or (eql *current-class* 'operator)
		 (operator-type-name-p *current-class*)))
    (dolist (id *current-identifiers*)
      (save-operator-att id *current-attribute*))
    (return-from new-attribute))

  (dolist (trace-class (cons *current-class* *extra-trace-classes*))
    (do-attribute-traces trace-class *current-attribute*))

  (when (not (eql *current-attribute* '*unknown*))

    (let* ((type-info (get-declared-type-info *current-class*))
	   (att-info-table
	    (when type-info (type-info-attribute-info type-info))))

      (when (and type-info
		 (not (member (list (if (type-info-global type-info)
				      '*global*
				      ;; ELSE
				      *current-segment-name*)
				    *current-class*)
			      *all-info-not-declared-types*
			      :test #'equal))
		 (or (not att-info-table)
		     (not (gethash *current-attribute* att-info-table))))
	(taql-warn-soft
	 "Attribute ~S was used for type ~S, but was not declared."
	 *current-attribute* *current-class*))))

  (add-attribute-to-model (get-data-model 'inferred *current-segment-name*)
			  *current-class*
			  *current-attribute*)

  t)

;; New-value takes as its argument a list of positive symbols that appeared
;; in the class position of a condition or action.  By positive symbol,
;; I mean something that didn't follow one of these relations: <>, <, >,
;; >=, <=, <=>.
;;
(defun new-value (values &optional (save-operator-augmentations t))
  (mapc #'(lambda (value)
	    (new-single-value value save-operator-augmentations))
	values))

;; When we get an argument that is a cons, we assume that it is a function
;; call.  We don't do any further checking/interence on a function call,
;; since we have no idea what kind of value it might return.  Can function
;; calls return structured object?  If not, then maybe we could treat a
;; function call more like a primitive value.
;;
;; When we get an aug* object, it comes from a buffered edit in TAQL.  We
;; want the value to be recorded as if the buffer wasn't there and we were
;; editing the object directly, and we put a hack in here to do that.  This
;; is important because the user doesn't see the buffer in their TCs -- it
;; looks like they ARE editing the object directly.  *last-aug*-class*
;; stores the class of the object the aug* is buffering edits for.  It
;; comes from either an add* object or an init-stuff* object.  We assume that
;; an aug* object is always associated with the nearest preceding add* or
;; init-stuff*.  The way the compiler is currently written, this will be the
;; case.
;;
(defun new-single-value (value &optional (save-operator-augmentations t))
  (when (and save-operator-augmentations
	     (or (eql *current-class* 'operator)
		 (operator-type-name-p *current-class*)))
    (when (and (eql *current-attribute* 'name)
	       (primitive-p value))
      ;; This is an attempt to be able to compile things like
      ;; (operator ^name cbl-bpi ^length.value 10).  If the length attribute
      ;; is declared in (defoperator cbl-bpi ...), then at least this lets
      ;; us compile it if the name attribute come before the length attribute.
      ;;
      (setq *current-class* (makesym 'operator* value)))
    (dolist (id *current-identifiers*)
      (save-operator-att-val id *current-attribute* value))
    (return-from new-single-value))

  (when (or (and (eql *current-class* 'add*)
		 (eql *current-attribute* 'class))
	    (and (eql *current-class* 'init-stuff*)
		 (eql *current-attribute* 'type)))
    (setq *last-aug*-class* value))

  (when (and (eql *current-class* 'add*)
	     (eql *current-attribute* 'id))
    (setq *last-aug*-id* value))

  (when (and (eql *current-class* 'aug*)
	     (not (eql *current-attribute* 'tc-id*)))
    (save-current-object-components)
    (new-class (list *last-aug*-class*))
    (when *last-aug*-id*
      (new-identifier (list *last-aug*-id*)))
    (new-attribute (list *current-attribute*))
    (new-value (list value))
    (restore-current-object-components))

  (dolist (trace-class (cons *current-class* *extra-trace-classes*))
    (do-simple-value-traces trace-class *current-attribute* value))

  (cond ((consp value)           ; we assume this is a function call
	 nil)

	((primitive-p value)
	 (dolist (trace-class (cons *current-class* *extra-trace-classes*))
	   (do-primitive-value-traces trace-class *current-attribute* value))
	 (check-primitive-value *current-class* *current-attribute* value
				'declared)
	 (check-primitive-value *current-class* *current-attribute* value
				'inferred))

	((variable-p value)
	 (record-variable-used-as-value
	  *current-class* *current-attribute* value))

	(t
	 (taql-warn
	  "Expected a variable, symbol, string, number or function call, but ~
           found ~S used as a value."
	  value))))

;; This routine assumes its value argument satisfies primitive-p.  It should
;; be called for every class/att/val that appears in a production when val
;; satisfies primitive-p.
;;
(defun check-primitive-value (class att value model-type)

  (let* ((data-model (if (eql model-type 'declared)
		       (get-declared-model class)
		       ;; ELSE
		       (get-data-model model-type *current-segment-name*)))
	 (value-types (and data-model
			   (get-value-types data-model class att
					    (eql model-type 'declared))))
	 (primitives
	  (when value-types
	    (get-primitives value-types))))

    (if (notany #'(lambda (prim-def)
		    (of-primitive-def-p prim-def value))
		primitives)
      (if (eql model-type 'inferred)
	(add-value-type data-model class att 'primitive)
	;; ELSE
	(if value-types
	  (taql-warn-soft
	   "Value ~S for (~S ^~S) is not of previously declared type ~S"
	   value class att value-types))))

    (when (eql model-type 'inferred)
      (add-used-value data-model class att value))))

;; This routine should be called for every class/att/id that appears in a
;; production.
;;
(defun check-structured-value (class att used-value-type-name model-type)

  (let* ((data-model (if (eql model-type 'declared)
		       (get-declared-model class)
		       ;; ELSE
		       (get-data-model model-type *current-segment-name*)))
	 (value-types (and data-model
			   (get-value-types data-model class att
					    (eql model-type 'declared))))
	 (non-primitives (when value-types
			   (get-non-primitives value-types))))

    (if (not (member used-value-type-name non-primitives))
      (if (eql model-type 'inferred)
	(add-value-type data-model class att used-value-type-name)
	;; ELSE
	(if value-types
	  (taql-warn-soft
	   "The value type for (~S ^~S) was previously declared to be ~S, ~
            but a structured object of type ~S appeared here as a value."
	   class att value-types used-value-type-name))))))

;;;----------------------------------------------------------------------
;;; Structured value type checking
;;;----------------------------------------------------------------------

;;; We store a list of class/attribute/value triples used in a production
;;; (even if the class or attribute is *unknown*) and value is a variable.
;;; After we have parsed the entire production, we use this list and
;;; the accumulated identifier->classes mapping to do value-type checking
;;; for structured attribute values.

;; Record that a defining instance of the identifier variable id was used in
;; an object of type class.  Do this even if class is *unknown*.
;;
;; We assume that id is a variable.  If id was previously used in
;; a defining instance with some different class, then the new classes is
;; added to a list of classes used for that variable.
;;
(defun record-variable-used-as-id (class id)
  (pushnew (if (operator-type-name-p class)
	     'operator
	     ;; ELSE
	     class)
	   (gethash id *id-class-mapping*)))

;; Store a class/attribute/value-id triple, if we haven't encountered
;; that triple in the production yet.  We also store the current value of
;; *extra-trace-classes* at the end of the entry, so that we'll trace the
;; right extra classes when the time comes.
;;
(defun record-variable-used-as-value (class attribute id)
  (pushnew `(,class ,attribute ,id ,@*extra-trace-classes*)
	   *class-attribute-value-set*
	   :test #'equal))

;; This should be called after a production has been completely parsed.
;; It checks the value types used for attributes that have structured
;; values in the production against previously declared or inferred
;; information.  Also, the new value type information is stored in the
;; inferred model of the current segment if that information is not
;; already there.
;;
;; If the production used the same id variable with objects of different
;; classes, then value types will be checked separately against each of the
;; classes used.  This will typically result in issuing soft warnings for
;; at least one of the classes used.
;;
;; We only check against the global declared model for entries for which
;; class, attribute and value type are known.  We check/enter all items into
;; the local inferred model, though.
;; 
;; An entry in *class-attribute-value-set* may have a non-nil cdddr.  If so,
;; the extra elements are additional class names to trace the entry under.
;; This is useful for operators, which we want to trace under both
;; "operator" and "operator*op-name".
;;
(defun check-structured-value-types ()
  (dolist (triple *class-attribute-value-set*)
    (let* ((class (car triple))
	   (attribute (cadr triple))
	   (value-id (caddr triple))
	   (id-classes (gethash value-id *id-class-mapping*)))

      (dolist (id-class id-classes)
	(when (and (not (eql class '*unknown*))
		   (not (eql attribute '*unknown*))
		   (not (eql id-class '*unknown*)))
	  (check-structured-value class attribute id-class 'declared))
	(dolist (trace-class (cons class (cdddr triple)))
	  (do-structured-value-traces trace-class attribute value-id id-class))
	(check-structured-value class attribute id-class 'inferred)))))

;;;----------------------------------------------------------------------
;;; Trace-load code
;;;----------------------------------------------------------------------

;;; This section defines the commands trace-load and untrace-load.
;;;
;;;   (trace-load [ CLASS-TRACE [ ATT-TRACE [ VALUE-TRACE ] ] ])
;;;   (untrace-load)
;;;
;;; CLASS-TRACE and ATT-TRACE must be either symbolic class/attribute names,
;;; the symbol *unknown*, or one of the symbols * or ?.
;;;
;;; VALUE-TRACE must be a primitive value, one of the symbols * or ?, or
;;; a list of the form (primitive TYPE-NAME*), (object TYPE-NAME*), or
;;; (function FUNCTION-NAME*).
;;;
;;; ATT-TRACE and VALUE-TRACE default to * when not given.
;;;
;;; The effect of trace-load is to print a trace message when the items
;;; matching the trace specification are encountered in a production
;;; when it is loaded.  This can be very convenient when using
;;; print-data-model to print the data model inferred from a large production
;;; set.  Trace-load can be used to locate the TCs that selected classes,
;;; attributes, and values appeared in.
;;;
;;; The symbol *unknown* stands for classes, attributes, and value types
;;; that TAQL could not determine uniquely.  For example, TAQL treats
;;; variable classes and attributes as *unknown*.
;;;
;;; The symbols * and ? are wildcards.  The difference is that for *, the
;;; associated component need not exist to match, while for ? it does.
;;; For example, (trace-load state color *) will print a trace message
;;; every time TAQL encounters the color attribute used in a state object,
;;; even if no value was given.  It will only print the class and attribute
;;; name in the trace message.  On the other hand, (trace-load state color ?)
;;; will only print a trace message if some value appear for the color
;;; attribute on a state.  It will print the class, attribute, and value.
;;;
;;; There are three special value trace specifiers:
;;;
;;;    (primitive TYPE-NAME*) -- This matches primitive values that
;;;        are of one of the specified primitive types.  If no type
;;;        name are listed, then all primitive values match.
;;;
;;;    (object TYPE-NAME*) -- This matches structured values that
;;;        are of one of the specified types.  If no type names are
;;;        listed, then all structured values match.
;;;
;;;    (primitive TYPE-NAME*) -- This matches values that are function
;;;        calls (e.g. compute) to functions named in the list.  If no
;;;        function names are given, then all function calls match.
;;;
;;; Trace-load with no arguments prints out the currently active trace
;;; specifications.
;;;
;;; Untrace-load removes all current trace specifications.

(defmacro untrace-load (&body body)
  `(untrace-load-aux ',body))

(defun untrace-load-aux (body)
  (if body
    (taql-warn2 "Untrace-load does not take any arguments:  ~S"
		(cons 'untrace-load body))
    ;; ELSE
    (setq *trace-load-list* nil))

  t)

(defmacro trace-load (&body body)
  `(trace-load-aux ',body))

(defun trace-load-aux (body)
  (cond ((null body)
	 (if (null *trace-load-list*)
	   (format t "~%Trace-load is not currently tracing anything.")
	   ;; ELSE
	   (format t "~%Trace-load is currently tracing these items:~
                      ~:{~%     ~S ~S ~S~}"
		   *trace-load-list*)))

	((> (length body) 3)
	 (taql-warn2 "Trace-load takes at most three arguments."))

	(t
	 (let ((class (car body))
	       (att (if (cdr body)
			(cadr body)
			;; ELSE
			'*))
	       (value (if (cddr body)
			(caddr body)
			;; ELSE
			'*))
	       (error-detected nil))

	   (when (not (symbolp class))
	     (setq error-detected t)
	     (taql-warn2 "Trace-load:  The class specifier must be a ~
                          symbolic class name, *, ?, or *unknown*.  Found ~
                          ~S instead."
			 class))

	   (when (not (symbolp att))
	     (setq error-detected t)
	     (taql-warn2 "Trace-load:  The attribute specifier must be a ~
                          symbolic attribute name, *, ?, or *unknown*.  ~
                          Found ~S instead."
			 class))

	   (when (and (eql class '*)
		      (eql att '*)
		      (eql value '*))
	     (setq error-detected t)
	     (taql-warn2 "Trace-load:  At least one of the class, attribute, ~
                          and value specifiers must be something other than ~
                          *"))

	   (cond ((primitive-p value)
		  ;; Ok, do nothing.
		  )
		 ((and (consp value)
		       (member (car value) '(object function primitive)))
		  (case (car value)
		    ((object function)
		     (cond
		      ((find-if-not #'symbolp (cdr value))
		       (setq error-detected t)
		       (taql-warn2 "Trace-load:  All items in a ~(~S~) list ~
                                    must be symbols."
				   (car value)))
		      ((and (eql (car value) 'object)
			    (find-if #'operator-type-name-p
				     (cdr value)))
		       (setq error-detected t)
		       (taql-warn2 "Trace-load:  No item in an object list ~
                                    can begin with 'operator*'."))))
		    (primitive
		     (let ((non-prim
			    (find-if-not #'get-primitive (cdr value))))
		       (when non-prim
			 (setq error-detected t)
			 (taql-warn2 "Trace-load:  ~S is not declared as the ~
                                      name of a primitive type."
				     non-prim))))))
		 (t
		  (setq error-detected t)
		  (taql-warn2 "Trace-load:  The value specifier must be a ~
                               primitive value, *, ?, (primitive TYPE-NAME*), ~
                               (object TYPE-NAME*), or (function ~
                               FUNCTION-NAME*).  Found ~S instead."
			      value)))

	   (when (not error-detected)
	     (pushnew (list class att value)
		      *trace-load-list*
		      :test #'equal)))))
  t)

(defun do-class-traces (class)
  (when (member-if #'(lambda (trace-spec)
		       (and (eql (cadr trace-spec) '*)
			    (eql (caddr trace-spec) '*)
			    (match-class class trace-spec)))
		   *trace-load-list*)
    (trace-load-message "Used class ~S" class)))

(defun do-attribute-traces (class att)
  (when (member-if #'(lambda (trace-spec)
		       (and (eql (caddr trace-spec) '*)
			    (not (eql (cadr trace-spec) '*))
			    (match-class class trace-spec)
			    (match-attribute att trace-spec)))
		   *trace-load-list*)
    (trace-load-message "Used ~S ^~S" class att)))

(defun do-simple-value-traces (class att value)
  (when (member-if #'(lambda (trace-spec)
		       (and (not (eql (caddr trace-spec) '*))
			    (match-class class trace-spec)
			    (match-attribute att trace-spec)
			    (match-value value trace-spec)))
		   *trace-load-list*)
    (trace-load-message "Used ~S ^~S ~S" class att value)))

(defun do-primitive-value-traces (class att value)
  (dolist (trace-spec *trace-load-list*)
    (when (and (match-class class trace-spec)
	       (match-attribute att trace-spec))
      (let ((val-part (caddr trace-spec)))
	(when (and (consp val-part)
		   (eql (car val-part) 'primitive))
	  (dolist (prim-type (or (cdr val-part)
				 '(primitive)))
	    (when (of-primitive-type-p prim-type value)
	      (trace-load-message "Used ~S ^~S ~S (value type ~S)"
				  class att value prim-type))))))))

(defun do-structured-value-traces (class att value-id value-type)
  (dolist (trace-spec *trace-load-list*)
    (when (and (match-class class trace-spec)
	       (match-attribute att trace-spec))
      (let ((val-part (caddr trace-spec)))
	(when (and (consp val-part)
		   (eql (car val-part) 'object))
	  (if (cdr val-part)
	    (when (member value-type (cdr val-part))
	      (trace-load-message "Used ~S ^~S ~S (value type ~S)"
				  class att value-id value-type))
	    ;; ELSE
	    (trace-load-message "Used ~S ^~S ~S (value type ~S)"
				class att value-id value-type)))))))

(defun match-class (arg trace-spec)
  (let ((part (car trace-spec)))
    (or (eql part arg)
	(eql part '*)
	(eql part '?))))

(defun match-attribute (arg trace-spec)
  (let ((part (cadr trace-spec)))
    (or (eql part arg)
	(eql part '*)
	(eql part '?))))

(defun match-value (arg trace-spec)
  (let ((part (caddr trace-spec)))
    (or (eql part arg)
	(eql part '*)
	(eql part '?)
	(and (consp arg)
	     (consp part)
	     (eql (car part) 'function)
	     (or (null (cdr part))
		 (member (car arg) (cdr part)))))))

(defun trace-load-message (format-string &rest args)
  (fresh-line)
  (format t "Trace-load: ~S.." *current-taql-name*)
  (eval `(format t ',format-string
	  ,@(mapcar #'(lambda (arg) (list 'quote arg)) args)))
  (terpri))

;;;----------------------------------------------------------------------
;;; End trace-load code
;;;----------------------------------------------------------------------
	 
(defun save-operator-att (id attribute)
  (pushnew (list attribute) (gethash id *saved-operator-augs*) :key #'car))
	 
(defun save-operator-att-val (id attribute value)
  (save-operator-att id attribute)
  (let ((item (assoc attribute (gethash id *saved-operator-augs*))))
    (pushnew value (cdr item) :test #'equal)))

(defun check-saved-operators ()
  (maphash #'check-saved-operator *saved-operator-augs*))

(defun check-saved-operator (id attribute-alist)
  (let* ((name-att (assoc 'name attribute-alist))
	 (op-name (if (or (null (cdr name-att))
			  (cddr name-att)
			  (not (symbolp (cadr name-att)))
			  (variable-p (cadr name-att)))
		    '*unknown*
		    ;; ELSE
		    (cadr name-att)))
	 (op-type (makesym 'operator* op-name))
	 (op-type-info (get-declared-type-info op-type))
	 (check-on-specific nil))

    (when (and (eql id *proposed-operator-id*)
	       (not (eql op-name '*unknown*)))
      (let ((required-atts (required-operator-atts op-type-info)))
	(dolist (att required-atts)
	  (when (not (assoc att attribute-alist))
	    (taql-warn-soft
	     "Operator ~S was proposed without its required ~S argument."
	     op-name att)))))

    (new-class (list 'operator))
    (new-identifier (list id))

    (let ((*extra-trace-classes*
	   (cons op-type *extra-trace-classes*)))
      (dolist (att-plus-vals attribute-alist)
	(let* ((att-name (car att-plus-vals))
	       (declared-for-generic
		(attribute-exists (get-declared-type-info 'operator) att-name))
	       (declared-for-specific
		(attribute-exists op-type-info att-name)))
	  
	  (if (and declared-for-generic
		   (not declared-for-specific))
	    (progn
	      (new-attribute (list att-name) nil)
	      (when (cdr att-plus-vals)
		(new-value (cdr att-plus-vals) nil)))
	    ;; ELSE
	    (push att-plus-vals check-on-specific)))))

    ;; Don't register operator**unknown* as a new class if the only
    ;; attributes it was used with were declared under the general
    ;; operator type.  This will prevent print-data-model from listing
    ;; (defoperator *unknown* ...) when only general operator attributes
    ;; are used for operators whose name is not known.  Do give the class
    ;; trace in any case, though.
    ;;
    (if (or check-on-specific
	    (not (eql op-name '*unknown*)))
      (new-class (list op-type))
      ;; ELSE just do the class trace
      (do-class-traces op-type))

    (let ((*extra-trace-classes*
	   (cons 'operator *extra-trace-classes*)))
      (dolist (att-plus-vals check-on-specific)
	(new-attribute (list (car att-plus-vals)) nil)
	(when (cdr att-plus-vals)
	  (new-value (cdr att-plus-vals) nil))))))

(defun attribute-exists (type-info att-name)
  (when type-info
    (let ((att-info-table (type-info-attribute-info type-info)))
      (when att-info-table
	(gethash att-name att-info-table)))))

;; We assume that type-info is for a specific operator type
;; (that is, its name has the form operator*op-name).  The 'required' field
;; of attribute info is only guaranteed to have a meaningful value for such
;; types.
;;
;; Type-info can also be nil, in which case nil is returned.
;;
(defun required-operator-atts (type-info)
  (when type-info
    (let ((att-info-table (type-info-attribute-info type-info)))
      (when att-info-table
	(let ((result nil))
	  (maphash #'(lambda (att-name att-info)
		       (when (attribute-info-required att-info)
			 (push att-name result)))
		   att-info-table)
	  result)))))

;; Adds a type-info structure for type-name in data-model, if there isn't
;; one already.  Returns the type-info structure (whether new or old).
;;
;; For primitive types, the primitive argument must be the function name
;; or item list defining the type.  Otherwise it must be nil.
;;
;; It is an error to try to redefine a non-primitive type as primitive,
;; or vice-versa.
;;
;; This will NOT overwrite any existing definition for the type.
;;
(defun add-type-to-model (data-model type-name primitive)
  (let ((existing-type-info (gethash type-name data-model)))

    (cond (existing-type-info
	   (cond ((and primitive
		       (not (type-info-primitive existing-type-info)))
		  (taql-warn
		   "Attempted to redefine type ~S as primitive.  It was ~
                    previously declared or inferred to be a structured type."
		   type-name))
		 ((and (not primitive)
		       (type-info-primitive existing-type-info))
		  (taql-warn
		   "Attempted to redefine type ~S as a structured type.  It ~
                    was previously declared or inferred to be a primitive ~
                    type."
		   type-name)))
	   existing-type-info)

	  (t
	   (setf (gethash type-name data-model)
		 (make-type-info :name type-name
				 :primitive primitive))))))

;; Adds att-name to the attributes used with type-name in data-model,
;; creating an object type named type-name first if necessary.
;; Returns the attribute-info structure for the added attribute (whether it
;; was pre-existing or not).
;;
(defun add-attribute-to-model (data-model type-name att-name)
  (let* ((type-info (add-type-to-model data-model type-name nil)))
    (add-attribute-to-type-info type-info att-name)))

;; Adds att-name to the attributes used with type-name in type-info.
;; Returns the attribute-info structure for the added attribute (whether it
;; was pre-existing or not).
;;
(defun add-attribute-to-type-info (type-info att-name)
  (let ((att-info-table
	 (or (type-info-attribute-info type-info)
	     (setf (type-info-attribute-info type-info)
		   (new-attribute-info-table)))))

    (or (gethash att-name att-info-table)
	(setf (gethash att-name att-info-table)
	      (make-attribute-info :name att-name)))))

;; This adds value-type-name to the list of names of acceptable value types
;; for attribute att-name in type type-name.
;;
;; Information on type-name, att-name, and/or value-type-name need not
;; pre-exist.  It just stores the type name of the value type without
;; checking to see whether a type with that name exists.  This makes
;; mutually recursive type definitions possible.
;;
;; We don't guarantee that this function will return any particular value.
;;
(defun add-value-type (data-model type-name att-name value-type-name)
  (pushnew value-type-name
	   (attribute-info-value-types
	    (add-attribute-to-model data-model type-name att-name))))

;; This adds value to the list of primitive values that have been used
;; for att-name in type-name.  Information on type-name and att-name.
;; need not pre-exist.  The updated list of used values is returned.
;;
(defun add-used-value (data-model type-name att-name value)
  (pushnew value
	   (attribute-info-used-values
	    (add-attribute-to-model data-model type-name att-name))
	   :test #'primitive-eql))

;; Return a list of the primitive definitions of the types named in type-names
;; that are declared primitive in the global declared model.  If none of
;; the types are primitive, return nil.
;;
(defun get-primitives (type-names)
  (let ((result nil))
    (dolist (type-name type-names)
      (let ((prim-def (get-primitive type-name)))
	(when prim-def
	  (push prim-def result))))
    result))

;; Return a sublist of type-names:  the names of types that are not
;; declared primitive in the global declared model.  The result includes both
;; types that are declared non-primitive and that are not declared at all.
;;
(defun get-non-primitives (type-names)
  (let ((result nil))
    (dolist (type-name type-names)
      (when (not (get-primitive type-name))
	(push type-name result)))
    result))

;; If type-name is declared as a primitive type in the global declared model,
;; return the primitive spec stored for it there.  Else return nil.
;;
(defun get-primitive (type-name)
  (let ((type-info (gethash type-name (get-data-model 'declared '*global*))))
    (when type-info
      (type-info-primitive type-info))))

;; Return non-nil iff type-name is defined globally as a primitive type and
;; value is of that type.
;;
(defun of-primitive-type-p (type-name value)
  (let ((primitive (get-primitive type-name)))
    (and primitive
	 (of-primitive-def-p primitive value))))

;; Return non-nil iff value is of the primitive type defined by primitive-def.
;; Primitive-def must either be a list of primitive values or the name of
;; a function that recognizes values of the desired type.
;;
(defun of-primitive-def-p (primitive-def value)
  (if (listp primitive-def)
    (member value primitive-def :test #'primitive-eql)
    ;; ELSE
    (eval `(,primitive-def ',value))))

;; This returns the data model that contains the declared type-info for
;; type-name that should be used for type checking (local if exists, else
;; global).  Return nil if there is no such declaration.
;;
(defun get-declared-model (type-name)
  (let* ((local-model (get-data-model 'declared *current-segment-name*))
	 (type-info (gethash type-name local-model)))

    (if (and type-info
	     (not (type-info-global type-info)))
      local-model
      ;; ELSE
      (let* ((global-model (get-data-model 'declared '*global*))
	     (type-info (gethash type-name global-model)))
	(when type-info
	  global-model)))))

;; This returns the declared type-info for type-name that should be used for
;; type checking (local if exists, else global).  Return nil if there is no
;; such declaration.
;;
(defun get-declared-type-info (type-name)
  (let ((type-info (gethash type-name
			    (get-data-model 'declared
					    *current-segment-name*))))
    (if (and type-info
	     (not (type-info-global type-info)))
      type-info
      ;; ELSE
      (gethash type-name (get-data-model 'declared '*global*)))))

;; Return the value type names of att-name for type-name in data-model, if
;; known.  Else return nil.
;;
;; If the warn-on-undeclared argument is non-nil, then we
;;
;;    1. Issue a soft warning when we detect an undeclared value type.
;;    2. Return only the declared value type names.
;;
;; It would be better to do this checking in a way that we didn't have to
;; check whenever we look up the value types, and only issued the warning
;; once.
;;
(defun get-value-types (data-model type-name att-name warn-on-undeclared)
  (let* ((type-info (gethash type-name data-model))
	 (value-types
	  (when type-info
	    (let ((att-info-table (type-info-attribute-info type-info)))
	      (when att-info-table
		(let ((att-info (gethash att-name att-info-table)))
		  (when att-info
		    (attribute-info-value-types att-info))))))))
    (if warn-on-undeclared
      (let ((declared-value-types nil))
	(dolist (value-type value-types)
	  (if (get-declared-type-info value-type)
	    (push value-type declared-value-types)
	    ;; ELSE
	    (taql-warn-soft
	     "The value type for (~S ^~S) was defined to be ~
              ~S, but type ~S has not been declared."
	     type-name att-name value-types value-type)))
	declared-value-types)
      ;; ELSE
      value-types)))

;; This looks up the value type for *current-class*/*current-attribute*
;; in the global declared data model, and returns its name.  It is an
;; error if the type cannot be determined, or if the declared type is
;; a primitive type, or if the declare type is a union containing more than
;; one non-primitive type.  This function is used to determine class
;; information when compiling attribute paths and structured object specs.
;;
;; If it can't determine a valid object type, it issues a warning and
;; returns the symbol *unknown*.
;;
;; Maybe it would be better to write something like this in pseudo-sp.lisp,
;; where the error messages could more explicitly direct people to look at
;; an attribute path, or structured value spec.
;;
(defun get-structured-value-type-name ()
  (let ((type-name *current-class*)
	(att-name *current-attribute*))

    (cond
     ((eql type-name '*unknown*)
      (if (eql att-name '*unknown*)
	(taql-warn "Could not determine value type for attribute:  could not ~
                    find constant class symbol.")
	;; ELSE
	(taql-warn "Could not determine value type for attribute ~S:  could ~
                    not find constant class symbol."
		   att-name))
      '*unknown*)
     ((eql att-name '*unknown*)
      (taql-warn "Could not determine value type for attribute in type ~S:  ~
                  could not find constant attribute symbol."
		 type-name)
      '*unknown*)
     (t
      (let ((type-info (get-declared-type-info type-name)))
	(if type-info
	  (let ((att-info-table (type-info-attribute-info type-info)))
	    (if att-info-table
	      (let ((att-info (gethash att-name att-info-table)))
		(if att-info
		  (get-structured-value-type-name-aux type-name att-name
						      att-info)
		  ;; ELSE
		  (if (and (operator-type-name-p type-name)
			   (setq att-info
				 (attribute-exists
				  (get-declared-type-info 'operator)
				  att-name)))
		    (get-structured-value-type-name-aux 'operator att-name
							att-info)
		    ;; ELSE
		    (progn
		      (taql-warn
		       "Attribute ~S is not declared for type ~S, but its ~
                        value type must be known to complete compilation."
		       att-name type-name)
		      '*unknown*))))
	      ;; ELSE
	      (progn
		(if (type-info-primitive type-info)
		  (taql-warn
		   "The value type of (~S ^~S) must be known ~
                    in order to complete compilation, but type ~S was ~
                    declared as a primitive type."
		   type-name att-name type-name)
		  ;; ELSE
		  (taql-warn
		   "The value type of (~S ^~S) must be known ~
                    in order to complete compilation, but no attributes are ~
                    declared for type ~S."
		   type-name att-name type-name))
		'*unknown*)))
	  ;; ELSE
	  (progn
	    (taql-warn
	     "The value type of (~S ^~S) must be known ~
              in order to complete compilation, but type ~S has not been ~
              declared."
	     type-name att-name type-name)
	    '*unknown*)))))))

(defun get-structured-value-type-name-aux (type-name att-name att-info)
  (let ((value-types
	 (attribute-info-value-types att-info)))
    (if value-types
      (let ((non-primitives
	     (remove '*unknown*
		     (get-non-primitives value-types))))
	(cond ((not non-primitives)
	       (taql-warn
		"The declared value type ~S for (~S ^~S) ~
                 contains no non-primitive types, but a ~
                 structured value is used here."
		value-types type-name att-name)
	       '*unknown*)
	      ((cdr non-primitives)
	       (taql-warn
		"Multiple non-primitive value types ~S are ~
                 declared for (~S ^~S), but a single value ~
                 type is needed here in order to continue compilation."
		non-primitives type-name att-name)
	       '*unknown*)
	      (t
	       ;; No error, return the value type
	       (car non-primitives))))
      ;; ELSE
      (progn
	(taql-warn
	 "A value type is not declared for (~S ^~S) ~
          but the value type must be known to complete compilation."
	 type-name att-name)
	'*unknown*))))

;; This is the function that should always be used to get a data model from
;; a segment.  It takes two arguments:  the segment name and the model type,
;; which must be one of the symbols:  inferred, declared.
;;
;; A TAQL warning is issued if there is no segment with the given name.
;; In this case, it returns an empty data model, in an effort to avoid
;; cascading errors.
;;
;; Otherwise, if the named segment has a model of the specified type, it
;; is returned.  If it doesn't have a model of that type yet, it is given
;; an empty one, and that new model is returned.
;;
;; Note that get-data-model will always return a data-model, never nil.
;;
(defun get-data-model (type segment-name)
  (declare (type symbol type segment-name))

  (let ((segment (gethash segment-name *segment-table*)))
    (cond ((null segment)
	   (taql-warn "There is no data segment named ~S" segment-name)
	   (new-data-model))
	  ((eql type 'inferred)
	   (or (segment-inferred-data-model segment)
	       (setf (segment-inferred-data-model segment)
		     (new-data-model))))
	  ((eql type 'declared)
	   (or (segment-declared-data-model segment)
	       (setf (segment-declared-data-model segment)
		     (new-data-model))))
	  (t
	   (error "INTERNAL TAQL ERROR:  type must be inferred/declared, but ~
                   received ~S"
		  type)))))

(defun get-type-info (type-name model-type segment-name)
  (gethash type-name (get-data-model model-type segment-name)))

;; This is like get-type-info, but takes an argument that is a list of the
;; form ({declared | inferred} TYPE-NAME SEGMENT-NAME).
;;
(defun get-type-info-2 (triple)
  (get-type-info (cadr triple) (car triple) (caddr triple)))

(defun empty-type-info (type-info)
  (or (null type-info)
      (and (not (type-info-primitive type-info))
	   (or (null (type-info-attribute-info type-info))
	       (zerop (hash-table-count
		       (type-info-attribute-info type-info)))))))

(defun declared-global-p (type-name segment-name)
  (let ((type-info (get-type-info type-name 'declared segment-name)))
    (and type-info
	 (type-info-global type-info))))

(defun declared-local-p (type-name segment-name)
  (let ((type-info (get-type-info type-name 'declared segment-name)))
    (and type-info
	 (not (type-info-global type-info)))))

(defun empty-data-model (model)
  (or (null model)
      (zerop (hash-table-count model))))

;; This lets you optional specify an initial size for the data-model's hash
;; table (default 20), but it won't create a table smaller than 20.
;;
(defun new-data-model (&optional (size 20))
  (make-hash-table :size (max 20 size)))

(defun new-attribute-info-table ()
  (make-hash-table :size 10))

(defun reset-per-production-data-model-info ()
  (setq *segment-name-most-recently-added-to*
	*current-segment-name*)

  (setq *current-class* '*unknown*)
  (setq *current-identifiers* nil)
  (setq *current-attribute* '*unknown*)

  (setq *extra-trace-classes* nil)

  (setq *last-aug*-class* '*unknown*)
  (setq *last-aug*-id* nil)

  (setq *current-object-components-stack* nil)

  (setq *saved-operator-augs* (clrhash *saved-operator-augs*))

  (setq *id-class-mapping* (clrhash *id-class-mapping*))
  (setq *class-attribute-value-set* nil))

(defmacro all-info-declared (&body body)
  `(all-info-declared-aux ',body))

(defun all-info-declared-aux (body)
  (if (null (cdr body))
    (taql-warn2 "Usage:  (all-info-declared {:yes | :no} { TYPE-NAME | (SEGMENT-NAME TYPE-NAME+) }+)")
    ;; ELSE

    (let ((yes (member :yes body))
	  (no (member :no body)))
      
      (if (or (and yes no)
	      (and (not yes) (not no)))
	(taql-warn2 "Exactly one of :yes, :no must be given as an argument ~
                     to all-info-declared.")
	;; ELSE
	(dolist (arg body)
	  (cond
	   ((member arg '(:yes :no))
	    ;; Do nothing
	    )
	   ((and (not (symbolp arg))
		 (not (and (consp arg)
			   (cdr arg)
			   (every #'symbolp arg))))
	    (taql-warn2 "All-info-declared:  Expected TYPE-NAME or ~
                              (SEGMENT-NAME TYPE-NAME+), but found ~S"
			arg))
	   (t
	    (let ((segment-name (if (symbolp arg) '*global* (car arg)))
		  (type-names (if (symbolp arg) (list arg) (cdr arg))))
	      (dolist (type-name type-names)
		(let ((new-item (list segment-name type-name))
		      (type-info
		       (gethash type-name
				(get-data-model 'declared segment-name))))
		  (when (and type-info
			     (type-info-global type-info)
			     (not (eql segment-name '*global*)))
		    ;; We want to ignore global declarations stored in local
		    ;; segments.
		    (setq type-info nil))
		  (cond ((not type-info)
			 (if (eql segment-name '*global*)
			   (taql-warn2 "All-info-declared:  Type ~S is not ~
                                         declared globally."
				       type-name)
			   ;; ELSE
			   (taql-warn2 "All-info-declared:  Type ~S is not ~
                                         declared locally in segment ~S."
				       type-name segment-name)))
			((type-info-primitive type-info)
			 (taql-warn2
			  "All-info-declared takes only non-primitive type ~
                            names as arguments, but ~S was declared primitive."
			  arg))
			(no
			 (when *taql-default-mode*
			   (pushnew new-item
				    *sticky-all-info-not-declared-types*
				    :test #'equal))
			 (pushnew new-item *all-info-not-declared-types*
				  :test #'equal))
			(t
			 (when *taql-default-mode*
			   (setq *sticky-all-info-not-declared-types*
				 (delete new-item
					 *sticky-all-info-not-declared-types*
					 :test #'equal)))
			 (setq *all-info-not-declared-types*
			       (delete new-item *all-info-not-declared-types*
				       :test #'equal)))))))))))))
  t)

(defmacro all-types-declared (&body body)
  `(all-types-declared-aux ',body))

(defun all-types-declared-aux (body)
  (if (or (null body)
	  (cddr body)
	  (not (member (car body) '(:yes :no))))
    (taql-warn2 "Usage: (all-types-declared {:yes | :no})")
    ;; ELSE
    (progn
      (when *taql-default-mode*
	(setq *sticky-all-types-declared* (car body)))
      (setq *all-types-declared* (eql (car body) :yes))))
  t)

(defmacro all-operators-declared (&body body)
  `(all-operators-declared-aux ',body))

(defun all-operators-declared-aux (body)
  (if (or (null body)
	  (cddr body)
	  (not (member (car body) '(:yes :no))))
    (taql-warn2 "Usage: (all-operators-declared {:yes | :no})")
    ;; ELSE
    (progn
      (when *taql-default-mode*
	(setq *sticky-all-operators-declared* (car body)))
      (setq *all-operators-declared* (eql (car body) :yes))))
  t)

;; When we get an excise-task, we
;;
;;   - Invoke all-types-declared on *sticky-all-types-declared*.
;;   - Invoke all-operators-declared on *sticky-all-operators-declared*.
;;   - Reset *all-info-not-declared-types* to
;;     *sticky-all-info-not-declared-types*.
;;   - Do (all-info-declared :no goal problem-space state operator).
;;
;; We don't need to redefine sticky declarations.  We assume that sticky
;; declarations are retained as part of excising all segments (see
;; segment-excise-task-extras in segments.lisp).
;;
;; This must be called AFTER segment-excise-task-extras.
;;
(defun data-model-excise-task-extras ()
  (eval `(all-types-declared ,*sticky-all-types-declared*))
  (eval `(all-operators-declared ,*sticky-all-operators-declared*))
  
  (setq *all-info-not-declared-types*
	*sticky-all-info-not-declared-types*)

  (all-info-declared :no goal problem-space state operator))

;; Excise type information specified by a list of parsed data model spec
;; triples.  See the parse-model-spec-triple function for a description of
;; the form of a parsed triple.
;;
;; Calling this on a batch of arguments is much more efficient than calling
;; it iteratively on each component of the batch.
;;
;; The local declarations-removed variable stores a list of entries of
;; the form (TYPE-NAMES SEGMENT-NAME), where TYPE-NAMES is either :all or
;; a list of type names.  Each entry specifies a declaration that was removed
;; from a segment.
;;
(defun excise-type-info-for-data-models (parsed-model-specs)
  (when parsed-model-specs
    (let ((declarations-removed nil)
	  (global-segment nil)
	  (global-model nil))

      ;; Get rid of the specified declarations
      ;;
      (dolist (parsed-spec parsed-model-specs)
	(setq declarations-removed
	      (append (excise-type-info-for-data-models-aux parsed-spec)
		      declarations-removed)))

      ;; Restore sticky declarations.

      (when declarations-removed

	;; We always want to redeclare sticky declarations in the
	;; global segment (e.g. the Primitive type).
	;;
	(push '(:all *global*) declarations-removed)

	;; If any declarations were removed, get rid of the global declared
	;; model as well.  Recreate the global segment if for some reason
	;; it doesn't exist.
	;;
	(setq global-segment (add-segment '*global*))
	(setq global-model (new-data-model))
	(setf (segment-declared-data-model global-segment)
	      global-model)

	;; Reinstall sticky declarations that were removed.
	;; *sticky-declarations* is stored in reverse order of the order in
	;; which the declarations were originally performed, so we have to
	;; reverse it here.
	;;
	;; Rebind *taql-default-mode* to nil while we are redefining a sticky
	;; declaration so that it doesn't end up on the list of sticky
	;; declarations twice by accident.
	;;
	(dolist (decl-info (reverse *sticky-declarations*))
	  (when (member-if #'(lambda (removed-spec)
			       ;; Check that segment and type of the sticky
			       ;; declaration match those of one of the
			       ;; declarations removed.
			       (and (eql (car decl-info) (cadr removed-spec))
				    (or (eql (car removed-spec) :all)
					(member
					 (if (member (cadr decl-info)
						     '(defoperator
							defoperator-merge))
					   (makesym 'operator*
						    (caddr decl-info))
					   ;; ELSE
					   (caddr decl-info))
					 (car removed-spec)))))
			   declarations-removed)
	    (let ((*current-segment-name* (car decl-info))
		  (declaration (cdr decl-info))
		  (*taql-default-mode* nil)
		  (*inhibit-global-inheritance* t))
	      (add-segment *current-segment-name*)
	      (eval declaration))))
    
	;; Set the global declarations to the union of the global declarations 
	;; remaining in all segments.  This ensures that the global model
	;; doesn't retain any declarations that were supposed to have been
	;; excised.  At this point, the global model will only contain
	;; sticky global declarations that were excised.
	;;
	(dolist (segment-name (all-non-global-segment-names))
	  (maphash #'(lambda (type-name type-info)
		       (declare (ignore type-name))
		       (when (type-info-global type-info)
			 (merge-type-into-data-model type-info global-model)))
		   (get-data-model 'declared segment-name)))))))

;; This function should only be called from excise-type-info-for-data-models.
;; It removes specified type information regardless of whether it came
;; from a sticky declaration or not.
;;
(defun excise-type-info-for-data-models-aux (parsed-spec)
  (let ((which-types (car parsed-spec))
	(which-models (cdr parsed-spec))
	(declarations-removed nil))

    (dolist (model-spec which-models)
      (let* ((model-type (car model-spec))
	     (segment-name (cadr model-spec))
	     (segment (gethash segment-name *segment-table*)))

	(when segment

	  (case model-type
	    (declared
	     (let ((model (segment-declared-data-model segment))
		   (types-excised nil))
	       (when model
		 (if (eql which-types :all)
		   (progn
		     (when (> (hash-table-count model) 0)
		       (push (list :all segment-name) declarations-removed))
		     (setf (segment-declared-data-model segment) nil))
		   ;; ELSE
		   (progn
		     (dolist (type-name which-types)
		       (when (gethash type-name model)
			 (push type-name types-excised)
			 (remhash type-name model)))
		     (when types-excised
		       (when (zerop (hash-table-count model))
			 (setf (segment-declared-data-model segment) nil))
		       (push (list types-excised segment-name)
			     declarations-removed)))))))

	    (inferred
	     (if (eql which-types :all)
	       (setf (segment-inferred-data-model segment) nil)
	       ;; ELSE
	       (let ((model (segment-inferred-data-model segment)))
		 (when model
		   (dolist (type-name which-types)
		     (remhash type-name model))
		   (when (zerop (hash-table-count model))
		     (setf (segment-inferred-data-model segment) nil))))))

	    (t
	     (error "INTERNAL TAQL ERROR: Case selector fell through: ~S"
		    model-type))))))

    declarations-removed))

(defun init-data-model-stuff ()
  (setq *proposed-operator-id* nil)

  (setq *inhibit-global-inheritance* nil)

  (setq *id-class-mapping*
	(make-hash-table :size 50))

  (setq *saved-operator-augs*
	(make-hash-table :size 5))

  (setq *all-info-not-declared-types* nil)
  (setq *sticky-all-info-not-declared-types* nil)
  (setq *sticky-all-types-declared* :no)
  (setq *sticky-all-operators-declared* :no)
  (setq *sticky-declarations* nil)
  
  (all-types-declared :no)
  (all-operators-declared :no)

  ;; All declarations that exist only in the global segment, such as
  ;; Primitive, must be declared stickily.  Otherwise
  ;; excise-type-info-for-data-models will not work properly.
  ;;
  (eval
   '(progn
      (let ((*current-segment-name* '*global*)
	    (*taql-default-mode* t))
	(defprimitive primitive primitive-p)
	(defobject-merge goal :global yes)
	(defobject-merge problem-space :global yes)
	(defobject-merge state :global yes)
	(defobject-merge operator :global yes))))

  (all-info-declared :no goal problem-space state operator)

  t)

(eval-when (compile load eval)
  (soarsyntax))
