;;; TAQL Compiler, Segment Module
;;;
;;; Gregg Yost, Erik Altmann
;;; School of Computer Science
;;; Carnegie Mellon University
;;;
;;; Working file: /afs/cs/user/altmann/soar/taql/segments.lisp
;;; Created March 15, 1991
;;;
;;; This file implements the functions involved with TAQL's segment
;;; facility.
;;;
;;; EXPORTED ROUTINES:
;;;
;;;    - FILL THIS IN
;;;
;;; Known bugs/funnies:
;;;
;;; =======================================================================
;;; Modification history:
;;; =======================================================================
;;;
;;; 6-11-91 - gry - Changed so that tc-name 'segment property is removed
;;;   when a TC is excised under (excise-task) or (excise SEGMENT-NAME).
;;;
;;; 4-5-91 through 4-22-91 - gry - added space model support
;;;
;;; 3-15-91 - gry - Created.

;;; *** BEGIN CODE ***

(eval-when (compile load eval)
  (lispsyntax))

(eval-when (compile load eval)
   (if (find-package "COMMON-LISP-USER")
       (in-package "COMMON-LISP-USER")
       (in-package "USER")))

;; These are initialized in init-segment-stuff.
;;
;; Enveloped in eval-when -- TFMcG 5-Sep-91
(eval-when (compile eval load)
	   (proclaim
	    '(special
	      *current-segment-name*
	      *segment-name-most-recently-added-to*
	      *segment-table*
	      *segment-group-table*
	      *sticky-segments*
	      )))

;; This has to be defined before begin-segment, because begin-segment calls
;; end-segment, and the compiler has to know that it is a macro.
;;
(defmacro end-segment (&body body)
  `(end-segment-aux ',body))

(defun end-segment-aux (body)
  (if body
    (taql-warn2 "End-segment does not take any arguments:  ~S"
		(cons 'end-segment body))
    ;; ELSE
    (setq *current-segment-name* 'user))

  t)

;; Some places in the code temporarily rebind *current-segment-name* before
;; doing something that depends on what segment it is done in.  This is
;; safer than doing begin-segment/end-segment in the code, because it doesn't
;; leave *current-segment-name* in a funny state if someone interrupts the
;; program.  But it depends on begin-segment and end-segment not doing any
;; more than resetting *current-segment-name*, so we should try to preserve
;; that property.  For this sort of rebinding to work, the value assigned
;; to *current-segment-name* must be the name of an existing segment.
;;
(defmacro begin-segment (&body body)
  `(begin-segment-aux ',body))

(defun begin-segment-aux (body)
  (cond ((null body)
	 *current-segment-name*)
	((or (cddr body)                  ; more than one arg
	     (not (symbolp (car body))))  ; arg not a symbol
	 (taql-warn2 "Begin-segment takes at most one argument ~
                     (a symbolic segment name):  ~S"
		     (cons 'begin-segment body))
	 nil)
	((gethash (car body) *segment-group-table*)
	 (taql-warn2 "Begin-segment:  ~S is already defined as the name of ~
                      a segment group.  It cannot also be defined as the
                      name of a segment."
		     (car body))
	 nil)
	(t
	 (when (not (eql *current-segment-name* 'user))
	   (end-segment))
	 (add-segment (car body))
	 (setq *current-segment-name* (car body))
	 *current-segment-name*)))

;; Segment-group:
;;
;;   - With no args, return list of defined segment group names.
;;   - With 1 arg, return list of segments in the expansion of that group
;;     name (error if undefined).
;;   - With > 1 args, define first arg to be a group name for the segments
;;     and segments-groups named in the remaining arguments.  The first
;;     argument cannot be :all or :latest, and the remaining arguments must be
;;     already-defined names.  Return the name of the defined group.
;;
(defmacro segment-group (&body body)
  `(segment-group-aux ',body))

(defun segment-group-aux (body)
  (cond ((null body)
	 (let ((result nil))
	   (maphash #'(lambda (seg-group-name member-segs)
			(declare (ignore member-segs))
			(push seg-group-name result))
		    *segment-group-table*)
	   result))

	((null (cdr body))
	 (if (eql (car body) :all)
	   (all-non-global-segment-names)
	   ;; ELSE
	   (let ((segments (gethash (car body) *segment-group-table*)))
	     (if segments
	       segments
	       ;; ELSE
	       (progn
		 (taql-warn2 "Segment-group:  ~S is not the name of a defined ~
                            segment group."
			     (car body))
		 nil)))))

	((eql (car body) :all)
	 (taql-warn2 "Segment-group:  :all cannot be defined as the name of ~
                      a segment group.")
	 nil)

	((eql (car body) :latest)
	 (taql-warn2 "Segment-group:  :latest cannot be defined as the name ~
                      of a segment group.")
	 nil)

	((not (symbolp (car body)))
	 (taql-warn2 "Segment-group:  The name of a segment group must be a ~
                      symbol, but ~S is not."
		     (car body))
	 nil)

	(t
	 (let ((expansions (mapcar #'expand-segment-spec (cdr body))))
	   (if (member nil expansions)
	     ;; There was an error in one of the specs
	     nil
	     ;; ELSE
	     (progn
	       (setf (gethash (car body) *segment-group-table*)
		     (apply #'append expansions))
	       (car body)))))))

;; SEGMENT-SPEC ::= SEGMENT-GROUP-NAME | SEGMENT-NAME | :all
;;
(defun segment-spec-p (arg)
  (or (eql arg :all)
      (gethash arg *segment-table*)
      (gethash arg *segment-group-table*)))

;; Returns non-nil iff arg is the name of an existing segment
;;
(defun segment-name-p (arg)
  (gethash arg *segment-table*))

;; SEGMENT-SPEC ::= SEGMENT-GROUP-NAME | SEGMENT-NAME | :all | :latest
;;
;; :latest stands for the name of the segment something was most recently
;; added to (initially the user segment).
;;
;; Returns nil iff an error was detected.  Else returns a list of segment
;; names, expanded from its argument (a SEGMENT-SPEC).
;;
(defun expand-segment-spec (spec)
  (cond ((not (symbolp spec))
	 (taql-warn2
	  "A segment specifier must be a symbol, :all, or :latest, but ~
           found ~S"
	  spec)
	 nil)
	((eql spec :all)
	 (all-non-global-segment-names))
	((eql spec :latest)
	 *segment-name-most-recently-added-to*)
	((gethash spec *segment-table*)
	 (list spec))
	(t
	 (let ((segments (gethash spec *segment-group-table*)))
	   (if segments
	     segments
	     ;; ELSE
	     (progn
	       (taql-warn2 "~S is not the name of a loaded segment or ~
                            defined segment group."
			   spec)
	       nil))))))

(defun segment-tcs (segment-name)
  (let ((segment (gethash segment-name *segment-table*)))
    (when segment
      (segment-tc-names segment))))

;; The *global* segment isn't a real segment, we just use it as a convenient
;; place to store the global declared data model.  That makes it possible to
;; manipulate the global declared model with the same functions we use
;; for other models.  But we want to exclude it from the list of real segments.
;;
(defun all-non-global-segments ()
  (let ((segments nil))
    (maphash #'(lambda (name segment)
		 (when (not (eql name '*global*))
		   (push segment segments)))
	     *segment-table*)
    segments))

(defun all-non-global-segment-names ()
  (mapcar #'segment-name (all-non-global-segments)))

;; Add a new data segment named segment-name, returning the created segment.
;; Or, if a segment with that name already exists, return it.
;;
(defun add-segment (segment-name)
  (when *taql-default-mode*
    (pushnew segment-name *sticky-segments*))

  (or (gethash segment-name *segment-table*)
      (setf (gethash segment-name *segment-table*)
	    (make-segment :name segment-name))))

;; The TC is added to the current segment if its doesn't already have a
;; segment property.  Otherwise it is added to the segment that is named
;; on the segment property.  This is so that TCs that are re-evaluated during
;; editing/debugging remain in their original segment without user effort.
;; One implication is that we shouldn't remove the segment property when a
;; TC is excised, unless it is excised as part of an (excise-task) or
;; (excise SEGMENT-NAME).
;;
(defun add-tc-to-segment (tc-name)
  (let* ((segment-name (or (get tc-name 'segment) *current-segment-name*))
	 (segment (add-segment segment-name)))
    (setq *segment-name-most-recently-added-to* segment-name)
    (setf (get tc-name 'segment) segment-name)
    (push tc-name (segment-tc-names segment))))

;; We deliberately don't remove the segment property.  See add-tc-to-segment.
;;
;; 6-11-91 - gry:
;;
;; The forget-tc-was-in-segments argument
;; should be a list of segments that, if the TC was in that segment, then
;; TAQL will not retain the fact that it had been in that segment.  So, if
;; a TC with the same name is later reloaded, it will not be forced into its
;; old segment.  When a TC is excise as part of (excise-task) or
;; (excise SEGMENT-NAME), you want it to forget which segment the TC was in,
;; but otherwise you don't.
;;
(defun remove-tc-from-segment (tc-name forget-tc-was-in-segments)
  (when (not (default-tc-p tc-name))
    (let* ((segment-name (get tc-name 'segment))
	   (segment (gethash segment-name *segment-table*)))
      (when segment
	(setf (segment-tc-names segment)
	      (delete tc-name (segment-tc-names segment))))
      (when segment-name
	(when (member segment-name forget-tc-was-in-segments)
	  (remprop tc-name 'segment))))))

;; This doesn't excise the segment's TCs, we assume that that is done by the
;; caller if necessary.  Neither does it update the book-keeping involved
;; with a TC's segments.  Again, we assume that is done as part of excising
;; a TC (see taql-excise-tc in taql-compiler.lisp).  All this function does
;; is the extra work needed to excise a segment beyond excising its TCs.
;;
(defun taql-excise-segments (segment-names)
  (excise-data-model-aux (list (list 'all :all segment-names)))
  (dolist (segment-name segment-names)
    (let ((segment (gethash segment-name *segment-table*)))
      (when segment
	(remove-segment-from-space-models segment-name)
	(when (and (not (member segment-name *sticky-segments*))
		   (null (segment-tc-names segment))
		   (empty-data-model (segment-declared-data-model segment))
		   (empty-data-model (segment-inferred-data-model segment))
		   (not (segment-has-space-models segment-name)))
	  (remhash segment-name *segment-table*)
	  (remove-segment-from-segment-groups segment-name)
	  (when (eql segment-name *current-segment-name*)
	    (setq *current-segment-name* 'user))
	  (when (eql segment-name *segment-name-most-recently-added-to*)
	    (setq *segment-name-most-recently-added-to* 'user)))))))

;; We call this when we remove a segment from the segment table, so that
;; we don't have any "dangling pointers" to nonexistent segments.  If this
;; causes the segment list for a segment group to become empty, we remove
;; the segment group from the segment group table as well.
;;
(defun remove-segment-from-segment-groups (segment-name)
  (maphash #'(lambda (group-name expansion)
	       (when (member segment-name expansion)
		 (setf (gethash group-name *segment-group-table*)
		       (remove segment-name
			       (gethash group-name *segment-group-table*)))
		 (when (null (gethash group-name *segment-group-table*))
		   (remhash group-name *segment-group-table*))))
	   *segment-group-table*))

;; When we get an excise-task, we
;;
;;   - Excise all segment info.  This will leave empty segments around for
;;     all sticky segments.
;;   - Set the current segment name and the segment name most recently
;;     added to to User.
;;
(defun segment-excise-task-extras ()
  (taql-excise-segments (all-non-global-segment-names))

  (setq *current-segment-name* 'user)
  (setq *segment-name-most-recently-added-to* 'user)

  t)

;; This is called by init-taql.
;;
(defun init-segment-stuff ()
  (setq *sticky-segments* nil)

  (setq *segment-group-table*
	(make-hash-table :size 15))

  (setq *segment-table*
	(make-hash-table :size 15))

  ;; The soar and taql segments have to be created now.  If we wait,
  ;; then we'll get an error whenever a redefined Soar command (like
  ;; start-default) is executed, because mixed-args-to-soar-args will
  ;; try to use the soar and taql segment names in the implicit :exclude
  ;; arguments.
  ;;
  (let ((*taql-default-mode* t))
    (add-segment '*global*)
    (add-segment 'user)
    (add-segment 'soar)
    (add-segment 'taql))

  (setq *current-segment-name* 'user)
  (setq *segment-name-most-recently-added-to* 'user)

  t)

(eval-when (compile load eval)
  (soarsyntax))
