;;; TAQL Compiler, Data Type Module
;;;
;;; Gregg Yost, Erik Altmann
;;; School of Computer Science
;;; Carnegie Mellon University
;;;
;;; Working file: /afs/cs/user/altmann/soar/taql/data-types.lisp
;;; Created September 5, 1990
;;;
;;; This file implements the functions that support TAQL's data types.
;;;
;;; Known bugs/funnies:
;;; 
;;;    This file needs to be better organized.  We need to identify the
;;;    exported routines, and clean up the interface between this file
;;;    and pseudo-sp.lisp.
;;;
;;; =======================================================================
;;; Modification history:
;;; =======================================================================
;;;
;;; 6-11-91 - gry - Added use-taql-library command.
;;;
;;; 9-5-90 - gry - Created.

;;; *** BEGIN CODE ***

(eval-when (compile load eval)
  (lispsyntax))

(eval-when (compile load eval)
   (if (find-package "COMMON-LISP-USER")
       (in-package "COMMON-LISP-USER")
       (in-package "USER")))

;; This variable should be defined in the TAQL load file.
;;
;; Currently, the only TAQL libraries are those associated with TAQL's
;; data types, for example the eval-expr operator for the expression data
;; type.  Some of these libraries contain substantial amounts of code,
;; so we don't want to force people to load it unless they really need it.
;; If they do, they can load selected files from this directory and its
;; subdirectories.
;;
;; 6-11-91 - gry - See the use-taql-library command for the new, simpler
;;   way users should load libraries.
;;
(defvar *taql-library-directory-pathname* nil
  "Pathname of directory containing TAQL libraries, should be defined in the TAQL load file at each site.")

(defvar *loaded-taql-libraries* nil
  "A list of symbols identifying loaded TAQL libraries.")

(defmacro use-taql-library (&body body)
  `(use-taql-library-aux ',body))

(defun use-taql-library-aux (body)
  (cond ((or (null body)
	     (cdr body))
	 (taql-warn2 "Usage:  (use-taql-library LIBRARY-NAME)"))
	((not (symbolp (car body)))
	 (taql-warn2 "Use-taql-library:  Expected a symbolic library name as an argument, but got ~S"
		     (car body)))
	(t
	 (let* ((arg (car body))
		(filename (merge-pathnames
			   (make-pathname
			    :name (concatenate 'string
					       (string-downcase
						(symbol-name arg))
					       "-lib")
			    :type "taql")
			   *taql-library-directory-pathname*)))
	   (when (not (member arg *loaded-taql-libraries*))
	     (cond ((probe-file filename)
		    (load filename)
		    (push arg *loaded-taql-libraries*))
		   (t
		    (taql-warn2 "Use-taql-library:  No such library as ~(~S~)"
				arg)))))))
  t)

;;; The next two functions are part of pseudo-sp expansion, but they contain
;;; data-type specific code so they are included here.  The data-type
;;; specific aspect is that these functions collect and process data-type
;;; directives in the RHS and LHS of pseudo-sps, respectively.

(defun expand-pseudo-sp-rhs-with-directives (rhs)
  (multiple-value-bind (directives lhs rhs)
      (collect-directives rhs 'RHS)
    
    (when (assoc 'member* directives)
      (taql-warn "Member* directives can only be used in conditions."))
    
    (when (assoc 'equal* directives)
      (taql-warn "Equal* directives can only be used in conditions."))
    
    (multiple-value-setq (lhs rhs)
      (process-set-edits (cdr (assoc 'insert* directives)) lhs rhs))
    (multiple-value-setq (lhs rhs)
      (process-set-edits (cdr (assoc 'delete* directives)) lhs rhs))
    (multiple-value-setq (lhs rhs)
      (process-list-edits (cdr (assoc 'rplaca* directives)) lhs rhs))
    (multiple-value-setq (lhs rhs)
      (process-list-edits (cdr (assoc 'rplacd* directives)) lhs rhs))
    (multiple-value-setq (lhs rhs)
      (process-concatenates (cdr (assoc 'concatenate* directives)) lhs rhs))

    (multiple-value-bind (extra-lhs modified-rhs)
	(expand-pseudo-sp*production-part rhs 'RHS)
      (setq lhs (append lhs extra-lhs))
      (setq rhs modified-rhs))

    (values lhs rhs)))

(defun expand-pseudo-sp-lhs-with-directives (listified-lhs)
  (let ((tag (car listified-lhs))
        (top-conds (cadr listified-lhs))
        (conjunctions (cddr listified-lhs)))
    (multiple-value-bind (directives lhs rhs)
	(collect-directives top-conds 'LHS)
      
      (when (assoc 'insert* directives)
        (taql-warn "Insert* directives cannot be used in conditions."))
      
      (when (assoc 'delete* directives)
        (taql-warn "Delete* directives cannot be used in conditions."))
      
      (when (assoc 'Rplaca* directives)
        (taql-warn "Rplaca* directives cannot be used in conditions."))
      
      (when (assoc 'Rplacd* directives)
        (taql-warn "Rplacd* directives cannot be used in conditions."))
      
      (when (assoc 'concatenate* directives)
        (taql-warn "Concatenate* directives cannot be used in conditions."))
      
      (when (and (not (eql tag 'top-level))
                 (assoc 'equal* directives))
        (taql-warn "Equal* directives cannot be used in conjunctions."))
      
      (multiple-value-setq (lhs rhs)
        (process-member* (cdr (assoc 'member* directives)) lhs rhs))

      (multiple-value-bind (modified-lhs extra-rhs)
	  (expand-pseudo-sp*production-part lhs 'LHS)
	(setq lhs modified-lhs)
	(setq rhs (append rhs extra-rhs)))
      
      (multiple-value-bind (conj-lhs conj-rhs)
          (expand-pseudo-sp-lhs-conjunctions conjunctions)
        (setq lhs (append lhs conj-lhs))
        (setq rhs (append rhs conj-rhs)))
      
      (case tag
        (top-level
	 ;; Do equal* last, and only at the top level, so that it affects
	 ;; the whole LHS, conjunctions and all.
	 ;;
         (multiple-value-setq (lhs rhs)
           (process-equal* (cdr (assoc 'equal* directives)) lhs rhs)))
        (non-negated
         (setq lhs `({ ,@lhs })))
	(negated
	 (setq lhs `(- { ,@lhs })))
	(t
	 (error "INTERNAL TAQL ERROR: case selector ~S fell through" tag)))
      
      (values lhs rhs))))
         
;; Collect any directives in a production part (a set of conditions or
;; actions).  The SIDE argument must be either LHS or RHS, indicating
;; whether the part is from the LHS or RHS.
;;
;; It returns three values:
;;
;;   The directives.  The directives are returned as an assoc list,
;;       with each item in the list having the form
;;       (directive-name . list-of-directives).  Data macro calls in the
;;       directive's arguments are expanded, and the directive returned
;;       in the directive list is the expanded ground instance.
;;       An exception is the concatenate* directive, which does NOT
;;       have any data-macro arguments expanded.  The function
;;       process-concatenates will expand them later, but needs to have
;;       access to the unprocessed argument list.
;;   Conditions resulting from the expansion of data macros in the
;;       directives, plus the non-directive things in PART if SIDE = LHS.
;;   Actions resulting from the expansion of data macros in the
;;       directives, plus the non-directive things in PART if SIDE = RHS.
;;
(defun collect-directives (part side)
  (multiple-value-bind (directives new-part extra-lhs extra-rhs)
      (collect-directives-aux part)
    (case side
      (LHS
        (values
           directives
           (append new-part extra-lhs)
           extra-rhs))
      (RHS
        (values
           directives
           extra-lhs
           (append new-part extra-rhs)))
      (t
       (error "INTERNAL TAQL ERROR: case selector ~S fell through" side)))))

(defun collect-directives-aux (clauses)
  (cond ((null clauses)
         (values nil nil nil nil))
        (t
         (multiple-value-bind (directives1 clauses1 lhs1 rhs1)
             (collect-directives-aux (cdr clauses))
           (cond
             ((and (listp (car clauses))
                   (member (caar clauses)
                           '(equal* member* insert* delete* concatenate*
				    rplaca* rplacd*)))
              (multiple-value-bind (ground-call lhs2 rhs2)
		  (if (eql (caar clauses) 'concatenate*)
		    (values (car clauses) nil nil)
		    ;; ELSE
		    (expand-data-macros-in-list (car clauses)))
                (values
                  (let ((item (assoc (car ground-call) directives1)))
                    (if item
                      (push ground-call (cdr item))
                      ; ELSE
                      (push (list (car ground-call) ground-call)
                            directives1))
                    directives1)
                  clauses1
                  (append lhs2 lhs1)
                  (append rhs2 rhs1))))
             (t
              (values
                 directives1
                 (cons (car clauses) clauses1)
                 lhs1
                 rhs1)))))))

;; This should check the number and types of arguments, to the extent that
;; it can.  It assumes that its argument (call) is a list whose first element
;; is the name of a data macro.
;;
(defun expand-data-macro (call)
  (let ((LHS nil)
        (RHS nil)
        (args nil))

    (cond ((data-macro-expands-arguments (car call))
           (multiple-value-setq (args LHS RHS)
             (expand-data-macros-in-list (cdr call))))
          (t ; doesn't expand arguments
           (setq args (cdr call))))
         
    (multiple-value-bind (root-id top-lhs top-rhs)
        (apply (data-macro-expander (car call)) args)
      
      (values root-id (append top-lhs LHS) (append top-rhs RHS)))))

;; If arg is a data macro call, expand it, else return arg.
;;
(defun expand-if-data-macro (arg)
  (if (data-macro-call-p arg)
    (expand-data-macro arg)
    ; ELSE
    (values arg nil nil)))

;; Expand any data macro calls in lis, returning a list with any data macros
;; replaced by their root ids, plus any extra lhs/rhs items resulting from
;; expansions.
;;
(defun expand-data-macros-in-list (lis)
  (let ((lhs nil)
        (rhs nil)
        (root-ids nil))

    (dolist (arg lis)
      (multiple-value-bind (sub-root-id sub-lhs sub-rhs)
          (expand-if-data-macro arg)
        
        (setq lhs (append sub-lhs lhs))
        (setq rhs (append sub-rhs rhs))
        (setq root-ids (add-to-end (list sub-root-id) root-ids))))
  
    ;; Root-ids has been used by add-to-end, so has a special list
    ;; representation.  We take its CAR here to get at the normal
    ;; list representation.  Root-ids ends up being the argument LIS
    ;; with any data-macro calls replaced by the root-ids of their
    ;; expansion.
    ;;
    (setq root-ids (car root-ids))

    (values root-ids lhs rhs)))

(defun declare-data-macro (name type expander expands-arguments
                           compatible-relations)
  (cond ((not (symbolp name))
         (taql-warn2 "The name of a data macro must be a symbol: ~S" name))
        ((not (symbolp type))
         (taql-warn2 "The type of a data macro must be a symbol: ~S" type))
        ((not (functionp expander))
         (taql-warn2 "The expander of a data macro must be a function ~S"
                     expander))
        (t
         (setf (get name 'data-type) type)
         (setf (get name 'data-macro-expander) expander)
         (setf (get name 'data-macro-expands-arguments) expands-arguments)
         (setf (get name 'data-macro-compatible) compatible-relations)))
  t)

;; Return non-nil iff the argument is the name of a data-macro.
;;
(defun data-macro-p (arg)
  (and (atom arg)
       (get arg 'data-macro-expander)))

;; Return non-nil iff the argument is a data macro call.  We assume that
;; any list that starts with the name of a data macro and doesn't contain
;; ^ is a data macro call.  This permits tree-structured value specifications
;; where the class name is the same as the name of a data macro.
;;
(defun data-macro-call-p (arg)
  (and (consp arg)
       (data-macro-p (car arg))
       (not (member '^ arg))))

;; This function assumes that its argument is the name of a data macro.
;;
(defun data-macro-expander (name)
  (get name 'data-macro-expander))

;; This function assumes that its argument is the name of a data macro.
;;
(defun data-macro-expands-arguments (name)
  (get name 'data-macro-expands-arguments))

;; This function assumes that its argument is the name of a data macro.
;; It returns a list of LHS predicates calls to that macro can appear with.
;;
(defun data-macro-compatible (name)
  (get name 'data-macro-compatible))

;;; ----------------------------------------------------------------------

;;; Next come the functions that expand specific data macros and directives.

;;; Data macros for the list data type:

(defun expand-car (arg)
  (let ((root-id (genvar 't)))
    (when (not (variable-p arg))
      (taql-warn "Argument to car must be a variable or data macro call: ~S"
		 arg))
    (values root-id
            `((list ,arg ^ car ,root-id))
            nil)))

(defun expand-cdr (arg)
  (let ((root-id (genvar 't)))
    (when (not (variable-p arg))
      (taql-warn "Argument to cdr must be a variable or data macro call: ~S"
		 arg))
    (values root-id
            `((list ,arg ^ cdr ,root-id))
            nil)))

(defun expand-cons (arg1 arg2)
  (let* ((root-id (genvar 'l))
         (expansion
           `((list ,root-id ^ car ,arg1 ^ cdr ,arg2
                   ,.(if (eql *pseudo-sp-context* 'RHS)
                       (list '^ 'type* 'list))))))

    (when (and (not (variable-p arg2))
	       (not (null arg2)))
      ;; (null) expands to NIL, and we want to accept (cons <x> (null)),
      ;; hence the test for (not (null arg2)) here.
      (taql-warn
       "Second argument to cons must be a variable or data macro call: ~S"
       arg2))

    (if (eql *pseudo-sp-context* 'RHS)
      (values root-id nil expansion)
      ; ELSE LHS
      (values root-id expansion nil))))

;; The null list is just the symbol NIL.
;;
(defun expand-null ()
  (values nil
          nil
          nil))

(defun expand-list (&rest items)
  (expand-data-macro (convert-literal-to-cons items)))

(defun convert-literal-to-cons (arg)
  (cond ((null arg)
         '(null))
        ((atom arg)
         arg)
        (t
          `(cons ,(convert-literal-to-cons (car arg))
                 ,(convert-literal-to-cons (cdr arg))))))

;;; Data macros for the set data type:

(defun expand-empty-set ()
  (expand-set))

(defun expand-set (&rest args)
  (let ((root-id (genvar 's)))
    (cond ((eql *pseudo-sp-context* 'LHS)
           (values
              root-id
              (cond ((null args)
                     `((set ,root-id - ^ member)))
                    ((null (cdr args))  ; singleton set
                     `((set ,root-id ^ member ,@args
                                     - ^ member <> ,@args)))
                    (t
                     `((set ,root-id ^ member ,@args
                            - ^ member { ,@(not-equal-ify args) }))))
              nil))
          (t ; RHS
           (values
              root-id
              nil
              `((set ,root-id ^ type* set
                     ,@(multify 'member args))))))))

;;; Data macros for the expression data type:

;; This uses convert-term-to-soar from the function package to convert
;; the expression literal.  Convert-term-to-soar doesn't do any error
;; checking at the moment (for example to see if only valid operators
;; are used).
;;
;; In the long run, I'd like to support infix literals, with the standard
;; operator precedence.  One glitch here is that Soar's compute
;; functions *doesn't* use the standard precedence (it always evaluates
;; left to right), and the difference could be confusing.  But, I refuse
;; on moral grounds to use a right-to-left precedence.  Another problem
;; is that if people define new operators with sdefun, there would have to be
;; a way of defining their precedence for parsing as well.  This latter
;; problem is the real killer.
;;
(defun expand-expr (&rest args)
  (when (cdr args)
    (taql-warn "Expr takes only one argument: ~S" (cons 'expr args)))

  (let ((root-id (genvar 'e))
	(expanded-expr nil)
	(extra-lhs nil)
	(extra-rhs nil))

    (multiple-value-setq (expanded-expr extra-lhs extra-rhs)
      (expand-data-macros-in-expr (car args)))

    (cond ((eql *pseudo-sp-context* 'LHS)
	   (values
	    root-id
	    (append extra-lhs
		    (convert-term-to-soar expanded-expr root-id nil))
	    extra-rhs))
	  (t
	   (values
	    root-id
	    extra-lhs
	    (append extra-rhs
		    (convert-term-to-soar expanded-expr root-id t)))))))

;; Expand data macro calls that appear as values in an expression literal.
;;
(defun expand-data-macros-in-expr (expr)
  (cond ((atom expr)
	 (values expr nil nil))
	(t
	 (do ((args (cdr expr) (cdr args))
	      (result-args nil)
	      (extra-lhs nil)
	      (extra-rhs nil))
	     ((null args)
	      (values (cons (car expr) (reverse result-args))
		      extra-lhs
		      extra-rhs))
	   (multiple-value-bind (arg lhs-1 rhs-1)
	       (expand-if-data-macro (car args))
	     (setq extra-lhs (append lhs-1 extra-lhs))
	     (setq extra-rhs (append rhs-1 extra-rhs))
	     (cond ((listp arg)
		    (multiple-value-bind (arg-2 lhs-2 rhs-2)
			(expand-data-macros-in-expr arg)
		      (setq extra-lhs (append lhs-2 extra-lhs))
		      (setq extra-rhs (append rhs-2 extra-rhs))
		      (push arg-2 result-args)))
		   (t
		    (push arg result-args))))))))

;;; Data macros for the text data type:

;; Text1 automatically inserts spaces between its arguments, analogous
;; to Soar's write1 and  SoarIO's write1+
;;
(defun expand-text1 (&rest args)
  (expand-text args t))

;; Text2 doesn't insert spaces between its arguments, analogous
;; to Soar's write2 and  SoarIO's write2+
;;
(defun expand-text2 (&rest args)
  (expand-text args nil))
  
;; The Text2 and Text data macros are identical.  See the Soar manual
;; for a description of the special forms (such as (crlf)) that can
;; appear in text literals.
;;
(defun expand-text (items insert-spaces)
  (let ((expansion nil)
	(root-id (genvar 't)))

    (when (null items)
      (setq expansion `((text ,root-id ^ type atom ^ item ""
			               ^ last ,root-id))))

    (do* ((items items (cdr items))
	  (item (car items) (car items))
	  (previous-id nil current-id)
	  (current-id root-id (genvar 't)))
	 ((null items)
	  (when previous-id
	    (setq expansion
		  (cons `(text ,root-id ^ last ,previous-id)
			(nreverse expansion)))))

      (when previous-id
	(setq expansion
	      (cons `(text ,previous-id ^ next ,current-id)
		    expansion))
	(when insert-spaces
	  (setq previous-id current-id)
	  (setq current-id (genvar 't))
	  (setq expansion
		(cons `(text ,previous-id ^ type cc ^ item space
			                  ^ next ,current-id)
		      expansion))))

      (cond ((equal item '(crlf))
	     (setq expansion
		   (cons `(text ,current-id ^ type cc ^ item crlf)
			 expansion)))
	    ((equal item '(tab))
	     (setq expansion
		   (cons `(text ,current-id ^ type cc ^ item tab)
			 expansion)))
	    ((and (listp item)
		  (eql (car item) 'tabto))
	     (when (or (not (cdr item))
		       (cddr item))
	       (taql-warn "Tabto takes exactly one argument: ~S" item))
	     (setq expansion
		   (cons `(text ,current-id ^ type tabto ^ item ,(cadr item))
			 expansion)))
	    ((not (atom item))
	     (taql-warn
	      "Text literal must be an atom or (crlf), (tab), (tabto <n>): ~S"
	      item))
	    (t
	     (setq expansion
		   (cons `(text ,current-id ^ type atom ^ item ,item)
			 expansion)))))

    (if (eql *pseudo-sp-context* 'LHS)
      (values root-id expansion nil)
      ;; ELSE
      (values root-id nil expansion))))

;; Process-concatenate-common expands both the concatenate data macro
;; and the concatenate* directive.
;;
(defun expand-concatenate (&rest args)
  (process-concatenate-common (cons 'concatenate args) nil nil nil))

;;; Next come the functions that process specific directives.

;;; Directives for the set data type:

;; Process-member* takes the LHS and RHS of an SP-form production,
;; and a list of member* directives, and returns the modified LHS
;; and RHS as specified by those directives.  We assume that any data-macro
;; calls in the directive's arguments have already been expanded.  (See
;; collect-directives, which strips directives from an SP-format production,
;; and expands any data-macro arguments in them.)
;;
;; The member* set directive is only meaningful on the LHS.
;;
;; The directive's form is (member* <x> <set>), and matches if and only if
;; <x> is a member of <set>.  Set is assumed to be bound to the identifier of
;; a set object.
;; If <x> is otherwise unbound in the conditions, it will be bound to a member
;; of the specified set, if there is an appropriate matching member.  Both
;; <x> and <set> may be data macro calls.  In addition, <x> may be a constant.
;;
(defun process-member* (directives lhs rhs)
  (dolist (clause directives)

    (cond ((not (eql (length clause) 3))
           (taql-warn "member* takes exactly two arguments: ~S" clause))
          (t
           (setq lhs
                 (append
                  lhs
                  `((set ,(caddr clause) ^ member ,(cadr clause))))))))

  (values lhs rhs))

;; Process-set-edits takes the LHS and RHS of an SP-form production, and
;; a list of insert* and delete* directives, and returns the LHS and RHS
;; as modified by those directives.  We assume that any data-macro
;; calls in the directive's arguments have already been expanded.  (See
;; collect-directives, which strips directives from an SP-format production,
;; and expands any data-macro arguments in them.)
;;
;; The insert* and delete* set directives are only meaningful on the RHS.
;;
;; The directive forms are (insert* <x> <set>) and (delete* <x> <set>),
;; which insert and delete the specified elements from the specified set,
;; respectively.  Set is assumed to be bound to the identifier of a set
;; object.  Both <x> and <set> may be data macro calls.  In addition, <x>
;; may be a constant.
;;
(defun process-set-edits (directives lhs rhs)
  (dolist (clause directives)

    (cond ((not (eql (length clause) 3))
           (taql-warn "~S takes exactly two arguments: ~S"
                      (car clause) clause))
          (t
           (setq rhs
             (append
              rhs
              `((set ,(caddr clause)
                     ^ member ,(cadr clause) ,@(case (car clause)
                                                   (insert* '(+ &))
                                                   (delete* '(-))))))))))

  (values lhs rhs))

;;; Directives for the list data type:

;; Process-list-edits handles rplaca* and rplacd* directives for the list
;; data type.  These two directives can only be used on the RHS, and then
;; only where it would be sticky.  It can't be used in
;; propose-{task-}operator because the rule generated causes itself to
;; retract.
;;
;; Also, currently it is an undetected error to rplaca* or rplacd*
;; something with the same value that is already there.  This is
;; because the production it builds would both reject and accept the
;; same value.  This could be fixed by testing on the LHS that the
;; new value if different from the old value.  But, there is a
;; complication to doing that.  If the new value is the result of a
;; data macro call that creates new structure, then its value will
;; not be available for testing on the LHS.  Even this could be
;; worked around, by changing things so that data macros in the
;; arguments are not expanded until you get here.  Then, if the new
;; value is a data macro that creates new structure, don't put the <>
;; test on the LHS.  Note that for the rplaca* case, the data macro
;; could be ANY data macro that creates new structure, not just one
;; of the ones from the list data type.  So a quick fix would be
;; fragile.  What is needed is some general way of knowing whether a
;; given data macro is one that creates new structure or not.
;;
;; The first argument to rplaca*/rplacd* (the one that indicates which list
;; to modify) must access an existing list object (it can't be a Cons or
;; List).  We don't detect this.
;;
(defun process-list-edits (directives lhs rhs)
  (dolist (clause directives)

    (cond ((not (eql (length clause) 3))
           (taql-warn "~S takes exactly two arguments: ~S"
                      (car clause) clause))
          (t
	   (let ((old-value (genvar 'v)))
	     (setq lhs
		   (append
		    lhs
		    `((list ,(cadr clause)
			    ^ ,(case (car clause)
				 (rplaca* 'car)
				 (rplacd* 'cdr))
			    ,old-value))))
	     (setq rhs
		   (append
		    rhs
		    `((list ,(cadr clause)
			    ^ ,(case (car clause)
				 (rplaca* 'car)
				 (rplacd* 'cdr))
			    ,(caddr clause) +
			    ,old-value -))))))))

  (values lhs rhs))

;;; Directives for the text data type:

(defun process-concatenates (directives lhs rhs)
  (dolist (clause directives)
    (multiple-value-setq (lhs rhs)
      (process-concatenate-common clause lhs rhs t)))
  (values lhs rhs))

;; This function does the processing for both the concatenate data macro
;; and the concatenate* directive.  If it is being called for the
;; directive, the directive-p argument must be non-nil.
;;
;; The clause argument is the directive or data macro call to be processed.
;; Because arguments that are text literal data macros must be handled
;; specially, any data-macros in the arguments must NOT be expanded before
;; this function is called.
;;
;; When called with directive-p = NIL, should also have lhs = rhs = NIL.
;; Otherwise lhs and rhs are the lhs and rhs of the production containing
;; the directive.
;;
;; The concatenate data macro is an operator-like thing in that is does
;; rejects/adds on the text objects that must be sticky.  Therefore one
;; may only use concatenate in places where that would be true
;; (apply-operator edits, for example).
;;
;; In the generated actions, we remove the ^last pointer for all but the first
;; string in the concatenate.  If we tried to update the last pointers
;; instead, the rules could reinstantiate on the post-concatenation last
;; pointers.  A consequence of this is that a given string cannot be used
;; in the middle of a concatenate more than once.  (Even if we tried to
;; update last pointers, it seems difficult to maintain their consistency --
;; suppose we did <A> := (concatenate <A> <B>) and then
;; <A> := (concatenate <A> <C>).  How would we get <B>'s last pointer changed
;; to <C>.last after the second concatenate?)
;;
;; IMPORTANT USER NOTE:
;; If the first string in a concatenate is the
;; only non-literal string in the concatenate, the rule could
;; still reinstantiate given the above scheme.  In such cases, it is the
;; user's responsibility to prevent looping.  Possibilities are to insert
;; a literal null string (e.g., (text)) at the beginning of the concatenate,
;; or to test other task-specific conditions that will prevent the production
;; from firing again.  If the concatenate appears in a non-sliding edit
;; clause of apply-operator or result-superstate, there is no problem, as
;; the TAQL compiler ensures that such productions get only one chance to
;; fire per operator application.
;;
;; A further problem is that if there is a non-literal in other than the
;; last position, the production will retract after it fires
;; (since the last pointers of all but the last string have changed/been
;; deleted).  Thus, for example, except in this one case concatenate cannot
;; be used in propose-operator, even though it creates operators stickily --
;; the acceptable preference for the operator would retract.
;;
(defun process-concatenate-common (clause lhs rhs directive-p)
  (let ((func-name (car clause)))

    (when (eql *pseudo-sp-context* 'LHS)
      (taql-warn "~S cannot be used in conditions." func-name)
      (return-from process-concatenate-common
		   (if directive-p
		     (values lhs rhs)
		     ;; ELSE
		     (values (genvar 'junk) lhs rhs))))

    (cond ((not (cddr clause))
	   (taql-warn "~S takes at least two arguments: ~S" func-name clause))
	  (t
	   (let ((text-ids-0 nil)
		 (last-last-id nil)
		 (extra-lhs nil)
		 (extra-rhs nil))

	     (when (and directive-p (text-literal-p (cadr clause)))
	       (taql-warn
		"First argument to ~S cannot be a text literal: ~S"
		func-name clause))

	     (multiple-value-setq (text-ids-0 extra-lhs extra-rhs)
	       (expand-data-macros-in-list (cdr clause)))

	     (setq last-last-id
		   (get-last-text-id
		      (car (last clause))
		      (car (last text-ids-0))
		      extra-rhs))

	     (do* ((text-ids text-ids-0 (cdr text-ids))
		   (text-id (car text-ids) (car text-ids))
		   (original-args (cdr clause) (cdr original-args))
		   (original-arg (car original-args) (car original-args))
		   (arg-is-text-literal
		      (text-literal-p original-arg)
		      (text-literal-p original-arg))
		   (last-id
		      (get-last-text-id original-arg text-id extra-rhs)
		      (get-last-text-id original-arg text-id extra-rhs)))

		 ((null (cdr text-ids))
		  (setq extra-rhs
			(cons `(text ,(car text-ids-0) ^ last ,last-last-id)
			      extra-rhs))
		  (when (not arg-is-text-literal)
		    (setq extra-lhs
			  (cons `(text ,text-id ^ last ,last-last-id)
				extra-lhs))))

	       (when (and (listp original-arg)
			  (eql (car original-arg) 'concatenate))
		 (taql-warn "Concatenate calls cannot be nested: ~S"
			    clause))

	       (when (not arg-is-text-literal)
		 (setq extra-lhs
		       (cons `(text ,text-id ^ last ,last-id)
			     extra-lhs)))
	       (setq extra-rhs
		     (nconc `((text ,text-id ^ last ,last-id -)
			      (text ,last-id ^ next ,(cadr text-ids)))
			    extra-rhs)))

	     (setq lhs (append lhs extra-lhs))
	     (setq rhs (append rhs extra-rhs))

	     (if directive-p
	       (values lhs rhs)
	       ;; ELSE
	       (values (car text-ids-0) lhs rhs)))))))

(defun text-literal-p (arg)
  (and (listp arg)
       (member (car arg) '(text text1 text2))))

;; Return an identifier variable to represent the last element of a text
;; string.  Arg is the unexpanded argument in the concatenate call, and
;; represents a text string.  If it is not a text literal data macro call,
;; we just return a genvar.  If it is a data macro call, then the variable
;; we want already exists somewhere among the rhs actions of its expansion,
;; and we get it from there.  The id argument is the root-id that results
;; from expanding arg if it is a data macro.
;;
(defun get-last-text-id (arg id rhs)
  (cond ((not (text-literal-p arg))
	 (genvar 'l))
	(t
	 (let ((action
		(find-if #'(lambda (rhs-action)
			     (and (eql (cadr rhs-action) id)
				  (member 'last rhs-action)))
			 rhs)))
	   (cadr (member 'last action))))))

;;; The equal* directive, which is not associated with any specific
;;; data type:

;; Process equal* directives in an SP-format production's LHS and
;; RHS, returning the modified LHS and RHS.  In the result, variable
;; substitutions achieving the desired equalities have been
;; performed.  Process-equal* takes a directives argument, which is a
;; list of equal* directives to be processed.  We assume that any
;; data-macro calls in the directive's arguments have already been
;; expanded.  (See collect-directives, which strips directives from
;; an SP-format production, and expands any data-macro arguments in
;; them.)
;;
;; An equal* condition has the form (equal* arg1 arg2), where arg1
;; and arg2 are both either constants, variables or data-macro calls.
;; Arguments that are data-macros are expanded to conditions, and the
;; data-macro call in the equal* is replaced by the root variable of
;; the expansion.  Equal* constrains the variables specified by arg1
;; and arg2 to match the same thing (this is accomplished by
;; substituting one variable for the other throughout the conditions
;; and actions).
;;
;; If one of the arguments is a constant (eg a number or a symbol),
;; it is substituted for the variable throughout the production.  It
;; is an error to have equal* clauses that test that a variable is
;; equal to to different constants.  It is also an error to test that
;; a variable that appears in the identifier field of some condition
;; or action is equal to a constant.  Currently, TAQL does not detect
;; this error -- it will substitute the constant for the variable,
;; and Soar will later generate an error about having a constant in
;; the identifier field.
;;
;; A further restriction on the arguments to equal* is that at least
;; one of the arguments must be either a bound variable or a
;; data-macro call that expands to access some component of a bound
;; variable.  For example, (equal* (car (cdr <x>)) (cons 2 <y>)) is
;; legal if <x> and <y> are bound, but (equal* (list (2 3)) (cons <z>
;; <w>)) is not, because LIST and CONS do not simply access
;; components of their arguments.  I don't know if this restriction
;; will ever be removed.  The motivation for this restriction is an
;; implementation difficulty: the variable representing such
;; non-accessor data-macro calls would be unbound elsewhere in the
;; conditions, and would lead to unlinked conditions in the resulting
;; production.  TAQL does not detect such errors, rather they will be
;; detected at the Soar level as unlinked conditions.
;;
;; IMPLEMENTOR'S NOTE: If TAQL were to try to detect places where
;; both args to equal* were constants or constructor data-macros, it
;; would have to flag cases such as (equal* (car (cons <x> <y>)) <z>)
;; as errors too (where the constructor macro is not the top-level
;; call).
;;
(defun process-equal* (directives lhs rhs)
  (when (check-equal*-directives directives)
    (let ((equiv-classes (form-equal-equiv-classes directives)))
      (dolist (class equiv-classes)
        (setq lhs (subst-if
                   (car class)
                   #'(lambda (x)
                       (member x (cdr class)))
                   lhs))
        (setq rhs (subst-if
                   (car class)
                   #'(lambda (x)
                       (member x (cdr class)))
                   rhs)))))
  (values lhs rhs))

(defun check-equal*-directives (directives)
  (let ((found-error nil))
    (dolist (clause directives)
      (cond ((not (eql (length clause) 3))
             (taql-warn "Equal* test must have exactly two arguments: ~S"
                        clause)
             (setq found-error t))
            ((or (consp (cadr clause))
                 (consp (caddr clause)))
             ;; Any data-macros in the directive arguments have already been
             ;; expanded by this point.
             (taql-warn
                "Arguments to equal* must be symbols or data-macro calls: ~S"
                clause)
             (setq found-error t))))
    (not found-error)))

;; Return a list of lists, each sublist being a list of things that are
;; constrained to be equal to each other by the equals* clauses passed as
;; arguments.  For example,
;;  (form-equal-equiv-classes
;;       '((equal* <a> <b>) (equal* <b> <c>) (equal* <d> <e>)))
;; yields ((<a> <c> <b>) (<d> <e>)).
;;
;; If any equivalence class ends up with more than one constant in it, an
;; error message is printed and NIL is returned.  Otherwise, the list of
;; equivalence classes is returned, and in any class containing a constant,
;; that constant appears at the beginning of the list representing that class.
;;
(defun form-equal-equiv-classes (equals)
  (let ((classes nil))
    (dolist (clause equals)
      (let* ((item1 (cadr clause))
             (item2 (caddr clause)))
        (when (not (eql item1 item2))
          (let ((class1 (find item1 classes :test #'member))
                (class2 (find item2 classes :test #'member)))
            (if class1
              (if class2
                (when (not (eq class1 class2)) ; coalesce classes
                  (nconc class1 class2)
                  (setq classes (delete class2 classes :test #'eq)))
                ;; ELSE item2 not in any class yet, add to class1
                ;; Don't want to change the pointer to the head of the
                ;; class, so we make the new element the second element.
                (setf (cdr class1) (cons item2 (cdr class1))))
              ;; ELSE item1 is not in any class yet
              (if class2 ; add item1 to class2
                (setf (cdr class2) (cons item1 (cdr class2)))
                ;; ELSE neither item1 nor item2 is in a class yet, create a new
                ;; class
                (setq classes (cons (list item1 item2) classes))))))))
    (check-equivalence-classes classes)))

(defun check-equivalence-classes (classes)
  (let ((result nil))
    (dolist (class classes)
      (let ((consts (collect #'(lambda (x)
                                    (and (atom x)
                                         (not (variable-p x))))
                      class)))

        (when (> (length consts) 1)
          (taql-warn
            "Equal* constrains the following to all be equal,~% but more than one constant appears: ~S"
            class)
          (return-from check-equivalence-classes nil))

        (if consts
          (push (cons (car consts)
                      (delete (car consts) class))
                result)
          ;; ELSE
          (push class result))))
    result))

;;; Declare the data macro expanders.

;;;  A data-macro that can appear on the LHS must always be compatible
;;; with at least the = relation, since that is the default relation
;;; when none is given explicitly.  I suppose there could be some odd
;;; sort of data macro that, say, can only appear after <>.  But I
;;; can't think of any.

(eval-when (load eval)

  ;; List data type:

  (declare-data-macro 'car 'list #'expand-car t *lhs-relations*)
  (declare-data-macro 'cdr 'list #'expand-cdr t '(= <>))
  (declare-data-macro 'cons 'list #'expand-cons t '(=))
  (declare-data-macro 'null 'list #'expand-null t '(= <>))
  (declare-data-macro 'list 'list #'expand-list nil '(=))
  
  ;; Set data type:
  
  (declare-data-macro 'empty-set 'set #'expand-empty-set t '(=))
  (declare-data-macro 'set 'set #'expand-set t '(=))
  
  ;; Expression data type:
  
  (declare-data-macro 'expr 'expression #'expand-expr nil '(=))
  ;; Expr expands data macros that appear as values in the expression.
  
  ;; Text data type:
  
  (declare-data-macro 'text1 'text #'expand-text1 t '(=))
  (declare-data-macro 'text2 'text #'expand-text2 t '(=))
  (declare-data-macro 'text 'text #'expand-text2 t '(=))
  (declare-data-macro 'concatenate 'text #'expand-concatenate nil nil)
  ;; Concatenate is not permitted on the LHS, or anywhere else outside
  ;; of actions of TCs that make sticky edits.  It does expand data
  ;; macros in its arguments eventually, but needs to get its hands
  ;; on the unprocessed arguments first.

  )

(eval-when (compile load eval)
  (soarsyntax))
