;*- Mode:Lisp; Package:USER; Base:10 -*-

; core3.5.lisp
; Soar 5.2.2
; Soowon Lee	8/27/91

; Core file for function capability

(lispsyntax)

(trace-attributes '(
		    (operator arglist) (operator name)
		    (term contents) (term next)
		    (boolean name)
		    (column digit) (column right) (integer head) 
		    (real integer) (real fraction) (fraction head)
		    ))
(multi-attributes '(
		    (operator arg)
		    ))


; Functions which convert from an external expression notation to the Soar
; internal representation

; External abbreviations for arithmetic functions
; Full names can also be used
(setq *arithmetic-operators*
      '((+ . add)
	(- . subtract)
	(* . multiply)
	(/ . divide)
	(% . remainder)
	(= . equal)
	(<> . not-equal)
	(< . less-than)
	(> . greater-than)
	(<= . less-than-or-equal)
	(>= . greater-than-or-equal)
	)
)

; For TAQL
(setf (symbol-function 'function-genvar)
      (symbol-function ' #-taql soar-genvar
			 #+taql genvar))

; (term <t> ^contents <v> ^type value)
; (term <t> ^contents <v> ^type operator)
; Will have either a value xor an operator (for a subexpression)
(defun convert-term-to-soar (term tvar rhs-p)	; :DSM
  (cond ((integerp term) (convert-integer-to-soar    term tvar rhs-p))
	((numberp term)  (convert-real-to-soar       term tvar rhs-p))
	((consp term)    (convert-expression-to-soar term tvar rhs-p))
	(t               (convert-boolean-to-soar    term tvar)))
)

; :DSM
(defun convert-term-to-soar-io (term tvar fun-name)
  (cond ((integerp term) (convert-integer-to-soar-io    term tvar fun-name))
	((numberp term)  (convert-real-to-soar-io       term tvar fun-name))
	((consp term)    (convert-expression-to-soar-io term tvar ))
	(t               (convert-boolean-to-soar-io    term tvar fun-name)))
)

; (boolean <b> ^name <n>)
(defun convert-boolean-to-soar (boolean bvar)
  `((boolean ,bvar ^ name ,boolean))
)

; :DSM
(defun convert-boolean-to-soar-io (boolean bvar fun-name)
  (add-input fun-name 'boolean bvar 'name boolean)
)

; (operator <q> ^name <n> ^arglist <a> ^arg ...)
(defun convert-expression-to-soar (expr evar rhs-p)
  (let* ((slvar (function-genvar 'l))
	 (opr-pair (assoc (car expr) *arithmetic-operators*))
	 (name (if opr-pair (cdr opr-pair) (car expr))))
    (cons `(operator ,evar ^ name ,name ^ arglist ,slvar)
	  (convert-terms-to-soar (cdr expr) slvar evar rhs-p)))
)

; :DSM
(defun convert-expression-to-soar-io (expr evar)
  (let* ((slvar (soar::make-io-object-symbol 'l))
	 (opr-pair (assoc (car expr) *arithmetic-operators*))
	 (name (if opr-pair (cdr opr-pair) (car expr))))
    (add-input (car expr) 'operator evar 'name name)
    (add-input (car expr) 'operator evar 'arglist slvar)
    (convert-terms-to-soar-io (cdr expr) slvar evar (car expr)))
)

; If a list contains a real number as its argument, then return true
(defun real-member (l)
  (cond ((null l) nil)
        ((and (numberp (car l)) (not (integerp (car l)))) t)
	(t (real-member (cdr l)))))

; If any of arguments of a list is real number, then convert all arguments
; into real.
(defun type-conversion (l)
  (if (real-member l) (mapcar 'float l) l))
	
  
; Generate arguments for an operator/expression, each of which is a term
; Real number conversion is done if any of input arguments is real 
; Soowon Lee (4/12/89)
(defun convert-terms-to-soar (raw-terms lvar evar rhs-p)
  (let ((stvar (function-genvar 't))
	(slvar (function-genvar 'l))
;	(terms (type-conversion raw-terms))	; Type conversion is ignored
						; in this version
	(terms raw-terms))
    (cons (append `(term ,lvar)
		  (cond ((consp (car terms)) `(^ contents ,stvar ^ type operator))
			((soar::variablep (car terms)) `(^ contents ,(car terms) ^ type value))
			((eq (car terms) *ignore-argument-symbol*) nil)
			(t `(^ contents ,stvar ^ type value)))
		  (if (null (cdr terms))
		      `(^ next none)
		    `(^ next ,slvar)))
	  (append (if evar 
			(if rhs-p `((operator ,evar ^ arg ,lvar ,lvar &))
			          `((operator ,evar ^ arg ,lvar)))
			nil)
		  (if (or (soar::variablep (car terms))
			  (eq (car terms) *ignore-argument-symbol*))
		      nil
		    (convert-term-to-soar (car terms) stvar rhs-p))
		  (if (null (cdr terms))
		      nil
		    (convert-terms-to-soar (cdr terms) slvar evar rhs-p)))))
)

; :DSM
(defun convert-terms-to-soar-io (terms lvar evar fun-name)
  (let ((stvar (soar::make-io-object-symbol 't))
	(slvar (soar::make-io-object-symbol 'l)))
    (cond ((consp (car terms)) 
			(add-input fun-name 'term lvar 'contents stvar)
			(add-input fun-name 'term lvar 'type 'operator))
	  ((soar::variablep (car terms)) 
			(add-input fun-name 'term lvar 'contents (car terms))
			(add-input fun-name 'term lvar 'type 'value))
	  ((eq (car terms) *ignore-argument-symbol*) 
			(add-input fun-name 'term lvar))
	  (t		(add-input fun-name 'term lvar 'contents stvar)
	        	(add-input fun-name 'term lvar 'type 'value)))
    (if (null (cdr terms))	
			(add-input fun-name 'term lvar 'next 'none)
			(add-input fun-name 'term lvar 'next slvar))
    (if evar 		(add-input fun-name 'operator evar 'arg lvar))
    (if (or (soar::variablep (car terms))
	    (eq (car terms) *ignore-argument-symbol*))
         nil
	 (convert-term-to-soar-io (car terms) stvar fun-name))
    (if (null (cdr terms))
        nil
        (convert-terms-to-soar-io (cdr terms) slvar evar fun-name)))
)

; (integer <i> ^sign <z> ^head <h> ^tail <t>)
; (column <h> ^digit <d> ^anchor head ^right <t>)
; and so on
; Support negative integers - Soowon Lee 03/14/89

(defun convert-integer-to-soar (int ivar rhs-p)	 ; :DSM
  (append 
    (if (minusp int) `((integer ,ivar ^ sign negative))
		     `((integer ,ivar ^ sign positive)))
    (do* ((int (if (minusp int) (* -1 int) int) (floor int 10))
	  (pcvar nil cvar)
	  (cvar (function-genvar 'c) (function-genvar 'c))
	  (dvar (function-genvar 'd) (function-genvar 'd))
	  (soarint (if rhs-p `((integer ,ivar ^ tail ,cvar)
		     	       (column ,cvar ^ anchor tail tail &))
	                     `((integer ,ivar ^ tail ,cvar)
		     	       (column ,cvar ^ anchor tail)))
		   (append soarint
			   `((column ,pcvar ^ left ,cvar)
			     (column ,cvar ^ right ,pcvar)))))
         ((equal (floor int 10) 0)
	  (append soarint
		  (if rhs-p `((column ,cvar ^ anchor head head & ^ digit ,dvar)
		    	      (digit ,dvar ^ name ,int)
		    	      (integer ,ivar ^ head ,cvar))
		  	    `((column ,cvar ^ anchor head ^ digit ,dvar)
		    	      (digit ,dvar ^ name ,int)
		    	      (integer ,ivar ^ head ,cvar)))))
         (setq soarint (append soarint
			       `((column ,cvar ^ digit ,dvar)
			         (digit ,dvar ^ name ,(rem int 10))))))
  )
)

; :DSM
(defun convert-integer-to-soar-io (int ivar fun-name)
 (if (minusp int) (add-input fun-name 'integer ivar 'sign 'negative)
                  (add-input fun-name 'integer ivar 'sign 'positive))
 "use soar-io to put number INT into memory spot IVAR for function FUN-NAME"
 (do* ((int (if (minusp int) (* -1 int) int) (floor int 10))
       (pcvar nil cvar)
       (cvar (wme-value (add-input fun-name 'integer ivar 'tail 
				   (soar::make-io-object-symbol 'c)))
             (wme-value (add-input fun-name 'column pcvar 'left
				   (soar::make-io-object-symbol 'c))))
       (dvar (progn (add-input fun-name 'column cvar 'anchor 'tail)  
                    (wme-value (add-input fun-name 'column cvar 'digit)) )
             (progn (add-input fun-name 'column cvar 'right pcvar)
                    (wme-value (add-input fun-name 'column cvar 'digit)))) )
      ((equal (floor int 10) 0)
       (progn (add-input fun-name 'column cvar 'anchor 'head)
	      (add-input fun-name 'digit dvar 'name int)
	      (add-input fun-name 'integer ivar 'head cvar)))
      (add-input fun-name 'digit dvar 'name (rem int 10)))
)

; Define variables for precision calculation

(defvar *precision* 5)		; default = 5
(defvar precision-integer)	
(defvar precision-real)	

; (set-precision) returns current setting (Oct. 21, 89)
(defun set-precision (&optional precision)
  (cond ((numberp precision) 
  	   (setq *precision* precision)
  	   (setq precision-integer 1)
  	   (dotimes (k precision)
    	     (setq precision-integer (* 10 precision-integer))) ;default 100000
  	   (setq precision-real (float precision-integer))))    ;default 100000.0
  (eval *precision*)
)

(set-precision *precision*)

; Calculate fixed-digit fraction part from real
(defun get-fraction (real)
  (if (minusp real)
	(/ (round (* (- (truncate real) real) precision-integer)) 
	   precision-real)
	(/ (round (* (- real (truncate real)) precision-integer)) 
	   precision-real))
)

; Shift fraction one digit left 
(defun shift-fraction-left (fraction)
  (/ (round (* (- (* fraction 10) 
		  (truncate (* fraction 10))) precision-integer)) 
     precision-real)
)
	 
; (real <r> ^sign <z> ^integer <i> ^fraction <f>) (4/12/89 Soowon Lee)
; (integer <i> ^sign <z> ^head <h> ^tail <t>)
; (fraction <f> ^head <fh> ^tail <ft>)

(defun convert-real-to-soar (real rvar rhs-p)
  (let ((ivar (function-genvar 'i))
        (fvar (function-genvar 'f)))
    (append
      (if (minusp real) `((real ,rvar ^ sign negative))
 		        `((real ,rvar ^ sign positive)))
      `((real ,rvar ^ integer ,ivar))
      `((real ,rvar ^ fraction ,fvar))
      (convert-integer-to-soar (truncate real) ivar rhs-p)
      (do* ((fraction (get-fraction real)
		      (shift-fraction-left fraction))
	     (pcvar nil cvar)
	     (cvar (function-genvar 'c) (function-genvar 'c))
	     (dvar (function-genvar 'd) (function-genvar 'd))
	     (soarfraction (if rhs-p `((fraction ,fvar ^ head ,cvar)
		             	       (column ,cvar ^ anchor head head &))
				     `((fraction ,fvar ^ head ,cvar)
		             	       (column ,cvar ^ anchor head)))
		           (append soarfraction
			      `((column ,pcvar ^ right ,cvar)
			        (column ,cvar ^ left ,pcvar)))))
            ((= (- fraction (/ (truncate (* 10 fraction)) 10)) 0.0)
	     (append soarfraction
		     (if rhs-p `((column ,cvar ^ anchor tail tail & ^ digit ,dvar)
		       		 (digit ,dvar ^ name ,(truncate (* 10 fraction)))
		       		 (fraction ,fvar ^ tail ,cvar))
		     	       `((column ,cvar ^ anchor tail ^ digit ,dvar)
		       		(digit ,dvar ^ name ,(truncate (* 10 fraction)))
		       		(fraction ,fvar ^ tail ,cvar)))))
            (setq soarfraction (append soarfraction
			          `((column ,cvar ^ digit ,dvar)
			            (digit ,dvar ^ name 
				       ,(truncate (* 10 fraction)))))))
    )
  )
)

; :DSM
(defun convert-real-to-soar-io (real rvar fun-name)
  (let ((ivar (soar::make-io-object-symbol 'i))
        (fvar (soar::make-io-object-symbol 'f)))
    (if (minusp real) (add-input fun-name 'real rvar 'sign 'negative)
                      (add-input fun-name 'real rvar 'sign 'positive))
    (add-input fun-name 'real rvar 'integer ivar)
    (add-input fun-name 'real rvar 'fraction fvar)
    (convert-integer-to-soar-io (truncate real) ivar fun-name)
    (cond ((and (minusp real) (equal 0 (truncate real)))	; -1 < real < 0
		(delete-input fun-name 'integer ivar 'sign 'positive)	
		(add-input    fun-name 'integer ivar 'sign 'negative)))
    (do* ((fraction (get-fraction real)
		    (shift-fraction-left fraction))
	  (pcvar nil cvar)
	  (cvar (wme-value (add-input fun-name 'fraction fvar 'head 
				      (soar::make-io-object-symbol 'c)))
	        (wme-value (add-input fun-name 'column pcvar 'right
				      (soar::make-io-object-symbol 'c))))
	  (dvar (progn (add-input fun-name 'column cvar 'anchor 'head)
	               (wme-value (add-input fun-name 'column cvar 'digit)))
	        (progn (add-input fun-name 'column cvar 'left pcvar)
	               (wme-value (add-input fun-name 'column cvar 'digit)))))

         ((= (- fraction (/ (truncate (* 10 fraction)) 10)) 0.0)
          (progn (add-input fun-name 'column cvar 'anchor 'tail)
	         (add-input fun-name 'digit dvar 'name 
				     		  (truncate (* 10 fraction)))
	         (add-input fun-name 'fraction fvar 'tail cvar)))

	 (add-input fun-name 'digit dvar 'name (truncate (* 10 fraction)))
    )
  )
)


; Allow new functions to be defined in Soar, for example:
; (sdefun xor (<x1> <x2>)
;         (or (and <x1> (not <x2>)) (and (not <x1>) <x2>)))
; Accepts soar variables, boolean constants, and integers as arguments so
; can give a partial definition of a function, for example:
; (sdefun add (0 <x2>) <x2>)
; Gets converted into productions

; Symbol which marks a parameter as being ignored
; eg.: (sdefun if (true <x1> _) <x1>)
(setq *ignore-argument-symbol* '_)

; Symbol which marks a parameter as matching any argument
; If used, should not test any other arguments
; eg.: (sdefun and (* false) false)
(setq *any-argument-symbol* '*)

; Macro to allow defining new functions in Soar
(defmacro sdefun (name arguments expression)
  (sdefun-function name arguments expression)
)

; Macro to allow declaring new functions in Soar
(defmacro sfunction (name)
  (sfunction-function name)
)

; Function to allow declaring new functions in Soar
; sdefun will automatically declare it, but otherwise need to call explicitly
(defun sfunction-function (name)
  (unless (sp? (soar-function-name name nil "TYPE"))	; Soar 5.2.2
          (soar-function-type name))
)

; To define a new function create a production that marks it as a function
; and one which either generates a value directly or generates an operator
; to compute an appropriate expression in a subgoal
(defun sdefun-function (name arguments expression)
  (sfunction-function name)
  (if (consp expression)
      (soar-function-gsso name arguments expression)
    (if (eq (car arguments) *any-argument-symbol*)
	    (soar-function-any-value name (cdr arguments) expression)
      (soar-function-value name arguments expression)))
)

; Get a string corresponding to name of object
(defun soar-function-object-name (object)
  (if (numberp object)
      (format nil "~d" object)
    (symbol-name object))
)

; Generate a name for a production
(defun soar-function-name (name arguments tail)
  (intern
   (concatenate 'string "FUNCTION*"
		(let ((str (symbol-name name)))
		  (mapc #'(lambda (arg)
				(setq str
				      (concatenate 'string str "*"
						   (soar-function-object-name arg))))
			arguments)
		  str)
		"*"
		tail))
)

; Generate a production which marks the new function as ^type function
(defun soar-function-type (name)
  (eval
   `(sp ,(soar-function-name name nil "TYPE")
       (goal <g> ^ state <s> ^ operator <q>)		; Soar5.2
       (operator <q> ^ name ,name)
       -->
       (operator <q> ^ type function function &)	; :DSM
       (state <s> ^ done yes + &)			; Soar5.2
       )
   )
)

; Generate actions to be used in productions generated from sdefuns
(defun soar-function-actions (expression)
  (if (soar::variablep expression)
      `((operator <q> ^ value ,expression))
    (append '((operator <q> ^ value <v>) 
	      (state <s> ^ done yes + &)	; Soar5.2
	      )
	    (convert-term-to-soar expression '<v> t)))
)

; Generates a production which produces a value for a function
; Assumes expression a boolean, integer, or variable rather than an operator
(defun soar-function-value (name arguments expression)
  (eval
   (append
    `(sp ,(soar-function-name name arguments "VALUE")
	 (goal <g> ^ state <s> ^ operator <q>)
	 (operator <q> ^ name ,name ^ arglist <a0>)
	 )
    (convert-terms-to-soar arguments '<a0> nil nil)
    `(
      -->
      )
    (soar-function-actions expression)
    )
   )
)

; Special case code for when math any argument
; If pattern mathes more than once, don't want multiple values, so
; use a special symbol that each generates, and then key off of
; symbol once to generate value
(defun soar-function-any-value (name arguments expression)
  (soar-function-any-generate-any name arguments)
  (soar-function-any-generate-value name arguments expression)
)

; Generate symbol if pattern matches an argument
(defun soar-function-any-generate-any (name arguments)
  (eval
   (append
    `(sp ,(soar-function-name name arguments "GENERATE-ANY")
	 (goal <g> ^ state <s> ^ operator <q>)
	 (operator <q> ^ name ,name ^ arg <a>)
	 (term <a> ^ contents <v> ^ type value)
	 )
    (convert-term-to-soar (car arguments) '<v> nil)
    `(
      -->
      (operator <q> ^ any ,(car arguments))
      )
    )
   )
)

; Generate value if any argument has matched pattern
(defun soar-function-any-generate-value (name arguments expression)
  (eval
   (append
    `(sp ,(soar-function-name name arguments "GENERATE-VALUE")
	 (goal <g> ^ state <s> ^ operator <q>)
	 (operator <q> ^ name ,name ^ any ,(car arguments))
	 -->
	 )
    (soar-function-actions expression)
    )
   )
)

; Generate a production which generates an operator in a subgoal which
; evaluates the function's defining expression
(defun soar-function-gsso (name arguments expression)
  (eval
   (append
    `(sp ,(soar-function-name name arguments "GOAL-SPACE-STATE-OPERATOR")
	 (goal <g> ^ attribute operator ^ impasse no-change ^ object <sg>)
	 (goal <sg> ^ operator <sq>)
	 (operator <sq> ^ name ,name ^ arglist <a0>)
	 )
    (convert-terms-to-soar arguments '<a0> nil nil)
    `(
      -->
      (goal <g> ^ desired <d>)
      (desired <d> ^ evaluate <q>)
      )
    (convert-expression-to-soar expression '<q> t)
    `(
	(goal <g> ^ problem-space <p>)		; :DSM
	(problem-space <p> ^ name function)	; :DSM
	(goal <g> ^ state <s>)			; :DSM
	(state <s> ^ name function)		; :DSM
	(goal <g> ^ operator <q>)		; :DSM
      )
    )
   )
)


(soarsyntax)
