
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;; 
;; File:         eight with soar/io for soar5.1
;; Author:       Michael Hucka
;; Organization: University of Michigan
;; Created:      Wed Mar  8 13:32:14 1989
;; RCS           $Header: eight-ext.lisp,v 1.1 89/03/22 21:19:01 hucka Locked $
;; Contents:     Lisp support for external Eight Puzzle board.
;;       
;; Updated:      Clare Bates Congdon -- October 30, 1989 
;;               To test Soar 5.1 IO code
;;               G. A. Pelton -- February 11. 1991
;;               To remove *unbound* 
;;               G. A. Pelton -- February 13. 1991
;;               To fix the output routine
;;					
;; 
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;
;;  INPUT INTERFACE FUNCTIONS USED IN THIS FILE:
;;   current-input-link
;;   new-input-link
;;   add-input
;;   delete-input
;;   match-input
;;   wme-value
;;   wme-id
;;  NOT USED:
;;   get-input-link
;;   change-input-value
;;   change-input-value-and-old
;;   top-state
;;   wme-class
;;   wme-attribute
;;
;;  OUTPUT INTERFACE FUNCTIONS USED IN THIS FILE:
;;   (none)
;;  NOT USED:
;;   get-output-values
;;   get-output-augmentations
;;
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

(defvar *in-link-owner* 'board-link-fun)
(defvar *out-link-owner* 'board-link-fun)
(defvar *next-move* nil)

(set-input-functions (look-at-board))
         
#+(or)   ;replaced line. -KAM 11/1/89
(set-output-mappings ((move-tile tile)))
(set-output-mappings ((move-tile move-tile)))



;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;; INPUT
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

;; look-at-board is the input function used by Soar IO.  
;; Normally, after the first time it's called, it calls update-board which
;; modifies the bindings directly in WM based on the changes specified by the
;; last call to move-tile.  The exception to this rule is that the first time
;; it's called it must create the whole board.
;;
(defun look-at-board ()
  (let* ((input-item-class 'board)
	 (input-link-attribute 'board)
    	 (input-link (current-input-link 'board-link-fun)))
    (if (null input-link)		; First time called
      (create-board input-link-attribute)
      (update-board input-item-class input-link))))


;;
;; create-board sets up the board directly in Soar's WM.  Because the Eight
;; Puzzle needs quite a few explicit objects, this function is ugly and long.
;;
(defun create-board (input-link-attrib)
  (let (link link-item-id)

    ;; Create one input link which will point to the board object
    (setf link (new-input-link *in-link-owner* input-link-attrib))
    (setf input-item-id (wme-value link))

    ;; Create 9 cell binding augmentations off the board object
    (dotimes (i 9)
      (add-wme 'board input-item-id 'binding))

    ;; Now the ugliness starts.  We want to create cells with specific names
    ;;  and build specific relationships between cells to create a matrix which
    ;;  constitutes the board.  We also want 9 tile objects with specific names.
    ;;  (In this case the names are integers from 0-8.)  Linking these tiles
    ;;  to the cells are binding objects, somewhat like this:
    ;;
    ;;        (cell x ^cell y)         (tile t ^name i)
    ;;             \                       /
    ;;              \                     /
    ;;                (binding b ^cell x)
    ;;                (binding b ^tile t)
    ;;
    ;; The initial set of bindings constitute the initial board configuration.

    ;; Create the cell matrix and the initial configuration of tile/cell bindings:
    (add-wme 'cell 'c11 'cell 'c12)
    (add-wme 'cell 'c11 'cell 'c21)
    (add-wme 'cell 'c12 'cell 'c11)
    (add-wme 'cell 'c12 'cell 'c13)
    (add-wme 'cell 'c12 'cell 'c22)
    (add-wme 'cell 'c13 'cell 'c12)
    (add-wme 'cell 'c13 'cell 'c23)
    (add-wme 'cell 'c21 'cell 'c11)
    (add-wme 'cell 'c21 'cell 'c31)
    (add-wme 'cell 'c21 'cell 'c22)
    (add-wme 'cell 'c22 'cell 'c21) 
    (add-wme 'cell 'c22 'cell 'c12)
    (add-wme 'cell 'c22 'cell 'c23)
    (add-wme 'cell 'c22 'cell 'c32)
    (add-wme 'cell 'c23 'cell 'c22)
    (add-wme 'cell 'c23 'cell 'c13)
    (add-wme 'cell 'c23 'cell 'c33)
    (add-wme 'cell 'c31 'cell 'c32)
    (add-wme 'cell 'c31 'cell 'c21)
    (add-wme 'cell 'c32 'cell 'c31)
    (add-wme 'cell 'c32 'cell 'c22)
    (add-wme 'cell 'c32 'cell 'c33)
    (add-wme 'cell 'c33 'cell 'c32)
    (add-wme 'cell 'c33 'cell 'c23)

    (let* ((binding-augs (match-input :attribute 'binding))
	   (b0 (wme-value (nth 0 binding-augs))) 
	   (b1 (wme-value (nth 1 binding-augs)))
	   (b2 (wme-value (nth 2 binding-augs))) 
	   (b3 (wme-value (nth 3 binding-augs)))
	   (b4 (wme-value (nth 4 binding-augs))) 
	   (b5 (wme-value (nth 5 binding-augs)))
	   (b6 (wme-value (nth 6 binding-augs))) 
	   (b7 (wme-value (nth 7 binding-augs)))
	   (b8 (wme-value (nth 8 binding-augs))))
      (add-wme 'binding b0 'cell 'c11)
      (add-wme 'tile (wme-value (add-wme 'binding b0 'tile)) 'name 2)
      (add-wme 'binding b1 'cell 'c12)
      (add-wme 'tile (wme-value (add-wme 'binding b1 'tile)) 'name 1)
      (add-wme 'binding b2 'cell 'c13)
      (add-wme 'tile (wme-value (add-wme 'binding b2 'tile)) 'name 7)
      (add-wme 'binding b3 'cell 'c21)
      (add-wme 'tile (wme-value (add-wme 'binding b3 'tile)) 'name 8)
      (add-wme 'binding b4 'cell 'c22)
      (add-wme 'tile (wme-value (add-wme 'binding b4 'tile)) 'name 6)
      (add-wme 'binding b5 'cell 'c23)
      (add-wme 'tile (wme-value (add-wme 'binding b5 'tile)) 'name 0)
      (add-wme 'binding b6 'cell 'c31)
      (add-wme 'tile (wme-value (add-wme 'binding b6 'tile)) 'name 3)
      (add-wme 'binding b7 'cell 'c32)
      (add-wme 'tile (wme-value (add-wme 'binding b7 'tile)) 'name 4)
      (add-wme 'binding b8 'cell 'c33)
      (add-wme 'tile (wme-value (add-wme 'binding b8 'tile)) 'name 5)))
  (print-board "Initial board configuration:"))


;; 
;; update-board
;; The intent is that this gets called when Soar "looks" at the board, which
;; is when the input function gets called through Soar IO and corresponds to
;; the time when we want Soar to notice a change in the board.  So, we add
;; and delete objects to/from WM according to the moves stored up in the last
;; call to move-tile.
;;
(defun update-board (class link)
  (unless (null *next-move*)
    (let* ((blank-tile (first *next-move*))
	   (blank-binding-id (second *next-move*))
	   (tile-to-move (third *next-move*))
	   (tile-binding-id (fourth *next-move*)))
      (add-wme 'binding blank-binding-id 'tile tile-to-move)
      (add-wme 'binding tile-binding-id 'tile blank-tile)
      (del-wme 'binding blank-binding-id 'tile blank-tile)
      (del-wme 'binding tile-binding-id 'tile tile-to-move))
    ;; Mark the fact we've updated the board for this move.
    (setf *next-move* nil)
    (print-board "Current external board")))

;;
;; print-board
;; Function to print out the current board configuration.
;; Does this by looking into WM at the binding objects.
;;
(defun print-board (msg)
  (let ((cell-bindings (make-cell-tile-alist)))
    (format t "~%~s~%" msg)
    (format t "-------------~%")
    (format t "| ~s | ~s | ~s |~%" (find-tile-in-cell 'c11 cell-bindings)
	                           (find-tile-in-cell 'c21 cell-bindings)
				   (find-tile-in-cell 'c31 cell-bindings))
    (format t "----+---+----~%")
    (format t "| ~s | ~s | ~s |~%" (find-tile-in-cell 'c12 cell-bindings)
	                           (find-tile-in-cell 'c22 cell-bindings)
				   (find-tile-in-cell 'c32 cell-bindings))
    (format t "----+---+----~%")
    (format t "| ~s | ~s | ~s |~%" (find-tile-in-cell 'c13 cell-bindings)
	                           (find-tile-in-cell 'c23 cell-bindings)
				   (find-tile-in-cell 'c33 cell-bindings))
    (format t "-------------~%")))


(defun find-tile-in-cell (cell bindings-alist)
  (second (assoc cell bindings-alist)))

(defun make-cell-tile-alist ()
  (let ((list-of-tiles (match-input :class 'tile :attribute 'name))
	cell-tile-alist)
    (dolist (tile-wme list-of-tiles)
      (let* ((binding-wme (first (match-input :class 'binding
					      :attribute 'tile
					      :value (wme-id tile-wme))))
	     (cell-wme (first (match-input :class 'binding
					   :attribute 'cell
					   :id (wme-id binding-wme)))))
	(push (list (wme-value cell-wme) (wme-value tile-wme)) cell-tile-alist)))
    cell-tile-alist))

;;
;; Short helper routines.
;;

(defun add-wme (class id attrib &optional (value '*unbound*))
  (if (eq value '*unbound*)
    (add-input *in-link-owner* class id attrib)
  ;;else
    (add-input *in-link-owner* class id attrib value))
)

(defun del-wme (class id attrib value)
  (delete-input *in-link-owner* class id attrib value))




;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;; OUTPUT
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

;; move-tile is the output function used by Soar IO.
;; Expects to be given one item, the cell containing the tile to be moved into
;;  the spot currently occupied by the blank cell.  It doesn't actually
;;  perform the move, but rather stores it on the *next-move* global variable.  
;;  The move is actually performed when look-at-board calls update-board, to
;;  simulate the act of Soar "looking" at the board and "seeing" a new
;;  configuration. 
;;
;;

(defun move-tile (args)
  (let* ((move-tile-ID (first (get-output-values 'move-tile args)))
         (move-command (first (get-output-augmentations move-tile-ID args)))
	 (cell-of-tile-to-move (wme-value move-command))
	 (cell-binding-id (wme-id (first (match-input :class 'binding
						      :attribute 'cell
						      :value cell-of-tile-to-move))))
	 (tile-to-move (wme-value (first (match-input :class 'binding
						      :attribute 'tile
						      :id cell-binding-id))))
	 (blank-tile (wme-id (first (match-input :class 'tile
						 :attribute 'name
						 :value 0))))
	 (tile-binding-id (wme-id (first (match-input :class 'binding
						      :attribute 'tile
						      :value tile-to-move))))
	 (blank-binding-id (wme-id (first (match-input :class 'binding
						    :attribute 'tile
						    :value blank-tile)))))
    (format t "~%~% ---- Moving tile ~s into the blank space. ----~%" tile-to-move)
    (format t "next-move is ~s~%" 
	    (list blank-tile blank-binding-id tile-to-move tile-binding-id))
    (setf *next-move* 
	  (list blank-tile blank-binding-id tile-to-move tile-binding-id))))



